# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core.input_event import *
from elisa.core.media_uri import MediaUri
from elisa.core.common import application
from elisa.core.utils.i18n import install_translation

from elisa.plugins.base.models.file import DirectoryModel
from elisa.plugins.base.models.device import VolumeModel

from elisa.plugins.base.messages.device import NewDeviceDetected, \
                                               DeviceRemoved

from elisa.core.utils import notifying_list

from elisa.plugins.poblesec.filesystem import FilesystemPreviewListController
from elisa.plugins.poblesec.base.preview_list import PreviewListController                                         
from elisa.plugins.poblesec.section import SectionMenuController
from elisa.plugins.poblesec.link import Link
from elisa.plugins.poblesec.widgets.selector import ActionSelector, \
                                                    ActionListSelector
from elisa.plugins.poblesec.widgets.menu_item import MenuItemWidgetWithAction

import os.path


_ = install_translation('poblesec')

HOME_URI = MediaUri({'scheme': 'file', 'path': os.path.expanduser('~')})


class MenuItem(object):

    media_type = None
    label = None
    icon = None
    uri = None


class DirectoryConfigureController(PreviewListController):

    node_widget = MenuItemWidgetWithAction
    list_widget = ActionListSelector

    def initialize(self, **args):
        dfr = super(DirectoryConfigureController, self).initialize()

        uri = args['uri']
        self._refresh_cb = args['refresh_cb']
        self._refresh_item = args['refresh_item']

        self.model = notifying_list.List()
        mi = MenuItem()
        mi.label = _("Add in Music Section")
        mi.media_type = "music"
        mi.icon = "elisa.plugins.poblesec.music_section"
        mi.uri = uri
        self.model.append(mi)

        mi = MenuItem()
        mi.label = _("Add in Video Section")
        mi.media_type = "video"
        mi.icon = "elisa.plugins.poblesec.video_section"
        mi.uri = uri
        self.model.append(mi)

        mi = MenuItem()
        mi.label = _("Add in Pictures Section")
        mi.media_type = "pictures"
        mi.icon = "elisa.plugins.poblesec.pictures_section"
        mi.uri = uri
        self.model.append(mi)

        return dfr

    def nodes_setup(self):
        super(DirectoryConfigureController,self).nodes_setup()
        #FIXME: why the renderer and the list are defined with class members
        #and not the selector ?
        self.selector = ActionSelector()

        self.selector.create_cursor("item")
        self.selector.create_cursor("action")

        self.nodes.connect('item-clicked', self.node_clicked)
        self.nodes.connect('action-clicked', self.action_clicked)

        theme = self.frontend.get_theme()
        left_cap = theme.get_resource('elisa.plugins.poblesec.selector_left_cap')
        right_cap = theme.get_resource('elisa.plugins.poblesec.selector_right_cap')
        body = theme.get_resource('elisa.plugins.poblesec.selector_body')
        self.selector.set_left_cap(left_cap)
        self.selector.set_right_cap(right_cap)
        self.selector.set_body(body)

        self.nodes.set_selector(self.selector)

    def node_renderer(self, item, widget):
        super(DirectoryConfigureController,self).node_renderer(item, widget)
        widget.label.label = item.label
        self.frontend.load_from_theme(item.icon, widget.icon)
        path = item.uri.path
        if path in application.media_directories.directories[item.media_type]:
            resource = 'elisa.plugins.poblesec.directory_settings_added'
            label = ''
        else:
            resource = 'elisa.plugins.poblesec.directory_settings_add'
            label = _('Add')
        self.frontend.load_from_theme(resource, widget.action_button.icon)
        widget.action_button.label = label

    def handle_input(self, manager, input_event):
        if  self.nodes.focus:
            if input_event.value == EventValue.KEY_GO_LEFT:
                if self.nodes.action_selected:
                    self.nodes.select_list_item()
                    return True
            elif input_event.value == EventValue.KEY_GO_RIGHT:
                if not self.nodes.action_selected:
                    self.nodes.select_action_item()
                    return True
            elif input_event.value == EventValue.KEY_OK:
                if self.nodes.action_selected:
                    index = self.nodes.selected_item_index
                    item = self.model[index]
                    self.toggle_for_model(item, index)
                    return True

        return super(DirectoryConfigureController,self).handle_input(manager, 
                                                                    input_event)

    def action_clicked(self, widget, model):
        self.toggle_for_model(model)

    def toggle_for_model(self, model, index=-1):
        application.media_directories.toggle_directory_installation \
                                    (model.uri, model.media_type)
        # that will trigger a refresh of the list widgets using
        # self.model
        if index == -1:
            index = self.model.index(model)
        self.model[index] = model

        # Refresh the state of the item in the parent controller.
        self._refresh_cb(self._refresh_item)


class MediaDirectoryList(SectionMenuController):

    def initialize(self, media_type):
        dfr = super(MediaDirectoryList, self).initialize()
        application.media_directories.refresh_directory_list(media_type)
        self.media_type = media_type
        self.widget.connect('focus', self._focus_cb)
        return dfr

    def set_frontend(self, frontend):
        super(MediaDirectoryList, self).set_frontend(frontend)
        self.list_media_directory(self.media_type)

    def list_media_directory(self, media_type):  
        application.media_directories.refresh_directory_list(media_type)
        dirs = application.media_directories.get_directories(media_type)

        for directory in dirs:
            link = Link()
            uri = MediaUri("file://%s" % directory)
            link.controller_path = "/poblesec/%s/filesystem" % media_type

            # translate the type of the section into a "media type"
            # (types defined in elisa.plugins.base.models.file.FileModel)
            translation = {'music': 'audio',
                           'video': 'video',
                           'pictures': 'image'}

            file_media_type = translation[media_type]
            link.controller_args = {'uri': uri, 'media_type': file_media_type}
            if uri.path == HOME_URI.path:
                # Home directory
                link.label = _('Home')
            else:
                link.label = uri.label
            link.icon = "elisa.plugins.poblesec.%s_folders" % media_type
            self.model.append(link)
        
        add_directory = Link()
        add_directory.controller_path = "/poblesec/settings/directory_settings"
        add_directory.label = _("Add folders")
        add_directory.icon = "elisa.plugins.poblesec.directory_settings_add_folder"
        add_directory.controller_args = {'media_type': media_type}
        self.model.append(add_directory)

    def _focus_cb(self, widget, focus):
        if focus:
            self.model[:] = []
            self.list_media_directory(self.media_type)


class DirectorySettingsController(FilesystemPreviewListController):

    hide_files = True
    node_widget = MenuItemWidgetWithAction
    list_widget = ActionListSelector
    media_type = None

    def initialize(self, uri=None, media_type=media_type):
        dfr = super(DirectorySettingsController, self).initialize(uri)
        application.media_directories.refresh_directory_list(media_type)
        self.media_type = media_type

        if uri is None:
            def got_devices(devices):
                volumes = [device for device in devices.devices \
                           if isinstance(device, VolumeModel) \
                           and device.mount_point is not None]
                self.model.extend(volumes)

                application.bus.register(self._device_added_cb, NewDeviceDetected)
                application.bus.register(self._device_removed_cb, DeviceRemoved)

                return self

            def failed_get_devices(failure):
                self.warning('Failed to retrieve the list of devices: %s.' % \
                             failure)
                return self

            def add_home(result):
                home = DirectoryModel()
                home.name = _('Home')
                home.uri = HOME_URI
                self.model.append(home)
                return result

            def get_devices(result):
                devices_uri = MediaUri('volumes://localhost/?filter=file')
                devices, dfr_devices = application.resource_manager.get(devices_uri)
                return dfr_devices

            dfr.addCallback(add_home)
            dfr.addCallback(get_devices)
            dfr.addCallback(got_devices)
            dfr.addErrback(failed_get_devices)

        return dfr

    def _device_added_cb(self, message, sender):
        volume = message.model
        if volume.protocol == 'file' and volume.mount_point is not None:
            self.model.append(volume)

    def _device_removed_cb(self, message, sender):
        for volume in self.model:
            if isinstance(volume, VolumeModel) and volume.udi == message.udi:
                self.model.remove(volume)
                break

    def nodes_setup(self):
        super(DirectorySettingsController,self).nodes_setup()
        #FIXME: why the renderer and the list are defined with class members
        #and not the selector ?
        self.selector = ActionSelector()

        # connect to the signals
        self.nodes.connect('item-clicked', self.node_clicked)
        self.nodes.connect('action-clicked', self.action_clicked)

        self.selector.create_cursor("item")
        self.selector.create_cursor("action")

        theme = self.frontend.get_theme()
        left_cap = theme.get_resource('elisa.plugins.poblesec.selector_left_cap')
        right_cap = theme.get_resource('elisa.plugins.poblesec.selector_right_cap')
        body = theme.get_resource('elisa.plugins.poblesec.selector_body')
        self.selector.set_left_cap(left_cap)
        self.selector.set_right_cap(right_cap)
        self.selector.set_body(body)

        self.nodes.set_selector(self.selector)

    def node_renderer(self, item, widget):
        super(DirectorySettingsController,self).node_renderer(item, widget)

        application.media_directories.refresh_directory_list(self.media_type)

        if isinstance(item, VolumeModel):
            # This is necessary on Windows to convert the escaped backslashes
            # to forward slashes in the mount point.
            uri = MediaUri({'scheme': 'file', 'path': item.mount_point})
            path = uri.path
        else:
            path = item.uri.path

        if path in application.media_directories.all_directories:
            resource = 'elisa.plugins.poblesec.directory_settings_added'
            label = ''
        else:
            resource = 'elisa.plugins.poblesec.directory_settings_add'
            label = _('Add')
        self.frontend.load_from_theme(resource, widget.action_button.icon)
        widget.action_button.label = label

    def node_clicked(self, widget, item):
        if isinstance(item, VolumeModel):
            file_uri = MediaUri({'scheme': 'file', 'path': item.mount_point})
        else:
            file_uri = item.uri

        # we assume there was only one controller created with that path
        controllers = self.frontend.retrieve_controllers('/poblesec/browser')
        browser = controllers[0]
        dfr = browser.history.append_controller(self.path, item.name,
                                                uri=file_uri,
                                                media_type=self.media_type)

    def action_clicked(self, widget, model):
        self.configure_directory(model)

    def configure_directory(self, model, index=-1):
        if index == -1:
            index = self.model.index(model)

        if isinstance(model, VolumeModel):
            file_uri = MediaUri({'scheme': 'file', 'path': model.mount_point})
        else:
            file_uri = model.uri

        if self.media_type:
            application.media_directories.toggle_directory_installation \
                                                    (file_uri, self.media_type)
            self.model[index] = model
        else:
            browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
            path = '/poblesec/settings/directory_selection_settings'
            args = {'uri': file_uri,
                    'refresh_cb': self._refresh_cb,
                    'refresh_item': model}
            dfr = browser.history.append_controller(path, model.name, **args)

    def _refresh_cb(self, item):
        # This trick triggers a re-rendering of the item's widget.
        index = self.model.index(item)
        self.model[index] = item

    def handle_input(self, manager, input_event):
        if  self.nodes.focus:
            if input_event.value == EventValue.KEY_GO_LEFT:
                if self.nodes.action_selected:
                    self.nodes.select_list_item()
                    return True
            elif input_event.value == EventValue.KEY_GO_RIGHT:
                if not self.nodes.action_selected:
                    self.nodes.select_action_item()
                    return True
            elif input_event.value == EventValue.KEY_OK:
                if self.nodes.action_selected:
                    index = self.nodes.selected_item_index
                    model = self.nodes.model[index]
                    self.configure_directory(model, index)
                    return True
    
        return super(DirectorySettingsController,self).handle_input(manager, 
                                                                    input_event)
