/* -*- tab-width: 4 -*-
 *
 * Electric(tm) VLSI Design System
 *
 * File: Tool.java
 *
 * Copyright (c) 2003 Sun Microsystems and Static Free Software
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 */
package com.sun.electric.tool;

import com.sun.electric.database.network.NetworkTool;
import com.sun.electric.database.text.Pref;
import com.sun.electric.database.text.Setting;
import com.sun.electric.database.variable.Variable;
import com.sun.electric.database.variable.ElectricObject;
import com.sun.electric.tool.compaction.Compaction;
import com.sun.electric.tool.drc.DRC;
import com.sun.electric.tool.erc.ERC;
import com.sun.electric.tool.io.IOTool;
import com.sun.electric.tool.logicaleffort.LETool;
import com.sun.electric.tool.project.Project;
import com.sun.electric.tool.routing.Routing;
import com.sun.electric.tool.sc.SilComp;
import com.sun.electric.tool.simulation.Simulation;
import com.sun.electric.tool.user.User;
import com.sun.electric.tool.extract.Extract;
import com.sun.electric.tool.extract.ParasiticTool;
import com.sun.electric.tool.extract.LayerCoverageTool;
import com.sun.electric.tool.generator.layout.GateLayGenSettings;
import com.sun.electric.tool.cvspm.CVS;

import java.lang.reflect.Field;
import java.lang.reflect.Method;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;

/**
 * This class represents a Tool in Electric.  It's here mostly for the name
 * of the tool and the variables attached.  The User holds
 * variables that keep track of the currently selected object, and other
 * useful information.
 */
public class Tool implements Comparable
{
	// The name of this tool
	private String toolName;
	private int toolState;
	private int toolIndex;

	// the static list of all tools
	private static LinkedHashMap<String,Tool> tools = new LinkedHashMap<String,Tool>();
	private static List<Listener> listeners = new ArrayList<Listener>();
	private static int toolNumber = 0;

    /** Settings for this Tool */                           private final HashMap<String,Setting> settingsByXmlPath = new HashMap<String,Setting>();
	/** Preferences for this Tool */                        public Pref.Group prefs;

	/** set if tool is on */								private static final int TOOLON =             01;
	/** set if tool is running in background */				private static final int TOOLBG =             02;
	/** set if tool will fix errors */						private static final int TOOLFIX =            04;
//	/** set if tool is coded in interpretive language */	private static final int TOOLLANG =          010;
	/** set if tool functions incrementally */				private static final int TOOLINCREMENTAL =   020;
	/** set if tool does analysis */						private static final int TOOLANALYSIS =      040;
	/** set if tool does synthesis */						private static final int TOOLSYNTHESIS =    0100;

	/**
	 * The constructor for Tool is only called by subclasses.
	 * @param toolName the name of this tool.
	 */
	protected Tool(String toolName)
	{
		this.toolName = toolName;
		this.toolState = 0;
		this.toolIndex = toolNumber++;
		assert findTool(toolName) == null;
		tools.put(toolName, this);
        prefs = Pref.groupForPackage(this.getClass());  // per-package namespace for preferences
	}

	private void updateListeners()
	{
		listeners.clear();
		for (Tool t : tools.values())
		{
			if (t instanceof Listener && ((Listener)t).isOn())
				listeners.add((Listener)t);
		}
	}

	/**
	 * This is called once, at the start of Electric, to initialize the Tools.
	 * Because of Java's "lazy evaluation", the only way to force the Tool constructors to fire
	 * and build a proper list of Tools, each class must somehow be referenced.
	 * So, each Tool is listed here.  If a new Tool is created, this must be updated.
	 */
	public static void initAllTools()
	{
		User.getUserTool().init();
		Compaction.getCompactionTool().init();
        DRC.getDRCTool().init();
        ERC.getERCTool().init();
        Extract.getExtractTool().init();
        IOTool.getIOTool().init();
        LETool.getLETool().init();
		NetworkTool.getNetworkTool().init();
        ParasiticTool.getParasiticTool().init();
        Project.getProjectTool().init();
        Routing.getRoutingTool().init();
        SilComp.getSilCompTool().init();
        Simulation.getSimulationTool().init();
        LayerCoverageTool.getLayerCoverageTool().init();
        GateLayGenSettings.tool.init();
        CVS.getCVSTool().init();
        try {
            Class<?> staClass = Class.forName("com.sun.electric.plugins.sctiming.RunSTA");
            if (staClass != null) {
                Method staInit = staClass.getMethod("staticinit");
                staInit.invoke(Tool.class);
            }
        } catch (Exception e) {
//            System.out.println(e.getMessage());
        }

        for (Tool tool: tools.values())
            tool.initProjectSettings();
	}

	/**
	 * Method to find the Tool with a specified name.
	 * @param name the name of the desired Tool.
	 * @return the Tool with that name, or null if no tool matches.
	 */
	public static Tool findTool(String name)
	{
		return tools.get(name);
	}

	/**
	 * Method to return an Iterator over all of the Tools in Electric.
	 * @return an Iterator over all of the Tools in Electric.
	 */
	public static Iterator<Tool> getTools()
	{
		return tools.values().iterator();
	}

	/**
	 * Method to return the number of Tools.
	 * @return the number of Tools.
	 */
	public static int getNumTools()
	{
		return tools.size();
	}

	/**
	 * Method to return an Iterator over all of the Listener in Electric
	 * which are on.
	 * @return an Iterator over all of the Listeners in Electric which are on
	 */
	public static Iterator<Listener> getListeners()
	{
		return listeners.iterator();
	}

	/**
	 * Method to return the name of this Tool.
	 * @return the name of this Tool.
	 */
	public String getName() { return toolName; }

	/**
	 * Method to return the index of this Tool.
	 * Each tool has a 0-based index that can be used to access arrays of Tools.
	 * @return the index of this Tool.
	 */
	public int getIndex() { return toolIndex; }

	/**
	 * Method to set this Tool to be on.
	 * Tools that are "on" are running incrementally, and get slices and broadcasts.
	 */
	public void setOn()
	{
		toolState |= TOOLON;
		updateListeners();
	}

	/**
	 * Method to set this Tool to be off.
	 * Tools that are "on" are running incrementally, and get slices and broadcasts.
	 */
	public void clearOn()
	{
		toolState &= ~TOOLON;
		updateListeners();
	}

	/**
	 * Method to tell whether this Tool is on.
	 * Tools that are "on" are running incrementally, and get slices and broadcasts.
	 * @return true if this Tool is on.
	 */
	public boolean isOn() { return (toolState & TOOLON) != 0; }

	/**
	 * Method to set this Tool to be in the background.
	 */
	public void setBackground() { toolState |= TOOLBG; }

	/**
	 * Method to set this Tool to be in the foreground.
	 */
	public void clearBackground() { toolState &= ~TOOLBG; }

	/**
	 * Method to tell whether this Tool is in the background.
	 * @return true if this Tool is in the background.
	 */
	public boolean isBackground() { return (toolState & TOOLBG) != 0; }

	/**
	 * Method to set this Tool to fix errors.
	 */
	public void setFixErrors() { toolState |= TOOLFIX; }

	/**
	 * Method to set this Tool to fix errors.
	 */
	public void clearFixErrors() { toolState &= ~TOOLFIX; }

	/**
	 * Method to tell whether this Tool fixes errors.
	 * @return true if this Tool fixes errors.
	 */
	public boolean isFixErrors() { return (toolState & TOOLFIX) != 0; }

	/**
	 * Method to set this Tool to be incremental.
	 */
	public void setIncremental() { toolState |= TOOLINCREMENTAL; }

	/**
	 * Method to set this Tool to be incremental.
	 */
	public void clearIncremental() { toolState &= ~TOOLINCREMENTAL; }

	/**
	 * Method to tell whether this Tool is incremental.
	 * @return true if this Tool is incremental.
	 */
	public boolean isIncremental() { return (toolState & TOOLINCREMENTAL) != 0; }

	/**
	 * Method to set this Tool to be analysis.
	 */
	public void setAnalysis() { toolState |= TOOLANALYSIS; }

	/**
	 * Method to set this Tool to be analysis.
	 */
	public void clearAnalysis() { toolState &= ~TOOLANALYSIS; }

	/**
	 * Method to tell whether this Tool does analysis.
	 * @return true if this Tool does analysis.
	 */
	public boolean isAnalysis() { return (toolState & TOOLANALYSIS) != 0; }

	/**
	 * Method to set this Tool to be synthesis.
	 */
	public void setSynthesis() { toolState |= TOOLSYNTHESIS; }

	/**
	 * Method to set this Tool to be synthesis.
	 */
	public void clearSynthesis() { toolState &= ~TOOLSYNTHESIS; }

	/**
	 * Method to tell whether this Tool does synthesis.
	 * @return true if this Tool does synthesis.
	 */
	public boolean isSynthesis() { return (toolState & TOOLSYNTHESIS) != 0; }

    public String getProjectSettings() {
        return toolName+"Tool.";
    }

	/**
	 * Method to get a list of project Settings associatiated with this Tool
     * which should be written to disk libraries
	 * @return a collection of project Settings
	 */
    public Collection<Setting> getDiskSettings() {
        ArrayList<Setting> settings = new ArrayList<Setting>();
        for (Setting setting: settingsByXmlPath.values()) {
            if (!setting.isValidOption()) continue;
            if (setting.getValue().equals(setting.getFactoryValue())) continue;
            settings.add(setting);
        }
        Collections.sort(settings, Setting.SETTINGS_BY_PREF_NAME);
        return settings;
    }

    public Setting getSetting(String xmlPath) {
        return settingsByXmlPath.get(getProjectSettings() + xmlPath);
    }

    /**
     * Compares Tools by their definition order.
     * @param obj the other Tool.
     * @return a comparison between the Tools.
     */
	public int compareTo(Object obj)
	{
		Tool that = (Tool)obj;
		return this.toolIndex - that.toolIndex;
	}

	/**
	 * Returns a printable version of this Tool.
	 * @return a printable version of this Tool.
	 */
	public String toString()
	{
		return "Tool '" + toolName + "'";
	}

    /**
     * Subclasses override this method to create ProjectSettings by Tool.makeXXXSetting methods declared below.
     */
    protected void initProjectSettings() {}

	/**
	 * Factory methods to create a boolean project setting objects.
     * The created object is stored into a field of this Tool whose name is "cache"+name.
	 * @param name the name of this Pref.
	 * @param location the user-command that can affect this meaning option.
	 * @param description the description of this meaning option.
	 * @param factory the "factory" default value (if nothing is stored).
	 */
    protected void makeBooleanSetting(String name, String location, String description, boolean factory) {
        setCacheField(Setting.makeBooleanSetting(name, prefs, getProjectSettings(), null, location, description, factory));
    }

	/**
	 * Factory methods to create an integer project setting objects.
     * The created object is stored into a field of this Tool whose name is "cache"+name.
	 * @param name the name of this Pref.
	 * @param location the user-command that can affect this meaning option.
	 * @param description the description of this meaning option.
	 * @param factory the "factory" default value (if nothing is stored).
	 */
    protected void makeIntSetting(String name, String location, String description, int factory) {
        setCacheField(Setting.makeIntSetting(name, prefs, getProjectSettings(), null, location, description, factory));
    }

	/**
	 * Factory methods to create a long project setting objects.
     * The created object is stored into a field of this Tool whose name is "cache"+name.
	 * @param name the name of this Pref.
	 * @param location the user-command that can affect this meaning option.
	 * @param description the description of this meaning option.
	 * @param factory the "factory" default value (if nothing is stored).
	 */
    protected void makeLongSetting(String name, String location, String description, long factory) {
        setCacheField(Setting.makeLongSetting(name, prefs, getProjectSettings(), null, location, description, factory));
    }

	/**
	 * Factory methods to create a double project setting objects.
     * The created object is stored into a field of this Tool whose name is "cache"+name.
	 * @param name the name of this Pref.
	 * @param location the user-command that can affect this meaning option.
	 * @param description the description of this meaning option.
	 * @param factory the "factory" default value (if nothing is stored).
	 */
    protected void makeDoubleSetting(String name, String location, String description, double factory) {
        setCacheField(Setting.makeDoubleSetting(name, prefs, getProjectSettings(), null, location, description, factory));
    }

	/**
	 * Factory methods to create a string project setting objects.
     * The created object is stored into a field of this Tool whose name is "cache"+name.
	 * @param name the name of this Pref.
	 * @param location the user-command that can affect this meaning option.
	 * @param description the description of this meaning option.
	 * @param factory the "factory" default value (if nothing is stored).
	 */
    protected void makeStringSetting(String name, String location, String description, String factory) {
        setCacheField(Setting.makeStringSetting(name, prefs, getProjectSettings(), null, location, description, factory));
    }

    /**
     * Method to store Setting variable in a cache field of this Tool.
     * The name of a field for Setting with name "SettingName" is "cacheSettingName".
     * @param setting Setting variable.
     */
    private void setCacheField(Setting setting) {
        try {
            Field fld = getClass().getDeclaredField("cache" + setting.getPrefName());
            fld.setAccessible(true);
            fld.set(this, setting);
            Setting oldSetting = settingsByXmlPath.put(setting.getXmlPath(), setting);
            assert oldSetting == null;
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

	/**
	 * Method to set a variable on an ElectricObject in a new Job.
	 * @param obj the ElectricObject on which to set the variable.
	 * @param key the Variable key.
	 * @param newVal the new value of the Variable.
	 */
	public void setVarInJob(ElectricObject obj, Variable.Key key, Object newVal)
	{
		new SetVarJob(this, obj, key, newVal);
	}

	/**
	 * Class for scheduling a wiring task.
	 */
	private static class SetVarJob extends Job
	{
		ElectricObject obj;
		Variable.Key key;
		Object newVal;

		protected SetVarJob(Tool tool, ElectricObject obj, Variable.Key key, Object newVal)
		{
			super("Add Variable", tool, Job.Type.CHANGE, null, null, Job.Priority.USER);
			this.obj = obj;
			this.key = key;
			this.newVal = newVal;
			startJob();
		}

		public boolean doIt() throws JobException
		{
			obj.newVar(key, newVal);
			return true;
		}
	}

	/**
	 * The initialization method for this Tool.
	 * Gets overridden by tools that want to do initialization.
	 */
	public void init() {}

    /***********************************
     * Test interface
     ***********************************/
    public static boolean testAll()
    {
        return true; // nothing to test
    }
}
