/* Copyright (C) 2008 to 2010 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the c++-gtk-utils
   sub-directory); if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA, 02111-1307, USA.

However, it is not intended that the object code of a program whose
source code instantiates a template from this file or uses macros or
inline functions (of any length) should by reason only of that
instantiation or use be subject to the restrictions of use in the GNU
Lesser General Public License.  With that in mind, the words "and
macros, inline functions and instantiations of templates (of any
length)" shall be treated as substituted for the words "and small
macros and small inline functions (ten lines or less in length)" in
the fourth paragraph of section 5 of that licence.  This does not
affect any other reason why object code may be subject to the
restrictions in that licence (nor for the avoidance of doubt does it
affect the application of section 2 of that licence to modifications
of the source code in this file).

*/

#ifndef CGU_CALLBACK_H
#define CGU_CALLBACK_H

/**
 * @file callback.h
 * @brief This file provides classes encapsulating callbacks.
 *
 * \#include <c++-gtk-utils/callback.h>
 *
 * These classes encapsulate callbacks (they are closures).  They
 * comprise a generic callback creation and execution interface.
 * There is a basic Callback::Callback type, which is a pure closure
 * where all the arguments are bound in the constructor and is
 * completely opaque.  Callback::CallbackArg<T> is a class which takes
 * one argument of type T when the callback is dispatched, with any
 * other argument being bound at construction time - the type
 * parameter is the type of the unbound argument.  (The pure
 * Callback::Callback type is in fact just a typedef for
 * Callback::CallbackArg<void>: the two types are interchangeable.)
 * The function or method referred to must be one of void return type.
 *
 * They are particularly useful where a callback object may need to be
 * handed between threads.
 *
 * The pure Callback::Callback class can also be useful as the data
 * argument of a call to g_signal_connect_data(), in a case where a
 * better design arises when passing arguments known at connect time
 * by storing them in the callback object itself (say, where otherwise
 * you would pass a plain struct as the data argument), or for general
 * event passing when used together with g_idle_add_full() from glib
 * (see the Callback::post() functions).
 *
 * These classes are also used in the Emitter/EmitterArg classes in
 * emitter.h, which enable callbacks to be connected to an emitter and
 * provide for automatic disconnection where a class object whose
 * member a callback represents ceases to exist.
 *
 * The templated helper Callback::make() functions make it trivial to
 * create a callback object of the correct type.  A maximum of two
 * bound arguments to pass to the relevant function or class method is
 * provided for and a maximum of one free argument to pass at call
 * time - but this can be extended indefinitely.  If there is a free
 * argument, it must be the last argument of the relevant function or
 * method to be called.  Note that bound arguments of the relevant
 * function or method to be called should not normally comprise a
 * reference argument (T& or const T&).  Although the types of the
 * template parameters of the Callback::make() call can be qualified
 * by hand to avoid a type mismatch with a bound reference, it will
 * usually be unsafe to do so.  (See though the documentation on
 * Thread::JoinableHandle for a usage where where binding a reference
 * argument would be safe.)
 *
 * If a functor class is required (say for passing to a c++ algorithm
 * or container, or for automatic lifetime management of the Callback
 * object), the Callback::Functor and Callback::FunctorArg wrapper
 * classes can be used.  However, for many Std C++ algorithms where
 * anonymous functors created as temporaries can be used, the
 * std::ptr_fun() and MemFun::make() factory functions will be a more
 * obvious choice.
 *
 * Callback::SafeFunctor and Callback::SafeFunctorArg classes are the
 * same as Callback::Functor and Callback::FunctorArg classes, except
 * that objects of the safe version may be passed and copied between
 * threads and put in different containers in different threads (that
 * is, the reference count maintained with respect to the contained
 * callback object is thread-safe).  They use a SharedLockPtr object
 * to hold the referenced callback object.
 *
 * If the library is installed using the
 * --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat configuration options, any
 * Callback::Functor, Callback::FunctorArg, Callback::SafeFunctor or
 * Callback::SafeFunctorArg objects constructed on free store (usually
 * they won't be) will be constructed in glib memory slices.  A
 * contained Callback::Callback or Callback::CallbackArg object, which
 * will always be constructed on free store, will be constructed in
 * glib memory slices if the --with-glib-memory-slices-no-compat
 * configuration option is chosen.
 *
 * @b Usage
 *
 * For a class object my_obj of type MyObj, with a method void
 * MyObj::my_method(int, const char*), usage for a fully bound
 * callback or functor would be:
 *
 * @code 
 *   using namespace Cgu;
 *   int arg = 1;
 *   Callback::Callback* cb =
 *     Callback::make(my_obj, &MyObj::my_method, arg, "Hello\n");
 *   cb->dispatch();
 *   delete cb;
 *
 *   Callback::Functor f(Callback::make(my_obj, &MyObj::my_method, arg, "Hello\n"));
 *   f();
 * @endcode
 *
 * Or for a partially bound callback or functor:
 *
 * @code
 *   using namespace Cgu;
 *   int arg = 1;
 *   Callback::CallbackArg<const char*>* cb =
 *     Callback::make(my_obj, &MyObj::my_method, arg);
 *   cb->dispatch("Hello\n");
 *   delete cb;
 *
 *   Callback::FunctorArg<const char*> f(Callback::make(my_obj, &MyObj::my_method, arg));
 *   f("Hello\n");
 * @endcode
 *
 * The syntax for a static member function with a signature void
 * MyObj::my_func(int, const char*), or for a normal function, would
 * be the same except that the call to Callback::make would comprise:
 *
 * @code
 *   Callback::make(&MyObj::my_func, arg, "Hello\n");
 * @endcode
 * (fully bound), or
 * @code
 *   Callback::make(&MyObj::my_func, arg);
 * @endcode
 * (partially bound).
 *
 * Callback/CallbackArg classes do not provide for a return value.  If
 * a result is wanted, users should pass an unbound argument by
 * reference or pointer (or pointer to pointer).
 *
 * Although as mentioned above only one unbound argument is provided
 * for, as that argument can be a struct, any number of arguments can
 * be passed as members of a struct.
 *
 * @b Overloaded @b functions
 *
 * Note that creating callbacks for overloaded functions can give rise
 * to an ambiguity when using Callback::make(), arising from the fact
 * that the callback object may have an unbound argument.  For
 * example:
 *
 * @code
 *   class MyClass {
 *     ...
 *     void add(int i);
 *     void add(int i, int j);
 *     void add(double d);
 *   };
 *   MyClass obj;
 *   using namespace Cgu;
 *   Callback::Callback* cb1 = Callback::make(obj, &MyClass::add, 1, 2); // ok
 *   Callback::Callback* cb2 = Callback::make(obj, &MyClass::add, 1.0);  // ok
 *   Callback::Callback* cb3 = Callback::make(obj, &MyClass::add, 1);    // ambiguous - compilation failure
 * @endcode
 *
 * The third call to Callback::make() is ambiguous, as it could be
 * creating a callback for either the function MyClass::add(int) with
 * no unbound argument (that is, creating a Callback::Callback
 * object), or the function MyClass::add(int, int) with an unbound int
 * argument (that is, creating a Callback::CallbackArg<int> object).
 * This situation could be disambiguated by not using the
 * Callback::make() factory functions and instead creating by hand an
 * explicit Callback object of type Callback::Callback1<MyClass, int>
 * (derived from Callback::Callback) or type
 * Callback::CallbackArg1<MyClass, int, int> (derived from
 * Callback::CallbackArg<int>), but generally it is best to avoid
 * creating callbacks for overloaded functions giving rise to such
 * ambiguities.
 *
 * @b Posting @b of @b callbacks
 *
 * This file also provides Callback::post() functions which will
 * execute a callback in a glib main loop and can be used (amongst
 * other things) to pass an event from a worker thread to the main
 * program thread.  In that respect, it provides an alternative to the
 * Notifier class.  It is passed a pointer to a Callback::CallbackArg
 * object created with a call to Callback::make.
 *
 * To provide for thread-safe automatic disconnection of the callback
 * if the object whose method it represents is destroyed before the
 * callback executes in the main loop, include a Releaser as a public
 * member of that object and pass the Releaser object as the second
 * argument of Callback::post().  Note that for this to be race free,
 * the lifetime of the remote object whose method is to be invoked
 * must be determined by the thread to whose main loop the callback
 * has been attached.  When the main loop begins invoking the
 * execution of the callback, the remote object must either wholly
 * exist (in which case the callback will be invoked) or have been
 * destroyed (in which case the callback will be ignored), and not be
 * in some transient half-state governed by another thread.
 *
 * Advantages as against Notifier:
 * 
 * 1. If there are a lot of different events requiring callbacks to be
 *    dispatched in the program from worker threads to the main
 *    thread, this avoids having separate Notifier objects for each
 *    event.
 *
 * 2. It is easier to pass arguments with varying values - they can be
 *    passed as templated arguments to the Callback::make functions
 *    and no special synchronisation is normally required (the call to
 *    g_source_attach() invokes locking of the main loop which will
 *    have the effect of ensuring memory visibility).  With a Notifier
 *    object it may be necessary to use an asynchronous queue to pass
 *    variable values (or to bind a reference to the data, thus
 *    normally requiring separate synchronisation).
 *
 * 3. Although the callback would normally be sent for execution by
 *    the main program loop, and that is the default, it can be sent
 *    for execution by any thread which has its own
 *    GMainContext/GMainLoop objects.  Thus callbacks can be passed
 *    for execution between worker threads, or from the main program
 *    thread to worker threads, as well as from worker threads to the
 *    main program thread.
 *
 * Disadvantages as against Notifier:
 *
 * 1. Less efficient, as a new callback object has to be created on
 *    freestore every time the callback is invoked, together with a
 *    new Emitter object if a Releaser is used to track the callback.
 * 
 * 2. Multiple callbacks relevant to a single event cannot be invoked
 *    from a single call for the event - each callback has to be
 *    separately dispatched.
 */

/**
 * @namespace Cgu::Callback
 * @brief This namespace provides classes encapsulating callbacks.
 *
 * \#include <c++-gtk-utils/callback.h>
 *
 * These classes encapsulate callbacks (they are closures).  They
 * comprise a generic callback creation and execution interface.
 * There is a basic Callback::Callback type, which is a pure closure
 * where all the arguments are bound in the constructor and is
 * completely opaque.  Callback::CallbackArg<T> is a class which takes
 * one argument of type T when the callback is dispatched, with any
 * other argument being bound at construction time - the type
 * parameter is the type of the unbound argument.  (The pure
 * Callback::Callback type is in fact just a typedef for
 * Callback::CallbackArg<void>: the two types are interchangeable.)
 * The function or method referred to must be one of void return type.
 *
 * They are particularly useful where a callback object may need to be
 * handed between threads.
 *
 * The pure Callback::Callback class can also be useful as the data
 * argument of a call to g_signal_connect_data(), in a case where a
 * better design arises when passing arguments known at connect time
 * by storing them in the callback object itself (say, where otherwise
 * you would pass a plain struct as the data argument), or for general
 * event passing when used together with g_idle_add_full() from glib
 * (see the Callback::post() functions).
 *
 * These classes are also used in the Emitter/EmitterArg classes in
 * emitter.h, which enable callbacks to be connected to an emitter and
 * provide for automatic disconnection where a class object whose
 * member a callback represents ceases to exist.
 *
 * The templated helper Callback::make() functions make it trivial to
 * create a callback object of the correct type.  A maximum of two
 * bound arguments to pass to the relevant function or class method is
 * provided for and a maximum of one free argument to pass at call
 * time - but this can be extended indefinitely.  If there is a free
 * argument, it must be the last argument of the relevant function or
 * method to be called.  Note that bound arguments of the relevant
 * function or method to be called should not normally comprise a
 * reference argument (T& or const T&).  Although the types of the
 * template parameters of the Callback::make() call can be qualified
 * by hand to avoid a type mismatch with a reference, it will usually
 * be unsafe to do so.  (See though the documentation on
 * Thread::JoinableHandle for a usage where where binding a reference
 * argument would be safe.)
 *
 * If a functor class is required (say for passing to a c++ algorithm
 * or container, or for automatic lifetime management of the Callback
 * object), the Callback::Functor and Callback::FunctorArg wrapper
 * classes can be used.  Callback::SafeFunctor
 * Callback::SafeFunctorArg classes are the same, but may be passed
 * between threads.  However, for many Std C++ algorithms where
 * anonymous functors created as temporaries can be used, the
 * std::ptr_fun() and MemFun::make() factory functions will be a more
 * obvious choice.
 *
 * If the library is installed using the
 * --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat configuration options, any
 * Callback::Functor, Callback::FunctorArg, Callback::SafeFunctor or
 * Callback::SafeFunctorArg objects constructed on free store (usually
 * they won't be) will be constructed in glib memory slices.  A
 * contained Callback::Callback or Callback::CallbackArg object, which
 * will always be constructed on free store, will be constructed in
 * glib memory slices if the --with-glib-memory-slices-no-compat
 * configuration option is chosen.
 *
 * @b Usage
 *
 * For a class object my_obj of type MyObj, with a method void
 * MyObj::my_method(int, const char*), usage for a fully bound
 * callback or functor would be:
 *
 * @code 
 *   using namespace Cgu;
 *   int arg = 1;
 *   Callback::Callback* cb =
 *     Callback::make(my_obj, &MyObj::my_method, arg, "Hello\n");
 *   cb->dispatch();
 *   delete cb;
 *
 *   Callback::Functor f(Callback::make(my_obj, &MyObj::my_method, arg, "Hello\n"));
 *   f();
 * @endcode
 *
 * Or for a partially bound callback or functor:
 *
 * @code
 *   using namespace Cgu;
 *   int arg = 1;
 *   Callback::CallbackArg<const char*>* cb =
 *     Callback::make(my_obj, &MyObj::my_method, arg);
 *   cb->dispatch("Hello\n");
 *   delete cb;
 *
 *   Callback::FunctorArg<const char*> f(Callback::make(my_obj, &MyObj::my_method, arg));
 *   f("Hello\n");
 * @endcode
 *
 * The syntax for a static member function with a signature void
 * MyObj::my_func(int, const char*), or for a normal function, would
 * be the same except that the call to Callback::make would comprise:
 *
 * @code
 *   Callback::make(&MyObj::my_func, arg, "Hello\n");
 * @endcode
 * (fully bound), or
 * @code
 *   Callback::make(&MyObj::my_func, arg);
 * @endcode
 * (partially bound).
 *
 * Callback/CallbackArg classes do not provide for a return value.  If
 * a result is wanted, users should pass an unbound argument by
 * reference or pointer (or pointer to pointer).
 *
 * Although as mentioned above only one unbound argument is provided
 * for, as that argument can be a struct, any number of arguments can
 * be passed as members of a struct.
 *
 * @b Overloaded @b functions
 *
 * Note that creating callbacks for overloaded functions can give rise
 * to an ambiguity when using Callback::make(), arising from the fact
 * that the callback object may have an unbound argument.  For
 * example:
 *
 * @code
 *   class MyClass {
 *     ...
 *     void add(int i);
 *     void add(int i, int j);
 *     void add(double d);
 *   };
 *   MyClass obj;
 *   using namespace Cgu;
 *   Callback::Callback* cb1 = Callback::make(obj, &MyClass::add, 1, 2); // ok
 *   Callback::Callback* cb2 = Callback::make(obj, &MyClass::add, 1.0);  // ok
 *   Callback::Callback* cb3 = Callback::make(obj, &MyClass::add, 1);    // ambiguous - compilation failure
 * @endcode
 *
 * The third call to Callback::make() is ambiguous, as it could be
 * creating a callback for either the function MyClass::add(int) with
 * no unbound argument (that is, creating a Callback::Callback
 * object), or the function MyClass::add(int, int) with an unbound int
 * argument (that is, creating a Callback::CallbackArg<int> object).
 * This situation could be disambiguated by not using the
 * Callback::make() factory functions and instead creating by hand an
 * explicit Callback object of type Callback::Callback1<MyClass, int>
 * (derived from Callback::Callback) or type
 * Callback::CallbackArg1<MyClass, int, int> (derived from
 * Callback::CallbackArg<int>), but generally it is best to avoid
 * creating callbacks for overloaded functions giving rise to such
 * ambiguities.
 *
 * @b Posting @b of @b callbacks
 *
 * This namespace also provides Callback::post() functions which will
 * execute a callback in a glib main loop and can be used (amongst
 * other things) to pass an event from a worker thread to the main
 * program thread.  In that respect, it provides an alternative to the
 * Notifier class.  It is passed a pointer to a Callback::CallbackArg
 * object created with a call to Callback::make.
 *
 * To provide for thread-safe automatic disconnection of the callback
 * if the object whose method it represents is destroyed before the
 * callback executes in the main loop, include a Releaser as a public
 * member of that object and pass the Releaser object as the second
 * argument of Callback::post().  Note that for this to be race free,
 * the lifetime of the remote object whose method is to be invoked
 * must be determined by the thread to whose main loop the callback
 * has been attached.  When the main loop begins invoking the
 * execution of the callback, the remote object must either wholly
 * exist (in which case the callback will be invoked) or have been
 * destroyed (in which case the callback will be ignored), and not be
 * in some transient half-state governed by another thread.
 *
 * Advantages as against Notifier:
 * 
 * 1. If there are a lot of different events requiring callbacks to be
 *    dispatched in the program from worker threads to the main
 *    thread, this avoids having separate Notifier objects for each
 *    event.
 *
 * 2. It is easier to pass arguments with varying values - they can be
 *    passed as templated arguments to the Callback::make functions
 *    and no special synchronisation is normally required (the call to
 *    g_source_attach() invokes locking of the main loop which will
 *    have the effect of ensuring memory visibility).  With a Notifier
 *    object it may be necessary to use an asynchronous queue to pass
 *    variable values (or to bind a reference to the data, thus
 *    normally requiring separate synchronisation).
 *
 * 3. Although the callback would normally be sent for execution by
 *    the main program loop, and that is the default, it can be sent
 *    for execution by any thread which has its own
 *    GMainContext/GMainLoop objects.  Thus callbacks can be passed
 *    for execution between worker threads, or from the main program
 *    thread to worker threads, as well as from worker threads to the
 *    main program thread.
 *
 * Disadvantages as against Notifier:
 *
 * 1. Less efficient, as a new callback object has to be created on
 *    freestore every time the callback is invoked, together with a
 *    new Emitter object if a Releaser is used to track the callback.
 * 
 * 2. Multiple callbacks relevant to a single event cannot be invoked
 *    from a single call for the event - each callback has to be
 *    separately dispatched.
 */

#include <functional> // for std::less

#include <glib.h>

#include <c++-gtk-utils/shared_ptr.h>
#include <c++-gtk-utils/cgu_config.h>

namespace Cgu {

namespace Callback {

/*
   The CallbackArg class could be additionally templated to provide a
   return value, but that would affect the simplicity of the
   interface, and if a case were to arise where a result is needed, an
   alternative is for users to pass an argument by pointer (or pointer
   to pointer) rather than have a return value.
*/

/*
  Passing arguments: It will be seen that the Callback::make() helper
  functions and the constructors of the derived Callback classes all
  pass the bound arguments of the relevant referenced function (for
  storage in the derived Callback class) by value rather than by const
  reference.

  This is deliberate.  We are not starting with a clean sheet but can
  assume that the designer of the referenced function took an educated
  view about how any argument should be passed to it (and where by
  reference/pointer, she would have had to construct the object on
  freestore for storage in the Callback object in any event).
  Furthermore, the helper function and constructors are all templated
  inline functions.  Where the argument is a built-in type such as a
  integer type or a pointer, or a user type without any specialised
  copy constructor or with a move constructor, the compiler can carry
  out copy elision right from the call site of the Callback::make()
  helper function through to the point at which the argument is stored
  in the Callback object, wherever reference arguments do not get in
  the way.
*/

/* Declare the two basic interface types */

template <class FreeArg> class CallbackArg;
typedef CallbackArg<void> Callback;

/* now the class definitions */

/**
 * @class CallbackArg callback.h c++-gtk-utils/callback.h
 * @brief The callback interface class
 * @sa Callback namespace
 * @sa FunctorArg SafeFunctorArg
 *
 * This provides the basic interface class that users will generally
 * see, and is suitable for passing by void*, which if using glib
 * and/or gthreads and/or pthreads is almost certain to happen at some
 * stage.  The template type is the type of the unbound argument,
 * if any.  Callback::CallbackArg<void> is typedef'ed to
 * Callback::Callback.
 *
 * @b Usage
 *
 * For a class object my_obj of type MyObj, with a method void
 * MyObj::my_method(int, const char*), usage for a fully bound
 * callback would be:
 *
 * @code 
 *   using namespace Cgu;
 *   int arg = 1;
 *   Callback::Callback* cb =
 *     Callback::make(my_obj, &MyObj::my_method, arg, "Hello\n");
 *   cb->dispatch();
 *   delete cb;
 * @endcode
 *
 * Or for a partially bound callback:
 *
 * @code
 *   using namespace Cgu;
 *   int arg = 1;
 *   Callback::CallbackArg<const char*>* cb =
 *     Callback::make(my_obj, &MyObj::my_method, arg);
 *   cb->dispatch("Hello\n");
 *   delete cb;
 * @endcode
 *
 * Callback/CallbackArg classes do not provide for a return value.  If
 * a result is wanted, users should pass an unbound argument by
 * reference or pointer (or pointer to pointer).
 *
 * For further background, read this: Callback
 */

template <class FreeArg>
class CallbackArg {
public:
/**
 * This will execute the referenced function or class method
 * encapsulated by this class.  It will only throw if the dispatched
 * function or class method throws, or if the assignment operator of
 * the free or a bound argument throws and it is not a reference
 * argument.  It is thread safe if the referenced function or class
 * method is thread safe.
 * @param arg The argument to be passed to the referenced function or
 * class method, if any.  (This class is specialised for a void
 * argument, and in that case no argument is passed.)
 * @note We use dispatch() to execute the callback, because the
 * callback would normally be invoked through a base class pointer.
 * To invoke it through operator()(), use the FunctorArg wrapper
 * class.
 */
  virtual void dispatch(FreeArg arg) const = 0;

/**
 *  The constructor will not throw unless the copy constructor or
 *  assignment operator of an argument bound to the derived
 *  implementation class throws.
 */
  CallbackArg() {};

/**
 *  The destructor will not throw unless the destructor of an argument
 *  bound to the derived implementation class throws.
 */
  virtual ~CallbackArg() {};

/* these functions will be inherited by the derived callback classes */
#ifdef CGU_USE_GLIB_MEMORY_SLICES_NO_COMPAT
  CGU_GLIB_MEMORY_SLICES_FUNCS
#endif
};

#ifndef DOXYGEN_PARSING
template <>
class CallbackArg<void> {
public:
  virtual void dispatch() const = 0;
  CallbackArg() {};
  virtual ~CallbackArg() {};

/* these functions will be inherited by the derived callback classes */
#ifdef CGU_USE_GLIB_MEMORY_SLICES_NO_COMPAT
  CGU_GLIB_MEMORY_SLICES_FUNCS
#endif
};
#endif

/* The four basic functor types */

template <class FreeArg> class FunctorArg;
template <class FreeArg> class SafeFunctorArg;
typedef FunctorArg<void> Functor;
typedef SafeFunctorArg<void> SafeFunctor;

/* Functor friend functions */

// we can use built-in operator == when comparing pointers referencing
// different objects of the same type
/**
 * This comparison operator does not throw.
 */
template <class T>
bool operator==(const FunctorArg<T>& f1, const FunctorArg<T>& f2) {
  return (f1.cb_s.get() == f2.cb_s.get());
}

/**
 * This comparison operator does not throw.
 *
 * Since 1.2.5
 */
template <class T>
bool operator!=(const FunctorArg<T>& f1, const FunctorArg<T>& f2) {
  return !(f1 == f2);
}

// we must use std::less rather than the < built-in operator for
// pointers to objects not within the same array or object: "For
// templates greater, less, greater_equal, and less_equal, the
// specializations for any pointer type yield a total order, even if
// the built-in operators <, >, <=, >= do not." (para 20.3.3/8).
/**
 * This comparison operator does not throw.
 */
template <class T>
bool operator<(const FunctorArg<T>& f1, const FunctorArg<T>& f2) {
  return std::less<const CallbackArg<T>*>()(f1.cb_s.get(), f2.cb_s.get());
}

/**
 * This comparison operator does not throw.
 */
template <class T>
bool operator==(const SafeFunctorArg<T>& f1, const SafeFunctorArg<T>& f2) {
  return (f1.cb_s.get() == f2.cb_s.get());
}

/**
 * This comparison operator does not throw.
 *
 * Since 1.2.5
 */
template <class T>
bool operator!=(const SafeFunctorArg<T>& f1, const SafeFunctorArg<T>& f2) {
  return !(f1 == f2);
}

/**
 * This comparison operator does not throw.
 */
template <class T>
bool operator<(const SafeFunctorArg<T>& f1, const SafeFunctorArg<T>& f2) {
  return std::less<const CallbackArg<T>*>()(f1.cb_s.get(), f2.cb_s.get());
}

/* the functor classes */

/**
 * @class FunctorArg callback.h c++-gtk-utils/callback.h
 * @brief Functor class holding a Callback::CallbackArg object.
 * @sa SafeFunctorArg
 * @sa Callback namespace
 *
 * This class wraps a CallbackArg object.  The callback object is kept
 * by SharedPtr so the functor can be copied and offers automatic
 * lifetime management of the wrapped callback object, as well as
 * providing an operator()() function.  Ownership is taken of the
 * CallbackArg object passed to the constructor taking a CallbackArg
 * pointer, so that constructor should be treated like a shared
 * pointer constructor - only pass a newly allocated object to it (or
 * copy construct it or assign to it from another existing FunctorArg
 * object.).  The template type is the type of the unbound argument,
 * if any.  Callback::FunctorArg<void> is typedef'ed to
 * Callback::Functor.
 *
 * The constructor taking a Callback::CallbackArg pointer is not
 * marked explicit, so the results of Callback::make() can be passed
 * directly to a function taking a Callback::FunctorArg argument, and
 * implicit conversion will take place.
 *
 * @b Usage
 *
 * For a class object my_obj of type MyObj, with a method void
 * MyObj::my_method(int, const char*), usage for a fully bound functor
 * would be:
 *
 * @code 
 *   using namespace Cgu;
 *   int arg = 1;
 *   Callback::Functor f(Callback::make(my_obj, &MyObj::my_method, arg, "Hello\n"));
 *   f();
 * @endcode
 *
 * Or for a partially bound functor:
 *
 * @code
 *   using namespace Cgu;
 *   int arg = 1;
 *   Callback::FunctorArg<const char*> f(Callback::make(my_obj, &MyObj::my_method, arg));
 *   f("Hello\n");
 * @endcode
 *
 * Callback/CallbackArg classes do not provide for a return value.  If
 * a result is wanted, users should pass an unbound argument by
 * reference or pointer (or pointer to pointer).
 *
 * For further background, read this: Callback
 */

template <class FreeArg>
class FunctorArg {
  SharedPtr<const CallbackArg<FreeArg> > cb_s;
public:
/**
 * This will execute the referenced function or class method
 * encapsulated by this class.  It will only throw if the executed
 * function or class method throws, or if the assignment operator of
 * the free or a bound argument throws and it is not a reference
 * argument.  It is thread safe if the referenced function or class
 * method is thread safe.
 * @param arg The argument to be passed to the referenced function or
 * class method, if any.  (This class is specialised for a void
 * argument, and in that case no argument is passed.)
 */
  void operator()(FreeArg arg) const {if (cb_s.get()) cb_s->dispatch(arg);}

/**
 * This comparison operator does not throw (this friend function is
 * also called by operator!=<>() from version 1.2.5).
 */
  friend bool operator== <>(const FunctorArg&, const FunctorArg&);
/**
 * This comparison operator does not throw.
 */
  friend bool operator< <>(const FunctorArg&, const FunctorArg&);

 /**
  * Constructor of first FunctorArg holding the referenced callback.
  * As it is not marked explicit, it is also a type conversion
  * constructor.
  * @param cb The CallbackArg object which the functor is to manage.
  * @exception std::bad_alloc This might throw std::bad_alloc if
  * memory is exhausted and the system throws in that case.  Note that
  * if such an exception is thrown, then this constructor will clean
  * itself up and also delete the callback object passed to it.
  * @note std::bad_alloc will not be thrown if the library has been
  * installed using the --with-glib-memory-slices-no-compat
  * configuration option: instead glib will terminate the program if
  * it is unable to obtain memory from the operating system.
 * 
  */
  FunctorArg(const CallbackArg<FreeArg>* cb): cb_s(cb) {}

 /** 
  * The copy constructor does not throw.
  * @param f The assignee
  */ 
  FunctorArg(const FunctorArg& f): cb_s(f.cb_s) {}

 /**
  * Default constructor, where a Callback::CallbackArg object is to be
  * assigned later (via the type conversion constructor and the
  * implicit assignment operator).  This constructor does not throw.
  */
  FunctorArg() {};

/* Only has effect if --with-glib-memory-slices-compat or
   --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};

#ifndef DOXYGEN_PARSING
template <>
class FunctorArg<void> {
  SharedPtr<const Callback> cb_s;
public:
  void operator()() const {if (cb_s.get()) cb_s->dispatch();}

  friend bool operator== <>(const FunctorArg&, const FunctorArg&);
  friend bool operator< <>(const FunctorArg&, const FunctorArg&);

  FunctorArg(const Callback* cb_p): cb_s(cb_p) {}
  FunctorArg(const FunctorArg& f): cb_s(f.cb_s) {}
  FunctorArg() {};

/* Only has effect if --with-glib-memory-slices-compat or
   --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};
#endif

/**
 * @class SafeFunctorArg callback.h c++-gtk-utils/callback.h
 * @brief Functor class holding a Callback::CallbackArg object, with
 * thread-safe reference count.
 * @sa FunctorArg
 * @sa Callback namespace
 *
 * This class is the same as Callback::FunctorArg except that it will
 * provide synchronisation of the reference count between threads.
 * Use it where a functor wrapper object is to be passed between
 * threads.  The FunctorArg documentation gives details on usage.
 *
 * Callback::SafeFunctorArg<void> is typedef'ed to
 * Callback::SafeFunctor.
 *
 * For further background, read this: Callback
 */

template <class FreeArg>
class SafeFunctorArg {
  SharedLockPtr<const CallbackArg<FreeArg> > cb_s;
public:
/**
 * This will execute the referenced function or class method
 * encapsulated by this class.  It will only throw if the executed
 * function or class method throws, or if the assignment operator of
 * the free or a bound argument throws and it is not a reference
 * argument.  It is thread safe if the referenced function or class
 * method is thread safe.
 * @param arg The argument to be passed to the referenced function or
 * class method, if any.  (This class is specialised for a void
 * argument, and in that case no argument is passed.)
 */
  void operator()(FreeArg arg) const {if (cb_s.get()) cb_s->dispatch(arg);}

/**
 * This comparison operator does not throw (this friend function is
 * also called by operator!=<>() from version 1.2.5).
 */
  friend bool operator== <>(const SafeFunctorArg&, const SafeFunctorArg&);
/**
 * This comparison operator does not throw.
 */
  friend bool operator< <>(const SafeFunctorArg&, const SafeFunctorArg&);

 /**
  * Constructor of first SafeFunctorArg holding the referenced
  * callback.  As it is not marked explicit, it is also a type
  * conversion constructor.
  * @param cb The CallbackArg object which the functor is to manage.
  * @exception std::bad_alloc This might throw std::bad_alloc if
  * memory is exhausted and the system throws in that case.  Note that
  * if such an exception is thrown, then this constructor will clean
  * itself up and also delete the callback object passed to it.
  * @note std::bad_alloc will not be thrown if the library has been
  * installed using the --with-glib-memory-slices-no-compat
  * configuration option: instead glib will terminate the program if
  * it is unable to obtain memory from the operating system.
  */
  SafeFunctorArg(const CallbackArg<FreeArg>* cb): cb_s(cb) {}

 /** 
  * The copy constructor does not throw.
  * @param f The assignee
  */ 
  SafeFunctorArg(const SafeFunctorArg& f): cb_s(f.cb_s) {}

 /**
  * Default constructor, where a Callback::CallbackArg object is to be
  * assigned later (via the type conversion constructor and the
  * implicit assignment operator).  This constructor does not throw.
  * @note The reference count maintained with respect to the contained
  * callback object is thread-safe, so SafeFunctorArg objects may be
  * copied between threads by the implicit assignment operator and put
  * in different containers in different threads.  They use a
  * SharedLockPtr object to hold the referenced callback object.
  */
  SafeFunctorArg() {};

/* Only has effect if --with-glib-memory-slices-compat or
   --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};

#ifndef DOXYGEN_PARSING
template <>
class SafeFunctorArg<void> {
  SharedLockPtr<const Callback> cb_s;
public:
  void operator()() const {if (cb_s.get()) cb_s->dispatch();}

  friend bool operator== <>(const SafeFunctorArg&, const SafeFunctorArg&);
  friend bool operator< <>(const SafeFunctorArg&, const SafeFunctorArg&);

  SafeFunctorArg(const Callback* cb_p): cb_s(cb_p) {}
  SafeFunctorArg(const SafeFunctorArg& f): cb_s(f.cb_s) {}
  SafeFunctorArg() {};

/* Only has effect if --with-glib-memory-slices-compat or
   --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};
#endif

/* the callback implementation classes */

template <class T>
class Callback0: public Callback {
public:
  typedef void (T::* MemFunc)();
private:
  T* obj;
  MemFunc func;
public:
  void dispatch() const {(obj->*func)();}
  Callback0(T& obj_, MemFunc func_): obj(&obj_), func(func_) {}
};

template <class T, class FreeArg>
class CallbackArg0: public CallbackArg<FreeArg> {
public:
  typedef void (T::* MemFunc)(FreeArg);
private:
  T* obj;
  MemFunc func;
public:
  void dispatch(FreeArg free_arg) const {(obj->*func)(free_arg);}
  CallbackArg0(T& obj_, MemFunc func_): obj(&obj_), func(func_) {}
};

template <class T, class BoundArg1>
class Callback1: public Callback {
public:
  typedef void (T::* MemFunc)(BoundArg1);
private:
  T* obj;
  MemFunc func;
  BoundArg1 arg1;
public:
  void dispatch() const {(obj->*func)(arg1);}
  Callback1(T& obj_, MemFunc func_, BoundArg1 arg1_): obj(&obj_), func(func_), arg1(arg1_) {}
};

template <class T, class BoundArg1, class FreeArg>
class CallbackArg1: public CallbackArg<FreeArg> {
public:
  typedef void (T::* MemFunc)(BoundArg1, FreeArg);
private:
  T* obj;
  MemFunc func;
  BoundArg1 arg1;
public:
  void dispatch(FreeArg free_arg) const {(obj->*func)(arg1, free_arg);}
  CallbackArg1(T& obj_, MemFunc func_, BoundArg1 arg1_): obj(&obj_), func(func_), arg1(arg1_) {}
};

template <class T, class BoundArg1, class BoundArg2>
class Callback2: public Callback {
public:
  typedef void (T::* MemFunc)(BoundArg1, BoundArg2);
private:
  T* obj;
  MemFunc func;
  BoundArg1 arg1;
  BoundArg2 arg2;
public:
  void dispatch() const {(obj->*func)(arg1, arg2);}
  Callback2(T& obj_, MemFunc func_, BoundArg1 arg1_, BoundArg2 arg2_): obj(&obj_), func(func_),
                                                                       arg1(arg1_), arg2(arg2_) {}
};

template <class T, class BoundArg1, class BoundArg2, class FreeArg>
class CallbackArg2: public CallbackArg<FreeArg> {
public:
  typedef void (T::* MemFunc)(BoundArg1, BoundArg2, FreeArg);
private:
  T* obj;
  MemFunc func;
  BoundArg1 arg1;
  BoundArg2 arg2;
public:
  void dispatch(FreeArg free_arg) const {(obj->*func)(arg1, arg2, free_arg);}
  CallbackArg2(T& obj_, MemFunc func_, BoundArg1 arg1_, BoundArg2 arg2_): obj(&obj_), func(func_),
                                                                          arg1(arg1_), arg2(arg2_) {}
};

/* const versions, for binding to const methods */

template <class T>
class Callback0_const: public Callback {
public:
  typedef void (T::* MemFunc)() const;
private:
  const T* obj;
  MemFunc func;
public:
  void dispatch() const {(obj->*func)();}
  Callback0_const(const T& obj_, MemFunc func_): obj(&obj_), func(func_) {}
};

template <class T, class FreeArg>
class CallbackArg0_const: public CallbackArg<FreeArg> {
public:
  typedef void (T::* MemFunc)(FreeArg) const;
private:
  const T* obj;
  MemFunc func;
public:
  void dispatch(FreeArg free_arg) const {(obj->*func)(free_arg);}
  CallbackArg0_const(const T& obj_, MemFunc func_): obj(&obj_), func(func_) {}
};

template <class T, class BoundArg1>
class Callback1_const: public Callback {
public:
  typedef void (T::* MemFunc)(BoundArg1) const;
private:
  const T* obj;
  MemFunc func;
  BoundArg1 arg1;
public:
  void dispatch() const {(obj->*func)(arg1);}
  Callback1_const(const T& obj_, MemFunc func_, BoundArg1 arg1_): obj(&obj_), func(func_), arg1(arg1_) {}
};

template <class T, class BoundArg1, class FreeArg>
class CallbackArg1_const: public CallbackArg<FreeArg> {
public:
  typedef void (T::* MemFunc)(BoundArg1, FreeArg) const;
private:
  const T* obj;
  MemFunc func;
  BoundArg1 arg1;
public:
  void dispatch(FreeArg free_arg) const {(obj->*func)(arg1, free_arg);}
  CallbackArg1_const(const T& obj_, MemFunc func_, BoundArg1 arg1_): obj(&obj_), func(func_), arg1(arg1_) {}
};

template <class T, class BoundArg1, class BoundArg2>
class Callback2_const: public Callback {
public:
  typedef void (T::* MemFunc)(BoundArg1, BoundArg2) const;
private:
  const T* obj;
  MemFunc func;
  BoundArg1 arg1;
  BoundArg2 arg2;
public:
  void dispatch() const {(obj->*func)(arg1, arg2);}
  Callback2_const(const T& obj_, MemFunc func_, BoundArg1 arg1_, BoundArg2 arg2_): obj(&obj_), func(func_),
                                                                                   arg1(arg1_), arg2(arg2_) {}
};

template <class T, class BoundArg1, class BoundArg2, class FreeArg>
class CallbackArg2_const: public CallbackArg<FreeArg> {
public:
  typedef void (T::* MemFunc)(BoundArg1, BoundArg2, FreeArg) const;
private:
  const T* obj;
  MemFunc func;
  BoundArg1 arg1;
  BoundArg2 arg2;
public:
  void dispatch(FreeArg free_arg) const {(obj->*func)(arg1, arg2, free_arg);}
  CallbackArg2_const(const T& obj_, MemFunc func_, BoundArg1 arg1_, BoundArg2 arg2_): obj(&obj_), func(func_),
                                                                                      arg1(arg1_), arg2(arg2_) {}
};

/* for static class methods and non-class functions */

class Callback0_static: public Callback {
public:
  typedef void (*Func)();
private:
  Func func;
public:
  void dispatch() const {func();}
  Callback0_static(Func func_): func(func_) {}
};

template <class FreeArg>
class CallbackArg0_static: public CallbackArg<FreeArg> {
public:
  typedef void (*Func)(FreeArg);
private:
  Func func;
public:
  void dispatch(FreeArg free_arg) const {func(free_arg);}
  CallbackArg0_static(Func func_): func(func_) {}
};

template <class BoundArg1>
class Callback1_static: public Callback {
public:
  typedef void (*Func)(BoundArg1);
private:
  Func func;
  BoundArg1 arg1;
public:
  void dispatch() const {func(arg1);}
  Callback1_static(Func func_, BoundArg1 arg1_): func(func_), arg1(arg1_) {}
};

 template <class BoundArg1, class FreeArg>
class CallbackArg1_static: public CallbackArg<FreeArg> {
public:
  typedef void (*Func)(BoundArg1, FreeArg);
private:
  Func func;
  BoundArg1 arg1;
public:
  void dispatch(FreeArg free_arg) const {func(arg1, free_arg);}
  CallbackArg1_static(Func func_, BoundArg1 arg1_): func(func_), arg1(arg1_) {}
};

template <class BoundArg1, class BoundArg2>
class Callback2_static: public Callback {
public:
  typedef void (*Func)(BoundArg1, BoundArg2);
private:
  Func func;
  BoundArg1 arg1;
  BoundArg2 arg2;
public:
  void dispatch() const {func(arg1, arg2);}
  Callback2_static(Func func_, BoundArg1 arg1_, BoundArg2 arg2_): func(func_),
			                                          arg1(arg1_), arg2(arg2_) {}
};

 template <class BoundArg1, class BoundArg2, class FreeArg>
class CallbackArg2_static: public CallbackArg<FreeArg> {
public:
  typedef void (*Func)(BoundArg1, BoundArg2, FreeArg);
private:
  Func func;
  BoundArg1 arg1;
  BoundArg2 arg2;
public:
  void dispatch(FreeArg free_arg) const {func(arg1, arg2, free_arg);}
  CallbackArg2_static(Func func_, BoundArg1 arg1_, BoundArg2 arg2_): func(func_),
			                                             arg1(arg1_), arg2(arg2_) {}
};


/* Convenience functions making callback objects on freestore.  These
 * can for example be passed as the first argument of the
 * Thread::start() method in thread.h. They are also used by the
 * Callback::post() function.
*/

/**
 * A convenience function to make Callback::CallbackArg objects
 * @exception std::bad_alloc It might throw std::bad_alloc if memory
 * is exhausted and the system throws in that case.  This exception
 * will not be thrown if the library has been installed using the
 * --with-glib-memory-slices-no-compat configuration option (instead
 * glib will terminate the program if it is unable to obtain memory
 * from the operating system).
 */
template <class T>
Callback* make(T& t,
	       void (T::*func)()) {
  return new Callback0<T>(t, func);
}

/**
 * A convenience function to make Callback::CallbackArg objects
 * @exception std::bad_alloc It might throw std::bad_alloc if memory
 * is exhausted and the system throws in that case.  This exception
 * will not be thrown if the library has been installed using the
 * --with-glib-memory-slices-no-compat configuration option (instead
 * glib will terminate the program if it is unable to obtain memory
 * from the operating system).
 */
template <class T, class FreeArg>
CallbackArg<FreeArg>* make(T& t,
			   void (T::*func)(FreeArg)) {
  return new CallbackArg0<T, FreeArg>(t, func);
}

/**
 * A convenience function to make Callback::CallbackArg objects
 * @exception std::bad_alloc It might throw std::bad_alloc if memory
 * is exhausted and the system throws in that case (this exception
 * will not be thrown if the library has been installed using the
 * --with-glib-memory-slices-no-compat configuration option: instead
 * glib will terminate the program if it is unable to obtain memory
 * from the operating system).  It will also throw if the assignment
 * operator of a bound argument throws.
 */
template <class T, class BoundArg1>
Callback* make(T& t,
	       void (T::*func)(BoundArg1),
	       BoundArg1 arg1) {
  return new Callback1<T, BoundArg1>(t, func, arg1);
}

/**
 * A convenience function to make Callback::CallbackArg objects
 * @exception std::bad_alloc It might throw std::bad_alloc if memory
 * is exhausted and the system throws in that case (this exception
 * will not be thrown if the library has been installed using the
 * --with-glib-memory-slices-no-compat configuration option: instead
 * glib will terminate the program if it is unable to obtain memory
 * from the operating system).  It will also throw if the assignment
 * operator of a bound argument throws.
 */
template <class T, class BoundArg1, class FreeArg>
CallbackArg<FreeArg>* make(T& t,
			   void (T::*func)(BoundArg1, FreeArg),
			   BoundArg1 arg1) {
  return new CallbackArg1<T, BoundArg1, FreeArg>(t, func, arg1);
}

/**
 * A convenience function to make Callback::CallbackArg objects
 * @exception std::bad_alloc It might throw std::bad_alloc if memory
 * is exhausted and the system throws in that case (this exception
 * will not be thrown if the library has been installed using the
 * --with-glib-memory-slices-no-compat configuration option: instead
 * glib will terminate the program if it is unable to obtain memory
 * from the operating system).  It will also throw if the assignment
 * operator of a bound argument throws.
 */
template <class T, class BoundArg1, class BoundArg2>
Callback* make(T& t,
	       void (T::*func)(BoundArg1, BoundArg2),
	       BoundArg1 arg1,
	       BoundArg2 arg2) {
  return new Callback2<T, BoundArg1, BoundArg2>(t, func, arg1, arg2);
}

/**
 * A convenience function to make Callback::CallbackArg objects
 * @exception std::bad_alloc It might throw std::bad_alloc if memory
 * is exhausted and the system throws in that case (this exception
 * will not be thrown if the library has been installed using the
 * --with-glib-memory-slices-no-compat configuration option: instead
 * glib will terminate the program if it is unable to obtain memory
 * from the operating system).  It will also throw if the assignment
 * operator of a bound argument throws.
 */
template <class T, class BoundArg1, class BoundArg2, class FreeArg>
CallbackArg<FreeArg>* make(T& t,
			   void (T::*func)(BoundArg1, BoundArg2, FreeArg),
			   BoundArg1 arg1,
			   BoundArg2 arg2) {
  return new CallbackArg2<T, BoundArg1, BoundArg2, FreeArg>(t, func, arg1, arg2);
}

/* const versions, for binding to const methods */

/**
 * A convenience function to make Callback::CallbackArg objects
 * @exception std::bad_alloc It might throw std::bad_alloc if memory
 * is exhausted and the system throws in that case.  This exception
 * will not be thrown if the library has been installed using the
 * --with-glib-memory-slices-no-compat configuration option (instead
 * glib will terminate the program if it is unable to obtain memory
 * from the operating system).
 */
template <class T>
Callback* make(const T& t,
	       void (T::*func)() const) {
  return new Callback0_const<T>(t, func);
}

/**
 * A convenience function to make Callback::CallbackArg objects
 * @exception std::bad_alloc It might throw std::bad_alloc if memory
 * is exhausted and the system throws in that case.  This exception
 * will not be thrown if the library has been installed using the
 * --with-glib-memory-slices-no-compat configuration option (instead
 * glib will terminate the program if it is unable to obtain memory
 * from the operating system).
 */
template <class T, class FreeArg>
CallbackArg<FreeArg>* make(const T& t,
			   void (T::*func)(FreeArg) const) {
  return new CallbackArg0_const<T, FreeArg>(t, func);
}

/**
 * A convenience function to make Callback::CallbackArg objects
 * @exception std::bad_alloc It might throw std::bad_alloc if memory
 * is exhausted and the system throws in that case (this exception
 * will not be thrown if the library has been installed using the
 * --with-glib-memory-slices-no-compat configuration option: instead
 * glib will terminate the program if it is unable to obtain memory
 * from the operating system).  It will also throw if the assignment
 * operator of a bound argument throws.
 */
template <class T, class BoundArg1>
Callback* make(const T& t,
	       void (T::*func)(BoundArg1) const,
	       BoundArg1 arg1) {
  return new Callback1_const<T, BoundArg1>(t, func, arg1);
}

/**
 * A convenience function to make Callback::CallbackArg objects
 * @exception std::bad_alloc It might throw std::bad_alloc if memory
 * is exhausted and the system throws in that case (this exception
 * will not be thrown if the library has been installed using the
 * --with-glib-memory-slices-no-compat configuration option: instead
 * glib will terminate the program if it is unable to obtain memory
 * from the operating system).  It will also throw if the assignment
 * operator of a bound argument throws.
 */
template <class T, class BoundArg1, class FreeArg>
CallbackArg<FreeArg>* make(const T& t,
			   void (T::*func)(BoundArg1, FreeArg) const,
			   BoundArg1 arg1) {
  return new CallbackArg1_const<T, BoundArg1, FreeArg>(t, func, arg1);
}

/**
 * A convenience function to make Callback::CallbackArg objects
 * @exception std::bad_alloc It might throw std::bad_alloc if memory
 * is exhausted and the system throws in that case (this exception
 * will not be thrown if the library has been installed using the
 * --with-glib-memory-slices-no-compat configuration option: instead
 * glib will terminate the program if it is unable to obtain memory
 * from the operating system).  It will also throw if the assignment
 * operator of a bound argument throws.
 */
template <class T, class BoundArg1, class BoundArg2>
Callback* make(const T& t,
	       void (T::*func)(BoundArg1, BoundArg2) const,
	       BoundArg1 arg1,
	       BoundArg2 arg2) {
  return new Callback2_const<T, BoundArg1, BoundArg2>(t, func, arg1, arg2);
}

/**
 * A convenience function to make Callback::CallbackArg objects
 * @exception std::bad_alloc It might throw std::bad_alloc if memory
 * is exhausted and the system throws in that case (this exception
 * will not be thrown if the library has been installed using the
 * --with-glib-memory-slices-no-compat configuration option: instead
 * glib will terminate the program if it is unable to obtain memory
 * from the operating system).  It will also throw if the assignment
 * operator of a bound argument throws.
 */
template <class T, class BoundArg1, class BoundArg2, class FreeArg>
CallbackArg<FreeArg>* make(const T& t,
			   void (T::*func)(BoundArg1, BoundArg2, FreeArg) const,
			   BoundArg1 arg1,
			   BoundArg2 arg2) {
  return new CallbackArg2_const<T, BoundArg1, BoundArg2, FreeArg>(t, func, arg1, arg2);
}

/* for static class methods and non-class functions */

/**
 * A convenience function to make Callback::CallbackArg objects
 * @exception std::bad_alloc It might throw std::bad_alloc if memory
 * is exhausted and the system throws in that case.  This exception
 * will not be thrown if the library has been installed using the
 * --with-glib-memory-slices-no-compat configuration option (instead
 * glib will terminate the program if it is unable to obtain memory
 * from the operating system).
 */
inline Callback* make(void (*func)()) {
  return new Callback0_static(func);
}

/**
 * A convenience function to make Callback::CallbackArg objects
 * @exception std::bad_alloc It might throw std::bad_alloc if memory
 * is exhausted and the system throws in that case.  This exception
 * will not be thrown if the library has been installed using the
 * --with-glib-memory-slices-no-compat configuration option (instead
 * glib will terminate the program if it is unable to obtain memory
 * from the operating system).
 */
template <class FreeArg>
CallbackArg<FreeArg>* make(void (*func)(FreeArg)) {
  return new CallbackArg0_static<FreeArg>(func);
}

/**
 * A convenience function to make Callback::CallbackArg objects
 * @exception std::bad_alloc It might throw std::bad_alloc if memory
 * is exhausted and the system throws in that case (this exception
 * will not be thrown if the library has been installed using the
 * --with-glib-memory-slices-no-compat configuration option: instead
 * glib will terminate the program if it is unable to obtain memory
 * from the operating system).  It will also throw if the assignment
 * operator of a bound argument throws.
 */
template <class BoundArg1>
Callback* make(void (*func)(BoundArg1),
	       BoundArg1 arg1) {
  return new Callback1_static<BoundArg1>(func, arg1);
}

/**
 * A convenience function to make Callback::CallbackArg objects
 * @exception std::bad_alloc It might throw std::bad_alloc if memory
 * is exhausted and the system throws in that case (this exception
 * will not be thrown if the library has been installed using the
 * --with-glib-memory-slices-no-compat configuration option: instead
 * glib will terminate the program if it is unable to obtain memory
 * from the operating system).  It will also throw if the assignment
 * operator of a bound argument throws.
 */
template <class BoundArg1, class FreeArg>
CallbackArg<FreeArg>* make(void (*func)(BoundArg1, FreeArg),
			   BoundArg1 arg1) {
  return new CallbackArg1_static<BoundArg1, FreeArg>(func, arg1);
}

/**
 * A convenience function to make Callback::CallbackArg objects
 * @exception std::bad_alloc It might throw std::bad_alloc if memory
 * is exhausted and the system throws in that case (this exception
 * will not be thrown if the library has been installed using the
 * --with-glib-memory-slices-no-compat configuration option: instead
 * glib will terminate the program if it is unable to obtain memory
 * from the operating system).  It will also throw if the assignment
 * operator of a bound argument throws.
 */
template <class BoundArg1, class BoundArg2>
Callback* make(void (*func)(BoundArg1, BoundArg2),
	       BoundArg1 arg1,
	       BoundArg2 arg2) {
  return new Callback2_static<BoundArg1, BoundArg2>(func, arg1, arg2);
}

/**
 * A convenience function to make Callback::CallbackArg objects
 * @exception std::bad_alloc It might throw std::bad_alloc if memory
 * is exhausted and the system throws in that case (this exception
 * will not be thrown if the library has been installed using the
 * --with-glib-memory-slices-no-compat configuration option: instead
 * glib will terminate the program if it is unable to obtain memory
 * from the operating system).  It will also throw if the assignment
 * operator of a bound argument throws.
 */
template <class BoundArg1, class BoundArg2, class FreeArg>
CallbackArg<FreeArg>* make(void (*func)(BoundArg1, BoundArg2, FreeArg),
			   BoundArg1 arg1,
			   BoundArg2 arg2) {
  return new CallbackArg2_static<BoundArg1, BoundArg2, FreeArg>(func, arg1, arg2);
}

} // namespace Callback

class Releaser;

namespace Callback {

/**
 * Posts a callback for execution by a glib main loop.  It is
 * thread-safe provided g_thread_init() has been called.  This
 * function will not throw.
 * @param cb The callback object.  Ownership is taken of this object,
 * and it will be deleted when it has been finished with.
 * @param priority The priority to be given to the callback in the
 * main loop.  In ascending order of priorities, priorities are
 * G_PRIORITY_LOW, G_PRIORITY_DEFAULT_IDLE, G_PRIORITY_HIGH_IDLE,
 * G_PRIORITY_DEFAULT and G_PRIORITY_HIGH. The default is
 * G_PRIORITY_DEFAULT_IDLE.  This determines the order in which the
 * callback will appear in the event list in the main loop, not the
 * priority which the OS will adopt
 * @param context The glib main loop context in which the callback is
 * to be executed (the default of NULL will cause the callback to be
 * executed in the main program loop, and this is usually what is
 * wanted).
 * @note Cancellation of the receiving thread is blocked when the
 * callback executes.
 * 
 * Since 0.9.2 choice of main context available.
 */
void post(const Callback* cb, gint priority = G_PRIORITY_DEFAULT_IDLE,
	  GMainContext* context = 0);

/**
 * Posts a callback for execution by a glib main loop.  It is
 * thread-safe provided g_thread_init() has been called.
 * @param cb The callback object.  Ownership is taken of this object,
 * and it will be deleted when it has been finished with.
 * @param r A Releaser object for automatic disconnection of the
 * callback from the main loop.
 * @param priority The priority to be given to the callback in the
 * main loop.  In ascending order of priorities, priorities are
 * G_PRIORITY_LOW, G_PRIORITY_DEFAULT_IDLE, G_PRIORITY_HIGH_IDLE,
 * G_PRIORITY_DEFAULT and G_PRIORITY_HIGH. The default is
 * G_PRIORITY_DEFAULT_IDLE.  This determines the order in which the
 * callback will appear in the event list in the main loop, not the
 * priority which the OS will adopt.
 * @param context The glib main loop context in which the callback is
 * to be executed (the default of NULL will cause the callback to be
 * executed in the main program loop, and this is usually what is
 * wanted).
 * @exception std::bad_alloc This function might throw std::bad_alloc
 * if memory is exhausted and the system throws in that case.  If it
 * does so, the Callback object will be disposed of.
 * @note Cancellation of the receiving thread is blocked when the
 * callback executes.
 * 
 * Since 0.9.2 choice of main context available.
 */
void post(const Callback* cb, Releaser& r,
	  gint priority = G_PRIORITY_DEFAULT_IDLE, GMainContext* context = 0);

} // namespace Callback

} // namespace Cgu

#endif
