/* Copyright (C) 2005 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYING distributed with the source files.

*/

#ifndef THREAD_H
#define THREAD_H

#include "prog_defs.h"

#include <memory>

#include <pthread.h>
#include <sigc++/sigc++.h>

namespace { // this function is placed in anonymous namespace in thread.ccp
            // so that is is not exported at link time
extern "C" void* thread_func(void*);

} // anonymous namespace

namespace Thread {


class Thread {
  pthread_t thread;
  // private constructor - this class can only be created with Thread::start
  Thread(void) {}
  // and it cannot be copied - this class has single ownership semantics
  Thread(const Thread&);
  Thread& operator=(const Thread&);
public:
  friend void* thread_func(void*);

  // Use Thread::cancel() with great care - sometimes its use is unavoidable
  // but destructors for local objects may not be called if a thread exits
  // by virtue of a call to Thread::cancel() (that depends on the implementation)
  // - therefore for maximum portability only use it if there are plain
  // data structures in existence in local scope when it is called and
  // if there is anything in free store to be released implement some
  // clean-ups with pthread_cleanup_push()/pthread_cleanup_pop() this
  // should be controlled with pthread_setcancelstate() and/or the
  // CancelBlock class to choose the cancellation point
  void cancel(void) {pthread_cancel(thread);}

  void join(void) {pthread_join(thread, 0);}

  // threads have single ownership semantics - the thread will keep running even
  // if the return value goes out of scope (but it will no longer be possible to call
  // any of the methods in this class for it)
  static std::auto_ptr<Thread> start(const sigc::slot<void>& cb, bool joinable);
};

class CancelBlock {
  // CancelBlocks cannot be copied (it is meaningless)
  CancelBlock(const CancelBlock&);
  CancelBlock& operator=(const CancelBlock&);
public:
  int block(int& old_state) {return pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, &old_state);}
  int block(void) {int old_state; return block(old_state);}
  int unblock(int& old_state) {return pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, &old_state);}
  int unblock(void) {int old_state; return unblock(old_state);}
  CancelBlock(bool blocking = true);
  ~CancelBlock(void) {unblock();}
};

} // namespace Thread

#endif
