/* Copyright (C) 2001 2002 2003 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/


#include <vector>

#include <cstdlib>
#include <iostream>
#include <fstream>
#include <strstream>
#include <iomanip>
#include <list>
#include <cstring>

#include <unistd.h>
#include <limits.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <dirent.h>
#include <stdlib.h>
#include <signal.h>

#include <gdkmm/pixbuf.h>
#include <gdk/gdkkeysyms.h> // the key codes are here
#include <gtkmm/image.h>
#include <gtkmm/stock.h>
#include <glibmm/convert.h>
#include <glibmm/timer.h>

#include "fax_list.h"
#include "dialogs.h"
#include "fax_list_icons.h"
#include "window_icon.h"

#ifdef HAVE_GETTEXT
#include <libintl.h>
#endif

int FaxListDialog::is_fax_received_list = 0;
int FaxListDialog::is_fax_sent_list = 0;


FaxListDialog::FaxListDialog(Mode mode_, const int standard_size_):
                              Gtk::Window(Gtk::WINDOW_TOPLEVEL),
			      mode(mode_), standard_size(standard_size_),
			      fax_list_box(false, 0), table(2, 1, false),
                              close_button(Gtk::Stock::CLOSE),
			      button_box(Gtk::BUTTONBOX_SPREAD) {

  // notify the existence of this object
  if (mode == received) is_fax_received_list++;
  else is_fax_sent_list++;

  fax_list_scroll_window.set_policy(Gtk::POLICY_ALWAYS, Gtk::POLICY_ALWAYS);
  fax_list_scroll_window.add(tree_view);
  fax_list_scroll_window.set_shadow_type(Gtk::SHADOW_IN);

  // create the tree model:
  list_store_r = Gtk::ListStore::create(model_columns);
  // connect it to the tree view
  tree_view.set_model(list_store_r);
  // add the columns of the tree model to tree view
  tree_view.append_column(gettext("Fax"), model_columns.fax_name);
  tree_view.append_column(gettext("Description"), model_columns.fax_description);
  // single line selection
  tree_view.get_selection()->set_mode(Gtk::SELECTION_SINGLE);
  // populate the fax list
  get_fax_list_rows();

  // set up the tool bar
  {
    using namespace Gtk::Toolbar_Helpers;
    
    tool_bar.set_orientation(Gtk::ORIENTATION_HORIZONTAL);
    tool_bar.set_toolbar_style(Gtk::TOOLBAR_ICONS);

    // first make the image widgets
    Gtk::Image* print_image_p = manage(new Gtk::Image(Gdk::Pixbuf::create_from_xpm_data(print_xpm)));
    Gtk::Image* view_image_p = manage(new Gtk::Image(Gdk::Pixbuf::create_from_xpm_data(view_xpm)));
    Gtk::Image* describe_image_p = manage(new Gtk::Image(Gdk::Pixbuf::create_from_xpm_data(describe_xpm)));
    Gtk::Image* delete_image_p = manage(new Gtk::Image(Gdk::Pixbuf::create_from_xpm_data(delete_xpm)));
    Gtk::Image* refresh_image_p = manage(new Gtk::Image(Gdk::Pixbuf::create_from_xpm_data(refresh_xpm)));

    tool_bar.set_tooltips(true);

    ToolList& tool_list = tool_bar.tools();
    tool_list.push_back(ButtonElem(*print_image_p, SigC::slot(*this, &FaxListDialog::print_fax_prompt),
				   gettext("Print selected fax")));
    print_button_p = static_cast<Gtk::Button*>(tool_list.back().get_widget());
    print_button_p->set_relief(Gtk::RELIEF_NONE);
    print_button_p->set_sensitive(false);

    tool_list.push_back(ButtonElem(*view_image_p, SigC::slot(*this, &FaxListDialog::view_fax),
				   gettext("View selected fax")));
    view_button_p = static_cast<Gtk::Button*>(tool_list.back().get_widget());
    view_button_p->set_relief(Gtk::RELIEF_NONE);
    view_button_p->set_sensitive(false);

    tool_list.push_back(ButtonElem(*describe_image_p, SigC::slot(*this, &FaxListDialog::describe_fax_prompt),
				   gettext("Add/amend description for selected fax")));
    describe_button_p = static_cast<Gtk::Button*>(tool_list.back().get_widget());
    describe_button_p->set_relief(Gtk::RELIEF_NONE);
    describe_button_p->set_sensitive(false);

    tool_list.push_back(ButtonElem(*delete_image_p, SigC::slot(*this, &FaxListDialog::delete_fax_prompt),
				   gettext("Delete selected fax")));
    delete_button_p = static_cast<Gtk::Button*>(tool_list.back().get_widget());
    delete_button_p->set_relief(Gtk::RELIEF_NONE);
    delete_button_p->set_sensitive(false);

    tool_list.push_back(ButtonElem(*refresh_image_p, SigC::slot(*this, &FaxListDialog::refresh_slot),
				   gettext("Refresh fax list")));
    Gtk::Button* refresh_button_p = static_cast<Gtk::Button*>(tool_list.back().get_widget());
    refresh_button_p->set_relief(Gtk::RELIEF_NORMAL);
    refresh_button_p->set_sensitive(true);
  }

  button_box.add(close_button);

  table.attach(fax_list_scroll_window, 0, 1, 0, 1, Gtk::EXPAND | Gtk::FILL,
         Gtk::EXPAND | Gtk::FILL, standard_size/3, standard_size/3);
  table.attach(button_box, 0, 1, 1, 2, Gtk::EXPAND | Gtk::FILL,
	 Gtk::SHRINK, standard_size/3, standard_size/3);

  fax_list_box.pack_start(tool_bar, false, false);
  fax_list_box.pack_start(table, true, true);

  close_button.signal_clicked().connect(SigC::slot(*this, &FaxListDialog::close_slot));
  close_button.set_flags(Gtk::CAN_DEFAULT);
  close_button.grab_focus();

  // now connect up the signal which indicates a selection has been made
  tree_view.get_selection()->signal_changed().connect(SigC::slot(*this, &FaxListDialog::set_buttons_slot));

  table.set_border_width(standard_size/3);

  if (mode == received) set_title(gettext("efax-gtk: Received fax list"));
  else set_title(gettext("efax-gtk: Sent fax list"));
  set_position(Gtk::WIN_POS_CENTER);
  add(fax_list_box);

  set_default_size(standard_size * 15, standard_size * 12);

  set_icon(Gdk::Pixbuf::create_from_xpm_data(window_icon_xpm));

  show_all();
}

FaxListDialog::~FaxListDialog(void) {
  // notify the destruction of this object
  if (mode == received) is_fax_received_list--;
  else is_fax_sent_list--;
}

void FaxListDialog::get_fax_list_rows(void) {

  string dir(prog_config.homedir);
  if (mode == received) dir += "/faxin";
  else dir += "/faxsent";

  chdir(dir.c_str());

  DIR* dir_p;
  if ((dir_p = opendir(dir.c_str())) == 0) {
    string msg("Can't open directory ");
    msg += dir;
    msg += '\n';
    write_error(msg.c_str());
  }
  else {

    struct dirent* direntry;
    struct stat statinfo;
    list<string> dir_list;

    // first populate dir_list with the directory names in $HOME/faxin
    while ((direntry = readdir(dir_p)) != 0) {
      stat(direntry->d_name, &statinfo);
      if (S_ISDIR(statinfo.st_mode)
	  && strcmp(direntry->d_name, "oldfax")
	  && strcmp(direntry->d_name, ".")
	  && strcmp(direntry->d_name, "..")
	  && (mode == sent
	      || strcmp(direntry->d_name, prog_config.receive_dirname))) {
	dir_list.push_back(direntry->d_name);
      }
    }
    closedir(dir_p);

    // now insert the directory names in fax_list, with description (if any)
    // first clear the list store -- we need to clear it here before
    // testing for an empty dir_list, or on deleting the last fax in the list,
    // the clear won't take out the former last entry

    list_store_r->clear();

    if (!dir_list.empty()) {
      // first sort them
      dir_list.sort();
      
      // populate the list store, and get the fax description (if any)
      string filename;
      string line;
      ifstream file;
      list<string>::iterator iter;

      for (iter = dir_list.begin(); iter!= dir_list.end(); ++iter) {
      
	// get a list store row to insert the fax name
	Gtk::TreeModel::Row row = *(list_store_r->append());
	row[model_columns.fax_name] = *iter;

	// now see if there is a description
	filename = dir + '/';
	filename += *iter;
	filename += "/Description";

#ifdef HAVE_IOS_NOCREATE
	file.open(filename.c_str(), ios::in | ios::nocreate);
#else
	// we must have Std C++ so we probably don't need a ios::nocreate
	// flag on a read open to ensure uniqueness
	file.open(filename.c_str(), ios::in);
#endif

	if (file) {
	  while (getline(file, line) && line.empty());
        }
	if (!line.empty()) row[model_columns.fax_description] = Glib::locale_to_utf8(line);

	filename = "";
	line = "";
	file.clear();
	file.close();
      }
    }
  }
  // reset current directory
  string temp(prog_config.homedir + "/faxin");
  chdir(temp.c_str());
}

void FaxListDialog::close_slot(void) {
  hide_all();
  delete this; // this is completely safe as the dialog is self-owning and modeless
}


bool FaxListDialog::on_delete_event(GdkEventAny*) {
  close_slot();
  return true; // returning true prevents destroy sig being emitted
}

void FaxListDialog::set_buttons_slot(void) {

  // see if anything is selected
  if (tree_view.get_selection()->get_selected()) {
    print_button_p->set_relief(Gtk::RELIEF_NORMAL);
    print_button_p->set_sensitive(true);

    view_button_p->set_relief(Gtk::RELIEF_NORMAL);
    view_button_p->set_sensitive(true);

    describe_button_p->set_relief(Gtk::RELIEF_NORMAL);
    describe_button_p->set_sensitive(true);

    delete_button_p->set_relief(Gtk::RELIEF_NORMAL);
    delete_button_p->set_sensitive(true);
  }
    
  else {
    print_button_p->set_relief(Gtk::RELIEF_NONE);
    print_button_p->set_sensitive(false);

    view_button_p->set_relief(Gtk::RELIEF_NONE);
    view_button_p->set_sensitive(false);

    describe_button_p->set_relief(Gtk::RELIEF_NONE);
    describe_button_p->set_sensitive(false);

    delete_button_p->set_relief(Gtk::RELIEF_NONE);
    delete_button_p->set_sensitive(false);
  }
}

void FaxListDialog::describe_fax_prompt(void) {


  Gtk::TreeModel::iterator row_iter = tree_view.get_selection()->get_selected();
  if (row_iter) {
    Glib::ustring description;
    description = (*row_iter)[model_columns.fax_description];
    
    DescriptionDialog* dialog_p = new DescriptionDialog(standard_size, description, *this);
    if (!dialog_p) {
      cerr << "Memory allocation error in FaxListDialog::describe_fax_prompt()" << endl;
      exit(MEM_ERROR);
    }
    dialog_p->accepted.connect(SigC::slot(*this, &FaxListDialog::describe_fax));
    // there is no memory leak -- the memory will be deleted when DescriptionDialog closes
  }
}

void FaxListDialog::describe_fax(const Glib::ustring& description) {

  Gtk::TreeModel::iterator row_iter = tree_view.get_selection()->get_selected();
  if (row_iter) {
    (*row_iter)[model_columns.fax_description] = description;

    string filename(prog_config.homedir);
    if (mode == received) filename += "/faxin/";
    else filename += "/faxsent/";

    filename += (*row_iter)[model_columns.fax_name];
    filename += "/Description";
    ofstream file(filename.c_str(), ios::out);
    if (file) file << Glib::locale_from_utf8(description);
    else {
      string msg("Can't open file ");
      msg += filename;
      msg += '\n';
      write_error(msg.c_str());
    }
  }
}

void FaxListDialog::print_fax_prompt(void) {

  Gtk::TreeModel::iterator row_iter = tree_view.get_selection()->get_selected();
  if (row_iter && !prog_config.print_cmd.empty()) {
    string faxnumber((*row_iter)[model_columns.fax_name]);
    Glib::ustring msg(gettext("Print fax "));
    msg += faxnumber + gettext("?");
    
    PromptDialog* dialog_p = new PromptDialog(msg, gettext("Print fax"), standard_size, *this);
    if (!dialog_p) {
      cerr << "Memory allocation error in FaxListDialog::print_fax_prompt()" << endl;
      exit(MEM_ERROR);
    }
    dialog_p->accepted.connect(SigC::slot(*this, &FaxListDialog::print_fax));
    // there is no memory leak -- the memory will be deleted when PromptDialog closes
  }
}

void FaxListDialog::print_from_stdin(void) {

  // this method prints from stdin
  // it never returns if it succeeds, as it calls exec()

  // now set up the parms for the print program
  vector<string> print_parms;
  string print_name;
  string::size_type end_pos;
    
  if ((end_pos = prog_config.print_cmd.find_first_of(' ')) != string::npos) { // we have parms
    print_name.assign(prog_config.print_cmd, 0, end_pos);
    print_parms.push_back(print_name);
    // find start of next parm
    string::size_type start_pos = prog_config.print_cmd.find_first_not_of(' ', end_pos);
    while (start_pos != string::npos) {
      end_pos = prog_config.print_cmd.find_first_of(' ', start_pos);
      if (end_pos != string::npos) {
	print_parms.push_back(prog_config.print_cmd.substr(start_pos, end_pos - start_pos));
	start_pos = prog_config.print_cmd.find_first_not_of(' ', end_pos); // prepare for next interation
      }
      else {
	print_parms.push_back(prog_config.print_cmd.substr(start_pos, 
				   prog_config.print_cmd.size() - start_pos));
	start_pos = end_pos;
      }
    }
  }

  else { // just a print command without parameters to be passed
    print_name = prog_config.print_cmd;
    print_parms.push_back(print_name);
  }

  char** exec_parms = new char*[print_parms.size() + 1]; // this does not create a leak
                                                         // it will be deleted by the system
                                                         // when we exec()
  char**  temp_pp = exec_parms;
  vector<string>::iterator iter;
  for (iter = print_parms.begin(); iter != print_parms.end(); ++iter, ++temp_pp) {
    *temp_pp = new char[iter->size() + 1]; // this does not create a leak
	                                   // it will be deleted by the system
	                                   // when we exec()
    strcpy(*temp_pp, iter->c_str());
  }
  
  *temp_pp = 0;

  execvp(print_name.c_str(), exec_parms);

  // if we reached this point, then the execvp() call must have failed
  write_error("Can't find the print program - please check your installation\n"
	      "and the PATH environmental variable\n");
  Glib::usleep(500000);
}

void FaxListDialog::fax_to_ps(const std::string& basename, bool allow_shrink) {

  // this method turns fax tiffg3 files to postscript, and
  // sends the output to stdout
  // it never returns if it succeeds, as it calls exec()

  // first set up the parms for efix
  vector<string> efix_parms;
  string temp;
    
  efix_parms.push_back("efix");
  efix_parms.push_back("-ve");
  efix_parms.push_back("-r300");
  efix_parms.push_back("-ops");
  temp = "-p";
  temp += prog_config.page_dim;
  efix_parms.push_back(temp);

  if (allow_shrink && prog_config.print_shrink.compare("100")) { // if print_shrink is not 100
    temp = "-s0.";
    temp += prog_config.print_shrink;
    efix_parms.push_back(temp);
      
    ostrstream strm;
    if (!prog_config.page_size.compare("a4")) {

      strm << "-d";
      float val = 210 * (100 - atoi(prog_config.print_shrink.c_str()))/200.0;
      strm << static_cast<int>(val + 0.5);
      val = 297 * (100 - atoi(prog_config.print_shrink.c_str()))/200.0;
      strm << ',' << static_cast<int>(val + 0.5) << "mm" << ends;
      
      const char* dim = strm.str();
      efix_parms.push_back(dim);
      delete[] dim;
    }
    
    else if (!prog_config.page_size.compare("letter")) {

      strm << "-d";
      float val = 216 * (100 - atoi(prog_config.print_shrink.c_str()))/200.0;
      strm << static_cast<int>(val + 0.5);
      val = 279 * (100 - atoi(prog_config.print_shrink.c_str()))/200.0;
      strm << ',' << static_cast<int>(val + 0.5) << "mm" << ends;
      
      const char* dim = strm.str();
      efix_parms.push_back(dim);
      delete[] dim;
    }

    else if (!prog_config.page_size.compare("legal")) {

      strm << "-d";
      float val = 216 * (100 - atoi(prog_config.print_shrink.c_str()))/200.0;
      strm << static_cast<int>(val + 0.5);
      val = 356 * (100 - atoi(prog_config.print_shrink.c_str()))/200.0;
      strm << ',' << static_cast<int>(val + 0.5) << "mm" << ends;
      
      const char* dim = strm.str();
      efix_parms.push_back(dim);
      delete[] dim;
    }
  }

  int partnumber = 1;
  ostrstream strm;
  strm << basename.c_str() << setfill('0') << setw(3) << partnumber << ends;
  const char* file_name = strm.str();
  int result = access(file_name, R_OK);
    
  while (!result) {
    efix_parms.push_back(file_name);
    delete[] file_name;
	
    partnumber++;
    ostrstream strm;
    strm << basename.c_str() << setfill('0') << setw(3) << partnumber << ends;
    file_name = strm.str();
    result = access(file_name, R_OK);
  }
  delete[] file_name;

  char** exec_parms = new char*[efix_parms.size() + 1]; // this does not create a leak
                                                        // it will be deleted by the system
                                                        // when we exec()
  char**  temp_pp = exec_parms;
  vector<string>::iterator iter;
  for (iter = efix_parms.begin(); iter != efix_parms.end(); ++iter, ++temp_pp) {
    *temp_pp = new char[iter->size() + 1]; // this does not create a leak
	                                   // it will be deleted by the system
	                                   // when we exec()
    strcpy(*temp_pp, iter->c_str());
  }
  
  *temp_pp = 0;
  execvp("efix", exec_parms);

  // if we reached this point, then the execvp() call must have failed
  write_error("Can't find the efix program - please check your installation\n"
	      "and the PATH environmental variable\n");
  Glib::usleep(500000);
}

void FaxListDialog::write_from_stdin(int dest_fd) {

  ssize_t result;
  char buffer[PIPE_BUF];

  while ((result = read(0, buffer, PIPE_BUF)) > 0) {
    write(dest_fd, buffer, result);
  }
  Glib::usleep(50000);
}

void FaxListDialog::ps_viewer(const char* filename) {

  // views a postscript file
  // it never returns if it succeeds, as it calls exec()

  // set up the parms for the view program
  vector<string> view_parms;
  string view_name;
  string::size_type end_pos;
    
  if ((end_pos = prog_config.ps_view_cmd.find_first_of(' ')) != string::npos) { // we have parms
    view_name.assign(prog_config.ps_view_cmd, 0, end_pos);
    view_parms.push_back(view_name);
    // find start of next parm
    string::size_type start_pos = prog_config.ps_view_cmd.find_first_not_of(' ', end_pos);
    while (start_pos != string::npos) {
      end_pos = prog_config.ps_view_cmd.find_first_of(' ', start_pos);
      if (end_pos != string::npos) {
	view_parms.push_back(prog_config.ps_view_cmd.substr(start_pos, end_pos - start_pos));
	start_pos = prog_config.ps_view_cmd.find_first_not_of(' ', end_pos); // prepare for next interation
      }
      else {
	view_parms.push_back(prog_config.ps_view_cmd.substr(start_pos, 
				   prog_config.ps_view_cmd.size() - start_pos));
	start_pos = end_pos;
      }
    }
  }

  else { // just a print command without parameters to be passed
    view_name = prog_config.ps_view_cmd;
    view_parms.push_back(view_name);
  }

  view_parms.push_back(filename);

  char** exec_parms = new char*[view_parms.size() + 1];  // this does not create a leak
                                                         // it will be deleted by the system
                                                         // when we exec()
  char**  temp_pp = exec_parms;
  vector<string>::iterator iter;
  for (iter = view_parms.begin(); iter != view_parms.end(); ++iter, ++temp_pp) {
    *temp_pp = new char[iter->size() + 1]; // this does not create a leak
	                                   // it will be deleted by the system
	                                   // when we exec()
    strcpy(*temp_pp, iter->c_str());
  }
  
  *temp_pp = 0;

  execvp(view_name.c_str(), exec_parms);

  // if we reached this point, then the execvp() call must have failed
  write_error("Can't find the ps viewer program - please check your installation\n"
	      "and the PATH environmental variable\n");
  Glib::usleep(500000);
}

void FaxListDialog::print_fax(void) {

  // get the base name of the fax to be printed
  // do this before we fork to avoid a possible race with another
  // FaxListDialog function resetting the value of the fax list selection
  Gtk::TreeModel::iterator row_iter = tree_view.get_selection()->get_selected();
  if (row_iter) {
    string basename(prog_config.homedir);
    if (mode == received) basename += "/faxin/";
    else basename += "/faxsent/";

    basename += (*row_iter)[model_columns.fax_name];
    basename += '/';
    basename += (*row_iter)[model_columns.fax_name];
    basename += '.';

    // now launch a new process to control the printing process
    // the main program process needs to continue while the printing
    // is going on
    pid_t pid = fork();

    if (pid == -1) {
      write_error("Fork error - exiting\n");
      exit(FORK_ERROR);
    }
    if (!pid) { // child print process
      
      connect_to_stderr();
      Pipe_fifo fork_pipe(Pipe_fifo::block);
    
      pid_t pid = fork();

      if (pid == -1) {
	write_error("Fork error - exiting\n");
	_exit(FORK_ERROR);
      }
      if (!pid) { // process to exec to print_from_stdin()
      
	fork_pipe.connect_to_stdin();
	print_from_stdin();
	// the exec() call in print_from_stdin() must have failed if we reached here
	// so we must now make this process end - use _exit(), not exit()
	_exit(0);
      }

      // this is the parent process which will send postscript to stdout
      fork_pipe.connect_to_stdout();
      fax_to_ps(basename, true);
      // the exec() call in fax_to_ps()  must have failed if we reached here
      // so we must now make this process end - use _exit(), not exit()
      _exit(0);
    }
  }
}

void FaxListDialog::view_fax(void) {

  Gtk::TreeModel::iterator row_iter = tree_view.get_selection()->get_selected();
  if (row_iter && !prog_config.ps_view_cmd.empty()) {
    string basename(prog_config.homedir);
    if (mode == received) basename += "/faxin/";
    else basename += "/faxsent/";

    basename += (*row_iter)[model_columns.fax_name];
    basename += '/';
    basename += (*row_iter)[model_columns.fax_name];
    basename += '.';

    // now launch a new process to control the viewing process
    // the main program process needs to continue while the viewing
    // is going on
    
    pid_t pid = fork();
    
    if (pid == -1) {
      write_error("Fork error\n");
      exit(FORK_ERROR);
    }
    if (!pid) { // child view-control process

      // restore the default SIGCHLD signal handler with SA_RESTART
      // for systems (such as Linux) which require it
      struct sigaction sig_act;
      sig_act.sa_handler = SIG_DFL;
      sigemptyset(&sig_act.sa_mask);
#ifdef SA_RESTART
      sig_act.sa_flags = SA_RESTART;
#else
      sig_act.sa_flags = 0;
#endif
      sigaction(SIGCHLD, &sig_act, 0);

      connect_to_stderr();

      // get a temporary file
      char filename[] = "/tmp/efax-gtk-view.XXXXXX";
      int file_fd = mkstemp(filename);

      if (file_fd == -1) {
	write_error("Failed to make temporary file:\n"
		    "please check permissions in /tmp directory");
	_exit(0);
      }

      // now fork again to create a process in which we can place a pipe
      // and write postscript to file, and wait on before invoking the
      // postscript viewer program
      pid_t pid = fork();

      if (pid == -1) {
	write_error("Fork error - exiting\n");
	_exit(FORK_ERROR);
      }

      if (!pid) { // child process to print to file from
      
	// now create a pipe and proceed to fork to write the postscript to temporary file
	Pipe_fifo fork_pipe(Pipe_fifo::block);

	// now fork again to create the process which will write postscript to stdout
	pid_t pid = fork();

	if (pid == -1) {
	  write_error("Fork error - exiting\n");
	  _exit(FORK_ERROR);
	}

	if (!pid) { // child process which will send postscript to stdout
      
	  fork_pipe.connect_to_stdout();
	  fax_to_ps(basename, false);
	  // the exec() call in fax_to_ps()  must have failed if we reached here
	  // so we must now make this process end - use _exit(), not exit()
	  _exit(0);
	}
	// this is the parent process to write to stdin
	fork_pipe.connect_to_stdin();
	write_from_stdin(file_fd);
	// we must now make this process end - use _exit(), not exit()
	_exit(0);
      }
      // this is the main viewing process again
      // now create another process from which to call ps_viewer()
      // (we need to do this rather than call ps_viewer() in this
      // process so that we can delete the temporary file when it's
      // finished with, because if it succeeds ps_viewer() never returns)

      // first wait for the temporary file to be created
      wait(0);

      close(file_fd);

      pid = fork();

      if (pid == -1) {
	write_error("Fork error - exiting\n");
	_exit(FORK_ERROR);
      }

      if (!pid) { // child process to call ps_viewer() from
      
	ps_viewer(filename);
	// the exec() call in ps_viewer()  must have failed if we reached here
	// so we must now make this process end - use _exit(), not exit()
	_exit(0);
      }
      // now we are back to the main viewing process again
      // delete the temporary file

      // first wait till the ps_viewer() process is closed by the user exiting
      // the postscript viewing program
      wait(0);

      unlink(filename);
      // now terminate the process
      _exit(0);
    }
  }
}

void FaxListDialog::delete_fax_prompt(void) {

  Gtk::TreeModel::iterator row_iter = tree_view.get_selection()->get_selected();
  if (row_iter) {
    string faxnumber((*row_iter)[model_columns.fax_name]);
    Glib::ustring msg(gettext("Delete fax "));
    msg += faxnumber + (gettext("?"));
    
    PromptDialog* dialog_p = new PromptDialog(msg, gettext("Delete fax"), standard_size, *this);
    if (!dialog_p) {
      cerr << "Memory allocation error in FaxListDialog::delete_fax_prompt()" << endl;
      exit(MEM_ERROR);
    }
    dialog_p->accepted.connect(SigC::slot(*this, &FaxListDialog::delete_fax));
    // there is no memory leak -- the memory will be deleted when PromptDialog closes
  }
}

void FaxListDialog::delete_fax(void) {

  Gtk::TreeModel::iterator row_iter = tree_view.get_selection()->get_selected();
  if (row_iter) {
    // get the name of the fax to be moved and the new name
    string old_dirname(prog_config.homedir);
    string new_dirname(prog_config.homedir);
    if (mode == received) {
      old_dirname += "/faxin/";
      new_dirname += "/faxin/oldfax/";
    }
    else {
      old_dirname += "/faxsent/";
      new_dirname += "/faxsent/oldfax/";
    }
    old_dirname += (*row_iter)[model_columns.fax_name];
    new_dirname += (*row_iter)[model_columns.fax_name];

    // make a new directory to copy the files into
    mkdir(new_dirname.c_str(), S_IRUSR | S_IWUSR | S_IXUSR);

    // it should be possible to use link()/unlink() as the target and source
    // are sub-directories of $HOME/faxin or $HOME/faxsent.  If because of a
    // very odd filesystem arrangement this won't work, the method will do nothing

    vector<string> filelist;
    struct dirent* direntry;
    struct stat statinfo;

    DIR* dir_p;
    if ((dir_p = opendir(old_dirname.c_str())) == 0) {
      string msg("Can't open directory ");
      msg += old_dirname;
      msg += '\n';
      write_error(msg.c_str());
    }

    else {
      chdir(old_dirname.c_str());
      while ((direntry = readdir(dir_p)) != 0) {
	stat(direntry->d_name, &statinfo);
	if (S_ISREG(statinfo.st_mode)) {
	  filelist.push_back(direntry->d_name);
	}
      }

      closedir(dir_p);

      bool failure = false;
      vector<string>::iterator iter;
      string old_filename;
      string new_filename;

      for (iter = filelist.begin(); iter != filelist.end(); ++iter) {
	old_filename = old_dirname + '/';
	old_filename += *iter;
	new_filename = new_dirname + '/';
	new_filename += *iter;
	if (link(old_filename.c_str(), new_filename.c_str())) {
	  failure = true;
	  break;
	}
      }

      if (failure) { // recover position
	vector<string>::iterator recover_iter;
	for (recover_iter = filelist.begin(); recover_iter != iter; ++recover_iter) {
	  new_filename = new_dirname + '/';
	  new_filename += *recover_iter;
	  unlink(new_filename.c_str());
	}
	rmdir(new_dirname.c_str());
	string msg("Can't move directory ");
	msg += old_dirname;
	msg += " to ";
	msg += new_dirname;
	msg += '\n';
	write_error(msg.c_str());
      }
      else {
	for (iter = filelist.begin(); iter != filelist.end(); ++iter) {
	  old_filename = old_dirname + '/';
	  old_filename += *iter;
	  unlink(old_filename.c_str());
	}
	if (rmdir(old_dirname.c_str())) {
	  string msg("Can't delete directory ");
	  msg += old_dirname;
	  msg += "\nThe contents should have been moved to ";
	  msg += new_dirname;
	  msg += "\nand it should now be empty -- please check\n";
	  write_error(msg.c_str());
	}
	else get_fax_list_rows();
      }
      // reset current directory
      string temp(prog_config.homedir + "/faxin");
      chdir(temp.c_str());
    }
  }
}

void FaxListDialog::refresh_slot(void) {

  get_fax_list_rows();
  set_buttons_slot();
}

DescriptionDialog::DescriptionDialog(const int standard_size, const string& text, Gtk::Window& window):
                             Gtk::Window(Gtk::WINDOW_TOPLEVEL), in_run_loop(false),
			     ok_button(Gtk::Stock::OK), cancel_button(Gtk::Stock::CANCEL),
			     button_box(Gtk::BUTTONBOX_END, standard_size/2),
			     label(gettext("Fax description?")), table(3, 1, false),
                             parent(window) {

  button_box.add(cancel_button);
  button_box.add(ok_button);

  table.attach(label, 0, 1, 0, 1, Gtk::FILL | Gtk::EXPAND,
	 Gtk::FILL | Gtk::EXPAND, standard_size/2, standard_size/4);

  table.attach(entry, 0, 1, 1, 2, Gtk::FILL | Gtk::EXPAND,
	 Gtk::EXPAND, standard_size/2, standard_size/4);

  table.attach(button_box, 0, 1, 2, 3, Gtk::FILL | Gtk::EXPAND,
	 Gtk::SHRINK, standard_size/2, standard_size/4);

  ok_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &DescriptionDialog::selected), true));
  cancel_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &DescriptionDialog::selected), false));

  ok_button.set_flags(Gtk::CAN_DEFAULT);
  cancel_button.set_flags(Gtk::CAN_DEFAULT);

  add(table);
  
  entry.set_text(text);
  set_title(gettext("Fax description"));
  set_transient_for(parent);
  set_type_hint(Gdk::WINDOW_TYPE_HINT_DIALOG);
  parent.set_sensitive(false);
  set_modal(true);

  entry.set_size_request(standard_size * 9, standard_size);
  set_border_width(standard_size/2);

  entry.grab_focus();

  set_position(Gtk::WIN_POS_CENTER);
  set_resizable(false);

  set_icon(Gdk::Pixbuf::create_from_xpm_data(window_icon_xpm));

  show_all();

  // we now need to deselect what is in the entry
  entry.select_region(0,0);
}

void DescriptionDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

void DescriptionDialog::selected(bool accept) {
  parent.set_sensitive(true); // do this before we emit accepted()
  hide_all();
  if (accept) accepted(entry.get_text());
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

bool DescriptionDialog::on_delete_event(GdkEventAny*) {
  selected(false);
  return true; // returning true prevents destroy sig being emitted
}

bool DescriptionDialog::on_key_press_event(GdkEventKey* event_p) {

  if (event_p->keyval == GDK_Escape) selected(false);
  else if (event_p->keyval == GDK_Return && !cancel_button.has_focus()) selected(true);
  else Gtk::Window::on_key_press_event(event_p);
  return true; // processing ends here
}
