/* Copyright (C) 2001 2002 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#include <sys/types.h>
#include <unistd.h>

#include <vector>
#include <cstdlib>
#include <iostream>

#include <gtkmm/main.h>
#include <gdk/gdkkeysyms.h> // the key codes are here
#include <gdkmm/pixbuf.h>
#include <gtkmm/stock.h>
#include <glibmm/timer.h>

#include "dialogs.h"
#include "gpl.h"
#include "window_icon.h"

#ifdef HAVE_GETTEXT
#include <libintl.h>
#endif

FileReadSelectDialog::FileReadSelectDialog(const std::string& filename, int size, Gtk::Window& window):
                                           Gtk::FileSelection(gettext("File to fax")),
					   in_run_loop(false), standard_size(size),
					   view_button(gettext("View")),
					   parent(window) {

  hide_fileop_buttons();

  if (!filename.empty() && filename[0] == '/') set_filename(filename);
  else if (!prog_config.homedir.empty()) set_filename(prog_config.homedir + "/faxout/");

  get_action_area()->set_homogeneous(false);
  get_action_area()->pack_start(view_filler, true, false, 0);
  get_action_area()->pack_start(view_button, false, false, 0);

  set_transient_for(parent);
  set_type_hint(Gdk::WINDOW_TYPE_HINT_DIALOG);
  parent.set_sensitive(false);
  set_modal(true);
  set_position(Gtk::WIN_POS_CENTER);

  get_ok_button()->signal_clicked().connect(SigC::bind(SigC::slot(*this, &FileReadSelectDialog::selected), true));
  get_cancel_button()->signal_clicked().connect(SigC::bind(SigC::slot(*this, &FileReadSelectDialog::selected), false));
  view_button.signal_clicked().connect(SigC::slot(*this, &FileReadSelectDialog::view_file));
  view_button.set_flags(Gtk::CAN_DEFAULT);

  set_icon(Gdk::Pixbuf::create_from_xpm_data(window_icon_xpm));

  show_all();

  // now get and set the button size for view_button
  view_button.set_size_request(get_cancel_button()->get_width(), get_cancel_button()->get_height());
}

void FileReadSelectDialog::view_file(void) {

  // check pre-conditions
    if (get_filename().empty()
	|| get_filename()[get_filename().size() - 1] == '/'
	|| access(get_filename().c_str(), R_OK)) {
    beep();
    return;
  }

  pid_t pid = fork();

  if (pid == -1) {
    write_error("Fork error - exiting\n");
    exit(FORK_ERROR);
  }
  if (!pid) {  // child process - as soon as everything is set up we are going to do an exec()

    connect_to_stderr();

    // now set up the parms for the view program
    vector<string> view_parms;
    string view_name;
    string::size_type end_pos;
    
    if ((end_pos = prog_config.ps_view_cmd.find_first_of(' ')) != string::npos) { // we have parms
      view_name.assign(prog_config.ps_view_cmd, 0, end_pos);
      view_parms.push_back(view_name);
      // find start of next parm
      string::size_type start_pos = prog_config.ps_view_cmd.find_first_not_of(' ', end_pos);
      while (start_pos != string::npos) {
	end_pos = prog_config.ps_view_cmd.find_first_of(' ', start_pos);
	if (end_pos != string::npos) {
	  view_parms.push_back(prog_config.ps_view_cmd.substr(start_pos, end_pos - start_pos));
	  start_pos = prog_config.ps_view_cmd.find_first_not_of(' ', end_pos); // prepare for next interation
	}
	else {
	  view_parms.push_back(prog_config.ps_view_cmd.substr(start_pos, 
	     				     prog_config.ps_view_cmd.size() - start_pos));
	  start_pos = end_pos;
	}
      }
    }

    else { // just a view command without parameters to be passed
      view_name = prog_config.ps_view_cmd;
      view_parms.push_back(view_name);
    }

    view_parms.push_back(get_filename());

    char** exec_parms = new char*[view_parms.size() + 1]; // this does not create a leak
                                                          // it will be deleted by the system
                                                          // when the child process terminates
    char**  temp_pp = exec_parms;
    vector<string>::iterator iter;
    for (iter = view_parms.begin(); iter != view_parms.end(); ++iter, ++temp_pp) {
	*temp_pp = new char[iter->size() + 1]; // this does not create a leak
	                                       // it will be deleted by the system
	                                       // when the child process terminates
	strcpy(*temp_pp, iter->c_str());
    }
    
    *temp_pp = 0;
    execvp(view_name.c_str(), exec_parms);

    // if we reached this point, then the execvp() call must have failed
    write_error("Can't find the postscript viewer program - please check your installation\n"
		"and the PATH environmental variable\n");
    Glib::usleep(500000);
    // use _exit(), not exit()
    _exit(0);
  } // end of view program process
}

string FileReadSelectDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
  return result;
}

void FileReadSelectDialog::selected(bool accept) {
  if (accept) result = get_filename();
  finish();
}

void FileReadSelectDialog::finish(void) {
  parent.set_sensitive(true);
  hide_all();
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

bool FileReadSelectDialog::on_delete_event(GdkEventAny*) {
  finish();
  return true; // returning true prevents destroy sig being emitted
}

GplDialog::GplDialog(int standard_size): Gtk::Window(Gtk::WINDOW_TOPLEVEL), in_run_loop(false),
			     result(rejected), accept_button(gettext("Accept")),
			     reject_button(gettext("Reject")),
		             label(gettext("Do you accept the Conditions, Notices "
					   "and Disclaimers shown above?")),
			     table(3, 2, false) {

  editbox.set_editable(false);
  editbox.modify_font(Pango::FontDescription(prog_config.fixed_font));
  editbox.get_buffer()->set_text(copyright_msg);

  scrolled_window.set_shadow_type(Gtk::SHADOW_IN);
  scrolled_window.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
  scrolled_window.add(editbox);

  table.attach(scrolled_window, 0, 2, 0, 1, Gtk::FILL | Gtk::EXPAND,
	       Gtk::FILL | Gtk::EXPAND, 0, 0);
  table.attach(label, 0, 2, 1, 2, Gtk::FILL | Gtk::EXPAND,
	       Gtk::SHRINK, 0, standard_size/3);
  table.attach(accept_button, 0, 1, 2, 3, Gtk::SHRINK,
	       Gtk::SHRINK, 0, standard_size/3);
  table.attach(reject_button, 1, 2, 2, 3, Gtk::SHRINK,
	       Gtk::SHRINK, 0, standard_size/3);

  accept_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &GplDialog::selected), accepted));
  reject_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &GplDialog::selected), rejected));
  
  add(table);
  
  set_title(gettext("efax-gtk: Conditions, Notices and Disclaimers"));
  set_modal(true);

  accept_button.set_size_request(standard_size * 3, standard_size);
  reject_button.set_size_request(standard_size * 3, standard_size);
  set_default_size(standard_size * 25, standard_size * 16);
  
  set_border_width(standard_size/4);
  set_position(Gtk::WIN_POS_CENTER);
  
  grab_focus();
  accept_button.unset_flags(Gtk::CAN_FOCUS);
  reject_button.unset_flags(Gtk::CAN_FOCUS);
  editbox.unset_flags(Gtk::CAN_FOCUS);
  
  show_all();
}

int GplDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
  return result;
}

void GplDialog::selected(Result selection) {
  hide_all();
  result = selection;
  if (in_run_loop) Gtk::Main::quit(); // this will cause the run() method to return, with result as its return value
}

bool GplDialog::on_delete_event(GdkEventAny*) {
  selected(rejected);
  return true; // returning true prevents destroy sig being emitted
}

bool GplDialog::on_key_press_event(GdkEventKey* event_p) {

  int keycode = event_p->keyval;
  
  if (keycode == GDK_Escape) selected(rejected);
  
  else if (keycode == GDK_Home || keycode == GDK_End
	   || keycode == GDK_Up || keycode == GDK_Down
	   || keycode == GDK_Page_Up || keycode == GDK_Page_Down) {
    editbox.on_key_press_event(event_p);
    return false;
  }
  return true;
}

InfoDialog::InfoDialog(const Glib::ustring& text, const Glib::ustring& caption, int standard_size,
                       Gtk::MessageType message_type, Gtk::Window& window):
                             Gtk::MessageDialog(text, message_type, Gtk::BUTTONS_CLOSE),
			     in_run_loop(false), parent(window) {

  set_title(caption);
  set_transient_for(parent);
  set_type_hint(Gdk::WINDOW_TYPE_HINT_DIALOG);
  parent.set_sensitive(false);
  set_modal(true);

  get_action_area()->set_layout(Gtk::BUTTONBOX_SPREAD);
  Gtk::Button* button_p = static_cast<Gtk::Button*>(get_action_area()->children().front().get_widget());
  button_p->signal_clicked().connect(SigC::slot(*this, &InfoDialog::selected));

  set_position(Gtk::WIN_POS_CENTER);
  set_resizable(false);

  set_icon(Gdk::Pixbuf::create_from_xpm_data(window_icon_xpm));

  show_all();
}

void InfoDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

void InfoDialog::selected(void) {
  parent.set_sensitive(true);
  hide_all();
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

bool InfoDialog::on_delete_event(GdkEventAny*) {
  selected();
  return true; // returning true prevents destroy sig being emitted
}

bool InfoDialog::on_key_press_event(GdkEventKey* event_p) {
  if (event_p->keyval == GDK_Return || event_p->keyval == GDK_Escape) selected();
  return false;
}

PromptDialog::PromptDialog(const Glib::ustring& text, const Glib::ustring& caption, int standard_size, Gtk::Window& window):
                             Gtk::Window(Gtk::WINDOW_TOPLEVEL), in_run_loop(false),
			     ok_button(Gtk::Stock::OK), cancel_button(Gtk::Stock::CANCEL),
			     button_box(Gtk::BUTTONBOX_END, standard_size/2),
			     label(text), table(2, 1, false), parent(window) {

  label.set_line_wrap(true);

  button_box.add(cancel_button);
  button_box.add(ok_button);

  table.attach(label, 0, 1, 0, 1, Gtk::FILL | Gtk::EXPAND,
	 Gtk::FILL | Gtk::EXPAND, standard_size/2, standard_size/4);
  table.attach(button_box, 0, 1, 1, 2, Gtk::FILL | Gtk::EXPAND,
	 Gtk::SHRINK, standard_size/2, standard_size/4);

  ok_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &PromptDialog::selected), true));
  cancel_button.signal_clicked().connect(SigC::bind(SigC::slot(*this, &PromptDialog::selected), false));
  ok_button.set_flags(Gtk::CAN_DEFAULT);
  cancel_button.set_flags(Gtk::CAN_DEFAULT);

  add(table);
  
  set_title(caption);
  set_transient_for(parent);
  set_type_hint(Gdk::WINDOW_TYPE_HINT_DIALOG);
  parent.set_sensitive(false);
  set_modal(true);

  set_border_width(standard_size/2);
  ok_button.grab_focus();
  set_position(Gtk::WIN_POS_CENTER);
  set_resizable(false);

  set_icon(Gdk::Pixbuf::create_from_xpm_data(window_icon_xpm));

  show_all();
}

void PromptDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

void PromptDialog::selected(bool accept) {
  parent.set_sensitive(true); // do this before we emit accepted()
  hide_all();
  if (accept) accepted();
  else rejected();
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

bool PromptDialog::on_delete_event(GdkEventAny*) {
  selected(false);
  return true; // returning true prevents destroy sig being emitted
}

bool PromptDialog::on_key_press_event(GdkEventKey* event_p) {
  if (event_p->keyval == GDK_Escape) selected(false);
  if (event_p->keyval == GDK_Return || event_p->keyval == GDK_Tab) Gtk::Window::on_key_press_event(event_p);
  return false;
}
