/*
 * Eee Applet 
 * Copyright (C) 2008 Olivier Rolland <billl@users.sourceforge.net>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <glib.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>

#include <sys/socket.h>
#include <sys/un.h>

#include "eee-monitor-hotkeys.h"

#define EEE_MONITOR_HOTKEYS_GET_PRIVATE(obj) \
    (G_TYPE_INSTANCE_GET_PRIVATE((obj), EEE_TYPE_MONITOR_HOTKEYS, EeeMonitorHotkeysPriv))

enum
{
  HK_NONE,
  HK_VOL_MUTE,
  HK_VOL_DOWN,
  HK_VOL_UP,
  HK_BRN_DOWN,
  HK_BRN_UP
};

struct _EeeMonitorHotkeysPriv
{
  gint src, brn;
};

static void     eee_monitor_hotkeys_finalize (GObject    *object);
static gboolean eee_monitor_hotkeys_run      (EeeMonitor *monitor);

G_DEFINE_TYPE (EeeMonitorHotkeys, eee_monitor_hotkeys, EEE_TYPE_MONITOR)

static void
eee_monitor_hotkeys_init (EeeMonitorHotkeys *monitor)
{
  monitor->priv = EEE_MONITOR_HOTKEYS_GET_PRIVATE (monitor);
}

static void
eee_monitor_hotkeys_class_init (EeeMonitorHotkeysClass *klass)
{
  GObjectClass    *object_class;
  EeeMonitorClass *monitor_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = eee_monitor_hotkeys_finalize;

  monitor_class = EEE_MONITOR_CLASS (klass);
  monitor_class->run = eee_monitor_hotkeys_run;

  g_type_class_add_private (klass, sizeof (EeeMonitorHotkeysPriv));
}

static void
eee_monitor_hotkeys_finalize (GObject *object)
{
  EeeMonitorHotkeys *monitor;

  monitor = EEE_MONITOR_HOTKEYS (object);

  if (monitor->priv->src)
  {
    g_source_remove (monitor->priv->src);
    monitor->priv->src = 0;
  }

  G_OBJECT_CLASS (eee_monitor_hotkeys_parent_class)->finalize (object);
}

unsigned int
eee_monitor_hotkeys_parse (EeeMonitorHotkeys *monitor, const gchar *buffer)
{
  gint hotkey = HK_NONE;

  if (strncmp ("000000", buffer, 6))
    return HK_NONE;

  if (buffer[6] == '2')
  {
/*
    int  val;

    if ('0'<=buffer[7] && buffer[7]<='9')
      val = buffer[7] - '0';
    else if ('a'<=buffer[7] && buffer[7]<='f')
      val = buffer[7] -'a' +10;
    else if ('A'<= buffer[7] && buffer[7]<='F')
      val = buffer[7] -'A' +10;
    else
      val = bri_value;

    if ((val > bri_value) || val==15)
      hotkey = HK_BRN_UP;
    else if (val < bri_value || val==0)
      hotkey = HK_BRN_DOWN;

    bri_value = val;
*/
  }
  else
  {
    // buffer[8] = 0;
    hotkey = atoi(buffer);
    switch (hotkey)
    {
      case 13:
        hotkey = HK_VOL_MUTE;
        break;
      case 14:
        hotkey = HK_VOL_DOWN;
        break;
      case 15:
        hotkey = HK_VOL_UP;
        break;
    }
  }

  return (unsigned int) hotkey;
}

static gboolean
eee_monitor_hotkeys_watch (GIOChannel *channel, GIOCondition condition, EeeMonitorHotkeys *monitor)
{
  GIOStatus status;
  gchar *buffer;

  status = g_io_channel_read_line (channel, &buffer, NULL, NULL, NULL);
  if (status != G_IO_STATUS_NORMAL)
  {
    g_free (buffer);
    return status == G_IO_STATUS_AGAIN;
  }

  if (strncmp ("hotkey ATKD ", buffer, 12) == 0)
  {
    gint hotkey;

    hotkey = eee_monitor_hotkeys_parse (monitor, buffer + 12);
    switch (hotkey)
    {
        case HK_VOL_MUTE:
          system ("xte \"key XF86AudioMute\"");
          break;
        case HK_VOL_DOWN:
          system ("xte \"key XF86AudioLowerVolume\"");
          break;
        case HK_VOL_UP:
          system ("xte \"key XF86AudioRaiseVolume\"");
          break;
        case HK_BRN_DOWN:
          break;
        case HK_BRN_UP:
          break;
        default:
          break;
      }
  }

  return TRUE;
}

static void
eee_monitor_hotkeys_notify (EeeMonitorHotkeys *monitor)
{
  EEE_MONITOR_HOTKEYS (monitor)->priv->src = 0;
}

static gboolean
eee_monitor_hotkeys_run (EeeMonitor *monitor)
{
  int fd;
  struct sockaddr_un socket_addr;
  GIOChannel *channel;

  fd = socket (AF_UNIX, SOCK_STREAM, 0);
  if (fd < 0)
    return FALSE;

  memset (&socket_addr, 0, sizeof (socket_addr));
  socket_addr.sun_family = AF_UNIX;
  strcpy (socket_addr.sun_path, "/var/run/acpid.socket");

  if (connect (fd, (struct sockaddr *) &socket_addr, sizeof (socket_addr)) < 0)
  {
    close (fd);
    return FALSE;
  }

  fcntl (fd, F_SETFD, FD_CLOEXEC);
  setvbuf (stdout, NULL, _IOLBF, 0);
  
  channel = g_io_channel_unix_new (fd);
  g_io_channel_set_close_on_unref (channel, TRUE);
  g_io_channel_set_encoding (channel, NULL, NULL);

  EEE_MONITOR_HOTKEYS (monitor)->priv->src = g_io_add_watch_full (channel, G_PRIORITY_HIGH_IDLE,
      G_IO_IN | G_IO_HUP | G_IO_ERR, (GIOFunc) eee_monitor_hotkeys_watch, NULL,
      (GDestroyNotify) eee_monitor_hotkeys_notify);
  g_io_channel_unref (channel);

  return TRUE;
}

EeeMonitor *
eee_monitor_hotkeys_new (void)
{
  return g_object_new (EEE_TYPE_MONITOR_HOTKEYS, NULL);
}

