/* This file is part of EdiTeX, an editor of mathematical
 * expressions based on TeX syntax.
 * 
 * Copyright (C) 2002-2003 Luca Padovani <lpadovan@cs.unibo.it>,
 *                    2003 Paolo Marinelli <pmarinel@cs.unibo.it>.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For more information, please visit the project's home page
 * http://helm.cs.unibo.it/editex/
 * or send an email to <lpadovan@cs.unibo.it>
 */

#include <config.h>
#include <stdio.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <gtkmathview.h>
#include "guiGTK.h"

#define XLINK_NS_URI "http://www.w3.org/1999/xlink"

static GtkWidget* window;
static GtkWidget* main_area;
static GtkWidget* scrolled_area;

static gpointer context = NULL;
static gchar* doc_name = NULL;
static GdomeElement* first_selected = NULL;
static GdomeElement* root_selected = NULL;

static void create_widget_set(gpointer);
static GtkWidget* get_main_menu(void);
static void file_open(GtkWidget*, gpointer);
static void file_re_open(GtkWidget*, gpointer);
static void file_close(GtkWidget*, gpointer);
static void file_output_tex(GtkWidget*, gpointer);
static void options_set_font_size(GtkWidget*, gpointer);
static void options_change_font_size(GtkWidget*, gboolean);
static void options_verbosity(GtkWidget*, guint);
static void edit_delete_selection(GtkWidget*, gpointer);
static void edit_select_parent(GtkWidget*, gpointer);
static void edit_reset_selection(GtkWidget*, gpointer);
static void edit_reset(GtkWidget*, gpointer);
static void edit_insert(GtkWidget*, gpointer);
static void help_about(GtkWidget*, gpointer);

static GtkItemFactoryEntry menu_items[] = {
  { "/_File",                          NULL,         NULL,          0, "<Branch>" },
  { "/File/_Open...",                  "<control>O", file_open,     0, NULL },
  { "/File/_Reopen",                   NULL,         file_re_open,  0, NULL },
  { "/File/_Close",                    "<control>W", file_close,    0, NULL },
  { "/File/Output _TeX",               NULL,         file_output_tex, 0, NULL },
  { "/File/sep1",                      NULL,         NULL,          0, "<Separator>" },
  { "/File/_Quit",                     "<control>Q", gtk_main_quit, 0, NULL },

  { "/_Edit",                          NULL, NULL,                  0,  "<Branch>" },
  { "/Edit/Reset Selection",           NULL, edit_reset_selection,  0, NULL },
  { "/Edit/Delete Selection",          NULL, edit_delete_selection, 0, NULL },
  { "/Edit/Select Parent",             NULL, edit_select_parent,    0, NULL },
  { "/Edit/sep1",                      NULL,         NULL,          0, "<Separator>" },
  { "/Edit/_Reset",                    NULL, edit_reset,            0, NULL },
  { "/Edit/Insert...",                 "<control>I", edit_insert,   0, NULL },

  { "/_Options",                       NULL, NULL,                  0,  "<Branch>" },
  { "/Options/Default _Font Size",     NULL, NULL,                  0,  "<Branch>" },
  { "/Options/Default Font Size/Set...", NULL, options_set_font_size, 0,  NULL },
  { "/Options/Default Font Size/sep1", NULL, NULL,                  0,  "<Separator>" },
  { "/Options/Default Font Size/Larger", NULL, options_change_font_size, TRUE, NULL },
  { "/Options/Default Font Size/Smaller", NULL, options_change_font_size, FALSE, NULL },
  { "/Options/Verbosity",              NULL, NULL,                  0,  "<Branch>" },
  { "/Options/Verbosity/_Errors",      NULL, options_verbosity,     0,  "<RadioItem>" },
  { "/Options/Verbosity/_Warnings",    NULL, options_verbosity,     1,  "/Options/Verbosity/Errors" },
  { "/Options/Verbosity/_Info",        NULL, options_verbosity,     2,  "/Options/Verbosity/Errors" },
  { "/Options/Verbosity/_Debug",       NULL, options_verbosity,     3,  "/Options/Verbosity/Errors" },

  { "/_Help" ,        NULL,         NULL,          0, "<LastBranch>" },
  { "/Help/About...", NULL,         help_about,    0, NULL }
};

static void
quick_message(const char* msg)
{
  GtkWidget* dialog;
  GtkWidget* label;
  GtkWidget* okay_button;
     
  /* Create the widgets */
     
  dialog = gtk_dialog_new();
  label = gtk_label_new (msg);
  okay_button = gtk_button_new_with_label("OK");

  gtk_widget_set_usize(dialog, 300, 100);

  /* Ensure that the dialog box is destroyed when the user clicks ok. */
     
  gtk_signal_connect_object (GTK_OBJECT (okay_button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy), dialog);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
		     okay_button);
  
  /* Add the label, and show everything we've added to the dialog. */
  
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), label);
  gtk_widget_show_all (dialog);
}

static void
load_error_msg(const char* name)
{
  char* msg = g_strdup_printf("Could not load\n`%s'", name);
  quick_message(msg);
  g_free(msg);
}

static guint edit_timeout_id;
extern void edit_timeout(gpointer);

void
GUI_init(int* argc, char*** argv, char* title, guint width, guint height, gpointer c)
{
  gtk_init(argc, argv);

  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(window), title);
  gtk_window_set_default_size(GTK_WINDOW(window), width, height);
  gtk_signal_connect(GTK_OBJECT(window), "delete_event", (GtkSignalFunc) gtk_main_quit, NULL);
  create_widget_set(context);

  gtk_widget_show(window);

  context = c;
  /*edit_timeout_id = gtk_timeout_add(50, edit_timeout, context);*/
}

void
GUI_uninit()
{
  GdomeException exc = 0;

  if (first_selected != NULL)
    {
      gdome_el_unref(first_selected, &exc);
      g_assert(exc == 0);
      first_selected = NULL;
    }

  if (root_selected != NULL)
    {
      gdome_el_unref(root_selected, &exc);
      g_assert(exc == 0);
      root_selected = NULL;
    }

  context = NULL;
}

int
GUI_load_document(GdomeDocument* doc)
{
  GtkMathView* math_view;

  g_return_val_if_fail(doc != NULL, -1);
  g_return_val_if_fail(main_area != NULL, -1);
  g_return_val_if_fail(GTK_IS_MATH_VIEW(main_area), -1);

  math_view = GTK_MATH_VIEW(main_area);

  if (!gtk_math_view_load_doc(math_view, doc)) return -1;

  return 0;
}

void
GUI_freeze()
{
  gtk_math_view_freeze(GTK_MATH_VIEW(main_area));
}

void
GUI_thaw()
{
  gtk_math_view_thaw(GTK_MATH_VIEW(main_area));
}

void
GUI_unload_document()
{
  GtkMathView* math_view;

  g_return_if_fail(main_area != NULL);
  g_return_if_fail(GTK_IS_MATH_VIEW(main_area));

  math_view = GTK_MATH_VIEW(main_area);

  gtk_math_view_unload(math_view);

  if (doc_name != NULL) g_free(doc_name);
  doc_name = NULL;
}

void
GUI_run()
{
  gtk_main();
}

#if 0
void
GUI_set_font_manager(FontManagerId id)
{
  gboolean t1;
  GtkMathView* math_view;

  g_return_if_fail(id != FONT_MANAGER_UNKNOWN);
  g_return_if_fail(main_area != NULL);
  g_return_if_fail(GTK_IS_MATH_VIEW(main_area));

  t1 = id == FONT_MANAGER_T1;

  math_view = GTK_MATH_VIEW(main_area);

  gtk_math_view_freeze(math_view);

  if (id != gtk_math_view_get_font_manager_type(math_view))
    gtk_math_view_set_font_manager_type(math_view, id);

  gtk_widget_set_sensitive(anti_aliasing(math_view, GTK_CHECK_MENU_ITEM(anti_aliasing_item)->active);
      gtk_math_view_set_transparency(math_view, GTK_CHECK_MENU_ITEM(transparency_item)->active);
    }

  gtk_math_view_thaw(math_view);
}
#endif

static void
store_filename(GtkFileSelection* selector, GtkWidget* user_data)
{
  gchar* selected_filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION(user_data));
  if (selected_filename != NULL)
    GUI_load_document(selected_filename);
}

static void
file_close(GtkWidget* widget, gpointer data)
{
  GUI_unload_document();
}

static void
file_re_open(GtkWidget* widget, gpointer data)
{
  if (doc_name != NULL) {
    GUI_load_document(doc_name);
  }
}

static void
file_open(GtkWidget* widget, gpointer data)
{
  GtkWidget* fs = gtk_file_selection_new("Open File");

  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION(fs)->ok_button),
		      "clicked", GTK_SIGNAL_FUNC (store_filename), (gpointer) fs);
                             
  /* Ensure that the dialog box is destroyed when the user clicks a button. */
     
  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION(fs)->ok_button),
			     "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     (gpointer) fs);

  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION(fs)->cancel_button),
			     "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     (gpointer) fs);
     
  /* Display that dialog */
     
  gtk_widget_show (fs);
}

static void
file_output_tex(GtkWidget* widget, gpointer data)
{
  g_assert(context != NULL);
  edit_output_tex(context);
}

#if 0
static void
options_font_manager(GtkWidget* widget, FontManagerId id)
{
  g_return_if_fail(id != FONT_MANAGER_UNKNOWN);
  GUI_set_font_manager(id);
}
#endif

static void
options_verbosity(GtkWidget* widget, guint level)
{
  gtk_math_view_set_log_verbosity(GTK_MATH_VIEW(main_area), level);
}

static void
edit_delete_selection(GtkWidget* widget, gpointer data)
{
  if (root_selected != NULL)
    {
      GdomeException exc;
      gtk_math_view_freeze(GTK_MATH_VIEW(main_area));
      printf("about to remove element %p\n", root_selected);
      delete_element(root_selected);
      gdome_el_unref(root_selected, &exc);
      g_assert(exc == 0);
      root_selected = NULL;
      gtk_math_view_thaw(GTK_MATH_VIEW(main_area));
    }
}

static void
edit_select_parent(GtkWidget* widget, gpointer data)
{
  if (root_selected != NULL)
    {
      GdomeException exc = 0;
      GdomeElement* parent = gdome_n_parentNode(root_selected, &exc);
      g_assert(exc == 0);
      gdome_el_unref(root_selected, &exc);
      g_assert(exc == 0);
      root_selected = parent;
      /* gtk_math_view_set_selection(GTK_MATH_VIEW(main_area), root_selected); */
    }
}

static void
edit_reset_selection(GtkWidget* widget, gpointer data)
{
  if (root_selected != NULL)
    {
      GdomeException exc = 0;
      /* gtk_math_view_reset_selection(GTK_MATH_VIEW(main_area), root_selected); */
      gdome_el_unref(root_selected, &exc);
      g_assert(exc == 0);
      root_selected = NULL;
    }
}

static void
edit_reset(GtkWidget* widget, gpointer data)
{
  g_assert(context != NULL);
  edit_reset_tex(context);
}

static void
insert_tex(GtkWidget* widget, GtkEntry* entry)
{
  gchar* text;
  g_return_if_fail(entry != NULL);

  text = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, -1);
  edit_push_string(context, text);
  g_free(text);
}

static void
edit_insert(GtkWidget* widget, gpointer data)
{
  GtkWidget* dialog;
  GtkWidget* entry;
  GtkWidget* ok;
  GtkWidget* cancel;

  dialog = gtk_dialog_new();
  entry = gtk_entry_new();
  ok = gtk_button_new_with_label("OK");
  cancel = gtk_button_new_with_label("Cancel");

  gtk_signal_connect (GTK_OBJECT (ok), "clicked",
		      GTK_SIGNAL_FUNC (insert_tex), (gpointer) entry);

  gtk_signal_connect_object (GTK_OBJECT (ok), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy), (gpointer) dialog);

  gtk_signal_connect_object (GTK_OBJECT (ok), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy), (gpointer) dialog);

  gtk_signal_connect_object (GTK_OBJECT (cancel), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy), (gpointer) dialog);

  gtk_container_set_border_width (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), 5);

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), entry);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area), ok);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area), cancel);

  gtk_widget_show_all (dialog);
}

static void
help_about(GtkWidget* widget, gpointer data)
{
  GtkWidget* dialog;
  GtkWidget* label;
  GtkWidget* ok;

  dialog = gtk_dialog_new();
  label = gtk_label_new("\n    MathML Editor    \n    Copyright (C) 2003 Luca Padovani    \n");
  ok = gtk_button_new_with_label("Close");

  gtk_signal_connect_object (GTK_OBJECT (ok), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy), (gpointer) dialog);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
		     ok);

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), label);

  gtk_widget_show_all (dialog);
}

static void
change_default_font_size(GtkWidget* widget, GtkSpinButton* spin)
{
  g_return_if_fail(spin != NULL);
  gtk_math_view_set_font_size( GTK_MATH_VIEW(main_area), gtk_spin_button_get_value_as_int(spin));
}

static void
options_change_font_size(GtkWidget* widget, gboolean larger)
{
  gfloat size = gtk_math_view_get_font_size (GTK_MATH_VIEW(main_area));
  if (larger) size = size / 0.71;
  else size = size * 0.71;
  if (size < 1) size = 1;
  gtk_math_view_set_font_size (GTK_MATH_VIEW(main_area), (gint) size + 0.5);
}

static void
options_set_font_size(GtkWidget* widget, gpointer data)
{
  GtkWidget* dialog;
  GtkWidget* label;
  GtkWidget* ok;
  GtkWidget* cancel;
  GtkWidget* spin;
  GtkObject* adj;

  dialog = gtk_dialog_new();
  label = gtk_label_new("Default font size:");
  ok = gtk_button_new_with_label("OK");
  cancel = gtk_button_new_with_label("Cancel");

  adj = gtk_adjustment_new (gtk_math_view_get_font_size (GTK_MATH_VIEW(main_area)), 1, 200, 1, 1, 1);
  spin = gtk_spin_button_new (GTK_ADJUSTMENT(adj), 1, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);

  gtk_signal_connect (GTK_OBJECT (ok), "clicked",
		      GTK_SIGNAL_FUNC (change_default_font_size), (gpointer) spin);

  gtk_signal_connect_object (GTK_OBJECT (ok), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy), (gpointer) dialog);

  gtk_signal_connect_object (GTK_OBJECT (ok), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy), (gpointer) dialog);

  gtk_signal_connect_object (GTK_OBJECT (cancel), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy), (gpointer) dialog);

  gtk_container_set_border_width (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), 5);

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area), ok);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area), cancel);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), label);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), spin);

  gtk_widget_show_all (dialog);
}

static void
select_begin(GtkMathView* math_view, GdomeElement* first, gint state)
{
  GdomeException exc = 0;

  g_return_if_fail(math_view != NULL);
  g_return_if_fail(GTK_IS_MATH_VIEW(math_view));
  g_return_if_fail(first != NULL);

  gtk_math_view_freeze(math_view);

  if (root_selected != NULL)
    {
      gtk_math_view_unselect(math_view, root_selected);
      gdome_el_unref(root_selected, &exc);
      g_assert(exc == 0);
    }

  root_selected = first_selected = find_element_with_ref(first);

  if (root_selected != NULL)
    {
      gtk_math_view_select(math_view, root_selected);
      gdome_el_ref(root_selected, &exc);
      g_assert(exc == 0);
    }

  gtk_math_view_thaw(math_view);
}

static void
select_over(GtkMathView* math_view, GdomeElement* elem, gint state)
{
  GdomeElement* new_selected = NULL;
  GdomeException exc = 0;

  g_return_if_fail(math_view != NULL);
  g_return_if_fail(GTK_IS_MATH_VIEW(math_view));
  g_return_if_fail(elem != NULL);

  if (first_selected == NULL || elem == NULL)
    new_selected = NULL;
  else
    new_selected = find_common_ancestor_with_ref(first_selected, elem);

  if (new_selected != root_selected)
    {
      gtk_math_view_freeze(math_view);
      if (root_selected != NULL)
	{
	  gtk_math_view_unselect(math_view, root_selected);
	  gdome_el_unref(root_selected, &exc);
	  g_assert(exc == 0);
	}
      root_selected = new_selected;
      if (root_selected != NULL)
	gtk_math_view_select(math_view, root_selected);
      gtk_math_view_thaw(math_view);
    }
  else if (new_selected != NULL)
    {
      gdome_el_unref(new_selected, &exc);
      g_assert(exc == 0);
    }

}

static gboolean
key_press_event(gpointer c,
		GdkEventKey* event,
		GtkWidget* widget)
{
  g_return_val_if_fail(widget != NULL, FALSE);
  g_return_val_if_fail(event != NULL, FALSE);
  g_return_val_if_fail(context != NULL, FALSE);

  if (event->type != GDK_KEY_PRESS) return FALSE;

  switch (event->keyval)
    {
    case GDK_BackSpace:
      edit_drop(context, event->state & GDK_MOD1_MASK, event->state & GDK_CONTROL_MASK);
      break;
    case GDK_Tab:
      edit_complete(context);
      break;
    default:
      if ((event->state & (GDK_CONTROL_MASK | GDK_MOD1_MASK)) == 0 && event->keyval < 0x80)
	edit_push_char(context, event->keyval);
      return FALSE;
    }

  return TRUE;
}

static void
create_widget_set(gpointer context)
{
  GtkWidget* main_vbox;
  GtkWidget* menu_bar;

  main_vbox = gtk_vbox_new(FALSE, 1);
  gtk_container_border_width(GTK_CONTAINER(main_vbox), 1);
  gtk_container_add(GTK_CONTAINER(window), main_vbox);
  gtk_widget_show(main_vbox);

  menu_bar = get_main_menu();
  gtk_box_pack_start(GTK_BOX(main_vbox), menu_bar, FALSE, TRUE, 0);
  gtk_widget_show(menu_bar);

  main_area = gtk_math_view_new(NULL, NULL);
  gtk_widget_show(main_area);

  //gtk_math_view_set_log_verbosity(GTK_MATH_VIEW(main_area), 3);

  gtk_signal_connect_object (GTK_OBJECT (main_area),
			     "select_begin", GTK_SIGNAL_FUNC (select_begin),
			     (gpointer) main_area);

  gtk_signal_connect_object (GTK_OBJECT (main_area),
			     "select_over", GTK_SIGNAL_FUNC (select_over),
			     (gpointer) main_area);

  gtk_signal_connect_object (GTK_OBJECT(window),
			     "key_press_event", GTK_SIGNAL_FUNC(key_press_event),
			     context);

  gtk_widget_add_events(GTK_WIDGET(main_area), GDK_KEY_PRESS_MASK);

  scrolled_area = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_area),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);

  gtk_widget_show(scrolled_area);
  gtk_container_add(GTK_CONTAINER(scrolled_area), main_area);
  gtk_box_pack_start(GTK_BOX(main_vbox), scrolled_area, TRUE, TRUE, 0);

  gtk_widget_show(main_vbox);
}

GtkWidget*
get_main_menu()
{
  GtkItemFactory* item_factory;
  GtkAccelGroup* accel_group;
  GtkWidget* menu_item;

  gint nmenu_items = sizeof(menu_items) / sizeof(menu_items[0]);

  accel_group = gtk_accel_group_new();

  item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>", accel_group);

  gtk_item_factory_create_items(item_factory, nmenu_items, menu_items, NULL);

  gtk_window_add_accel_group(GTK_WINDOW(window), accel_group);

  return gtk_item_factory_get_widget(item_factory, "<main>");
}
