/*============================================================================
 *  Définitions des fonctions de base
 *  associées à la structure `ecs_maillage_t' décrivant un maillage.
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2009 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilité
 *============================================================================*/

#include "ecs_config.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <assert.h>

#include <bft_error.h>
#include <bft_file.h>
#include <bft_mem.h>
#include <bft_printf.h>
#include <bft_timer.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' système
 *----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_chaine_glob.h"
#include "ecs_def.h"
#include "ecs_fic.h"
#include "ecs_tab.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage global "Post-Traitement"
 *----------------------------------------------------------------------------*/

#include "ecs_post.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_entmail.h"
#include "ecs_entmail_pcp.h"
#include "ecs_entmail_post.h"
#include "ecs_famille_chaine.h"
#include "ecs_param_perio_glob.h"
#include "ecs_param_rc_glob.h"
#include "ecs_post.h"
#include "ecs_select_fac_glob.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_maillage_post.h"


/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associé au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_maillage.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' privés   du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_maillage_priv.h"



/*============================================================================
 *                              Fonctions privées
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Simplification d'un maillage extrait correspondant à un cas de
 *  coupe de faces : on ne conserve du maillage que l'entité des
 *  faces et de cette entité que les champs filiation et type,
 *  suffisants pour réaliser le post-traitement des variables
 *  centrées sur les faces.
 *----------------------------------------------------------------------------*/

static void
ecs_loc_maillage__epure_coupe(ecs_maillage_t    *maillage_coupe,
                              const ecs_int_t    nbr_dump,
                              const char        *nom_fic_dump)
{

  ecs_int_t      ient;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage_coupe != NULL);


  /* Libération du contenu des entités */
  /*-----------------------------------*/

  for (ient = ECS_ENTMAIL_DEB; ient < ECS_ENTMAIL_FIN; ient++) {

    if (ient != ECS_ENTMAIL_FAC && maillage_coupe->entmail[ient] != NULL)

      maillage_coupe->entmail[ient]
        = ecs_entmail__detruit(maillage_coupe->entmail[ient]);

  }

  ecs_entmail_pcp__epure_coupe(maillage_coupe->entmail[ECS_ENTMAIL_FAC]);

  /* Liste chaînée des familles */
  /*----------------------------*/

  for (ient = ECS_ENTMAIL_DEB; ient < ECS_ENTMAIL_FIN; ient++) {

    if (maillage_coupe->famille[ient] != NULL)

      ecs_famille_chaine__detruit(&maillage_coupe->famille[ient]);

  }


  if (nbr_dump > 0)
    ecs_maillage__imprime(maillage_coupe,
                          nom_fic_dump,
                          nbr_dump,
                          "Reduced clip mesh");

}


/*----------------------------------------------------------------------------
 *    Fonction d'affichage des infos d'une périodicité
 *----------------------------------------------------------------------------*/

static void
ecs_loc_maillage__print_perio(ecs_param_perio_t   param_perio)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  if (param_perio.type_perio == ECS_PERIO_TYPE_TRANS)
    bft_printf(_("\n  Periodicity type: pure translation\n"));
  else if (param_perio.type_perio == ECS_PERIO_TYPE_ROTA)
    bft_printf(_("\n  Periodicity type: translation + rotation\n"));
  else
    assert(   param_perio.type_perio == ECS_PERIO_TYPE_TRANS
           || param_perio.type_perio == ECS_PERIO_TYPE_NUL);


  if (   param_perio.type_perio == ECS_PERIO_TYPE_TRANS
      || param_perio.type_perio == ECS_PERIO_TYPE_ROTA)

    bft_printf(_("\n    Translation parameters\n"
                 "      - translation vector:     % 10.5e % 10.5e % 10.5e\n\n"),
               param_perio.translation[0],
               param_perio.translation[1],
               param_perio.translation[2]);

  if (   param_perio.type_perio == ECS_PERIO_TYPE_ROTA
      && ECS_ABS(param_perio.angle) < 1.e-6)

    bft_printf(_("\n    Rotation parameters\n"
                 "      - invariant point:        % 10.5e % 10.5e % 10.5e\n"
                 "      - rotation matrix:        % 10.5e % 10.5e % 10.5e\n"
                 "                                % 10.5e % 10.5e % 10.5e\n"
                 "                                % 10.5e % 10.5e % 10.5e\n\n"),
               param_perio.point_inv[0],
               param_perio.point_inv[1],
               param_perio.point_inv[2],
               param_perio.matrice[0][0], param_perio.matrice[0][1],
               param_perio.matrice[0][2],
               param_perio.matrice[1][0], param_perio.matrice[1][1],
               param_perio.matrice[1][2],
               param_perio.matrice[2][0], param_perio.matrice[2][1],
               param_perio.matrice[2][2]);

  else if (   param_perio.type_perio == ECS_PERIO_TYPE_ROTA
           && ECS_ABS(param_perio.angle) > 1.e-6)

    bft_printf(_("    Rotation parameters\n"
                 "      - invariant point:        % 10.5e % 10.5e % 10.5e\n"
                 "      - direction vector:       % 10.5e % 10.5e % 10.5e\n"
                 "      - angle (in degrees):     % 10.5e\n\n"),
               param_perio.point_inv[0],
               param_perio.point_inv[1],
               param_perio.point_inv[2],
               param_perio.direction[0],
               param_perio.direction[1],
               param_perio.direction[2],
               param_perio.angle);


}


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *   Fonction qui créé une structure définissant un maillage
 *----------------------------------------------------------------------------*/

ecs_maillage_t *
ecs_maillage__cree(ecs_entmail_t  **vect_entmail)
{
  ecs_maillage_t  *maillage;

  ecs_int_t  ient;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Structure globale du maillage */
  /*-------------------------------*/

  /* Allocation de la structure globale du maillage */

  BFT_MALLOC(maillage, 1, ecs_maillage_t);


  /* Initialisation du type de connectivité par défaut */

  maillage->typ_connect = ECS_MAILLAGE_CONNECT_NUL;


  /* Entités de maillage */
  /*---------------------*/

  for (ient = ECS_ENTMAIL_DEB; ient < ECS_ENTMAIL_FIN; ient++) {

    maillage->entmail[ient] = vect_entmail[ient];

  }


  /* Familles */
  /*----------*/

  for (ient = ECS_ENTMAIL_DEB; ient < ECS_ENTMAIL_FIN; ient++) {

    maillage->famille[ient] = NULL;

  }


  return maillage;


}


/*----------------------------------------------------------------------------
 *  Fonction libérant une structure `ecs_maillage_t' donnée en argument.
 *  Elle renvoie un pointeur NULL
 *----------------------------------------------------------------------------*/

ecs_maillage_t *
ecs_maillage__detruit(ecs_maillage_t  *this_maillage)
{

  ecs_int_t      ient;        /* Indice de boucle sur les entités de maillage*/


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Structures des entités de maillage */
  /*====================================*/

  /* Libération du contenu des entités */
  /*-----------------------------------*/

  for (ient = ECS_ENTMAIL_DEB; ient < ECS_ENTMAIL_FIN; ient++) {

    if (this_maillage->entmail[ient] != NULL) {

      this_maillage->entmail[ient]
        = ecs_entmail__detruit(this_maillage->entmail[ient]);

    }

  }


  /* Liste chaînée des familles */
  /*============================*/

  for (ient = ECS_ENTMAIL_DEB; ient < ECS_ENTMAIL_FIN; ient++) {

    if (this_maillage->famille[ient] != NULL) {

      ecs_famille_chaine__detruit(&this_maillage->famille[ient]);

    }

  }

  /* Structure globale du maillage */
  /*==============================*/

  BFT_FREE(this_maillage);


  return this_maillage;


}


/*----------------------------------------------------------------------------
 *  Fonction imprimant le contenu d'une structure `ecs_maillage_t' donnée
 *   dans le fichier de nom donné
 *   précédé par le `titre'
 *----------------------------------------------------------------------------*/

void
ecs_maillage__imprime(const ecs_maillage_t  *this_maillage,
                      const char            *nom_fichier_dump,
                      const ecs_int_t        nbr_imp,
                      const char            *titre)
{

  const char *nom_typ_c[3] = {
    "ECS_MAILLAGE_CONNECT_NUL",
    "ECS_MAILLAGE_CONNECT_NODALE",
    "ECS_MAILLAGE_CONNECT_DESCENDANTE"
  };


  const char *nom_ent_c[ECS_ENTMAIL_FIN] = {
    "ENTMAIL_SOM",
    "ENTMAIL_FAC",
    "ENTMAIL_CEL"
  };


  const char *nom_fam_c[ECS_ENTMAIL_FIN] = {
    "FAMILLE_SOM",
    "FAMILLE_FAC",
    "FAMILLE_CEL"
  };


  bft_file_t  *fic_imp;       /* Descripteur du fichier d'impression         */

  ecs_int_t    imp_col;

  ecs_int_t    ient;          /* Indice de boucle sur les entités de maillage*/


#define ECS_FCT_IMP_MAILLAGE_FAMILLE       "famille"


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_maillage != NULL);


  imp_col = 0;


  /* Ouverture du fichier d'impression */
  /*===================================*/

  fic_imp = bft_file_open(nom_fichier_dump,
                          BFT_FILE_MODE_APPEND, BFT_FILE_TYPE_TEXT);


  /* Message sur la sortie standard */
  /*================================*/

  bft_printf("\n\n%s %s\n%s\n\n",
             _("Dump mesh structure to file"),
             nom_fichier_dump, titre);


  /* Écriture du titre */
  /*===================*/

  bft_file_printf(fic_imp, "\n\n%s\n\n", titre );


  /* Impression des membres scalaires de la structure */
  /*==================================================*/

  ecs_fic__imprime_val(fic_imp, imp_col, "typ_connect",
                       ECS_TYPE_char, nom_typ_c[this_maillage->typ_connect]);


  /* Impressions des entités de maillage */
  /*=====================================*/


  for (ient = ECS_ENTMAIL_DEB; ient < ECS_ENTMAIL_FIN; ient++) {

    /* Impression du pointeur sur une entité principale */

    ecs_fic__imprime_ptr(fic_imp, imp_col,
                         nom_ent_c[ient],
                         (void *)this_maillage->entmail[ient]);


    if (this_maillage->entmail[ient] != NULL) {

      /* Impression du contenu d'une entité principale */

      ecs_entmail__imprime(this_maillage->entmail[ient],
                           imp_col + 1,
                           nbr_imp,
                           fic_imp);

    }

  }


  /* Impression des familles */
  /*=========================*/

  for (ient = ECS_ENTMAIL_DEB; ient < ECS_ENTMAIL_FIN; ient++) {

    /* Impression du pointeur sur une entité principale */

    ecs_fic__imprime_ptr(fic_imp, imp_col,
                         nom_fam_c[ient],
                         (void *)this_maillage->famille[ient]);


    if (this_maillage->famille[ient] != NULL) {

      ecs_famille_chaine__imprime(this_maillage->famille[ient],
                                  imp_col + 1,
                                  fic_imp);

    }

  }


  /* Fermeture du fichier d'impression */
  /*===================================*/

  bft_file_free(fic_imp);


}


/*----------------------------------------------------------------------------
 *  Fonction qui retourne le type d'entité de plus grande dimension
 *   contenue dans une structure `ecs_maillage_t'
 *----------------------------------------------------------------------------*/


ECS_ENTMAIL_E
ecs_maillage__ret_entmail_max(const ecs_maillage_t  *this_maillage)
{
  ECS_ENTMAIL_E entmail_max = ECS_ENTMAIL_DEB;
  ecs_int_t ient;

  for (ient = ECS_ENTMAIL_DEB; ient < ECS_ENTMAIL_FIN; ient++) {
    if (this_maillage->entmail[ient] != NULL) {
      if (ecs_entmail__ret_nbr_ele(this_maillage->entmail[ient]) > 0)
        entmail_max = (ECS_ENTMAIL_E) ient;
    }
  }

  return entmail_max;

}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie la taille en octets d'une structure `ecs_maillage_t'.
 *----------------------------------------------------------------------------*/

float
ecs_maillage__ret_taille(const ecs_maillage_t  *this_maillage)
{

  float        taille;

  ecs_int_t    ient;          /* Indice de boucle sur les entités de maillage*/


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_maillage != NULL);


  taille = sizeof(*this_maillage);


  for (ient = ECS_ENTMAIL_DEB; ient < ECS_ENTMAIL_FIN; ient++) {

    if (this_maillage->entmail[ient] != NULL) {

      taille += ecs_entmail__ret_taille(this_maillage->entmail[ient]);

    }

  }


  for (ient = ECS_ENTMAIL_DEB; ient < ECS_ENTMAIL_FIN; ient++) {

    if (this_maillage->famille[ient] != NULL) {

      taille += ecs_famille_chaine__ret_taille(this_maillage->famille[ient]);

    }

  }


  return taille;


}


/*----------------------------------------------------------------------------
 *  Fonction qui créé une structure maillage en connectivité nodale
 *   (les connectivités étant liées par les étiquettes)
 *   à partir du vecteur des entités de maillage
 *----------------------------------------------------------------------------*/

ecs_maillage_t *
ecs_maillage__cree_nodal(ecs_entmail_t  **vect_entmail)
{

  ecs_maillage_t  * maillage;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Création de la structure du maillage */
  /*======================================*/

  maillage = ecs_maillage__cree(vect_entmail);


  maillage->typ_connect = ECS_MAILLAGE_CONNECT_NODALE;


  /* Suppression des sommets ne participant pas à la connectivité */
  /* et fusion des éléments surfaciques confondus éventuels       */
  /*==============================================================*/

  ecs_maillage__nettoie_nodal(maillage);


  return maillage;

}


/*----------------------------------------------------------------------------
 *  Fusion des sommets confondus et suppression des éléments dégénérés
 *----------------------------------------------------------------------------*/

void
ecs_maillage__nettoie_descend(ecs_maillage_t  *this_maillage)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Suppression des sommets inutiles */

  ecs_entmail_pcp__nettoie_descend(this_maillage->entmail);


}


/*----------------------------------------------------------------------------
 *  Suppression des sommets ne participant pas à la connectivité
 *   et fusion des éléments surfaciques confondus éventuels
 *----------------------------------------------------------------------------*/

void
ecs_maillage__nettoie_nodal(ecs_maillage_t  *this_maillage)
{

  ecs_entmail_t * liste_entmail_connect[1];


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Suppression des sommets inutiles */

  ecs_entmail_pcp__nettoie_nodal(this_maillage->entmail);


  /*
    Fusion d'éléments surfaciques confondus éventuels (issus par exemple,
    de l'utilisation conjointe de couleurs de faces et faces de bord
    sous Simail, double `surface coating' sous I-DEAS, ...)
  */

  liste_entmail_connect[0] = NULL;

  if (this_maillage->entmail[ECS_ENTMAIL_CEL] != NULL) {

    if (this_maillage->entmail[ECS_ENTMAIL_FAC] != NULL)
      ecs_entmail_pcp__fusionne(this_maillage->entmail[ECS_ENTMAIL_FAC],
                                liste_entmail_connect);

  }

}


/*----------------------------------------------------------------------------
 *  Correction si nécessaire de l'orientation des éléments en
 *   connectivité nodale.
 *----------------------------------------------------------------------------*/

void
ecs_maillage__orient_nodal(ecs_maillage_t    *this_maillage,
                           ecs_tab_int_t     *liste_cel_err,
                           ecs_tab_int_t     *liste_cel_cor,
                           bool               correc_orient)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  ecs_entmail_pcp__orient_nodal(this_maillage->entmail,
                                liste_cel_err,
                                liste_cel_cor,
                                correc_orient);

}


/*----------------------------------------------------------------------------
 *  Fonction qui assigne la tête de la liste chaînée des familles donnée
 *   à la structure de maillage donnée
 *----------------------------------------------------------------------------*/

void
ecs_maillage__definit_famille(ecs_maillage_t   *maillage,
                              ecs_famille_t   **vect_famille)
{

  ecs_int_t ifam_ent;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL);


  for (ifam_ent = ECS_FAMILLE_DEB; ifam_ent < ECS_FAMILLE_FIN; ifam_ent++) {

    maillage->famille[ifam_ent] = vect_famille[ifam_ent];

  }


}


/*----------------------------------------------------------------------------
 *  Fonction réalisant, à partir d'une connectivité de maillage donnée,
 *   la connectivité descendante du maillage
 *----------------------------------------------------------------------------*/

void
ecs_maillage__connect_descend(ecs_maillage_t * maillage)
{
  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  ecs_int_t     ient;

  ecs_entmail_t  *entmail_cel;
  ecs_entmail_t  *entmail_fac_dec;

  ecs_entmail_t  *liste_entmail_connect[ECS_ENTMAIL_FIN];


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL);

  assert(maillage->typ_connect == ECS_MAILLAGE_CONNECT_NODALE);

  maillage->typ_connect = ECS_MAILLAGE_CONNECT_DESCENDANTE;

  entmail_cel = maillage->entmail[ECS_ENTMAIL_CEL];

  if (entmail_cel == NULL)
    return;

  liste_entmail_connect[0] = entmail_cel;
  for (ient = 1; ient < ECS_ENTMAIL_FIN; ient++) {
    liste_entmail_connect[ient] = NULL;
  }


  /* Décomposition des cellules en leurs faces */
  /*-------------------------------------------*/

  entmail_fac_dec = ecs_entmail_pcp__decompose_cel(entmail_cel);


  /* Concaténation, dans la même entité de maillage, des sous-éléments crées, */
  /*  avec les sous-éléments de même entité déjà existants (s'ils existent !) */
  /*--------------------------------------------------------------------------*/

  if (maillage->entmail[ECS_ENTMAIL_FAC] != NULL) {

    ecs_entmail_pcp__concatene(maillage->entmail[ECS_ENTMAIL_FAC],
                               entmail_fac_dec,
                               liste_entmail_connect);

    entmail_fac_dec = ecs_entmail__detruit(entmail_fac_dec);

  }
  else {

    maillage->entmail[ECS_ENTMAIL_FAC] = entmail_fac_dec;

  }


  /* Fusion des sous-éléments ayant la même définition topologique */
  /*---------------------------------------------------------------*/

  ecs_entmail_pcp__fusionne(maillage->entmail[ECS_ENTMAIL_FAC],
                            liste_entmail_connect);

}


/*----------------------------------------------------------------------------
 *  Fonction réalisant le tri des éléments suivant leur type géométrique
 *  La fonction affiche le nombre d'éléments par type géométrique
 *----------------------------------------------------------------------------*/

void
ecs_maillage__trie_typ_geo(ecs_maillage_t  *maillage)
{

  ecs_int_t      ient;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL);


  bft_printf("\n");


  for (ient = ECS_ENTMAIL_FAC; ient < ECS_ENTMAIL_FIN; ient++) {

    if (maillage->entmail[ient] != NULL) {

      ecs_entmail_pcp__trie_typ_geo(maillage->entmail[ient],
                                    ient);

    }

  }

  bft_printf("\n");

}


/*----------------------------------------------------------------------------
 *  Fonction qui définit un nouveau maillage
 *   par extraction d'une partie du maillage donné
 *
 *  Les éléments à extraire doivent être tous de même dimension :
 *  cellules ou faces ou arêtes ou sommets
 *
 *  On construit automatiquement une filiation.
 *----------------------------------------------------------------------------*/

ecs_maillage_t *
ecs_maillage__extrait(ecs_maillage_t       *maillage,
                      ECS_ENTMAIL_E         entmail_sel_e,
                      const ecs_tab_int_t  *liste_filtre,
                      const bool            herite_attributs)
{

  ecs_tab_bool_t   bool_elt_select;

  ecs_int_t        ient;

  ecs_entmail_t  **vect_entmail;

  ecs_maillage_t  *this_maillage_new;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL);

  assert(   maillage->typ_connect == ECS_MAILLAGE_CONNECT_NODALE
         || entmail_sel_e == ECS_ENTMAIL_FAC);


  /* Construction de la liste des éléments du maillage d'origine à extraire */
  /*------------------------------------------------------------------------*/

  bool_elt_select = ecs_entmail_pcp__selectionne_lst(maillage->entmail,
                                                     entmail_sel_e,
                                                     liste_filtre);


  /* Extraction des éléments sélectionnés du maillage d'origine */
  /*  pour l'ensemble des entités de maillage                   */
  /*------------------------------------------------------------*/

  vect_entmail = ecs_entmail_pcp__extrait(maillage->entmail,
                                          entmail_sel_e,
                                          bool_elt_select,
                                          herite_attributs);


  if (bool_elt_select.val != NULL)
    BFT_FREE(bool_elt_select.val);


  /* Création de la structure "maillage" */
  /*-------------------------------------*/

  this_maillage_new = ecs_maillage__cree(vect_entmail);

  BFT_FREE(vect_entmail);


  /*--------------------------------------------------------------------------*/
  /* 4ème étape                                                               */
  /*--------------------------------------------------------------------------*/
  /*                                                                          */
  /* Recopie des descripteurs de familles                                     */
  /*                                                                          */
  /*--------------------------------------------------------------------------*/

  if (herite_attributs == true) {

    for (ient = ECS_ENTMAIL_FAC; ient < ECS_ENTMAIL_FIN; ient++) {

      if (   this_maillage_new->entmail[ient] != NULL
          && maillage->famille[ient] != NULL) {

        this_maillage_new->famille[ient]
          = ecs_famille_chaine__copie(maillage->famille[ient]);
      }

    }

  }

  return this_maillage_new;
}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie une liste d'entités portant des familles
 *   non vides dans un maillage donné
 *
 *  La partie sélectionnée correspond aux éléments :
 *  -       ayant        une des couleurs données dans la liste des couleurs
 *  - et/ou appartenant à un des groupes  donnés  dans la liste des groupes
 *
 *  La sélection peut être inversée (en restant dans les entités de
 *  famille non vides)
 *----------------------------------------------------------------------------*/

ecs_tab_int_t
ecs_maillage__selectionne_fam(ecs_maillage_t        *maillage,
                              ECS_ENTMAIL_E          entmail_sel_e,
                              const ecs_tab_int_t   *liste_filtre,
                              const ecs_tab_int_t    liste_couleur,
                              const ecs_tab_char_t   liste_groupe,
                              const bool             inv_selection)
{

  ecs_tab_bool_t  bool_fam_select;

  size_t          ind;

  ecs_tab_int_t   liste_elt_select;

  const bool      inv_bool[2] = {true, false};

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL);


  if (liste_couleur.nbr > 0 || liste_groupe.nbr > 0)

    bft_printf(_("  Selection of elements\n"
                 "   by their color or group membership\n"
                 "  -----------------------------------\n\n"));


  liste_elt_select.nbr = 0;
  liste_elt_select.val = NULL;


  /* Construction de la liste des familles à sélectionner */
  /*------------------------------------------------------*/

  if (maillage->famille[entmail_sel_e] == NULL)
    return liste_elt_select;


  bool_fam_select
    = ecs_famille_chaine__indic_fam_att(maillage->famille[entmail_sel_e],
                                        liste_couleur,
                                        liste_groupe);

  if (bool_fam_select.nbr < 2) {
    BFT_FREE(bool_fam_select.val);
    return liste_elt_select;
  }

  if (liste_couleur.nbr == 0 && liste_groupe.nbr == 0) {
    for (ind = 1; ind < bool_fam_select.nbr; ind++)
      bool_fam_select.val[ind] = true;
  }

  bool_fam_select.val[0] = false;

  if (inv_selection == true) {

    for (ind = 1; ind < bool_fam_select.nbr; ind++)
      bool_fam_select.val[ind] = inv_bool[bool_fam_select.val[ind]];

  }


  /* Construction de la liste des éléments du maillage à sélectionner */
  /*------------------------------------------------------------------*/

  liste_elt_select = ecs_entmail_pcp__liste_ent_fam(maillage->entmail,
                                                    entmail_sel_e,
                                                    liste_filtre,
                                                    &bool_fam_select);


  bft_printf(_("  Number of selected elements: %d\n"),
             liste_elt_select.nbr);


  /* Libération mémoire et retour */
  /*------------------------------*/

  bool_fam_select.nbr = 0;
  BFT_FREE(bool_fam_select.val);

  return liste_elt_select;

}


/*----------------------------------------------------------------------------
 *  Fonction qui concatène dans un maillage récepteur donné,
 *   un maillage à concaténer donné
 *----------------------------------------------------------------------------*/

void
ecs_maillage__concatene_nodal(ecs_maillage_t  *maillage_recept,
                              ecs_maillage_t  *maillage_concat)
{

  ecs_int_t        ient;
  ecs_int_t        ient_connect;
  ecs_int_t        iliste;

  ecs_entmail_t   *liste_entmail_connect[2 * (ECS_ENTMAIL_FIN - 1)];


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage_recept != NULL);
  assert(maillage_concat != NULL);

  assert(maillage_recept->typ_connect == ECS_MAILLAGE_CONNECT_NODALE);
  assert(maillage_concat->typ_connect == ECS_MAILLAGE_CONNECT_NODALE);


  /*--------------------------------------------------------------------------*/
  /* 1ère étape                                                               */
  /*--------------------------------------------------------------------------*/
  /*                                                                          */
  /* Concaténation et fusion des éléments provenant des différentes           */
  /*  entités de maillage extraites                                           */
  /*                                                                          */
  /*--------------------------------------------------------------------------*/


  for (ient = ECS_ENTMAIL_DEB; ient < ECS_ENTMAIL_FIN; ient++) {


    if (maillage_concat->entmail[ient] != NULL) {


      if (maillage_recept->entmail[ient] != NULL) {

        /* - en connectivité descendante,                                   */
        /*      l'entité de maillage `ient' ne participe qu'a la définition */
        /*   de l'entité de maillage `ient + 1'                             */
        /* - en connectivité nodale,                                        */
        /*   seule l'entité de maillage sur les sommets participe           */
        /*    à la définition de toutes les autres entités de maillage      */


        for (ient_connect = ECS_ENTMAIL_DEB;
             ient_connect < 2 * (ECS_ENTMAIL_FIN - 1); ient_connect++)

          liste_entmail_connect[ient_connect] = NULL;


        iliste = 0;

        if ((ECS_ENTMAIL_E)ient == ECS_ENTMAIL_SOM) {

          for (ient_connect = ECS_ENTMAIL_FAC;
               ient_connect < ECS_ENTMAIL_FIN; ient_connect++ ) {

            if (maillage_concat->entmail[ient_connect] != NULL) {

              liste_entmail_connect[iliste++]
                = maillage_concat->entmail[ient_connect];

            }

          }

        } /* Fin : si l'entité de maillage concerne les sommets */


        ecs_entmail_pcp__concatene(maillage_recept->entmail[ient],
                                   maillage_concat->entmail[ient],
                                   liste_entmail_connect);

        maillage_concat->entmail[ient]
          = ecs_entmail__detruit(maillage_concat->entmail[ient]);

      }
      else {

        maillage_recept->entmail[ient]
          = maillage_concat->entmail[ient];

        maillage_concat->entmail[ient] = NULL;

      }


    }  /* else : rien à faire */


  }  /* Fin : boucle sur `ient' */


  ecs_maillage__detruit(maillage_concat);

}


/*----------------------------------------------------------------------------
 *  Création et initialisation d'un maillage extrait correspondant à un cas de
 *  coupe de faces : on extrait un maillage correspondant à la zone
 *  sélectionnée par liste, puis on découpe éventuellement les faces de plus
 *  de 4 arêtes de ce maillage en triangles; finalement, on l'ordonne
 *  selon le numéro de type de face.
 *
 *  On remplit un tableau de pointeurs sur les structures maillage associées
 *  à chaque niveau de découpage de la coupe.
 *----------------------------------------------------------------------------*/

ecs_maillage_t *
ecs_maillage__cree_coupe(const char           *nom_coupe,
                         ecs_maillage_t       *maillage,
                         const ecs_tab_int_t   liste_fac,
                         const ecs_int_t       nbr_dump,
                         const char           *nom_fic_dump,
                         const bool            herite_attributs,
                         ecs_post_type_t       type_post,
                         ecs_post_t           *cas_post)
{

  ecs_maillage_t  * maillage_extrait;

  ecs_tab_int_t     liste_couleur;
  ecs_tab_char_t    liste_groupe;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  assert(maillage != NULL);

  /* Initialisations */
  /*-----------------*/

  liste_couleur.nbr = 0;
  liste_couleur.val = NULL;

  liste_groupe.nbr = 0;
  liste_groupe.val = NULL;

  maillage_extrait = NULL;

  /* Si aucun post-traitement, rien à faire */

  if (cas_post == NULL)
    return NULL;


  /* Extraction des faces sélectionnées */
  /*------------------------------------*/

  maillage_extrait = ecs_maillage__extrait(maillage,
                                           ECS_ENTMAIL_FAC,
                                           &liste_fac,
                                           herite_attributs);

  maillage_extrait->typ_connect = maillage->typ_connect;


  if (nbr_dump > 0)
    ecs_maillage__imprime(maillage_extrait,
                          nom_fic_dump,
                          nbr_dump,
                          nom_coupe);


  /* Tri selon le type des faces */
  /*-----------------------------*/

  bft_printf(_("\nCreating mesh for output: %s"
               "\n-------------------------\n\n"),
             nom_coupe);

  ecs_entmail_pcp__trie_typ_geo(maillage_extrait->entmail[ECS_ENTMAIL_FAC],
                                ECS_ENTMAIL_FAC);


  if (nbr_dump > 0)
    ecs_maillage__imprime(maillage_extrait,
                          nom_fic_dump,
                          nbr_dump,
                          "Extracted nodal mesh");


  /* Sortie de la géométrie pour post-traitement */

  ecs_maillage_post__ecr(nom_coupe,
                         maillage_extrait,
                         ECS_ENTMAIL_FAC,
                         type_post,
                         cas_post);

  /* Simplification du maillage extrait (gain mémoire) */

  ecs_loc_maillage__epure_coupe(maillage_extrait,
                                nbr_dump,
                                nom_fic_dump);


  /* Renvoie un pointeur sur le maillage extrait simplifié */

  return maillage_extrait;
}


/*----------------------------------------------------------------------------
 *  Fonction qui crée la liste des faces intérieures à un maillage.
 *  On ne compte pas ici les faces périodiques parmi les faces intérieures,
 *  cette fonction étant destinée à filtrer le post-triatment, et les
 *  faces périodiques étant déjà affichables par ailleurs.
 *----------------------------------------------------------------------------*/

ecs_tab_int_t
ecs_maillage__liste_fac_int(ecs_maillage_t  *maillage)
{
  return ecs_entmail_pcp__liste_fac_int(maillage->entmail);
}


/*----------------------------------------------------------------------------
 *  Fonction qui construit la liste des cellules attachées à une liste
 *  de faces fournie en argument.
 *----------------------------------------------------------------------------*/

ecs_tab_int_t
ecs_maillage__liste_cel_fac(ecs_maillage_t       *maillage,
                            const ecs_tab_int_t   liste_fac)
{

  return ecs_entmail_pcp__liste_cel_fac(maillage->entmail,
                                        liste_fac);

}


/*----------------------------------------------------------------------------
 *  Fonction qui calcule les coordonnées min et max du domaine
 *----------------------------------------------------------------------------*/

void
ecs_maillage__calc_coo_ext(ecs_maillage_t  *maillage)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL);


  assert(maillage->entmail != NULL);

  assert(maillage->entmail[ECS_ENTMAIL_SOM] != NULL);


  ecs_entmail_pcp__calc_coo_ext(maillage->entmail[ECS_ENTMAIL_SOM]);

}


/*----------------------------------------------------------------------------
 *  Fonction qui modifie les coordonnées du maillage
 *----------------------------------------------------------------------------*/

void
ecs_maillage__transf_coo(ecs_maillage_t  *maillage,
                         const double     matrice[3][4])
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL);


  assert(maillage->entmail != NULL);

  assert(maillage->entmail[ECS_ENTMAIL_SOM] != NULL);


  ecs_entmail_pcp__transf_coo(maillage->entmail[ECS_ENTMAIL_SOM],
                              matrice);

}


/*----------------------------------------------------------------------------
 *  Fonction qui recolle les faces non conformes
 *
 *  Les listes des faces nouvelles ou modifiées sont construites (et allouées)
 *  ici; les structures liste_fac_new et liste_fac_mod correspondantes sont
 *  donc vides en entrée; idem pour liste_fac_err qui indiquera les indices
 *  des faces pour lesquelles le découpage en sous-faces a échoué
 *
 *  On renvoie 1 si l'on a appelé effectivement le recollement, 0 sinon
 *  (i.e. si la sélection de faces à recoller est vide).
 *----------------------------------------------------------------------------*/

int
ecs_maillage__recolle(ecs_maillage_t          *maillage,
                      const ecs_select_fac_t   select_fac_rc,
                      ecs_tab_int_t           *liste_fac_new,
                      ecs_tab_int_t           *liste_fac_mod,
                      ecs_tab_int_t           *liste_fac_err,
                      const ecs_param_rc_t     param_rc)
{
  size_t            nbr_fac;
  ecs_tab_bool_t    bool_elt_select;

  ecs_tab_int_t     liste_fac_isolee;
  ecs_tab_int_t     liste_fac_de_bord;

  double            start_time[2], end_time[2];

  int               val_ret;

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL);

  start_time[0] = bft_timer_wtime();
  start_time[1] = bft_timer_cpu_time();

  /* Message d'information pour l'utilisateur */
  /*------------------------------------------*/

  if (param_rc.param_perio == NULL)

    bft_printf("\n\n%s", _("Joining of non-conforming faces\n"
                           "-------------------------------\n"));
  else {

    bft_printf(_("\n\nProcessing periodicity %1d\n"
                 "----------------------------\n"),
               (*(param_rc.param_perio)).num_perio);

    ecs_loc_maillage__print_perio(*param_rc.param_perio);

  }


  /* Construction de la liste des faces du maillage d'origine */
  /*  concernées par le recollement                           */
  /*----------------------------------------------------------*/

  ecs_entmail_pcp__compte_typ_fac(maillage->entmail,
                                  NULL,
                                  NULL,
                                  &liste_fac_de_bord,
                                  &liste_fac_isolee);

  nbr_fac = ecs_entmail__ret_nbr_ele(maillage->entmail[ECS_ENTMAIL_FAC]);

  /* Si l'on a des faces "libres", on veut les traiter aussi, sauf
     dans le cas de la périodicité */

  if (param_rc.param_perio != NULL) {
    liste_fac_isolee.nbr = 0;
    BFT_FREE(liste_fac_isolee.val);
  }

  if (liste_fac_de_bord.nbr > 0 && liste_fac_isolee.nbr > 0) {

    size_t ind_isol, ind_bord, ind_filtre;
    ecs_int_t  *liste_filtre;

    ind_bord = liste_fac_de_bord.nbr;

    BFT_MALLOC(liste_filtre,
               liste_fac_de_bord.nbr + liste_fac_isolee.nbr,
               ecs_int_t);

    ind_bord = 0;
    ind_isol = 0;
    ind_filtre = 0;

    while (   ind_bord < liste_fac_de_bord.nbr
           && ind_isol < liste_fac_isolee.nbr) {

      if (  liste_fac_de_bord.val[ind_bord]
          < liste_fac_isolee.val[ind_isol])
        liste_filtre[ind_filtre++] = liste_fac_de_bord.val[ind_bord++];
      else
        liste_filtre[ind_filtre++] = liste_fac_isolee.val[ind_isol++];

    }

    while (ind_bord < liste_fac_de_bord.nbr)
      liste_filtre[ind_filtre++] = liste_fac_de_bord.val[ind_bord++];

    while (ind_isol < liste_fac_isolee.nbr)
      liste_filtre[ind_filtre++] = liste_fac_isolee.val[ind_isol++];

    assert(ind_filtre == liste_fac_de_bord.nbr + liste_fac_isolee.nbr);

    BFT_FREE(liste_fac_de_bord.val);
    BFT_FREE(liste_fac_isolee.val);

    liste_fac_de_bord.val = liste_filtre;

    liste_fac_de_bord.nbr += liste_fac_isolee.nbr;
    liste_fac_isolee.nbr = 0;

  }

  else if (liste_fac_de_bord.nbr == 0 && liste_fac_isolee.nbr > 0) {

    liste_fac_de_bord.nbr = liste_fac_isolee.nbr;
    liste_fac_de_bord.val = liste_fac_isolee.val;

    liste_fac_isolee.nbr = 0;
    liste_fac_isolee.val = NULL;

  }

  bool_elt_select
    = ecs_entmail_pcp__selectionne(maillage->entmail,
                                   ECS_ENTMAIL_FAC,
                                   &liste_fac_de_bord,
                                   select_fac_rc.liste_couleur_fac,
                                   select_fac_rc.liste_groupe_fac,
                                   select_fac_rc.inv_selection);

  BFT_FREE(liste_fac_de_bord.val);

  /* Recollement des faces concernées */
  /*----------------------------------*/

  val_ret = ecs_entmail_pcp__recolle(maillage->entmail,
                                     NULL,
                                     &bool_elt_select,
                                     liste_fac_new,
                                     liste_fac_mod,
                                     liste_fac_err,
                                     param_rc);


  if (bool_elt_select.val != NULL)
    BFT_FREE(bool_elt_select.val);

  end_time[0] = bft_timer_wtime();
  end_time[1] = bft_timer_cpu_time();

  bft_printf(_("\n  Wall-clock time: %f s; CPU time: %f s\n\n"),
               (double)(end_time[0] - start_time[0]),
               (double)(end_time[1] - start_time[1]));

  return val_ret;
}


/*----------------------------------------------------------------------------
 *  Fonction qui recolle des faces non conformes dont la visibilité
 *  est connue (par exemple une filiation);
 *
 *  Les listes des faces nouvelles ou modifiées sont construites (et allouées)
 *  ici; les structures liste_fac_new et liste_fac_mod correspondantes sont
 *  donc vides en entrée; idem pour liste_fac_err qui indiquera les indices
 *  des faces pour lesquelles le découpage en sous-faces a échoué
 *
 *  On renvoie 1 si l'on a effectivement appelé le recollement, 0 sinon
 *  (i.e. si le champ de type "visibilité" est introuvable).
 *----------------------------------------------------------------------------*/

int
ecs_maillage__recolle_vis(ecs_maillage_t  *maillage,
                          ecs_tab_int_t   *liste_fac_new,
                          ecs_tab_int_t   *liste_fac_mod,
                          ecs_tab_int_t   *liste_fac_err)
{
  ecs_param_rc_t  param_rc;
  double          start_time[2], end_time[2];

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  param_rc.num_rc        = -1;
  param_rc.fraction_dist = 0.1;
  param_rc.epsilon_plan  = 0.8;
  param_rc.param_perio   = NULL;
  param_rc.semi_conforme = false;


  assert(maillage != NULL);

  start_time[0] = bft_timer_wtime();
  start_time[1] = bft_timer_cpu_time();


  /* Recollement des faces concernées */
  /*----------------------------------*/

  return ecs_entmail_pcp__recolle(maillage->entmail,
                                  ECS_CHAMP_NOM_CONNECT,
                                  NULL,
                                  liste_fac_new,
                                  liste_fac_mod,
                                  liste_fac_err,
                                  param_rc);

  end_time[0] = bft_timer_wtime();
  end_time[1] = bft_timer_cpu_time();

  bft_printf(_("\n  Wall-clock time: %f s; CPU time: %f s\n\n"),
               (double)(end_time[0] - start_time[0]),
               (double)(end_time[1] - start_time[1]));
}


/*----------------------------------------------------------------------------
 *  Fonction qui construit les familles
 *----------------------------------------------------------------------------*/

void
ecs_maillage__cree_famille(ecs_maillage_t  *maillage,
                           ECS_ENTMAIL_E    ent_deb_e,
                           ECS_ENTMAIL_E    ent_fin_e)
{

  ecs_int_t       ient;
  ecs_int_t       ifam_ent;

  ecs_entmail_t *   vect_entmail[ECS_ENTMAIL_FIN];
  ecs_famille_t * * vect_famille;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(ent_fin_e <  ECS_ENTMAIL_FIN);
  assert(ent_deb_e <= ent_fin_e  );


  for (ient = ECS_ENTMAIL_DEB; ient < ECS_ENTMAIL_FIN; ient++)
    vect_entmail[ient] = NULL;


  for (ient = ent_deb_e; ient < (ecs_int_t)ent_fin_e + 1; ient++) {

    if (maillage->entmail[ient] != NULL)
      vect_entmail[ient] = maillage->entmail[ient];

  }


  vect_famille = ecs_entmail_pcp__cree_famille(vect_entmail);


  for (ifam_ent = ent_deb_e; ifam_ent < (ecs_int_t)ent_fin_e + 1; ifam_ent++)
    maillage->famille[ifam_ent] = vect_famille[ifam_ent];

  BFT_FREE(vect_famille);
}


/*----------------------------------------------------------------------------
 *  Fonction qui détruit les familles
 *----------------------------------------------------------------------------*/

void
ecs_maillage__detruit_famille(ecs_maillage_t  *maillage)
{

  ecs_int_t       ient;
  ecs_int_t       ifam_ent;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Destruction des champs "famille" */
  /*----------------------------------*/

  for (ient = ECS_ENTMAIL_DEB; ient < ECS_ENTMAIL_FIN; ient++) {

    if (maillage->entmail[ient] != NULL ) {

      ecs_entmail_pcp__detruit_att_nom(maillage->entmail[ient],
                                       ECS_CHAMP_NOM_FAMILLE);

    }

  }


  /* Destruction des familles */
  /*--------------------------*/

  for (ifam_ent  = (ecs_int_t)ECS_FAMILLE_FIN - 1;
       ifam_ent >= (ecs_int_t)ECS_FAMILLE_DEB;  ifam_ent--) {

    if (maillage->famille[ifam_ent] != NULL) {

      ecs_famille_chaine__detruit(&maillage->famille[ifam_ent]);

    }

  }


}


/*----------------------------------------------------------------------------
 *  Fonction qui construit les attributs "groupe" et "couleur"
 *   à partir des familles
 *----------------------------------------------------------------------------*/

void
ecs_maillage__cree_attribut(ecs_maillage_t  *maillage,
                            ECS_ENTMAIL_E    ent_deb_e,
                            ECS_ENTMAIL_E    ent_fin_e)
{

  ecs_int_t        ient;
  ecs_int_t        ifam_ent;

  ecs_famille_t    * famille;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(ent_fin_e <  ECS_ENTMAIL_FIN);
  assert(ent_deb_e <= ent_fin_e  );


  /* Concaténation des familles des différentes entités */

  famille = NULL;

  for (ifam_ent =  (ecs_int_t)ent_fin_e;
       ifam_ent >= (ecs_int_t)ent_deb_e; ifam_ent--) {

    if (maillage->famille[ifam_ent] != NULL) {

      ecs_famille_chaine__ajoute(&famille,
                                 maillage->famille[ifam_ent]);

      maillage->famille[ifam_ent] = NULL;


    }

  }


  if (famille != NULL) {

    for (ient =  (ecs_int_t)ent_fin_e;
         ient >= (ecs_int_t)ent_deb_e; ient--) {

      if (maillage->entmail[ient] != NULL ) {

        ecs_entmail_pcp__cree_attribut(maillage->entmail[ient],
                                       famille);

      }

    }

  }


  ecs_famille_chaine__detruit(&famille);
}


/*----------------------------------------------------------------------------
 *  Fonction qui supprime les attributs "groupe" et "couleur"
 *----------------------------------------------------------------------------*/

void
ecs_maillage__supprime_attributs(ecs_maillage_t  *this_maillage)
{

  ecs_int_t  ient;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Structures des entités de maillage */
  /*====================================*/

  /* Libération du contenu des entités */
  /*-----------------------------------*/

  for (ient = ECS_ENTMAIL_DEB; ient < ECS_ENTMAIL_FIN; ient++) {

    if (this_maillage->entmail[ient] != NULL) {

      ecs_entmail_pcp__suppr_attribut(this_maillage->entmail[ient]);

    }

  }

}


/*----------------------------------------------------------------------------
 *  Vérification d'un maillage
 *----------------------------------------------------------------------------*/

bool
ecs_maillage__verif(ecs_maillage_t   *maillage,
                    const ecs_int_t   nbr_dump,
                    const char       *nom_fic_dump,
                    ecs_post_t       *cas_post)
{
  ecs_tab_int_t     liste_fac_erreur;
  ecs_tab_int_t     liste_fac_de_bord;
  ecs_tab_int_t     liste_fac_isolee;

  bool              bool_coherent;

  ecs_maillage_t   *maillage_coupe = NULL;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL);

  bool_coherent = true;


  /* On vérifie qu'on est bien en connectivité descendante */

  assert(maillage->typ_connect == ECS_MAILLAGE_CONNECT_DESCENDANTE);

  assert(maillage->entmail != NULL);

  assert(maillage->entmail[ECS_ENTMAIL_CEL] != NULL);
  assert(maillage->entmail[ECS_ENTMAIL_FAC] != NULL);
  assert(maillage->entmail[ECS_ENTMAIL_SOM] != NULL);


  /* Création des coupes (pour post-traitement) */

  if (cas_post != NULL) {

    ecs_entmail_pcp__compte_typ_fac(maillage->entmail,
                                    &liste_fac_erreur,
                                    NULL,
                                    &liste_fac_de_bord,
                                    &liste_fac_isolee);


    maillage_coupe = ecs_maillage__cree_coupe(_(ECS_MAILLAGE_NOM_BORD),
                                              maillage,
                                              liste_fac_de_bord,
                                              nbr_dump,
                                              nom_fic_dump,
                                              true,
                                              ECS_POST_TYPE_BORD,
                                              cas_post);

    BFT_FREE(liste_fac_de_bord.val);
    liste_fac_de_bord.nbr = 0;


    /* Affichage des faces isolées ou de connectivité excessive */

    if (liste_fac_erreur.nbr > 0) {
      ecs_maillage_post__ecr_fac_liste(_(ECS_MAILLAGE_NOM_FAC_ERR_CONNECT),
                                       maillage,
                                       liste_fac_erreur,
                                       false,
                                       ECS_POST_TYPE_ERREUR,
                                       cas_post);
      BFT_FREE(liste_fac_erreur.val);
      liste_fac_erreur.nbr = 0;
    }

    if (liste_fac_isolee.nbr > 0) {
      ecs_maillage_post__ecr_fac_liste(_(ECS_MAILLAGE_NOM_FAC_ISOLEE),
                                       maillage,
                                       liste_fac_isolee,
                                       false,
                                       ECS_POST_TYPE_INFO,
                                       cas_post);
      BFT_FREE(liste_fac_isolee.val);
      liste_fac_isolee.nbr = 0;
    }

  }

  /* Vérification */

  bool_coherent = ecs_entmail_pcp__verif(maillage->entmail,
                                         cas_post);


  /* Destruction des coupes */

  if (maillage_coupe != NULL)
    maillage_coupe = ecs_maillage__detruit(maillage_coupe);


  return bool_coherent;
}


/*============================================================================*/

