package org.eclipse.webdav.dom;

/*
 * (c) Copyright IBM Corp. 2000, 2001.
 * All Rights Reserved.
 */

import java.util.*;
import org.eclipse.webdav.Policy;
import org.w3c.dom.*;

/**
 * An element editor for the WebDAV propertyupdate element. See RFC2518
 * section 12.13 for the element's definition.
 * <p>
 * <b>Note:</b> This class/interface is part of an interim API that is still under 
 * development and expected to change significantly before reaching stability. 
 * It is being made available at this early stage to solicit feedback from pioneering 
 * adopters on the understanding that any code that uses this API will almost 
 * certainly be broken (repeatedly) as the API evolves.
 * </p>
 *
 * @see Prop
 */
public class PropertyUpdate extends ElementEditor {
	/**
	 * An editor for a generic element that is a child of a WebDAV prop
	 * element, which, in turn, is a child of a WebDAV set or remove
	 * element.
	 */
	public class Directive extends ElementEditor {

		/**
		 * Creates a new editor on the given element.
		 *
		 * @param element a generic element
		 * @throws        MalformedElementException if there is reason
		 *                to believe that this editor's underlying
		 *                element is not well formed
		 */
		public Directive(Element element) throws MalformedElementException {

			super(element);

			Node parent = element.getParentNode();
			ensureDAVElement(parent, "prop", Policy.bind("ensure.expectingPropElmt"));

			Node grandparent = parent.getParentNode();
			ensure(
				isDAVElement(grandparent, "remove") || isDAVElement(grandparent, "set"), 
				Policy.bind("ensure.expectingRemoveOrSetElmt")); 
		}

		/**
		 * Returns this editor's underlying element, which may be any
		 * generic element.
		 *
		 * @return this editor's underlying element
		 */
		public Element getProperty() throws MalformedElementException {

			return root;
		}

		/**
		 * Returns <code>true</code> if this directive's property
		 * decends from a WebDAV remove element and <code>false</code>
		 * if it decends from a WebDAV set element.
		 *
		 * @return a boolean indicating whether this directive is a
		 *           remove or not
		 * @throws MalformedElementException if there is reason to
		 *         believe that this editor's underlying element is not
		 *         well formed
		 */
		public boolean isRemove() throws MalformedElementException {

			Node parent = root.getParentNode();
			ensureDAVElement(parent, "prop", Policy.bind("ensure.expectingPropElmt"));

			Node grandparent = parent.getParentNode();

			return isDAVElement(grandparent, "remove");
		}

		/**
		 * Returns <code>true</code> if this directive's property
		 * decends from a WebDAV set element and <code>false</code>
		 * if it decends from a WebDAV remove element.
		 *
		 * @return a boolean indicating whether this directive is a
		 *           set or not
		 * @throws MalformedElementException if there is reason to
		 *         believe that this editor's underlying element is not
		 *         well formed
		 */
		public boolean isSet() throws MalformedElementException {

			Node parent = root.getParentNode();
			ensureDAVElement(parent, "prop", Policy.bind("ensure.expectingPropElmt"));

			Node grandparent = parent.getParentNode();

			return isDAVElement(grandparent, "set");
		}
	}
/**
 * Creates a new editor on the given WebDAV propertyupdate element. The
 * element is assumed to be well formed.
 *
 * @param root a propertyupdate element
 * @throws        MalformedElementException if there is reason to
 *                believe that the element is not well formed
 */
public PropertyUpdate(Element root) throws MalformedElementException {

	super(root, "propertyupdate");
}
/**
 * Adds a remove to the given propertyupdate and returns an editor on
 * its prop.
 *
 * @return an editor on a new prop
 */
public Prop addRemove() {

	Element remove = appendChild(root, "remove");
	Element prop = appendChild(remove, "prop");

	Prop result = null;
	try {
		result = new Prop(prop);
	} catch (MalformedElementException e) {
		Assert.isTrue(false, Policy.bind("assert.internalError"));
	}

	return result;
}
/**
 * Adds a set to the given propertyupdate and returns an editor on its
 * prop.
 *
 * @return an editor on a new prop
 */
public Prop addSet() {

	Element set = appendChild(root, "set");
	Element prop = appendChild(set, "prop");

	try {
		return new Prop(prop);
	} catch (MalformedElementException e) {
		Assert.isTrue(false, Policy.bind("assert.internalError"));
		return null; // Never reached.
	}
}
/**
 * Creates a new WebDAV propertyupdate element and sets it as the root
 * of the given document. Returns an editor on the new propertyupdate
 * element. The document must not be <code>null</code>, and must not
 * already have a root element.
 *
 * @param document the document that will become the root of a new
 *                 propertyupdate element
 * @return         an element editor on a propertyupdate element
 */
public static PropertyUpdate create(Document document){
	Assert.isNotNull(document);
	Assert.isTrue(document.getDocumentElement() == null);

	Element element = create(document, "propertyupdate");

	PropertyUpdate result = null;
	try {
		result = new PropertyUpdate(element);
	} catch(MalformedElementException e){
		Assert.isTrue(false, Policy.bind("assert.internalError"));
	}

	return result;
}
/**
 * Returns an <code>Enumeration</code> over this propertyupdate's set
 * and remove property elements.
 *
 * @return an <code>Enumeration</code> of
 *         <code>PropertyUpdate.Directive</code>s
 * @throws MalformedElementException if there is reason to believe that
 *         this editor's underlying element is not well formed
 */
public Enumeration getSetsAndRemoves() throws MalformedElementException {

	Node setOrRemove = getFirstChild(root, new String[] {"remove", "set"});
	ensureNotNull(Policy.bind("ensure.missingRemoveOrSetElmt"), setOrRemove);
	Node property = null;

	while (setOrRemove != null && property == null) {
		Node prop = getFirstChild((Element) setOrRemove, "prop");
		ensureNotNull(Policy.bind("ensure.missingPropElmt"), prop);
		property = getChildElement((Element) prop, true);

		if (property == null)
			setOrRemove = 
				getNextSibling((Element) setOrRemove, new String[] {"remove", "set"}); 
	}

	final Node a = setOrRemove;
	final Node c = property;

	Enumeration e = new Enumeration() {
		Node currentSetOrRemove = a;
		Node currentProperty = c;

		public boolean hasMoreElements() {

			return currentProperty != null;
		}

		public Object nextElement() {

			if (!hasMoreElements())
				throw new NoSuchElementException();
			Directive result = null;
			try {
				result = new Directive((Element) currentProperty);
			} catch (MalformedElementException ex) {
				Assert.isTrue(false, Policy.bind("assert.internalError"));
			}
			currentProperty = getNextSibling((Element) currentProperty);
			while (currentSetOrRemove != null && currentProperty == null) {
				currentSetOrRemove = 
					getNextSibling((Element) currentSetOrRemove, new String[] {"remove", "set"}); 
				if (currentSetOrRemove != null) {
					Node prop = getFirstChild((Element) currentSetOrRemove, "prop");
					if (prop != null)
						currentProperty = getChildElement((Element) prop, true);
				}
			}
			return result;
		}
	};

	return e;
}
}
