/*  dvdisaster: Additional error correction for optical media.
 *  Copyright (C) 2004,2005 Carsten Gnoerlich.
 *  Project home page: http://www.dvdisaster.com
 *  Email: carsten@dvdisaster.com  -or-  cgnoerlich@fsfe.org
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA,
 *  or direct your browser at http://www.gnu.org.
 */

#include "dvdisaster.h"

#ifdef SYS_LINUX
#include <sys/wait.h>
#endif

#ifdef SYS_MINGW
#include "windows.h"
#include "shellapi.h"
#endif

/***
 *** Ask user to specify his browser
 ***/

#ifdef SYS_LINUX
#define SEARCH_BUTTON 1

typedef struct
{  GtkWidget *dialog;
   GtkWidget *entry;
   GtkWidget *search;
   GtkWidget *filesel;
   GtkWidget *fileok;
   GtkWidget *filecancel;
   char *url;
} browser_dialog_info;

static void response_cb(GtkWidget *widget, int response, gpointer data)
{  browser_dialog_info *bdi = (browser_dialog_info*)data; 

   switch(response)
   {  case GTK_RESPONSE_ACCEPT:
	if(Closure->browser) g_free(Closure->browser);
	Closure->browser = g_strdup(gtk_entry_get_text(GTK_ENTRY(bdi->entry)));
	ShowHTML(bdi->url);
	break;

      case GTK_RESPONSE_REJECT:
        break;
   }
   gtk_widget_destroy(widget);
   if(bdi->filesel)
     gtk_widget_destroy(bdi->filesel);
}

static void search_cb(GtkWidget *widget, gpointer data)
{  browser_dialog_info *bdi = (browser_dialog_info*)data; 

   if(widget == bdi->search) 
   {  bdi->filesel = gtk_file_selection_new(_utf("windowtitle|Choose a browser"));
      bdi->fileok = GTK_FILE_SELECTION(bdi->filesel)->ok_button;
      bdi->filecancel = GTK_FILE_SELECTION(bdi->filesel)->cancel_button;
      ReverseCancelOK(GTK_DIALOG(bdi->filesel));
      gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(bdi->filesel));
      g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(bdi->filesel)->ok_button), "clicked", 
		       G_CALLBACK(search_cb), bdi);
    
      g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(bdi->filesel)->cancel_button), "clicked", 
		       G_CALLBACK(search_cb), bdi);
      
      gtk_widget_show(bdi->filesel);
   }

   if(widget == bdi->fileok)
   {
      if(Closure->browser) g_free(Closure->browser);
      Closure->browser = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(bdi->filesel)));
      ShowHTML(bdi->url);
      gtk_widget_destroy(bdi->filesel);
      gtk_widget_destroy(bdi->dialog);
      return;
   }

   if(widget == bdi->filecancel)
     gtk_widget_destroy(bdi->filesel);
}

static void browser_dialog(char *url)
{  GtkWidget *dialog, *vbox, *hbox, *label, *entry, *button;
   browser_dialog_info *bdi = g_malloc0(sizeof(browser_dialog_info));

   /* Create the dialog */

   dialog = gtk_dialog_new_with_buttons(_utf("windowtitle|Browser required"), 
				       Closure->window, GTK_DIALOG_DESTROY_WITH_PARENT,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, 
				       GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);
   bdi->dialog = dialog;
   if(url)
   {  bdi->url = g_strdup(url);
   }

   vbox = gtk_vbox_new(FALSE, 0);
   gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), vbox, FALSE, FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);

   /* Insert the contents */

   label = gtk_label_new(NULL);
   gtk_label_set_markup(GTK_LABEL(label), _utf("<b>Could not find a suitable browser.</b>\n\n"
                                               "Which browser would you like to use\n"
                                               "for reading the online documentation?\n\n"
			                       "Please enter its name (e.g. mozilla) or\n"
			                       "use the \"Search\" button for a file dialog.\n")),
			      gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 10);

   hbox = gtk_hbox_new(FALSE, 0);
   gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 10);

   bdi->entry = entry = gtk_entry_new();
   gtk_box_pack_start(GTK_BOX(hbox), entry, FALSE, FALSE, 10);

   bdi->search = button = gtk_button_new_with_label(_utf("Search"));
   g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(search_cb), bdi);
   gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 10);

   /* Show it */

   g_signal_connect(dialog, "response", G_CALLBACK(response_cb), bdi);

   gtk_widget_show_all(dialog);
}
#endif /* SYS_LINUX */

/***
 *** Show the manual in an external browser
 ***/

/*
 * Check the child processes exit status
 * to find whether the browser could be invoked.
 */

typedef struct
{  pid_t pid;
   char *url;
   GtkWidget *msg;
   int seconds;
} browser_info;


static void msg_destroy_cb(GtkWidget *widget, gpointer data)
{  browser_info *bi = (browser_info*)data;

   bi->msg = NULL; 
}

#ifdef SYS_LINUX
static gboolean browser_timeout_func(gpointer data)
{  browser_info *bi = (browser_info*)data;
   int status;

   waitpid(bi->pid, &status, WNOHANG);

   if(WIFEXITED(status))
   {  if(bi->msg) 
        gtk_widget_destroy(bi->msg);

      if(WEXITSTATUS(status) == 111)
	 browser_dialog(bi->url);

      if(bi->url) g_free(bi->url);
      g_free(bi);
      return FALSE;
   }

   bi->seconds++;
   if(bi->seconds == 10 && bi->msg)
   {  gtk_widget_destroy(bi->msg);
      bi->msg = NULL;
   }

   return bi->seconds > 60 ? FALSE : TRUE;
}
#endif /* SYS_LINUX */

#ifdef SYS_MINGW
static gboolean browser_timeout_func(gpointer data)
{  browser_info *bi = (browser_info*)data;
   
   bi->seconds++;
   if(bi->seconds >= 10)
   {  if(bi->msg)
      {  gtk_widget_destroy(bi->msg);
         bi->msg = NULL;
      }
      return FALSE;
   }

   return TRUE;
}
#endif /* SYS_MINGW */

/*
 * Invoke the browser
 */

void ShowHTML(char *target)
{  browser_info *bi = g_malloc0(sizeof(browser_info));
   struct stat mystat;
   char index_path[strlen(Closure->docDir)+80];
   const char *lang;
#ifdef SYS_LINUX
   pid_t pid; 
#endif

   /* If no target is given, select between translations of the manual. */

   if(!target)
   {  lang = g_getenv("LANG");
      if(!strncmp(lang, "de", 2))
           sprintf(index_path,"%s/de/index.html",Closure->docDir); 
      else sprintf(index_path,"%s/en/index.html",Closure->docDir); 
      
      target = index_path;

      if(stat(target, &mystat) == -1)
      {  CreateMessage(_("Documentation file\n%s\nnot found.\n"), GTK_MESSAGE_ERROR, target);
         g_free(bi);
         return;
      }
   }
   else bi->url = g_strdup(target);

   /* Lock the help button and show a message for 10 seconds. */

   TimedInsensitive(Closure->helpButton, 10000);
   bi->msg = CreateMessage(_("Please hang on until the browser comes up!"), GTK_MESSAGE_INFO);
   g_signal_connect(G_OBJECT(bi->msg), "destroy", G_CALLBACK(msg_destroy_cb), bi);

#ifdef SYS_MINGW
   /* Okay, Billy wins big time here ;-) */

   ShellExecute(NULL, "open", target, "", "c:\\", SW_SHOWNORMAL);
   g_timeout_add(1000, browser_timeout_func, (gpointer)bi);
#endif

#ifdef SYS_LINUX
   /* Fork and start the browser */

   bi->pid = pid = fork();

   if(pid == -1)
   {  printf("fork failed\n");
      return;
   }

   g_timeout_add(1000, browser_timeout_func, (gpointer)bi);

   /* Try the browser from .dvdisaster first,
      then make some guesses on available browsers. */

   if(pid == 0)
   {  char *argv[10];
      int argc = 0;

      argv[argc++] = Closure->browser;
      argv[argc++] = target;
      argv[argc++] = NULL;
      execvp(argv[0], argv);
#if 1 /* set to 0 for testing only! */
      argv[0] = "mozilla";
      execvp(argv[0], argv);

      argv[0] = "firefox";
      execvp(argv[0], argv);

      argv[0] = "konqueror";
      execvp(argv[0], argv);

      argv[0] = "netscape";
      execvp(argv[0], argv);

      argv[0] = "opera";
      execvp(argv[0], argv);
#endif
      _exit(111);
   }
#endif /* SYS_LINUX */
}
