/*  dvdisaster: Additional error correction for optical media.
 *  Copyright (C) 2004,2005 Carsten Gnoerlich.
 *  Project home page: http://www.dvdisaster.com
 *  Email: carsten@dvdisaster.com  -or-  cgnoerlich@fsfe.org
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA,
 *  or direct your browser at http://www.gnu.org.
 */

#include "dvdisaster.h"

#ifdef SYS_MINGW
 #include <windows.h>
 #include <winnls.h>
#endif

/*
 * A special marker for a unread sector 
 */

void PrepareDeadSector()
{   char *end_marker;
    int end_length; 

    if(!Closure->deadSector)
      Closure->deadSector = g_malloc(2048);

    if(Closure->fillUnreadable >= 0)
      memset(Closure->deadSector, Closure->fillUnreadable, 2048);
    else
    {  memset(Closure->deadSector, 0 ,2048);
       g_sprintf(Closure->deadSector,
		 "dvdisaster dead sector marker\n"
		 "This sector could not be read from the image.\n"
		 "Its contents have been substituted by the dvdisaster read routine.\n");

       end_marker = "dvdisaster dead sector end marker\n";
       end_length = strlen(end_marker);
       memcpy(Closure->deadSector+2046-end_length, end_marker, end_length); 
    }
}

/*
 * See if we can handle the specified .ecc file
 */

int VerifyVersion(EccHeader *eh, int fatal)
{  
   if(Closure->version < eh->neededVersion)
   {  if(fatal)
       Stop(_("\n* This .ecc file requires dvdisaster V%d.%d.%d or higher.\n"
	      "* Please visit http://www.dvdisaster.com for an upgrade."),
	    eh->neededVersion/10000,
	    (eh->neededVersion%10000)/100,
	    eh->neededVersion%100);

      return 1;
   }

   return 0;
}

/*
 * Append the given file suffix if none is already present
 */

char *ApplyAutoSuffix(char *filename, char *suffix)
{  char *out;

   if(!filename || !*filename || strrchr(filename, '.')) 
     return filename;

   out = g_strdup_printf("%s.%s",filename,suffix);
   g_free(filename);
   
   return out;
}

/***
 *** Open the image and ecc files and fill in the respective Info structs.
 ***/

void OpenImageAndEcc(ImageInfo **iptr, EccInfo **eptr, int mode)
{  ImageInfo *ii = NULL;
   EccInfo   *ei = NULL; 

   /*** Open the ecc file  */

   if(eptr)
   {
      *eptr = ei = g_malloc0(sizeof(EccInfo));
      ei->eh = g_malloc0(sizeof(EccHeader));

      if(!(ei->file = fopen(Closure->eccName, mode & WRITEABLE_ECC ? "w" : "r")))
      {  FreeEccInfo(ei);
         *eptr = NULL;
	 ei = NULL;

	 if(mode & PRINT_MODE)  /* missing ecc file no problem in print mode */
	 {  goto open_image;
	 }
         else  Stop(_("Can't open %s:\n%s"),Closure->eccName,strerror(errno));
      }

#ifdef SYS_MINGW
      setmode(fileno(ei->file), O_BINARY);
#endif

      if(!(mode & WRITEABLE_ECC))
      {   int n = fread(ei->eh, sizeof(EccHeader), 1, ei->file);

          if(n != 1)
	    Stop(_("Can't read ecc header:\n%s"),strerror(errno));

	  /*** See if we can use the ecc file */

	  if(   ei->eh->method_flags[0] != 1 || ei->eh->method_flags[1] 
	     || ei->eh->method_flags[2] || ei->eh->method_flags[3])
	    Stop(_("\nEcc file has been created with version 0.40.7.\n"
		   "Please use version 0.41.1 or higher to create the .ecc files.\n"
		   "I pledge not to change the file format again until version 2.0 :-)\n"));

	  VerifyVersion(ei->eh,1);

	  ei->sectors = uchar_to_gint64(ei->eh->sectors);
	  rewind(ei->file);
      }
   }

   /*** Open the image file */

open_image:

   if(iptr)
   {  gint64 img_size;
      int n,file_flags;
      gint64 fp_sector;
      unsigned char buf[2048];
      struct MD5Context md5ctxt;

      *iptr = ii = g_malloc0(sizeof(ImageInfo));

      if(!(mode & PRINT_MODE))
	PrintLog(_("\nOpening %s"), Closure->imageName);

      if(!LargeStat(Closure->imageName, &img_size))
      {  if(mode & PRINT_MODE)  /* no error in print mode */ 
	 {  FreeImageInfo(ii);
 	    *iptr = NULL;
	    return;
	 }

         PrintLog(_(": not present.\n"));
         g_free(ii); *iptr=NULL;
	 Stop(_("Image file %s not present."),Closure->imageName);
      }

      file_flags = mode & WRITEABLE_IMAGE ? O_RDWR : O_RDONLY;

      if(!(ii->file = LargeOpen(Closure->imageName, file_flags, IMG_PERMS)))
      {  if(Closure->guiMode)
	 {  g_free(ii); *iptr=NULL;
	    PrintLog(": %s", strerror(errno));
	    Stop(_("Can't open %s:\n%s"),Closure->imageName,strerror(errno));
	 }
         else Stop(": %s", strerror(errno));
      }

      ii->size = img_size;
      CalcSectors(img_size, &ii->sectors, &ii->inLast);

      if(!(mode & PRINT_MODE))
	PrintLog(_(": %lld medium sectors.\n"), ii->sectors);


      /*** Calculate md5sum of the footprint sector.
	   Use sector specified by .ecc file if possible, 
	   else use built-in default  */

      if(!ei || (mode & WRITEABLE_ECC))
	   fp_sector = FOOTPRINT_SECTOR;
      else fp_sector = ei->eh->fpSector;

      LargeSeek(ii->file, fp_sector*2048);

      MD5Init(&md5ctxt);
      n = LargeRead(ii->file, buf, 2048);
      if(n != 2048)
	Stop(_("could not read image sector %lld (only %d bytes):\n%s"),
	     fp_sector,n,strerror(errno));
      MD5Update(&md5ctxt, buf, n);
      MD5Final(ii->mediumFP, &md5ctxt);
      if(!memcmp(buf,Closure->deadSector,2048))  /* No sector, no md5sum */
	memset(ii->mediumFP, 0 ,16);
      LargeSeek(ii->file, 0);                     /* rewind */
   }
}

/*
 * Scan the image for missing blocks.
 * If the ecc file is present, also compare the CRC sums.
 * If CREATE_CRC is requested, calculate the CRC sums.
 */

#define CRCBUFSIZE (1024*256)

void ScanImage(ImageInfo *ii, EccInfo *ei, int mode, GtkWidget *pbar)
{  EccHeader eh; 
   unsigned char buf[2048];
   guint32 crcbuf[CRCBUFSIZE];
   int crcidx = 0;
   struct MD5Context image_md5;
   gint64 eh_sectors = 0;
   gint64 s, first_missing, last_missing;
   gint64 prev_missing = 0;
   gint64 prev_crc_errors = 0;
   int last_percent,current_missing;
   char *msg;

   /* Read the ecc file header */

   if(ei && mode != CREATE_CRC)
   {   rewind(ei->file);
       fread(&eh, sizeof(EccHeader), 1, ei->file);
       eh_sectors = uchar_to_gint64(eh.sectors);
   }     

   /* Position behind the ecc file header,
      initialize CRC buffer pointers */

   if(ei)
   {  if(fseeko(ei->file, (gint64)sizeof(EccHeader), SEEK_SET) == -1)
         Stop(_("fseek failed skipping ecc header: %s"),strerror(errno));

      crcidx = (mode & CREATE_CRC) ? 0 : CRCBUFSIZE;
      MD5Init(&ei->md5Ctxt);            /* md5sum of CRC portion of ecc file */
   }

   /* Prepare for scanning the image and calculating its md5sum */

   MD5Init(&image_md5);              /* md5sum of image file itself */
   LargeSeek(ii->file, 0);            /* rewind image file */   
      
   if(mode & PRINT_MODE)
        msg = _("- testing sectors  : %3d%%");
   else msg = _("Scanning image sectors: %3d%%");

   last_percent = 0;
   ii->sectorsMissing = 0;
   first_missing = last_missing = -1;

   /* Go through all sectors and look for the "dead sector marker" */
   
   for(s=0; s<ii->sectors; s++)
   {  int n,percent;

      /* Check for user interruption */

      if(Closure->stopActions)   
      {  ii->sectorsMissing += ii->sectors - s;
         return;
      }

      /* Read the next sector */

      n = LargeRead(ii->file, buf, 2048);
      if(n != 2048 && (s != ii->sectors - 1 || n != ii->inLast))
	Stop(_("premature end in image (only %d bytes): %s\n"),n,strerror(errno));

      /* Look for the dead sector marker */

      current_missing = !memcmp(buf, Closure->deadSector, n);
      if(current_missing)
      {  if(first_missing < 0) first_missing = s;
         last_missing = s;
	 ii->sectorsMissing++;
      }

      /* Report dead sectors. Combine subsequent missing sectors into one report. */

      if(mode & PRINT_MODE)
	if(!current_missing || s==ii->sectors-1)
	{  if(first_missing>=0)
	    {   if(first_missing == last_missing)
		     PrintCLI(_("* missing sector   : %lld\n"), first_missing);
		else PrintCLI(_("* missing sectors  : %lld - %lld\n"), first_missing, last_missing);
	      first_missing = -1;
	   }
	}

      if(ei)   /* Do something with the CRC portion of the .ecc file */
      {
	 /* If creation of the CRC32 is requested, do that. */

	 if(mode & CREATE_CRC)
	 {  crcbuf[crcidx++] = Crc32(buf, 2048);

	    if(crcidx >= CRCBUFSIZE)  /* write out CRC buffer contents */
	    {  size_t size = CRCBUFSIZE*sizeof(guint32);

	       MD5Update(&ei->md5Ctxt, (unsigned char*)crcbuf, size);
	       if(fwrite(crcbuf, 1, size, ei->file) != size)
		 Stop(_("Error writing CRC information: %s"),strerror(errno));

	       crcidx = 0;
	    }
	 }

	 /* else do the CRC32 check. Missing sectors are skipped in the CRC report. */
	 
	 else if(s < eh_sectors)
	 {  guint32 crc = Crc32(buf, 2048); 

            /* If the CRC buf is exhausted, refill. */

            if(crcidx >= CRCBUFSIZE)
	    {  size_t remain = ii->sectors-s;
	       size_t size;

	       if(remain < CRCBUFSIZE)
		    size = remain*sizeof(guint32);
	       else size = CRCBUFSIZE*sizeof(guint32);

	       if(fread(crcbuf, 1, size, ei->file) != size)
		 Stop(_("Error reading CRC information: %s"),strerror(errno));

	       crcidx = 0;
	    }

	    if(crc != crcbuf[crcidx++] && !current_missing)
	    {  PrintCLI(_("* CRC error, sector: %lld\n"), s);
	       ii->crcErrors++;
	    }
	 }
      }

      MD5Update(&image_md5, buf, n);  /* update image md5sum */

      if(Closure->guiMode && mode & PRINT_MODE) 
	    percent = (COMPARE_IMAGE_SEGMENTS*s)/ii->sectors;
      else  percent = (100*s)/ii->sectors;
      if(last_percent != percent) 
      {  PrintProgress(msg,percent);

         if(pbar)
	   SetProgress(pbar, percent, 100);

	 if(Closure->guiMode && mode & PRINT_MODE)
	 {  AddCompareValues(percent, ii->sectorsMissing, ii->crcErrors,
			     ii->sectorsMissing - prev_missing,
			     ii->crcErrors - prev_crc_errors);
	    prev_missing = ii->sectorsMissing;
	    prev_crc_errors = ii->crcErrors;
	 }

	 last_percent = percent;
      }
   }

   /*** Flush the rest of the CRC buffer */

   if((mode & CREATE_CRC) && crcidx)
   {  size_t size = crcidx*sizeof(guint32);

      MD5Update(&ei->md5Ctxt, (unsigned char*)crcbuf, size);
      if(fwrite(crcbuf, 1, size, ei->file) != size)
	Stop(_("Error writing CRC information: %s"),strerror(errno));
   }

   /*** The image md5sum can only be calculated if all blocks have been successfully read. */

   MD5Final(ii->mediumSum, &image_md5);
   PrintProgress(msg,100);

   LargeSeek(ii->file, 0);
}

/*
 * Free the ImageInfo and EccInfo stuff
 */

void FreeImageInfo(ImageInfo *ii)
{
   if(ii->file) 
     if(!LargeClose(ii->file))
       Stop(_("Error closing image file:\n%s"), strerror(errno));

   g_free(ii);
}

void FreeEccInfo(EccInfo *ei)
{
   if(ei->file) 
     if(fclose(ei->file))
       Stop(_("Error closing error correction file:\n%s"), strerror(errno));

   if(ei->eh) g_free(ei->eh);
   g_free(ei);
}

/*
 * Create ecc information for the medium
 */

/* Remove the image file if requested */

void UnlinkImage(GtkWidget *label)
{
   if(LargeUnlink(Closure->imageName))
   {    PrintLog(_("\nImage file %s deleted.\n"),Closure->imageName);

        if(Closure->guiMode)
	  SetLabelText(GTK_LABEL(label),
		       _("\nImage file %s deleted.\n"), Closure->imageName);
   }
   else 
   {  if(!Closure->guiMode)
       PrintLog("\n");

       ModalWarning(GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, NULL,
		    _("Image file %s not deleted: %s\n"),
		    Closure->imageName, strerror(errno));
   }
}

void CreateEcc(void)
{  Method *method = FindMethod(Closure->methodName); 

   /*** GUI mode does its own FindMethod() before calling us
	so the following Stop() will never execute in GUI mode */

   if(!method) Stop(_("\nMethod %s not available.\n"
		      "Use -m without parameters for a method list.\n"), 
	            Closure->methodName);

   method->create(method);
}

/*
 * Fix the medium with ecc information
 */

void FixEcc(void)
{  EccInfo *ei;
   Method *method; 
   char method_name[5];

   /*** Peek into Ecc header to determine the method */

   OpenImageAndEcc(NULL, &ei, READABLE_ECC);
   memcpy(method_name, ei->eh->method, 4); method_name[4] = 0;
   FreeEccInfo(ei);

   /*** GUI mode has already verified that the method exists before
	calling us so the following Stop() will never execute in GUI mode. */

   method = FindMethod(method_name);
   if(!method) Stop(_("\nUnknown method %s.\n"),method_name);

   /*** Dispatch to proper method */
   
   method->fix(method);
}

/*
 * The all-famous main() loop 
 */

typedef enum
{  MODE_NONE, 

   MODE_CREATE,
   MODE_HELP,
   MODE_FIX,
   MODE_PRINT, 
   MODE_READ, 
   MODE_SCAN,
   MODE_SEQUENCE, 

   MODE_BYTESET, 
   MODE_ERASE, 
   MODE_LIST_ASPI,
   MODE_RANDOM_ERR, 
   MODE_RANDOM_IMAGE,
   MODE_SEND_CDB,
   MODE_SHOW_SECTOR, 
   MODE_TRUNCATE,
   MODE_ZERO_UNREADABLE,

   MODIFIER_ADAPTIVE_READ, 
   MODIFIER_AUTO_SUFFIX,
   MODIFIER_CACHE_SIZE, 
   MODIFIER_CLV_SPEED,    /* unused */ 
   MODIFIER_CAV_SPEED,    /* unused */
   MODIFIER_DAO, 
   MODIFIER_DEBUG,
   MODIFIER_FILL_UNREADABLE,
   MODIFIER_KEEP_STYLE,
   MODIFIER_PARSE_UDF,
   MODIFIER_RANDOM_SEED,
   MODIFIER_SPEED_WARNING, 
   MODIFIER_SPINUP_DELAY, 
   MODIFIER_SPLIT_FILES,
   MODIFIER_TRUNCATE
} run_mode;

int main(int argc, char *argv[])
{  int mode = MODE_NONE; 
   int sequence = MODE_NONE;
   char *debug_arg = NULL;
   char *default_device;
   char *read_range = NULL;
#ifdef WITH_NLS_YES
   char *locale_test,src_locale_path[strlen(SRCDIR)+10];
#endif

#ifdef WITH_MEMDEBUG_YES
    atexit(check_memleaks);
#endif

    /*** Setup the global closure. */

    InitClosure();

    /*** Setup the locale.
         Try the local source directory first (it may be more recent),
         then fall back to the global installation directory.
         The test phrase is supposed to be translated with "yes",
         _independent_ of the actual locale! */

#ifdef WITH_NLS_YES
#ifdef SYS_MINGW
    if(!g_getenv("LANG"))  /* Unix style setting has precedence */
    {  LANGID lang_id;

       /* Try to get locale from Windows
	  and set the respective environment variables. */

       lang_id = GetUserDefaultLangID(); 
       switch(PRIMARYLANGID(lang_id))
       {  case 0x07:  /* German */
	     g_setenv("LANG", "de_DE", 1);
#ifdef WIN_CONSOLE
	     g_setenv("OUTPUT_CHARSET", "CP850", 1);
#else
	     g_setenv("OUTPUT_CHARSET", "iso-8859-1", 1);
#endif
	     break;

          case 0x10:  /* Italian */
	     g_setenv("LANG", "it_IT", 1);
#ifdef WIN_CONSOLE
	     g_setenv("OUTPUT_CHARSET", "CP850", 1);
#else
	     g_setenv("OUTPUT_CHARSET", "iso-8859-1", 1);
#endif
	     break;
       }
    }
#endif /* SYS_MINGW */

    setlocale(LC_CTYPE, "");
    setlocale(LC_MESSAGES, "");
    textdomain("dvdisaster");

    /* Try local source directory first */

    g_sprintf(src_locale_path,"%s/locale",SRCDIR);
    bindtextdomain("dvdisaster", src_locale_path);

    /* TRANSLATORS: 
       This is a dummy entry which is supposed to translate into "ok".
       Please do not return anything else here. */
    locale_test = gettext("test phrase for verifying the locale installation");

    /* Try current directory. Might work if we are starting
       from the root dir of a binary distribution. */

    if(strcmp(locale_test, "ok"))
    {  char buf[256];

       if(getcwd(buf, 256))
       {  char locale_path[strlen(buf)+20];
          g_sprintf(locale_path,"%s/locale", buf);
          bindtextdomain("dvdisaster", locale_path);
	  locale_test = gettext("test phrase for verifying the locale installation");
       }
    }

#ifdef SYS_MINGW
    /* Try the directory where our executable comes from.
       Only possible under Windows. */

    if(strcmp(locale_test, "ok"))
    {  char locale_path[strlen(Closure->binDir)+20];

       g_sprintf(locale_path,"%s/locale", Closure->binDir);
       bindtextdomain("dvdisaster", locale_path);
       locale_test = gettext("test phrase for verifying the locale installation");
    }
#endif

    /* Last resort: fall back to global locale */

    if(strcmp(locale_test, "ok"))
      bindtextdomain("dvdisaster", LOCALEDIR);  
#endif /* WITH_NLS_YES */

   /*** Collect the Method list */

   CollectMethods();

   /*** The following test may go wrong if the compiler chooses the
	wrong packing. */

   if(sizeof(EccHeader) != 4096)
     Stop(_("sizeof(EccHeader) is %d, but must be 4096.\n"),sizeof(EccHeader));

   /*** Determine the default device (OS dependent!) */

   default_device = DefaultDevice();
   if(!Closure->device)
        Closure->device = default_device;
   else g_free(default_device);

   /*** Parse the options */
   
   for(;;)
   {  int option_index,c;
      static struct option long_options[] =
      { {"adaptive-read", 0, 0, MODIFIER_ADAPTIVE_READ},
	{"auto-suffix", 0, 0,  MODIFIER_AUTO_SUFFIX},
	{"byteset", 1, 0, MODE_BYTESET },
	{"cache-size", 1, 0, MODIFIER_CACHE_SIZE },
	{"cav", 1, 0, MODIFIER_CAV_SPEED },
	{"clv", 1, 0, MODIFIER_CLV_SPEED },
	{"create", 0, 0, 'c'},
	{"dao", 0, 0, MODIFIER_DAO },
	{"debug", 0, 0, MODIFIER_DEBUG }, 
	{"device", 0, 0, 'd'},
        {"ecc", 1, 0, 'e'},
	{"erase", 1, 0, MODE_ERASE },
	{"fill-unreadable", 1, 0, MODIFIER_FILL_UNREADABLE },
	{"fix", 0, 0, 'f'},
	{"help", 0, 0, 'h'},
        {"image", 1, 0, 'i'},
	{"jump", 1, 0, 'j'},
	{"keep-style", 0, 0, MODIFIER_KEEP_STYLE },
#ifdef SYS_MINGW
	{"list", 0, 0, 'l' },
#endif
	{"method", 1, 0, 'm' },
	{"parse-udf", 0, 0, MODIFIER_PARSE_UDF },
        {"prefix", 1, 0, 'p'},
	{"random-errors", 0, 0, MODE_RANDOM_ERR },
	{"random-image", 1, 0, MODE_RANDOM_IMAGE },
	{"random-seed", 1, 0, MODIFIER_RANDOM_SEED },
	{"read", 2, 0,'r'},
	{"redundancy", 1, 0, 'n'},
	{"scan", 2, 0,'s'},
	{"send-cdb", 1, 0, MODE_SEND_CDB},
	{"show-sector", 1, 0, MODE_SHOW_SECTOR},
	{"speed-warning", 2, 0, MODIFIER_SPEED_WARNING},
	{"spinup-delay", 1, 0, MODIFIER_SPINUP_DELAY},
	{"split-files", 0, 0, MODIFIER_SPLIT_FILES},
	{"test", 0, 0, 't'},
	{"truncate", 2, 0, MODIFIER_TRUNCATE},
	{"unlink", 0, 0, 'u'},
       	{"version", 0, 0, 'v'},
	{"zero-unreadable", 0, 0, MODE_ZERO_UNREADABLE},
        {0, 0, 0, 0}
      };

      c = getopt_long(argc, argv, 
		      "cd:e:fhi:j:lm::n:p:r::s::tuv",
		      long_options, &option_index);

      if(c == -1) break;

      switch(c)
      {  case 'r': mode = MODE_SEQUENCE; sequence |= 1<<MODE_READ; 
	           if(optarg) read_range = g_strdup(optarg);
		   break;

         case 's': mode = MODE_SEQUENCE; sequence |= 1<<MODE_SCAN; 
	           if(optarg) read_range = g_strdup(optarg); 
		   break;
         case 'c': mode = MODE_SEQUENCE; sequence |= 1<<MODE_CREATE; break;
         case 'f': mode = MODE_SEQUENCE; sequence |= 1<<MODE_FIX; break;
         case 't': mode = MODE_SEQUENCE; sequence |= 1<<MODE_PRINT; break;
         case 'u': Closure->unlinkImage = TRUE; break;
         case 'h': mode = MODE_HELP; break;
         case 'i': if(optarg) 
	           {  g_free(Closure->imageName);
		      Closure->imageName = g_strdup(optarg); 
		   }
	           break;
         case 'j': if(optarg) Closure->sectorSkip = atoi(optarg) & ~0xf;
	           if(Closure->sectorSkip<0) Closure->sectorSkip = 0;
		   break;
	 case 'l': mode = MODE_LIST_ASPI; break;
         case 'm': if(optarg) 
	           {  g_free(Closure->methodName);
	              Closure->methodName = g_strdup(optarg); 
                   }
	           else {  ListMethods(); FreeClosure(); exit(EXIT_SUCCESS); }
	           break;
         case 'n': if(optarg) Closure->redundancy = g_strdup(optarg); break;
         case 'e': if(optarg) 
	           {  g_free(Closure->eccName);
		      Closure->eccName = g_strdup(optarg);
		   }
	           break;
         case 'p': if(optarg) 
	           {  Closure->eccName = g_malloc(strlen(optarg)+5);
		      Closure->imageName = g_malloc(strlen(optarg)+5);
		      g_sprintf(Closure->eccName,"%s.ecc",optarg);
		      g_sprintf(Closure->imageName,"%s.img",optarg);
		   }
	           break;
         case 'd': if(optarg) 
	           {  g_free(Closure->device);
	              Closure->device = g_strdup(optarg); 
	              break;
                   }
         case 'v': PrintCLI(_("\ndvdisaster version %s\n\n"), VERSION);
	 	   FreeClosure();
	           exit(EXIT_SUCCESS); 
	           break;

         case  0 : break; /* flag argument */

         case MODIFIER_ADAPTIVE_READ:
	   Closure->adaptiveRead = TRUE;
	   break;
         case MODIFIER_AUTO_SUFFIX:
	   Closure->autoSuffix = TRUE;
	   break;
         case MODIFIER_CACHE_SIZE:
	   Closure->cacheMB = atoi(optarg);
	   if(Closure->cacheMB <   1) 
	     Stop(_("--cache-size must at least be 1MB; 16MB or higher is recommended.")); 
	   if(Closure->cacheMB > 2048) 
	     Stop(_("--cache-size maximum is 2048MB.")); 
	   break;
         case MODIFIER_DAO: 
	   Closure->noTruncate = 1; 
	   break;
         case MODIFIER_FILL_UNREADABLE:
	   if(optarg) Closure->fillUnreadable = strtol(optarg, NULL, 0);
	   break;
         case MODIFIER_TRUNCATE: 
	   if(optarg)                  /* debugging truncate mode */
	   {  mode = MODE_TRUNCATE;
	      debug_arg = g_strdup(optarg);
	   }
	   else Closure->truncate = 1; /* truncate confirmation for fix mode */
	   break;
         case MODIFIER_KEEP_STYLE:
	   Closure->keepStyle = 1;
	   break;
         case MODIFIER_DEBUG:
	   Closure->debugMode = TRUE;
	   break;
         case MODIFIER_PARSE_UDF:
	   Closure->parseUDF = TRUE;
	   break;
         case MODIFIER_RANDOM_SEED:
	   if(optarg) Closure->randomSeed = atoi(optarg);
	   break;
         case MODIFIER_SPINUP_DELAY:
	   if(optarg) Closure->spinupDelay = atoi(optarg);
	   break;
         case MODIFIER_SPEED_WARNING:
	   if(optarg) Closure->speedWarning = atoi(optarg);
	   else Closure->speedWarning=10;
	   break;
         case MODIFIER_SPLIT_FILES:
	   Closure->splitFiles = 1;
	   break;
         case MODIFIER_CLV_SPEED:
	   Closure->driveSpeed = atoi(optarg);
	   break;
         case MODIFIER_CAV_SPEED:
	   Closure->driveSpeed = -atoi(optarg);
	   break;
         case MODE_SEND_CDB: 
	   mode = MODE_SEND_CDB;
	   debug_arg = g_strdup(optarg);
	   break;
         case MODE_ERASE: 
	   mode = MODE_ERASE;
	   debug_arg = g_strdup(optarg);
	   break;
         case MODE_RANDOM_ERR:
	   mode = MODE_RANDOM_ERR;
	   break;
         case MODE_RANDOM_IMAGE:
	   mode = MODE_RANDOM_IMAGE;
	   debug_arg = g_strdup(optarg);
	   break;
         case MODE_BYTESET:
	   mode = MODE_BYTESET;
	   debug_arg = g_strdup(optarg);
	   break;
         case MODE_SHOW_SECTOR:
	   mode = MODE_SHOW_SECTOR;
	   debug_arg = g_strdup(optarg);
	   break;
         case MODE_ZERO_UNREADABLE:
	   mode = MODE_ZERO_UNREADABLE;
	   break;
         case '?': mode = MODE_HELP; break;
         default: PrintCLI(_("?? illegal getopt return value %d\n"),c); break;
      }
   }

   /*** Don't allow debugging option if --debug wasn't given */
   
   if(!Closure->debugMode)
     switch(mode)
     {  case MODE_BYTESET:
        case MODE_ERASE:
        case MODE_SEND_CDB:
        case MODE_SHOW_SECTOR:
        case MODE_RANDOM_ERR:
        case MODE_RANDOM_IMAGE:
        case MODE_TRUNCATE:
        case MODE_ZERO_UNREADABLE:
	  mode = MODE_HELP;
          break;
     }
	  
   PrepareDeadSector();

   /*** Parse the sector ranges for --read and --scan */

   if(read_range)
   {  char *dashpos = strchr(read_range,'-');

      if(dashpos)
      {  *dashpos = 0;

         Closure->readStart = atoi(read_range);
	 if(strlen(dashpos+1) == 3 && !strncmp(dashpos+1, "end", 3))
	   Closure->readEnd = -1;
	 else Closure->readEnd = atoi(dashpos+1);
      }
      else Closure->readStart = Closure->readEnd = atoi(read_range);
      g_free(read_range);
   }

   /*** Apply auto suffixes 
        (--read etc. may be processed before --auto-suffix,
         so we must defer autosuffixing until all args have been read) */

   if(Closure->autoSuffix)
   {  Closure->eccName   = ApplyAutoSuffix(Closure->eccName, "ecc");
      Closure->imageName = ApplyAutoSuffix(Closure->imageName, "img");
   }

   /*** Dispatch action depending on mode.
        The major modes can be executed in sequence 
	(although not all combinations may be really useful) */

   switch(mode)
   {  case MODE_SEQUENCE:
	if(sequence & 1<<MODE_SCAN)
	  ReadMediumLinear((gpointer)1);

	if(sequence & 1<<MODE_READ)
	{  if(Closure->adaptiveRead) 
	        ReadMediumAdaptive((gpointer)0);
	   else ReadMediumLinear((gpointer)0);
	}

	if(sequence & 1<<MODE_CREATE)
	  CreateEcc();

	if(sequence & 1<<MODE_FIX)
	  FixEcc();

	if(sequence & 1<<MODE_PRINT)
	  Compare();
	break;

      case MODE_BYTESET:
         Byteset(debug_arg);
	 break;

      case MODE_ERASE:
         Erase(debug_arg);
	 break;

      case MODE_SEND_CDB:
	 SendCDB(debug_arg);
	 break;

      case MODE_SHOW_SECTOR:
	 ShowSector(debug_arg);
	 break;

      case MODE_RANDOM_ERR:
	 RandomError(Closure->imageName);
	 break;

      case MODE_RANDOM_IMAGE:
	 RandomImage(Closure->imageName, debug_arg);
	 break;

      case MODE_TRUNCATE:
	 TruncateImage(debug_arg);
	 break;

      case MODE_ZERO_UNREADABLE:
	 ZeroUnreadable();
	 break;

#ifdef SYS_MINGW
      case MODE_LIST_ASPI:
	 ListAspiDrives();
	 break;
#endif
      default:
	break;
   }

   if(debug_arg) g_free(debug_arg);

   /*** If no mode was selected, print the help screen. */

   if(mode == MODE_HELP)
   {  
#ifndef QUIET
     /* TRANSLATORS: Program options like -r and --read are not to be translated
	to avoid confusion when discussing the program in international forums. */
      PrintCLI(_("\nCommon usage examples:\n"
	     "  dvdisaster -r,--read   # Read the medium image to hard disc.\n"
	     "                         # Use -rn-m to read a certain sector range, e.g. -r100-200\n"
	     "  dvdisaster -c,--create # Create .ecc information for the medium image.\n"
	     "  dvdisaster -f,--fix    # Try to fix medium image using .ecc information.\n"
	     "  dvdisaster -s,--scan   # Scan the medium for read errors.\n"
	     "  dvdisaster -t,--test   # Test integrity of the .img and .ecc files.\n"
	     "  dvdisaster -u,--unlink # Delete .img files (when other actions complete)\n\n"));

      PrintCLI(_("Drive and file specification:\n"
	     "  -d,--device device     - read from given device   (default: %s)\n"
	     "  -p,--prefix prefix     - prefix of .img/.ecc file (default: medium.*  )\n"
	     "  -i,--image  imagefile  - name of image file       (default: medium.img)\n"
	     "  -e,--ecc    eccfile    - name of parity file      (default: medium.ecc)\n"),
	     Closure->device);

#ifdef SYS_MINGW
      PrintCLI(_("  -l,--list              - list drives available under ASPI manager\n\n"));
#else
      PrintCLI("\n");
#endif

      PrintCLI(_("Tweaking options (see manual before using!)\n"
	     "  -j,--jump n            - jump n sectors forward after a read error (default: 16)\n"
	     "  -n,--redundancy n%%     - sets redundancy for error correction (default: 14.3%%)\n"
	     "  --adaptive-read        - use optimized strategy for reading damaged media\n"
	     "  --auto-suffix          - automatically add .img and .ecc file suffixes\n"
	     "  --cache-size n         - image cache size in MB during -c mode (default: 32MB)\n"
	     "  --dao                  - assume DAO disc; do not trim image end\n"
	     "  --fill-unreadable n    - fill unreadable sectors with byte n\n"
	     "  --parse-udf            - use information from ISO/UDF filesystem\n"
	     "  --speed-warning n      - print warning if speed changes by more than n percent\n"
	     "  --spinup-delay n       - wait n seconds for drive to spin up\n"
	     "  --split-files          - split image into files <= 2GB\n\n"));

      PrintCLI(_("Graphical user interface options\n"
	     "  --keep-style           - do not override the style settings\n\n"));

      if(Closure->debugMode)
	PrintCLI(_("Debugging options (purposefully undocumented and possibly harmful)\n"
	     "  --debug           - enables the following options\n"
	     "  --byteset s,i,b   - set byte i in sector s to b\n"
	     "  --erase sector    - erase the given sector\n"
	     "  --erase n-m       - erase sectors n - m, inclusively\n"
	     "  --random-errors   - seed image with (correctable) random errors\n"
	     "  --random-image n  - create image with n sectors or random numbers\n"
	     "  --random-seed n   - random seed for above function\n"
 	     "  --send-cdb arg    - executes given cdb at drive; kills system if used wrong\n"  
	     "  --show-sector n   - shows hexdump of the given sector\n"
	     "  --truncate n      - truncates image to n sectors\n"
	     "  --zero-unreadable - replace the \"unreadable sector\" markers with zeros\n\n"));
#endif

      FreeClosure();
      exit(EXIT_FAILURE);
   }

   /* If no mode was selected at the command line, start the graphical user interface. */

   if(mode == MODE_NONE)
   {  
      Closure->guiMode = TRUE;
      ReadDotfile();
      CreateMainWindow(&argc, &argv);
   }
   
   FreeClosure();

   exit(exitCode);
}
