(module image mzscheme

  ;; The color struct:
  (require "private/color.ss")
  (provide (all-from "private/color.ss"))

  ;; Flag to indicate whether we've tried to load
  ;;  the MrEd-using image functions
  (define tried? #f)

  ;; This macro is used once to export all functions
  ;;  from "private/image.ss":
  (define-syntax (re-export stx)
    (syntax-case stx ()
      [(_ (id . impl) ...) 
       (with-syntax ([(got-id ...) (generate-temporaries #'(id ...))])
	 #'(begin 
	     (provide id ...)
	     (re-export-one get-procs! id got-id . impl) ...
	     (define got-id #f) ...
	     (define (get-procs!)
	       (unless tried?
		 (set! tried? #t)
		 (with-handlers ([not-break-exn? (lambda (x) #f)])
		   (set! got-id (dynamic-require '(lib "image.ss" "lang" "private") 'id))
		   ...)))))]))

  ;; This macro is used once for each individual function,
  ;;  and the use is generated by `re-export':
  (define-syntax (re-export-one stx)
    (syntax-case stx ()
      [(_ get-procs! id got-id args alt ...)
       (with-syntax ([alt (if (null? (syntax-e #'(alt ...)))
			      #'(raise-type-error 'id "image" 0 . args)
			      #'(alt ... . args))])
			      
	 #'(define (id . args)
	     (get-procs!)
	     (if got-id
		 (got-id . args)
		 alt)))]))

  ;; Each re-export is the name, formals for the argument,
  ;;  and if the first argument isn't required to be an image,
  ;;  a function to apply when the real version is unavailable
  ;;  (i.e., because there's no MrEd)
  (re-export (image? (x) (lambda (x) #f))
	     (image=? (a b))
	     
	     (image-width (a))
	     (image-height (a))
	     (image+ (a d))
	     (offset-image+ (a b c d))
	     (offset-masked-image+ (a b c d e))
	     
	     (filled-rect (w h c) (fake-p 'filled-rect))
	     (outline-rect (w h c) (fake-p 'outline-rect))
	     (filled-circle (w h c) (fake-p 'filled-circle))
	     (outline-circle (w h c) (fake-p 'outline-circle))
	     
	     (image-inside? (a b))
	     (find-image (a b))
	     
	     (image->color-list (a))
	     (color-list->image (l w h) fake-color-list->image))


  (define (check name p? v desc)
    (unless (p? v)
      (raise-type-error
       name
       desc
       v)))
  
  (define ((fake-p who) w h color)
    (check who real? w "positive integer")
    (check who real? h "positive integer")
    (check who symbol? color "symbol")
    (raise (make-exn:misc:unsupported 
	    (format "~a: not supported" who)
	    (current-continuation-marks))))

  (define (color-list? l)
    (and (list? l) (andmap color? l)))

  (define (posi? i)
    (and (number? i) (integer? i) (positive? i) (exact? i)))

  (define (fake-color-list->image cl w h)
    (check 'color-list->image color-list? cl "list-of-colors")
    (check 'color-list->image posi? w "positive exact integer")
    (check 'color-list->image posi? h "positive exact integer")
    (unless (and (< 0 w 10000) (< 0 h 10000))
      (error (format "cannot make ~a x ~a image" w h)))
    (unless (= (* w h) (length cl))
      (error (format "given width times given height is ~a, but the given color list has ~a items"
		     (* w h) (length cl))))
    (raise (make-exn:misc:unsupported 
	    "color-list->image: not supported"
	    (current-continuation-marks)))))
