/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * Copyright 2002-2005 Todd Kulesza
 * Copyright 2009 Neil Williams  <linux@codehelp.co.uk>
 *
 * Authors:
 * 		Todd Kulesza <todd@dropline.net>
 * 		Neil Williams  <linux@codehelp.co.uk>
 */

#include <config.h>

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <string.h>
#include <glib.h>
#include <libgnomevfs/gnome-vfs-cancellation.h>
#include <libsoup/soup.h>

#include "blog_advogato.h"
#include "blog_atom.h"
#include "blog_blogger.h"
#include "blog_lj.h"
#include "blog_mt.h"
#include "drivel.h"
#include "drivel_request.h"
#include "msg_queue.h"
#include "login.h"
#include "journal.h"
#include "xmlrpc.h"
#include "network.h"

#define USER_AGENT "GNOME-Drivel/" VERSION

typedef struct _LJChallenge LJChallenge;

extern GMutex *net_mutex;
extern gboolean verbose;
extern DrivelClient *dc;
static SoupSession *session = NULL;

/* Update the network progress dialog to explain to the user what is happening */

void
update_status_msg (DrivelRequestType type)
{
	guint id;
	gchar *msg;
	
	msg = g_strdup (_("Done"));
	switch (type)
	{
		case REQUEST_TYPE_LOGIN:
		{
			/* Translators: this particular string needs to be short
			 * in length. There isn't much room. */
			msg = g_strdup (_("Logging in . . "));
			break;
		}
		case REQUEST_TYPE_GETPICTURE:
		{
			msg = g_strdup (_("Downloading user pictures"));
			break;
		}
		case REQUEST_TYPE_POSTEVENT:
		{
			msg = g_strdup (_("Posting journal entry"));
			break;
		}
		case REQUEST_TYPE_EDITEVENT:
		{
			msg = g_strdup (_("Updating journal entry"));
			break;
		}
		case REQUEST_TYPE_GETEVENTS:
		{
			msg = g_strdup (_("Retrieving journal entries"));
			break;
		}
		case REQUEST_TYPE_GETDAYCOUNTS:
		{
			msg = g_strdup (_("Retrieving journal history"));
			break;
		}
		case REQUEST_TYPE_EDITFRIENDS:
		{
			msg = g_strdup (_("Updating Friends list"));
			break;
		}
		case REQUEST_TYPE_CHECKFRIENDS:
		case REQUEST_TYPE_GETFRIENDS:
		{
			msg = g_strdup (_("Retrieving Friends list"));
			break;
		}
		case REQUEST_TYPE_GETCATEGORIES:
		case REQUEST_TYPE_GETPOSTCATEGORIES:
		{
			msg = g_strdup (_("Retrieving categories"));
			break;
		}
		case REQUEST_TYPE_SETPOSTCATEGORIES:
		{
			msg = g_strdup (_("Setting categories"));
			break;
		}
		case REQUEST_TYPE_PUBLISH:
		{
			msg = g_strdup (_("Publishing journal entry"));
			break;
		}
		case REQUEST_TYPE_DELETEEVENT:
		{
			msg = g_strdup (_("Deleting journal entry"));
			break;
		}
		case REQUEST_TYPE_PING:
		{
			msg = g_strdup (_("Notifying Technorati"));
			break;
		}
		case REQUEST_TYPE_GETFRIENDGROUPS:
		{
			msg = g_strdup (_("Retrieving security groups"));
			break;
		}
		case REQUEST_TYPE_SETFRIENDGROUPS:
		{
			msg = g_strdup (_("Updating security groups"));
			break;
		}
		case REQUEST_TYPE_NONE:
			break;
		/* no default here, this is deliberate.
		 * Ensure each case is covered. */
	}
	
	if (dc->statusbar)
	{
		id = gtk_statusbar_get_context_id (dc->statusbar, PACKAGE);
		gtk_statusbar_push (dc->statusbar, id, msg);
	}
	g_free (msg);
	return;
}

static void
authenticate_cb (SoupSession *session, SoupMessage *msg,
		 SoupAuth *auth, gboolean retrying,
		 gpointer user_data)
{
	debug ("setting username and password for HTTP_BASIC auth");

	if (!retrying)
		soup_auth_authenticate (auth, dc->user->username, dc->user->password);

	return;
}

/* If the user has setup a proxy, return a SoupURI to it */
static SoupURI *
get_proxy_uri (DrivelClient *dc)
{
	SoupURI *proxy_uri = NULL;

	if (dc->proxy && dc->proxy_url)
	{
		debug ("Setting up proxy server:");
		debug (dc->proxy_url);
		if (!dc->proxy_port)
			dc->proxy_port = 8080;
		
		proxy_uri = soup_uri_new (dc->proxy_url);
		soup_uri_set_port (proxy_uri, dc->proxy_port);
	}

	return proxy_uri;
}

static gboolean
clear_progress (gpointer data)
{
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR(dc->progressbar), 0.0);
	return FALSE;
}

static void
wrote_headers_cb (SoupMessage *msg, gpointer data)
{
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR(dc->progressbar), 0.05);
}

static void
got_headers_cb (SoupMessage *msg, gpointer data)
{
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR(dc->progressbar), 0.05);
}

static void
wrote_chunk_cb (SoupMessage *msg, gpointer data)
{
	static gint content_length = 0;
	static gint bytes_transferred = 0;

	/* FIXME: This doesn't actually work; the request body will
	 * always have the full length, and wrote_chunk is only
	 * emitted when using chunked encoding anyway, which drivel
	 * doesn't do (and which many servers don't support for
	 * requests anyway).
	 */

	if (bytes_transferred >= content_length)
	{
		content_length = soup_message_headers_get_content_length (msg->request_headers);
		bytes_transferred = msg->request_body->length;
	}
	else
	{
		bytes_transferred += msg->request_body->length;
	}

	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR(dc->progressbar),
		bytes_transferred / content_length);
	g_timeout_add (3000, clear_progress, NULL);

	return;
}

static void
got_chunk_cb (SoupMessage *msg, SoupBuffer *chunk, gpointer data)
{
	static gint content_length = 0;
	static gint bytes_transferred = 0;

	if (bytes_transferred >= content_length)
	{
		content_length = soup_message_headers_get_content_length (msg->response_headers);
		bytes_transferred = chunk->length;
	}
	else
	{
		bytes_transferred += chunk->length;
	}

	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR(dc->progressbar),
		bytes_transferred / content_length);
	g_timeout_add (3000, clear_progress, NULL);

	return;
}

/* Add a SoupMessage to the queue for processing */
void
net_enqueue_msg (SoupMessage *msg)
{
	g_return_if_fail (msg);

	/* Add the User-Agent header to every request */
	soup_message_headers_append (msg->request_headers, "User-Agent", USER_AGENT);

	/* Add the ljfastserver cookie if we have a paid account */
	if (dc->net->fast_servers)
		soup_message_headers_append (msg->request_headers, "Cookie", "ljfastserver=1");
	
	g_signal_connect (G_OBJECT (msg), "wrote-headers",
			  G_CALLBACK (wrote_headers_cb), NULL);
	g_signal_connect (G_OBJECT (msg), "got-headers",
			  G_CALLBACK (got_headers_cb), NULL);
	g_signal_connect (G_OBJECT (msg), "wrote-chunk",
			  G_CALLBACK (wrote_chunk_cb), NULL);
	g_signal_connect (G_OBJECT (msg), "got-chunk",
			  G_CALLBACK (got_chunk_cb), NULL);

	/* Queue up the message */
	soup_session_queue_message (session, msg, NULL, NULL);
	
	return;
}

void
net_requeue_msg (SoupMessage *msg)
{
	g_return_if_fail (msg);

	soup_session_requeue_message (session, msg);

	return;
}

/* Check for an error response */
SoupXmlrpcResponse*
net_msg_get_response (SoupXmlrpcMessage *msg)
{
	SoupXmlrpcResponse *response = NULL;

	debug ("net_msg_get_response");
	if (!SOUP_STATUS_IS_SUCCESSFUL (SOUP_MESSAGE (msg)->status_code))
	{
		display_error_dialog (dc, _("Server error"), _("Network connection failed"));
	}
	else
	{
		if (!(response = soup_xmlrpc_message_parse_response (msg)))
		{
			display_error_dialog (dc, _("Server error"), _("Could not understand server response"));
		}
		else
		{
			/* If the response is a fault, display the error */
			if (soup_xmlrpc_response_is_fault (response))
			{
				SoupXmlrpcValue *value;
				GHashTable *table;
				gchar *error;
				glong drupal6;

				value = soup_xmlrpc_response_get_value (response);
				soup_xmlrpc_value_get_struct (value, &table);
				value = g_hash_table_lookup (table, "faultString");
				soup_xmlrpc_value_get_string (value, &error);
				value = g_hash_table_lookup (table, "faultCode");
				soup_xmlrpc_value_get_int (value, &drupal6);
				if (drupal6 == 1)
				{
					/* handle another drupal6 bug */
					if (verbose)
						print_response_xmlrpc (response);
					g_message ("Drupal bug: faultCode: %ld. faultString: %s", drupal6, error);
					update_status_msg (REQUEST_TYPE_NONE);
					g_free (error);
					g_hash_table_destroy (table);
					g_object_unref (response);
					response = NULL;
					return NULL;
				}
				if (drupal6 <= -32601 && drupal6 >= -32610)
				{
					/* handle a drupal6 bug */
					if (verbose)
						print_response_xmlrpc (response);
					g_message ("Drupal bug: faultCode: %ld. faultString: %s", drupal6, error);
					update_status_msg (REQUEST_TYPE_NONE);
					g_free (error);
					g_hash_table_destroy (table);
					g_object_unref (response);
					response = NULL;
					return NULL;
				}
				display_error_dialog (dc, _("Server error"), error);

				g_free (error);
				g_hash_table_destroy (table);
				g_object_unref (response);
				response = NULL;
			}
		}
	}

	update_status_msg (REQUEST_TYPE_NONE);
	return response;
}

/* Initialize the global SoupSession */
gint
net_start_session (void)
{
	SoupURI *proxy;

	g_return_val_if_fail (!session, -1);

	proxy = get_proxy_uri (dc);
	session = soup_session_async_new_with_options (SOUP_SESSION_PROXY_URI,
						       proxy,
						       SOUP_SESSION_USER_AGENT,
						       USER_AGENT,
						       NULL);
	if (proxy)
		soup_uri_free (proxy);
	g_signal_connect (G_OBJECT (session), "authenticate",
			  G_CALLBACK (authenticate_cb), NULL);

	return 0;
}

/* send a ping to technorati */
/* void */
/* net_ping_technorati (DrivelClient *dc) */
/* { */
/* 	DrivelRequest *dr; */
/* 	gchar *packet; */
/* 	const gchar *name; */
	
/* 	debug ("net_ping_technorati ()"); */
	
/* 	if (!gconf_client_get_bool (dc->client, dc->gconf->technorati, NULL)) */
/* 		return; */
	
/* 	/\* Use the LiveJournal description as the Technorati title for LJ users *\/ */
/* 	if ((dc->user->api == BLOG_API_LJ) &&  */
/* 		(dc->active_journal->type == JOURNAL_TYPE_USER)) */
/* 		name = dc->active_journal->description; */
/* 	else */
/* 		name = dc->active_journal->name; */
	
/* 	packet = xmlrpc_build_packet ("weblogUpdates.ping",  */
/* 			XMLRPC_TYPE_STRING, name, */
/* 			XMLRPC_TYPE_STRING, dc->active_journal->uri_view, */
/* 			-1); */
	
/* 	dr = drivel_request_new_with_items (REQUEST_TYPE_PING, */
/* 			REQUEST_PROTOCOL_POST,  */
/* 			BLOG_API_GENERIC,  */
/* 			"http://rpc.technorati.com/rpc/ping", */
/* 			g_strdup ("xml"), packet, */
/* 			NULL); */
	
/* /\* 	net_enqueue_request (dc, dr); *\/ */
	
/* 	return; */
/* } */
