#!/usr/bin/env bash

######################################################################
### Common settings/variables/functionality used by the testcases  ###
######################################################################

set -eo pipefail

function Log 
{
  LogLevel=$1
  LogMessage=$2

  echo "$LOG_SOURCE [$LogLevel] $(date '+%H:%M:%S'): $LogMessage";
}

function LogDebug 
{
  LogMessage=$1
  Log "DEBUG" "$LogMessage";
}

function LogInfo 
{
  LogMessage=$1
  Log "INFO" "$LogMessage";
}

function LogWarning 
{
  LogMessage=$1
  Log "WARNING" "$LogMessage";
}

function LogErrorAndTerminate
{
  LogMessage=$1
  Log "ERROR" "$LogMessage" 1>&2;
  
  # makes shure that the shell terminates even from a subshell
  kill -SIGPIPE "$$" 
}

function GetUbuntuVersionIdentifier 
{
  echo "$(source /etc/os-release && echo "$VERSION_ID")";
}

function GetSystemArchitecture
{
  ARCH="$(uname -i)";

  case "$ARCH" in 
    x86_64)
      echo "x64";
      ;;
    aarch64) 
      echo "arm64";
      ;;
    *) 
      LogErrorAndTerminate "Unknown/Unsupported architecture '$ARCH'.";
      ;; 
  esac
}

function GetDotnetRuntimeIdentifier
{
  echo "ubuntu.$(GetUbuntuVersionIdentifier)-$(GetSystemArchitecture)";
}

function GetSdkVersion
{
  dotnet --version;
}

function SetupEnvironmentVariables
{
  LogInfo "Setup Environment Variables"
  
  export DOTNET_SDK_VERSION="$(GetSdkVersion)";
  LogDebug "DOTNET_SDK_VERSION = $DOTNET_SDK_VERSION";

  export DOTNET_MAJOR_VERSION_NUMBER="$(echo "$DOTNET_SDK_VERSION" | cut -d '.' -f 1)"
  LogDebug "DOTNET_MAJOR_VERSION_NUMBER = $DOTNET_MAJOR_VERSION_NUMBER";

  export DOTNET_MINOR_VERSION_NUMBER="$(echo "$DOTNET_SDK_VERSION" | cut -d '.' -f 2)"
  LogDebug "DOTNET_MINOR_VERSION_NUMBER = $DOTNET_MINOR_VERSION_NUMBER";

  export DOTNET_REVISION_VERSION_NUMBER="$(echo "$DOTNET_SDK_VERSION" | cut -d '.' -f 3 | cut -d '-' -f 1)"
  LogDebug "DOTNET_REVISION_VERSION_NUMBER = $DOTNET_REVISION_VERSION_NUMBER";

  export DOTNET_PRERELEASE_SUFFIX="$(echo "$DOTNET_SDK_VERSION" | cut -d- -f 2)"
  LogDebug "DOTNET_PRERELEASE_SUFFIX = $DOTNET_PRERELEASE_SUFFIX";

  export DOTNET_VERSION_NAME="net$DOTNET_MAJOR_VERSION_NUMBER.$DOTNET_MINOR_VERSION_NUMBER"
  LogDebug "DOTNET_VERSION_NAME = $DOTNET_VERSION_NAME";

  DOTNET_RUNTIME_REVISION_NUMBER=$(echo "$DOTNET_REVISION_VERSION_NUMBER" | grep -Po '(?<=^1)[1-9][0-9]$|(?<=^10)[0-9]$')
  export DOTNET_RUNTIME_VERSION="${DOTNET_MAJOR_VERSION_NUMBER}.${DOTNET_MINOR_VERSION_NUMBER}.${DOTNET_RUNTIME_REVISION_NUMBER}"
  LogDebug "DOTNET_RUNTIME_VERSION = $DOTNET_RUNTIME_VERSION";

  export DOTNET_RUNTIME_IDENTIFIER="$(GetDotnetRuntimeIdentifier)"
  LogDebug "DOTNET_RUNTIME_IDENTIFIER = $DOTNET_RUNTIME_IDENTIFIER";

  export DOTNET_LANGUAGES=("C#" "F#" "VB")
  LogDebug "$(echo "DOTNET_LANGUAGES = (${DOTNET_LANGUAGES[@]})")";

  # supress welcome banner on first run of a dotnet command
  export DOTNET_NOLOGO=true 
  LogDebug "DOTNET_NOLOGO = $DOTNET_NOLOGO";
  export DOTNET_SKIP_FIRST_TIME_EXPERIENCE=true 
  LogDebug "DOTNET_SKIP_FIRST_TIME_EXPERIENCE = $DOTNET_SKIP_FIRST_TIME_EXPERIENCE";
}

function ShowDotnetDebugInfo
{
  LogDebug "Show Debug Infos"
  echo "$ dotnet --info"
  dotnet --info
}

function ChangeToAutopkgtestTmpFolder
{
  # to run the test without the autopkgtest runner
  if [[ -z $AUTOPKGTEST_TMP ]]; then
    export AUTOPKGTEST_TMP="$(mktemp -d /tmp/autopkgtest.XXXXXX)"
    LogDebug "Created autopkgtest temporary folder '$AUTOPKGTEST_TMP'."
  fi

  cd "$AUTOPKGTEST_TMP"
  LogInfo "Changed to autopkgtest temporary folder '$AUTOPKGTEST_TMP'."
}

# Set LOG_SOURCE to this script
export LOG_SOURCE='debian/tests/.tests.rc'

SetupEnvironmentVariables
ShowDotnetDebugInfo

# Set LOG_SOURCE to the script that called this script. This assumes that this 
# script gets called by a test inside the debian/tests folder.
export LOG_SOURCE="$(echo $0 | grep -oE "debian/.+$")"
