/* doscan - Denial Of Service Capable Auditing of Networks
 * Copyright (C) 2003 Florian Weimer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * This protocol module scans for hosts which are vulnerable to
 * CAN-2003-0717, (Microsoft Security Bulletin MS03-043, "Buffer
 * Overrun in Messenger Service Could Allow Code Execution").
 *
 * The detection code is based on Nessus plugin 11890:
 *
 *   # This script is (C) Tenable Network Security
 *   # 10/22/2003 updated by KK Liu 10/22/2003
 *   #  - check messenger service, if not on - exit
 *   #  - check Windows OS
 */

#include "config.h"
#include "event_queue.h"
#include "scan_udp.h"
#include "opt.h"
#include "proto.h"
#include "results.h"

#include <cstdio>
#include <cstdlib>
#include <memory>
#include <string>
#include <unistd.h>

static bool ms_start (subnets&);
static void ms_open (scan_host_t *);

void
proto_ms_03_043_register (void)
{
  proto_register ("ms_ms03-043", ms_start, ms_open);
}

static void ms_open (scan_host_t *)
{
}

struct scan_ms_03_043 : scan_udp_single::handler {
  scan_ms_03_043* clone() const
  {
    return new scan_ms_03_043(*this);
  }

  void create_query(ipv4_t host, std::string& data)
  {
    // Apparently, Microsoft Windows ignores retransmits with the same
    // sequence number.  Therefore, we use a different random value
    // for each request.

    unsigned seq = random();

    data.clear();
    data.reserve(200);

    static const char ch_part1[] = {
    0x04, 0x00, 0x28, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xf8, 0x91, 0x7b, 0x5a, 0x00, 0xff, 0xd0, 0x11, 0xa9, 0xb2, 0x00, 0xc0,
    0x4f, 0xb6, 0xe6, 0xfc, 0x04, 0x00,
    };
    static const std::string part1(ch_part1, sizeof(ch_part1));

    data += part1;
    data += seq & 0xff;
    data += (seq >> 8) & 0xff;

    static const char ch_part2[] = {
      0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e,
      0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    };
    static const std::string part2(ch_part2, sizeof(ch_part2));

    data += part2;
    data += seq & 0xff;
    data += (seq >> 8) & 0xff;

    static const char ch_part3[] = {
      0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x34, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00,
      0x00, 0x00, 'T', 'E', 'N', 'A', 'B', 'L', 'E', 0x00, 0x01, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
      0x00, 't', 's', 't', 0x00,
    };
    static const std::string part3 (ch_part3, sizeof(ch_part3));

    data += part3;
  }

  void reply_received(ipv4_t host, unsigned short remote_port,
                      const std::string& data)
  {
    if (data.size() < 4) {
      results_add(ticks_get_cached(), host, RESULTS_ERROR_NOMATCH,
                  data.data(), data.size());
    }

    const std::string::const_iterator p = data.end() - 4;
    const std::string error_code(p, data.end());
    static const std::string vulnerable("\xf7\x06\x00\x00", 4);
    static const std::string not_vulnerable("\x03\x00\x01\x1c", 4);
    static const std::string unix_error_code("\034\001\000\001", 4);

    if (error_code == vulnerable) {
      static const char msg[] = "vulnerable to MS03-043";
      results_add(ticks_get_cached(), host, 0, msg, sizeof(msg) - 1);
    } else if (error_code == not_vulnerable) {
      if (opt_verbose) {
        static const char msg[] = "not vulnerable to MS03-043";
        results_add(ticks_get_cached(), host, 0, msg, sizeof(msg) - 1);
      }
    } else if (error_code == unix_error_code) {
      if (opt_verbose) {
        static const char msg[] = "UNIX DCE RPC implementation";
        results_add(ticks_get_cached(), host, 0, msg, sizeof(msg) - 1);
      }
    } else {
      results_add(ticks_get_cached(), host, 0, &*p, 4);
    }
  }

  void timeout(ipv4_t host, ticks_t ticks)
  {
  }
};

static bool
ms_start (subnets& nets)
{
  if (opt_receive && opt_receive[0]) {
    fprintf (stderr, "%s: --receive is not supported by this module\n",
             opt_program);
    exit (EXIT_FAILURE);
  }

  if (opt_banner_size) {
    fprintf (stderr, "%s: --banner is not supported by this module.\n",
             opt_program);
    exit (EXIT_FAILURE);
  }

  if (opt_send && (opt_send[0] != 0)) {
    fprintf (stderr, "%s: --send is not supported by this module.\n",
             opt_program);
    exit (EXIT_FAILURE);
  }

  unsigned seed = getpid() << 8;
  seed ^= time(0);
  srandom(seed);

  std::auto_ptr<event_queue> q(event_queue::create(1));
  scan_ms_03_043 ms;
  scan_udp_single scanner(*q, nets, ms, opt_fd_count,
                          opt_add_timeout, opt_add_burst,
                          5, opt_connect_timeout / 5);

  q->run();

  return false;
}

// arch-tag: fea5e549-9743-4179-8a6a-be28a63b8225
