import java.io.*;
import java.util.*;
import gnu.regexp.*;

/**
 * The main class for generating sitemap-style indices. 
 * 
 * <br><br>$Id: SitemapGen.java,v 1.32 2002/09/24 14:32:49 howama Exp $
 * 
 * @author Mark Howard
 */
public class SitemapGen extends AnyIndexGen{
    /** The log file */
    private IGLog log;
    private static final int LOGLEVEL =9;
    /** Project data */
    private Project proj;
    /** File List object */
    private FileList list;
    /** The root of the directories tree */
    private IGDirectory treeRoot;
    /** Replacer for the smaller strings */
    private IGReplace replacer;

    private String output;
    /**
     * Constructor will set up links to various objects.
     *
     * Assumptions: 
     *   - Project settings have been loaded
     *   - Log has been started
     *   
     * @param iglog		An instantiated log object
     * @param projData 	Project data object
     */
    SitemapGen(IGLog iglog, Project projData) throws REException,
	    IllegalVariableException {
        log = iglog;
        proj = projData;
        log.add(9, "SitemapGen Object Created");
        log.add(IGLog.VERSION, "SitemapGen.java $Revision: 1.32 $");
        output = proj.getStringSetting("OUTPUT_FILENAME");

	replacer = new IGReplace(log, proj);
    }

    /**
     * Starts the generation process.
     */
    public void generate() throws IllegalVariableException, IOException,
	    REException, org.apache.oro.text.regex.MalformedPatternException {
        if (LOGLEVEL > 8){
            log.add(IGLog.PROCEDURE, "start of sitemap generate method");
        }
        list = new FileList(log, proj);

        // Add the files to the list
        IGKeySet wishlist = IGReplace.getWanted(log, proj);
	list.makeList(wishlist);

        // Generate a tree containing the data in a more useful format
        generateTree();

        // Write the output
        writeOutput();
        log.add(IGLog.PROGRESS, "End of Sitemap Generation");
    }

    /**
     * Writes the data to the output file.
     */
    private void writeOutput() throws REException, IOException,
	    IllegalVariableException{
        log.add(IGLog.PROCEDURE, "SitemapGen: writeOutput");
		IGLog.ui.startOutputGenerate();

	if (! outFileAcceptable(log, output)) {
	    return;
	}

	BufferedWriter out = new BufferedWriter(new FileWriter(output));
	out.write(replacer.makeCode("SM_START_TXT"));
	String[] curdir = null;
	IGReplace linkReplacer = new IGReplace(log, proj,
		proj.getStringSetting("SM_LINKS_TXT"), IGKey.REPLACE_TYPE_FILE,
		"SM_LINKS_TXT");
	IGReplace dirEnterReplacer = new IGReplace(log, proj,
		proj.getStringSetting("SM_ENTER_FOLDER"),
		IGKey.REPLACE_TYPE_DIRECTORY, "SM_ENTER_FOLDER");
	IGReplace dirLeaveReplacer = new IGReplace(log, proj,
		proj.getStringSetting("SM_LEAVE_FOLDER"),
		IGKey.REPLACE_TYPE_DIRECTORY, "SM_LEAVE_FOLDER");

	IGDirectory start = treeRoot;
	while (start.files.isEmpty() && start.subDirectories.size() == 1) {
	    start = (IGDirectory) start.subDirectories.get(
		    start.subDirectories.firstKey());
	}

	if (start.files.isEmpty() &&  start.subDirectories.size() < 1) {
	    log.addWarning(110, "EMPTY_INDEX", null);
	} else {
	    writeDirectory("START", start, out, linkReplacer, dirEnterReplacer,
		    dirLeaveReplacer);
	}

	out.write(replacer.makeCode("SM_END_TXT"));
	out.close();
    }


    /**
     * Writes one directory to output including files in that directory and,
     * recursively, the subdirectories.
     * @param workingDir the relative directory or url to go before the current
     * directory name in all links
     * @param directory the directory to be written to output
     * @param out where the output should go
     */
    private void writeDirectory( String workingDir,
                                 IGDirectory directory,
                                 BufferedWriter out,
                                 IGReplace linkReplacer,
                                 IGReplace dirEnterReplacer,
                                 IGReplace dirLeaveReplacer
                               ) throws IOException{
        log.add(IGLog.PROCEDURE, "writeDirectory "+directory.topDir);
        out.write( dirEnterReplacer.makeCode(directory) );

        while (!directory.files.isEmpty()){
            IGFile current = (IGFile) directory.files.first();
            directory.files.remove( current );
            out.write( linkReplacer.makeCode( current ) );
        }
        Object[] subDirs = directory.subDirectories.values().toArray();
        for (int i = 0; i<subDirs.length; i++){
            writeDirectory(workingDir+directory.topDir, (IGDirectory) subDirs[i], out, linkReplacer, dirEnterReplacer, dirLeaveReplacer);
        }
        out.write( dirLeaveReplacer.makeCode(directory) );
    }

    /**
     * Uses the file list to generate a tree of IGDirectory classes for
     * temporarily storing references to the file details, in the same format as
     * the output will be
     * @see IGDirectory
     */
    private void generateTree(){
        log.add(IGLog.PROCEDURE, "Sitemap: GenerateTree" );
        treeRoot = new IGDirectory("", "");
        ListIterator currentIter = list.getList().listIterator(0);
        //		log.add(9, "i+curriter - "+ (IGFile) currentIter.next() );
        while (currentIter.hasNext()){
            IGDirectory currentDir = treeRoot;
            //		log.add(9, "dir"+currentDir);
            IGFile current = (IGFile) currentIter.next();
            log.add(9, "Adding file to structure "+ current.getLocation());
            for (int i = 0; i < (((String[])current.get( IGKey.FILE_PARTS )).length - 1); i++){
                if (currentDir.subDirectories.containsKey( ((String[]) current.get(IGKey.FILE_PARTS ))[i] )){
                    currentDir = (IGDirectory) currentDir.subDirectories.get( ((String[])current.get(IGKey.FILE_PARTS))[i] );
                }else{
                    // TODO: fix 2nd param
                    IGDirectory newDir = new IGDirectory(
                                             ((String[])current.get(IGKey.FILE_PARTS))[i],
                                             IGMisc.relative( currentDir.fullRelDir + '/' + ((String[])current.get(IGKey.FILE_PARTS))[i], output ));
                    currentDir.subDirectories.put( ((String[])current.get(IGKey.FILE_PARTS))[i], newDir );
                    currentDir = newDir;
                }
            }
            currentDir.files.add( current ) ;
        };
    }

}

/*
 * $Log: SitemapGen.java,v $
 * Revision 1.32  2002/09/24 14:32:49  howama
 * various changes, bugfixes, etc
 *
 * Revision 1.31  2002/09/22 23:13:29  blsecres
 * Propagated exceptions that were unnecessarily caught.
 *
 * Revision 1.30  2002/09/11 13:22:25  howama
 * bug fixes
 *
 * Revision 1.29  2002/09/07 15:44:59  howama
 * bug fix. now detects invalid directories
 *
 * Revision 1.28  2002/09/05 14:01:41  howama
 * new log methods for erros
 *
 * Revision 1.27  2002/09/04 20:08:49  howama
 * user interface work
 *
 * Revision 1.26  2002/09/01 10:59:29  howama
 * More work and random bug fixes.
 *
 * Revision 1.25  2002/08/31 22:00:21  blsecres
 * Provided fallbacks for missing ordered list sort criteria.
 * Implemented insignificant article removal for sorting purposes.
 * Modified parsers to only provide information asked for in IGKeySet.
 *
 * Revision 1.24  2002/08/31 15:27:57  howama
 * various changes
 *
 * Revision 1.23  2002/08/30 02:27:07  blsecres
 * Initial ordered list support.
 * User configurable sorting.
 *
 * Revision 1.22  2002/08/28 06:33:49  howama
 * various
 *
 * Revision 1.21  2002/08/23 13:58:01  howama
 * propogated illegalvariable exceptions rather than catching them
 *
 * Revision 1.20  2002/08/23 12:22:07  howama
 * fixed
 *
 * Revision 1.19  2002/08/23 10:41:58  howama
 * more work
 *
 * Revision 1.18  2002/08/21 15:13:12  howama
 * more work
 *
 * Revision 1.17  2002/08/20 13:22:31  howama
 * more work
 *
 * Revision 1.16  2002/08/20 09:18:20  howama
 * various
 *
 * Revision 1.15  2002/08/19 09:47:27  howama
 * updated placeholders and replcement code
 *
 * Revision 1.14  2002/08/18 14:15:47  howama
 * upodates
 *
 * Revision 1.13  2002/08/12 11:05:48  howama
 * changes IGDirectory structure
 *
 * Revision 1.12  2002/08/11 19:50:45  howama
 * various changes. Now uses IGDirectory objects rather than sort methods.
 *
 * Revision 1.11  2002/08/10 15:03:23  howama
 * various
 *
 * Revision 1.10  2002/08/09 18:14:33  howama
 * log items
 *
 * Revision 1.9  2002/08/09 16:07:49  howama
 * various changes
 *
 * Revision 1.8  2002/08/09 13:02:39  howama
 * more code
 *
 * Revision 1.7  2002/08/08 20:10:57  howama
 * updates
 *
 * Revision 1.6  2002/08/08 17:24:43  howama
 * updates
 *
 * Revision 1.5  2002/07/24 09:24:21  howama
 * updated javadoc docs
 *
 * Revision 1.4  2002/07/12 14:07:30  howama
 * updates
 *
 * Revision 1.3  2002/07/11 18:20:10  howama
 * Now extends anyIndexGen base class
 *
 * Revision 1.2  2002/07/05 10:52:01  howama
 * Added log tag
 * 
 */
