"This module provides plotting functionality (wrapper for Viper)."

__author__ = "Joachim B. Haga <jobh@broadpark.no>"
__date__ = "2008-03-05"
__copyright__ = "Copyright (C) 2008 Joachim B. Haga"
__license__  = "GNU LGPL Version 2.1"

# Modified by Martin Sandve Alnaes, 2008.
# Modified by Anders Logg, 2008-2009.
# Last changed: 2010-06-01

import os
import dolfin.cpp as cpp
from donothing import DoNothing
from dolfin.function.function import Function

__all__ = ['Viper', 'plot', 'update', 'interactive', 'save_plot', 'figure']

# Intelligent plot command that handles projections
def dolfin_plot(object, *args, **kwargs):
    "Plot given object (Mesh, MeshFunction or Function)"

    from viper.viper_dolfin import plot as viper_plot

    # Plot mesh
    if isinstance(object, cpp.Mesh):
        return viper_plot(object, *args, **kwargs)

    # Plot mesh function
    if isinstance(object, cpp.MeshFunctionUInt) or \
       isinstance(object, cpp.MeshFunctionInt)  or \
       isinstance(object, cpp.MeshFunctionDouble):
        return viper_plot(object, *args, **kwargs)

    # Plot function
    if isinstance(object, cpp.Function):
        return viper_plot(object, *args, **kwargs)

    # Plot expression
    if isinstance(object, cpp.Expression):
        if "mesh" not in kwargs:
            raise TypeError, "expected a mesh when plotting an expression."
        return viper_plot(object, *args, **kwargs)

    # Plot function plot data
    if isinstance(object, cpp.FunctionPlotData):
        return viper_plot(object, *args, **kwargs)

    # Plot boundary condition
    if isinstance(object, cpp.DirichletBC):
        bc = object
        V = bc.function_space()
        v = Function(V)
        bc.apply(v.vector())
        return viper_plot(v, *args, **kwargs)

    # Try projecting function
    from dolfin.fem.project import project
    try:
        print "Object cannot be plotted directly, projecting to piecewise linears."
        u = project(object)
        return plot(u, *args, **kwargs)
    except:
        raise RuntimeError, ("Don't know how to plot given object and projection failed: " + str(object))

# Check DOLFIN_NOPLOT
do_nothing = False
if os.environ.has_key('DOLFIN_NOPLOT'):
    cpp.info("DOLFIN_NOPLOT set, plotting disabled.")
    do_nothing = True
else:

    # Check for Viper
    try:
        for x in __all__:
            exec ('from viper.viper_dolfin import %s' % x)
        plot = dolfin_plot
    except ImportError, details:
        cpp.info(str(details))
        cpp.warning("Unable to import Viper, plotting disabled.")
        do_nothing= True

# Ignore all plot calls
if do_nothing:
    for x in __all__:
        exec('%s = DoNothing' % x)
