"""
Energy norm error estimators based on ErrorEstimatorBase.

"""

__author__ = "Marie E. Rognes (meg@simula.no)"
__copyright__ = "Copyright (C) 2009 - Marie E. Rognes"
__license__  = "GNU GPL version 3 or any later version"

# First added:  2009-10-20
# Last changed: 2010-05-09
#
# Modified by Anders Logg, 2009.

__all__ = ["EnergyNormEstimator", "BankWeiserEstimator"]

import numpy
from math import sqrt

from ufl.algorithms import preprocess

from dolfin import FunctionSpace, TestFunction, CellSize, Function
from dolfin import avg, inner, dx, ds, dS, assemble, info, lhs, replace

from dolfin.adaptivity.errorestimator import ErrorEstimatorBase
from dolfin.adaptivity.residual import compute_residual_representation, bank_weiser


class EnergyNormEstimator(ErrorEstimatorBase):
    """
    \eta_T = h_T^2 ||R_T||_T + h^T ||R_dT||_dT

    """

    def __init__(self, F, parameters=None):
        ErrorEstimatorBase.__init__(self, F, parameters)
        self.error_indicators = None

    def estimate_error(self, u_h):

        # Compute residual representation
        (R_T, R_dT) = compute_residual_representation(self.F, u_h)

        # Define error indicator form
        mesh = R_T.function_space().mesh()
        Constants = FunctionSpace(mesh, "DG", 0)
        v = TestFunction(Constants)

        h = CellSize(mesh)

        # FIXME: Need to be more careful with the domain boundary
        # terms here
        form = v*h*h*inner(R_T, R_T)*dx + v*h*inner(R_dT, R_dT)*ds \
               + 2*avg(v)*avg(h)*inner(avg(R_dT), avg(R_dT))*dS

        # Assemble (squared) error indicators
        indicators = assemble(form).array()

        # Sum indicators to obtain error
        error = sqrt(sum(indicators))

        # Take square root of indicators
        error_indicators = Function(Constants)
        error_indicators.vector()[:] = numpy.sqrt(indicators)

        # Store indicators for later use
        self.error_indicators = error_indicators

        return error

    def assemble_error_indicators(self, u_h):
        info("Computing error indicators")
        if self.error_indicators is None:
            self.estimate_error(u_h)
        return self.error_indicators


    def __str__(self):
        return "residual"

class BankWeiserEstimator(ErrorEstimatorBase):
    """
    Residual based error estimator based on local problems solved on
    each cell
    """

    def __init__(self, F, parameters=None):
        ErrorEstimatorBase.__init__(self, F, parameters)
        self.error_indicators = None

    def estimate_error(self, u_h):

        # Compute residual representation
        (R_T, R_dT) = compute_residual_representation(self.F, u_h)

        # Compute bank_weiser representation
        R = bank_weiser(self.F, R_T, R_dT)

        # Define piecewise constant test function
        Constants = FunctionSpace(R_T.function_space().mesh(), "DG", 0)
        w = TestFunction(Constants)

        # FIXME: (Bi)linear only
        a = lhs(self.F)

        preprocessed = preprocess(a)
        (v, u) = preprocessed.form_data().original_arguments
        a = replace(a, {v: w*R, u:R})

        # Assemble indicators
        indicators = assemble(a)

        # Sum indicators to obtain error
        error = sqrt(sum(indicators))

        # Take square root of indicators
        error_indicators = Function(Constants)
        error_indicators.vector()[:] = numpy.sqrt(indicators)

        # Store indicators for later use
        self.error_indicators = error_indicators

        return error

    def assemble_error_indicators(self, u_h):
        info("Computing error indicators")
        if self.error_indicators is None:
            self.estimate_error(u_h)
        return self.error_indicators

    def __str__(self):
        return "bank_weiser"
