// Copyright (C) 2008 Kent-Andre Mardal.
// Licensed under the GNU LGPL Version 2.1.
//
// First added:  2008-08-25
// Last changed: 2009-08-11
//
// Modified by Anders Logg, 2008.

#include <stdexcept>
#include <iostream>
#include <cmath>
#include <climits>
#include <dolfin/common/utils.h>
#include "Vector.h"
#include "BlockVector.h"

using namespace dolfin;

//-----------------------------------------------------------------------------
BlockVector::BlockVector(uint n_, bool owner_): owner(owner_), n(n_)
{
  vectors = new Vector*[n];
  if (owner)
  {
    for (uint i = 0; i < n; i++)
      vectors[i] = new Vector();
  }
}
//-----------------------------------------------------------------------------
BlockVector::~BlockVector()
{
  if (owner)
  {
    for (uint i = 0; i < n; i++)
      delete vectors[i];
  }
  delete [] vectors;
}
//-----------------------------------------------------------------------------
BlockVector* BlockVector::copy() const
{
  BlockVector* x= new BlockVector(n);
  for (uint i = 0; i < n; i++)
    x->set(i,*(this->get(i).copy()));
  return x;
}
//-----------------------------------------------------------------------------
SubVector BlockVector::operator()(uint i)
{
  SubVector sv(i,*this);
  return sv;
}
//-----------------------------------------------------------------------------
dolfin::uint BlockVector::size() const
{
  return n;
}
//-----------------------------------------------------------------------------
void BlockVector::axpy(double a, const BlockVector& x)
{
  for (uint i = 0; i < n; i++)
    this->get(i).axpy(a, x.get(i));
}
//-----------------------------------------------------------------------------
double BlockVector::inner(const BlockVector& x) const
{
  double value = 0.0;
  for (uint i = 0; i < n; i++)
    value += this->get(i).inner(x.get(i));
  return value;
}
//-----------------------------------------------------------------------------
double BlockVector::norm(std::string norm_type) const
{
  double value = 0.0;
  if(norm_type == "l1")
  {
    for (uint i = 0; i < n; i++)
      value += this->get(i).norm(norm_type);
  }
  else if(norm_type == "l2")
  {
    for (uint i = 0; i < n; i++)
      value += std::pow(this->get(i).norm(norm_type), 2);
    value = sqrt(value);
  }
  else
  {
    double tmp= 0.0;
    for (uint i = 0; i < n; i++)
    {
      tmp = this->get(i).norm(norm_type);
      if (tmp > value)
        value = tmp;
    }
  }
  return value;
}
//-----------------------------------------------------------------------------
double BlockVector::min() const
{
  double value = 100000000; //FIXME use MAXFLOAT or something
  double tmp = 0.0;
  for (uint i = 0; i < n; i++)
  {
    tmp = this->get(i).min();
    if (tmp < value)
      value = tmp;
  }
  return value;
}
//-----------------------------------------------------------------------------
double BlockVector::max() const
{
  double value = -1.0; //FIXME use MINFLOAT or something
  double tmp = 0.0;
  for (uint i = 0; i < n; i++)
  {
    tmp = this->get(i).min();
    if (tmp > value)
      value = tmp;
  }
  return value;
}
//-----------------------------------------------------------------------------
const BlockVector& BlockVector::operator*= (double a)
{
  for(uint i = 0; i < n; i++)
    this->get(i) *= a;
  return *this;
}
//-----------------------------------------------------------------------------
const BlockVector& BlockVector::operator/= (double a)
{
  for(uint i = 0; i < n; i++)
    this->get(i) /= a;
  return *this;
}
//-----------------------------------------------------------------------------
const BlockVector& BlockVector::operator+= (const BlockVector& y)
{
  axpy(1.0, y);
  return *this;
}
//-----------------------------------------------------------------------------
const BlockVector& BlockVector::operator-= (const BlockVector& y)
{
  axpy(-1.0, y);
  return *this;
}
//-----------------------------------------------------------------------------
const BlockVector& BlockVector::operator= (const BlockVector& x)
{
  for(uint i = 0; i < n; i++)
    this->get(i) = x.get(i);
  return *this;
}
//-----------------------------------------------------------------------------
const BlockVector& BlockVector::operator= (double a)
{
  for(uint i = 0; i < n; i++)
    this->get(i) = a;
  return *this;
}
//-----------------------------------------------------------------------------
std::string BlockVector::str(bool verbose) const
{
  std::stringstream s;

  if (verbose)
  {
    s << str(false) << std::endl << std::endl;

    for (uint i = 0; i < n; i++)
    {
      s << "  BlockVector " << i << std::endl << std::endl;
      s << indent(indent(get(i).str(true))) << std::endl;
    }
  }
  else
  {
    s << "<BlockVector containing " << n << " blocks>";
  }

  return s.str();
}
//-----------------------------------------------------------------------------
void BlockVector::set(uint i, Vector& v)
{
//  matrices[i*n+j] = m.copy(); //FIXME. not obvious that copy is the right thing
  vectors[i] = &v; //FIXME. not obvious that copy is the right thing
}
//-----------------------------------------------------------------------------
const Vector& BlockVector::get(uint i) const
{
  return *(vectors[i]);
}
//-----------------------------------------------------------------------------
Vector& BlockVector::get(uint i)
{
  return *(vectors[i]);
}
//-----------------------------------------------------------------------------
// SubVector
//-----------------------------------------------------------------------------
SubVector::SubVector(uint n_, BlockVector& bv_)
  : n(n_),bv(bv_)
{
  // Do nothing
}
//-----------------------------------------------------------------------------
SubVector::~SubVector()
{
  // Do nothing
}
//-----------------------------------------------------------------------------
const SubVector& SubVector::operator=(Vector& v)
{
  bv.set(n, v);
  return *this;
}
/*
//-----------------------------------------------------------------------------
Vector& SubVector::operator()
{
  return bm.get(row, col);
}
*/
//-----------------------------------------------------------------------------
