"""This module defines a jit function that wraps the jit function for
the chosen form compiler."""

__author__ = "Johan Hake (hake@simula.no)"
__date__ = "2008-12-04 -- 2009-12-11"
__copyright__ = "Copyright (C) 2008-2009 Johan Hake"
__license__  = "GNU LGPL Version 2.1"

# Modified by Anders Logg, 2008-2009.

__all__ = ["jit"]

# Import SWIG-generated extension module (DOLFIN C++)
import dolfin.cpp as cpp
from dolfin.cpp import MPI, info, parameters

# Set UFL log level
import ufl
ufl.set_level(ufl.INFO)

# Import check for correct swig version from instant
from instant import check_swig_version, get_swig_version

def mpi_jit_decorator(local_jit, *args, **kwargs):
    """A decorator for jit compilation

    Use this function as a decorator to any jit compiler function.  In
    a paralell run, this function will first call the jit compilation
    function on the first process. When this is done, and the module
    is in the cache, it will call the jit compiler on the remaining
    processes, which will then use the cached module.

    Usage:

        @mpi_jit_decorator
        def jit_something(something):
            ....

    """
    def mpi_jit(*args, **kwargs):

        # Just call JIT compiler when running in serial
        if MPI.num_processes() == 1:
            return local_jit(*args, **kwargs)

        # Compile first on process 0
        if MPI.process_number() == 0:
            output = local_jit(*args, **kwargs)

        # Wait for the first process to finish
        MPI.barrier()

        # Then compile on all other processes (which may then just read the cache)
        if not MPI.process_number() == 0:
            output = local_jit(*args,**kwargs)

        return output

    # Return the decorated jit function
    return mpi_jit

@mpi_jit_decorator
def jit(form, form_compiler_parameters=None, common_cell=None):
    """Just-in-time compile any provided form.

    It uses the jit function from the form compiler registered by
    parameters["form_compiler"]["name"].
    """

    # Check that the form compiler will use the same swig version
    # that PyDOLFIN was compiled with
    if not check_swig_version(cpp.__swigversion__,same=True):
        raise OSError, """\
PyDOLFIN was not compiled with the present version of swig.
Install swig version %s or recompiled PyDOLFIN with present swig
"""%cpp.__swigversion__

    # Import form compiler
    form_compiler_name = cpp.parameters["form_compiler"]["name"]
    try:
        form_compiler = __import__(form_compiler_name)
    except ImportError, message:
        print message
        raise RuntimeError, "Could not import %s form compiler." % form_compiler_name

    # Prepare form compiler parameters
    p = form_compiler.default_parameters()

    # Set parameters from global DOLFIN parameter set
    for key, value in parameters["form_compiler"].iteritems():
        p[key] = value

    # Override with local parameters if any
    if form_compiler_parameters:
        p.update(form_compiler_parameters)

    # Get jit function
    try:
        jit_compile = form_compiler.jit
    except AttributeError:
        raise RuntimeError, "Form compiler must implement the jit function."

    return jit_compile(form, parameters=p, common_cell=common_cell)
