#ifndef ZipFile_h
#include "ZipFile.h"
#endif

#ifndef Unzip_h
#include "Unzip.h"
#endif

#ifndef std_iostream
#define std_iostream
#include <iostream>
#endif

#include <stdio.h>

using namespace doctorj;
using namespace std;

namespace doctorj 
{
    struct ZipEntry {
        string name;            // file name
        int uncomp;             // uncompressed
        int comp;               // compressed
        int offset;             // where in the file
    };
}


ZipFile::ZipFile(const string& name) : name_(name), stream_(name)
{
    stream_.read(22, -22, SEEK_END);

    unsigned int magic = stream_.getU4();
    stream_.skip(8);
    u4 central_directory_size = stream_.getU4();

    stream_.read(central_directory_size + 22, -((int) central_directory_size + 22), SEEK_END);

    for (magic = stream_.getU4(); magic == 0x02014b50; magic = stream_.getU4()) {
        readEntry();
    }
}

ZipFile::~ZipFile()
{
    vector<ZipEntry*>::iterator it   = entries_.begin();
    vector<ZipEntry*>::iterator stop = entries_.end();
    while (it != stop) {
        ZipEntry* ze = *it;
        delete ze;
        ++it;
    }
}

void ZipFile::readEntry()
{
    // we'll get all of the following, not that we use it.
    /* u2 version_made_by                 = */ stream_.getU2(); // unused
    /* u2 version_needed_to_extract       = */ stream_.getU2(); // unused
    /* u2 general_purpose_bits            = */ stream_.getU2(); // unused
    /* u2 compression_method              = */ stream_.getU2(); // unused
    /* u2 last_mod_file_time              = */ stream_.getU2(); // used
    /* u2 last_mod_file_date              = */ stream_.getU2(); // used
    /* u4 crc32                           = */ stream_.getU4(); // unused
    u4 compressed_size                 = stream_.getU4(); // used
    u4 uncompressed_size               = stream_.getU4(); // used
    u2 file_name_length                = stream_.getU2(); // used
    u2 extra_field_length              = stream_.getU2(); // used
    u2 file_comment_length             = stream_.getU2(); // used
    /* u2 disk_number_start               = */ stream_.getU2(); // unused
    /* u2 internal_file_attributes        = */ stream_.getU2(); // unused
    /* u4 external_file_attributes        = */ stream_.getU4(); // unused
    u4 relative_offset_of_local_header = stream_.getU4(); // used
    /* u4 date_time                       = ((u4) last_mod_file_date) << 16 | last_mod_file_time; */

    string namestr(stream_.position(), file_name_length);
    
    stream_.skip(file_name_length + extra_field_length + file_comment_length);

//     cout << "entry:" << endl;
//     cout << "    isdir   = " << (bool)(namestr[file_name_length - 1] == '/') << endl;
//     cout << "    name    = " << namestr << endl;
//     cout << "    offset  = " << relative_offset_of_local_header << endl;
//     cout << "    uncomp  = " << uncompressed_size << endl;
//     cout << "    comp    = " << compressed_size << endl;

    if (namestr[file_name_length - 1] != '/') {
        ZipEntry* ze = new ZipEntry();
        ze->name   = namestr;
        ze->offset = relative_offset_of_local_header;
        ze->uncomp = uncompressed_size;
        ze->comp   = compressed_size;
        entries_.push_back(ze);
    }
}

vector<string> ZipFile::entryNames() const
{
    vector<string> names;

    vector<ZipEntry*>::const_iterator it   = entries_.begin();
    vector<ZipEntry*>::const_iterator stop = entries_.end();
    while (it != stop) {
        ZipEntry* ze = *it;
        names.push_back(ze->name);
        ++it;
    }

    return names;
}

int ZipFile::getBytes(const string& name, char** bytes)
{
//     cout << "getBytes(" << name << ")" << endl;
    
    vector<ZipEntry*>::const_iterator it   = entries_.begin();
    vector<ZipEntry*>::const_iterator stop = entries_.end();
    ZipEntry* entry = NULL;
    while (!entry && it != stop) {
        ZipEntry* ze = *it;
        if (ze->name == name) {
            entry = ze;
        }
        ++it;
    }
    
    if (!entry) {
        cerr << "entry not found for name " << name << endl;
        return -1;
    }

    stream_.read(30, entry->offset, SEEK_SET);
    
    // get everything, but throw most of it away.
    /* u4 magic                     = */ stream_.getU4(); // unused
    /* u2 version_needed_to_extract = */ stream_.getU2(); // unused
    /* u2 general_purpose_bits      = */ stream_.getU2(); // unused
    u2 compression_method        = stream_.getU2(); // used
    /* u2 time                      = */ stream_.getU2(); // unused
    /* u2 date                      = */ stream_.getU2(); // unused
    /* u4 crc32                     = */ stream_.getU4(); // unused
    /* u4 compressed_size           = */ stream_.getU4(); // unused
    /* u4 uncompressed_size         = */ stream_.getU4(); // unused
    u2 filename_length           = stream_.getU2(); // used
    u2 extra_field_length        = stream_.getU2(); // used

    stream_.read(filename_length + extra_field_length);

    stream_.skip(filename_length + extra_field_length);

    *bytes = new char[entry->uncomp];

    if (compression_method == 0) {
        // just stored, not compressed.
        Unzip::restore(stream_.file(), *bytes, entry->uncomp);
        // int status = Unzip::restore(stream_.file(), *bytes, entry->uncomp);
        return entry->uncomp;
    }
    else {
        // it must be inflate/deflate (method == 8).
        int status = Unzip::deflate(stream_.file(), *bytes);
        if (status) {
            cerr << "problems deflating the file." << endl;
            return 0;
        }
        else {
//             cout << "bytes (" << (void*)*bytes << ")" << endl;
            
//             for (int i = 0; i < 5 && i < entry->uncomp; ++i) {
//                 cout << "buffer[" << i << "] (" << (int)(*bytes)[i] << ") = '" << (*bytes)[i] << "'" << endl;
//             }
            return entry->uncomp;
        }
    }
}
