#ifndef Yagol_h
#define Yagol_h

#ifndef StringUtilities_h
#include "StringUtilities.h"
#endif

#ifndef Debug_h
#include "Debug.h"
#endif

#ifndef std_string
#define std_string
#include <string>
#endif

#ifndef std_iostream
#define std_iostream
#include <iostream>
#endif

#ifndef std_algorithm
#define std_algorithm
#include <algorithm>
#endif

#ifndef std_vector
#define std_vector
#include <vector>
#endif

#ifndef std_map
#define std_map
#include <map>
#endif

#ifndef yagol_deque
#define yagol_deque
#include <deque>
#endif

#ifndef yagol_set
#define yagol_set
#include <set>
#endif

#ifndef yagol_list
#define yagol_list
#include <list>
#endif

#include <ctype.h>

#ifdef DEBUGGING
#    define DEBUG_YAGOL(a)     if (DebugOptions::get()->isEnabled('y'))    a
#else
#    define DEBUG_YAGOL(a)     if (false)                                  a
#endif

using namespace std;

namespace yagol
{
    /**
     * A minimal exception pertaining to the yagol module.
     */
    class Exception
    {
    public:

        friend ostream& operator<<(ostream& os, const yagol::Exception& e);

        /**
         * An exception without a message. The message will be set to "yagol
         * Exception".
         */
        Exception();

        /**
         * An exception with a message.
         */
        Exception(const string& msg);

        /**
         * Copy constructor.
         */
        Exception(const yagol::Exception& rhs);

        /**
         * Destructor.
         */
        virtual ~Exception();

        /**
         * Assignment operator.
         */
        virtual Exception& operator=(const yagol::Exception& rhs);

        /**
         * Writes the message to the output stream.
         */
        virtual void print(ostream& os) const;

        /**
         * Returns the message.
         */
        virtual string message() const;

    private:

        string message_;

    };


    template < class Type >
    class InvalidValueException : public Exception 
    {
    public:
        /**
         * Creates the exception for the given value, which is not in the array
         * of valid values.
         */
        InvalidValueException(const Type& value, Type* const valids, int nValids);

        /**
         * Destructor.
         */
        virtual ~InvalidValueException();

        /**
         * Returns the value.
         */
        virtual Type value() const;

        /**
         * Returns the message.
         */
        virtual string message() const;

    private:
        /**
         * The value.
         */
        Type value_;
        
        /**
         * The possible (valid) values.
         */
        Type* valids_;

        /**
         * The number of possible (valid) values.
         */
        int nValids_;

    };

    template < class Type >
    InvalidValueException<Type>::InvalidValueException(const Type& value, Type* const valids, int nValids) :
            value_(value), valids_(valids), nValids_(nValids)
    {
    }

    template < class Type >
    InvalidValueException<Type>::~InvalidValueException()
    {
    }

    template < class Type >
    Type InvalidValueException<Type>::value() const
    {
        return value_;
    }

    template < class Type >
    string InvalidValueException<Type>::message() const
    {
        string msg("value ");
        msg += doctorj::StringUtilities::toString(value());
        msg += " is not in the valid values (";
        if (nValids_ == 1) {
            msg += doctorj::StringUtilities::toString(valids_[0]);
        }
        else if (nValids_ == 2) {
            msg += doctorj::StringUtilities::toString(valids_[0]);
            msg += " and ";
            msg += doctorj::StringUtilities::toString(valids_[1]);
        }
        else {
            for (int i = 0; i < nValids_ - 1; ++i) {
                msg += doctorj::StringUtilities::toString(valids_[i]);
                msg += ", ";
            }
            msg += " and ";
            msg += doctorj::StringUtilities::toString(valids_[nValids_ - 1]);
        }
        msg += ")";
        return msg;
    }


    template < class Type >
    class OutOfRangeException : public Exception 
    {
    public:
        /**
         * Creates the exception for the given value, which exceeds the given
         * limit.
         */
        OutOfRangeException(const Type& value, const Type& limit);

        /**
         * Destructor.
         */
        virtual ~OutOfRangeException();

        /**
         * Returns the value.
         */
        virtual Type value() const;

        /**
         * Returns the limit.
         */
        virtual Type limit() const;

    private:
        /**
         * The value.
         */
        Type value_;
        
        /**
         * The limit.
         */
        Type limit_;

    };

    template < class Type >
    OutOfRangeException<Type>::OutOfRangeException(const Type& value, const Type& limit) :
            value_(value), limit_(limit)
    {
    }

    template < class Type >
    OutOfRangeException<Type>::~OutOfRangeException()
    {
    }

    template < class Type >
    Type OutOfRangeException<Type>::value() const
    {
        return value_;
    }

    template < class Type >
    Type OutOfRangeException<Type>::limit() const
    {
        return limit_;
    }


    template < class Type >
    class AboveRangeException : public OutOfRangeException<Type>
    {
    public:
        /**
         * Creates the exception for the given value, which is above the given
         * upper bound.
         */
        AboveRangeException(Type value, Type upper);

        /**
         * Destructor.
         */
        virtual ~AboveRangeException();

        /**
         * Returns the message.
         */
        virtual string message() const;

    };

    template < class Type >
    AboveRangeException<Type>::AboveRangeException(Type value, Type upper) :
            OutOfRangeException<Type>(value, upper)
    {
    }

    template < class Type >
    AboveRangeException<Type>::~AboveRangeException()
    {
    }

    template < class Type >
    string AboveRangeException<Type>::message() const
    {
        return "value " + doctorj::StringUtilities::toString(value()) + 
            " is above the upper bound " + doctorj::StringUtilities::toString(limit());
    }


    template < class Type >
    class BelowRangeException : public OutOfRangeException<Type>
    {
    public:
        /**
         * Creates the exception for the given value, which is below the given
         * lower bound.
         */
        BelowRangeException(Type value, Type lower);

        /**
         * Destructor.
         */
        virtual ~BelowRangeException();

        /**
         * Returns the message.
         */
        virtual string message() const;

    };

    template < class Type >
    BelowRangeException<Type>::BelowRangeException(Type value, Type lower) :
            OutOfRangeException<Type>(value, lower)
    {
    }

    template < class Type >
    BelowRangeException<Type>::~BelowRangeException()
    {
    }

    template < class Type >
    string BelowRangeException<Type>::message() const
    {
        return "value " + doctorj::StringUtilities::toString(value()) + 
            " is below the lower bound " + doctorj::StringUtilities::toString(limit());
    }


    class FileNotReadableException : public yagol::Exception
    {
    public:
    
        /**
         * Takes the name of the file that was not readable.
         */
        FileNotReadableException(const string& filename);

        /**
         * Copy constructor.
         */
        FileNotReadableException(const FileNotReadableException& rhs);

        /**
         * Destructor.
         */
        virtual ~FileNotReadableException();

        /**
         * Assignment operator.
         */
        virtual FileNotReadableException& operator=(const FileNotReadableException& rhs);

    };


    class FileNotWritableException : public Exception
    {
    public:
    
        /**
         * Takes the name of the file that was not writable.
         */
        FileNotWritableException(const string& filename);

        /**
         * Copy constructor.
         */
        FileNotWritableException(const FileNotWritableException& rhs);

        /**
         * Destructor.
         */
        virtual ~FileNotWritableException();

        /**
         * Assignment operator.
         */
        virtual FileNotWritableException& operator=(const FileNotWritableException& rhs);

    };


    class NoSuchFileException : public Exception
    {
    public:
    
        /**
         * Takes the name of the file that could not be found
         */
        NoSuchFileException(const string& name);

        /**
         * Copy constructor.
         */
        NoSuchFileException(const NoSuchFileException& rhs);

        /**
         * Destructor.
         */
        virtual ~NoSuchFileException();

        /**
         * Assignment operator.
         */
        virtual NoSuchFileException& operator=(const NoSuchFileException& rhs);
    };


    class NotAFileException : public yagol::Exception
    {
    public:
    
        /**
         * Takes the name of the object that was not a file.
         */
        NotAFileException(const string& name);

        /**
         * Copy constructor.
         */
        NotAFileException(const NotAFileException& rhs);

        /**
         * Destructor.
         */
        virtual ~NotAFileException();

        /**
         * Assignment operator.
         */
        virtual NotAFileException& operator=(const NotAFileException& rhs);
    };


    class OptionList;
    
    /**
     * A single option in its simplest form, with no value associated with it.
     */
    class Option
    {
    public:

        friend ostream& operator<<(ostream& os, const Option& opt);

        static const string DELIMITER;

        /**
         * Constructor using a name ("size") and a description ("the size of the
         * pholoylm").
         */
        Option(const string& name, const string& description, bool strictArgs = false);

        /**
         * Copy constructor.
         */
        Option(const Option& rhs);

        /**
         * Destructor.
         */
        virtual ~Option();

        /**
         * Processes (sets) this option, from a string. The precondition is that the
         * name has already been checked to ensure that it matches the value's tag.
         */
        virtual void setValue(const string& value) = 0;
    
        /**
         * Returns the name of this option.
         */
        virtual string name() const;

        /**
         * Returns the description.
         */
        virtual string description() const;

        /**
         * Returns the default value(s), as a string, suitable for a
         * configuration file.
         */
        virtual string defaultValueConfigString() const = 0;

        /**
         * Writes this option in a --help style format.
         */
        virtual ostream& print(ostream& os) const;

        /**
         * Writes the description, and the default value, if any. That is,
         * prints everything except for the name.
         */
        virtual ostream& printDescription(ostream& os) const;

        /**
         * Returns whether the argument appears to be a tagged option, that is,
         * the argument begins with the delimiter.
         */
        static bool matchesDelimiter(const string& arg);

        /**
         * Returns the length of the delimiter.
         */
        static int delimiterLength();

        /**
         * Called when the option is successfully processed from the command
         * line. This is a no-op by default, returning true. Subclasses should
         * throw an exception to abort processing.
         */
        virtual bool onProcessed();

        /**
         * Called when the option is successfully processed from a name/value
         * pair in a configuration file. This is a no-op by default, returning
         * true, which means to continue processing. Subclasses should throw an
         * exception to abort processing.
         */
        virtual bool onRead();

        /**
         * Called after all options have been successfully processed. This is a
         * no-op by default. Subclasses should throw an exception to abort
         * processing.
         */
        virtual void onComplete(const OptionList& opts);

        /**
         * Returns whether the name matches, for command-line arguments.
         */
        virtual bool tagMatches(const string& tag) const;

        /**
         * Returns whether the name matches, for RC file labels (names).
         */
        virtual bool labelMatches(const string& label) const;

        /**
         * The precondition for this method is that the argument string (arg)
         * had a valid delimiter. This method checks the argument string for a
         * match, and "consumes" arguments from the argv list. Returns the
         * number of arguments consumed.
         */
        virtual int consume(const string& arg, int position, int argc, char** argv) = 0;

        /**
         * The precondition for this method is that the argument string (arg)
         * had a valid delimiter. This method uses the given value to set itself.
         */
        virtual void setValueFromConfigFile(const string& value);

        /**
         * Adds the given option as being tied to this one. If this option is
         * set, so too will the other option.
         */
        virtual void addLinkedOption(Option* const opt);

        /**
         * Writes the low-level details about this object to the given stream.
         */
        virtual void debugPrint(ostream& os) const = 0;
        
        /**
         * Sets whether this option exists only in configuration files, not on
         * the command line.
         */
        virtual void setConfigOnly(bool cfgOnly);

        /**
         * Returns whether this option exists only in configuration files, not on
         * the command line.
         */
        virtual bool isConfigOnly() const;
        
        /**
         * Sets whether this option exists only on the command line, not in
         * configuration files.
         */
        virtual void setCommandLineOnly(bool clOnly);

        /**
         * Returns whether this option exists only on the command line, not in
         * configuration files.
         */
        virtual bool isCommandLineOnly() const;

        /**
         * Writes the configuration.
         */
        virtual void writeConfig(ostream& os) const;

        /**
         * Adds the given alias.
         */
        virtual void addAlias(const string& alias);

    protected:

        /**
         * Returns whether to throw an exception when something that appears to
         * be a tag is passed as an argument.
         */
        virtual bool strictArgs() const;

        /**
         * Checks the argument as matching the delimiter, and throws an
         * exception if it appears to be a tag.
         */
        virtual void checkArgument(const string& arg) throw(yagol::Exception);

        /**
         * Returns the options that inherit from this one.
         */
        vector<Option*>& getLinkedOptions();

    private:

        /**
         * Options that inherit from this one.
         */
        vector<Option*> linkedOptions_;

        string name_;
    
        string description_;

        /**
         * Whether to throw an exception when something that appears to be a tag
         * is passed as an argument.
         */
        bool strictArgs_;

        /**
         * Whether this option is only in configuration files, not on the
         * command line.
         */
        bool configOnly_;

        /**
         * Whether this option is only on the command line, not in configuration
         * files.
         */
        bool cmdLineOnly_;

        /**
         * The aliases for this option.
         */
        vector<string> aliases_;

    };



    /**
     * Corresponds to yagol::OptionGeneric. The template parameter defines the
     * type of variable (a C++ built-in type) for the option.
     */
    template < class Type >
    class ArgCallback 
    {
    public:

        /**
         * Usage of this contructor denotes an object tied to the option.
         */
        ArgCallback(Type* const value);

        /**
         * This contructor has no object tied to the option.
         */
        ArgCallback();

        /**
         * Destructor does not delete the object tied to the option.
         */
        virtual ~ArgCallback();

        /**
         * Called after the option is processed, with its value. By default,
         * this is a no-op that returns true, which means to continue processing
         * the argument list.
         */
        virtual bool onProcessed(const Type& v);

        /**
         * Called after the option is read, with its value. By default, this is
         * a no-op that returns true, which means to continue processing the
         * argument list.
         */
        virtual bool onRead(const Type& v);

        /**
         * Called after all of the options are processed, with the value of this
         * option. This is a no-op by default.
         */
        virtual void onComplete(const OptionList& opts);

        /**
         * Returns the object tied to this option.
         */
        virtual Type* tiedObject();

    private:
        
        /**
         * The object tied to this option.
         */
        Type* value_;

    };


    template < class Type >
    ArgCallback<Type>::ArgCallback(Type* const value) : value_(value)
    {
    }

    template < class Type >
    ArgCallback<Type>::ArgCallback() : value_(NULL)
    {
    }

    template < class Type >
    ArgCallback<Type>::~ArgCallback()
    {
        // nothing.
    }

    template < class Type >
    bool ArgCallback<Type>::onProcessed(const Type& v)
    {
        return true;
    }

    template < class Type >
    bool ArgCallback<Type>::onRead(const Type& v)
    {
        return true;
    }

    template < class Type >
    void ArgCallback<Type>::onComplete(const OptionList& opts)
    {
    }

    template < class Type >
    Type* ArgCallback<Type>::tiedObject()
    {
        return value_;
    }


    /**
     * Handles generic data types as a user-specified option. An instance of this
     * class for some type Type must have the following defined:
     *
     *     Type(const Type&)           // (copy constructor)
     *     StringUtilities::fromString<Type>(const string&)
     *     operator<<(ostream&, T)     // output (insertion) operator
     *     operator=(const Type&)      // assignment operator
     *
     * All of the C++ built-in data types are supported, as well as the STL string
     * class.
     */
    template < class Type >
    class OptionGeneric : public Option
    {
    public:
    
        /**
         * Constructor using an name ("size"), a description ("the size of the
         * pholoylm"), and the associated value, of type Type. The input value of
         * this will be considered to be the default value. This value will be set
         * upon successful processing of the argument list.
         */
        OptionGeneric(const string& name, 
                      const string& description,
                      Type* const value,
                      bool strictArgs = false);

        /**
         * Constructor using an name ("size"), a description ("the size of the
         * pholoylm"), and an associated callback object, for type Type. The
         * input value of the object tied to the callback will be considered to
         * be the default value. This value will be set upon successful
         * processing of the argument list.
         */
        OptionGeneric(const string& name, 
                      const string& description,
                      yagol::ArgCallback<Type>* const callback,
                      bool strictArgs = false);

        /**
         * Copy constructor. See the class comment for details about methods that
         * must be defined.
         */
        OptionGeneric(const OptionGeneric<Type>& rhs);

        /**
         * Destructor.
         */
        virtual ~OptionGeneric();

        /**
         * Checks the argument name, and if it matches, sets the value, starting at
         * the given position in the argument list. The position is advanced by the
         * number of arguments "consumed" by this object.
         */
        virtual int consume(const string& arg, int position, int argc, char** argv) throw(yagol::Exception);
    
        /**
         * Processes (sets) this option, from a string. The precondition is that the
         * name has already been checked to ensure that it matches the value's tag.
         */
        virtual void setValue(const string& value);

        /**
         * Sets all linked options to the given value.
         */
        virtual void setLinkedOptions(const string& value);

        /**
         * Returns the value of the tied object.
         */
        virtual Type value() const;

        /**
         * Writes the description, and the default value, if any. That is,
         * prints everything except for the name.
         */
        virtual ostream& printDescription(ostream& os) const;

        /**
         * Returns the default value.
         */
        virtual Type defaultValue() const;

        /**
         * Returns the default value, as a string, suitable for a configuration
         * file.
         */
        virtual string defaultValueConfigString() const;

        /**
         * Called when the option is successfully processed. This is delegated
         * to the arg callback object, if any; otherwise it propagates to the
         * method in the base class.
         */
        virtual bool onProcessed();

        /**
         * Called when the option is successfully read (from a configuration
         * file). This is delegated to the arg callback object, if any;
         * otherwise it propagates to the method in the base class.
         */
        virtual bool onRead();

        /**
         * Called after all options have been successfully processed. This is
         * delegated to the arg callback object, if any; otherwise it propagates
         * to the method in the base class.
         */
        virtual void onComplete(const OptionList& opts);

        /**
         * Redeclared here, in order to override behavior for boolean options,
         * which can match tags of the form "tag" or "no-?tag".
         */
        virtual bool tagMatches(const string& tag) const;

        /**
         * Writes the low-level details about this object to the given stream.
         */
        virtual void debugPrint(ostream& os) const;

    private:
    
        Type defvalue_;

        Type* value_;

        ArgCallback<Type>* callback_;

    };

    template < class Type >
    OptionGeneric<Type>::OptionGeneric(const string& name, 
                                       const string& description,
                                       Type* const value,
                                       bool strictArgs /* = false */) :
            Option(name, description, strictArgs),
         defvalue_(*value),
         value_(value),
         callback_(NULL)
    {
    }

    template < class Type >
    OptionGeneric<Type>::OptionGeneric(const string& name, 
                                       const string& description,
                                       yagol::ArgCallback<Type>* const callback,
                                       bool strictArgs /* = false */) :
            Option(name, description, strictArgs),
            defvalue_(callback->tiedObject() ? *(callback->tiedObject()) : 0),
            value_(callback->tiedObject()),
            callback_(callback)
    {
    }

    template < class Type >
    OptionGeneric<Type>::OptionGeneric(const OptionGeneric<Type>& rhs) :
            Option(rhs),
            defvalue_(rhs.defvalue_),
            value_(rhs.value_),
            callback_(rhs.callback_)
    {
    }

    template < class Type >
    OptionGeneric<Type>::~OptionGeneric()
    {
    }

    template < class Type >
    int OptionGeneric<Type>::consume(const string& arg, int position, int argc, char** argv) throw(yagol::Exception)
    {
        int nArgs = 1;
        ++position;

        if (position < argc) {
            // there needs to be a following argument, which is the value
            string argstr(argv[position]);
            checkArgument(argstr);
            setValue(argstr);
            position++;         // and now consume the value
            nArgs++;
        }
        else {
            string msg("argument expected for ");
            msg += name();
            throw yagol::Exception(msg);
        }
        return nArgs;
    }

    template < class Type >
    void OptionGeneric<Type>::setValue(const string& value)
    {
        if (value_) {
            doctorj::StringUtilities::fromString(value, *value_);
        }
        setLinkedOptions(value);
    }

    template < class Type >
    void OptionGeneric<Type>::setLinkedOptions(const string& value)
    {
        for (vector<Option*>::iterator it = getLinkedOptions().begin(), stop = getLinkedOptions().end();
             it != stop; ++it) {
            Option* opt = *it;
            opt->setValue(value);
        }
    }

    template < class Type >
    Type OptionGeneric<Type>::value() const
    {
        return *value_;
    }

    template < class Type >
    ostream& OptionGeneric<Type>::printDescription(ostream& os) const
    {
        os << description() << " (default = " << defvalue_ << ")";
        return os;
    }

    template < class Type >
    Type OptionGeneric<Type>::defaultValue() const
    {
        return defvalue_;
    }

    template < class Type >
    string OptionGeneric<Type>::defaultValueConfigString() const
    {
        return doctorj::StringUtilities::toString(defvalue_);
    }

    template < class Type >
    bool OptionGeneric<Type>::onProcessed()
    {
        if (callback_ == NULL) {
            return Option::onProcessed();
        }
        else {
            return callback_->onProcessed(*value_);
        }
    }

    template < class Type >
    bool OptionGeneric<Type>::onRead()
    {
        if (callback_ == NULL) {
            return Option::onRead();
        }
        else {
            return callback_->onRead(*value_);
        }
    }

    template < class Type >
    void OptionGeneric<Type>::onComplete(const OptionList& opts)
    {
        if (callback_ == NULL) {
            Option::onComplete(opts);
        }
        else {
            callback_->onComplete(opts);
        }
    }

    template < class Type >
    bool OptionGeneric< Type >::tagMatches(const string& tag) const
    {
        return Option::tagMatches(tag);
    }


    template < class Type >
    void OptionGeneric< Type >::debugPrint(ostream& os) const
    {
        os << "{ " << name() << ", " << description() << ", " << value() << " }";
    }


    // ----------------------------------------------------------------------
    // string
    // ----------------------------------------------------------------------

    /**
     * Strings get quoted.
     */
    template <>
    inline ostream& OptionGeneric<string>::printDescription(ostream& os) const
    {
        os << description() << " (default = \"" << defvalue_ << "\")";
        return os;
    }


    // ----------------------------------------------------------------------
    // bool
    // ----------------------------------------------------------------------

    /**
     * "true" => true and "yes" => true.
     */
    template <>
    inline void OptionGeneric<bool>::setValue(const string& value)
    {
        string aslower = doctorj::StringUtilities::toLower(value);
        *value_ = aslower == "true" || aslower == "yes";

        setLinkedOptions(value);
    }

    /**
     * prints "true" and "false", not "1" and "0".
     */
    template <>
    inline ostream& OptionGeneric<bool>::printDescription(ostream& os) const
    {
        os << description() << " (default = " << (defvalue_ ? "true" : "false") << ")";
        return os;
    }

    /**
     * special case, because:
     * instead of:          option is:
     *     --foo true   ->  --foo
     *     --foo false  ->  --no-foo
     *     --foo false  ->  --nofoo
     */
    template <>
    inline int OptionGeneric<bool>::consume(const string& arg, int position, int argc, char** argv) throw(yagol::Exception)
    {
        if (Option::tagMatches(arg)) {
            // positive (true) ("--foo")
            if (value_) {
                *value_ = true;
            }
            setLinkedOptions("true");
            return 1;
        }
        else if (arg.substr(0, 3) == "no-" || arg.substr(0, 2) == "no") {
            // negative (false) ("--no-?foo")
            if (value_) {
                *value_ = false;
            }
            setLinkedOptions("false");
            return 1;
        }

        // we shouldn't get here, since the precondition is to match the tag.
        return 0;
    }

    /**
     * Matches "tag" and "no-?tag", i.e., prefixed with negative.
     */
    template <>
    inline bool OptionGeneric<bool>::tagMatches(const string& tag) const
    {
        return Option::tagMatches(tag) ||
            (tag.substr(0, 3) == "no-" && Option::tagMatches(tag.substr(3))) ||
            (tag.substr(0, 2) == "no"  && Option::tagMatches(tag.substr(2)));
    }

    template <>
    inline string OptionGeneric<bool>::defaultValueConfigString() const
    {
        return defvalue_ ? "true" : "false";
    }


    /**
     * Handles a vector of values of the given parameterized type.
     */
    template < class Container >
    class OptionMultiValues : public Option
    {
    public:
    
        /**
         * Constructor using an name ("file"), a description ("the data file"), and
         * the associated file name. The input value of this will be considered to
         * be the default value. This value will be set upon successful processing
         * of the argument list.
         */
        OptionMultiValues(const string& name, 
                          const string& description,
                          Container* const values,
                          bool strictArgs = false);
        
        /**
         * Constructor using an name ("size"), a description ("the size of the
         * pholoylm"), and an associated callback object, for a vector of type
         * Type, which is a C++ built-in data type. The input value of the
         * object tied to the callback will be considered to be the default
         * value. This value will be set upon successful processing of the
         * argument list.
         */
        OptionMultiValues(const string& name, 
                          const string& description,
                          yagol::ArgCallback<vector <typename Container::value_type> >* const callback,
                          bool strictArgs = false);

        /**
         * Copy constructor.
         */
        OptionMultiValues(const OptionMultiValues& rhs);

        /**
         * Destructor.
         */
        virtual ~OptionMultiValues();

        /**
         * Checks the argument name, and if it matches, gets the file name and
         * validates it. Starts at the given position in the argument list. The
         * return value is the number of arguments "consumed" by this object,
         * which will be either to the end of the arguments, or to "--end".
         */
        virtual int consume(const string& arg, int position, int argc, char** argv) throw(yagol::Exception);
        
        /**
         * Adds the value to the list.
         */
        virtual void setValue(const string& value) throw(yagol::Exception);
    
        /**
         * Writes this option in a --help style format.
         */
        virtual ostream& printDescription(ostream& os) const;

        /**
         * Returns the default values, as a string, suitable for a configuration
         * file. Whitespace separates the values.
         */
        virtual string defaultValueConfigString() const;

        /**
         * Called when the option is successfully processed. This is delegated
         * to the arg callback object, if any; otherwise it propagates to the
         * method in the base class.
         */
        virtual bool onProcessed();

        /**
         * Called when the option is successfully read (from a configuration
         * file). This is delegated to the arg callback object, if any;
         * otherwise it propagates to the method in the base class.
         */
        virtual bool onRead();

        /**
         * Called after all options have been successfully processed. This is
         * delegated to the arg callback object, if any; otherwise it propagates
         * to the method in the base class.
         */
        virtual void onComplete(const OptionList& opts);

        /**
         * The precondition for this method is that the argument string (arg)
         * had a valid delimiter. This method uses the given value to set
         * itself. A multi-value option splits the given value on whitespace,
         * and uses each of the substrings to set itself.
         */
        virtual void setValueFromConfigFile(const string& value);

        /**
         * Replaces the current collection of values with the given ones, which
         * are converted from strings to the type for this option.
         */
        virtual void replaceValues(const vector<string>& values);

        /**
         * Returns the matching end of tag marker for the given tag. Default is
         * "--end".
         */
        virtual string endTag(const string& tag) const;

        /**
         * Returns whether the name matches, for command-line arguments.
         */
        virtual bool tagMatches(const string& tag) const;

        /**
         * Writes the low-level details about this object to the given stream.
         */
        virtual void debugPrint(ostream& os) const;

        /**
         * Writes the configuration.
         */
        virtual void writeConfig(ostream& os) const;

    private:

        /**
         * The default values.
         */
        Container defvalues_;
    
        /**
         * The applicable values.
         */
        Container* values_;

        /**
         * The object to delegated to in the <code>onProcessed</code> and
         * <code>onComplete</code> methods.
         */
        ArgCallback<vector <typename Container::value_type> >* callback_;

    };


    template < class Container >
    OptionMultiValues<Container>::OptionMultiValues(const string& name, 
                                                          const string& description,
                                                          Container* const values,
                                                          bool strictArgs /* = false */) :  
            Option(name, description, strictArgs), 
         defvalues_(*values),
         values_(values),
         callback_(NULL)
    {
    }

    template < class Container >
    OptionMultiValues<Container>::OptionMultiValues(const string& name, 
                                                          const string& description,
                                                          yagol::ArgCallback<vector< typename Container::value_type > >* const callback,
                                                          bool strictArgs /* = false */) :  
            Option(name, description, strictArgs), 
         defvalues_(*(callback->tiedObject())),
         values_(callback->tiedObject()),
         callback_(callback)
    {
    }
    
    template < class Container >
    OptionMultiValues<Container>::OptionMultiValues(const OptionMultiValues<Container>& rhs)
            : Option(rhs), defvalues_(rhs.defvalues_), values_(rhs.values_)
    {
    }

    template < class Container >
    OptionMultiValues<Container>::~OptionMultiValues()
    {
    }

    template < class Container >
    int OptionMultiValues<Container>::consume(const string& arg, int position, int argc, char** argv) throw(yagol::Exception)
    {
        int nArgs = 1;
        string endtag = endTag(arg);

        // collect the values until we get the end tag
        vector<string> values;
        for (int p = position + 1; 
             p < argc && !doctorj::StringUtilities::startsWith(argv[p], "--end");
             ++p, ++nArgs) {
            values.push_back(string(argv[p]));
        }

        replaceValues(values);
        
        // advance through the end of list tag
        ++nArgs;
        return nArgs;
    }

    template < class Container >
    string OptionMultiValues<Container>::endTag(const string& tag) const
    {
        string sw("start-of-");
        string t = doctorj::StringUtilities::startsWith(tag, sw) ? tag.substr(sw.length()) : tag;
        return "--end";
    }

    template < class Container >
    void OptionMultiValues<Container>::setValue(const string& value) throw(yagol::Exception)
    {
        typename Container::value_type val;
        doctorj::StringUtilities::fromString(value, val);
        values_->push_back(val);
    }

    template < class Container >
    string OptionMultiValues<Container>::defaultValueConfigString() const
    {
        string str;
        for (typename Container::const_iterator it = defvalues_.begin(); it != defvalues_.end(); ++it) {
            if (it != defvalues_.begin()) {
                str += " ";
            }
            str += doctorj::StringUtilities::toString(*it);
        }
        return str;
    }

    template < class Container >
    ostream& OptionMultiValues<Container>::printDescription(ostream& os) const
    {
        os << description();
        if (defvalues_.size() > 0) {
            os << " (default = ";
            for (typename Container::const_iterator it = defvalues_.begin(); it != defvalues_.end(); ++it) {
                if (it != defvalues_.begin()) {
                    os << ", ";
                }
                os << *it;
            }
            os << ")";
        }
        return os;
    }

    template < class Container >
    bool OptionMultiValues<Container>::onProcessed()
    {
        if (callback_ == NULL) {
            return Option::onProcessed();
        }
        else {
            return callback_->onProcessed(*values_);
        }
    }

    template < class Container >
    bool OptionMultiValues<Container>::onRead()
    {
        if (callback_ == NULL) {
            return Option::onRead();
        }
        else {
            return callback_->onRead(*values_);
        }
    }

    template < class Container >
    void OptionMultiValues<Container>::onComplete(const OptionList& opts)
    {
        if (callback_ == NULL) {
            Option::onComplete(opts);
        }
        else {
            callback_->onComplete(opts);
        }
    }

    template < class Container >
    void OptionMultiValues<Container>::setValueFromConfigFile(const string& value)
    {
        cout << "OptionMultiValues<Container>::setValueFromConfigFile(" << value << ")" << endl;
        vector<string> words;
        doctorj::StringUtilities::getWords(value, &words);
        replaceValues(words);
    }

    template < class Container >
    void OptionMultiValues<Container>::replaceValues(const vector<string>& values)
    {
        //!!! add a recurse guard for linked options?

        // empty the current values, since we'll be pushing all the new ones
        // onto the collection
        values_->erase(values_->begin(), values_->end());
        
        vector<string>::const_iterator vit   = values.begin();
        vector<string>::const_iterator vstop = values.end();
        for (; vit != vstop; ++vit) {
            string val = *vit;
            checkArgument(val);
            setValue(val);
        }

        // any linked value gets the same treatment
        vector<Option*>::iterator lit   = getLinkedOptions().begin();
        vector<Option*>::iterator lstop = getLinkedOptions().end();
        for (; lit != lstop; ++lit) {
            Option* opt = *lit;
            // the linked option should be of the same type as this one:
            OptionMultiValues< Container >* other = dynamic_cast<OptionMultiValues< Container >*>(opt);
            other->replaceValues(values);
        }
    }

    template < class Container >
    bool OptionMultiValues<Container>::tagMatches(const string& tag) const
    {
        return Option::tagMatches(tag) || tag == (string("start-of-") + name());
    }

    template < class Container >
    void OptionMultiValues<Container>::debugPrint(ostream& os) const
    {
        os << "{ " << name() << ", " << description() << endl;
        typename Container::const_iterator it   = values_->begin();
        typename Container::const_iterator stop = values_->end();
        for (; it != stop; ++it) {
            typename Container::value_type val = *it;
            os << ", " << val;
        }
        os << " }";
    }

    template < class Container >
    void OptionMultiValues<Container>::writeConfig(ostream& os) const
    {
        os << "# " << description() << endl;
        // should probably insert # after newlines in the description:
        
        string nm = name();
        typename Container::const_iterator it   = defvalues_.begin();
        typename Container::const_iterator stop = defvalues_.end();
        if (it == stop) {
            os << nm << " = " << endl;
        }
        else {
            while (it != stop) {
                os << nm;
                if (it == defvalues_.begin()) {
                    os << " = ";
                }
                else {
                    os << " += ";
                }
                os << doctorj::StringUtilities::toString(*it) << endl;;
                ++it;
            }
        }
    }


    /**
     * Handles files specified as a user-defined option.
     */
    class OptionFile : public OptionGeneric<string>
    {
    public:
    
        /**
         * Constructor using an name ("file"), a description ("the data file"),
         * and the associated file name. The input value of this will be
         * considered to be the default value. This value will be set upon
         * successful processing of the argument list.
         */
        OptionFile(const string& name, 
                   const string& description,
                   string* const filename,
                   bool strictArgs = false);

        /**
         * Copy constructor.
         */
        OptionFile(const OptionFile& rhs);

        /**
         * Destructor.
         */
        virtual ~OptionFile();

        /**
         * Checks the file, e.g., for write permission for output files. Throws
         * the exception on failure.
         */
        virtual void validate(const string& filename) const throw(yagol::Exception) = 0;

        /**
         * Validates the file.
         */
        virtual bool onProcessed();

    };


    /**
     * Handles an input file specified as a user-defined option. The validate
     * method will throw an exception if the file does not exist or is not
     * readable.
     */
    class OptionInputFile : public OptionFile
    {
    public:
    
        /**
         * Constructor using an input name ("file"), a description ("the data
         * file"), and the associated file name. The input value of this will be
         * considered to be the default value. This value will be set upon
         * successful processing of the argument list.
         */
        OptionInputFile(const string& name, 
                        const string& description,
                        string* const filename,
                        bool strictArgs = false);

        /**
         * Copy constructor.
         */
        OptionInputFile(const OptionInputFile& rhs);

        /**
         * Destructor.
         */
        virtual ~OptionInputFile();

        /**
         * Checks that such a file exists and is readable. Throws
         * yagol::FileNotReadableException, yagol::NoSuchFileException,
         * and yagol::NotA_FileException.
         */
        virtual void validate(const string& filename) const throw(yagol::Exception);

    };



    /**
     * Handles an output file specified as a user-defined option. The validate
     * method will throw an exception if the file does not exist or is not
     * readable.
     */
    class OptionOutputFile : public OptionFile
    {
    public:
    
        /**
         * Constructor using an output name ("file"), a description ("the data
         * file"), and the associated file name. The output value of this will
         * be considered to be the default value. This value will be set upon
         * successful processing of the argument list.
         */
        OptionOutputFile(const string& name, 
                         const string& description,
                         string* const filename,
                         bool strictArgs = false);

        /**
         * Copy constructor.
         */
        OptionOutputFile(const OptionOutputFile& rhs);

        /**
         * Destructor.
         */
        virtual ~OptionOutputFile();

        /**
         * Checks that the file either does not exist; or exists, and is a
         * writable file. Throws yagol::FileNotWritableException and
         * yagol::NotAFileException.
         */
        virtual void validate(const string& filename) const throw(yagol::Exception);

    };


    /**
     * Handles post-processing of an argument.
     */
    class Processor 
    {
    public:

        /**
         * Default constructor.
         */
        Processor();

        /**
         * Does nothing.
         */
        virtual ~Processor();

#ifdef ON_PROCESSED_DOES_NOT_TAKE_TYPE_AS_AN_ARGUMENT
        /**
         * Called after the option is processed, with its value. By default,
         * this is a no-op that returns true, which means to continue processing
         * the argument list.
         */
        virtual bool onProcessed(const Type& v);
#endif

        /**
         * Called after all of the options are processed, with the value of this
         * option. This is a no-op by default.
         */
        virtual void onComplete(const OptionList& opts);

    };


    
    template < class Type >
    class GroupValidator : public ArgCallback< Type > 
    {
    public:
        /**
         * Denotes the array of valid values.
         */
        GroupValidator(Type* const value, Type* const valids, int nValids);

        /**
         * Destructor.
         */
        virtual ~GroupValidator();

        /**
         * Called after the option is processed, with its value. Checks the
         * given value against the valid values. Returns false if the value is
         * not in the array of valid values.
         */
        virtual bool onProcessed(const Type& v);

    private:

        Type* valids_;

        int nValids_;

    };

    template < class Type >
    GroupValidator<Type>::GroupValidator(Type* const value, Type* const valids, int nValids) :
            ArgCallback<Type>(value), valids_(valids), nValids_(nValids)
    {
    }

    template < class Type >
    GroupValidator<Type>::~GroupValidator()
    {
    }
    
    template < class Type >
    bool GroupValidator<Type>::onProcessed(const Type& v) 
    {
        for (int i = 0; i < nValids_; ++i) {
            if (v == valids_[i]) {
                return true;
            }
        }
        throw InvalidValueException<Type>(v, valids_, nValids_);
        return false;
    }


    template < class Type >
    class RangeValidator : public ArgCallback< Type > 
    {
    public:
        /**
         * Denotes the lower and upper bound.
         */
        RangeValidator(Type* const value, Type lower, Type upper);

        /**
         * Destructor.
         */
        virtual ~RangeValidator();

        /**
         * Called after the option is processed, with its value. Checks the
         * given value against the lower and upper bounds. Returns false if the
         * value is out of bounds.
         */
        virtual bool onProcessed(const Type& v);

    private:

        Type lower_;

        Type upper_;

    };

    template < class Type >
    RangeValidator<Type>::RangeValidator(Type* const value, Type lower, Type upper) :
            ArgCallback<Type>(value), lower_(lower), upper_(upper)
    {
    }

    template < class Type >
    RangeValidator<Type>::~RangeValidator()
    {
    }
    
    template < class Type >
    bool RangeValidator<Type>::onProcessed(const Type& v) 
    {
        if (v < lower_) {
            throw BelowRangeException<Type>(v, lower_);
        }
        else if (v > upper_) {
            throw AboveRangeException<Type>(v, upper_);
        }
        else {
            return true;
        }
    }

    
    /**
     * Processes a set of options and command-line arguments, which are in the long
     * POSIX format (e.g., "--size 14"). Usage is straightforward:<p>
     *
     * <pre>
     *     - create your variables that will be tied to the command line
     *     - create a yagol object with the program name and description
     *     - add the options, tied to your variables
     *     - process the command-line arguments.
     * </pre>
     * 
     * See Example.t.cpp for example usage.<p>
     *
     * What it does, that you might not expect it to:<p>
     * 
     * </pre>
     *     - assume that any argument that begins with the delimiter character is a
     *       valid parameter
     *
     *     - validate input and output files (see yagol::OptionInputFile and
     *       yagol::OptionOutputFile)
     *
     *     - "boolean" variables are denoted by the "no" prefix, e.g.: "--change"
     *       and "--nochange"
     *
     *     - accept short arguments, such as "--s" instead of "--size"
     *
     *     - accept multiple names for the same option, such as both "--length"
     *       and "--duration" being synonymous
     *
     * What it doesn't do, that you might expect it to:<p>
     * 
     * <pre>
     *     - support arguments with optional parameters
     * </pre>
     */
    class OptionList
    {
    public:

        /**
         * Dumps the list of options. For developer usage.
         */
        friend ostream& operator<<(ostream& os, const OptionList& opts);

        /**
         * Options are added after construction. The <code>strictArgs</code>
         * denotes whether to treat command-line arguments that look like tags
         * (that is, they start with the delimiter) as erroneous if they do not
         * match any option. If so, an exception will be thrown during
         * processing.
         */
        OptionList(bool strictArgs = false);

        /**
         * Destructor deletes all options.
         */
        virtual ~OptionList();

        /**
         * Processes the argument list from argv, of size argc. The arguments
         * that are not "consumed" can be retrieved via
         * <code>getUnprocessedArgs</code>.
         */
        virtual void process(int argc, char** argv) throw(yagol::Exception);
    
        /**
         * Writes the usage of each option to the given output stream. The
         * format is that as for --help:<p>
         *
         * <pre>
         * --opt     Option description (default = default)
         * </pre>
         *
         * Note that there is one tab between the option and the description.
         * Also note that there is no "Option:" line, since these lists can be
         * chained together.
         */
        virtual void writeUsage(ostream& os) const;

        /**
         * Writes all the options, including those in the configuration file.
         */
        virtual void writeConfigUsage(ostream& os) const;

        /**
         * Adds the option, associated with the given argument variable. See
         * yagol::OptionGeneric for the methods that must be defined for Type.
         */
        template < typename Type >
        Option* addOption(const string& name, 
                          const string& description, 
                          Type* const value)
        {
            //cout << "OPLST  addOption(" << name << ", " << description << ", " << *value << ")" << endl;
            return addOption(new OptionGeneric<Type>(name, description, value, strictArgs_));
        }

        /**
         * Adds the multi-value option, associated with the given argument
         * variable. See yagol::OptionMultiValues.
         */
        template < typename Type >
        Option* addOption(const string& name, 
                          const string& description, 
                          vector<Type>* const values)
        {
            return addOption(new OptionMultiValues<vector<Type> >(name, description, values, strictArgs_));
        }

        /**
         * Adds the multi-value option, associated with the given argument
         * variable. See yagol::OptionMultiValues.
         */
        template < typename Type >
        Option* addOption(const string& name, 
                          const string& description, 
                          list<Type>* const values)
        {
            return addOption(new OptionMultiValues<list<Type> >(name, description, values, strictArgs_));
        }

        /**
         * Adds the multi-value option, associated with the given argument
         * variable. See yagol::OptionMultiValues.
         */
        template < typename Type >
        Option* addOption(const string& name, 
                          const string& description, 
                          deque<Type>* const values)
        {
            return addOption(new OptionMultiValues<deque<Type> >(name, description, values, strictArgs_));
        }

        /**
         * Adds the option, associated with the given callback object for a
         * built-in C++ data type (including "string"). See
         * yagol::OptionGeneric for the methods that must be defined for Type.
         */
        template < class Type >
        Option* addOptionCb(const string& name, 
                            const string& description, 
                            yagol::ArgCallback<Type>* const callback)
        {
            return addOption(new OptionGeneric<Type>(name, description, callback, strictArgs_));
        }

        /**
         * Adds the option, associated with the given callback object for a
         * vector of a built-in C++ data type (including "string"). See
         * yagol::OptionGeneric for the methods that must be defined for Type.
         */
        template < class Type >
        Option* addOptionCb(const string& name, 
                            const string& description, 
                            yagol::ArgCallback<vector< Type > >* const callback)
        {
            return addOption(new OptionMultiValues<vector<Type> >(name, description, callback, strictArgs_));
        }

        /**
         * Adds the option, associated with the given argument variable. See
         * yagol::OptionGeneric for the methods that must be defined for Type.
         */
        template < class Type >
        Option* addOptionRanged(const string& name, 
                                const string& description, 
                                Type lower,
                                Type upper,
                                Type* const value)
        {
            ArgCallback<Type>* validator = new RangeValidator<Type>(value, lower, upper);
            return addOption(new OptionGeneric<Type>(name, description, validator, strictArgs_));
        }

        /**
         * Adds the multi-value option, associated with the given argument
         * variable. See yagol::OptionMultiValues.
         */
        template < class Container >
        Option* addOptionRanged(const string& name, 
                                const string& description, 
                                typename Container::value_type lower,
                                typename Container::value_type upper,
                                vector<typename Container::value_type>* const values)
        {
            ArgCallback<vector< typename Container::value_type > >* validator = 
                new RangeValidator<vector< typename Container::value_type > >(values, lower, upper);
            return addOption(new OptionMultiValues<Container>(name, description, validator, strictArgs_));
        }

        /**
         * Adds the option, associated with the given argument variable and
         * valid for the given array of values.
         */
        template < class Type >
        Option* addOptionRanged(const string& name, 
                                const string& description, 
                                Type valids[],
                                int nValids,
                                Type* const value)
        {
            ArgCallback<Type>* validator = new GroupValidator<Type>(value, valids, nValids);
            return addOption(new OptionGeneric<Type>(name, description, validator, strictArgs_));
        }

        /**
         * Adds the option, associated with the given argument variable and
         * valid for the given array of values.
         */
        template < class Container >
        Option* addOptionRanged(const string& name, 
                                const string& description, 
                                typename Container::value_type valids[],
                                int nValids,
                                vector<typename Container::value_type>* const values)
        {
            ArgCallback<vector< typename Container::value_type > >* validator = 
                new GroupValidator<vector< typename Container::value_type > >(value, valids, nValids);
            return addOption(new OptionMultiValues<Container>(name, description, validator, strictArgs_));
        }

        /**
         * Adds the option. See the yagol::Option hierarchy for valid option
         * types. Note that ownership of the option object is taken by this
         * object, so <code>opt</code> will be deleted when this yagol object is
         * destroyed. Returns the option that was added.
         */
        virtual Option* addOption(Option* const opt);

        /**
         * Adds the processor.
         */
        virtual void addProcessor(Processor* const proc);

        /**
         * Makes the first option take on the value of the second option, if it
         * is not set explicitly.
         */
        virtual void addLink(const string& from, const string& to);

        /**
         * Returns the options (arguments) that were not consumed by the options.
         */
        virtual vector<char*> getUnprocessedArgs() const;

        /**
         * Checks for existence of an option with the same name in this option
         * set. Throws an exception when matches are found.
         */
        virtual void checkName(const string& name) const throw(yagol::Exception);

        /**
         * Returns the options for the given name. Note that this must be a full
         * match.
         */
        virtual vector<Option*> findByName(const string& name) const;

        /**
         * Returns the first option for the given name. Note that this must be a
         * full match.
         * 
         * @see findByName
         */
        virtual Option* findFirstByName(const string& name) const;

        /**
         * Writes the option list to the given stream. For developer usage only.
         */
        virtual ostream& print(ostream& os) const;

        /**
         * Returns whether we produced help/usage.
         */
        virtual bool showedHelp() const;

        /**
         * Sets whether to treat as an error when an argument looks like an
         * option, but is not.
         */
        virtual void setStrictArguments(bool strictArgs);

        /**
         * Returns the first option that is an exact match for the given name.
         */
        virtual Option* findExactMatch(const string& name) const;

        /**
         * Returns the first option that is an approximate match for the given
         * name. Will produce an error message if more than one option matches.
         */
        virtual Option* findApproximateMatch(const string& name, 
                                             vector<string>* const others) const;

    protected:

        virtual vector<Option*>& options();

        /**
         * Returns the names of all options that match the tag.
         *
         * @param tag The tag to compare the options against.
         */
        virtual vector<string> findMatches(const string& tag) const;

        /**
         * Complains about the number of matches, which is presumed to be more
         * than one, then throws an exception.
         */
        void complainAboutMatches(const vector<string>& matches, const string& tag) const
             throw(yagol::Exception);
        
    private:
        
        /**
         * The option handlers registered for this set. Note that an ordered map
         * would be much faster for a lot of the functionality we need.
         */
        vector<Option*> options_;

        /**
         * Whether to insist that arguments NOT match the tag pattern.
         */
        bool strictArgs_;

        /**
         * The options (arguments) not consumed by processing the options.
         */
        vector<char*> unprocessedArgs_;

        /**
         * Called after processing the argument list.
         */
        vector<Processor*> postProcessors_;

        /**
         * Denotes whether we produced help/usage.
         */
        mutable bool showedHelp_;
    };


    class VersionDisplayer;
    
    /**
     * In addition to the processing and usage functionality of
     * yagol::OptionList, processes configuration files prior to the
     * command-line processing, and has a mapping from known names (tags) to
     * values. See Features.t.cpp in the test directory for a plethora of
     * examples.
     */
    class AppOptionSet : public OptionList
    {
    public:

        enum Options { 
            STRICT_ARGUMENTS = 1 << 0, 
            HELP_ON_ERROR    = 1 << 1
        };
        
        /**
         * Program name, usage summary and description are used for displaying
         * the usage block. Options are added after construction.
         */
        AppOptionSet(const string& progName,
                     const string& progDesc = string(),
                     const string& usageSummary = string());

        /**
         * <code>strictArgs</code> defines whether to enforce strict arguments,
         * that is, whether to treat arguments that look like tags as errors if
         * an option for them cannot be found. The <code>helpOnError</code>
         * denotes whether to show help and terminate upon unsuccessful
         * processing. The program name, usage summary and description are used
         * for displaying the usage block. Options are added after construction.
         */
        AppOptionSet(bool strictArgs,
                     bool helpOnError,
                     const string& progName,
                     const string& progDesc = string(),
                     const string& usageSummary = string());

        /**
         * @param options A bit-ored set of Option enumerations.
         */
        AppOptionSet(int options,
                     const string& progName,
                     const string& progDesc = string(),
                     const string& usageSummary = string());

        /**
         * Destructor deletes all options.
         */
        virtual ~AppOptionSet();
    
        /**
         * Processes the argument list from argv, of size argc.
         */
        virtual void process(int argc, char** argv) throw(yagol::Exception);
    
        /**
         * Writes the usage to the given output stream. The format is that as for
         * --help:<p>
         *
         * <pre>
         * Usage: PROGRAM NAME
         * PROGRAM DESCRIPTION
         *
         * Options:
         * --opt     Option description (default = default)
         * </pre>
         *
         * Note that there is one tab between the option and the description.
         */
        virtual void writeUsage(ostream& os) const;

        /**
         * Returns the program name, suitable for the usage block.
         */
        virtual string programName() const;

        /**
         * Returns the description of the program, suitable for the usage block.
         */
        virtual string programDescription() const;

        /**
         * This method is called to set options from the configuration (RC)
         * files, prior to reading the environment variables and the command
         * line itself.
         */
        virtual void processConfigFiles();

        /**
         * This method is called to set options from the environment variables,
         * prior to the command line itself.
         */
        virtual void processEnvironmentVariables();

        /**
         * Handles exceptions thrown by the processor. If
         * <code>helpOnError_</code> is set to true, the usage statement is
         * written to standard error, and <code>hadError_</code> is set to
         * true. Otherwise, the exception is thrown.
         */
        virtual void onError(const yagol::Exception& e) throw(yagol::Exception);

        /**
         * Returns whether there was an error during processing.
         */
        virtual bool hadError() const;

        /**
         * Adds the configuration file. The one added will follow any previously
         * added configuration files. Note that the usage of "~" as the first
         * character of the string will be converted to the value of the HOME
         * environment variable.
         */
        virtual void addConfigFile(const string& file);

        /**
         * Adds the environment variable. The one added will follow any
         * previously added environment variables.
         */
        virtual void addEnvironmentVariable(const string& envVar);

        /**
         * Returns the fully-expanded name for the file, by converting a leading
         * tilde (~) into the values for the HOME environment variable, or
         * HOMEDRIVE and HOMEPATH under Windows.
         */
        virtual string expandFileName(const string& fname) const;

        /**
         * Returns the usage summary.
         */
        virtual string usageSummary() const;

        /**
         * Sets the package and version, and adds the "--version" option.
         */
        virtual void setVersion(const string& package, const string& version);

        /**
         * Sets whether to display help when an error occurs, such as an invalid
         * option.
         */
        virtual void setHelpOnError(bool helpOnError);

        /**
         * Returns whether the version was shown.
         */
        virtual bool displayedVersion() const;
        
    protected:

        /**
         * Reads the configuration file and forwards the name/value pairs to the
         * appropriate options.
         */
        virtual void readConfigFile(const string& file);

        /**
         * Reads the environment variable, splits it by whitespace, and
         * processes it as a command line.
         */
        virtual void readEnvironmentVariable(const string& envVar);

        void init(const string& progName,
                  const string& progDesc,
                  bool helpOnError, 
                  const string& usageSummary);

    private:

        /**
         * The name of the program (application).
         */
        string progName_;

        /**
         * The description of the program (application).
         */
        string progDesc_;

        /**
         * Whether to show help when processing fails.
         */
        bool helpOnError_;

        /**
         * Whether there was an error when processing the arguments.
         */
        bool hadError_;
        
        /**
         * The configuration files.
         */
        vector<string> configFiles_;

        /**
         * The environment variables.
         */
        vector<string> envVars_;

        /**
         * Summary of usage.
         */
        string usageSummary_;

        /**
         * Whether to show the version.
         */
        bool displayVersion_;

        /**
         * Whether the version was shown.
         */
        bool displayedVersion_;

        /**
         * Displays version information.
         */
        VersionDisplayer* versionDisplayer_;

    };
}

#endif //! Yagol_h
