#ifndef StringEdit_h
#define StringEdit_h

#ifndef std_string
#define std_string
#include <string>
#endif

#ifndef std_vector
#define std_vector
#include <vector>
#endif

#ifndef std_map
#define std_map
#include <map>
#endif

using namespace std;

namespace doctorj
{
    class File;

    class EditableString
    {
    public:
        static int INVALID_INDEX;
    
        EditableString(const string& str, char* const head);

        EditableString(const EditableString& rhs);

        EditableString* clone();

        virtual ~EditableString();

        void dump() const;

        void insert(int pos, const string& s);

        void remove(int pos, int len);

        void replace(int apos, int alen, const string& bstr);

        void move(int fromindex, int fromlen, int toindex, int toposition = -999);

        void insert(char* const pos, const string& s);

        void remove(char* const pos, int len);

        void replace(char* const apos, int alen, const string& bstr);

        void moveInsert(char* const apos, int alen, char* const bpos);

        void moveAfter(char* const apos, int alen, char* const bpos);
        
        string str() const;

        char* head() const;

        int index(char* const pos) const;

    protected:
        void shiftEachPosition(int from, int to, int amount);

        void shiftPosition(int index, int amount);

    private:
        string str_;

        vector<int> positions_;

        int npositions_;

        char* head_;
    
    };

    
    /**
     * Each instantiation of this class represents an edit of a string.
     */
    class StringEdit
    {
    public:
        /**
         * The given string will replace <code>len</code> number of characters,
         * starting at the given location. An empty string is the same as a
         * deletion; length of 0 is an insertion.
         */
        StringEdit();

        /**
         * Does not free any memory.
         */
        virtual ~StringEdit();

        /**
         * Runs the edit, and updates the editor.
         */
        virtual void execute(EditableString* const estr) const = 0;

        /**
         * Returns the location of the change to the string.
         */
        virtual char* position() const = 0;

        /**
         * Returns the last location of the change to the string.
         */
        virtual char* endPosition() const = 0;

        /**
         * Displays the text at the location where it would be changed.
         */
        virtual void showUnedited(File* const file);

        /**
         * Displays the text with this edit applied.
         */
        virtual void showEdited(File* const file);

    };

    
    /**
     * Each instantiation of this class represents an edit of a string.
     */
    class StringEditSingle : public StringEdit
    {
    public:
        /**
         * Creates an edit for the given position.
         */
        StringEditSingle(char* const pos);

        /**
         * Does not free any memory.
         */
        virtual ~StringEditSingle();

        /**
         * Runs the edit, and updates the editor.
         */
        virtual void execute(EditableString* const estr) const = 0;

        /**
         * Returns the location where the string should be edited.
         */
        virtual char* position() const;

        /**
         * Returns the end of the location where the string should be edited.
         */
        virtual char* endPosition() const;

        /**
         * Returns the number of characters to be replaced.
         */
        virtual int length() const = 0;

    private:
        /**
         * The edit position.
         */
        char* pos_;

    };

    
    /**
     * Each instantiation of this class represents the deletion of part of a
     * string.
     */
    class StringEditDelete : public StringEditSingle
    {
    public:
        /**
         * Removes the text between <code>from</code> and <code>to</code>,
         * inclusive, not exceeding the upper and lower bounds.
         */
        StringEditDelete(char* const from, 
                         char* const to, 
                         char* const lowerBound, 
                         char* const upperBound);

        /**
         * Removes the text between <code>from</code> and <code>to</code>,
         * inclusive.
         */
        StringEditDelete(char* const from, 
                         char* const to);

        /**
         * Removes the text between <code>from</code> and <code>from +
         * len</code>, inclusive.
         */
        StringEditDelete(char* const from, int len);

        /**
         * Does not free any memory.
         */
        virtual ~StringEditDelete();

        /**
         * Runs the edit, and updates the editor.
         */
        virtual void execute(EditableString* const estr) const;

        /**
         * Returns the length.
         */
        virtual int length() const;

        /**
         * Displays the text with this edit applied. Shows only the line of the
         * start position, since end could be on another line.
         */
        virtual void showEdited(File* const file);

    private:
        int len_;

    };


    /**
     * Each instantiation of this class represents the deletion of part of a
     * string.
     */
    class StringEditDeleteLines : public StringEditSingle
    {
    public:
        /**
         * Removes the lines between <code>from</code> and <code>to</code>,
         * inclusive, not exceeding the upper and lower bounds.
         */
        StringEditDeleteLines(char* const from, 
                              char* const to, 
                              char* const lowerBound, 
                              char* const upperBound);

        /**
         * Does not free any memory.
         */
        virtual ~StringEditDeleteLines();

        virtual void showEdited(File* const file);

        /**
         * Runs the edit, and updates the editor.
         */
        virtual void execute(EditableString* const estr) const;

        /**
         * Returns the length.
         */
        virtual int length() const;

    private:
        int len_;

    };


    /**
     * Each instantiation of this class represents an edit of a string.
     */
    class StringEditInsert : public StringEditSingle
    {
    public:
        /**
         * The given string will be inserted, starting at the given location.
         */
        StringEditInsert(const string& str, char* pos);

        /**
         * Does not free any memory.
         */
        virtual ~StringEditInsert();

        /**
         * Returns the string that is replacing the current contents.
         */
        string str() const;

        /**
         * Returns the number of characters to be replaced.
         */
        int length() const;

        /**
         * Displays the text with this edit applied.
         */
        virtual void showEdited(File* const file);

        /**
         * Displays the text at the location where it would be changed.
         */
        virtual void showUnedited(File* const file);

        /**
         * Displays the existing text at the given line.
         */
        virtual void showUneditedLine(File* const file, char* const cstart);

        virtual void execute(EditableString* const estr) const;
             
    private:
        string str_;

    };


    /**
     * Each instantiation of this class represents an edit of a string.
     */
    class StringEditInsertLine : public StringEditInsert
    {
    public:
        /**
         * The given string will be inserted, starting at the given location.
         */
        StringEditInsertLine(const string& str, char* pos);

        /**
         * Does not free any memory.
         */
        virtual ~StringEditInsertLine();

        /**
         * Displays the text at the location where it would be changed. Since
         * this is a new line (i.e., there is no existing text where this line
         * is going in) this just says "<<none>>".
         */
        virtual void showUneditedLine(File* const file, char* const cstart);

    };

    
    /**
     * Each instantiation of this class represents an edit of a string.
     */
    class StringEditMove : public StringEdit
    {
    public:
        /**
         * The given string will be inserted, starting at the given location.
         */
        StringEditMove(char* from, int len, char* to);

        virtual ~StringEditMove();

        /**
         * Runs the edit, and updates the editor.
         */
        virtual void execute(EditableString* const estr) const;

        /**
         * Returns the location where the string should be edited.
         */
        virtual char* position() const;

        /**
         * Returns the end of the location where the string should be edited.
         */
        virtual char* endPosition() const;

    protected:
        char* from_;

        int length_;

        char* to_;

    };


    /**
     * Each instantiation of this class represents an edit of a string.
     */
    class StringEditMoveAfter : public StringEditMove
    {
    public:
        /**
         * The given string will be inserted, starting at the given location.
         */
        StringEditMoveAfter(char* from, int len, char* to);

        virtual ~StringEditMoveAfter();

        /**
         * Runs the edit, and updates the editor.
         */
        virtual void execute(EditableString* const estr) const;

    };


    /**
     * Each instantiation of this class represents an edit of a string.
     */
    class StringEditMoveBefore : public StringEditMove
    {
    public:
        /**
         * The given string will be inserted, starting at the given location.
         */
        StringEditMoveBefore(char* from, int len, char* to);

        virtual ~StringEditMoveBefore();

    };


    class StringEditMulti : public StringEdit
    {
    public:
        StringEditMulti(const vector<StringEdit*>& edits);

        StringEditMulti();

        virtual ~StringEditMulti();

        /**
         * Adds the edit.
         */
        virtual void add(StringEdit* const edit);
        
        /**
         * Runs the edit, but does not update the editor.
         */
        virtual void apply(EditableString* const estr) const;

        /**
         * Runs the edit, and updates the editor.
         */
        virtual void execute(EditableString* const estr) const;

        /**
         * Returns the location of the change to the string.
         */
        virtual char* position() const;

        /**
         * Returns the last location of the change to the string.
         */
        virtual char* endPosition() const;

    private:
        /**
         * The edits.
         */
        vector<StringEdit*> edits_;

    };


    /**
     * Each instantiation of this class represents an edit of a string.
     */
    class StringEditReplace : public StringEditSingle
    {
    public:
        /**
         * The given string will replace <code>len</code> number of characters,
         * starting at the given location. An empty string is the same as a
         * deletion; length of 0 is an insertion.
         */
        StringEditReplace(const string& str, char* pos, int len);

        /**
         * Does not free any memory.
         */
        virtual ~StringEditReplace();

        /**
         * Returns the string is replacing the current contents.
         */
        string str() const;

        /**
         * Returns the number of characters to be replaced.
         */
        int length() const;

        virtual void execute(EditableString* const estr) const;
        
    private:
        string str_;

        int len_;

    };


    /**
     * This class edits a string. It is not an editor in the Emacs sense. (Then
     * again, nothing else is.)
     */
    class StringEditor
    {
    public:
        /**
         * Edits the given string.
         */
        StringEditor(char* str);

        /**
         * Creates a new editor, using the values of the old one. 
         */
        StringEditor(const StringEditor& rhs);

        /**
         * Deletes the edits added.
         */
        virtual ~StringEditor();

        /**
         * Returns whether there are pending edits.
         */
        bool hasEdits() const;

        /**
         * Returns the new string.
         */
        string str() const;

        /**
         * Adds an edit.
         */
        void add(StringEdit* const edit);

        /**
         * Executes all edits.
         */
        void execute();

        /**
         * Returns the current string, with the given edit applied. The existing
         * string and edits are not changed.
         */
        string preview(const StringEdit& edit);

        /**
         * Returns the index.
         */
        virtual int index(char* const pos) const;

        /**
         * Returns the original string.
         */
        virtual string originalStr() const;

    protected:
        /**
         * Executes the given edit.
         */
        void execute(const StringEdit& edit);

        /**
         * So that this can be subclassed. Subclasses should call
         * <code>init</code> before methods are invoked.
         */
        StringEditor();

        /**
         * Applies the given edit to the given string.
         */
        void apply(const StringEdit& edit);

        /**
         * Sets the original string.
         */
        void init(char* str);

    private:

        EditableString* estr_;

        char* orig_;

        /**
         * The edits to be executed.
         */
        vector<StringEdit*> edits_;

    };

}

#endif //! StringEdit_h
