#ifndef Platform_h
#include "Platform.h"
#endif

#ifndef File_h
#include "File.h"
#endif

#ifndef StringUtilities_h
#include "StringUtilities.h"
#endif

#ifndef UtilDebug_h
#include "UtilDebug.h"
#endif

#ifndef std_algorithm
#define std_algorithm
#include <algorithm>
#endif

#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>
#include <dirent.h>

using namespace std;
using namespace doctorj;

bool Platform::isDirectory(const string& name)
{
    struct stat st;
    int status = stat(name.c_str(), &st);
    return status == 0 && (st.st_mode & S_IFDIR);
}

bool Platform::isRegularFile(const string& name)
{
    struct stat st;
    int status = stat(name.c_str(), &st);
    return status == 0 && (st.st_mode & S_IFREG);
}

bool Platform::isLink(const string& name)
{
    struct stat st;
    int status = stat(name.c_str(), &st);
    return status == 0 && (st.st_mode & S_IFLNK);
}

bool Platform::isReadableFile(const string& name)
{
    return access(name.c_str(), F_OK) == 0;
}

// there is far too much repeated code in here.

void Platform::findFiles(const string& dirname, 
                         const string& ext, 
                         vector<string>* const files)
{
    FileFinder ff(files, ext);
    ff.processDirectory(dirname);
}

void Platform::getFiles(const string& dirname, 
                        vector<string>* const files)
{
    FileFilter ff(files);
    ff.processDirectory(dirname);
}

void Platform::getFiles(const string& dirname, 
                        const string& ext0,
                        const string& ext1,
                        vector<string>* const files)
{
    FileFilter ff(files, ext0);
    ff.addExtension(ext1);
    ff.processDirectory(dirname);
}

void Platform::getFiles(const string& dirname, 
                        const string& ext,
                        vector<string>* const files)
{
    FileFilter ff(files, ext);
    ff.processDirectory(dirname);
}

void Platform::getEntries(const string& dirname, 
                          vector<string>* const entries)
{
    FileFilter ff(entries);
    ff.processDirectory(dirname);
}

bool Platform::getEnv(const string& name, string* const value /* = NULL */)
{
    char* val = getenv(name.c_str());
    if (val) {
        if (value) {
            *value = val;
        }
        return true;
    }
    else {
        return false;
    }
}

string Platform::getCurrentDirectory()
{
    char dir[1024];
    getcwd(dir, 1024);
    return dir;
}

void Platform::renameFile(const string& from, const string& to)
{
    rename(from.c_str(), to.c_str());
}

bool Platform::ensureDirectoryExists(const string& path)
{
    DEBUG_UTIL(cout << "PLFRM  ensureDirectoryExists(" << path << ")" << endl);
    vector<string> pieces = StringUtilities::split(path, "/");
    string dir;
    for (int i = 0; i < pieces.size(); ++i) {
        DEBUG_UTIL(cout << "PLFRM  pieces[" << i << "] = '" << pieces[i] << "'" << endl);
        if (i == 0) {
            if (pieces[i].size() == 0) {
                DEBUG_UTIL(cout << "PLFRM  initial piece is empty" << endl);
            }
            else {
                DEBUG_UTIL(cout << "PLFRM  initial piece is " << pieces[i] << endl);
                dir = pieces[i];
            }
        }
        else if (pieces[i].size() == 0) {
            DEBUG_UTIL(cout << "PLFRM  skipping empty directory piece" << endl);
        }
        else {
            dir += "/" + pieces[i];
            DEBUG_UTIL(cout << "PLFRM  directory is '" << dir << "'" << endl);
        }
        DEBUG_UTIL(cout << "PLFRM  at i " << i << ", dir = '" << dir << "'" << endl);

        if (dir.length() > 0) {
            if (isDirectory(dir)) {
                DEBUG_UTIL(cout << "PLFRM  isDirectory(" << dir << ") = true" << endl);
            }
            else {
                DEBUG_UTIL(cout << "PLFRM  isDirectory(" << dir << ") = false" << endl);
                if (mkdir(dir.c_str(), 0775)) {
                    DEBUG_UTIL(cout << "PLFRM  mkdir did not work" << endl);
                    return false;
                }
                else {
                    DEBUG_UTIL(cout << "PLFRM  mkdir worked" << endl);
                }
            }
        }
    }

    return true;
}


FileProcessor::FileProcessor(vector<string>* const matches) : matches_(matches)
{
}

FileProcessor::~FileProcessor()
{
}

void FileProcessor::processDirectory(const string& dirname)
{
    // POSIX version:
    DIR* dir = opendir(dirname.c_str());
    if (dir) {
        dirent* de = NULL;
        while ((de = readdir(dir)) != NULL) {
            string name = de->d_name;
            if (name != "." && name != "..") {
                string fullname = dirname;
                if (fullname[fullname.length() - 1] != '/') {
                    fullname += "/";
                }
                fullname += name;
                handle(fullname);
            }
        }
        closedir(dir);
    }
}


FileFilter::FileFilter(vector<string>* const matches, const string& ext /* = string("") */) : FileProcessor(matches) 
{
    if (ext.length() > 0) {
        extns_.push_back(ext);
    }
}

FileFilter::~FileFilter()
{
}

void FileFilter::addExtension(const string& ext)
{ 
    extns_.push_back(ext);
}

bool FileFilter::matches(const string& fname) const
{
    return extns_.size() == 0 || find(extns_.begin(), extns_.end(), FileName::extension(fname)) != extns_.end();
}

void FileFilter::handle(const string& fname)
{
    if (matches(fname)) {
        matches_->push_back(fname);
    }
}


FileFinder::FileFinder(vector<string>* const matches, const string& ext /* = string("") */)
        : FileFilter(matches, ext)
{
}

FileFinder::~FileFinder()
{
}

void FileFinder::handle(const string& fname)
{
    if (Platform::isDirectory(fname)) {
        processDirectory(fname);
    }
    else {
        FileFilter::handle(fname);
    }
}
