#ifndef ErrorNameStandard_h
#define ErrorNameStandard_h

#ifndef NamingError_h
#include "NamingError.h"
#endif

namespace doctorj
{
    // trailing underscores to avoid collisions (e.g., "PACKAGE").
    extern char PACKAGE_[];
    extern char PRIVATE_[];
    extern char PROTECTED_[];
    extern char PUBLIC_[];
    extern char FIELD_[];

    template <bool IsStatic, bool IsFinal, char* Access, char* Type>
    class ErrorNameStandard : public NamingError
    {
    public:
        ErrorNameStandard();

        /**
         * Creates the error for the item with the offensive name.
         */
        ErrorNameStandard(Reporter* const reporter, 
                          AstItem* const item, 
                          const string& name, 
                          const string& standard);

        virtual ~ErrorNameStandard();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

    protected:
        /**
         * Returns what type of thing this refers to, in lowercase format.
         */
        virtual string typeAsLowercase() const;
   
        /**
         * Returns what type of thing this refers to, in capitalized format.
         */
        virtual string typeAsCapitalized() const;

    private:
        /**
         * The name of the offensive item.
         */
        string name_;

        /**
         * The standard for this type of item.
         */
        string standard_;
        
    };

    string convert(const string& str, bool toUpper);

    template <bool IsStatic, bool IsFinal, char* Access, char* Type>
    ErrorNameStandard<IsStatic, IsFinal, Access, Type>::ErrorNameStandard()
    {
    }

    template <bool IsStatic, bool IsFinal, char* Access, char* Type>
    ErrorNameStandard<IsStatic, IsFinal, Access, Type>::ErrorNameStandard(Reporter* const reporter,
                                                                          AstItem* const item, 
                                                                          const string& name,
                                                                          const string& standard) :
            NamingError(reporter, item), name_(name), standard_(standard)
    {
    }

    template <bool IsStatic, bool IsFinal, char* Access, char* Type>
    ErrorNameStandard<IsStatic, IsFinal, Access, Type>::~ErrorNameStandard()
    {
    }

    template <bool IsStatic, bool IsFinal, char* Access, char* Type>
    string ErrorNameStandard<IsStatic, IsFinal, Access, Type>::title() const
    {
        return "Format of " + typeAsCapitalized() + " Name";
    }

    template <bool IsStatic, bool IsFinal, char* Access, char* Type>
    void ErrorNameStandard<IsStatic, IsFinal, Access, Type>::writeDescription(ostream& os) const
    {
        os << "Name of " << typeAsLowercase() << " name is inconsistent with the convention." << endl;
    }

    template <bool IsStatic, bool IsFinal, char* Access, char* Type>
    string ErrorNameStandard<IsStatic, IsFinal, Access, Type>::message() const
    {
        return "name of " + typeAsLowercase() + 
            " '" + name_ + "' is inconsistent with the naming standard: " + standard_;
    }

    template <bool IsStatic, bool IsFinal, char* Access, char* Type>
    string ErrorNameStandard<IsStatic, IsFinal, Access, Type>::typeAsCapitalized() const
    {
        string s(IsStatic ? "Static" : "Instance");
        string f(IsFinal  ? "Final"  : "Nonfinal");
        string a(convert(Access, true));
        string t(convert(Type, true));
        return s + " " + f + " " + a + " " + t;
    }

    template <bool IsStatic, bool IsFinal, char* Access, char* Type>
    string ErrorNameStandard<IsStatic, IsFinal, Access, Type>::typeAsLowercase() const
    {
        string s(IsStatic ? "static" : "instance");
        string f(IsFinal  ? "final"  : "nonfinal");
        string a(convert(Access, false));
        string t(convert(Type, false));
        return s + " " + f + " " + a + " " + t;
    }


    class ErrorInstanceFinalPackageFieldName : public ErrorNameStandard<false, true, PACKAGE_, FIELD_> {
    public:
        ErrorInstanceFinalPackageFieldName();

        ErrorInstanceFinalPackageFieldName(Reporter* const reporter, AstItem* const item, const string& name, const string& standard);

        virtual ~ErrorInstanceFinalPackageFieldName();

    };


    class ErrorInstanceFinalPrivateFieldName : public ErrorNameStandard<false, true, PRIVATE_, FIELD_> {
    public:
        ErrorInstanceFinalPrivateFieldName();

        ErrorInstanceFinalPrivateFieldName(Reporter* const reporter, AstItem* const item, const string& name, const string& standard);

        virtual ~ErrorInstanceFinalPrivateFieldName();

    };


    class ErrorInstanceFinalProtectedFieldName : public ErrorNameStandard<false, true, PROTECTED_, FIELD_> {
    public:
        ErrorInstanceFinalProtectedFieldName();

        ErrorInstanceFinalProtectedFieldName(Reporter* const reporter, AstItem* const item, const string& name, const string& standard);

        virtual ~ErrorInstanceFinalProtectedFieldName();

    };


    class ErrorInstanceFinalPublicFieldName : public ErrorNameStandard<false, true, PUBLIC_, FIELD_> {
    public:
        ErrorInstanceFinalPublicFieldName();

        ErrorInstanceFinalPublicFieldName(Reporter* const reporter, AstItem* const item, const string& name, const string& standard);

        virtual ~ErrorInstanceFinalPublicFieldName();

    };


    class ErrorInstanceNonfinalPackageFieldName : public ErrorNameStandard<false, false, PACKAGE_, FIELD_> {
    public:
        ErrorInstanceNonfinalPackageFieldName();

        ErrorInstanceNonfinalPackageFieldName(Reporter* const reporter, AstItem* const item, const string& name, const string& standard);

        virtual ~ErrorInstanceNonfinalPackageFieldName();

    };


    class ErrorInstanceNonfinalPrivateFieldName : public ErrorNameStandard<false, false, PRIVATE_, FIELD_> {
    public:
        ErrorInstanceNonfinalPrivateFieldName();

        ErrorInstanceNonfinalPrivateFieldName(Reporter* const reporter, AstItem* const item, const string& name, const string& standard);

        virtual ~ErrorInstanceNonfinalPrivateFieldName();

    };


    class ErrorInstanceNonfinalProtectedFieldName : public ErrorNameStandard<false, false, PROTECTED_, FIELD_> {
    public:
        ErrorInstanceNonfinalProtectedFieldName();

        ErrorInstanceNonfinalProtectedFieldName(Reporter* const reporter, AstItem* const item, const string& name, const string& standard);

        virtual ~ErrorInstanceNonfinalProtectedFieldName();

    };


    class ErrorInstanceNonfinalPublicFieldName : public ErrorNameStandard<false, false, PUBLIC_, FIELD_> {
    public:
        ErrorInstanceNonfinalPublicFieldName();

        ErrorInstanceNonfinalPublicFieldName(Reporter* const reporter, AstItem* const item, const string& name, const string& standard);

        virtual ~ErrorInstanceNonfinalPublicFieldName();

    };


    class ErrorStaticFinalPackageFieldName : public ErrorNameStandard<true, true, PACKAGE_, FIELD_> {
    public:

        ErrorStaticFinalPackageFieldName();

        ErrorStaticFinalPackageFieldName(Reporter* const reporter, AstItem* const item, const string& name, const string& standard);

        virtual ~ErrorStaticFinalPackageFieldName();

    };


    class ErrorStaticFinalPrivateFieldName : public ErrorNameStandard<true, true, PRIVATE_, FIELD_> {
    public:
        ErrorStaticFinalPrivateFieldName();

        ErrorStaticFinalPrivateFieldName(Reporter* const reporter, AstItem* const item, const string& name, const string& standard);

        virtual ~ErrorStaticFinalPrivateFieldName();

    };


    class ErrorStaticFinalProtectedFieldName : public ErrorNameStandard<true, true, PROTECTED_, FIELD_> {
    public:
        ErrorStaticFinalProtectedFieldName();

        ErrorStaticFinalProtectedFieldName(Reporter* const reporter, AstItem* const item, const string& name, const string& standard);

        virtual ~ErrorStaticFinalProtectedFieldName();

    };


    class ErrorStaticFinalPublicFieldName : public ErrorNameStandard<true, true, PUBLIC_, FIELD_> {
    public:
        ErrorStaticFinalPublicFieldName();

        ErrorStaticFinalPublicFieldName(Reporter* const reporter, AstItem* const item, const string& name, const string& standard);

        virtual ~ErrorStaticFinalPublicFieldName();

    };


    class ErrorStaticNonfinalPackageFieldName : public ErrorNameStandard<true, false, PACKAGE_, FIELD_> {
    public:
        
        ErrorStaticNonfinalPackageFieldName();

        ErrorStaticNonfinalPackageFieldName(Reporter* const reporter, AstItem* const item, const string& name, const string& standard);

        virtual ~ErrorStaticNonfinalPackageFieldName();

    };


    class ErrorStaticNonfinalPrivateFieldName : public ErrorNameStandard<true, false, PRIVATE_, FIELD_> {
    public:
        ErrorStaticNonfinalPrivateFieldName();

        ErrorStaticNonfinalPrivateFieldName(Reporter* const reporter, AstItem* const item, const string& name, const string& standard);

        virtual ~ErrorStaticNonfinalPrivateFieldName();

    };


    class ErrorStaticNonfinalProtectedFieldName : public ErrorNameStandard<true, false, PROTECTED_, FIELD_> {
    public:
        ErrorStaticNonfinalProtectedFieldName();

        ErrorStaticNonfinalProtectedFieldName(Reporter* const reporter, AstItem* const item, const string& name, const string& standard);

        virtual ~ErrorStaticNonfinalProtectedFieldName();

    };


    class ErrorStaticNonfinalPublicFieldName : public ErrorNameStandard<true, false, PUBLIC_, FIELD_> {
    public:

        ErrorStaticNonfinalPublicFieldName();

        ErrorStaticNonfinalPublicFieldName(Reporter* const reporter, AstItem* const item, const string& name, const string& standard);

        virtual ~ErrorStaticNonfinalPublicFieldName();

    };

}

#endif //! ErrorNameStandard_h
