#ifndef Java_h
#define Java_h

#ifndef AST_h
#include "AST.h"
#endif

#ifndef Log_h
#include "Log.h"
#endif

#ifndef Processor_h
#include "Processor.h"
#endif

#ifndef std_map
#define std_map
#include <map>
#endif

#ifndef std_vector
#define std_vector
#include <vector>
#endif

#ifndef std_string
#define std_string
#include <string>
#endif

#ifndef Debug_h
#include "Debug.h"
#endif

#ifndef stdio_h
#include "stdio.h"
#endif

#ifdef DEBUGGING
#define DEBUG_JAVA_CODE(a)      if (DebugOptions::get()->isEnabled('j')) { a;                  }
#define JAVALOG(a)              if (DebugOptions::get()->isEnabled('j')) { LOG(4, a);          }
#define JAVALOGF(fmt, args...)  if (DebugOptions::get()->isEnabled('j')) { LOGF(4, fmt, args); }
#else
#define DEBUG_JAVA_CODE(a)  
#define JAVALOG(a)          
#define JAVALOGF(fmt, args...)
#endif


namespace doctorj
{
    class Reporter;
    
    class JavaArgumentType
    {
    public:
        JavaArgumentType();

        virtual ~JavaArgumentType();

        virtual bool matches(AstFormalParameter* const fp) const;

    };


    class JavaArgumentTypeList
    {
    public:
        JavaArgumentTypeList();

        virtual ~JavaArgumentTypeList();

        virtual void add(JavaArgumentType* const argType);

        virtual int count() const;

        /**
         * Returns whether this argument type list matches the given method
         * declarator.
         */
        virtual bool matches(AstMethodDeclarator* const md) const;

    private:
        vector<JavaArgumentType*> arguments_;

    };


    /**
     * Wraps a method.
     */
    class JavaMethod
    {
    public:
        JavaMethod(AstMethodHeader* const header);

        JavaMethod(AstMethodDeclaration* const method);
        
        virtual ~JavaMethod();

        virtual bool isMatch(const string& name, const JavaArgumentTypeList& argTypes) const;

        /**
         * Returns the type, which is of class AstItem, or AstVoid, or NULL.
         */
        AstItem* getReturnType() const;

        /**
         * Returns whether this method is declared as abstract.
         */
        bool isAbstract() const;

        /**
         * Returns the name of the method.
         */
        string getName() const;

    private:

        AstMethodHeader* header_;

    };


    /**
     * A type declaration (class or interface).
     */
    class JavaType 
    {
    public:
        JavaType();

        virtual ~JavaType();

        /**
         * Returns whether the class has a "local" method for the given name and
         * argument types. A local method is one that is defined in the class
         * itself, as opposed to being inherited.
         */
        virtual bool hasLocalMethod(const string& name, const JavaArgumentTypeList& argTypes) const;

        /**
         * Returns the methods (as JavaMethod objects) for this type.
         */
        virtual vector<JavaMethod> getMethods() const = 0;

    };


    /**
     * A class declaration.
     */
    class JavaClass : public JavaType
    {
    public:
        JavaClass(AstClassDeclaration* const cd);
        
        virtual ~JavaClass();

        virtual vector<JavaMethod> getMethods() const;

    private:
        AstClassDeclaration* cd_;
    };


    class JavaClassLoader
    {
    public:
        static JavaClassLoader* get(const string& classPath);
        
        JavaClassLoader(const string& classPath);

        virtual ~JavaClassLoader();
        
        bool getTypesInPackage(const string& pkgName, vector<string>* const types);

    protected:

        /**
         * Looks through the given list of types for the public one. There can
         * be at most one public type in a list of type declarations within a
         * compilation unit.
         */
        bool getPublicType(AstTypeDeclarationList* const typelist, vector<string>* const types);

        //-------------------------------------------------------
        // class path processing

        bool readClassFile(const string& cfname, const string& pkgName, vector<string>* const types);

        bool readClassFiles(const vector<string>& classes, const string& pkgName, vector<string>* const types);

        bool readClassPath(const string& pkgName, vector<string>* const types);
        
        bool readArchiveFile(const string& archive, const string& pkgName, vector<string>* const types);

        void readPackages();

    private:
        static JavaClassLoader* instance_;

        string classPath_;

    };


    /**
     * Represents the AstEqualityExpression.
     */
    class JavaEqualityExpression
    {
    public:
        static bool hasTrueLiteralInExpression(AstEqualityExpression* const ee);

        static bool hasFalseLiteralInExpression(AstEqualityExpression* const ee);

    };


    /**
     * Creates various types of Java objects, acting as a bridge between this
     * module and the bc and ast modules.
     */
    class JavaFactory
    {
    public:
//         static JavaType* makeType(BcTypeDeclaration* const btd);
    };


    /**
     * Wraps a field declaration.
     */
    class JavaFieldDeclaration
    {
    public:
        template <class ErrorType>
        static void complainAboutVariableList(Reporter* const reporter, AstFieldDeclaration* const fd) 
        {
            AstVariableDeclaratorList* vars = fd->getVariableDeclaratorList();
            int count = vars->getVariableDeclaratorCount();
            for (int i = 0; i < count; ++i) {
                AstVariableDeclarator* vd = vars->getVariableDeclarator(i);
                ErrorType* err = new ErrorType(reporter, vd, vd->name());
                err->process();
            }
        }
    };


    /**
     * Represent the AST as a cleaner grammar.
     */
    class JavaGrammar
    {
    public:        
        static JavaArgumentTypeList getArgumentTypes(AstMethodInvocationSuperArgs* const mi);

        static JavaArgumentTypeList getArgumentTypes(AstMethodInvocationSuperNoArgs* const mi);

        static bool isPrimitiveType(AstItem* const varType);

    };


    /**
     * A class declaration.
     */
    class JavaInterface : public JavaType
    {
    public:
        JavaInterface(AstInterfaceDeclaration* const id);
        
        virtual ~JavaInterface();

        virtual vector<JavaMethod> getMethods() const;

    private:
        AstInterfaceDeclaration* id_;
    };


    /**
     * Represents the AST as a more cleanly structured grammar.
     */
    class JavaItem
    {
    public:
        static JavaType* getContainingType(AstItem* const item);
        
        /**
         * Returns the name(s) of the item. Only variable declarations may have
         * multiple names.
         */
        static vector<string> namesOf(AstItem* const item);

        /**
         * Returns the main item that is identified with the given item. For
         * example, of a method, usually the method declarator is identified as
         * the "method", not the block of code associated with it.
         */
        static AstItem* primaryItem(AstItem* const item);

        /**
         * Returns the list of modifiers for the given item.
         */
        static AstModifierList* getModifierList(AstItem* const item);

        /**
         * Returns the type for this item. The type is in the way the public
         * views it, e.g., "method", "field", "ctor", not the grammar-ish way we
         * do "MethodDeclarationNoMods".
         */
        static string typeOf(AstItem* const item);

        /**
         * Returns the first AstLeaf in the vector that can be downcast to the
         * given Type.
         */
        template <class Type>
        static Type firstOfType(const vector<AstLeaf*>& vec) {
            vector<AstLeaf*>::const_iterator stop = vec.end();
            vector<AstLeaf*>::const_iterator it   = vec.begin();
            for (; it != stop; ++it) {
                AstLeaf* leaf = *it;
                Type dct = dynamic_cast<Type>(leaf);
                if (dct) {
                    return dct;
                }
            }
            // not found:
            return NULL;
        }

        static int lineCount(AstItem* const item);
        
        static int lineCount(AstItem* const from, AstItem* const to);

        /**
         * Returns whether the item is a semicolon, and there is no comment
         * denoting intent.
         */
        static bool isEmptyStatement(AstItem* const statement);

        /**
         * Returns the trailing noncode.
         */
        static AstNoncode* getTrailingNoncode(AstItem* const item);

    };


    /**
     * Associates a name with a list of types.
     */
    class JavaPackage
    {
    public:
        JavaPackage(const string& pkgName);

        virtual ~JavaPackage();

        /**
         * Returns the name of this package.
         */
        string getName() const;

        /**
         * Adds a type to this package.
         */
        void addType(const string& type);
        
        /**
         * Passes back a copy of the list of the types in this package.
         */
        void getTypes(vector<string>* const types) const;

    private:
        string pkgName_;
        
        vector<string> types_;

    };


    class JavaSourceLoader
    {
    public:
        static JavaSourceLoader* get(AstItem* const item, const string& sourcePath);
        
        JavaSourceLoader(AstItem* const item, const string& sourcePath);

        virtual ~JavaSourceLoader();
        
        bool getTypesInPackage(const string& pkgName, vector<string>* const types);

    protected:

        /**
         * Looks through the given list of types for the public one. There can
         * be at most one public type in a list of type declarations within a
         * compilation unit.
         */
        bool getPublicType(AstTypeDeclarationList* const typelist, vector<string>* const types);

        //-------------------------------------------------------
        // compilation unit processing

        /**
         * Reads the compilation unit, adding the public type if the package
         * name of the compilation unit matches the given one.
         */
        bool readCompilationUnit(AstCompilationUnit* const cu, const string& pkgName, vector<string>* const types);

        /**
         * Reads all compilation units, adding the public type of each
         * compilation unit with a package name matching the given one.
         */
        bool readCompilationUnits(const string& pkgName, vector<string>* const types);

        //-------------------------------------------------------
        // source path processing

        /**
         * Reads everything in the source path. The source path consists of
         * colon-delimited files and directories. Files are .java files, which
         * will be parsed and their public types added if their package name
         * matches the given one. If a directory "/usr/local/foo" is given and
         * the package is "bar.baz", then all .java files in
         * "/usr/local/foo/bar/baz" will be parsed and their public types added
         * if the package name matches.
         */
        bool readSourcePath(const string& pkgName, vector<string>* const types);

        /**
         * Reads the given .java file.
         */
        bool readSourceFile(const string& sourceFile, const string& pkgName, vector<string>* const types);

    private:
        static JavaSourceLoader* instance_;
        
        AstProject* project_;

        vector<JavaPackage*>* packages_;

        string sourcePath_;

    };


    class JavaLoader
    {
    public:
        static JavaLoader* get(AstItem* const item, const string& sourcePath, const string& classPath);
        
        JavaLoader(AstItem* const item, const string& sourcePath, const string& classPath);

        virtual ~JavaLoader();
        
        bool getTypesInPackage(const string& pkgName, vector<string>* const types);

    protected:

    private:
        static JavaLoader* instance_;
        
        JavaSourceLoader sourceLoader_;

        JavaClassLoader classLoader_;

    };


    /**
     * Represents a modifier list. Also represents the lack of a modifier list,
     * in which case access will be "package", for example.
     */
    class JavaModifierList
    {
    public:
        /**
         * Wraps the modifier list of the given method. This object is still
         * valid if there is no modifier list associated with the method.
         */
        JavaModifierList(AstMethodDeclaration* const method);

        /**
         * Wraps the modifier list of the given class. This object is still
         * valid if there is no modifier list associated with the class.
         */
        JavaModifierList(AstClassDeclaration* const cls);

        /**
         * Wraps the modifier list of the given interface. This object is still
         * valid if there is no modifier list associated with the interface.
         */
        JavaModifierList(AstInterfaceDeclaration* const id);

        /**
         * Wraps the modifier list of the given field. This object is still
         * valid if there is no modifier list associated with the field.
         */
        JavaModifierList(AstFieldDeclarationMods* const field);

        /**
         * Wraps the given modifier list. This object is still valid if the
         * modifier list is NULL.
         */
        JavaModifierList(AstModifierList* const modifiers);

        virtual ~JavaModifierList();

        /**
         * Returns whether static is in this modifier list.
         */
        bool containsStatic() const;

        /**
         * Returns whether final is in this modifier list.
         */
        bool containsFinal() const;
        
        /**
         * Returns the object (AstLeaf) that denotes access type. Returns NULL
         * if this modifier list does not exist or contains no specified access.
         */
        AstLeaf* getAccess() const;
        
        /**
         * Returns a string denoting access type. Returns "package" if this
         * modifier list does not exist or contains no specified access.
         */
        string getAccessString() const;

        /**
         * Returns the string describing the access in the modifier list.
         */
        static string accessOf(AstModifierList* const mods);

        /**
         * Returns the string describing the finality (final or nonfinal) in
         * the modifier list.
         */
        static string finalityOf(AstModifierList* const mods);

        /**
         * Returns the string describing the ownership (static or instance) in
         * the modifier list.
         */
        static string ownershipOf(AstModifierList* const mods);
        
        /**
         * Returns the object in this modifier list that denotes access type.
         * Returns NULL if the object has no specified access.
         */
        static AstLeaf* getAccess(AstModifierList* const mods);

        /**
         * Returns the (first) modifier that can be downcast to AstAbstract.
         * Returns NULL if none exists.
         */
        static AstAbstract* getAbstractModifier(AstModifierList* const mods);

        /**
         * Returns the (first) modifier that can be downcast to AstStatic. Returns
         * NULL if none exists.
         */
        static AstStatic* getStaticModifier(AstModifierList* const mods);

        /**
         * Returns the (first) modifier that can be downcast to AstFinal. Returns
         * NULL if none exists.
         */
        static AstFinal* getFinalModifier(AstModifierList* const mods);

    private:
        AstModifierList* modifiers_;

    };


    /**
     * A sequence of noncode, i.e., comments and whitespace.
     */
    class JavaNoncode
    {
    public:
        static AstComment* getFirstComment(AstNoncode* const nc);

        static AstJavadocComment* getFirstJavadocComment(AstNoncode* const nc);
    };


    class JavaParameter
    {
    public:
        JavaParameter();

        JavaParameter(AstFormalParameter* const fp);

        virtual ~JavaParameter();

        static bool isMatch(AstFormalParameter* const param, AstItem* const item);

        static bool isMatch(AstFormalParameter* const param, const string& name);

        /**
         * Returns the type, as a string.
         */
        string getType() const;

        /**
         * Returns whether this parameter is declared as final.
         */
        bool isFinal() const;

        /**
         * Returns the names of the variables.
         */
        string getName() const;

    private:
        AstFormalParameter* fp_;

    };


    class JavaParameterList
    {
    public:
        JavaParameterList();

        virtual ~JavaParameterList();

        static AstFormalParameter* getMatching(AstFormalParameterList* const params, AstItem* const item);

        static AstFormalParameter* getMatching(AstFormalParameterList* const params, const string& name);

    };


    /**
     * Wraps a method.
     */
    class JavaSwitchStatement
    {
    public:
        static AstSwitchLabelDefault* getFirstSwitchLabelDefault(AstSwitchLabelList* const sll);
        
        static AstSwitchLabelDefault* getFirstSwitchLabelDefault(AstSwitchBlockStatementGroup* const sbsg);
        
        static AstSwitchLabelDefault* getFirstSwitchLabelDefault(AstSwitchBlockStatementGroupList* const sbsgl);

        static AstSwitchLabelDefault* getFirstSwitchLabelDefault(AstSwitchStatement* const ss);

    };


    class JavaSymbol
    {
    public:
        JavaSymbol();

        virtual ~JavaSymbol();

        /**
         * Returns the declaration of the given variable.
         */
        static AstItem* findVariableDeclaration(AstName* const name);

        /**
         * Returns the type of the given variable, based on its declaration.
         */
        static AstItem* getVariableType(AstName* const name);
        
    };


    class JavaSymbolTable
    {
    public:
        JavaSymbolTable(AstItem* const item, JavaSymbolTable* const parentTable = NULL);

        virtual ~JavaSymbolTable();

        AstItem* getItem() const;
        
        vector<JavaSymbolTable*> getSubTables() const;

        JavaSymbolTable* getParentTable() const;

        JavaSymbolTable* getChildTable(AstItem* const item) const;

        void printAll(ostream& os, int depth = 0) const;

        void addDeclaration(AstItem* const item);

        void addVariableDeclaration(AstVariableDeclaratorId* const id);

        void addMethodDeclaration(AstItem* const item);

        void addTypeDeclaration(AstItem* const item);

        void addReference(AstItem* const item);

        void addReference(AstClassMemberDeclaration* const cmd);

        vector<AstItem*> findUnreferenced() const;

        AstItem* lookup(AstName* const name);
        
    private:
        AstItem* item_;

        map<AstItem*, int> variables_;
        
        map<AstItem*, int> types_;
        
        map<AstItem*, int> methods_;
        
        vector<JavaSymbolTable*> subTables_;

        JavaSymbolTable* parentTable_;

    };


    class JavaSymbolReader : public Processor
    {
    public:
        JavaSymbolReader();

        virtual ~JavaSymbolReader();

        virtual JavaSymbolTable* processCompilationUnit(AstCompilationUnit* const ac);

        virtual void process(AstAbstractMethodDeclaration* const aa);
             
        virtual void process(AstBlock* const ab);

        virtual void process(AstBlockNoStatements* const ab);

        virtual void process(AstBlockWStatements* const ab);

        virtual void process(AstCatchClause* const ac);

        virtual void process(AstClassDeclaration* const ac);

        virtual void process(AstClassDeclarationModsBaseless* const ac);

        virtual void process(AstClassDeclarationModsExtends* const ac);

        virtual void process(AstClassDeclarationModsExtendsImplements* const ac);

        virtual void process(AstClassDeclarationModsImplements* const ac);

        virtual void process(AstClassDeclarationNoModsBaseless* const ac);

        virtual void process(AstClassDeclarationNoModsExtends* const ac);

        virtual void process(AstClassDeclarationNoModsExtendsImplements* const ac);

        virtual void process(AstClassDeclarationNoModsImplements* const ac);

        virtual void process(AstClassMemberDeclaration* const ac);

        virtual void process(AstClassMemberDeclarationModsExtendsImplements* const ac);

        virtual void process(AstClassMemberDeclarationModsExtendsNoImplements* const ac);

        virtual void process(AstClassMemberDeclarationModsNoExtendsImplements* const ac);

        virtual void process(AstClassMemberDeclarationModsNoExtendsNoImplements* const ac);

        virtual void process(AstClassMemberDeclarationNoModsExtendsImplements* const ac);

        virtual void process(AstClassMemberDeclarationNoModsExtendsNoImplements* const ac);

        virtual void process(AstClassMemberDeclarationNoModsNoExtendsImplements* const ac);

        virtual void process(AstClassMemberDeclarationNoModsNoExtendsNoImplements* const ac);

        virtual void process(AstConstructorDeclaration* const ac);

        virtual void process(AstConstructorDeclarationModsNoThrows* const ac);

        virtual void process(AstConstructorDeclarationModsThrows* const ac);

        virtual void process(AstConstructorDeclarationNoModsNoThrows* const ac);

        virtual void process(AstConstructorDeclarationNoModsThrows* const ac);

        virtual void process(AstFor* const af);

        virtual void process(AstForStatement* const af);

        virtual void process(AstForStatementNoInitConditionNoUpdate* const af);

        virtual void process(AstForStatementNoInitConditionUpdate* const af);

        virtual void process(AstForStatementNoInitNoConditionNoUpdate* const af);

        virtual void process(AstForStatementNoInitNoConditionUpdate* const af);

        virtual void process(AstForStatementStmtsConditionNoUpdate* const af);

        virtual void process(AstForStatementStmtsConditionUpdate* const af);

        virtual void process(AstForStatementStmtsNoConditionNoUpdate* const af);

        virtual void process(AstForStatementStmtsNoConditionUpdate* const af);

        virtual void process(AstForStatementVarsConditionNoUpdate* const af);

        virtual void process(AstForStatementVarsConditionUpdate* const af);

        virtual void process(AstForStatementVarsNoConditionNoUpdate* const af);

        virtual void process(AstForStatementVarsNoConditionUpdate* const af);

        virtual void process(AstFormalParameter* const af);

        virtual void process(AstFormalParameterFinal* const af);

        virtual void process(AstFormalParameterNonFinal* const af);

        virtual void process(AstInterfaceDeclaration* const ai);

        virtual void process(AstInterfaceDeclarationModsExtends* const ai);

        virtual void process(AstInterfaceDeclarationModsNoExtends* const ai);

        virtual void process(AstInterfaceDeclarationNoModsExtends* const ai);

        virtual void process(AstInterfaceDeclarationNoModsNoExtends* const ai);

        virtual void process(AstLocalVariableDeclaration* const al);

        virtual void process(AstLocalVariableDeclarationFinal* const al);

        virtual void process(AstLocalVariableDeclarationNonFinal* const al);

        virtual void process(AstMethodDeclaration* const am);

        virtual void process(AstMethodDeclarationBlock* const am);

        virtual void process(AstMethodDeclarationSemicolon* const am);

        virtual void process(AstTryStatementCatches* const at);

        virtual void process(AstTryStatementCatchesFinally* const at);

        virtual void process(AstTryStatementFinally* const at);

    private:
        JavaSymbolTable* symbolTable_;

    };

    
    /**
     * Wraps a LocalVariableDeclaration.
     */
    class JavaVariable
    {
    public:
        JavaVariable(AstLocalVariableDeclaration* const lvd);
        
        virtual ~JavaVariable();

        /**
         * Returns the type, as a string.
         */
        string getType() const;

        /**
         * Returns whether this/there variable(s) is/are declared as final.
         */
        bool isFinal() const;

        /**
         * Returns the names of the variables.
         */
        vector<string> getNames() const;

    private:
        AstLocalVariableDeclaration* lvd_;

    };
}

#endif //! Java_h
