#ifndef AST_h
#include "AST.h"
#endif

#ifndef Processor_h
#include "Processor.h"
#endif

#ifndef File_h
#include "File.h"
#endif

#ifndef JavadocParser_h
#include "JavadocParser.h"
#endif

#ifndef AstDebug_h
#include "AstDebug.h"
#endif

#ifndef Parser_h
#include "Parser.h"
#endif

#ifndef Platform_h
#include "Platform.h"
#endif

#ifndef StringUtilities_h
#include "StringUtilities.h"
#endif

using namespace doctorj;
using namespace std;

extern Parser parser;
extern AstCompilationUnit* compUnit;


AstAbstract::AstAbstract(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstAbstract::~AstAbstract()
{
}

string AstAbstract::type() const
{
    return "ABSTRACT";
}

void AstAbstract::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstAbstract::length() const
{
    return 8;
}

AstAbstractMethodDeclaration::AstAbstractMethodDeclaration(AstMethodHeader* const mh, 
             AstSemicolon* const s,
             File* const srcfile) :
        AstNode(mh, s, srcfile), methodHeader_(mh), semicolon_(s)
{
}

AstAbstractMethodDeclaration::~AstAbstractMethodDeclaration()
{
}

AstMethodHeader* AstAbstractMethodDeclaration::getMethodHeader() const
{
    return methodHeader_;
}

AstSemicolon* AstAbstractMethodDeclaration::getSemicolon() const
{
    return semicolon_;
}

void AstAbstractMethodDeclaration::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstAbstractMethodDeclaration::type() const
{
    return "AbstractMethodDeclaration";
}

AstAdditiveExpression::AstAdditiveExpression(AstItem* const lhex,
                                             AstItem* const rhex,
                                             File* const srcfile) :
        AstNode(lhex, rhex, srcfile), lhsExpression_(lhex), rhsExpression_(rhex)
{
}

AstAdditiveExpression::~AstAdditiveExpression()
{
}

AstItem* AstAdditiveExpression::getLhsExpression() const
{
    return lhsExpression_;
}

AstItem* AstAdditiveExpression::getRhsExpression() const
{
    return rhsExpression_;
}

AstAdditiveExpressionMinus::AstAdditiveExpressionMinus(AstItem* const lhex,
                                                       AstMinus* const mi,
                                                       AstItem* const rhex,
                                                       File* const srcfile) :
        AstAdditiveExpression(lhex, rhex, srcfile), minus_(mi)
{
    append(mi, lhex);
}

AstAdditiveExpressionMinus::~AstAdditiveExpressionMinus()
{
}

AstMinus* AstAdditiveExpressionMinus::getMinus() const
{
    return minus_;
}

void AstAdditiveExpressionMinus::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstAdditiveExpressionMinus::type() const
{
    return "AdditiveExpressionMinus";
}

AstAdditiveExpressionPlus::AstAdditiveExpressionPlus(AstItem* const lhex,
                                                     AstPlus* const pl,
                                                     AstItem* const rhex,
                                                     File* const srcfile) :
        AstAdditiveExpression(lhex, rhex, srcfile), plus_(pl)
{
    append(pl, lhex);
}

AstAdditiveExpressionPlus::~AstAdditiveExpressionPlus()
{
}

AstPlus* AstAdditiveExpressionPlus::getPlus() const
{
    return plus_;
}

void AstAdditiveExpressionPlus::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstAdditiveExpressionPlus::type() const
{
    return "AdditiveExpressionPlus";
}

AstAndand::AstAndand(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstAndand::~AstAndand()
{
}

string AstAndand::type() const
{
    return "ANDAND";
}

void AstAndand::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstAndand::length() const
{
    return 2;
}

AstAnd::AstAnd(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstAnd::~AstAnd()
{
}

string AstAnd::type() const
{
    return "AND";
}

void AstAnd::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstAnd::length() const
{
    return 1;
}

AstAndeq::AstAndeq(char* position, File* const srcfile) :
        AstAssignmentOperator(position, srcfile)
{
}

AstAndeq::~AstAndeq()
{
}

string AstAndeq::type() const
{
    return "ANDEQ";
}

void AstAndeq::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstAndeq::length() const
{
    return 2;
}

AstAndExpression::AstAndExpression(AstItem* const le, 
                                   AstAnd* const a,
                                   AstItem* const re,
                                   File* const srcfile /* = NULL */) :
        AstNode(le, a, re, srcfile), lhsExpression_(le), and_(a), rhsExpression_(re)
{
}

AstAndExpression::~AstAndExpression()
{
}

AstItem* AstAndExpression::getLhsExpression() const
{
    return lhsExpression_;
}

AstAnd* AstAndExpression::getAnd() const
{
    return and_;
}

AstItem* AstAndExpression::getRhsExpression() const
{
    return rhsExpression_;
}

void AstAndExpression::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstAndExpression::type() const
{
    return "AndExpression";
}

AstArrayAccess::AstArrayAccess(AstDimensionExpression* const diex,
                               File* const srcfile) :
        AstNode(diex, srcfile), dimensionExpression_(diex)
{
}

AstArrayAccess::~AstArrayAccess()
{
}

AstDimensionExpression* AstArrayAccess::getDimensionExpression() const
{
    return dimensionExpression_;
}

AstArrayAccessName::AstArrayAccessName(AstName* const na,
                                 AstDimensionExpression* const diex,
                                 File* const srcfile) :
        AstArrayAccess(diex, srcfile), name_(na)
{
    prepend(na);
}

AstArrayAccessName::~AstArrayAccessName()
{
}

AstName* AstArrayAccessName::getName() const
{
    return name_;
}

void AstArrayAccessName::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstArrayAccessName::type() const
{
    return "ArrayAccessName";
}

AstArrayAccessPrimary::AstArrayAccessPrimary(AstItem* const pr,
                                       AstDimensionExpression* const diex,
                                       File* const srcfile) :
        AstArrayAccess(diex, srcfile), primary_(pr)
{
    prepend(pr);
}

AstArrayAccessPrimary::~AstArrayAccessPrimary()
{
}

AstItem* AstArrayAccessPrimary::getPrimary() const
{
    return primary_;
}

void AstArrayAccessPrimary::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstArrayAccessPrimary::type() const
{
    return "ArrayAccessPrimary";
}

AstArrayCreationExpression::AstArrayCreationExpression(AstNew* const ne,
                                                       File* const srcfile) :
        AstNode(ne, srcfile), new_(ne)
{
}

AstArrayCreationExpression::~AstArrayCreationExpression()
{
}

AstNew* AstArrayCreationExpression::getNew() const
{
    return new_;
}

AstArrayCreationExpressionNameDimsInit::AstArrayCreationExpressionNameDimsInit(AstNew* const ne,
                                                                         AstName* const na,
                                                                         AstDimensionList* const dili,
                                                                         AstArrayInitializer* const arin,
                                                                         File* const srcfile) :
        AstArrayCreationExpression(ne, srcfile), name_(na), dimensionList_(dili), arrayInitializer_(arin)
{
    append(na, ne);
    append(dili, na);
    append(arin, dili);
}

AstArrayCreationExpressionNameDimsInit::~AstArrayCreationExpressionNameDimsInit()
{
}

AstName* AstArrayCreationExpressionNameDimsInit::getName() const
{
    return name_;
}

AstDimensionList* AstArrayCreationExpressionNameDimsInit::getDimensionList() const
{
    return dimensionList_;
}

AstArrayInitializer* AstArrayCreationExpressionNameDimsInit::getArrayInitializer() const
{
    return arrayInitializer_;
}

void AstArrayCreationExpressionNameDimsInit::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstArrayCreationExpressionNameDimsInit::type() const
{
    return "ArrayCreationExpressionNameDimsInit";
}

AstArrayCreationExpressionNameExprDims::AstArrayCreationExpressionNameExprDims(AstNew* const ne,
                                                                         AstName* const na,
                                                                         AstDimensionExpressionList* const diexli,
                                                                         AstDimensionList* const dili,
                                                                         File* const srcfile) :
        AstArrayCreationExpression(ne, srcfile), name_(na), dimensionExpressionList_(diexli), dimensionList_(dili)
{
    append(na, ne);
    append(diexli, na);
    append(dili, diexli);
}

AstArrayCreationExpressionNameExprDims::~AstArrayCreationExpressionNameExprDims()
{
}

AstName* AstArrayCreationExpressionNameExprDims::getName() const
{
    return name_;
}

AstDimensionExpressionList* AstArrayCreationExpressionNameExprDims::getDimensionExpressionList() const
{
    return dimensionExpressionList_;
}

AstDimensionList* AstArrayCreationExpressionNameExprDims::getDimensionList() const
{
    return dimensionList_;
}

void AstArrayCreationExpressionNameExprDims::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstArrayCreationExpressionNameExprDims::type() const
{
    return "ArrayCreationExpressionNameExprDims";
}

AstArrayCreationExpressionNameExprNoDims::AstArrayCreationExpressionNameExprNoDims(AstNew* const ne,
                                                                             AstName* const na,
                                                                             AstDimensionExpressionList* const diexli,
                                                                             File* const srcfile) :
        AstArrayCreationExpression(ne, srcfile), name_(na), dimensionExpressionList_(diexli)
{
    append(na, ne);
    append(diexli, na);
}

AstArrayCreationExpressionNameExprNoDims::~AstArrayCreationExpressionNameExprNoDims()
{
}

AstName* AstArrayCreationExpressionNameExprNoDims::getName() const
{
    return name_;
}

AstDimensionExpressionList* AstArrayCreationExpressionNameExprNoDims::getDimensionExpressionList() const
{
    return dimensionExpressionList_;
}

void AstArrayCreationExpressionNameExprNoDims::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstArrayCreationExpressionNameExprNoDims::type() const
{
    return "ArrayCreationExpressionNameExprNoDims";
}

AstArrayCreationExpressionPrimitiveDimsInit::AstArrayCreationExpressionPrimitiveDimsInit(AstNew* const ne,
                                                                                   AstPrimitiveType* const prty,
                                                                                   AstDimensionList* const dili,
                                                                                   AstArrayInitializer* const arin,
                                                                                   File* const srcfile) :
        AstArrayCreationExpression(ne, srcfile), primitiveType_(prty), dimensionList_(dili), arrayInitializer_(arin)
{
    append(prty, ne);
    append(dili, prty);
    append(arin, dili);
}

AstArrayCreationExpressionPrimitiveDimsInit::~AstArrayCreationExpressionPrimitiveDimsInit()
{
}

AstPrimitiveType* AstArrayCreationExpressionPrimitiveDimsInit::getPrimitiveType() const
{
    return primitiveType_;
}

AstDimensionList* AstArrayCreationExpressionPrimitiveDimsInit::getDimensionList() const
{
    return dimensionList_;
}

AstArrayInitializer* AstArrayCreationExpressionPrimitiveDimsInit::getArrayInitializer() const
{
    return arrayInitializer_;
}

void AstArrayCreationExpressionPrimitiveDimsInit::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstArrayCreationExpressionPrimitiveDimsInit::type() const
{
    return "ArrayCreationExpressionPrimitiveDimsInit";
}

AstArrayCreationExpressionPrimitiveExprDims::AstArrayCreationExpressionPrimitiveExprDims(AstNew* const ne,
                                                                                   AstPrimitiveType* const prty,
                                                                                   AstDimensionExpressionList* const diexli,
                                                                                   AstDimensionList* const dili,
                                                                                   File* const srcfile) :
        AstArrayCreationExpression(ne, srcfile), primitiveType_(prty), dimensionExpressionList_(diexli), dimensionList_(dili)
{
    append(prty, ne);
    append(diexli, prty);
    append(dili, diexli);
}

AstArrayCreationExpressionPrimitiveExprDims::~AstArrayCreationExpressionPrimitiveExprDims()
{
}

AstPrimitiveType* AstArrayCreationExpressionPrimitiveExprDims::getPrimitiveType() const
{
    return primitiveType_;
}

AstDimensionExpressionList* AstArrayCreationExpressionPrimitiveExprDims::getDimensionExpressionList() const
{
    return dimensionExpressionList_;
}

AstDimensionList* AstArrayCreationExpressionPrimitiveExprDims::getDimensionList() const
{
    return dimensionList_;
}

void AstArrayCreationExpressionPrimitiveExprDims::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstArrayCreationExpressionPrimitiveExprDims::type() const
{
    return "ArrayCreationExpressionPrimitiveExprDims";
}

AstArrayCreationExpressionPrimitiveExprNoDims::AstArrayCreationExpressionPrimitiveExprNoDims(AstNew* const ne,
                                                                                       AstPrimitiveType* const prty,
                                                                                       AstDimensionExpressionList* const diexli,
                                                                                       File* const srcfile) :
        AstArrayCreationExpression(ne, srcfile), primitiveType_(prty), dimensionExpressionList_(diexli)
{
    append(prty, ne);
    append(diexli, prty);
}

AstArrayCreationExpressionPrimitiveExprNoDims::~AstArrayCreationExpressionPrimitiveExprNoDims()
{
}

AstPrimitiveType* AstArrayCreationExpressionPrimitiveExprNoDims::getPrimitiveType() const
{
    return primitiveType_;
}

AstDimensionExpressionList* AstArrayCreationExpressionPrimitiveExprNoDims::getDimensionExpressionList() const
{
    return dimensionExpressionList_;
}

void AstArrayCreationExpressionPrimitiveExprNoDims::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstArrayCreationExpressionPrimitiveExprNoDims::type() const
{
    return "ArrayCreationExpressionPrimitiveExprNoDims";
}

AstArrayInitializerComma::AstArrayInitializerComma(AstLbrace* const l, 
                                                   AstComma* const c,
                                                   AstRbrace* const r,
                                                   File* const srcfile) :
        AstArrayInitializer(l, NULL, c, r, srcfile)
{
}

AstArrayInitializerComma::~AstArrayInitializerComma()
{
}

void AstArrayInitializerComma::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstArrayInitializerComma::type() const
{
    return "ArrayInitializerComma";
}

AstArrayInitializer::AstArrayInitializer(AstLbrace* const l, 
                                         AstVariableInitializerList* const vil,
                                         AstComma* const c,
                                         AstRbrace* const r,
                                         File* const srcfile) :
        AstNode(l, vil, c, r, srcfile),
     lbrace_(l), 
     variableInitializerList_(vil), 
     comma_(c),
     rbrace_(r)
{
}

AstArrayInitializer::~AstArrayInitializer()
{
}

AstLbrace* AstArrayInitializer::getLbrace() const
{
    return lbrace_;
}

AstVariableInitializerList* AstArrayInitializer::getVariableInitializerList() const
{
    return variableInitializerList_;
}

AstComma* AstArrayInitializer::getComma() const
{
    return comma_;
}

AstRbrace* AstArrayInitializer::getRbrace() const
{
    return rbrace_;
}

AstArrayInitializerEmpty::AstArrayInitializerEmpty(AstLbrace* const l, 
                                                   AstRbrace* const r,
                                                   File* const srcfile) :
        AstArrayInitializer(l, NULL, NULL, r, srcfile)
{
}

AstArrayInitializerEmpty::~AstArrayInitializerEmpty()
{
}

void AstArrayInitializerEmpty::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstArrayInitializerEmpty::type() const
{
    return "ArrayInitializerEmpty";
}

AstArrayInitializerVarsComma::AstArrayInitializerVarsComma(AstLbrace* const l, 
                                                           AstVariableInitializerList* const vil,
                                                           AstComma* const c,
                                                           AstRbrace* const r,
                                                           File* const srcfile) :
        AstArrayInitializer(l, vil, c, r, srcfile)
{
}

AstArrayInitializerVarsComma::~AstArrayInitializerVarsComma()
{
}

void AstArrayInitializerVarsComma::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstArrayInitializerVarsComma::type() const
{
    return "ArrayInitializerVarsComma";
}

AstArrayInitializerVars::AstArrayInitializerVars(AstLbrace* const l, 
                                                 AstVariableInitializerList* const vil,
                                                 AstRbrace* const r,
                                                 File* const srcfile) :
        AstArrayInitializer(l, vil, NULL, r, srcfile)
{
}

AstArrayInitializerVars::~AstArrayInitializerVars()
{
}

void AstArrayInitializerVars::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstArrayInitializerVars::type() const
{
    return "ArrayInitializerVars";
}

AstArrayNameType::AstArrayNameType(AstName* const n, 
                             AstDimensionList* const dl,
                             File* const srcfile) :
        AstArrayType(n, dl, srcfile), name_(n)
{
}

AstArrayNameType::~AstArrayNameType()
{
}

AstName* AstArrayNameType::getName() const
{
    return name_;
}

void AstArrayNameType::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstArrayNameType::type() const
{
    return "ArrayNameType";
}

AstArrayPrimitiveType::AstArrayPrimitiveType(AstPrimitiveType* const pt, 
                                       AstDimensionList* const dl,
                                       File* const srcfile) :
        AstArrayType(pt, dl, srcfile), primitiveType_(pt)
{
}

AstArrayPrimitiveType::~AstArrayPrimitiveType()
{
}

AstPrimitiveType* AstArrayPrimitiveType::getPrimitiveType() const
{
    return primitiveType_;
}

void AstArrayPrimitiveType::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstArrayPrimitiveType::type() const
{
    return "ArrayPrimitiveType";
}

AstArrayType::AstArrayType(AstItem* const item0, 
                     AstDimensionList* const dl,
                     File* const srcfile) :
        AstNode(item0, dl, srcfile), type_(item0), dimensionList_(dl)
{
}

AstArrayType::~AstArrayType()
{
}

AstItem* AstArrayType::getType() const
{
    return type_;
}

AstDimensionList* AstArrayType::getDimensionList() const
{
    return dimensionList_;
}

void AstArrayType::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstArrayType::type() const
{
    return "ArrayType";
}

AstArrayTypeDotClass::AstArrayTypeDotClass(AstArrayType* const at, 
                                     AstDot* const d,
                                     AstClass* const c,
                                     File* const srcfile) :
        AstNode(at, d, c, srcfile), arrayType_(at), dot_(d), class_(c)
{
}

AstArrayTypeDotClass::~AstArrayTypeDotClass()
{
}

AstArrayType* AstArrayTypeDotClass::getArrayType() const
{
    return arrayType_;
}

AstDot* AstArrayTypeDotClass::getDot() const
{
    return dot_;
}

AstClass* AstArrayTypeDotClass::getClass() const
{
    return class_;
}

void AstArrayTypeDotClass::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstArrayTypeDotClass::type() const
{
    return "ArrayTypeDotClass";
}

AstAssert::AstAssert(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstAssert::~AstAssert()
{
}

string AstAssert::type() const
{
    return "ASSERT";
}

void AstAssert::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstAssert::length() const
{
    return 6;
}

AstAssertStatement::AstAssertStatement(AstAssert* const as,
                                       AstItem* const exon,
                                       AstColon* const co,
                                       AstItem* const extw,
                                       AstSemicolon* const se,
                                       File* const srcfile) :
        AstNode(as, exon, co, extw, se, srcfile), 
     assert_(as), 
     expressionOne_(exon), 
     colon_(co), 
     expressionTwo_(extw), 
     semicolon_(se)
{
}

AstAssertStatement::~AstAssertStatement()
{
}

AstAssert* AstAssertStatement::getAssert() const
{
    return assert_;
}

AstItem* AstAssertStatement::getExpressionOne() const
{
    return expressionOne_;
}

AstColon* AstAssertStatement::getColon() const
{
    return colon_;
}

AstItem* AstAssertStatement::getExpressionTwo() const
{
    return expressionTwo_;
}

AstSemicolon* AstAssertStatement::getSemicolon() const
{
    return semicolon_;
}

AstAssertStatementOneExpression::AstAssertStatementOneExpression(AstAssert* const as,
                                                                 AstItem* const ex,
                                                                 AstSemicolon* const se,
                                                                 File* const srcfile) :
        AstAssertStatement(as, ex, NULL, NULL, se, srcfile)
{
}

AstAssertStatementOneExpression::~AstAssertStatementOneExpression()
{
}

void AstAssertStatementOneExpression::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstAssertStatementOneExpression::type() const
{
    return "AssertStatementOneExpression";
}

AstAssertStatementTwoExpressions::AstAssertStatementTwoExpressions(AstAssert* const as,
                                                                   AstItem* const exon,
                                                                   AstColon* const co,
                                                                   AstItem* const extw,
                                                                   AstSemicolon* const se,
                                                                   File* const srcfile) :
        AstAssertStatement(as, exon, co, extw, se, srcfile)
{
}

AstAssertStatementTwoExpressions::~AstAssertStatementTwoExpressions()
{
}

void AstAssertStatementTwoExpressions::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstAssertStatementTwoExpressions::type() const
{
    return "AssertStatementTwoExpressions";
}

AstAssignmentArray::AstAssignmentArray(AstArrayAccess* const arac,
                                 AstAssignmentOperator* const asop,
                                 AstItem* const ex,
                                 File* const srcfile) :
        AstAssignment(asop, ex, srcfile), arrayAccess_(arac)
{
    prepend(arac);
}

AstAssignmentArray::~AstAssignmentArray()
{
}

AstArrayAccess* AstAssignmentArray::getArrayAccess() const
{
    return arrayAccess_;
}

void AstAssignmentArray::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstAssignmentArray::type() const
{
    return "AssignmentArray";
}

AstAssignment::AstAssignment(AstAssignmentOperator* const asop,
                             AstItem* const ex,
                             File* const srcfile) :
        AstNode(asop, ex, srcfile), assignmentOperator_(asop), expression_(ex)
{
}

AstAssignment::~AstAssignment()
{
}

AstAssignmentOperator* AstAssignment::getAssignmentOperator() const
{
    return assignmentOperator_;
}

AstItem* AstAssignment::getExpression() const
{
    return expression_;
}

AstAssignmentField::AstAssignmentField(AstFieldAccess* const fiac,
                                 AstAssignmentOperator* const asop,
                                 AstItem* const ex,
                                 File* const srcfile) :
        AstAssignment(asop, ex, srcfile), fieldAccess_(fiac)
{
    prepend(fiac);
}

AstAssignmentField::~AstAssignmentField()
{
}

AstFieldAccess* AstAssignmentField::getFieldAccess() const
{
    return fieldAccess_;
}

void AstAssignmentField::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstAssignmentField::type() const
{
    return "AssignmentField";
}

AstAssignmentName::AstAssignmentName(AstName* const na,
                                     AstAssignmentOperator* const asop,
                                     AstItem* const ex,
                                     File* const srcfile) :
        AstAssignment(asop, ex, srcfile), name_(na)
{
    prepend(na);
}

AstAssignmentName::~AstAssignmentName()
{
}

AstName* AstAssignmentName::getName() const
{
    return name_;
}

void AstAssignmentName::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstAssignmentName::type() const
{
    return "AssignmentName";
}

AstAssignmentOperator::AstAssignmentOperator(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstAssignmentOperator::~AstAssignmentOperator()
{
}

AstBlock::AstBlock(AstLbrace* const l, 
                   AstBlockStatementList* const bsl,
                   AstRbrace* const r,
                   File* const srcfile) :
        AstNode(l, bsl, r, srcfile), lbrace_(l), blockStatementList_(bsl), rbrace_(r)
{
}

AstBlock::~AstBlock()
{
}

AstLbrace* AstBlock::getLbrace() const
{
    return lbrace_;
}

AstBlockStatementList* AstBlock::getBlockStatementList() const
{
    return blockStatementList_;
}

AstRbrace* AstBlock::getRbrace() const
{
    return rbrace_;
}

AstBlockNoStatements::AstBlockNoStatements(AstLbrace* const l, 
                                           AstRbrace* const r,
                                           File* const srcfile) :
        AstBlock(l, NULL, r, srcfile)
{
}

AstBlockNoStatements::~AstBlockNoStatements()
{
}

void AstBlockNoStatements::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstBlockNoStatements::type() const
{
    return "BlockNoStatements";
}

AstBlockStatementList::AstBlockStatementList(AstItem* const bs, 
             File* const srcfile) :
        AstNode(bs, srcfile)
{
    blockStatements_.push_back(bs);
}

AstBlockStatementList::~AstBlockStatementList()
{
}

AstItem* AstBlockStatementList::getBlockStatement(int index) const
{
    return blockStatements_[index];
}

int AstBlockStatementList::getBlockStatementCount() const
{
    return blockStatements_.size();
}

void AstBlockStatementList::add(AstItem* const bs)
{
    blockStatements_.push_back(bs);
    AstNode::append(bs);
}

void AstBlockStatementList::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstBlockStatementList::type() const
{
    return "BlockStatementList";
}

AstBlockWStatements::AstBlockWStatements(AstLbrace* const l, 
                                         AstBlockStatementList* const bsl,
                                         AstRbrace* const r,
                                         File* const srcfile) : AstBlock(l, bsl, r, srcfile)
{
}

AstBlockWStatements::~AstBlockWStatements()
{
}

void AstBlockWStatements::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstBlockWStatements::type() const
{
    return "BlockWStatements";
}

AstBoolean::AstBoolean(char* position, File* const srcfile) :
        AstPrimitiveType(position, srcfile)
{
}

AstBoolean::~AstBoolean()
{
}

string AstBoolean::type() const
{
    return "BOOLEAN";
}

void AstBoolean::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstBoolean::length() const
{
    return 7;
}

AstBooleanLiteral::AstBooleanLiteral(const string& text, char* position, File* const srcfile) :
        AstLiteral(text, position, srcfile)
{
}

AstBooleanLiteral::~AstBooleanLiteral()
{
}

string AstBooleanLiteral::type() const
{
    return "BOOLEAN_LITERAL";
}

void AstBooleanLiteral::getProcessed(Processor* const proc)
{
    proc->process(this);
}

AstBreak::AstBreak(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstBreak::~AstBreak()
{
}

string AstBreak::type() const
{
    return "BREAK";
}

void AstBreak::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstBreak::length() const
{
    return 5;
}

AstBreakStatement::AstBreakStatement(AstBreak* const b, 
                                     AstIdentifier* const i,
                                     AstSemicolon* const s,
                                     File* const srcfile) :
        AstNode(b, i, s, srcfile), break_(b), identifier_(i), semicolon_(s)
{
}

AstBreakStatement::~AstBreakStatement()
{
}

AstBreak* AstBreakStatement::getBreak() const
{
    return break_;
}

AstIdentifier* AstBreakStatement::getIdentifier() const
{
    return identifier_;
}

AstSemicolon* AstBreakStatement::getSemicolon() const
{
    return semicolon_;
}

void AstBreakStatement::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstBreakStatement::type() const
{
    return "BreakStatement";
}

AstBreakStatementId::AstBreakStatementId(AstBreak* const b, 
                                         AstIdentifier* const i,
                                         AstSemicolon* const s,
                                         File* const srcfile) : 
        AstBreakStatement(b, i, s, srcfile)
{
}

AstBreakStatementId::~AstBreakStatementId()
{
}

void AstBreakStatementId::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstBreakStatementId::type() const
{
    return "BreakStatementId";
}

AstBreakStatementNoId::AstBreakStatementNoId(AstBreak* const b, 
                                             AstSemicolon* const s,
                                             File* const srcfile) :
        AstBreakStatement(b, NULL, s, srcfile)
{
}

AstBreakStatementNoId::~AstBreakStatementNoId()
{
}

void AstBreakStatementNoId::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstBreakStatementNoId::type() const
{
    return "BreakStatementNoId";
}

AstByte::AstByte(char* position, File* const srcfile) :
        AstIntegralType(position, srcfile)
{
}

AstByte::~AstByte()
{
}

string AstByte::type() const
{
    return "BYTE";
}

void AstByte::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstByte::length() const
{
    return 4;
}

AstCase::AstCase(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstCase::~AstCase()
{
}

string AstCase::type() const
{
    return "CASE";
}

void AstCase::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstCase::length() const
{
    return 4;
}

AstCastExpression::AstCastExpression(AstLparen* const lp,
                                     AstRparen* const rp,
                                     AstItem* const unex,
                                     File* const srcfile) :
        AstNode(lp, rp, unex, srcfile), lparen_(lp), rparen_(rp), unaryExpression_(unex)
{
}

AstCastExpression::~AstCastExpression()
{
}

AstLparen* AstCastExpression::getLparen() const
{
    return lparen_;
}

AstRparen* AstCastExpression::getRparen() const
{
    return rparen_;
}

AstItem* AstCastExpression::getUnaryExpression() const
{
    return unaryExpression_;
}

AstCastExpressionExpr::AstCastExpressionExpr(AstLparen* const lp,
                                       AstItem* const ex,
                                       AstRparen* const rp,
                                       AstItem* const unex,
                                       File* const srcfile) :
        AstCastExpression(lp, rp, unex, srcfile), expression_(ex)
{
    append(ex, lp);
}

AstCastExpressionExpr::~AstCastExpressionExpr()
{
}

AstItem* AstCastExpressionExpr::getExpression() const
{
    return expression_;
}

void AstCastExpressionExpr::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstCastExpressionExpr::type() const
{
    return "CastExpressionExpr";
}

AstCastExpressionNameDims::AstCastExpressionNameDims(AstLparen* const lp,
                                               AstName* const na,
                                               AstDimensionList* const dili,
                                               AstRparen* const rp,
                                               AstItem* const unex,
                                               File* const srcfile) :
        AstCastExpression(lp, rp, unex, srcfile), name_(na), dimensionList_(dili)
{
    append(na, lp);
    append(dili, na);
}

AstCastExpressionNameDims::~AstCastExpressionNameDims()
{
}

AstName* AstCastExpressionNameDims::getName() const
{
    return name_;
}

AstDimensionList* AstCastExpressionNameDims::getDimensionList() const
{
    return dimensionList_;
}

void AstCastExpressionNameDims::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstCastExpressionNameDims::type() const
{
    return "CastExpressionNameDims";
}

AstCastExpressionPrimitiveDims::AstCastExpressionPrimitiveDims(AstLparen* const lp,
                                                         AstPrimitiveType* const prty,
                                                         AstDimensionList* const dili,
                                                         AstRparen* const rp,
                                                         AstItem* const unex,
                                                         File* const srcfile) :
        AstCastExpression(lp, rp, unex, srcfile), primitiveType_(prty), dimensionList_(dili)
{
    append(prty, lp);
    append(dili, prty);
}

AstCastExpressionPrimitiveDims::~AstCastExpressionPrimitiveDims()
{
}

AstPrimitiveType* AstCastExpressionPrimitiveDims::getPrimitiveType() const
{
    return primitiveType_;
}

AstDimensionList* AstCastExpressionPrimitiveDims::getDimensionList() const
{
    return dimensionList_;
}

void AstCastExpressionPrimitiveDims::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstCastExpressionPrimitiveDims::type() const
{
    return "CastExpressionPrimitiveDims";
}

AstCastExpressionPrimitiveNoDims::AstCastExpressionPrimitiveNoDims(AstLparen* const lp,
                                                             AstPrimitiveType* const prty,
                                                             AstRparen* const rp,
                                                             AstItem* const unex,
                                                             File* const srcfile) :
        AstCastExpression(lp, rp, unex, srcfile), primitiveType_(prty)
{
    append(prty, lp);
}

AstCastExpressionPrimitiveNoDims::~AstCastExpressionPrimitiveNoDims()
{
}

AstPrimitiveType* AstCastExpressionPrimitiveNoDims::getPrimitiveType() const
{
    return primitiveType_;
}

void AstCastExpressionPrimitiveNoDims::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstCastExpressionPrimitiveNoDims::type() const
{
    return "CastExpressionPrimitiveNoDims";
}

AstCatchClause::AstCatchClause(AstCatch* const c, 
                         AstLparen* const l,
                         AstFormalParameter* const fp,
                         AstRparen* const r,
                         AstBlock* const b,
                         File* const srcfile) :
        AstNode(c, l, fp, r, b, srcfile),
        catch_(c), 
        lparen_(l), 
        formalParameter_(fp), 
        rparen_(r), 
        block_(b)
{
}

AstCatchClause::~AstCatchClause()
{
}

AstCatch* AstCatchClause::getCatch() const
{
    return catch_;
}

AstLparen* AstCatchClause::getLparen() const
{
    return lparen_;
}

AstFormalParameter* AstCatchClause::getFormalParameter() const
{
    return formalParameter_;
}

AstRparen* AstCatchClause::getRparen() const
{
    return rparen_;
}

AstBlock* AstCatchClause::getBlock() const
{
    return block_;
}

void AstCatchClause::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstCatchClause::type() const
{
    return "CatchClause";
}

AstCatchClauseList::AstCatchClauseList(AstCatchClause* const cc, 
             File* const srcfile) :
        AstNode(cc, srcfile)
{
    catchClauses_.push_back(cc);
}

AstCatchClauseList::~AstCatchClauseList()
{
}

AstCatchClause* AstCatchClauseList::getCatchClause(int index) const
{
    return catchClauses_[index];
}

int AstCatchClauseList::getCatchClauseCount() const
{
    return catchClauses_.size();
}

void AstCatchClauseList::add(AstCatchClause* const cc)
{
    catchClauses_.push_back(cc);
    AstNode::append(cc);
}

void AstCatchClauseList::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstCatchClauseList::type() const
{
    return "CatchClauseList";
}

AstCatch::AstCatch(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstCatch::~AstCatch()
{
}

string AstCatch::type() const
{
    return "CATCH";
}

void AstCatch::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstCatch::length() const
{
    return 5;
}

AstCComment::AstCComment(char* start, char* end, File* const srcfile) :
        AstComment(start, end, srcfile)
{
}

AstCComment::~AstCComment()
{
}

string AstCComment::type() const 
{
    return "C comment"; 
}

void AstCComment::getProcessed(Processor* const p)
{
    p->process(this);
}

AstCharacterLiteral::AstCharacterLiteral(const string& text, char* position, File* const srcfile) :
        AstLiteral(text, position, srcfile)
{
}

AstCharacterLiteral::~AstCharacterLiteral()
{
}

string AstCharacterLiteral::type() const
{
    return "CHARACTER_LITERAL";
}

void AstCharacterLiteral::getProcessed(Processor* const proc)
{
    proc->process(this);
}

AstChar::AstChar(char* position, File* const srcfile) :
        AstIntegralType(position, srcfile)
{
}

AstChar::~AstChar()
{
}

string AstChar::type() const
{
    return "CHAR";
}

void AstChar::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstChar::length() const
{
    return 4;
}

AstClassBody::AstClassBody(AstLbrace* const lb,
                           AstClassBodyDeclarationList* const clbodeli,
                           AstRbrace* const rb,
                           File* const srcfile) :
        AstNode(lb, clbodeli, rb, srcfile),
     lbrace_(lb), 
     classBodyDeclarationList_(clbodeli), 
     rbrace_(rb)
{
}

AstClassBody::~AstClassBody()
{
}

AstLbrace* AstClassBody::getLbrace() const
{
    return lbrace_;
}

AstClassBodyDeclarationList* AstClassBody::getClassBodyDeclarationList() const
{
    return classBodyDeclarationList_;
}

AstRbrace* AstClassBody::getRbrace() const
{
    return rbrace_;
}

AstClassBodyDeclarationList::AstClassBodyDeclarationList(AstItem* const cbd, 
                                                   File* const srcfile) :
        AstNode(cbd, srcfile)
{
    classBodyDeclarations_.push_back(cbd);
}

AstClassBodyDeclarationList::~AstClassBodyDeclarationList()
{
}

AstItem* AstClassBodyDeclarationList::getClassBodyDeclaration(int index) const
{
    return classBodyDeclarations_[index];
}

int AstClassBodyDeclarationList::getClassBodyDeclarationCount() const
{
    return classBodyDeclarations_.size();
}

void AstClassBodyDeclarationList::add(AstItem* const cbd)
{
    classBodyDeclarations_.push_back(cbd);
    AstNode::append(cbd);
}

void AstClassBodyDeclarationList::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassBodyDeclarationList::type() const
{
    return "ClassBodyDeclarationList";
}

AstClassBodyDecls::AstClassBodyDecls(AstLbrace* const lb,
                               AstClassBodyDeclarationList* const clbodeli,
                               AstRbrace* const rb,
                               File* const srcfile) :
        AstClassBody(lb, clbodeli, rb, srcfile)
{
}

AstClassBodyDecls::~AstClassBodyDecls()
{
}

void AstClassBodyDecls::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassBodyDecls::type() const
{
    return "ClassBodyDecls";
}

AstClassBodyEmpty::AstClassBodyEmpty(AstLbrace* const lb,
                               AstRbrace* const rb,
                               File* const srcfile) :
        AstClassBody(lb, NULL, rb, srcfile)
{
}

AstClassBodyEmpty::~AstClassBodyEmpty()
{
}

void AstClassBodyEmpty::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassBodyEmpty::type() const
{
    return "ClassBodyEmpty";
}

AstClass::AstClass(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstClass::~AstClass()
{
}

string AstClass::type() const
{
    return "CLASS";
}

void AstClass::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstClass::length() const
{
    return 5;
}

AstClassDeclaration::AstClassDeclaration(AstClass* const cl,
                                         AstIdentifier* const id,
                                         AstClassBody* const clbo,
                                         File* const srcfile) :
        AstNode(cl, id, clbo, srcfile), 
     modifierList_(NULL),
     class_(cl), 
     identifier_(id), 
     extends_(NULL),
     name_(NULL),
     implementsInterfaceList_(NULL),
     classBody_(clbo)
{
}

AstClassDeclaration::AstClassDeclaration(AstModifierList* const moli,
                                         AstClass* const cl,
                                         AstIdentifier* const id,
                                         AstExtends* const ex,
                                         AstName* const na,
                                         AstImplementsInterfaceList* const iminli,
                                         AstClassBody* const clbo,
                                         File* const srcfile) :
        AstNode(cl, id, clbo, srcfile), 
     modifierList_(moli),
     class_(cl), 
     identifier_(id), 
     extends_(ex),
     name_(na),
     implementsInterfaceList_(iminli),
     classBody_(clbo)
{
}

AstClassDeclaration::~AstClassDeclaration()
{
}

AstModifierList* AstClassDeclaration::getModifierList() const
{
    return modifierList_;
}

AstClass* AstClassDeclaration::getClass() const
{
    return class_;
}

AstIdentifier* AstClassDeclaration::getIdentifier() const
{
    return identifier_;
}

AstExtends* AstClassDeclaration::getExtends() const
{
    return extends_;
}

AstName* AstClassDeclaration::getName() const
{
    return name_;
}

AstImplementsInterfaceList* AstClassDeclaration::getImplementsInterfaceList() const
{
    return implementsInterfaceList_;
}

AstClassBody* AstClassDeclaration::getClassBody() const
{
    return classBody_;
}

AstClassDeclarationModsBaseless::AstClassDeclarationModsBaseless(AstModifierList* const moli,
                                                                 AstClass* const cl,
                                                                 AstIdentifier* const id,
                                                                 AstClassBody* const clbo,
                                                                 File* const srcfile) :
        AstClassDeclaration(moli, cl, id, NULL, NULL, NULL, clbo, srcfile)
{
    prepend(moli);
}

AstClassDeclarationModsBaseless::~AstClassDeclarationModsBaseless()
{
}

void AstClassDeclarationModsBaseless::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassDeclarationModsBaseless::type() const
{
    return "ClassDeclarationModsBaseless";
}

AstClassDeclarationModsExtends::AstClassDeclarationModsExtends(AstModifierList* const moli,
                                                               AstClass* const cl,
                                                               AstIdentifier* const id,
                                                               AstExtends* const ex,
                                                               AstName* const na,
                                                               AstClassBody* const clbo,
                                                               File* const srcfile) :
        AstClassDeclaration(moli, cl, id, ex, na, NULL, clbo, srcfile)
{
    prepend(moli);
    append(ex, id);
    append(na, ex);
}

AstClassDeclarationModsExtends::~AstClassDeclarationModsExtends()
{
}

void AstClassDeclarationModsExtends::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassDeclarationModsExtends::type() const
{
    return "ClassDeclarationModsExtends";
}

AstClassDeclarationModsExtendsImplements::
AstClassDeclarationModsExtendsImplements(AstModifierList* const moli,
                                         AstClass* const cl,
                                         AstIdentifier* const id,
                                         AstExtends* const ex,
                                         AstName* const na,
                                         AstImplementsInterfaceList* const iminli,
                                         AstClassBody* const clbo,
                                         File* const srcfile) :
        AstClassDeclaration(moli, cl, id, ex, na, iminli, clbo, srcfile)
{
    prepend(moli);
    append(ex, id);
    append(na, ex);
    append(iminli, na);
}

AstClassDeclarationModsExtendsImplements::~AstClassDeclarationModsExtendsImplements()
{
}

void AstClassDeclarationModsExtendsImplements::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassDeclarationModsExtendsImplements::type() const
{
    return "ClassDeclarationModsExtendsImplements";
}

AstClassDeclarationModsImplements::AstClassDeclarationModsImplements(AstModifierList* const moli,
                                                                     AstClass* const cl,
                                                                     AstIdentifier* const id,
                                                                     AstImplementsInterfaceList* const iminli,
                                                                     AstClassBody* const clbo,
                                                                     File* const srcfile) :
        AstClassDeclaration(moli, cl, id, NULL, NULL, iminli, clbo, srcfile)
{
    prepend(moli);
    append(iminli, id);
}

AstClassDeclarationModsImplements::~AstClassDeclarationModsImplements()
{
}

void AstClassDeclarationModsImplements::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassDeclarationModsImplements::type() const
{
    return "ClassDeclarationModsImplements";
}

AstClassDeclarationNoModsBaseless::AstClassDeclarationNoModsBaseless(AstClass* const cl,
                                                               AstIdentifier* const id,
                                                               AstClassBody* const clbo,
                                                               File* const srcfile) :
        AstClassDeclaration(cl, id, clbo, srcfile)
{
}

AstClassDeclarationNoModsBaseless::~AstClassDeclarationNoModsBaseless()
{
}

void AstClassDeclarationNoModsBaseless::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassDeclarationNoModsBaseless::type() const
{
    return "ClassDeclarationNoModsBaseless";
}

AstClassDeclarationNoModsExtends::AstClassDeclarationNoModsExtends(AstClass* const cl,
                                                             AstIdentifier* const id,
                                                             AstExtends* const ex,
                                                             AstName* const na,
                                                             AstClassBody* const clbo,
                                                             File* const srcfile) :
        AstClassDeclaration(cl, id, clbo, srcfile), extends_(ex), name_(na)
{
    append(ex, id);
    append(na, ex);
}

AstClassDeclarationNoModsExtends::~AstClassDeclarationNoModsExtends()
{
}

AstExtends* AstClassDeclarationNoModsExtends::getExtends() const
{
    return extends_;
}

AstName* AstClassDeclarationNoModsExtends::getName() const
{
    return name_;
}

void AstClassDeclarationNoModsExtends::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassDeclarationNoModsExtends::type() const
{
    return "ClassDeclarationNoModsExtends";
}

AstClassDeclarationNoModsExtendsImplements::AstClassDeclarationNoModsExtendsImplements(AstClass* const cl,
                                                                                 AstIdentifier* const id,
                                                                                 AstExtends* const ex,
                                                                                 AstName* const na,
                                                                                 AstImplementsInterfaceList* const iminli,
                                                                                 AstClassBody* const clbo,
                                                                                 File* const srcfile) :
        AstClassDeclaration(cl, id, clbo, srcfile), extends_(ex), name_(na), implementsInterfaceList_(iminli)
{
    append(ex, id);
    append(na, ex);
    append(iminli, na);
}

AstClassDeclarationNoModsExtendsImplements::~AstClassDeclarationNoModsExtendsImplements()
{
}

AstExtends* AstClassDeclarationNoModsExtendsImplements::getExtends() const
{
    return extends_;
}

AstName* AstClassDeclarationNoModsExtendsImplements::getName() const
{
    return name_;
}

AstImplementsInterfaceList* AstClassDeclarationNoModsExtendsImplements::getImplementsInterfaceList() const
{
    return implementsInterfaceList_;
}

void AstClassDeclarationNoModsExtendsImplements::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassDeclarationNoModsExtendsImplements::type() const
{
    return "ClassDeclarationNoModsExtendsImplements";
}

AstClassDeclarationNoModsImplements::AstClassDeclarationNoModsImplements(AstClass* const cl,
                                                                   AstIdentifier* const id,
                                                                   AstImplementsInterfaceList* const iminli,
                                                                   AstClassBody* const clbo,
                                                                   File* const srcfile) :
        AstClassDeclaration(cl, id, clbo, srcfile), implementsInterfaceList_(iminli)
{
    append(iminli, id);
}

AstClassDeclarationNoModsImplements::~AstClassDeclarationNoModsImplements()
{
}

AstImplementsInterfaceList* AstClassDeclarationNoModsImplements::getImplementsInterfaceList() const
{
    return implementsInterfaceList_;
}

void AstClassDeclarationNoModsImplements::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassDeclarationNoModsImplements::type() const
{
    return "ClassDeclarationNoModsImplements";
}

AstClassInstanceCreationExpression::AstClassInstanceCreationExpression(AstNew* const ne,
                                                                       AstLparen* const lp,
                                                                       AstExpressionList* const exli,
                                                                       AstRparen* const rp,
                                                                       AstClassBody* const clbo,
                                                                       File* const srcfile) :
        AstNode(ne, lp, exli, rp, clbo, srcfile), 
     new_(ne), 
     lparen_(lp), 
     expressionList_(exli), 
     rparen_(rp), 
     classBody_(clbo)
{
}

AstClassInstanceCreationExpression::~AstClassInstanceCreationExpression()
{
}

AstNew* AstClassInstanceCreationExpression::getNew() const
{
    return new_;
}

AstLparen* AstClassInstanceCreationExpression::getLparen() const
{
    return lparen_;
}

AstExpressionList* AstClassInstanceCreationExpression::getExpressionList() const
{
    return expressionList_;
}

AstRparen* AstClassInstanceCreationExpression::getRparen() const
{
    return rparen_;
}

AstClassBody* AstClassInstanceCreationExpression::getClassBody() const
{
    return classBody_;
}

AstClassInstanceCreationExpressionNameExprBody::AstClassInstanceCreationExpressionNameExprBody(AstNew* const ne,
                                                                                               AstName* const na,
                                                                                               AstLparen* const lp,
                                                                                               AstExpressionList* const exli,
                                                                                               AstRparen* const rp,
                                                                                               AstClassBody* const clbo,
                                                                                               File* const srcfile) :
        AstClassInstanceCreationExpression(ne, lp, exli, rp, clbo, srcfile)
{
    append(na, ne);
}

AstClassInstanceCreationExpressionNameExprBody::~AstClassInstanceCreationExpressionNameExprBody()
{
}

AstName* AstClassInstanceCreationExpressionNameExprBody::getName() const
{
    return name_;
}

void AstClassInstanceCreationExpressionNameExprBody::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassInstanceCreationExpressionNameExprBody::type() const
{
    return "ClassInstanceCreationExpressionNameExprBody";
}

AstClassInstanceCreationExpressionNameExprNoBody::AstClassInstanceCreationExpressionNameExprNoBody(AstNew* const ne,
                                                                                                   AstName* const na,
                                                                                                   AstLparen* const lp,
                                                                                                   AstExpressionList* const exli,
                                                                                                   AstRparen* const rp,
                                                                                                   File* const srcfile) :
        AstClassInstanceCreationExpression(ne, lp, exli, rp, NULL, srcfile), name_(na)
{
    append(na, ne);
}

AstClassInstanceCreationExpressionNameExprNoBody::~AstClassInstanceCreationExpressionNameExprNoBody()
{
}

AstName* AstClassInstanceCreationExpressionNameExprNoBody::getName() const
{
    return name_;
}

void AstClassInstanceCreationExpressionNameExprNoBody::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassInstanceCreationExpressionNameExprNoBody::type() const
{
    return "ClassInstanceCreationExpressionNameExprNoBody";
}

AstClassInstanceCreationExpressionNameIdExprBody::AstClassInstanceCreationExpressionNameIdExprBody(AstName* const na,
                                                                                                   AstDot* const d,
                                                                                                   AstNew* const ne,
                                                                                                   AstIdentifier* const id,
                                                                                                   AstLparen* const lp,
                                                                                                   AstExpressionList* const exli,
                                                                                                   AstRparen* const rp,
                                                                                                   AstClassBody* const clbo,
                                                                                                   File* const srcfile) :
        AstClassInstanceCreationExpression(ne, lp, exli, rp, clbo, srcfile), 
     name_(na), 
     dot_(d), 
     identifier_(id)
{
    prepend(na);
    append(d, na);
    append(id, ne);
}

AstClassInstanceCreationExpressionNameIdExprBody::~AstClassInstanceCreationExpressionNameIdExprBody()
{
}

AstName* AstClassInstanceCreationExpressionNameIdExprBody::getName() const
{
    return name_;
}

AstDot* AstClassInstanceCreationExpressionNameIdExprBody::getDot() const
{
    return dot_;
}

AstIdentifier* AstClassInstanceCreationExpressionNameIdExprBody::getIdentifier() const
{
    return identifier_;
}

void AstClassInstanceCreationExpressionNameIdExprBody::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassInstanceCreationExpressionNameIdExprBody::type() const
{
    return "ClassInstanceCreationExpressionNameIdExprBody";
}

AstClassInstanceCreationExpressionNameIdExprNoBody::AstClassInstanceCreationExpressionNameIdExprNoBody(AstName* const na,
                                                                                                       AstDot* const d,
                                                                                                       AstNew* const ne,
                                                                                                       AstIdentifier* const id,
                                                                                                       AstLparen* const lp,
                                                                                                       AstExpressionList* const exli,
                                                                                                       AstRparen* const rp,
                                                                                                       File* const srcfile) :
        AstClassInstanceCreationExpression(ne, lp, exli, rp, NULL, srcfile), 
     name_(na), 
     dot_(d), 
     identifier_(id)
{
    prepend(na);
    append(d, na);
    append(id, ne);
}

AstClassInstanceCreationExpressionNameIdExprNoBody::~AstClassInstanceCreationExpressionNameIdExprNoBody()
{
}

AstName* AstClassInstanceCreationExpressionNameIdExprNoBody::getName() const
{
    return name_;
}

AstDot* AstClassInstanceCreationExpressionNameIdExprNoBody::getDot() const
{
    return dot_;
}

AstIdentifier* AstClassInstanceCreationExpressionNameIdExprNoBody::getIdentifier() const
{
    return identifier_;
}

void AstClassInstanceCreationExpressionNameIdExprNoBody::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassInstanceCreationExpressionNameIdExprNoBody::type() const
{
    return "ClassInstanceCreationExpressionNameIdExprNoBody";
}

AstClassInstanceCreationExpressionNameIdNoExprBody::AstClassInstanceCreationExpressionNameIdNoExprBody(AstName* const na,
                                                                                                       AstDot* const d,
                                                                                                       AstNew* const ne,
                                                                                                       AstIdentifier* const id,
                                                                                                       AstLparen* const lp,
                                                                                                       AstRparen* const rp,
                                                                                                       AstClassBody* const clbo,
                                                                                                       File* const srcfile) :
        AstClassInstanceCreationExpression(ne, lp, NULL, rp, clbo, srcfile), 
     name_(na), 
     dot_(d), 
     identifier_(id)
{
    prepend(na);
    append(d, na);
    append(id, ne);
}

AstClassInstanceCreationExpressionNameIdNoExprBody::~AstClassInstanceCreationExpressionNameIdNoExprBody()
{
}

AstName* AstClassInstanceCreationExpressionNameIdNoExprBody::getName() const
{
    return name_;
}

AstDot* AstClassInstanceCreationExpressionNameIdNoExprBody::getDot() const
{
    return dot_;
}

AstIdentifier* AstClassInstanceCreationExpressionNameIdNoExprBody::getIdentifier() const
{
    return identifier_;
}

void AstClassInstanceCreationExpressionNameIdNoExprBody::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassInstanceCreationExpressionNameIdNoExprBody::type() const
{
    return "ClassInstanceCreationExpressionNameIdNoExprBody";
}

AstClassInstanceCreationExpressionNameIdNoExprNoBody::AstClassInstanceCreationExpressionNameIdNoExprNoBody(AstName* const na,
                                                                                                           AstDot* const d,
                                                                                                           AstNew* const ne,
                                                                                                           AstIdentifier* const id,
                                                                                                           AstLparen* const lp,
                                                                                                           AstRparen* const rp,
                                                                                                           File* const srcfile) :
        AstClassInstanceCreationExpression(ne, lp, NULL, rp, NULL, srcfile), 
     name_(na), 
     dot_(d), 
     identifier_(id)
{
    prepend(na);
    append(d, na);
    append(id, ne);
}

AstClassInstanceCreationExpressionNameIdNoExprNoBody::~AstClassInstanceCreationExpressionNameIdNoExprNoBody()
{
}

AstName* AstClassInstanceCreationExpressionNameIdNoExprNoBody::getName() const
{
    return name_;
}

AstDot* AstClassInstanceCreationExpressionNameIdNoExprNoBody::getDot() const
{
    return dot_;
}

AstIdentifier* AstClassInstanceCreationExpressionNameIdNoExprNoBody::getIdentifier() const
{
    return identifier_;
}

void AstClassInstanceCreationExpressionNameIdNoExprNoBody::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassInstanceCreationExpressionNameIdNoExprNoBody::type() const
{
    return "ClassInstanceCreationExpressionNameIdNoExprNoBody";
}

AstClassInstanceCreationExpressionNameNoExprBody::AstClassInstanceCreationExpressionNameNoExprBody(AstNew* const ne,
                                                                                                   AstName* const na,
                                                                                                   AstLparen* const lp,
                                                                                                   AstRparen* const rp,
                                                                                                   AstClassBody* const clbo,
                                                                                                   File* const srcfile) :
        AstClassInstanceCreationExpression(ne, lp, NULL, rp, clbo, srcfile), name_(na)
{
    append(na, ne);
}

AstClassInstanceCreationExpressionNameNoExprBody::~AstClassInstanceCreationExpressionNameNoExprBody()
{
}

AstName* AstClassInstanceCreationExpressionNameNoExprBody::getName() const
{
    return name_;
}

void AstClassInstanceCreationExpressionNameNoExprBody::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassInstanceCreationExpressionNameNoExprBody::type() const
{
    return "ClassInstanceCreationExpressionNameNoExprBody";
}

AstClassInstanceCreationExpressionNameNoExprNoBody::AstClassInstanceCreationExpressionNameNoExprNoBody(AstNew* const ne,
                                                                                                       AstName* const na,
                                                                                                       AstLparen* const lp,
                                                                                                       AstRparen* const rp,
                                                                                                       File* const srcfile) :
        AstClassInstanceCreationExpression(ne, lp, NULL, rp, NULL, srcfile), name_(na)
{
    append(na, ne);
}

AstClassInstanceCreationExpressionNameNoExprNoBody::~AstClassInstanceCreationExpressionNameNoExprNoBody()
{
}

AstName* AstClassInstanceCreationExpressionNameNoExprNoBody::getName() const
{
    return name_;
}

void AstClassInstanceCreationExpressionNameNoExprNoBody::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassInstanceCreationExpressionNameNoExprNoBody::type() const
{
    return "ClassInstanceCreationExpressionNameNoExprNoBody";
}

AstClassInstanceCreationExpressionPrimaryExprBody::AstClassInstanceCreationExpressionPrimaryExprBody(AstItem* const pr,
                                                                                                     AstDot* const d,
                                                                                                     AstNew* const ne,
                                                                                                     AstIdentifier* const id,
                                                                                                     AstLparen* const lp,
                                                                                                     AstExpressionList* const exli,
                                                                                                     AstRparen* const rp,
                                                                                                     AstClassBody* const clbo,
                                                                                                     File* const srcfile) :
        AstClassInstanceCreationExpression(ne, lp, exli, rp, clbo, srcfile),
     primary_(pr),
     dot_(d),
     identifier_(id)
{
    prepend(pr);
    append(d, pr);
    append(id, ne);
}

AstClassInstanceCreationExpressionPrimaryExprBody::~AstClassInstanceCreationExpressionPrimaryExprBody()
{
}

AstItem* AstClassInstanceCreationExpressionPrimaryExprBody::getPrimary() const
{
    return primary_;
}

AstDot* AstClassInstanceCreationExpressionPrimaryExprBody::getDot() const
{
    return dot_;
}

AstIdentifier* AstClassInstanceCreationExpressionPrimaryExprBody::getIdentifier() const
{
    return identifier_;
}

void AstClassInstanceCreationExpressionPrimaryExprBody::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassInstanceCreationExpressionPrimaryExprBody::type() const
{
    return "ClassInstanceCreationExpressionPrimaryExprBody";
}

AstClassInstanceCreationExpressionPrimaryExprNoBody::AstClassInstanceCreationExpressionPrimaryExprNoBody(AstItem* const pr,
                                                                                                         AstDot* const d,
                                                                                                         AstNew* const ne,
                                                                                                         AstIdentifier* const id,
                                                                                                         AstLparen* const lp,
                                                                                                         AstExpressionList* const exli,
                                                                                                         AstRparen* const rp,
                                                                                                         File* const srcfile) :
        AstClassInstanceCreationExpression(ne, lp, exli, rp, NULL, srcfile),
     primary_(pr), 
     dot_(d),
     identifier_(id)
{
    prepend(pr);
    append(d, pr);
    append(id, ne);
}

AstClassInstanceCreationExpressionPrimaryExprNoBody::~AstClassInstanceCreationExpressionPrimaryExprNoBody()
{
}

AstItem* AstClassInstanceCreationExpressionPrimaryExprNoBody::getPrimary() const
{
    return primary_;
}

AstDot* AstClassInstanceCreationExpressionPrimaryExprNoBody::getDot() const
{
    return dot_;
}

AstIdentifier* AstClassInstanceCreationExpressionPrimaryExprNoBody::getIdentifier() const
{
    return identifier_;
}

void AstClassInstanceCreationExpressionPrimaryExprNoBody::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassInstanceCreationExpressionPrimaryExprNoBody::type() const
{
    return "ClassInstanceCreationExpressionPrimaryExprNoBody";
}

AstClassInstanceCreationExpressionPrimaryNoExprBody::AstClassInstanceCreationExpressionPrimaryNoExprBody(AstItem* const pr,
                                                                                                         AstDot* const d,
                                                                                                         AstNew* const ne,
                                                                                                         AstIdentifier* const id,
                                                                                                         AstLparen* const lp,
                                                                                                         AstRparen* const rp,
                                                                                                         AstClassBody* const clbo,
                                                                                                         File* const srcfile) :
        AstClassInstanceCreationExpression(ne, lp, NULL, rp, clbo, srcfile),
     primary_(pr), 
     dot_(d),
     identifier_(id)
{
    prepend(pr);
    append(d, pr);
    append(id, ne);
}

AstClassInstanceCreationExpressionPrimaryNoExprBody::~AstClassInstanceCreationExpressionPrimaryNoExprBody()
{
}

AstItem* AstClassInstanceCreationExpressionPrimaryNoExprBody::getPrimary() const
{
    return primary_;
}

AstDot* AstClassInstanceCreationExpressionPrimaryNoExprBody::getDot() const
{
    return dot_;
}

AstIdentifier* AstClassInstanceCreationExpressionPrimaryNoExprBody::getIdentifier() const
{
    return identifier_;
}

void AstClassInstanceCreationExpressionPrimaryNoExprBody::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassInstanceCreationExpressionPrimaryNoExprBody::type() const
{
    return "ClassInstanceCreationExpressionPrimaryNoExprBody";
}

AstClassInstanceCreationExpressionPrimaryNoExprNoBody::AstClassInstanceCreationExpressionPrimaryNoExprNoBody(AstItem* const pr,
                                                                                                       AstDot* const d,
                                                                                                       AstNew* const ne,
                                                                                                       AstIdentifier* const id,
                                                                                                       AstLparen* const lp,
                                                                                                       AstRparen* const rp,
                                                                                                       File* const srcfile) :
        AstClassInstanceCreationExpression(ne, lp, NULL, rp, NULL, srcfile), 
     primary_(pr), 
     dot_(d), 
     identifier_(id)
{
    prepend(pr);
    append(d, pr);
    append(id, ne);
}

AstClassInstanceCreationExpressionPrimaryNoExprNoBody::~AstClassInstanceCreationExpressionPrimaryNoExprNoBody()
{
}

AstItem* AstClassInstanceCreationExpressionPrimaryNoExprNoBody::getPrimary() const
{
    return primary_;
}

AstDot* AstClassInstanceCreationExpressionPrimaryNoExprNoBody::getDot() const
{
    return dot_;
}

AstIdentifier* AstClassInstanceCreationExpressionPrimaryNoExprNoBody::getIdentifier() const
{
    return identifier_;
}

void AstClassInstanceCreationExpressionPrimaryNoExprNoBody::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassInstanceCreationExpressionPrimaryNoExprNoBody::type() const
{
    return "ClassInstanceCreationExpressionPrimaryNoExprNoBody";
}

AstClassMemberDeclaration::AstClassMemberDeclaration(AstModifierList* const moli,
                                                     AstClass* const cl,
                                                     AstIdentifier* const id,
                                                     AstExtends* const ex,
                                                     AstName* const na,
                                                     AstImplementsInterfaceList* const iminli,
                                                     AstClassBody* const clbo,
                                                     File* const srcfile) :
        AstNode(moli, cl, id, ex, na, iminli, clbo, srcfile), 
     modifierList_(moli), 
     class_(cl), 
     identifier_(id), 
     extends_(ex), 
     name_(na), 
     implementsInterfaceList_(iminli), 
     classBody_(clbo)
{
}

AstClassMemberDeclaration::~AstClassMemberDeclaration()
{
}

AstModifierList* AstClassMemberDeclaration::getModifierList() const
{
    return modifierList_;
}

AstClass* AstClassMemberDeclaration::getClass() const
{
    return class_;
}

AstIdentifier* AstClassMemberDeclaration::getIdentifier() const
{
    return identifier_;
}

AstExtends* AstClassMemberDeclaration::getExtends() const
{
    return extends_;
}

AstName* AstClassMemberDeclaration::getName() const
{
    return name_;
}

AstImplementsInterfaceList* AstClassMemberDeclaration::getImplementsInterfaceList() const
{
    return implementsInterfaceList_;
}

AstClassBody* AstClassMemberDeclaration::getClassBody() const
{
    return classBody_;
}

AstClassMemberDeclarationModsExtendsImplements::AstClassMemberDeclarationModsExtendsImplements(AstModifierList* const moli,
                                                                                               AstClass* const cl,
                                                                                               AstIdentifier* const id,
                                                                                               AstExtends* const ex,
                                                                                               AstName* const na,
                                                                                               AstImplementsInterfaceList* const iminli,
                                                                                               AstClassBody* const clbo,
                                                                                               File* const srcfile) :
        AstClassMemberDeclaration(moli, cl, id, ex, na, iminli, clbo, srcfile)
{
}

AstClassMemberDeclarationModsExtendsImplements::~AstClassMemberDeclarationModsExtendsImplements()
{
}

void AstClassMemberDeclarationModsExtendsImplements::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassMemberDeclarationModsExtendsImplements::type() const
{
    return "ClassMemberDeclarationModsExtendsImplements";
}

AstClassMemberDeclarationModsExtendsNoImplements::AstClassMemberDeclarationModsExtendsNoImplements(AstModifierList* const moli,
                                                                                                   AstClass* const cl,
                                                                                                   AstIdentifier* const id,
                                                                                                   AstExtends* const ex,
                                                                                                   AstName* const na,
                                                                                                   AstClassBody* const clbo,
                                                                                                   File* const srcfile) :
        AstClassMemberDeclaration(moli, cl, id, ex, na, NULL, clbo, srcfile)
{
}

AstClassMemberDeclarationModsExtendsNoImplements::~AstClassMemberDeclarationModsExtendsNoImplements()
{
}

void AstClassMemberDeclarationModsExtendsNoImplements::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassMemberDeclarationModsExtendsNoImplements::type() const
{
    return "ClassMemberDeclarationModsExtendsNoImplements";
}

AstClassMemberDeclarationModsNoExtendsImplements::AstClassMemberDeclarationModsNoExtendsImplements(AstModifierList* const moli,
                                                                                                   AstClass* const cl,
                                                                                                   AstIdentifier* const id,
                                                                                                   AstImplementsInterfaceList* const iminli,
                                                                                                   AstClassBody* const clbo,
                                                                                                   File* const srcfile) :
        AstClassMemberDeclaration(moli, cl, id, NULL, NULL, iminli, clbo, srcfile)
{
}

AstClassMemberDeclarationModsNoExtendsImplements::~AstClassMemberDeclarationModsNoExtendsImplements()
{
}

void AstClassMemberDeclarationModsNoExtendsImplements::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassMemberDeclarationModsNoExtendsImplements::type() const
{
    return "ClassMemberDeclarationModsNoExtendsImplements";
}

AstClassMemberDeclarationModsNoExtendsNoImplements::AstClassMemberDeclarationModsNoExtendsNoImplements(AstModifierList* const moli,
                                                                                                       AstClass* const cl,
                                                                                                       AstIdentifier* const id,
                                                                                                       AstClassBody* const clbo,
                                                                                                       File* const srcfile) :
        AstClassMemberDeclaration(moli, cl, id, NULL, NULL, NULL, clbo, srcfile)
{
}

AstClassMemberDeclarationModsNoExtendsNoImplements::~AstClassMemberDeclarationModsNoExtendsNoImplements()
{
}

void AstClassMemberDeclarationModsNoExtendsNoImplements::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassMemberDeclarationModsNoExtendsNoImplements::type() const
{
    return "ClassMemberDeclarationModsNoExtendsNoImplements";
}

AstClassMemberDeclarationNoModsExtendsImplements::AstClassMemberDeclarationNoModsExtendsImplements(AstClass* const cl,
                                                                                                   AstIdentifier* const id,
                                                                                                   AstExtends* const ex,
                                                                                                   AstName* const na,
                                                                                                   AstImplementsInterfaceList* const iminli,
                                                                                                   AstClassBody* const clbo,
                                                                                                   File* const srcfile) :
        AstClassMemberDeclaration(NULL, cl, id, ex, na, iminli, clbo, srcfile)
{
}

AstClassMemberDeclarationNoModsExtendsImplements::~AstClassMemberDeclarationNoModsExtendsImplements()
{
}

void AstClassMemberDeclarationNoModsExtendsImplements::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassMemberDeclarationNoModsExtendsImplements::type() const
{
    return "ClassMemberDeclarationNoModsExtendsImplements";
}

AstClassMemberDeclarationNoModsExtendsNoImplements::AstClassMemberDeclarationNoModsExtendsNoImplements(AstClass* const cl,
                                                                                                       AstIdentifier* const id,
                                                                                                       AstExtends* const ex,
                                                                                                       AstName* const na,
                                                                                                       AstClassBody* const clbo,
                                                                                                       File* const srcfile) :
        AstClassMemberDeclaration(NULL, cl, id, ex, na, NULL, clbo, srcfile)
{
}

AstClassMemberDeclarationNoModsExtendsNoImplements::~AstClassMemberDeclarationNoModsExtendsNoImplements()
{
}

void AstClassMemberDeclarationNoModsExtendsNoImplements::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassMemberDeclarationNoModsExtendsNoImplements::type() const
{
    return "ClassMemberDeclarationNoModsExtendsNoImplements";
}

AstClassMemberDeclarationNoModsNoExtendsImplements::AstClassMemberDeclarationNoModsNoExtendsImplements(AstClass* const cl,
                                                                                                       AstIdentifier* const id,
                                                                                                       AstImplementsInterfaceList* const iminli,
                                                                                                       AstClassBody* const clbo,
                                                                                                       File* const srcfile) :
        AstClassMemberDeclaration(NULL, cl, id, NULL, NULL, iminli, clbo, srcfile)
{
}

AstClassMemberDeclarationNoModsNoExtendsImplements::~AstClassMemberDeclarationNoModsNoExtendsImplements()
{
}

void AstClassMemberDeclarationNoModsNoExtendsImplements::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassMemberDeclarationNoModsNoExtendsImplements::type() const
{
    return "ClassMemberDeclarationNoModsNoExtendsImplements";
}

AstClassMemberDeclarationNoModsNoExtendsNoImplements::AstClassMemberDeclarationNoModsNoExtendsNoImplements(AstClass* const cl,
                                                                                                           AstIdentifier* const id,
                                                                                                           AstClassBody* const clbo,
                                                                                                           File* const srcfile) :
        AstClassMemberDeclaration(NULL, cl, id, NULL, NULL, NULL, clbo, srcfile)
{
}

AstClassMemberDeclarationNoModsNoExtendsNoImplements::~AstClassMemberDeclarationNoModsNoExtendsNoImplements()
{
}

void AstClassMemberDeclarationNoModsNoExtendsNoImplements::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstClassMemberDeclarationNoModsNoExtendsNoImplements::type() const
{
    return "ClassMemberDeclarationNoModsNoExtendsNoImplements";
}

AstColon::AstColon(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstColon::~AstColon()
{
}

string AstColon::type() const
{
    return "COLON";
}

void AstColon::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstColon::length() const
{
    return 1;
}

AstComma::AstComma(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstComma::~AstComma()
{
}

string AstComma::type() const
{
    return "COMMA";
}

void AstComma::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstComma::length() const
{
    return 1;
}

AstComment::AstComment(char* start, char* end, File* const srcfile) :
        AstNode(srcfile),
        start_(start),
        end_(end)
{
}

AstComment::~AstComment()
{
}

string AstComment::fulltext() const
{
    return string(start_, end_ - start_);
}

char* AstComment::position() const
{
    return start_;
}

char* AstComment::endPosition() const
{
    return end_;
}

AstComp::AstComp(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstComp::~AstComp()
{
}

string AstComp::type() const
{
    return "COMP";
}

void AstComp::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstComp::length() const
{
    return 1;
}

AstCompilationUnit::AstCompilationUnit(AstPackageDeclaration* const pd, 
                                       AstImportDeclarationList* const idl,
                                       AstTypeDeclarationList* const tdl,
                                       File* const srcfile) :
        AstNode(srcfile), 
     packageDeclaration_(pd),
     importDeclarationList_(idl),
     typeDeclarationList_(tdl),
     lastNoncode_(NULL)
{
    append(pd);
    append(idl);
    append(tdl);
}

AstCompilationUnit::~AstCompilationUnit()
{
}

AstPackageDeclaration* AstCompilationUnit::getPackageDeclaration() const
{
    return packageDeclaration_;
}

AstImportDeclarationList* AstCompilationUnit::getImportDeclarationList() const
{
    return importDeclarationList_;
}

AstTypeDeclarationList* AstCompilationUnit::getTypeDeclarationList() const
{
    return typeDeclarationList_;
}

void AstCompilationUnit::setTrailingNoncode(AstNoncode* const nc)
{
     lastNoncode_ = nc;
     append(nc);
}

AstNoncode* AstCompilationUnit::trailingNoncode() const
{
    return lastNoncode_;
}

AstCompilationUnitEmpty::AstCompilationUnitEmpty(File* const srcfile) :
        AstCompilationUnit(NULL, NULL, NULL, srcfile)
{
}

AstCompilationUnitEmpty::~AstCompilationUnitEmpty()
{
}

void AstCompilationUnitEmpty::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstCompilationUnitEmpty::type() const
{
    return "CompilationUnitEmpty";
}

AstCompilationUnitImp::AstCompilationUnitImp(AstImportDeclarationList* const idl, 
                                       File* const srcfile) :
        AstCompilationUnit(NULL, idl, NULL, srcfile)
{
}

AstCompilationUnitImp::~AstCompilationUnitImp()
{
}

void AstCompilationUnitImp::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstCompilationUnitImp::type() const
{
    return "CompilationUnitImp";
}

AstCompilationUnitImpTypes::AstCompilationUnitImpTypes(AstImportDeclarationList* const idl, 
                                                       AstTypeDeclarationList* const tdl,
                                                       File* const srcfile) :
        AstCompilationUnit(NULL, idl, tdl, srcfile)
{
}

AstCompilationUnitImpTypes::~AstCompilationUnitImpTypes()
{
}

void AstCompilationUnitImpTypes::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstCompilationUnitImpTypes::type() const
{
    return "CompilationUnitImpTypes";
}

AstCompilationUnitPkg::AstCompilationUnitPkg(AstPackageDeclaration* const pd, 
                                             File* const srcfile) :
        AstCompilationUnit(pd, NULL, NULL, srcfile)
{
}

AstCompilationUnitPkg::~AstCompilationUnitPkg()
{
}

void AstCompilationUnitPkg::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstCompilationUnitPkg::type() const
{
    return "CompilationUnitPkg";
}

AstCompilationUnitPkgImp::AstCompilationUnitPkgImp(AstPackageDeclaration* const pd, 
                                                   AstImportDeclarationList* const idl,
                                                   File* const srcfile) :
        AstCompilationUnit(pd, idl, NULL, srcfile)
{
}

AstCompilationUnitPkgImp::~AstCompilationUnitPkgImp()
{
}

void AstCompilationUnitPkgImp::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstCompilationUnitPkgImp::type() const
{
    return "CompilationUnitPkgImp";
}

AstCompilationUnitPkgImpTypes::AstCompilationUnitPkgImpTypes(AstPackageDeclaration* const pd, 
                                                             AstImportDeclarationList* const idl,
                                                             AstTypeDeclarationList* const tdl,
                                                             File* const srcfile) :
        AstCompilationUnit(pd, idl, tdl, srcfile)
{
}

AstCompilationUnitPkgImpTypes::~AstCompilationUnitPkgImpTypes()
{
}

void AstCompilationUnitPkgImpTypes::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstCompilationUnitPkgImpTypes::type() const
{
    return "CompilationUnitPkgImpTypes";
}

AstCompilationUnitPkgTypes::AstCompilationUnitPkgTypes(AstPackageDeclaration* const pd, 
                                                 AstTypeDeclarationList* const tdl,
                                                 File* const srcfile) :
        AstCompilationUnit(pd, NULL, tdl, srcfile)
{
}

AstCompilationUnitPkgTypes::~AstCompilationUnitPkgTypes()
{
}

void AstCompilationUnitPkgTypes::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstCompilationUnitPkgTypes::type() const
{
    return "CompilationUnitPkgTypes";
}

AstCompilationUnitTypes::AstCompilationUnitTypes(AstTypeDeclarationList* const tdl, 
                                           File* const srcfile) :
        AstCompilationUnit(NULL, NULL, tdl, srcfile)
{
}

AstCompilationUnitTypes::~AstCompilationUnitTypes()
{
}

void AstCompilationUnitTypes::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstCompilationUnitTypes::type() const
{
    return "CompilationUnitTypes";
}

AstConditionalAndExpression::AstConditionalAndExpression(AstItem* const le, 
                                                         AstAndand* const a,
                                                         AstItem* const re,
                                                         File* const srcfile /* = NULL */) :
        AstNode(le, a, re, srcfile), lhsExpression_(le), andand_(a), rhsExpression_(re)
{
}

AstConditionalAndExpression::~AstConditionalAndExpression()
{
}

AstItem* AstConditionalAndExpression::getLhsExpression() const
{
    return lhsExpression_;
}

AstAndand* AstConditionalAndExpression::getAndand() const
{
    return andand_;
}

AstItem* AstConditionalAndExpression::getRhsExpression() const
{
    return rhsExpression_;
}

void AstConditionalAndExpression::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstConditionalAndExpression::type() const
{
    return "ConditionalAndExpression";
}

AstConditionalArithExpression::AstConditionalArithExpression(AstItem* const ie, 
                                                             AstQuestion* const q,
                                                             AstItem* const te,
                                                             AstColon* const c,
                                                             AstItem* const ee,
                                                             File* const srcfile /* = NULL */) :
        AstNode(ie, q, te, c, ee, srcfile),
     ifExpression_(ie), 
     question_(q), 
     thenExpression_(te), 
     colon_(c), 
     elseExpression_(ee)
{
}

AstConditionalArithExpression::~AstConditionalArithExpression()
{
}

AstItem* AstConditionalArithExpression::getIfExpression() const
{
    return ifExpression_;
}

AstQuestion* AstConditionalArithExpression::getQuestion() const
{
    return question_;
}

AstItem* AstConditionalArithExpression::getThenExpression() const
{
    return thenExpression_;
}

AstColon* AstConditionalArithExpression::getColon() const
{
    return colon_;
}

AstItem* AstConditionalArithExpression::getElseExpression() const
{
    return elseExpression_;
}

void AstConditionalArithExpression::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstConditionalArithExpression::type() const
{
    return "ConditionalArithExpression";
}

AstConditionalOrExpression::AstConditionalOrExpression(AstItem* const le, 
                                                       AstOror* const o,
                                                       AstItem* const re,
                                                       File* const srcfile /* = NULL */) :
        AstNode(le, o, re, srcfile), lhsExpression_(le), oror_(o), rhsExpression_(re)
{
}

AstConditionalOrExpression::~AstConditionalOrExpression()
{
}

AstItem* AstConditionalOrExpression::getLhsExpression() const
{
    return lhsExpression_;
}

AstOror* AstConditionalOrExpression::getOror() const
{
    return oror_;
}

AstItem* AstConditionalOrExpression::getRhsExpression() const
{
    return rhsExpression_;
}

void AstConditionalOrExpression::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstConditionalOrExpression::type() const
{
    return "ConditionalOrExpression";
}

AstConst::AstConst(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstConst::~AstConst()
{
}

string AstConst::type() const
{
    return "CONST";
}

void AstConst::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstConst::length() const
{
    return 5;
}

AstConstructorBodyBlock::AstConstructorBodyBlock(AstLbrace* const lb,
                                                 AstBlockStatementList* const blstli,
                                                 AstRbrace* const rb,
                                                 File* const srcfile) :
        AstConstructorBody(lb, NULL, blstli, rb, srcfile)
{
}

AstConstructorBodyBlock::~AstConstructorBodyBlock()
{
}

void AstConstructorBodyBlock::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstConstructorBodyBlock::type() const
{
    return "ConstructorBodyBlock";
}

AstConstructorBody::AstConstructorBody(AstLbrace* const lb,
                                       AstExplicitConstructorInvocation* const excoin,
                                       AstBlockStatementList* const blstli,
                                       AstRbrace* const rb,
                                       File* const srcfile) :
        AstNode(lb, excoin, blstli, rb, srcfile),
     lbrace_(lb), 
     explicitConstructorInvocation_(excoin), 
     blockStatementList_(blstli), 
     rbrace_(rb)
{
}

AstConstructorBody::~AstConstructorBody()
{
}

AstLbrace* AstConstructorBody::getLbrace() const
{
    return lbrace_;
}

AstExplicitConstructorInvocation* AstConstructorBody::getExplicitConstructorInvocation() const
{
    return explicitConstructorInvocation_;
}

AstBlockStatementList* AstConstructorBody::getBlockStatementList() const
{
    return blockStatementList_;
}

AstRbrace* AstConstructorBody::getRbrace() const
{
    return rbrace_;
}

AstConstructorBodyEmpty::AstConstructorBodyEmpty(AstLbrace* const lb,
                                           AstRbrace* const rb,
                                           File* const srcfile) :
        AstConstructorBody(lb, NULL, NULL, rb, srcfile)
{
}

AstConstructorBodyEmpty::~AstConstructorBodyEmpty()
{
}

void AstConstructorBodyEmpty::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstConstructorBodyEmpty::type() const
{
    return "ConstructorBodyEmpty";
}

AstConstructorBodyExplicitBlock::AstConstructorBodyExplicitBlock(AstLbrace* const lb,
                                                           AstExplicitConstructorInvocation* const excoin,
                                                           AstBlockStatementList* const blstli,
                                                           AstRbrace* const rb,
                                                           File* const srcfile) :
        AstConstructorBody(lb, excoin, blstli, rb, srcfile)
{
}

AstConstructorBodyExplicitBlock::~AstConstructorBodyExplicitBlock()
{
}

void AstConstructorBodyExplicitBlock::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstConstructorBodyExplicitBlock::type() const
{
    return "ConstructorBodyExplicitBlock";
}

AstConstructorBodyExplicit::AstConstructorBodyExplicit(AstLbrace* const lb,
                                                 AstExplicitConstructorInvocation* const excoin,
                                                 AstRbrace* const rb,
                                                 File* const srcfile) :
        AstConstructorBody(lb, excoin, NULL, rb, srcfile)
{
}

AstConstructorBodyExplicit::~AstConstructorBodyExplicit()
{
}

void AstConstructorBodyExplicit::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstConstructorBodyExplicit::type() const
{
    return "ConstructorBodyExplicit";
}

AstConstructorDeclaration::AstConstructorDeclaration(AstModifierList* const moli,
                                                     AstConstructorDeclarator* const code,
                                                     AstThrowsNameList* const thnali,
                                                     AstConstructorBody* const cobo,
                                                     File* const srcfile) :
        AstNode(moli, code, thnali, cobo, srcfile), 
     modifierList_(moli), 
     constructorDeclarator_(code), 
     throwsNameList_(thnali), 
     constructorBody_(cobo)
{
}

AstConstructorDeclaration::~AstConstructorDeclaration()
{
}

AstModifierList* AstConstructorDeclaration::getModifierList() const
{
    return modifierList_;
}

AstConstructorDeclarator* AstConstructorDeclaration::getConstructorDeclarator() const
{
    return constructorDeclarator_;
}

AstThrowsNameList* AstConstructorDeclaration::getThrowsNameList() const
{
    return throwsNameList_;
}

AstConstructorBody* AstConstructorDeclaration::getConstructorBody() const
{
    return constructorBody_;
}

AstConstructorDeclarationModsNoThrows::AstConstructorDeclarationModsNoThrows(AstModifierList* const moli,
                                                                             AstConstructorDeclarator* const code,
                                                                             AstConstructorBody* const cobo,
                                                                             File* const srcfile) :
        AstConstructorDeclaration(moli, code, NULL, cobo, srcfile)
{
}

AstConstructorDeclarationModsNoThrows::~AstConstructorDeclarationModsNoThrows()
{
}

void AstConstructorDeclarationModsNoThrows::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstConstructorDeclarationModsNoThrows::type() const
{
    return "ConstructorDeclarationModsNoThrows";
}

AstConstructorDeclarationModsThrows::AstConstructorDeclarationModsThrows(AstModifierList* const moli,
                                                                   AstConstructorDeclarator* const code,
                                                                   AstThrowsNameList* const thnali,
                                                                   AstConstructorBody* const cobo,
                                                                   File* const srcfile) :
        AstConstructorDeclaration(moli, code, thnali, cobo, srcfile)
{
}

AstConstructorDeclarationModsThrows::~AstConstructorDeclarationModsThrows()
{
}

void AstConstructorDeclarationModsThrows::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstConstructorDeclarationModsThrows::type() const
{
    return "ConstructorDeclarationModsThrows";
}

AstConstructorDeclarationNoModsNoThrows::AstConstructorDeclarationNoModsNoThrows(AstConstructorDeclarator* const code,
                                                                                 AstConstructorBody* const cobo,
                                                                                 File* const srcfile) :
        AstConstructorDeclaration(NULL, code, NULL, cobo, srcfile)
{
}

AstConstructorDeclarationNoModsNoThrows::~AstConstructorDeclarationNoModsNoThrows()
{
}

void AstConstructorDeclarationNoModsNoThrows::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstConstructorDeclarationNoModsNoThrows::type() const
{
    return "ConstructorDeclarationNoModsNoThrows";
}

AstConstructorDeclarationNoModsThrows::AstConstructorDeclarationNoModsThrows(AstConstructorDeclarator* const code,
                                                                             AstThrowsNameList* const thnali,
                                                                             AstConstructorBody* const cobo,
                                                                             File* const srcfile) :
        AstConstructorDeclaration(NULL, code, thnali, cobo, srcfile)
{
}

AstConstructorDeclarationNoModsThrows::~AstConstructorDeclarationNoModsThrows()
{
}

void AstConstructorDeclarationNoModsThrows::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstConstructorDeclarationNoModsThrows::type() const
{
    return "ConstructorDeclarationNoModsThrows";
}

AstConstructorDeclarator::AstConstructorDeclarator(AstIdentifier* const id,
                                                   AstLparen* const lp,
                                                   AstFormalParameterList* const fopali,
                                                   AstRparen* const rp,
                                                   File* const srcfile) :
        AstNode(id, lp, fopali, rp, srcfile), 
     identifier_(id), 
     lparen_(lp), 
     formalParameterList_(fopali), 
     rparen_(rp)
{
}

AstConstructorDeclarator::~AstConstructorDeclarator()
{
}

AstIdentifier* AstConstructorDeclarator::getIdentifier() const
{
    return identifier_;
}

AstLparen* AstConstructorDeclarator::getLparen() const
{
    return lparen_;
}

AstFormalParameterList* AstConstructorDeclarator::getFormalParameterList() const
{
    return formalParameterList_;
}

AstRparen* AstConstructorDeclarator::getRparen() const
{
    return rparen_;
}

AstConstructorDeclaratorEmpty::AstConstructorDeclaratorEmpty(AstIdentifier* const id,
                                                             AstLparen* const lp,
                                                             AstRparen* const rp,
                                                             File* const srcfile) :
        AstConstructorDeclarator(id, lp, NULL, rp, srcfile)
{
}

AstConstructorDeclaratorEmpty::~AstConstructorDeclaratorEmpty()
{
}

void AstConstructorDeclaratorEmpty::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstConstructorDeclaratorEmpty::type() const
{
    return "ConstructorDeclaratorEmpty";
}

AstConstructorDeclaratorParams::AstConstructorDeclaratorParams(AstIdentifier* const id,
                                                               AstLparen* const lp,
                                                               AstFormalParameterList* const fopali,
                                                               AstRparen* const rp,
                                                               File* const srcfile) :
        AstConstructorDeclarator(id, lp, fopali, rp, srcfile)
{
}

AstConstructorDeclaratorParams::~AstConstructorDeclaratorParams()
{
}

void AstConstructorDeclaratorParams::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstConstructorDeclaratorParams::type() const
{
    return "ConstructorDeclaratorParams";
}

AstContinue::AstContinue(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstContinue::~AstContinue()
{
}

string AstContinue::type() const
{
    return "CONTINUE";
}

void AstContinue::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstContinue::length() const
{
    return 8;
}

AstContinueStatement::AstContinueStatement(AstContinue* const c, 
                                           AstIdentifier* const i,
                                           AstSemicolon* const s,
                                           File* const srcfile) :
        AstNode(c, i, s, srcfile), continue_(c), identifier_(i), semicolon_(s)
{
}

AstContinueStatement::~AstContinueStatement()
{
}

AstContinue* AstContinueStatement::getContinue() const
{
    return continue_;
}

AstIdentifier* AstContinueStatement::getIdentifier() const
{
    return identifier_;
}

AstSemicolon* AstContinueStatement::getSemicolon() const
{
    return semicolon_;
}

void AstContinueStatement::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstContinueStatement::type() const
{
    return "ContinueStatement";
}

AstContinueStatementId::AstContinueStatementId(AstContinue* const c, 
                                               AstIdentifier* const i,
                                               AstSemicolon* const s,
                                               File* const srcfile) :
        AstContinueStatement(c, i, s, srcfile)
{
}

AstContinueStatementId::~AstContinueStatementId()
{
}

void AstContinueStatementId::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstContinueStatementId::type() const
{
    return "ContinueStatementId";
}

AstContinueStatementNoId::AstContinueStatementNoId(AstContinue* const c, 
                                                   AstSemicolon* const s,
                                                   File* const srcfile) :
        AstContinueStatement(c, NULL, s, srcfile)
{
}

AstContinueStatementNoId::~AstContinueStatementNoId()
{
}

void AstContinueStatementNoId::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstContinueStatementNoId::type() const
{
    return "ContinueStatementNoId";
}

AstCppComment::AstCppComment(char* start, char* end, File* const srcfile) :
        AstComment(start, end, srcfile)
{
}

AstCppComment::~AstCppComment()
{
}

string AstCppComment::type() const 
{
    return "C++ comment";
}

void AstCppComment::getProcessed(Processor* const p)
{
    p->process(this);
}

AstDefault::AstDefault(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstDefault::~AstDefault()
{
}

string AstDefault::type() const
{
    return "DEFAULT";
}

void AstDefault::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstDefault::length() const
{
    return 7;
}

AstDimension::AstDimension(AstLbracket* const item0, 
                     AstRbracket* const item1,
                     File* const srcfile) :
        AstNode(item0, item1, srcfile), lbracket_(item0), rbracket_(item1)
{
}

AstDimension::~AstDimension()
{
}

AstLbracket* AstDimension::getLbracket() const
{
    return lbracket_;
}

AstRbracket* AstDimension::getRbracket() const
{
    return rbracket_;
}

void AstDimension::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstDimension::type() const
{
    return "Dimension";
}

AstDimensionExpression::AstDimensionExpression(AstLbracket* const l, 
             AstItem* const e,
             AstRbracket* const r,
             File* const srcfile) :
        AstNode(l, e, r, srcfile), lbracket_(l), expression_(e), rbracket_(r)
{
}

AstDimensionExpression::~AstDimensionExpression()
{
}

AstLbracket* AstDimensionExpression::getLbracket() const
{
    return lbracket_;
}

AstItem* AstDimensionExpression::getExpression() const
{
    return expression_;
}

AstRbracket* AstDimensionExpression::getRbracket() const
{
    return rbracket_;
}

void AstDimensionExpression::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstDimensionExpression::type() const
{
    return "DimensionExpression";
}

AstDimensionExpressionList::AstDimensionExpressionList(AstDimensionExpression* const de, 
             File* const srcfile) :
        AstNode(de, srcfile)
{
    dimensionExpressions_.push_back(de);
}

AstDimensionExpressionList::~AstDimensionExpressionList()
{
}

AstDimensionExpression* AstDimensionExpressionList::getDimensionExpression(int index) const
{
    return dimensionExpressions_[index];
}

int AstDimensionExpressionList::getDimensionExpressionCount() const
{
    return dimensionExpressions_.size();
}

void AstDimensionExpressionList::add(AstDimensionExpression* const de)
{
    dimensionExpressions_.push_back(de);
    AstNode::append(de);
}

void AstDimensionExpressionList::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstDimensionExpressionList::type() const
{
    return "DimensionExpressionList";
}

AstDimensionList::AstDimensionList(AstDimension* const d, 
             File* const srcfile) :
        AstNode(d, srcfile)
{
    dimensions_.push_back(d);
}

AstDimensionList::~AstDimensionList()
{
}

AstDimension* AstDimensionList::getDimension(int index) const
{
    return dimensions_[index];
}

int AstDimensionList::getDimensionCount() const
{
    return dimensions_.size();
}

void AstDimensionList::add(AstDimension* const d)
{
    dimensions_.push_back(d);
    AstNode::append(d);
}

void AstDimensionList::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstDimensionList::type() const
{
    return "DimensionList";
}

AstDiv::AstDiv(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstDiv::~AstDiv()
{
}

string AstDiv::type() const
{
    return "DIV";
}

void AstDiv::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstDiv::length() const
{
    return 1;
}

AstDiveq::AstDiveq(char* position, File* const srcfile) :
        AstAssignmentOperator(position, srcfile)
{
}

AstDiveq::~AstDiveq()
{
}

string AstDiveq::type() const
{
    return "DIVEQ";
}

void AstDiveq::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstDiveq::length() const
{
    return 2;
}

AstDo::AstDo(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstDo::~AstDo()
{
}

string AstDo::type() const
{
    return "DO";
}

void AstDo::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstDo::length() const
{
    return 2;
}

AstDoStatement::AstDoStatement(AstDo* const d, 
                               AstItem* const st,
                               AstWhile* const w,
                               AstLparen* const l,
                               AstItem* const e,
                               AstRparen* const r,
                               AstSemicolon* const se,
                               File* const srcfile) :
        AstNode(d, st, w, l, e, r, se, srcfile),
     do_(d), 
     statement_(st), 
     while_(w), 
     lparen_(l), 
     expression_(e), 
     rparen_(r), 
     semicolon_(se)
{
}

AstDoStatement::~AstDoStatement()
{
}

AstDo* AstDoStatement::getDo() const
{
    return do_;
}

AstItem* AstDoStatement::getStatement() const
{
    return statement_;
}

AstWhile* AstDoStatement::getWhile() const
{
    return while_;
}

AstLparen* AstDoStatement::getLparen() const
{
    return lparen_;
}

AstItem* AstDoStatement::getExpression() const
{
    return expression_;
}

AstRparen* AstDoStatement::getRparen() const
{
    return rparen_;
}

AstSemicolon* AstDoStatement::getSemicolon() const
{
    return semicolon_;
}

void AstDoStatement::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstDoStatement::type() const
{
    return "DoStatement";
}

AstDot::AstDot(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstDot::~AstDot()
{
}

string AstDot::type() const
{
    return "DOT";
}

void AstDot::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstDot::length() const
{
    return 1;
}

AstDouble::AstDouble(char* position, File* const srcfile) :
        AstFloatingPointType(position, srcfile)
{
}

AstDouble::~AstDouble()
{
}

string AstDouble::type() const
{
    return "DOUBLE";
}

void AstDouble::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstDouble::length() const
{
    return 6;
}

AstElse::AstElse(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstElse::~AstElse()
{
}

string AstElse::type() const
{
    return "ELSE";
}

void AstElse::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstElse::length() const
{
    return 4;
}

AstEq::AstEq(char* position, File* const srcfile) :
        AstAssignmentOperator(position, srcfile)
{
}

AstEq::~AstEq()
{
}

string AstEq::type() const
{
    return "EQ";
}

void AstEq::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstEq::length() const
{
    return 1;
}

AstEqeq::AstEqeq(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstEqeq::~AstEqeq()
{
}

string AstEqeq::type() const
{
    return "EQEQ";
}

void AstEqeq::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstEqeq::length() const
{
    return 2;
}

AstEqualityExpression::AstEqualityExpression(AstItem* const lhex,
                                             AstItem* const rhex,
                                             File* const srcfile) :
        AstNode(lhex, rhex, srcfile), lhsExpression_(lhex), rhsExpression_(rhex)
{
}

AstEqualityExpression::~AstEqualityExpression()
{
}

AstItem* AstEqualityExpression::getLhsExpression() const
{
    return lhsExpression_;
}

AstItem* AstEqualityExpression::getRhsExpression() const
{
    return rhsExpression_;
}

AstEqualityExpressionEq::AstEqualityExpressionEq(AstItem* const lhex,
                                                 AstEqeq* const eq,
                                                 AstItem* const rhex,
                                                 File* const srcfile) :
        AstEqualityExpression(lhex, rhex, srcfile), eqeq_(eq)
{
    append(eq, lhex);
}

AstEqualityExpressionEq::~AstEqualityExpressionEq()
{
}

AstEqeq* AstEqualityExpressionEq::getEqeq() const
{
    return eqeq_;
}

void AstEqualityExpressionEq::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstEqualityExpressionEq::type() const
{
    return "EqualityExpressionEq";
}

AstEqualityExpressionNe::AstEqualityExpressionNe(AstItem* const lhex,
                                                 AstNoteq* const no,
                                                 AstItem* const rhex,
                                                 File* const srcfile) :
        AstEqualityExpression(lhex, rhex, srcfile), noteq_(no)
{
    append(no, lhex);
}

AstEqualityExpressionNe::~AstEqualityExpressionNe()
{
}

AstNoteq* AstEqualityExpressionNe::getNoteq() const
{
    return noteq_;
}

void AstEqualityExpressionNe::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstEqualityExpressionNe::type() const
{
    return "EqualityExpressionNe";
}

AstExplicitConstructorInvocation::AstExplicitConstructorInvocation(AstLparen* const lp,
                                                                   AstRparen* const rp,
                                                                   AstSemicolon* const se,
                                                                   File* const srcfile) :
        AstNode(lp, rp, se, srcfile), lparen_(lp), rparen_(rp), semicolon_(se)
{
}

AstExplicitConstructorInvocation::~AstExplicitConstructorInvocation()
{
}

AstLparen* AstExplicitConstructorInvocation::getLparen() const
{
    return lparen_;
}

AstRparen* AstExplicitConstructorInvocation::getRparen() const
{
    return rparen_;
}

AstSemicolon* AstExplicitConstructorInvocation::getSemicolon() const
{
    return semicolon_;
}

AstExplicitConstructorInvocationPrimarySuper::AstExplicitConstructorInvocationPrimarySuper(AstItem* const pr,
                                                                                     AstDot* const d,
                                                                                     AstSuper* const su,
                                                                                     AstLparen* const lp,
                                                                                     AstRparen* const rp,
                                                                                     AstSemicolon* const se,
                                                                                     File* const srcfile) :
        AstExplicitConstructorInvocation(lp, rp, se, srcfile), primary_(pr), dot_(d), super_(su)
{
    prepend(pr);
    append(d, pr);
    append(su, d);
}

AstExplicitConstructorInvocationPrimarySuper::~AstExplicitConstructorInvocationPrimarySuper()
{
}

AstItem* AstExplicitConstructorInvocationPrimarySuper::getPrimary() const
{
    return primary_;
}

AstDot* AstExplicitConstructorInvocationPrimarySuper::getDot() const
{
    return dot_;
}

AstSuper* AstExplicitConstructorInvocationPrimarySuper::getSuper() const
{
    return super_;
}

void AstExplicitConstructorInvocationPrimarySuper::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstExplicitConstructorInvocationPrimarySuper::type() const
{
    return "ExplicitConstructorInvocationPrimarySuper";
}

AstExplicitConstructorInvocationPrimarySuperExpressions::AstExplicitConstructorInvocationPrimarySuperExpressions(AstItem* const pr,
                                                                                                           AstDot* const d,
                                                                                                           AstSuper* const su,
                                                                                                           AstLparen* const lp,
                                                                                                           AstExpressionList* const exli,
                                                                                                           AstRparen* const rp,
                                                                                                           AstSemicolon* const se,
                                                                                                           File* const srcfile) :
        AstExplicitConstructorInvocation(lp, rp, se, srcfile), primary_(pr), dot_(d), super_(su), expressionList_(exli)
{
    prepend(pr);
    append(d, pr);
    append(su, d);
    append(exli, lp);
}

AstExplicitConstructorInvocationPrimarySuperExpressions::~AstExplicitConstructorInvocationPrimarySuperExpressions()
{
}

AstItem* AstExplicitConstructorInvocationPrimarySuperExpressions::getPrimary() const
{
    return primary_;
}

AstDot* AstExplicitConstructorInvocationPrimarySuperExpressions::getDot() const
{
    return dot_;
}

AstSuper* AstExplicitConstructorInvocationPrimarySuperExpressions::getSuper() const
{
    return super_;
}

AstExpressionList* AstExplicitConstructorInvocationPrimarySuperExpressions::getExpressionList() const
{
    return expressionList_;
}

void AstExplicitConstructorInvocationPrimarySuperExpressions::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstExplicitConstructorInvocationPrimarySuperExpressions::type() const
{
    return "ExplicitConstructorInvocationPrimarySuperExpressions";
}

AstExplicitConstructorInvocationPrimaryThis::AstExplicitConstructorInvocationPrimaryThis(AstItem* const pr,
                                                                                   AstDot* const d,
                                                                                   AstThis* const th,
                                                                                   AstLparen* const lp,
                                                                                   AstRparen* const rp,
                                                                                   AstSemicolon* const se,
                                                                                   File* const srcfile) :
        AstExplicitConstructorInvocation(lp, rp, se, srcfile), primary_(pr), dot_(d), this_(th)
{
    prepend(pr);
    append(d, pr);
    append(th, d);
}

AstExplicitConstructorInvocationPrimaryThis::~AstExplicitConstructorInvocationPrimaryThis()
{
}

AstItem* AstExplicitConstructorInvocationPrimaryThis::getPrimary() const
{
    return primary_;
}

AstDot* AstExplicitConstructorInvocationPrimaryThis::getDot() const
{
    return dot_;
}

AstThis* AstExplicitConstructorInvocationPrimaryThis::getThis() const
{
    return this_;
}

void AstExplicitConstructorInvocationPrimaryThis::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstExplicitConstructorInvocationPrimaryThis::type() const
{
    return "ExplicitConstructorInvocationPrimaryThis";
}

AstExplicitConstructorInvocationPrimaryThisExpressions::AstExplicitConstructorInvocationPrimaryThisExpressions(AstItem* const pr,
                                                                                                         AstDot* const d,
                                                                                                         AstThis* const th,
                                                                                                         AstLparen* const lp,
                                                                                                         AstExpressionList* const exli,
                                                                                                         AstRparen* const rp,
                                                                                                         AstSemicolon* const se,
                                                                                                         File* const srcfile) :
        AstExplicitConstructorInvocation(lp, rp, se, srcfile), primary_(pr), dot_(d), this_(th), expressionList_(exli)
{
    prepend(pr);
    append(d, pr);
    append(th, d);
    append(exli, lp);
}

AstExplicitConstructorInvocationPrimaryThisExpressions::~AstExplicitConstructorInvocationPrimaryThisExpressions()
{
}

AstItem* AstExplicitConstructorInvocationPrimaryThisExpressions::getPrimary() const
{
    return primary_;
}

AstDot* AstExplicitConstructorInvocationPrimaryThisExpressions::getDot() const
{
    return dot_;
}

AstThis* AstExplicitConstructorInvocationPrimaryThisExpressions::getThis() const
{
    return this_;
}

AstExpressionList* AstExplicitConstructorInvocationPrimaryThisExpressions::getExpressionList() const
{
    return expressionList_;
}

void AstExplicitConstructorInvocationPrimaryThisExpressions::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstExplicitConstructorInvocationPrimaryThisExpressions::type() const
{
    return "ExplicitConstructorInvocationPrimaryThisExpressions";
}

AstExplicitConstructorInvocationSuper::AstExplicitConstructorInvocationSuper(AstSuper* const su,
                                                                       AstLparen* const lp,
                                                                       AstRparen* const rp,
                                                                       AstSemicolon* const se,
                                                                       File* const srcfile) :
        AstExplicitConstructorInvocation(lp, rp, se, srcfile), super_(su)
{
    prepend(su);
}

AstExplicitConstructorInvocationSuper::~AstExplicitConstructorInvocationSuper()
{
}

AstSuper* AstExplicitConstructorInvocationSuper::getSuper() const
{
    return super_;
}

void AstExplicitConstructorInvocationSuper::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstExplicitConstructorInvocationSuper::type() const
{
    return "ExplicitConstructorInvocationSuper";
}

AstExplicitConstructorInvocationSuperExpressions::AstExplicitConstructorInvocationSuperExpressions(AstSuper* const su,
                                                                                             AstLparen* const lp,
                                                                                             AstExpressionList* const exli,
                                                                                             AstRparen* const rp,
                                                                                             AstSemicolon* const se,
                                                                                             File* const srcfile) :
        AstExplicitConstructorInvocation(lp, rp, se, srcfile), super_(su), expressionList_(exli)
{
    prepend(su);
    append(exli, lp);
}

AstExplicitConstructorInvocationSuperExpressions::~AstExplicitConstructorInvocationSuperExpressions()
{
}

AstSuper* AstExplicitConstructorInvocationSuperExpressions::getSuper() const
{
    return super_;
}

AstExpressionList* AstExplicitConstructorInvocationSuperExpressions::getExpressionList() const
{
    return expressionList_;
}

void AstExplicitConstructorInvocationSuperExpressions::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstExplicitConstructorInvocationSuperExpressions::type() const
{
    return "ExplicitConstructorInvocationSuperExpressions";
}

AstExplicitConstructorInvocationThis::AstExplicitConstructorInvocationThis(AstThis* const th,
                                                                     AstLparen* const lp,
                                                                     AstRparen* const rp,
                                                                     AstSemicolon* const se,
                                                                     File* const srcfile) :
        AstExplicitConstructorInvocation(lp, rp, se, srcfile), this_(th)
{
    prepend(th);
}

AstExplicitConstructorInvocationThis::~AstExplicitConstructorInvocationThis()
{
}

AstThis* AstExplicitConstructorInvocationThis::getThis() const
{
    return this_;
}

void AstExplicitConstructorInvocationThis::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstExplicitConstructorInvocationThis::type() const
{
    return "ExplicitConstructorInvocationThis";
}

AstExplicitConstructorInvocationThisExpressions::AstExplicitConstructorInvocationThisExpressions(AstThis* const th,
                                                                                           AstLparen* const lp,
                                                                                           AstExpressionList* const exli,
                                                                                           AstRparen* const rp,
                                                                                           AstSemicolon* const se,
                                                                                           File* const srcfile) :
        AstExplicitConstructorInvocation(lp, rp, se, srcfile), this_(th), expressionList_(exli)
{
    prepend(th);
    append(exli, lp);
}

AstExplicitConstructorInvocationThisExpressions::~AstExplicitConstructorInvocationThisExpressions()
{
}

AstThis* AstExplicitConstructorInvocationThisExpressions::getThis() const
{
    return this_;
}

AstExpressionList* AstExplicitConstructorInvocationThisExpressions::getExpressionList() const
{
    return expressionList_;
}

void AstExplicitConstructorInvocationThisExpressions::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstExplicitConstructorInvocationThisExpressions::type() const
{
    return "ExplicitConstructorInvocationThisExpressions";
}

AstExpressionList::AstExpressionList(AstItem* const e, 
             File* const srcfile) :
        AstNode(e, srcfile)
{
    expressions_.push_back(e);
}

AstExpressionList::~AstExpressionList()
{
}

AstItem* AstExpressionList::getExpression(int index) const
{
    return expressions_[index];
}

int AstExpressionList::getExpressionCount() const
{
    return expressions_.size();
}

void AstExpressionList::add(AstItem* const e)
{
    expressions_.push_back(e);
    AstNode::append(e);
}

AstComma* AstExpressionList::getComma(int index) const
{
    return commas_[index];
}

int AstExpressionList::getCommaCount() const
{
    return commas_.size();
}

void AstExpressionList::add(AstComma* const c)
{
    commas_.push_back(c);
    AstNode::append(c);
}

void AstExpressionList::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstExpressionList::type() const
{
    return "ExpressionList";
}

AstExpressionStatement::AstExpressionStatement(AstItem* const e, 
             AstSemicolon* const s,
             File* const srcfile /* = NULL */) :
        AstNode(e, s, srcfile), expression_(e), semicolon_(s)
{
}

AstExpressionStatement::~AstExpressionStatement()
{
}

AstItem* AstExpressionStatement::getExpression() const
{
    return expression_;
}

AstSemicolon* AstExpressionStatement::getSemicolon() const
{
    return semicolon_;
}

void AstExpressionStatement::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstExpressionStatement::type() const
{
    return "ExpressionStatement";
}

AstExtends::AstExtends(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstExtends::~AstExtends()
{
}

string AstExtends::type() const
{
    return "EXTENDS";
}

void AstExtends::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstExtends::length() const
{
    return 7;
}

AstExtendsInterfaceList::AstExtendsInterfaceList(AstExtends* const e, 
             AstNameList* const nl,
             File* const srcfile) :
        AstNode(e, nl, srcfile), extends_(e), nameList_(nl)
{
}

AstExtendsInterfaceList::~AstExtendsInterfaceList()
{
}

AstExtends* AstExtendsInterfaceList::getExtends() const
{
    return extends_;
}

AstNameList* AstExtendsInterfaceList::getNameList() const
{
    return nameList_;
}

void AstExtendsInterfaceList::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstExtendsInterfaceList::type() const
{
    return "ExtendsInterfaceList";
}

AstFalseLiteral::AstFalseLiteral(char* position, File* const srcfile) :
        AstBooleanLiteral("false", position, srcfile)
{
}

AstFalseLiteral::~AstFalseLiteral()
{
}

string AstFalseLiteral::type() const
{
    return "FALSE_LITERAL";
}

void AstFalseLiteral::getProcessed(Processor* const proc)
{
    proc->process(this);
}

AstFieldAccess::AstFieldAccess(AstIdentifier* const id,
                               File* const srcfile) :
        AstNode(id, srcfile), identifier_(id)
{
}

AstFieldAccess::~AstFieldAccess()
{
}

AstIdentifier* AstFieldAccess::getIdentifier() const
{
    return identifier_;
}

AstFieldAccessNameSuper::AstFieldAccessNameSuper(AstName* const na,
                                                 AstDot* const d0,
                                                 AstSuper* const su,
                                                 AstDot* const d1,
                                                 AstIdentifier* const id,
                                                 File* const srcfile) :
        AstFieldAccess(id, srcfile), name_(na), firstDot_(d0), super_(su), secondDot_(d1)
{
    prepend(na);
    append(d0, na);
    append(su, d0);
    append(d1, su);
}

AstFieldAccessNameSuper::~AstFieldAccessNameSuper()
{
}

AstName* AstFieldAccessNameSuper::getName() const
{
    return name_;
}

AstDot* AstFieldAccessNameSuper::getFirstDot() const
{
    return firstDot_;
}

AstSuper* AstFieldAccessNameSuper::getSuper() const
{
    return super_;
}

AstDot* AstFieldAccessNameSuper::getSecondDot() const
{
    return secondDot_;
}

void AstFieldAccessNameSuper::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstFieldAccessNameSuper::type() const
{
    return "FieldAccessNameSuper";
}

AstFieldAccessPrimary::AstFieldAccessPrimary(AstItem* const pr,
                                       AstDot* const d,
                                       AstIdentifier* const id,
                                       File* const srcfile) :
        AstFieldAccess(id, srcfile), primary_(pr), dot_(d)
{
    prepend(pr);
    append(d, pr);
}

AstFieldAccessPrimary::~AstFieldAccessPrimary()
{
}

AstItem* AstFieldAccessPrimary::getPrimary() const
{
    return primary_;
}

AstDot* AstFieldAccessPrimary::getDot() const
{
    return dot_;
}

void AstFieldAccessPrimary::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstFieldAccessPrimary::type() const
{
    return "FieldAccessPrimary";
}

AstFieldAccessSuper::AstFieldAccessSuper(AstSuper* const su,
                                   AstDot* const d,
                                   AstIdentifier* const id,
                                   File* const srcfile) :
        AstFieldAccess(id, srcfile), super_(su), dot_(d)
{
    prepend(su);
    append(d, su);
}

AstFieldAccessSuper::~AstFieldAccessSuper()
{
}

AstSuper* AstFieldAccessSuper::getSuper() const
{
    return super_;
}

AstDot* AstFieldAccessSuper::getDot() const
{
    return dot_;
}

void AstFieldAccessSuper::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstFieldAccessSuper::type() const
{
    return "FieldAccessSuper";
}

AstFieldDeclaration::AstFieldDeclaration(AstModifierList* const moli,
                                         AstItem* const ty,
                                         AstVariableDeclaratorList* const vadeli,
                                         AstSemicolon* const se,
                                         File* const srcfile) :
        AstNode(moli, ty, vadeli, se, srcfile), 
     modifierList_(moli), 
     type_(ty), 
     variableDeclaratorList_(vadeli), 
     semicolon_(se)
{
}

AstFieldDeclaration::~AstFieldDeclaration()
{
}

AstModifierList* AstFieldDeclaration::getModifierList() const
{
    return modifierList_;
}

AstItem* AstFieldDeclaration::getType() const
{
    return type_;
}

AstVariableDeclaratorList* AstFieldDeclaration::getVariableDeclaratorList() const
{
    return variableDeclaratorList_;
}

AstSemicolon* AstFieldDeclaration::getSemicolon() const
{
    return semicolon_;
}

AstFieldDeclarationMods::AstFieldDeclarationMods(AstModifierList* const moli,
                                                 AstItem* const ty,
                                                 AstVariableDeclaratorList* const vadeli,
                                                 AstSemicolon* const se,
                                                 File* const srcfile) :
        AstFieldDeclaration(moli, ty, vadeli, se, srcfile)
{
}

AstFieldDeclarationMods::~AstFieldDeclarationMods()
{
}

void AstFieldDeclarationMods::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstFieldDeclarationMods::type() const
{
    return "FieldDeclarationMods";
}

AstFieldDeclarationNoMods::AstFieldDeclarationNoMods(AstItem* const ty,
                                                     AstVariableDeclaratorList* const vadeli,
                                                     AstSemicolon* const se,
                                                     File* const srcfile) :
        AstFieldDeclaration(NULL, ty, vadeli, se, srcfile)
{
}

AstFieldDeclarationNoMods::~AstFieldDeclarationNoMods()
{
}

void AstFieldDeclarationNoMods::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstFieldDeclarationNoMods::type() const
{
    return "FieldDeclarationNoMods";
}

AstFinal::AstFinal(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstFinal::~AstFinal()
{
}

string AstFinal::type() const
{
    return "FINAL";
}

void AstFinal::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstFinal::length() const
{
    return 5;
}

AstFinallyBlock::AstFinallyBlock(AstFinally* const f, 
             AstBlock* const b,
             File* const srcfile) :
        AstNode(f, b, srcfile), finally_(f), block_(b)
{
}

AstFinallyBlock::~AstFinallyBlock()
{
}

AstFinally* AstFinallyBlock::getFinally() const
{
    return finally_;
}

AstBlock* AstFinallyBlock::getBlock() const
{
    return block_;
}

void AstFinallyBlock::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstFinallyBlock::type() const
{
    return "FinallyBlock";
}

AstFinally::AstFinally(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstFinally::~AstFinally()
{
}

string AstFinally::type() const
{
    return "FINALLY";
}

void AstFinally::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstFinally::length() const
{
    return 7;
}

AstFloat::AstFloat(char* position, File* const srcfile) :
        AstFloatingPointType(position, srcfile)
{
}

AstFloat::~AstFloat()
{
}

string AstFloat::type() const
{
    return "FLOAT";
}

void AstFloat::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstFloat::length() const
{
    return 5;
}

AstFloatingPointLiteral::AstFloatingPointLiteral(const string& text, char* position, File* const srcfile) :
        AstLiteral(text, position, srcfile)
{
}

AstFloatingPointLiteral::~AstFloatingPointLiteral()
{
}

string AstFloatingPointLiteral::type() const
{
    return "FLOATING_POINT_LITERAL";
}

void AstFloatingPointLiteral::getProcessed(Processor* const proc)
{
    proc->process(this);
}

AstFloatingPointType::AstFloatingPointType(char* position, File* const srcfile) :
        AstNumericType(position, srcfile)
{
}

AstFloatingPointType::~AstFloatingPointType()
{
}

AstFor::AstFor(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstFor::~AstFor()
{
}

string AstFor::type() const
{
    return "FOR";
}

void AstFor::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstFor::length() const
{
    return 3;
}

AstFormalParameter::AstFormalParameter(AstFinal* const f, 
                                       AstItem* const t, 
                                       AstVariableDeclaratorId* const vdi,
                                       File* const srcfile) :
        AstNode(f, t, vdi, srcfile), final_(f), type_(t), variableDeclaratorId_(vdi)
{
}

AstFormalParameter::~AstFormalParameter()
{
}

AstFinal* AstFormalParameter::getFinal() const
{
    return final_;
}

AstItem* AstFormalParameter::getType() const
{
    return type_;
}

AstVariableDeclaratorId* AstFormalParameter::getVariableDeclaratorId() const
{
    return variableDeclaratorId_;
}

string AstFormalParameter::name() const
{
    return variableDeclaratorId_->name();
}

string AstFormalParameter::dataType() const
{
    return getType()->text();
}

AstFormalParameterFinal::AstFormalParameterFinal(AstFinal* const f, 
                                                 AstItem* const t,
                                                 AstVariableDeclaratorId* const vdi,
                                                 File* const srcfile) :
        AstFormalParameter(f, t, vdi, srcfile)
{
}

AstFormalParameterFinal::~AstFormalParameterFinal()
{
}

void AstFormalParameterFinal::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstFormalParameterFinal::type() const
{
    return "FormalParameterFinal";
}

AstFormalParameterList::AstFormalParameterList(AstFormalParameter* const fp, 
                                         File* const srcfile) :
        AstNode(fp, srcfile)
{
    formalParameters_.push_back(fp);
}

AstFormalParameterList::~AstFormalParameterList()
{
}

AstFormalParameter* AstFormalParameterList::getFormalParameter(int index) const
{
    return formalParameters_[index];
}

int AstFormalParameterList::getFormalParameterCount() const
{
    return formalParameters_.size();
}

void AstFormalParameterList::add(AstFormalParameter* const fp)
{
    formalParameters_.push_back(fp);
    AstNode::append(fp);
}

AstComma* AstFormalParameterList::getComma(int index) const
{
    return commas_[index];
}

int AstFormalParameterList::getCommaCount() const
{
    return commas_.size();
}

void AstFormalParameterList::add(AstComma* const c)
{
    commas_.push_back(c);
    AstNode::append(c);
}

void AstFormalParameterList::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstFormalParameterList::type() const
{
    return "FormalParameterList";
}

AstFormalParameterNonFinal::AstFormalParameterNonFinal(AstItem* const t,
                                                       AstVariableDeclaratorId* const vdi,
                                                       File* const srcfile) :
        AstFormalParameter(NULL, t, vdi, srcfile)
{
}

AstFormalParameterNonFinal::~AstFormalParameterNonFinal()
{
}

void AstFormalParameterNonFinal::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstFormalParameterNonFinal::type() const
{
    return "FormalParameterNonFinal";
}

AstForStatement::AstForStatement(AstFor* const fo,
                                 AstLparen* const lp,
                                 AstSemicolon* const se0,
                                 AstSemicolon* const se1,
                                 AstRparen* const rp,
                                 AstItem* const st,
                                 File* const srcfile) :
        AstNode(fo, lp, se0, se1, rp, st, srcfile),
     for_(fo),
     lparen_(lp),
     firstSemicolon_(se0),
     secondSemicolon_(se1),
     rparen_(rp),
     statement_(st)
{
}

AstForStatement::~AstForStatement()
{
}

AstFor* AstForStatement::getFor() const
{
    return for_;
}

AstLparen* AstForStatement::getLparen() const
{
    return lparen_;
}

AstSemicolon* AstForStatement::getFirstSemicolon() const
{
    return firstSemicolon_;
}

AstSemicolon* AstForStatement::getSecondSemicolon() const
{
    return secondSemicolon_;
}

AstRparen* AstForStatement::getRparen() const
{
    return rparen_;
}

AstItem* AstForStatement::getStatement() const
{
    return statement_;
}

AstForStatementNoInitConditionNoUpdate::AstForStatementNoInitConditionNoUpdate(AstFor* const fo,
                                                                               AstLparen* const lp,
                                                                               AstSemicolon* const se0,
                                                                               AstItem* const ex,
                                                                               AstSemicolon* const se1,
                                                                               AstRparen* const rp,
                                                                               AstItem* const st,
                                                                               File* const srcfile) :
        AstForStatement(fo, lp, se0, se1, rp, st, srcfile), expression_(ex)
{
    append(ex, se0);
}

AstForStatementNoInitConditionNoUpdate::~AstForStatementNoInitConditionNoUpdate()
{
}

AstItem* AstForStatementNoInitConditionNoUpdate::getExpression() const
{
    return expression_;
}

void AstForStatementNoInitConditionNoUpdate::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstForStatementNoInitConditionNoUpdate::type() const
{
    return "ForStatementNoInitConditionNoUpdate";
}

AstForStatementNoInitConditionUpdate::AstForStatementNoInitConditionUpdate(AstFor* const fo,
                                                                           AstLparen* const lp,
                                                                           AstSemicolon* const se0,
                                                                           AstItem* const ex,
                                                                           AstSemicolon* const se1,
                                                                           AstStatementExpressionList* const cstexli,
                                                                           AstRparen* const rp,
                                                                           AstItem* const st,
                                                                           File* const srcfile) :
        AstForStatement(fo, lp, se0, se1, rp, st, srcfile), expression_(ex), conditionStatementExpressionList_(cstexli)
{
    append(ex, se0);
    append(cstexli, se1);
}

AstForStatementNoInitConditionUpdate::~AstForStatementNoInitConditionUpdate()
{
}

AstItem* AstForStatementNoInitConditionUpdate::getExpression() const
{
    return expression_;
}

AstStatementExpressionList* AstForStatementNoInitConditionUpdate::getConditionStatementExpressionList() const
{
    return conditionStatementExpressionList_;
}

void AstForStatementNoInitConditionUpdate::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstForStatementNoInitConditionUpdate::type() const
{
    return "ForStatementNoInitConditionUpdate";
}

AstForStatementNoInitNoConditionNoUpdate::AstForStatementNoInitNoConditionNoUpdate(AstFor* const fo,
                                                                                   AstLparen* const lp,
                                                                                   AstSemicolon* const se0,
                                                                                   AstSemicolon* const se1,
                                                                                   AstRparen* const rp,
                                                                                   AstItem* const st,
                                                                                   File* const srcfile) :
        AstForStatement(fo, lp, se0, se1, rp, st, srcfile)
{
}

AstForStatementNoInitNoConditionNoUpdate::~AstForStatementNoInitNoConditionNoUpdate()
{
}

void AstForStatementNoInitNoConditionNoUpdate::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstForStatementNoInitNoConditionNoUpdate::type() const
{
    return "ForStatementNoInitNoConditionNoUpdate";
}

AstForStatementNoInitNoConditionUpdate::AstForStatementNoInitNoConditionUpdate(AstFor* const fo,
                                                                               AstLparen* const lp,
                                                                               AstSemicolon* const se0,
                                                                               AstSemicolon* const se1,
                                                                               AstStatementExpressionList* const cstexli,
                                                                               AstRparen* const rp,
                                                                               AstItem* const st,
                                                                               File* const srcfile) :
        AstForStatement(fo, lp, se0, se1, rp, st, srcfile), conditionStatementExpressionList_(cstexli)
{
    append(cstexli, se1);
}

AstForStatementNoInitNoConditionUpdate::~AstForStatementNoInitNoConditionUpdate()
{
}

AstStatementExpressionList* AstForStatementNoInitNoConditionUpdate::getConditionStatementExpressionList() const
{
    return conditionStatementExpressionList_;
}

void AstForStatementNoInitNoConditionUpdate::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstForStatementNoInitNoConditionUpdate::type() const
{
    return "ForStatementNoInitNoConditionUpdate";
}

AstForStatementStmtsConditionNoUpdate::AstForStatementStmtsConditionNoUpdate(AstFor* const fo,
                                                                             AstLparen* const lp,
                                                                             AstStatementExpressionList* const istexli,
                                                                             AstSemicolon* const se0,
                                                                             AstItem* const ex,
                                                                             AstSemicolon* const se1,
                                                                             AstRparen* const rp,
                                                                             AstItem* const st,
                                                                             File* const srcfile) :
        AstForStatement(fo, lp, se0, se1, rp, st, srcfile), initStatementExpressionList_(istexli), expression_(ex)
{
    append(istexli, lp);
    append(ex, se0);
}

AstForStatementStmtsConditionNoUpdate::~AstForStatementStmtsConditionNoUpdate()
{
}

AstStatementExpressionList* AstForStatementStmtsConditionNoUpdate::getInitStatementExpressionList() const
{
    return initStatementExpressionList_;
}

AstItem* AstForStatementStmtsConditionNoUpdate::getExpression() const
{
    return expression_;
}

void AstForStatementStmtsConditionNoUpdate::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstForStatementStmtsConditionNoUpdate::type() const
{
    return "ForStatementStmtsConditionNoUpdate";
}

AstForStatementStmtsConditionUpdate::AstForStatementStmtsConditionUpdate(AstFor* const fo,
                                                                         AstLparen* const lp,
                                                                         AstStatementExpressionList* const istexli,
                                                                         AstSemicolon* const se0,
                                                                         AstItem* const ex,
                                                                         AstSemicolon* const se1,
                                                                         AstStatementExpressionList* const cstexli,
                                                                         AstRparen* const rp,
                                                                         AstItem* const st,
                                                                         File* const srcfile) :
        AstForStatement(fo, lp, se0, se1, rp, st, srcfile), initStatementExpressionList_(istexli), expression_(ex), conditionStatementExpressionList_(cstexli)
{
    append(istexli, lp);
    append(ex, se0);
    append(cstexli, se1);
}

AstForStatementStmtsConditionUpdate::~AstForStatementStmtsConditionUpdate()
{
}

AstStatementExpressionList* AstForStatementStmtsConditionUpdate::getInitStatementExpressionList() const
{
    return initStatementExpressionList_;
}

AstItem* AstForStatementStmtsConditionUpdate::getExpression() const
{
    return expression_;
}

AstStatementExpressionList* AstForStatementStmtsConditionUpdate::getConditionStatementExpressionList() const
{
    return conditionStatementExpressionList_;
}

void AstForStatementStmtsConditionUpdate::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstForStatementStmtsConditionUpdate::type() const
{
    return "ForStatementStmtsConditionUpdate";
}

AstForStatementStmtsNoConditionNoUpdate::AstForStatementStmtsNoConditionNoUpdate(AstFor* const fo,
                                                                                 AstLparen* const lp,
                                                                                 AstStatementExpressionList* const istexli,
                                                                                 AstSemicolon* const se0,
                                                                                 AstSemicolon* const se1,
                                                                                 AstRparen* const rp,
                                                                                 AstItem* const st,
                                                                                 File* const srcfile) :
        AstForStatement(fo, lp, se0, se1, rp, st, srcfile), initStatementExpressionList_(istexli)
{
    append(istexli, lp);
}

AstForStatementStmtsNoConditionNoUpdate::~AstForStatementStmtsNoConditionNoUpdate()
{
}

AstStatementExpressionList* AstForStatementStmtsNoConditionNoUpdate::getInitStatementExpressionList() const
{
    return initStatementExpressionList_;
}

void AstForStatementStmtsNoConditionNoUpdate::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstForStatementStmtsNoConditionNoUpdate::type() const
{
    return "ForStatementStmtsNoConditionNoUpdate";
}

AstForStatementStmtsNoConditionUpdate::AstForStatementStmtsNoConditionUpdate(AstFor* const fo,
                                                                             AstLparen* const lp,
                                                                             AstStatementExpressionList* const istexli,
                                                                             AstSemicolon* const se0,
                                                                             AstSemicolon* const se1,
                                                                             AstStatementExpressionList* const cstexli,
                                                                             AstRparen* const rp,
                                                                             AstItem* const st,
                                                                             File* const srcfile) :
        AstForStatement(fo, lp, se0, se1, rp, st, srcfile), initStatementExpressionList_(istexli), conditionStatementExpressionList_(cstexli)
{
    append(istexli, lp);
    append(cstexli, se1);
}

AstForStatementStmtsNoConditionUpdate::~AstForStatementStmtsNoConditionUpdate()
{
}

AstStatementExpressionList* AstForStatementStmtsNoConditionUpdate::getInitStatementExpressionList() const
{
    return initStatementExpressionList_;
}

AstStatementExpressionList* AstForStatementStmtsNoConditionUpdate::getConditionStatementExpressionList() const
{
    return conditionStatementExpressionList_;
}

void AstForStatementStmtsNoConditionUpdate::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstForStatementStmtsNoConditionUpdate::type() const
{
    return "ForStatementStmtsNoConditionUpdate";
}

AstForStatementVarsConditionNoUpdate::AstForStatementVarsConditionNoUpdate(AstFor* const fo,
                                                                           AstLparen* const lp,
                                                                           AstLocalVariableDeclaration* const lovade,
                                                                           AstSemicolon* const se0,
                                                                           AstItem* const ex,
                                                                           AstSemicolon* const se1,
                                                                           AstRparen* const rp,
                                                                           AstItem* const st,
                                                                           File* const srcfile) :
        AstForStatement(fo, lp, se0, se1, rp, st, srcfile), localVariableDeclaration_(lovade), expression_(ex)
{
    append(lovade, lp);
    append(ex, se0);
}

AstForStatementVarsConditionNoUpdate::~AstForStatementVarsConditionNoUpdate()
{
}

AstLocalVariableDeclaration* AstForStatementVarsConditionNoUpdate::getLocalVariableDeclaration() const
{
    return localVariableDeclaration_;
}

AstItem* AstForStatementVarsConditionNoUpdate::getExpression() const
{
    return expression_;
}

void AstForStatementVarsConditionNoUpdate::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstForStatementVarsConditionNoUpdate::type() const
{
    return "ForStatementVarsConditionNoUpdate";
}

AstForStatementVarsConditionUpdate::AstForStatementVarsConditionUpdate(AstFor* const fo,
                                                                       AstLparen* const lp,
                                                                       AstLocalVariableDeclaration* const lovade,
                                                                       AstSemicolon* const se0,
                                                                       AstItem* const ex,
                                                                       AstSemicolon* const se1,
                                                                       AstStatementExpressionList* const cstexli,
                                                                       AstRparen* const rp,
                                                                       AstItem* const st,
                                                                       File* const srcfile) :
        AstForStatement(fo, lp, se0, se1, rp, st, srcfile),
     localVariableDeclaration_(lovade),
     expression_(ex),
     conditionStatementExpressionList_(cstexli)
{
    append(lovade, lp);
    append(ex, se0);
    append(cstexli, se1);
}

AstForStatementVarsConditionUpdate::~AstForStatementVarsConditionUpdate()
{
}

AstLocalVariableDeclaration* AstForStatementVarsConditionUpdate::getLocalVariableDeclaration() const
{
    return localVariableDeclaration_;
}

AstItem* AstForStatementVarsConditionUpdate::getExpression() const
{
    return expression_;
}

AstStatementExpressionList* AstForStatementVarsConditionUpdate::getConditionStatementExpressionList() const
{
    return conditionStatementExpressionList_;
}

void AstForStatementVarsConditionUpdate::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstForStatementVarsConditionUpdate::type() const
{
    return "ForStatementVarsConditionUpdate";
}

AstForStatementVarsNoConditionNoUpdate::AstForStatementVarsNoConditionNoUpdate(AstFor* const fo,
                                                                               AstLparen* const lp,
                                                                               AstLocalVariableDeclaration* const lovade,
                                                                               AstSemicolon* const se0,
                                                                               AstSemicolon* const se1,
                                                                               AstRparen* const rp,
                                                                               AstItem* const st,
                                                                               File* const srcfile) :
        AstForStatement(fo, lp, se0, se1, rp, st, srcfile), localVariableDeclaration_(lovade)
{
    append(lovade, lp);
}

AstForStatementVarsNoConditionNoUpdate::~AstForStatementVarsNoConditionNoUpdate()
{
}

AstLocalVariableDeclaration* AstForStatementVarsNoConditionNoUpdate::getLocalVariableDeclaration() const
{
    return localVariableDeclaration_;
}

void AstForStatementVarsNoConditionNoUpdate::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstForStatementVarsNoConditionNoUpdate::type() const
{
    return "ForStatementVarsNoConditionNoUpdate";
}

AstForStatementVarsNoConditionUpdate::AstForStatementVarsNoConditionUpdate(AstFor* const fo,
                                                                     AstLparen* const lp,
                                                                     AstLocalVariableDeclaration* const lovade,
                                                                     AstSemicolon* const se0,
                                                                     AstSemicolon* const se1,
                                                                     AstStatementExpressionList* const cstexli,
                                                                     AstRparen* const rp,
                                                                     AstItem* const st,
                                                                     File* const srcfile) :
        AstForStatement(fo, lp, se0, se1, rp, st, srcfile), localVariableDeclaration_(lovade), conditionStatementExpressionList_(cstexli)
{
    append(lovade, lp);
    append(cstexli, se1);
}

AstForStatementVarsNoConditionUpdate::~AstForStatementVarsNoConditionUpdate()
{
}

AstLocalVariableDeclaration* AstForStatementVarsNoConditionUpdate::getLocalVariableDeclaration() const
{
    return localVariableDeclaration_;
}

AstStatementExpressionList* AstForStatementVarsNoConditionUpdate::getConditionStatementExpressionList() const
{
    return conditionStatementExpressionList_;
}

void AstForStatementVarsNoConditionUpdate::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstForStatementVarsNoConditionUpdate::type() const
{
    return "ForStatementVarsNoConditionUpdate";
}

AstGoto::AstGoto(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstGoto::~AstGoto()
{
}

string AstGoto::type() const
{
    return "GOTO";
}

void AstGoto::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstGoto::length() const
{
    return 4;
}

AstGt::AstGt(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstGt::~AstGt()
{
}

string AstGt::type() const
{
    return "GT";
}

void AstGt::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstGt::length() const
{
    return 1;
}

AstGteq::AstGteq(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstGteq::~AstGteq()
{
}

string AstGteq::type() const
{
    return "GTEQ";
}

void AstGteq::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstGteq::length() const
{
    return 2;
}

AstIdentifier::AstIdentifier(const string& text, char* position, File* const srcfile) :
        AstValue(text, position, srcfile)
{
}

AstIdentifier::~AstIdentifier()
{
}

string AstIdentifier::type() const
{
    return "IDENTIFIER";
}

void AstIdentifier::getProcessed(Processor* const proc)
{
    proc->process(this);
}

AstIf::AstIf(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstIf::~AstIf()
{
}

string AstIf::type() const
{
    return "IF";
}

void AstIf::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstIf::length() const
{
    return 2;
}

AstIfThenElseStatement::AstIfThenElseStatement(AstIf* const i, 
                                               AstLparen* const l,
                                               AstItem* const ex,
                                               AstRparen* const r,
                                               AstItem* const s0,
                                               AstElse* const el,
                                               AstItem* const s1,
                                               File* const srcfile) :
        AstNode(i, l, ex, r, s0, el, s1, srcfile),
     if_(i), 
     lparen_(l), 
     expression_(ex), 
     rparen_(r), 
     firstStatement_(s0), 
     else_(el), 
     secondStatement_(s1)
{
}

AstIfThenElseStatement::~AstIfThenElseStatement()
{
}

AstIf* AstIfThenElseStatement::getIf() const
{
    return if_;
}

AstLparen* AstIfThenElseStatement::getLparen() const
{
    return lparen_;
}

AstItem* AstIfThenElseStatement::getExpression() const
{
    return expression_;
}

AstRparen* AstIfThenElseStatement::getRparen() const
{
    return rparen_;
}

AstItem* AstIfThenElseStatement::getFirstStatement() const
{
    return firstStatement_;
}

AstElse* AstIfThenElseStatement::getElse() const
{
    return else_;
}

AstItem* AstIfThenElseStatement::getSecondStatement() const
{
    return secondStatement_;
}

void AstIfThenElseStatement::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstIfThenElseStatement::type() const
{
    return "IfThenElseStatement";
}

AstIfThenStatement::AstIfThenStatement(AstIf* const i, 
                                       AstLparen* const l,
                                       AstItem* const e,
                                       AstRparen* const r,
                                       AstItem* const s,
                                       File* const srcfile) :
        AstNode(i, l, e, r, s, srcfile),
     if_(i), 
     lparen_(l), 
     expression_(e), 
     rparen_(r), 
     statement_(s)
{
}

AstIfThenStatement::~AstIfThenStatement()
{
}

AstIf* AstIfThenStatement::getIf() const
{
    return if_;
}

AstLparen* AstIfThenStatement::getLparen() const
{
    return lparen_;
}

AstItem* AstIfThenStatement::getExpression() const
{
    return expression_;
}

AstRparen* AstIfThenStatement::getRparen() const
{
    return rparen_;
}

AstItem* AstIfThenStatement::getStatement() const
{
    return statement_;
}

void AstIfThenStatement::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstIfThenStatement::type() const
{
    return "IfThenStatement";
}

AstImplements::AstImplements(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstImplements::~AstImplements()
{
}

string AstImplements::type() const
{
    return "IMPLEMENTS";
}

void AstImplements::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstImplements::length() const
{
    return 10;
}

AstImplementsInterfaceList::AstImplementsInterfaceList(AstImplements* const i, 
             AstNameList* const nl,
             File* const srcfile) :
        AstNode(i, nl, srcfile), implements_(i), nameList_(nl)
{
}

AstImplementsInterfaceList::~AstImplementsInterfaceList()
{
}

AstImplements* AstImplementsInterfaceList::getImplements() const
{
    return implements_;
}

AstNameList* AstImplementsInterfaceList::getNameList() const
{
    return nameList_;
}

void AstImplementsInterfaceList::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstImplementsInterfaceList::type() const
{
    return "ImplementsInterfaceList";
}

AstImport::AstImport(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstImport::~AstImport()
{
}

string AstImport::type() const
{
    return "IMPORT";
}

void AstImport::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstImport::length() const
{
    return 6;
}

AstImportDeclaration::AstImportDeclaration(AstImport* const i, 
                                           AstName* const n,
                                           AstSemicolon* const s,
                                           File* const srcfile) :
        AstNode(i, n, s, srcfile), import_(i), name_(n), semicolon_(s)
{
}

AstImportDeclaration::~AstImportDeclaration()
{
}

AstImport* AstImportDeclaration::getImport() const
{
    return import_;
}

AstName* AstImportDeclaration::getName() const
{
    return name_;
}

AstSemicolon* AstImportDeclaration::getSemicolon() const
{
    return semicolon_;
}

string AstImportDeclaration::importee() const
{
    return getName()->fullName();
}

AstImportDeclarationList::AstImportDeclarationList(AstImportDeclaration* const id, 
                                             File* const srcfile) :
        AstNode(id, srcfile)
{
    importDeclarations_.push_back(id);
}

AstImportDeclarationList::~AstImportDeclarationList()
{
}

AstImportDeclaration* AstImportDeclarationList::getImportDeclaration(int index) const
{
    return importDeclarations_[index];
}

int AstImportDeclarationList::getImportDeclarationCount() const
{
    return importDeclarations_.size();
}

void AstImportDeclarationList::add(AstImportDeclaration* const id)
{
    importDeclarations_.push_back(id);
    AstNode::append(id);
}

void AstImportDeclarationList::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstImportDeclarationList::type() const
{
    return "ImportDeclarationList";
}

AstImportDeclarationOnDemand::AstImportDeclarationOnDemand(AstImport* const i, 
                                                           AstName* const n,
                                                           AstDot* const d,
                                                           AstMult* const m,
                                                           AstSemicolon* const s,
                                                           File* const srcfile) :
        AstImportDeclaration(i, n, s, srcfile),
     dot_(d), 
     mult_(m)
{
    append(d, n);
    append(m, d);
}

AstImportDeclarationOnDemand::~AstImportDeclarationOnDemand()
{
}

AstDot* AstImportDeclarationOnDemand::getDot() const
{
    return dot_;
}

AstMult* AstImportDeclarationOnDemand::getMult() const
{
    return mult_;
}

void AstImportDeclarationOnDemand::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstImportDeclarationOnDemand::type() const
{
    return "ImportDeclarationOnDemand";
}

string AstImportDeclarationOnDemand::importee() const
{
    return getName()->fullName() + ".*";
}

AstImportDeclarationSingle::AstImportDeclarationSingle(AstImport* const i, 
                                                 AstName* const n,
                                                 AstSemicolon* const s,
                                                 File* const srcfile) :
        AstImportDeclaration(i, n, s, srcfile)
{
}

AstImportDeclarationSingle::~AstImportDeclarationSingle()
{
}

void AstImportDeclarationSingle::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstImportDeclarationSingle::type() const
{
    return "ImportDeclarationSingle";
}

AstInstanceof::AstInstanceof(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstInstanceof::~AstInstanceof()
{
}

string AstInstanceof::type() const
{
    return "INSTANCEOF";
}

void AstInstanceof::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstInstanceof::length() const
{
    return 10;
}

AstInt::AstInt(char* position, File* const srcfile) :
        AstIntegralType(position, srcfile)
{
}

AstInt::~AstInt()
{
}

string AstInt::type() const
{
    return "INT";
}

void AstInt::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstInt::length() const
{
    return 3;
}

AstIntegerLiteral::AstIntegerLiteral(const string& text, char* position, File* const srcfile) :
        AstLiteral(text, position, srcfile)
{
}

AstIntegerLiteral::~AstIntegerLiteral()
{
}

string AstIntegerLiteral::type() const
{
    return "INTEGER_LITERAL";
}

void AstIntegerLiteral::getProcessed(Processor* const proc)
{
    proc->process(this);
}

AstIntegralType::AstIntegralType(char* position, File* const srcfile) :
        AstNumericType(position, srcfile)
{
}

AstIntegralType::~AstIntegralType()
{
}

AstInterfaceBody::AstInterfaceBody(AstLbrace* const lb,
                                   AstInterfaceMemberDeclarationList* const inmedeli,
                                   AstRbrace* const rb,
                                   File* const srcfile) :
        AstNode(lb, inmedeli, rb, srcfile), lbrace_(lb), interfaceMemberDeclarationList_(inmedeli), rbrace_(rb)
{
}

AstInterfaceBody::~AstInterfaceBody()
{
}

AstLbrace* AstInterfaceBody::getLbrace() const
{
    return lbrace_;
}

AstInterfaceMemberDeclarationList* AstInterfaceBody::getInterfaceMemberDeclarationList() const
{
    return interfaceMemberDeclarationList_;
}

AstRbrace* AstInterfaceBody::getRbrace() const
{
    return rbrace_;
}

AstInterfaceBodyDecls::AstInterfaceBodyDecls(AstLbrace* const lb,
                                             AstInterfaceMemberDeclarationList* const inmedeli,
                                             AstRbrace* const rb,
                                             File* const srcfile) :
        AstInterfaceBody(lb, inmedeli, rb, srcfile)
{
}

AstInterfaceBodyDecls::~AstInterfaceBodyDecls()
{
}

void AstInterfaceBodyDecls::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstInterfaceBodyDecls::type() const
{
    return "InterfaceBodyDecls";
}

AstInterfaceBodyNoDecls::AstInterfaceBodyNoDecls(AstLbrace* const lb,
                                                 AstRbrace* const rb,
                                                 File* const srcfile) :
        AstInterfaceBody(lb, NULL, rb, srcfile)
{
}

AstInterfaceBodyNoDecls::~AstInterfaceBodyNoDecls()
{
}

void AstInterfaceBodyNoDecls::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstInterfaceBodyNoDecls::type() const
{
    return "InterfaceBodyNoDecls";
}

AstInterface::AstInterface(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstInterface::~AstInterface()
{
}

string AstInterface::type() const
{
    return "INTERFACE";
}

void AstInterface::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstInterface::length() const
{
    return 9;
}

AstInterfaceDeclaration::AstInterfaceDeclaration(AstModifierList* const moli,
                                                 AstInterface* const in,
                                                 AstIdentifier* const id,
                                                 AstExtendsInterfaceList* const exinli,
                                                 AstInterfaceBody* const inbo,
                                                 File* const srcfile) :
        AstNode(moli, in, id, inbo, srcfile), 
     modifierList_(moli), 
     interface_(in), 
     identifier_(id), 
     extendsInterfaceList_(exinli),
     interfaceBody_(inbo)
{
}

AstInterfaceDeclaration::~AstInterfaceDeclaration()
{
}

AstModifierList* AstInterfaceDeclaration::getModifierList() const
{
    return modifierList_;
}

AstInterface* AstInterfaceDeclaration::getInterface() const
{
    return interface_;
}

AstIdentifier* AstInterfaceDeclaration::getIdentifier() const
{
    return identifier_;
}

AstExtendsInterfaceList* AstInterfaceDeclaration::getExtendsInterfaceList() const
{
    return extendsInterfaceList_;
}

AstInterfaceBody* AstInterfaceDeclaration::getInterfaceBody() const
{
    return interfaceBody_;
}

AstInterfaceDeclarationModsExtends::AstInterfaceDeclarationModsExtends(AstModifierList* const moli,
                                                                       AstInterface* const in,
                                                                       AstIdentifier* const id,
                                                                       AstExtendsInterfaceList* const exinli,
                                                                       AstInterfaceBody* const inbo,
                                                                       File* const srcfile) :
        AstInterfaceDeclaration(moli, in, id, exinli, inbo, srcfile)
{
}

AstInterfaceDeclarationModsExtends::~AstInterfaceDeclarationModsExtends()
{
}

void AstInterfaceDeclarationModsExtends::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstInterfaceDeclarationModsExtends::type() const
{
    return "InterfaceDeclarationModsExtends";
}

AstInterfaceDeclarationModsNoExtends::AstInterfaceDeclarationModsNoExtends(AstModifierList* const moli,
                                                                           AstInterface* const in,
                                                                           AstIdentifier* const id,
                                                                           AstInterfaceBody* const inbo,
                                                                           File* const srcfile) :
        AstInterfaceDeclaration(moli, in, id, NULL, inbo, srcfile)
{
}

AstInterfaceDeclarationModsNoExtends::~AstInterfaceDeclarationModsNoExtends()
{
}

void AstInterfaceDeclarationModsNoExtends::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstInterfaceDeclarationModsNoExtends::type() const
{
    return "InterfaceDeclarationModsNoExtends";
}

AstInterfaceDeclarationNoModsExtends::AstInterfaceDeclarationNoModsExtends(AstInterface* const in,
                                                                           AstIdentifier* const id,
                                                                           AstExtendsInterfaceList* const exinli,
                                                                           AstInterfaceBody* const inbo,
                                                                           File* const srcfile) :
        AstInterfaceDeclaration(NULL, in, id, exinli, inbo, srcfile)
{
}

AstInterfaceDeclarationNoModsExtends::~AstInterfaceDeclarationNoModsExtends()
{
}

void AstInterfaceDeclarationNoModsExtends::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstInterfaceDeclarationNoModsExtends::type() const
{
    return "InterfaceDeclarationNoModsExtends";
}

AstInterfaceDeclarationNoModsNoExtends::AstInterfaceDeclarationNoModsNoExtends(AstInterface* const in,
                                                                               AstIdentifier* const id,
                                                                               AstInterfaceBody* const inbo,
                                                                               File* const srcfile) :
        AstInterfaceDeclaration(NULL, in, id, NULL, inbo, srcfile)
{
}

AstInterfaceDeclarationNoModsNoExtends::~AstInterfaceDeclarationNoModsNoExtends()
{
}

void AstInterfaceDeclarationNoModsNoExtends::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstInterfaceDeclarationNoModsNoExtends::type() const
{
    return "InterfaceDeclarationNoModsNoExtends";
}

AstInterfaceMemberDeclarationList::AstInterfaceMemberDeclarationList(AstNode* const imd, 
                                                               File* const srcfile) :
        AstNode(imd, srcfile)
{
    interfaceMemberDeclarations_.push_back(imd);
}

AstInterfaceMemberDeclarationList::~AstInterfaceMemberDeclarationList()
{
}

AstNode* AstInterfaceMemberDeclarationList::getInterfaceMemberDeclaration(int index) const
{
    return interfaceMemberDeclarations_[index];
}

int AstInterfaceMemberDeclarationList::getInterfaceMemberDeclarationCount() const
{
    return interfaceMemberDeclarations_.size();
}

void AstInterfaceMemberDeclarationList::add(AstNode* const imd)
{
    interfaceMemberDeclarations_.push_back(imd);
    AstNode::append(imd);
}

void AstInterfaceMemberDeclarationList::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstInterfaceMemberDeclarationList::type() const
{
    return "InterfaceMemberDeclarationList";
}

AstItem::AstItem(File* const srcfile) : srcfile_(srcfile), parent_(NULL)
{
}

AstItem::~AstItem()
{
    // srcfile_ is shared, so don't delete it.
}

void AstItem::process(const vector<Processor*>& processors)
{
    for (vector<Processor*>::const_iterator it = processors.begin(); it != processors.end(); ++it) {
        Processor* const proc = *it;
        getProcessed(proc);
    }
}

File* AstItem::sourceFile() const
{
    return srcfile_;
}

AstNode* AstItem::parent() const
{
    return parent_;
}

void AstItem::setParent(AstNode* const pt)
{
    parent_ = pt;
}

AstItem* AstItem::previous() const
{
    AstNode* pt = parent();
    if (pt) {
        return pt->findPrevious(this);
    }
    else {
        return NULL;
    }
}

AstItem* AstItem::next() const
{
    AstNode* pt = parent();
    if (pt) {
        return pt->findNext(this);
    }
    else {
        return NULL;
    }
}

int AstItem::line() const
{
    return srcfile_->lineOf(position());
}

int AstItem::column() const
{
    return srcfile_->columnOf(position());
}

string AstItem::text() const 
{
    char* start = position();
    char* end   = endPosition();
    return string(start, 1 + end - start);
}

int AstItem::endLine() const
{
    return srcfile_->lineOf(endPosition());
}

int AstItem::endColumn() const
{
    return srcfile_->columnOf(endPosition());
}

AstJavadocComment::AstJavadocComment(char* start, char* end, File* const srcfile) :
        AstComment(start, end, srcfile), descriptionStart_(NULL), descriptionEnd_(NULL)
{
    JavadocParser jp(start_, end_);
    jp.getDescription(&descriptionStart_, &descriptionEnd_);
    vector<TagCmt> tags;
    jp.getTagCmts(&tags);
    vector<TagCmt>::iterator it   = tags.begin();
    vector<TagCmt>::iterator stop = tags.end();
    while (it != stop) {
        TagCmt tc = *it;
        if (tc.descStart && tc.descEnd) {
            // got a description
            taggedComments_.push_back(new AstTaggedDescribedComment(tc.start, tc.end, tc.descStart, tc.descEnd, srcfile));
        }
        else {
            // no description
            taggedComments_.push_back(new AstTaggedComment(tc.start, tc.end, srcfile));
        }
        ++it;
    }
}

AstJavadocComment::~AstJavadocComment()
{
}

string AstJavadocComment::type() const 
{
    return "Javadoc comment";
}

vector<AstTaggedComment*> AstJavadocComment::taggedComments() const
{
    return taggedComments_;
}

string AstJavadocComment::description() const
{
    if (descriptionStart_ && descriptionEnd_) {
        return string(descriptionStart_, 1 + descriptionEnd_ - descriptionStart_);
    }
    else {
        return "";
    }
}

char* AstJavadocComment::descriptionStart() const
{
    return descriptionStart_;
}

char* AstJavadocComment::descriptionEnd() const
{
    return descriptionEnd_;
}

AstStringLeaf AstJavadocComment::descriptionLeaf() const
{
    return AstStringLeaf(descriptionStart_, descriptionEnd_, sourceFile());
}

void AstJavadocComment::getProcessed(Processor* const p)
{
    p->process(this);
}

AstLabeledStatement::AstLabeledStatement(AstIdentifier* const i, 
             AstColon* const c,
             AstItem* const s,
             File* const srcfile) :
        AstNode(i, c, s, srcfile), identifier_(i), colon_(c), statement_(s)
{
}

AstLabeledStatement::~AstLabeledStatement()
{
}

AstIdentifier* AstLabeledStatement::getIdentifier() const
{
    return identifier_;
}

AstColon* AstLabeledStatement::getColon() const
{
    return colon_;
}

AstItem* AstLabeledStatement::getStatement() const
{
    return statement_;
}

void AstLabeledStatement::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstLabeledStatement::type() const
{
    return "LabeledStatement";
}

AstLbrace::AstLbrace(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstLbrace::~AstLbrace()
{
}

string AstLbrace::type() const
{
    return "LBRACE";
}

void AstLbrace::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstLbrace::length() const
{
    return 1;
}

AstLbracket::AstLbracket(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstLbracket::~AstLbracket()
{
}

string AstLbracket::type() const
{
    return "LBRACKET";
}

void AstLbracket::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstLbracket::length() const
{
    return 1;
}

AstLeaf::AstLeaf(char* position, File* const srcfile) :
        AstItem(srcfile),
        position_(position), 
        leadingNoncode_(NULL)
{
}

AstLeaf::~AstLeaf()
{
}

char* AstLeaf::position() const
{
    return position_;
}

AstNoncode* AstLeaf::leadingNoncode() const
{
    return leadingNoncode_;
}

void AstLeaf::setLeadingNoncode(AstNoncode* const nc)
{
    leadingNoncode_ = nc;
}

AstNoncode* AstLeaf::trailingNoncode() const
{
    if (AstItem* nxt = next()) {
        return nxt->leadingNoncode();
    }
    else {
        // cout << "leaf<" << type() << "> has no next item. returning NULL" << endl;
        return NULL;
    }
}

char* AstLeaf::endPosition() const
{
    return position() + length() - 1;
}

AstLiteral::AstLiteral(const string& text, char* position, File* const srcfile) :
        AstValue(text, position, srcfile)
{
}

AstLiteral::~AstLiteral()
{
}

AstLocalVariableDeclaration::AstLocalVariableDeclaration(AstFinal* const f, 
                                                         AstItem* const t, 
                                                         AstVariableDeclaratorList* const vdl,
                                                         File* const srcfile) :
        AstNode(f, t, vdl, srcfile), final_(f), type_(t), variableDeclaratorList_(vdl)
{
}

AstLocalVariableDeclaration::~AstLocalVariableDeclaration()
{
}

AstFinal* AstLocalVariableDeclaration::getFinal() const
{
    return final_;
}

AstItem* AstLocalVariableDeclaration::getType() const
{
    return type_;
}

AstVariableDeclaratorList* AstLocalVariableDeclaration::getVariableDeclaratorList() const
{
    return variableDeclaratorList_;
}

AstLocalVariableDeclarationFinal::AstLocalVariableDeclarationFinal(AstFinal* const f, 
                                                                   AstItem* const t,
                                                                   AstVariableDeclaratorList* const vdl,
                                                                   File* const srcfile) :
        AstLocalVariableDeclaration(f, t, vdl, srcfile)
{
}

AstLocalVariableDeclarationFinal::~AstLocalVariableDeclarationFinal()
{
}

void AstLocalVariableDeclarationFinal::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstLocalVariableDeclarationFinal::type() const
{
    return "LocalVariableDeclarationFinal";
}

AstLocalVariableDeclarationNonFinal::AstLocalVariableDeclarationNonFinal(AstItem* const t, 
                                                                         AstVariableDeclaratorList* const vdl,
                                                                         File* const srcfile) :
        AstLocalVariableDeclaration(NULL, t, vdl, srcfile)
{
}

AstLocalVariableDeclarationNonFinal::~AstLocalVariableDeclarationNonFinal()
{
}

void AstLocalVariableDeclarationNonFinal::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstLocalVariableDeclarationNonFinal::type() const
{
    return "LocalVariableDeclarationNonFinal";
}

AstLocalVariableDeclarationStatement::AstLocalVariableDeclarationStatement(AstLocalVariableDeclaration* const lvd, 
             AstSemicolon* const s,
             File* const srcfile) :
        AstNode(lvd, s, srcfile), localVariableDeclaration_(lvd), semicolon_(s)
{
}

AstLocalVariableDeclarationStatement::~AstLocalVariableDeclarationStatement()
{
}

AstLocalVariableDeclaration* AstLocalVariableDeclarationStatement::getLocalVariableDeclaration() const
{
    return localVariableDeclaration_;
}

AstSemicolon* AstLocalVariableDeclarationStatement::getSemicolon() const
{
    return semicolon_;
}

void AstLocalVariableDeclarationStatement::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstLocalVariableDeclarationStatement::type() const
{
    return "LocalVariableDeclarationStatement";
}

AstLong::AstLong(char* position, File* const srcfile) :
        AstIntegralType(position, srcfile)
{
}

AstLong::~AstLong()
{
}

string AstLong::type() const
{
    return "LONG";
}

void AstLong::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstLong::length() const
{
    return 4;
}

AstLparen::AstLparen(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstLparen::~AstLparen()
{
}

string AstLparen::type() const
{
    return "LPAREN";
}

void AstLparen::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstLparen::length() const
{
    return 1;
}

AstLshift::AstLshift(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstLshift::~AstLshift()
{
}

string AstLshift::type() const
{
    return "LSHIFT";
}

void AstLshift::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstLshift::length() const
{
    return 2;
}

AstLshifteq::AstLshifteq(char* position, File* const srcfile) :
        AstAssignmentOperator(position, srcfile)
{
}

AstLshifteq::~AstLshifteq()
{
}

string AstLshifteq::type() const
{
    return "LSHIFTEQ";
}

void AstLshifteq::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstLshifteq::length() const
{
    return 3;
}

AstLt::AstLt(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstLt::~AstLt()
{
}

string AstLt::type() const
{
    return "LT";
}

void AstLt::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstLt::length() const
{
    return 1;
}

AstLteq::AstLteq(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstLteq::~AstLteq()
{
}

string AstLteq::type() const
{
    return "LTEQ";
}

void AstLteq::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstLteq::length() const
{
    return 2;
}

AstMethodDeclarationBlock::AstMethodDeclarationBlock(AstMethodHeader* const mehe,
                                               AstBlock* const bl,
                                               File* const srcfile) :
        AstMethodDeclaration(mehe, srcfile), block_(bl)
{
    append(bl, mehe);
}

AstMethodDeclarationBlock::~AstMethodDeclarationBlock()
{
}

AstBlock* AstMethodDeclarationBlock::getBlock() const
{
    return block_;
}

void AstMethodDeclarationBlock::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMethodDeclarationBlock::type() const
{
    return "MethodDeclarationBlock";
}

AstMethodDeclaration::AstMethodDeclaration(AstMethodHeader* const mehe,
                                           File* const srcfile) :
        AstNode(mehe, srcfile), methodHeader_(mehe)
{
}

AstMethodDeclaration::~AstMethodDeclaration()
{
}

AstMethodHeader* AstMethodDeclaration::getMethodHeader() const
{
    return methodHeader_;
}

AstMethodDeclarationSemicolon::AstMethodDeclarationSemicolon(AstMethodHeader* const mehe,
                                                       AstSemicolon* const se,
                                                       File* const srcfile) :
        AstMethodDeclaration(mehe, srcfile), semicolon_(se)
{
    append(se, mehe);
}

AstMethodDeclarationSemicolon::~AstMethodDeclarationSemicolon()
{
}

AstSemicolon* AstMethodDeclarationSemicolon::getSemicolon() const
{
    return semicolon_;
}

void AstMethodDeclarationSemicolon::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMethodDeclarationSemicolon::type() const
{
    return "MethodDeclarationSemicolon";
}

AstMethodDeclarator::AstMethodDeclarator(AstIdentifier* const i, 
                                         AstLparen* const l,
                                         AstFormalParameterList* const fpl,
                                         AstRparen* const r,
                                         File* const srcfile) :
        AstNode(i, l, fpl, r, srcfile),
     identifier_(i), 
     lparen_(l), 
     formalParameterList_(fpl),
     rparen_(r)
{
}

AstMethodDeclarator::~AstMethodDeclarator()
{
}

AstIdentifier* AstMethodDeclarator::getIdentifier() const
{
    return identifier_;
}

AstLparen* AstMethodDeclarator::getLparen() const
{
    return lparen_;
}

AstFormalParameterList* AstMethodDeclarator::getFormalParameterList() const
{
    return formalParameterList_;
}

AstRparen* AstMethodDeclarator::getRparen() const
{
    return rparen_;
}

AstDimension* AstMethodDeclarator::getDimension(int index) const
{
    return dimensions_[index];
}

int AstMethodDeclarator::getDimensionCount() const
{
    return dimensions_.size();
}

void AstMethodDeclarator::add(AstDimension* const d)
{
    dimensions_.push_back(d);
    AstNode::append(d);
}

AstMethodDeclaratorNoArgs::AstMethodDeclaratorNoArgs(AstIdentifier* const i, 
                                                     AstLparen* const l,
                                                     AstRparen* const r,
                                                     File* const srcfile) :
        AstMethodDeclarator(i, l, NULL, r, srcfile)
{
}

AstMethodDeclaratorNoArgs::~AstMethodDeclaratorNoArgs()
{
}

void AstMethodDeclaratorNoArgs::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMethodDeclaratorNoArgs::type() const
{
    return "MethodDeclaratorNoArgs";
}

AstMethodDeclaratorWArgs::AstMethodDeclaratorWArgs(AstIdentifier* const i, 
                                                   AstLparen* const l,
                                                   AstFormalParameterList* const fpl,
                                                   AstRparen* const r,
                                                   File* const srcfile) :
        AstMethodDeclarator(i, l, fpl, r, srcfile)
{
}

AstMethodDeclaratorWArgs::~AstMethodDeclaratorWArgs()
{
}

void AstMethodDeclaratorWArgs::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMethodDeclaratorWArgs::type() const
{
    return "MethodDeclaratorWArgs";
}

AstMethodHeader::AstMethodHeader(AstModifierList* const moli,
                                 AstItem* const ty,
                                 AstMethodDeclarator* const mede,
                                 AstThrowsNameList* const thnali,
                                 File* const srcfile) :
        AstNode(moli, ty, mede, thnali, srcfile), 
     modifierList_(moli), 
     type_(ty), 
     methodDeclarator_(mede), 
     throwsNameList_(thnali)
{
}

AstMethodHeader::~AstMethodHeader()
{
}

AstModifierList* AstMethodHeader::getModifierList() const
{
    return modifierList_;
}

AstItem* AstMethodHeader::getType() const
{
    return type_;
}

AstMethodDeclarator* AstMethodHeader::getMethodDeclarator() const
{
    return methodDeclarator_;
}

AstThrowsNameList* AstMethodHeader::getThrowsNameList() const
{
    return throwsNameList_;
}

string AstMethodHeader::name() const
{
    AstIdentifier* id = methodDeclarator_->getIdentifier();
    return id->text();
}

AstMethodHeaderTypeModsNoThrows::AstMethodHeaderTypeModsNoThrows(AstModifierList* const moli,
                                                                 AstItem* const ty,
                                                                 AstMethodDeclarator* const mede,
                                                                 File* const srcfile) :
        AstMethodHeader(moli, ty, mede, NULL, srcfile)
{
}

AstMethodHeaderTypeModsNoThrows::~AstMethodHeaderTypeModsNoThrows()
{
}

void AstMethodHeaderTypeModsNoThrows::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMethodHeaderTypeModsNoThrows::type() const
{
    return "MethodHeaderTypeModsNoThrows";
}

AstMethodHeaderTypeModsThrows::AstMethodHeaderTypeModsThrows(AstModifierList* const moli,
                                                             AstItem* const ty,
                                                             AstMethodDeclarator* const mede,
                                                             AstThrowsNameList* const thnali,
                                                             File* const srcfile) :
        AstMethodHeader(moli, ty, mede, thnali, srcfile)
{
}

AstMethodHeaderTypeModsThrows::~AstMethodHeaderTypeModsThrows()
{
}

void AstMethodHeaderTypeModsThrows::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMethodHeaderTypeModsThrows::type() const
{
    return "MethodHeaderTypeModsThrows";
}

AstMethodHeaderTypeNoModsNoThrows::AstMethodHeaderTypeNoModsNoThrows(AstItem* const ty,
                                                                     AstMethodDeclarator* const mede,
                                                                     File* const srcfile) :
        AstMethodHeader(NULL, ty, mede, NULL, srcfile)
{
}

AstMethodHeaderTypeNoModsNoThrows::~AstMethodHeaderTypeNoModsNoThrows()
{
}

void AstMethodHeaderTypeNoModsNoThrows::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMethodHeaderTypeNoModsNoThrows::type() const
{
    return "MethodHeaderTypeNoModsNoThrows";
}

AstMethodHeaderTypeNoModsThrows::AstMethodHeaderTypeNoModsThrows(AstItem* const ty,
                                                                 AstMethodDeclarator* const mede,
                                                                 AstThrowsNameList* const thnali,
                                                                 File* const srcfile) :
        AstMethodHeader(NULL, ty, mede, thnali, srcfile)
{
}

AstMethodHeaderTypeNoModsThrows::~AstMethodHeaderTypeNoModsThrows()
{
}

void AstMethodHeaderTypeNoModsThrows::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMethodHeaderTypeNoModsThrows::type() const
{
    return "MethodHeaderTypeNoModsThrows";
}

AstMethodHeaderVoidModsNoThrows::AstMethodHeaderVoidModsNoThrows(AstModifierList* const moli,
                                                                 AstVoid* const vo,
                                                                 AstMethodDeclarator* const mede,
                                                                 File* const srcfile) :
        AstMethodHeader(moli, vo, mede, NULL, srcfile), void_(vo)
{
}

AstMethodHeaderVoidModsNoThrows::~AstMethodHeaderVoidModsNoThrows()
{
}

AstVoid* AstMethodHeaderVoidModsNoThrows::getVoid() const
{
    return void_;
}

void AstMethodHeaderVoidModsNoThrows::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMethodHeaderVoidModsNoThrows::type() const
{
    return "MethodHeaderVoidModsNoThrows";
}

AstMethodHeaderVoidModsThrows::AstMethodHeaderVoidModsThrows(AstModifierList* const moli,
                                                       AstVoid* const vo,
                                                       AstMethodDeclarator* const mede,
                                                       AstThrowsNameList* const thnali,
                                                       File* const srcfile) :
        AstMethodHeader(moli, vo, mede, thnali, srcfile), void_(vo)
{
}

AstMethodHeaderVoidModsThrows::~AstMethodHeaderVoidModsThrows()
{
}

AstVoid* AstMethodHeaderVoidModsThrows::getVoid() const
{
    return void_;
}

void AstMethodHeaderVoidModsThrows::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMethodHeaderVoidModsThrows::type() const
{
    return "MethodHeaderVoidModsThrows";
}

AstMethodHeaderVoidNoModsNoThrows::AstMethodHeaderVoidNoModsNoThrows(AstVoid* const vo,
                                                                     AstMethodDeclarator* const mede,
                                                                     File* const srcfile) :
        AstMethodHeader(NULL, vo, mede, NULL, srcfile), void_(vo)
{
}

AstMethodHeaderVoidNoModsNoThrows::~AstMethodHeaderVoidNoModsNoThrows()
{
}

AstVoid* AstMethodHeaderVoidNoModsNoThrows::getVoid() const
{
    return void_;
}

void AstMethodHeaderVoidNoModsNoThrows::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMethodHeaderVoidNoModsNoThrows::type() const
{
    return "MethodHeaderVoidNoModsNoThrows";
}

AstMethodHeaderVoidNoModsThrows::AstMethodHeaderVoidNoModsThrows(AstVoid* const vo,
                                                                 AstMethodDeclarator* const mede,
                                                                 AstThrowsNameList* const thnali,
                                                                 File* const srcfile) :
        AstMethodHeader(NULL, vo, mede, thnali, srcfile), void_(vo)
{
}

AstMethodHeaderVoidNoModsThrows::~AstMethodHeaderVoidNoModsThrows()
{
}

AstVoid* AstMethodHeaderVoidNoModsThrows::getVoid() const
{
    return void_;
}

void AstMethodHeaderVoidNoModsThrows::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMethodHeaderVoidNoModsThrows::type() const
{
    return "MethodHeaderVoidNoModsThrows";
}

AstMethodInvocation::AstMethodInvocation(AstLparen* const lp,
                                         AstRparen* const rp,
                                         File* const srcfile) :
        AstNode(lp, rp, srcfile), lparen_(lp), rparen_(rp)
{
}

AstMethodInvocation::~AstMethodInvocation()
{
}

AstLparen* AstMethodInvocation::getLparen() const
{
    return lparen_;
}

AstRparen* AstMethodInvocation::getRparen() const
{
    return rparen_;
}

AstMethodInvocationNameArgs::AstMethodInvocationNameArgs(AstName* const na,
                                                   AstLparen* const lp,
                                                   AstExpressionList* const exli,
                                                   AstRparen* const rp,
                                                   File* const srcfile) :
        AstMethodInvocation(lp, rp, srcfile), name_(na), expressionList_(exli)
{
    prepend(na);
    append(exli, lp);
}

AstMethodInvocationNameArgs::~AstMethodInvocationNameArgs()
{
}

AstName* AstMethodInvocationNameArgs::getName() const
{
    return name_;
}

AstExpressionList* AstMethodInvocationNameArgs::getExpressionList() const
{
    return expressionList_;
}

void AstMethodInvocationNameArgs::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMethodInvocationNameArgs::type() const
{
    return "MethodInvocationNameArgs";
}

AstMethodInvocationNameNoArgs::AstMethodInvocationNameNoArgs(AstName* const na,
                                                       AstLparen* const lp,
                                                       AstRparen* const rp,
                                                       File* const srcfile) :
        AstMethodInvocation(lp, rp, srcfile), name_(na)
{
    prepend(na);
}

AstMethodInvocationNameNoArgs::~AstMethodInvocationNameNoArgs()
{
}

AstName* AstMethodInvocationNameNoArgs::getName() const
{
    return name_;
}

void AstMethodInvocationNameNoArgs::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMethodInvocationNameNoArgs::type() const
{
    return "MethodInvocationNameNoArgs";
}

AstMethodInvocationNameSuperArgs::AstMethodInvocationNameSuperArgs(AstName* const na,
                                                                   AstDot* const d0,
                                                                   AstSuper* const su,
                                                                   AstDot* const d1,
                                                                   AstIdentifier* const id,
                                                                   AstLparen* const lp,
                                                                   AstExpressionList* const exli,
                                                                   AstRparen* const rp,
                                                                   File* const srcfile) :
        AstMethodInvocation(lp, rp, srcfile),
     name_(na),
     firstDot_(d0),
     super_(su),
     secondDot_(d1),
     identifier_(id),
     expressionList_(exli)
{
    prepend(na);
    append(d0, na);
    append(su, d0);
    append(d1, su);
    append(id, d1);
    append(exli, lp);
}

AstMethodInvocationNameSuperArgs::~AstMethodInvocationNameSuperArgs()
{
}

AstName* AstMethodInvocationNameSuperArgs::getName() const
{
    return name_;
}

AstDot* AstMethodInvocationNameSuperArgs::getFirstDot() const
{
    return firstDot_;
}

AstSuper* AstMethodInvocationNameSuperArgs::getSuper() const
{
    return super_;
}

AstDot* AstMethodInvocationNameSuperArgs::getSecondDot() const
{
    return secondDot_;
}

AstIdentifier* AstMethodInvocationNameSuperArgs::getIdentifier() const
{
    return identifier_;
}

AstExpressionList* AstMethodInvocationNameSuperArgs::getExpressionList() const
{
    return expressionList_;
}

void AstMethodInvocationNameSuperArgs::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMethodInvocationNameSuperArgs::type() const
{
    return "MethodInvocationNameSuperArgs";
}

AstMethodInvocationNameSuperNoArgs::AstMethodInvocationNameSuperNoArgs(AstName* const na,
                                                                       AstDot* const d0,
                                                                       AstSuper* const su,
                                                                       AstDot* const d1,
                                                                       AstIdentifier* const id,
                                                                       AstLparen* const lp,
                                                                       AstRparen* const rp,
                                                                       File* const srcfile) :
        AstMethodInvocation(lp, rp, srcfile), name_(na), firstDot_(d0), super_(su), secondDot_(d1), identifier_(id)
{
    prepend(na);
    append(d0, na);
    append(su, d0);
    append(d1, su);
    append(id, d1);
}

AstMethodInvocationNameSuperNoArgs::~AstMethodInvocationNameSuperNoArgs()
{
}

AstName* AstMethodInvocationNameSuperNoArgs::getName() const
{
    return name_;
}

AstDot* AstMethodInvocationNameSuperNoArgs::getFirstDot() const
{
    return firstDot_;
}

AstSuper* AstMethodInvocationNameSuperNoArgs::getSuper() const
{
    return super_;
}

AstDot* AstMethodInvocationNameSuperNoArgs::getSecondDot() const
{
    return secondDot_;
}

AstIdentifier* AstMethodInvocationNameSuperNoArgs::getIdentifier() const
{
    return identifier_;
}

void AstMethodInvocationNameSuperNoArgs::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMethodInvocationNameSuperNoArgs::type() const
{
    return "MethodInvocationNameSuperNoArgs";
}

AstMethodInvocationPrimary::AstMethodInvocationPrimary(AstItem* const pr,
                                                       AstDot* const d,
                                                       AstIdentifier* const id,
                                                       AstLparen* const lp,
                                                       AstRparen* const rp,
                                                       File* const srcfile) :
        AstMethodInvocation(lp, rp, srcfile), primary_(pr), dot_(d), identifier_(id)
{
    prepend(pr);
    append(d, pr);
    append(id, d);
}

AstMethodInvocationPrimary::~AstMethodInvocationPrimary()
{
}

AstItem* AstMethodInvocationPrimary::getPrimary() const
{
    return primary_;
}

AstDot* AstMethodInvocationPrimary::getDot() const
{
    return dot_;
}

AstIdentifier* AstMethodInvocationPrimary::getIdentifier() const
{
    return identifier_;
}

AstMethodInvocationPrimaryArgs::AstMethodInvocationPrimaryArgs(AstItem* const pr,
                                                               AstDot* const d,
                                                               AstIdentifier* const id,
                                                               AstLparen* const lp,
                                                               AstExpressionList* const exli,
                                                               AstRparen* const rp,
                                                               File* const srcfile) :
        AstMethodInvocationPrimary(pr, d, id, lp, rp, srcfile), expressionList_(exli)
{
    append(exli, lp);
}

AstMethodInvocationPrimaryArgs::~AstMethodInvocationPrimaryArgs()
{
}

AstExpressionList* AstMethodInvocationPrimaryArgs::getExpressionList() const
{
    return expressionList_;
}

void AstMethodInvocationPrimaryArgs::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMethodInvocationPrimaryArgs::type() const
{
    return "MethodInvocationPrimaryArgs";
}

AstMethodInvocationPrimaryNoArgs::AstMethodInvocationPrimaryNoArgs(AstItem* const pr,
                                                                   AstDot* const d,
                                                                   AstIdentifier* const id,
                                                                   AstLparen* const lp,
                                                                   AstRparen* const rp,
                                                                   File* const srcfile) :
        AstMethodInvocationPrimary(pr, d, id, lp, rp, srcfile)
{
}

AstMethodInvocationPrimaryNoArgs::~AstMethodInvocationPrimaryNoArgs()
{
}

void AstMethodInvocationPrimaryNoArgs::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMethodInvocationPrimaryNoArgs::type() const
{
    return "MethodInvocationPrimaryNoArgs";
}

AstMethodInvocationSuper::AstMethodInvocationSuper(AstSuper* const su,
                                                   AstDot* const d,
                                                   AstIdentifier* const id,
                                                   AstLparen* const lp,
                                                   AstRparen* const rp,
                                                   File* const srcfile) :
        AstMethodInvocation(lp, rp, srcfile), super_(su), dot_(d), identifier_(id)
{
    prepend(su);
    append(d, su);
    append(id, d);
}

AstMethodInvocationSuper::~AstMethodInvocationSuper()
{
}

AstSuper* AstMethodInvocationSuper::getSuper() const
{
    return super_;
}

AstDot* AstMethodInvocationSuper::getDot() const
{
    return dot_;
}

AstIdentifier* AstMethodInvocationSuper::getIdentifier() const
{
    return identifier_;
}

AstMethodInvocationSuperArgs::AstMethodInvocationSuperArgs(AstSuper* const su,
                                                           AstDot* const d,
                                                           AstIdentifier* const id,
                                                           AstLparen* const lp,
                                                           AstExpressionList* const exli,
                                                           AstRparen* const rp,
                                                           File* const srcfile) :
        AstMethodInvocationSuper(su, d, id, lp, rp, srcfile), expressionList_(exli)
{
    append(exli, lp);
}

AstMethodInvocationSuperArgs::~AstMethodInvocationSuperArgs()
{
}

AstExpressionList* AstMethodInvocationSuperArgs::getExpressionList() const
{
    return expressionList_;
}

void AstMethodInvocationSuperArgs::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMethodInvocationSuperArgs::type() const
{
    return "MethodInvocationSuperArgs";
}

AstMethodInvocationSuperNoArgs::AstMethodInvocationSuperNoArgs(AstSuper* const su,
                                                               AstDot* const d,
                                                               AstIdentifier* const id,
                                                               AstLparen* const lp,
                                                               AstRparen* const rp,
                                                               File* const srcfile) :
        AstMethodInvocationSuper(su, d, id, lp, rp, srcfile)
{
}

AstMethodInvocationSuperNoArgs::~AstMethodInvocationSuperNoArgs()
{
}

void AstMethodInvocationSuperNoArgs::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMethodInvocationSuperNoArgs::type() const
{
    return "MethodInvocationSuperNoArgs";
}

AstMinus::AstMinus(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstMinus::~AstMinus()
{
}

string AstMinus::type() const
{
    return "MINUS";
}

void AstMinus::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstMinus::length() const
{
    return 1;
}

AstMinuseq::AstMinuseq(char* position, File* const srcfile) :
        AstAssignmentOperator(position, srcfile)
{
}

AstMinuseq::~AstMinuseq()
{
}

string AstMinuseq::type() const
{
    return "MINUSEQ";
}

void AstMinuseq::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstMinuseq::length() const
{
    return 2;
}

AstMinusminus::AstMinusminus(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstMinusminus::~AstMinusminus()
{
}

string AstMinusminus::type() const
{
    return "MINUSMINUS";
}

void AstMinusminus::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstMinusminus::length() const
{
    return 2;
}

AstMod::AstMod(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstMod::~AstMod()
{
}

string AstMod::type() const
{
    return "MOD";
}

void AstMod::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstMod::length() const
{
    return 1;
}

AstModeq::AstModeq(char* position, File* const srcfile) :
        AstAssignmentOperator(position, srcfile)
{
}

AstModeq::~AstModeq()
{
}

string AstModeq::type() const
{
    return "MODEQ";
}

void AstModeq::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstModeq::length() const
{
    return 2;
}

AstModifierList::AstModifierList(AstLeaf* const m, 
                           File* const srcfile) :
        AstNode(m, srcfile)
{
    modifiers_.push_back(m);
}

AstModifierList::~AstModifierList()
{
}

AstLeaf* AstModifierList::getModifier(int index) const
{
    return modifiers_[index];
}

int AstModifierList::getModifierCount() const
{
    return modifiers_.size();
}

void AstModifierList::add(AstLeaf* const m)
{
    modifiers_.push_back(m);
    AstNode::append(m);
}

void AstModifierList::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstModifierList::type() const
{
    return "ModifierList";
}

const vector<AstLeaf*>& AstModifierList::getModifiers() const
{
    return modifiers_;
}

AstMult::AstMult(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstMult::~AstMult()
{
}

string AstMult::type() const
{
    return "MULT";
}

void AstMult::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstMult::length() const
{
    return 1;
}

AstMulteq::AstMulteq(char* position, File* const srcfile) :
        AstAssignmentOperator(position, srcfile)
{
}

AstMulteq::~AstMulteq()
{
}

string AstMulteq::type() const
{
    return "MULTEQ";
}

void AstMulteq::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstMulteq::length() const
{
    return 2;
}

AstMultiplicativeExpression::AstMultiplicativeExpression(AstItem* const lhex,
                                                         AstItem* const rhex,
                                                         File* const srcfile) :
        AstNode(lhex, rhex, srcfile), lhsExpression_(lhex), rhsExpression_(rhex)
{
}

AstMultiplicativeExpression::~AstMultiplicativeExpression()
{
}

AstItem* AstMultiplicativeExpression::getLhsExpression() const
{
    return lhsExpression_;
}

AstItem* AstMultiplicativeExpression::getRhsExpression() const
{
    return rhsExpression_;
}

AstMultiplicativeExpressionDiv::AstMultiplicativeExpressionDiv(AstItem* const lhex,
                                                               AstDiv* const di,
                                                               AstItem* const rhex,
                                                               File* const srcfile) :
        AstMultiplicativeExpression(lhex, rhex, srcfile), div_(di)
{
    append(di, lhex);
}

AstMultiplicativeExpressionDiv::~AstMultiplicativeExpressionDiv()
{
}

AstDiv* AstMultiplicativeExpressionDiv::getDiv() const
{
    return div_;
}

void AstMultiplicativeExpressionDiv::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMultiplicativeExpressionDiv::type() const
{
    return "MultiplicativeExpressionDiv";
}

AstMultiplicativeExpressionMod::AstMultiplicativeExpressionMod(AstItem* const lhex,
                                                               AstMod* const mo,
                                                               AstItem* const rhex,
                                                               File* const srcfile) :
        AstMultiplicativeExpression(lhex, rhex, srcfile), mod_(mo)
{
    append(mo, lhex);
}

AstMultiplicativeExpressionMod::~AstMultiplicativeExpressionMod()
{
}

AstMod* AstMultiplicativeExpressionMod::getMod() const
{
    return mod_;
}

void AstMultiplicativeExpressionMod::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMultiplicativeExpressionMod::type() const
{
    return "MultiplicativeExpressionMod";
}

AstMultiplicativeExpressionMult::AstMultiplicativeExpressionMult(AstItem* const lhex,
                                                                 AstMult* const mu,
                                                                 AstItem* const rhex,
                                                                 File* const srcfile) :
        AstMultiplicativeExpression(lhex, rhex, srcfile), mult_(mu)
{
    append(mu, lhex);
}

AstMultiplicativeExpressionMult::~AstMultiplicativeExpressionMult()
{
}

AstMult* AstMultiplicativeExpressionMult::getMult() const
{
    return mult_;
}

void AstMultiplicativeExpressionMult::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstMultiplicativeExpressionMult::type() const
{
    return "MultiplicativeExpressionMult";
}

AstName::AstName(AstIdentifier* const id, File* const srcfile)
        : AstNode(id, srcfile)
{
}

AstName::~AstName()
{
}

string AstName::fullName() const
{
    list<AstItem*>::const_iterator stop = components().end();
    list<AstItem*>::const_iterator start = components().begin();
    string name;
    for (list<AstItem*>::const_iterator it = start; it != stop; ++it) {
        AstItem* item = *it;
        AstDot* dot = dynamic_cast<AstDot*>(item);
        if (dot) {
            name += ".";
        }
        else {
            AstIdentifier* id = dynamic_cast<AstIdentifier*>(item);
            if (id) {
                name += id->text();
            }
        }
    }
    return name;
}

string AstName::type() const
{
    return "Name";
}

void AstName::getProcessed(Processor* const proc)
{
    proc->process(this);
}

AstNameDotClass::AstNameDotClass(AstName* const item0, 
             AstDot* const item1,
             AstClass* const item2,
             File* const srcfile) :
        AstNode(item0, item1, item2, srcfile), name_(item0), dot_(item1), class_(item2)
{
}

AstNameDotClass::~AstNameDotClass()
{
}

AstName* AstNameDotClass::getName() const
{
    return name_;
}

AstDot* AstNameDotClass::getDot() const
{
    return dot_;
}

AstClass* AstNameDotClass::getClass() const
{
    return class_;
}

void AstNameDotClass::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstNameDotClass::type() const
{
    return "NameDotClass";
}

AstNameDotThis::AstNameDotThis(AstName* const item0, 
             AstDot* const item1,
             AstThis* const item2,
             File* const srcfile) :
        AstNode(item0, item1, item2, srcfile), name_(item0), dot_(item1), this_(item2)
{
}

AstNameDotThis::~AstNameDotThis()
{
}

AstName* AstNameDotThis::getName() const
{
    return name_;
}

AstDot* AstNameDotThis::getDot() const
{
    return dot_;
}

AstThis* AstNameDotThis::getThis() const
{
    return this_;
}

void AstNameDotThis::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstNameDotThis::type() const
{
    return "NameDotThis";
}

AstNameList::AstNameList(AstName* const n, 
             File* const srcfile) :
        AstNode(n, srcfile)
{
    names_.push_back(n);
}

AstNameList::~AstNameList()
{
}

AstName* AstNameList::getName(int index) const
{
    return names_[index];
}

int AstNameList::getNameCount() const
{
    return names_.size();
}

void AstNameList::add(AstName* const n)
{
    names_.push_back(n);
    AstNode::append(n);
}

AstComma* AstNameList::getComma(int index) const
{
    return commas_[index];
}

int AstNameList::getCommaCount() const
{
    return commas_.size();
}

void AstNameList::add(AstComma* const c)
{
    commas_.push_back(c);
    AstNode::append(c);
}

void AstNameList::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstNameList::type() const
{
    return "NameList";
}

AstNative::AstNative(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstNative::~AstNative()
{
}

string AstNative::type() const
{
    return "NATIVE";
}

void AstNative::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstNative::length() const
{
    return 6;
}

AstNew::AstNew(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstNew::~AstNew()
{
}

string AstNew::type() const
{
    return "NEW";
}

void AstNew::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstNew::length() const
{
    return 3;
}

AstNode::AstNode(File* const srcfile /* = NULL */) : AstItem(srcfile)
{
}

AstNode::AstNode(AstItem* const it0,
                 File* const srcfile /* = NULL */) : AstItem(srcfile)
{
    append(it0);
}

AstNode::AstNode(AstItem* const it0, 
                 AstItem* const it1,
                 File* const srcfile /* = NULL */) : AstItem(srcfile)
{
    append(it0);
    append(it1);
}

AstNode::AstNode(AstItem* const it0, 
                 AstItem* const it1,
                 AstItem* const it2,
                 File* const srcfile /* = NULL */) : AstItem(srcfile)
{
    append(it0);
    append(it1);
    append(it2);
}

AstNode::AstNode(AstItem* const it0, 
                 AstItem* const it1,
                 AstItem* const it2,
                 AstItem* const it3,
                 File* const srcfile /* = NULL */) : AstItem(srcfile)
{
    append(it0);
    append(it1);
    append(it2);
    append(it3);
}

AstNode::AstNode(AstItem* const it0, 
                 AstItem* const it1,
                 AstItem* const it2,
                 AstItem* const it3,
                 AstItem* const it4,
                 File* const srcfile /* = NULL */) : AstItem(srcfile)
{
    append(it0);
    append(it1);
    append(it2);
    append(it3);
    append(it4);
}

AstNode::AstNode(AstItem* const it0, 
                 AstItem* const it1,
                 AstItem* const it2,
                 AstItem* const it3,
                 AstItem* const it4,
                 AstItem* const it5,
                 File* const srcfile /* = NULL */) : AstItem(srcfile)
{
    append(it0);
    append(it1);
    append(it2);
    append(it3);
    append(it4);
    append(it5);
}

AstNode::AstNode(AstItem* const it0, 
                 AstItem* const it1,
                 AstItem* const it2,
                 AstItem* const it3,
                 AstItem* const it4,
                 AstItem* const it5,
                 AstItem* const it6,
                 File* const srcfile /* = NULL */) : AstItem(srcfile)
{
    append(it0);
    append(it1);
    append(it2);
    append(it3);
    append(it4);
    append(it5);
    append(it6);
}

AstNode::AstNode(AstItem* const it0, 
                 AstItem* const it1,
                 AstItem* const it2,
                 AstItem* const it3,
                 AstItem* const it4,
                 AstItem* const it5,
                 AstItem* const it6,
                 AstItem* const it7,
                 File* const srcfile /* = NULL */) : AstItem(srcfile)
{
    append(it0);
    append(it1);
    append(it2);
    append(it3);
    append(it4);
    append(it5);
    append(it6);
    append(it7);
}

AstNode::AstNode(AstItem* const it0, 
                 AstItem* const it1,
                 AstItem* const it2,
                 AstItem* const it3,
                 AstItem* const it4,
                 AstItem* const it5,
                 AstItem* const it6,
                 AstItem* const it7,
                 AstItem* const it8,
                 File* const srcfile /* = NULL */) : AstItem(srcfile)
{
    append(it0);
    append(it1);
    append(it2);
    append(it3);
    append(it4);
    append(it5);
    append(it6);
    append(it7);
    append(it8);
}

AstNode::AstNode(AstItem* const it0, 
                 AstItem* const it1,
                 AstItem* const it2,
                 AstItem* const it3,
                 AstItem* const it4,
                 AstItem* const it5,
                 AstItem* const it6,
                 AstItem* const it7,
                 AstItem* const it8,
                 AstItem* const it9,
                 File* const srcfile /* = NULL */) : AstItem(srcfile)
{
    append(it0);
    append(it1);
    append(it2);
    append(it3);
    append(it4);
    append(it5);
    append(it6);
    append(it7);
    append(it8);
    append(it9);
}

AstNode::~AstNode()
{
    list<AstItem*>::iterator stop = components_.end();
    for (list<AstItem*>::iterator cit = components_.begin(); cit != stop; ++cit) {
        AstItem* component = *cit;
        if (component) {
            delete component;
        }
    }
}

template <typename IterType>
static AstItem* findItem(const IterType& start, 
                         const IterType& stop, 
                         const AstItem* const it)
{
    // Note that "prev" may actually be "next", depending on the direction of
    // the iterator (const_iterator vs. const_reverse_iterator.

    AstItem* prev = NULL;

    for (IterType cit = start; cit != stop; ++cit) {
        AstItem* itm = *cit;
        if (itm == it) {
            return prev;
        }
        else {
            prev = itm;
        }
    }

    // we didn't find it
    return NULL;
}

AstItem* AstNode::findPrevious(const AstItem* const it) const
{
    list<AstItem*>::const_iterator stop  = components().end();
    list<AstItem*>::const_iterator start = components().begin();
    return findItem(start, stop, it);
}

AstItem* AstNode::findNext(const AstItem* const it) const
{
    // same as findPreviousNode, but iterating backward.
    list<AstItem*>::const_reverse_iterator stop  = components().rend();
    list<AstItem*>::const_reverse_iterator start = components().rbegin();
    return findItem(start, stop, it);
}

AstNoncode* AstNode::leadingNoncode() const
{
    // this is the leading noncode block from the first component.
    if (components().begin() == components().end()) {
        return NULL;
    }
    else {
        list<AstItem*>::const_iterator it = components().begin();
        AstItem*    first = *it;
        AstNoncode* nc    = first->leadingNoncode();
        return nc;
    }
}

AstNoncode* AstNode::trailingNoncode() const
{
    // this is the leading noncode block from the next node
    if (AstItem* nxt = next()) {
        return nxt->leadingNoncode();
    }
    else {
        return NULL;
    }
}

char* AstNode::position() const
{
    if (components().begin() == components().end()) {
        DEBUG_ANY(cout << "warning: seeking position without components for " << type() << endl);
        // this shouldn't happen!
        return NULL;
    }
    else {
        return (*components().begin())->position();
    }
}

const list<AstItem*>& AstNode::components() const
{
    return components_;
}

list<AstItem*>& AstNode::components()
{
    return components_;
}

list<AstItem*>::iterator AstNode::findPosition(AstItem* const item) 
{
    for (list<AstItem*>::iterator it = components().begin(); it != components().end(); ++it) {
        AstItem* i = *it;
        if (i == item) {
            return it;
        }
    }
    return components().end();
}

/**
 * Adds the given item to the list, after the other, or at the end of the
 * list.
 */
void AstNode::append(AstItem* const item, AstItem* const prev /* = NULL */) 
{
    if (item) {
        item->setParent(this);
        if (prev) {
            list<AstItem*>::iterator it = findPosition(prev);
            if (it != components().end()) {
                ++it;
            }
            components().insert(it, item);
        }
        else {
            components().push_back(item);
        }
    }
}

/**
 * Inserts the item, before <code>next</code>, or at the head of the list if
 * <code>next</code> is NULL.
 */
void AstNode::prepend(AstItem* const item, AstItem* const next /* = NULL */) 
{
    if (item) {
        item->setParent(this);
        if (next) {
            list<AstItem*>::iterator it = findPosition(next);
            components().insert(it, item);
        }
        else {
            components().push_front(item);
        }
    }
}

char* AstNode::endPosition() const
{
    list<AstItem*>::const_iterator it = components().end();
    --it;
    AstItem* item = *it;
    return item ? item->endPosition() : NULL;
}

AstNoncode::AstNoncode(char* start, char* end, File* const srcfile) :
        AstNode(srcfile), 
     start_(start),
     end_(end),
     comments_(NULL)
{    
    DEBUG_NONCODE({
        cout << "Noncode(" << (unsigned int)(start) << ", " << (unsigned int)(end) << ")" << endl;
        string nc(start, end - start);
        int stline = srcfile->lineOf(start);
        int stcol  = srcfile->columnOf(start);
        int endline = srcfile->lineOf(end);
        int endcol  = srcfile->columnOf(end);
        cout << "got noncode (" << stline << ":" << stcol << " - "
             << endline << ":" << endcol
             << " = '" << nc
             << "')" << endl;
    });
}

AstNoncode::~AstNoncode()
{
    // comments are deleted by the superclass.
}

void AstNoncode::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstNoncode::type() const
{
    return "noncode";
}

string AstNoncode::text() const
{
    string txt(start_, end_ - start_);
    return txt;
}

vector<AstComment*> AstNoncode::comments() const
{
    if (!comments_) {
        comments_ = new vector<AstComment*>;

        // parsing it here... what a mess, but it should be as fast as can be
        // expected

        for (char* p = start_; p < end_; ++p) {
            if (*p == '/') {
                AstComment* cmt = NULL;
                char* start = p;
                if (*(p + 1) == '*') {
                    p += 2;
                    while (!(*p == '*' && *(p + 1) == '/')) {
                        p++;
                    }
                    p += 2;
                    char* to = p;
                    if (*(start + 2) == '*') {
                        cmt = new AstJavadocComment(start, to, sourceFile());
                    }
                    else {
                        cmt = new AstCComment(start, to, sourceFile());
                    }
                }
                else if (*(p + 1) == '/') {
                    while (++p < end_ && *p != '\n') {
                        // nothing
                    }
                    char* to = p;
                    cmt = new AstCppComment(start, to, sourceFile());
                }
                if (cmt) {
                    comments_->push_back(cmt);
                    // ugh. have to cast to a nonconst object.
                    AstNoncode* ncthis = const_cast<AstNoncode*>(this);
                    ncthis->append(cmt);
                }
            }
        }
    }
    return *comments_;
}

const list<AstItem*>& AstNoncode::components() const
{
    comments();
    return SUPERCLASS::components();
}

list<AstItem*>& AstNoncode::components()
{
    comments();
    return SUPERCLASS::components();
}

char* AstNoncode::position() const
{
    return start_;
}

char* AstNoncode::endPosition() const
{
    return end_;
}

AstNot::AstNot(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstNot::~AstNot()
{
}

string AstNot::type() const
{
    return "NOT";
}

void AstNot::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstNot::length() const
{
    return 1;
}

AstNoteq::AstNoteq(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstNoteq::~AstNoteq()
{
}

string AstNoteq::type() const
{
    return "NOTEQ";
}

void AstNoteq::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstNoteq::length() const
{
    return 2;
}

AstNullLiteral::AstNullLiteral(char* position, File* const srcfile) :
        AstLiteral("null", position, srcfile)
{
}

AstNullLiteral::~AstNullLiteral()
{
}

string AstNullLiteral::type() const
{
    return "NULL_LITERAL";
}

void AstNullLiteral::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstNullLiteral::length() const
{
    return 4;
}

AstNumericType::AstNumericType(char* position, File* const srcfile) :
        AstPrimitiveType(position, srcfile)
{
}

AstNumericType::~AstNumericType()
{
}

AstOr::AstOr(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstOr::~AstOr()
{
}

string AstOr::type() const
{
    return "OR";
}

void AstOr::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstOr::length() const
{
    return 1;
}

AstOreq::AstOreq(char* position, File* const srcfile) :
        AstAssignmentOperator(position, srcfile)
{
}

AstOreq::~AstOreq()
{
}

string AstOreq::type() const
{
    return "OREQ";
}

void AstOreq::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstOreq::length() const
{
    return 2;
}

AstOrExpression::AstOrExpression(AstItem* const le, 
                                 AstOr* const o,
                                 AstItem* const re,
                                 File* const srcfile /* = NULL */) :
        AstNode(le, o, re, srcfile), lhsExpression_(le), or_(o), rhsExpression_(re)
{
}

AstOrExpression::~AstOrExpression()
{
}

AstItem* AstOrExpression::getLhsExpression() const
{
    return lhsExpression_;
}

AstOr* AstOrExpression::getOr() const
{
    return or_;
}

AstItem* AstOrExpression::getRhsExpression() const
{
    return rhsExpression_;
}

void AstOrExpression::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstOrExpression::type() const
{
    return "OrExpression";
}

AstOror::AstOror(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstOror::~AstOror()
{
}

string AstOror::type() const
{
    return "OROR";
}

void AstOror::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstOror::length() const
{
    return 2;
}

AstPackage::AstPackage(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstPackage::~AstPackage()
{
}

string AstPackage::type() const
{
    return "PACKAGE";
}

void AstPackage::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstPackage::length() const
{
    return 7;
}

AstPackageDeclaration::AstPackageDeclaration(AstPackage* const p, 
             AstName* const n,
             AstSemicolon* const s,
             File* const srcfile) :
        AstNode(p, n, s, srcfile), package_(p), name_(n), semicolon_(s)
{
}

AstPackageDeclaration::~AstPackageDeclaration()
{
}

AstPackage* AstPackageDeclaration::getPackage() const
{
    return package_;
}

AstName* AstPackageDeclaration::getName() const
{
    return name_;
}

AstSemicolon* AstPackageDeclaration::getSemicolon() const
{
    return semicolon_;
}

void AstPackageDeclaration::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstPackageDeclaration::type() const
{
    return "PackageDeclaration";
}

AstPlus::AstPlus(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstPlus::~AstPlus()
{
}

string AstPlus::type() const
{
    return "PLUS";
}

void AstPlus::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstPlus::length() const
{
    return 1;
}

AstPluseq::AstPluseq(char* position, File* const srcfile) :
        AstAssignmentOperator(position, srcfile)
{
}

AstPluseq::~AstPluseq()
{
}

string AstPluseq::type() const
{
    return "PLUSEQ";
}

void AstPluseq::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstPluseq::length() const
{
    return 2;
}

AstPlusplus::AstPlusplus(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstPlusplus::~AstPlusplus()
{
}

string AstPlusplus::type() const
{
    return "PLUSPLUS";
}

void AstPlusplus::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstPlusplus::length() const
{
    return 2;
}

AstPostdecrementExpression::AstPostdecrementExpression(AstItem* const e, 
             AstMinusminus* const m,
             File* const srcfile /* = NULL */) :
        AstNode(e, m, srcfile), expression_(e), minusminus_(m)
{
}

AstPostdecrementExpression::~AstPostdecrementExpression()
{
}

AstItem* AstPostdecrementExpression::getExpression() const
{
    return expression_;
}

AstMinusminus* AstPostdecrementExpression::getMinusminus() const
{
    return minusminus_;
}

void AstPostdecrementExpression::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstPostdecrementExpression::type() const
{
    return "PostdecrementExpression";
}

AstPostincrementExpression::AstPostincrementExpression(AstItem* const e, 
             AstPlusplus* const p,
             File* const srcfile /* = NULL */) :
        AstNode(e, p, srcfile), expression_(e), plusplus_(p)
{
}

AstPostincrementExpression::~AstPostincrementExpression()
{
}

AstItem* AstPostincrementExpression::getExpression() const
{
    return expression_;
}

AstPlusplus* AstPostincrementExpression::getPlusplus() const
{
    return plusplus_;
}

void AstPostincrementExpression::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstPostincrementExpression::type() const
{
    return "PostincrementExpression";
}

AstPredecrementExpression::AstPredecrementExpression(AstMinusminus* const m, 
             AstItem* const e,
             File* const srcfile /* = NULL */) :
        AstNode(m, e, srcfile), minusminus_(m), expression_(e)
{
}

AstPredecrementExpression::~AstPredecrementExpression()
{
}

AstMinusminus* AstPredecrementExpression::getMinusminus() const
{
    return minusminus_;
}

AstItem* AstPredecrementExpression::getExpression() const
{
    return expression_;
}

void AstPredecrementExpression::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstPredecrementExpression::type() const
{
    return "PredecrementExpression";
}

AstPreincrementExpression::AstPreincrementExpression(AstPlusplus* const p, 
             AstItem* const e,
             File* const srcfile /* = NULL */) :
        AstNode(p, e, srcfile), plusplus_(p), expression_(e)
{
}

AstPreincrementExpression::~AstPreincrementExpression()
{
}

AstPlusplus* AstPreincrementExpression::getPlusplus() const
{
    return plusplus_;
}

AstItem* AstPreincrementExpression::getExpression() const
{
    return expression_;
}

void AstPreincrementExpression::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstPreincrementExpression::type() const
{
    return "PreincrementExpression";
}

AstPrimaryParenExpression::AstPrimaryParenExpression(AstLparen* const l, 
                                               AstItem* const e,
                                               AstRparen* const r,
                                               File* const srcfile) :
        AstNode(l, e, r, srcfile), lparen_(l), expression_(e), rparen_(r)
{
}

AstPrimaryParenExpression::~AstPrimaryParenExpression()
{
}

AstLparen* AstPrimaryParenExpression::getLparen() const
{
    return lparen_;
}

AstItem* AstPrimaryParenExpression::getExpression() const
{
    return expression_;
}

AstRparen* AstPrimaryParenExpression::getRparen() const
{
    return rparen_;
}

void AstPrimaryParenExpression::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstPrimaryParenExpression::type() const
{
    return "PrimaryParenExpression";
}

AstPrimitiveDotClass::AstPrimitiveDotClass(AstPrimitiveType* const pt, 
                                     AstDot* const d,
                                     AstClass* const c,
                                     File* const srcfile) :
        AstNode(pt, d, c, srcfile), primitiveType_(pt), dot_(d), class_(c)
{
}

AstPrimitiveDotClass::~AstPrimitiveDotClass()
{
}

AstPrimitiveType* AstPrimitiveDotClass::getPrimitiveType() const
{
    return primitiveType_;
}

AstDot* AstPrimitiveDotClass::getDot() const
{
    return dot_;
}

AstClass* AstPrimitiveDotClass::getClass() const
{
    return class_;
}

void AstPrimitiveDotClass::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstPrimitiveDotClass::type() const
{
    return "PrimitiveDotClass";
}

AstPrimitiveType::AstPrimitiveType(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstPrimitiveType::~AstPrimitiveType()
{
}

AstPrivate::AstPrivate(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstPrivate::~AstPrivate()
{
}

string AstPrivate::type() const
{
    return "PRIVATE";
}

void AstPrivate::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstPrivate::length() const
{
    return 7;
}

AstProject::AstProject(AstCompilationUnit* const cu) : AstNode(cu)
{
    compilationUnits_.push_back(cu);
}

AstProject::AstProject(double javaVersion, const vector<char*>& files) : javaVersion_(javaVersion)
{
    process(files);
}

AstProject::~AstProject()
{
}

AstCompilationUnit* AstProject::getCompilationUnit(int index) const
{
    return compilationUnits_[index];
}

int AstProject::getCompilationUnitCount() const
{
    return compilationUnits_.size();
}

void AstProject::add(AstCompilationUnit* const cu)
{
    compilationUnits_.push_back(cu);
    AstNode::append(cu);
}

void AstProject::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstProject::type() const
{
    return "Project";
}

void AstProject::process(const string& fname)
{
    process(new File(fname));
}

void AstProject::process(File* const file)
{
    files_.push_back(file);
    parser.parse(compUnit, file);
    if (compUnit) {
        add(compUnit);
    }
    else {
        DEBUG_PARSER(cerr << "doctorj: no compilation unit; error during parsing" << endl);
    }
}

void AstProject::process(const vector<char*>& files)
{
    int nFiles = files.size();
    DEBUG_ANY(cout << nFiles << " files (non-options)" << endl);

    parser.setJavaVersion(javaVersion_);
    for (int i = 0; i < nFiles; ++i) {
        DEBUG_ANY(cout << "doctorj: file " << files[i] << ": " << i << " of " << nFiles << endl);
        try {
            string name = files[i];
            if (Platform::isDirectory(name)) {
                DEBUG_ANY(cout << "processing directory..." << endl);
                vector<string> javas;
                Platform::findFiles(name, "java", &javas);
                vector<string>::iterator it   = javas.begin();
                vector<string>::iterator stop = javas.end();
                while (it != stop) {
                    string fname = *it;
                    process(fname);
                    ++it;
                }
            }
            else if (Platform::isReadableFile(name)) {
                process(name);
            }
            else if (Platform::isRegularFile(name)) {
                cerr << "doctorj: " << name << ": file not readable" << endl;
            }
            else {
                cerr << "doctorj: " << name << ": no such file or directory" << endl;
            }
        }
        catch (const FileException& fe) {
            cout << "error opening " << files[i] << endl;
        }
    }
}    

vector<File*> AstProject::files() const
{
    return files_;
}

AstProtected::AstProtected(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstProtected::~AstProtected()
{
}

string AstProtected::type() const
{
    return "PROTECTED";
}

void AstProtected::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstProtected::length() const
{
    return 9;
}

AstPublic::AstPublic(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstPublic::~AstPublic()
{
}

string AstPublic::type() const
{
    return "PUBLIC";
}

void AstPublic::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstPublic::length() const
{
    return 6;
}

AstQuestion::AstQuestion(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstQuestion::~AstQuestion()
{
}

string AstQuestion::type() const
{
    return "QUESTION";
}

void AstQuestion::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstQuestion::length() const
{
    return 1;
}

AstRbrace::AstRbrace(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstRbrace::~AstRbrace()
{
}

string AstRbrace::type() const
{
    return "RBRACE";
}

void AstRbrace::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstRbrace::length() const
{
    return 1;
}

AstRbracket::AstRbracket(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstRbracket::~AstRbracket()
{
}

string AstRbracket::type() const
{
    return "RBRACKET";
}

void AstRbracket::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstRbracket::length() const
{
    return 1;
}

AstRelationalExpression::AstRelationalExpression(AstItem* const lhex,
                                                 AstItem* const rhex,
                                                 File* const srcfile) :
        AstNode(lhex, rhex, srcfile), lhsExpression_(lhex), rhsExpression_(rhex)
{
}

AstRelationalExpression::~AstRelationalExpression()
{
}

AstItem* AstRelationalExpression::getLhsExpression() const
{
    return lhsExpression_;
}

AstItem* AstRelationalExpression::getRhsExpression() const
{
    return rhsExpression_;
}

AstRelationalExpressionGt::AstRelationalExpressionGt(AstItem* const lhex,
                                                     AstGt* const gt,
                                                     AstItem* const rhex,
                                                     File* const srcfile) :
        AstRelationalExpression(lhex, rhex, srcfile), gt_(gt)
{
    append(gt, lhex);
}

AstRelationalExpressionGt::~AstRelationalExpressionGt()
{
}

AstGt* AstRelationalExpressionGt::getGt() const
{
    return gt_;
}

void AstRelationalExpressionGt::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstRelationalExpressionGt::type() const
{
    return "RelationalExpressionGt";
}

AstRelationalExpressionGteq::AstRelationalExpressionGteq(AstItem* const lhex,
                                                         AstGteq* const gt,
                                                         AstItem* const rhex,
                                                         File* const srcfile) :
        AstRelationalExpression(lhex, rhex, srcfile), gteq_(gt)
{
    append(gt, lhex);
}

AstRelationalExpressionGteq::~AstRelationalExpressionGteq()
{
}

AstGteq* AstRelationalExpressionGteq::getGteq() const
{
    return gteq_;
}

void AstRelationalExpressionGteq::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstRelationalExpressionGteq::type() const
{
    return "RelationalExpressionGteq";
}

AstRelationalExpressionInstanceof::AstRelationalExpressionInstanceof(AstItem* const lhex,
                                                                     AstInstanceof* const in,
                                                                     AstItem* const rhex,
                                                                     File* const srcfile) :
        AstRelationalExpression(lhex, rhex, srcfile), instanceof_(in)
{
    append(in, lhex);
}

AstRelationalExpressionInstanceof::~AstRelationalExpressionInstanceof()
{
}

AstInstanceof* AstRelationalExpressionInstanceof::getInstanceof() const
{
    return instanceof_;
}

void AstRelationalExpressionInstanceof::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstRelationalExpressionInstanceof::type() const
{
    return "RelationalExpressionInstanceof";
}

AstRelationalExpressionLt::AstRelationalExpressionLt(AstItem* const lhex,
                                                     AstLt* const lt,
                                                     AstItem* const rhex,
                                                     File* const srcfile) :
        AstRelationalExpression(lhex, rhex, srcfile), lt_(lt)
{
    append(lt, lhex);
}

AstRelationalExpressionLt::~AstRelationalExpressionLt()
{
}

AstLt* AstRelationalExpressionLt::getLt() const
{
    return lt_;
}

void AstRelationalExpressionLt::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstRelationalExpressionLt::type() const
{
    return "RelationalExpressionLt";
}

AstRelationalExpressionLteq::AstRelationalExpressionLteq(AstItem* const lhex,
                                                         AstLteq* const lt,
                                                         AstItem* const rhex,
                                                         File* const srcfile) :
        AstRelationalExpression(lhex, rhex, srcfile), lteq_(lt)
{
    append(lt, lhex);
}

AstRelationalExpressionLteq::~AstRelationalExpressionLteq()
{
}

AstLteq* AstRelationalExpressionLteq::getLteq() const
{
    return lteq_;
}

void AstRelationalExpressionLteq::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstRelationalExpressionLteq::type() const
{
    return "RelationalExpressionLteq";
}

AstReturn::AstReturn(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstReturn::~AstReturn()
{
}

string AstReturn::type() const
{
    return "RETURN";
}

void AstReturn::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstReturn::length() const
{
    return 6;
}

AstReturnStatement::AstReturnStatement(AstReturn* const r, 
                                       AstItem* const e,
                                       AstSemicolon* const s,
                                       File* const srcfile) :
        AstNode(r, e, s, srcfile), return_(r), expression_(e), semicolon_(s)
{
}

AstReturnStatement::~AstReturnStatement()
{
}

AstReturn* AstReturnStatement::getReturn() const
{
    return return_;
}

AstItem* AstReturnStatement::getExpression() const
{
    return expression_;
}

AstSemicolon* AstReturnStatement::getSemicolon() const
{
    return semicolon_;
}

void AstReturnStatement::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstReturnStatement::type() const
{
    return "ReturnStatement";
}

AstReturnStatementEmpty::AstReturnStatementEmpty(AstReturn* const r, 
                                                 AstSemicolon* const s,
                                                 File* const srcfile) :
        AstReturnStatement(r, NULL, s, srcfile)
{
}

AstReturnStatementEmpty::~AstReturnStatementEmpty()
{
}

void AstReturnStatementEmpty::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstReturnStatementEmpty::type() const
{
    return "ReturnStatementEmpty";
}

AstReturnStatementExpr::AstReturnStatementExpr(AstReturn* const r, 
                                               AstItem* const e,
                                               AstSemicolon* const s,
                                               File* const srcfile) :
        AstReturnStatement(r, e, s, srcfile)
{
}

AstReturnStatementExpr::~AstReturnStatementExpr()
{
}

void AstReturnStatementExpr::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstReturnStatementExpr::type() const
{
    return "ReturnStatementExpr";
}

AstRparen::AstRparen(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstRparen::~AstRparen()
{
}

string AstRparen::type() const
{
    return "RPAREN";
}

void AstRparen::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstRparen::length() const
{
    return 1;
}

AstRshift::AstRshift(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstRshift::~AstRshift()
{
}

string AstRshift::type() const
{
    return "RSHIFT";
}

void AstRshift::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstRshift::length() const
{
    return 2;
}

AstRshifteq::AstRshifteq(char* position, File* const srcfile) :
        AstAssignmentOperator(position, srcfile)
{
}

AstRshifteq::~AstRshifteq()
{
}

string AstRshifteq::type() const
{
    return "RSHIFTEQ";
}

void AstRshifteq::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstRshifteq::length() const
{
    return 3;
}

AstSemicolon::AstSemicolon(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstSemicolon::~AstSemicolon()
{
}

string AstSemicolon::type() const
{
    return "SEMICOLON";
}

void AstSemicolon::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstSemicolon::length() const
{
    return 1;
}

AstShiftExpression::AstShiftExpression(AstItem* const lhex,
                                       AstItem* const rhex,
                                       File* const srcfile) :
        AstNode(lhex, rhex, srcfile), lhsExpression_(lhex), rhsExpression_(rhex)
{
}

AstShiftExpression::~AstShiftExpression()
{
}

AstItem* AstShiftExpression::getLhsExpression() const
{
    return lhsExpression_;
}

AstItem* AstShiftExpression::getRhsExpression() const
{
    return rhsExpression_;
}

AstShiftExpressionLshift::AstShiftExpressionLshift(AstItem* const lhex,
                                                   AstLshift* const ls,
                                                   AstItem* const rhex,
                                                   File* const srcfile) :
        AstShiftExpression(lhex, rhex, srcfile), lshift_(ls)
{
    append(ls, lhex);
}

AstShiftExpressionLshift::~AstShiftExpressionLshift()
{
}

AstLshift* AstShiftExpressionLshift::getLshift() const
{
    return lshift_;
}

void AstShiftExpressionLshift::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstShiftExpressionLshift::type() const
{
    return "ShiftExpressionLshift";
}

AstShiftExpressionRshift::AstShiftExpressionRshift(AstItem* const lhex,
                                                   AstRshift* const rs,
                                                   AstItem* const rhex,
                                                   File* const srcfile) :
        AstShiftExpression(lhex, rhex, srcfile), rshift_(rs)
{
    append(rs, lhex);
}

AstShiftExpressionRshift::~AstShiftExpressionRshift()
{
}

AstRshift* AstShiftExpressionRshift::getRshift() const
{
    return rshift_;
}

void AstShiftExpressionRshift::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstShiftExpressionRshift::type() const
{
    return "ShiftExpressionRshift";
}

AstShiftExpressionUrshift::AstShiftExpressionUrshift(AstItem* const lhex,
                                                     AstUrshift* const ur,
                                                     AstItem* const rhex,
                                                     File* const srcfile) :
        AstShiftExpression(lhex, rhex, srcfile), urshift_(ur)
{
    append(ur, lhex);
}

AstShiftExpressionUrshift::~AstShiftExpressionUrshift()
{
}

AstUrshift* AstShiftExpressionUrshift::getUrshift() const
{
    return urshift_;
}

void AstShiftExpressionUrshift::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstShiftExpressionUrshift::type() const
{
    return "ShiftExpressionUrshift";
}

AstShort::AstShort(char* position, File* const srcfile) :
        AstIntegralType(position, srcfile)
{
}

AstShort::~AstShort()
{
}

string AstShort::type() const
{
    return "SHORT";
}

void AstShort::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstShort::length() const
{
    return 5;
}

AstStatementExpressionList::AstStatementExpressionList(AstItem* const se, 
             File* const srcfile) :
        AstNode(se, srcfile)
{
    statementExpressions_.push_back(se);
}

AstStatementExpressionList::~AstStatementExpressionList()
{
}

AstItem* AstStatementExpressionList::getStatementExpression(int index) const
{
    return statementExpressions_[index];
}

int AstStatementExpressionList::getStatementExpressionCount() const
{
    return statementExpressions_.size();
}

void AstStatementExpressionList::add(AstItem* const se)
{
    statementExpressions_.push_back(se);
    AstNode::append(se);
}

AstComma* AstStatementExpressionList::getComma(int index) const
{
    return commas_[index];
}

int AstStatementExpressionList::getCommaCount() const
{
    return commas_.size();
}

void AstStatementExpressionList::add(AstComma* const c)
{
    commas_.push_back(c);
    AstNode::append(c);
}

void AstStatementExpressionList::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstStatementExpressionList::type() const
{
    return "StatementExpressionList";
}

AstStatic::AstStatic(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstStatic::~AstStatic()
{
}

string AstStatic::type() const
{
    return "STATIC";
}

void AstStatic::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstStatic::length() const
{
    return 6;
}

AstStaticInitializer::AstStaticInitializer(AstStatic* const s, 
                                           AstBlock* const b,
                                           File* const srcfile) :
        AstNode(s, b, srcfile), static_(s), block_(b)
{
}

AstStaticInitializer::~AstStaticInitializer()
{
}

AstStatic* AstStaticInitializer::getStatic() const
{
    return static_;
}

AstBlock* AstStaticInitializer::getBlock() const
{
    return block_;
}

void AstStaticInitializer::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstStaticInitializer::type() const
{
    return "StaticInitializer";
}

AstStrictfp::AstStrictfp(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstStrictfp::~AstStrictfp()
{
}

string AstStrictfp::type() const
{
    return "STRICTFP";
}

void AstStrictfp::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstStrictfp::length() const
{
    return 8;
}

AstStringLeaf::AstStringLeaf(char* start, char* end, File* const srcfile) :
        AstLeaf(start, srcfile),
        end_(end)
{
}

AstStringLeaf::~AstStringLeaf()
{
}

void AstStringLeaf::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstStringLeaf::length() const
{
    return 1 + end_ - position();
}

string AstStringLeaf::str() const
{
    char* start = position();
    return string(start, 1 + end_ - start);
}

char* AstStringLeaf::endPosition() const
{
    return end_;
}

bool AstStringLeaf::operator==(const string& s) const
{
    return str() == s;
}

bool AstStringLeaf::operator!=(const string& s) const
{
    return str() != s;
}

string AstStringLeaf::type() const
{
    return "StringLeaf";
}

AstStringLiteral::AstStringLiteral(const string& text, char* position, File* const srcfile) :
        AstLiteral(text, position, srcfile)
{
}

AstStringLiteral::~AstStringLiteral()
{
}

string AstStringLiteral::type() const
{
    return "STRING_LITERAL";
}

void AstStringLiteral::getProcessed(Processor* const proc)
{
    proc->process(this);
}

AstSuper::AstSuper(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstSuper::~AstSuper()
{
}

string AstSuper::type() const
{
    return "SUPER";
}

void AstSuper::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstSuper::length() const
{
    return 5;
}

AstSwitchBlock::AstSwitchBlock(AstLbrace* const lb,
                               AstSwitchBlockStatementGroupList* const swblstgrli,
                               AstSwitchLabelList* const swlali,
                               AstRbrace* const rb,
                               File* const srcfile) :
        AstNode(lb, swblstgrli, swlali, rb, srcfile), 
     lbrace_(lb), 
     switchBlockStatementGroupList_(swblstgrli), 
     switchLabelList_(swlali), 
     rbrace_(rb)
{
}

AstSwitchBlock::~AstSwitchBlock()
{
}

AstLbrace* AstSwitchBlock::getLbrace() const
{
    return lbrace_;
}

AstSwitchBlockStatementGroupList* AstSwitchBlock::getSwitchBlockStatementGroupList() const
{
    return switchBlockStatementGroupList_;
}

AstSwitchLabelList* AstSwitchBlock::getSwitchLabelList() const
{
    return switchLabelList_;
}

AstRbrace* AstSwitchBlock::getRbrace() const
{
    return rbrace_;
}

AstSwitchBlockEmpty::AstSwitchBlockEmpty(AstLbrace* const lb,
                                         AstRbrace* const rb,
                                         File* const srcfile) :
        AstSwitchBlock(lb, NULL, NULL, rb, srcfile)
{
}

AstSwitchBlockEmpty::~AstSwitchBlockEmpty()
{
}

void AstSwitchBlockEmpty::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstSwitchBlockEmpty::type() const
{
    return "SwitchBlockEmpty";
}

AstSwitchBlockGroups::AstSwitchBlockGroups(AstLbrace* const lb,
                                           AstSwitchBlockStatementGroupList* const swblstgrli,
                                           AstRbrace* const rb,
                                           File* const srcfile) :
        AstSwitchBlock(lb, swblstgrli, NULL, rb, srcfile)
{
}

AstSwitchBlockGroups::~AstSwitchBlockGroups()
{
}

void AstSwitchBlockGroups::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstSwitchBlockGroups::type() const
{
    return "SwitchBlockGroups";
}

AstSwitchBlockGroupsLabels::AstSwitchBlockGroupsLabels(AstLbrace* const lb,
                                                       AstSwitchBlockStatementGroupList* const swblstgrli,
                                                       AstSwitchLabelList* const swlali,
                                                       AstRbrace* const rb,
                                                 File* const srcfile) :
        AstSwitchBlock(lb, swblstgrli, swlali, rb, srcfile)
{
}

AstSwitchBlockGroupsLabels::~AstSwitchBlockGroupsLabels()
{
}

void AstSwitchBlockGroupsLabels::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstSwitchBlockGroupsLabels::type() const
{
    return "SwitchBlockGroupsLabels";
}

AstSwitchBlockLabels::AstSwitchBlockLabels(AstLbrace* const lb,
                                           AstSwitchLabelList* const swlali,
                                           AstRbrace* const rb,
                                           File* const srcfile) :
        AstSwitchBlock(lb, NULL, swlali, rb, srcfile)
{
}

AstSwitchBlockLabels::~AstSwitchBlockLabels()
{
}

AstSwitchLabelList* AstSwitchBlockLabels::getSwitchLabelList() const
{
    return switchLabelList_;
}

void AstSwitchBlockLabels::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstSwitchBlockLabels::type() const
{
    return "SwitchBlockLabels";
}

AstSwitchBlockStatementGroup::AstSwitchBlockStatementGroup(AstSwitchLabelList* const sll, 
             AstBlockStatementList* const bsl,
             File* const srcfile) :
        AstNode(sll, bsl, srcfile), switchLabelList_(sll), blockStatementList_(bsl)
{
}

AstSwitchBlockStatementGroup::~AstSwitchBlockStatementGroup()
{
}

AstSwitchLabelList* AstSwitchBlockStatementGroup::getSwitchLabelList() const
{
    return switchLabelList_;
}

AstBlockStatementList* AstSwitchBlockStatementGroup::getBlockStatementList() const
{
    return blockStatementList_;
}

void AstSwitchBlockStatementGroup::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstSwitchBlockStatementGroup::type() const
{
    return "SwitchBlockStatementGroup";
}

AstSwitchBlockStatementGroupList::AstSwitchBlockStatementGroupList(AstSwitchBlockStatementGroup* const sbsg, 
             File* const srcfile) :
        AstNode(sbsg, srcfile)
{
    switchBlockStatementGroups_.push_back(sbsg);
}

AstSwitchBlockStatementGroupList::~AstSwitchBlockStatementGroupList()
{
}

AstSwitchBlockStatementGroup* AstSwitchBlockStatementGroupList::getSwitchBlockStatementGroup(int index) const
{
    return switchBlockStatementGroups_[index];
}

int AstSwitchBlockStatementGroupList::getSwitchBlockStatementGroupCount() const
{
    return switchBlockStatementGroups_.size();
}

void AstSwitchBlockStatementGroupList::add(AstSwitchBlockStatementGroup* const sbsg)
{
    switchBlockStatementGroups_.push_back(sbsg);
    AstNode::append(sbsg);
}

void AstSwitchBlockStatementGroupList::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstSwitchBlockStatementGroupList::type() const
{
    return "SwitchBlockStatementGroupList";
}

AstSwitch::AstSwitch(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstSwitch::~AstSwitch()
{
}

string AstSwitch::type() const
{
    return "SWITCH";
}

void AstSwitch::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstSwitch::length() const
{
    return 6;
}

AstSwitchLabelCase::AstSwitchLabelCase(AstCase* const ca,
                                 AstItem* const ex,
                                 AstColon* const co,
                                 File* const srcfile) :
        AstSwitchLabel(co, srcfile), case_(ca), expression_(ex)
{
    prepend(ca);
    append(ex, ca);
}

AstSwitchLabelCase::~AstSwitchLabelCase()
{
}

AstCase* AstSwitchLabelCase::getCase() const
{
    return case_;
}

AstItem* AstSwitchLabelCase::getExpression() const
{
    return expression_;
}

void AstSwitchLabelCase::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstSwitchLabelCase::type() const
{
    return "SwitchLabelCase";
}

AstSwitchLabel::AstSwitchLabel(AstColon* const co,
                               File* const srcfile) :
        AstNode(co, srcfile), colon_(co)
{
}

AstSwitchLabel::~AstSwitchLabel()
{
}

AstColon* AstSwitchLabel::getColon() const
{
    return colon_;
}

AstSwitchLabelDefault::AstSwitchLabelDefault(AstDefault* const de,
                                       AstColon* const co,
                                       File* const srcfile) :
        AstSwitchLabel(co, srcfile), default_(de)
{
    prepend(de);
}

AstSwitchLabelDefault::~AstSwitchLabelDefault()
{
}

AstDefault* AstSwitchLabelDefault::getDefault() const
{
    return default_;
}

void AstSwitchLabelDefault::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstSwitchLabelDefault::type() const
{
    return "SwitchLabelDefault";
}

AstSwitchLabelList::AstSwitchLabelList(AstSwitchLabel* const sl, 
             File* const srcfile) :
        AstNode(sl, srcfile)
{
    switchLabels_.push_back(sl);
}

AstSwitchLabelList::~AstSwitchLabelList()
{
}

AstSwitchLabel* AstSwitchLabelList::getSwitchLabel(int index) const
{
    return switchLabels_[index];
}

int AstSwitchLabelList::getSwitchLabelCount() const
{
    return switchLabels_.size();
}

void AstSwitchLabelList::add(AstSwitchLabel* const sl)
{
    switchLabels_.push_back(sl);
    AstNode::append(sl);
}

void AstSwitchLabelList::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstSwitchLabelList::type() const
{
    return "SwitchLabelList";
}

AstSwitchStatement::AstSwitchStatement(AstSwitch* const s, 
             AstLparen* const l,
             AstItem* const e,
             AstRparen* const r,
             AstSwitchBlock* const sb,
             File* const srcfile) :
        AstNode(s, l, e, r, sb, srcfile),
        switch_(s), 
        lparen_(l), 
        expression_(e), 
        rparen_(r), 
        switchBlock_(sb)
{
}

AstSwitchStatement::~AstSwitchStatement()
{
}

AstSwitch* AstSwitchStatement::getSwitch() const
{
    return switch_;
}

AstLparen* AstSwitchStatement::getLparen() const
{
    return lparen_;
}

AstItem* AstSwitchStatement::getExpression() const
{
    return expression_;
}

AstRparen* AstSwitchStatement::getRparen() const
{
    return rparen_;
}

AstSwitchBlock* AstSwitchStatement::getSwitchBlock() const
{
    return switchBlock_;
}

void AstSwitchStatement::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstSwitchStatement::type() const
{
    return "SwitchStatement";
}

AstSynchronized::AstSynchronized(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstSynchronized::~AstSynchronized()
{
}

string AstSynchronized::type() const
{
    return "SYNCHRONIZED";
}

void AstSynchronized::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstSynchronized::length() const
{
    return 12;
}

AstSynchronizedStatement::AstSynchronizedStatement(AstSynchronized* const s, 
             AstLparen* const l,
             AstItem* const e,
             AstRparen* const r,
             AstBlock* const b,
             File* const srcfile) :
        AstNode(s, l, e, r, b, srcfile),
        synchronized_(s), 
        lparen_(l), 
        expression_(e), 
        rparen_(r), 
        block_(b)
{
}

AstSynchronizedStatement::~AstSynchronizedStatement()
{
}

AstSynchronized* AstSynchronizedStatement::getSynchronized() const
{
    return synchronized_;
}

AstLparen* AstSynchronizedStatement::getLparen() const
{
    return lparen_;
}

AstItem* AstSynchronizedStatement::getExpression() const
{
    return expression_;
}

AstRparen* AstSynchronizedStatement::getRparen() const
{
    return rparen_;
}

AstBlock* AstSynchronizedStatement::getBlock() const
{
    return block_;
}

void AstSynchronizedStatement::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstSynchronizedStatement::type() const
{
    return "SynchronizedStatement";
}

AstTaggedComment::AstTaggedComment(char* tagStart, char* tagEnd, 
                                   File* const srcfile) :
        AstComment(tagStart, tagEnd, srcfile), tagStart_(tagStart), tagEnd_(tagEnd)
{
}

AstTaggedComment::AstTaggedComment(char* start, char* end,
                                   char* tagStart, char* tagEnd, 
                                   File* const srcfile) :
        AstComment(start, end, srcfile), tagStart_(tagStart), tagEnd_(tagEnd)
{
}

AstTaggedComment::~AstTaggedComment()
{
}

string AstTaggedComment::tag() const
{
    return string(tagStart_, 1 + tagEnd_ - tagStart_);
}

string AstTaggedComment::type() const 
{
    return "Tagged comment";
}

ostream& AstTaggedComment::print(ostream& os) const
{
    os << "TaggedComment" << endl;
    os << "type       : "  << type()        << endl;
    os << "tag        : '" << tag()         << "'" << endl;
    return os;
}

char* AstTaggedComment::tagStart() const
{
    return tagStart_;
}

char* AstTaggedComment::tagEnd() const
{
    return tagEnd_;
}

AstStringLeaf AstTaggedComment::tagLeaf() const
{
    return AstStringLeaf(tagStart_, tagEnd_, sourceFile());
}

int AstTaggedComment::countTargets() const
{
    return 0;
}

void AstTaggedComment::getProcessed(Processor* const p)
{
    p->process(this);
}

/**
 * Tag targets may be one of three forms:
 *
 * @html <a href="http://www.foo.org/something.html">An HTML Target</a>
 * @quoted "A Quoted Target"
 * @word Word
 */
enum TagTargetType { HTML, QUOTED, WORD };

AstTaggedDescribedComment::AstTaggedDescribedComment(char* tagStart, char* tagEnd, 
                                                     char* descriptionStart, char* descriptionEnd, 
                                                     File* const srcfile) :
        AstTaggedComment(tagStart, descriptionEnd, tagStart, tagEnd, srcfile),
     descriptionStart_(descriptionStart),
     descriptionEnd_(descriptionEnd),
     targetStart_(NULL),
     targetEnd_(NULL),
     isParsed_(false)
{
}

AstTaggedDescribedComment::~AstTaggedDescribedComment()
{
}

string AstTaggedDescribedComment::target() const
{
    if (!isParsed_) {
        // cast "this" to nonconst to call the nonconst parse method (ugh.)
        AstTaggedDescribedComment* ncthis = const_cast<AstTaggedDescribedComment*>(this);
        ncthis->parse();
    }

    if (targetStart_ && targetEnd_) {
        return string(targetStart_, 1 + targetEnd_ - targetStart_);
    }
    else {
        return "";
    }
}

string AstTaggedDescribedComment::description() const
{
    if (descriptionStart_ && descriptionEnd_) {
        return string(descriptionStart_, 1 + descriptionEnd_ - descriptionStart_);
    }
    else {
        return "";
    }
}

string AstTaggedDescribedComment::type() const 
{
    return "Tagged Described comment";
}

void AstTaggedDescribedComment::parse()
{
    char* start = descriptionStart_;
    char* end = descriptionEnd_;
    TagTargetType type;
    if (!strncasecmp(start, "<a", 2)) {
        type = HTML;
    }
    else if (*start == '"') {
        type = QUOTED;
    }
    else {
        type = WORD;
    }

    // also handle targets with balanced parentheses, for example:
    //     @see set(int, double, java.net.Socket)

    bool inParen = false;

    char* p = start;
    while (p <= end) {
        if (*p == '\\' && p < end - 1) {
            ++p;
        }
        else if (type == WORD) {
            if (*p == '(') {
                inParen = true;
            }
            else if (inParen && *p == ')') {
                inParen = false;
            }
            if (!inParen) {
                if (p == end) {
                    // we'll never get a space, because we're at the end
                    targetStart_ = start;
                    targetEnd_ = p;
                    break;
                }
                else if (isspace(*p)) {
                    // we have a space between the target and the next word
                    targetStart_ = start;
                    targetEnd_ = p - 1;
                    break;
                }
            }
        }
        else if (type == HTML && *p == '>' && toupper(*(p - 1)) == 'A') {
            // HTML target
            targetStart_ = start;
            targetEnd_ = p;
            break;
        }
        else if (type == QUOTED && *p == '"') {
            // quoted target
            targetStart_ = start;
            targetEnd_ = p;
            break;
        }
        ++p;
    }

    if (type == HTML || type == QUOTED && start && !targetEnd_) {
        // special case for a target that is unbalanced HTML or quotes
        targetStart_ = start;
        targetEnd_ = end;
    }

    isParsed_ = true;
}

ostream& AstTaggedDescribedComment::print(ostream& os) const
{
    os << "TaggedDescribedComment" << endl;
    os << "orig       : "  << string(descriptionStart_, descriptionEnd_ - descriptionStart_ + 1) << endl;
    os << "type       : "  << type()        << endl;
    os << "tag        : '" << tag()         << "'" << endl;
    os << "description: '" << description() << "'" << endl;
    os << "target     : '" << target()      << "'" << endl;
    return os;
}

char* AstTaggedDescribedComment::descriptionStart() const
{
    return descriptionStart_;
}

char* AstTaggedDescribedComment::descriptionEnd() const
{
    return descriptionEnd_;
}

char* AstTaggedDescribedComment::targetStart() const
{
    if (!isParsed_) {
        AstTaggedDescribedComment* ncthis = const_cast<AstTaggedDescribedComment*>(this);
        ncthis->parse();
    }
    return targetStart_;
}

char* AstTaggedDescribedComment::targetEnd() const
{
    if (!isParsed_) {
        AstTaggedDescribedComment* ncthis = const_cast<AstTaggedDescribedComment*>(this);
        ncthis->parse();
    }
    return targetEnd_;
}

AstStringLeaf AstTaggedDescribedComment::targetLeaf() const
{
    return AstStringLeaf(targetStart_, targetEnd_, sourceFile());
}

int AstTaggedDescribedComment::countTargets() const
{
    // check that the description after the targetEnd has at least one word.
    char* tgtEnd = targetEnd();
    char* descEnd = descriptionEnd();

    if (tgtEnd == NULL) {
        // how would this happen?
        return 0;
    }
    else if (tgtEnd >= descEnd) {
        // only one target
        return 1;
    }
    else {
        string descAfterTarget(tgtEnd, 1 + descEnd - tgtEnd);
        return StringUtilities::countWords(descAfterTarget);
    }
}

void AstTaggedDescribedComment::getProcessed(Processor* const p)
{
    p->process(this);
}

AstThis::AstThis(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstThis::~AstThis()
{
}

string AstThis::type() const
{
    return "THIS";
}

void AstThis::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstThis::length() const
{
    return 4;
}

AstThrow::AstThrow(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstThrow::~AstThrow()
{
}

string AstThrow::type() const
{
    return "THROW";
}

void AstThrow::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstThrow::length() const
{
    return 5;
}

AstThrows::AstThrows(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstThrows::~AstThrows()
{
}

string AstThrows::type() const
{
    return "THROWS";
}

void AstThrows::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstThrows::length() const
{
    return 6;
}

AstThrowsNameList::AstThrowsNameList(AstThrows* const t, 
                               AstNameList* const nl,
                               File* const srcfile) :
        AstNode(t, nl, srcfile), throws_(t), nameList_(nl)
{
}

AstThrowsNameList::~AstThrowsNameList()
{
}

AstThrows* AstThrowsNameList::getThrows() const
{
    return throws_;
}

AstNameList* AstThrowsNameList::getNameList() const
{
    return nameList_;
}

void AstThrowsNameList::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstThrowsNameList::type() const
{
    return "ThrowsNameList";
}

AstThrowStatement::AstThrowStatement(AstThrow* const t, 
             AstItem* const e,
             AstSemicolon* const s,
             File* const srcfile) :
        AstNode(t, e, s, srcfile), throw_(t), expression_(e), semicolon_(s)
{
}

AstThrowStatement::~AstThrowStatement()
{
}

AstThrow* AstThrowStatement::getThrow() const
{
    return throw_;
}

AstItem* AstThrowStatement::getExpression() const
{
    return expression_;
}

AstSemicolon* AstThrowStatement::getSemicolon() const
{
    return semicolon_;
}

void AstThrowStatement::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstThrowStatement::type() const
{
    return "ThrowStatement";
}

AstTransient::AstTransient(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstTransient::~AstTransient()
{
}

string AstTransient::type() const
{
    return "TRANSIENT";
}

void AstTransient::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstTransient::length() const
{
    return 9;
}

AstTrueLiteral::AstTrueLiteral(char* position, File* const srcfile) :
        AstBooleanLiteral("true", position, srcfile)
{
}

AstTrueLiteral::~AstTrueLiteral()
{
}

string AstTrueLiteral::type() const
{
    return "TRUE_LITERAL";
}

void AstTrueLiteral::getProcessed(Processor* const proc)
{
    proc->process(this);
}

AstTry::AstTry(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstTry::~AstTry()
{
}

string AstTry::type() const
{
    return "TRY";
}

void AstTry::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstTry::length() const
{
    return 3;
}

AstTryStatementCatches::AstTryStatementCatches(AstTry* const tr,
                                               AstBlock* const bl,
                                               AstCatchClauseList* const caclli,
                                               File* const srcfile) :
        AstTryStatement(tr, bl, caclli, NULL, srcfile)
{
}

AstTryStatementCatches::~AstTryStatementCatches()
{
}

void AstTryStatementCatches::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstTryStatementCatches::type() const
{
    return "TryStatementCatches";
}

AstTryStatementCatchesFinally::AstTryStatementCatchesFinally(AstTry* const tr,
                                                             AstBlock* const bl,
                                                             AstCatchClauseList* const caclli,
                                                             AstFinallyBlock* const fibl,
                                                             File* const srcfile) :
        AstTryStatement(tr, bl, caclli, fibl, srcfile)
{
}

AstTryStatementCatchesFinally::~AstTryStatementCatchesFinally()
{
}

void AstTryStatementCatchesFinally::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstTryStatementCatchesFinally::type() const
{
    return "TryStatementCatchesFinally";
}

AstTryStatement::AstTryStatement(AstTry* const tr,
                                 AstBlock* const bl,
                                 AstCatchClauseList* const caclli,
                                 AstFinallyBlock* const fibl,
                                 File* const srcfile) :
        AstNode(tr, bl, caclli, fibl, srcfile), try_(tr), block_(bl), catchClauseList_(caclli), finallyBlock_(fibl)
{
}

AstTryStatement::~AstTryStatement()
{
}

AstTry* AstTryStatement::getTry() const
{
    return try_;
}

AstBlock* AstTryStatement::getBlock() const
{
    return block_;
}

AstCatchClauseList* AstTryStatement::getCatchClauseList() const
{
    return catchClauseList_;
}

AstFinallyBlock* AstTryStatement::getFinallyBlock() const
{
    return finallyBlock_;
}

AstTryStatementFinally::AstTryStatementFinally(AstTry* const tr,
                                               AstBlock* const bl,
                                               AstFinallyBlock* const fibl,
                                               File* const srcfile) :
        AstTryStatement(tr, bl, NULL, fibl, srcfile)
{
}

AstTryStatementFinally::~AstTryStatementFinally()
{
}

void AstTryStatementFinally::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstTryStatementFinally::type() const
{
    return "TryStatementFinally";
}

AstTypeDeclarationList::AstTypeDeclarationList(AstItem* const td, 
             File* const srcfile) :
        AstNode(td, srcfile)
{
    typeDeclarations_.push_back(td);
}

AstTypeDeclarationList::~AstTypeDeclarationList()
{
}

AstItem* AstTypeDeclarationList::getTypeDeclaration(int index) const
{
    return typeDeclarations_[index];
}

int AstTypeDeclarationList::getTypeDeclarationCount() const
{
    return typeDeclarations_.size();
}

void AstTypeDeclarationList::add(AstItem* const td)
{
    typeDeclarations_.push_back(td);
    AstNode::append(td);
}

void AstTypeDeclarationList::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstTypeDeclarationList::type() const
{
    return "TypeDeclarationList";
}

AstUnaryExpressionComp::AstUnaryExpressionComp(AstComp* const co,
                                         AstItem* const ex,
                                         File* const srcfile) :
        AstUnaryExpression(ex, srcfile), comp_(co)
{
    prepend(co);
}

AstUnaryExpressionComp::~AstUnaryExpressionComp()
{
}

AstComp* AstUnaryExpressionComp::getComp() const
{
    return comp_;
}

void AstUnaryExpressionComp::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstUnaryExpressionComp::type() const
{
    return "UnaryExpressionComp";
}

AstUnaryExpression::AstUnaryExpression(AstItem* const ex,
                                       File* const srcfile) :
        AstNode(ex, srcfile), expression_(ex)
{
}

AstUnaryExpression::~AstUnaryExpression()
{
}

AstItem* AstUnaryExpression::getExpression() const
{
    return expression_;
}

AstUnaryExpressionMinus::AstUnaryExpressionMinus(AstMinus* const mi,
                                           AstItem* const ex,
                                           File* const srcfile) :
        AstUnaryExpression(ex, srcfile), minus_(mi)
{
    prepend(mi);
}

AstUnaryExpressionMinus::~AstUnaryExpressionMinus()
{
}

AstMinus* AstUnaryExpressionMinus::getMinus() const
{
    return minus_;
}

void AstUnaryExpressionMinus::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstUnaryExpressionMinus::type() const
{
    return "UnaryExpressionMinus";
}

AstUnaryExpressionNot::AstUnaryExpressionNot(AstNot* const no,
                                       AstItem* const ex,
                                       File* const srcfile) :
        AstUnaryExpression(ex, srcfile), not_(no)
{
    prepend(no);
}

AstUnaryExpressionNot::~AstUnaryExpressionNot()
{
}

AstNot* AstUnaryExpressionNot::getNot() const
{
    return not_;
}

void AstUnaryExpressionNot::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstUnaryExpressionNot::type() const
{
    return "UnaryExpressionNot";
}

AstUnaryExpressionPlus::AstUnaryExpressionPlus(AstPlus* const pl,
                                         AstItem* const ex,
                                         File* const srcfile) :
        AstUnaryExpression(ex, srcfile), plus_(pl)
{
    prepend(pl);
}

AstUnaryExpressionPlus::~AstUnaryExpressionPlus()
{
}

AstPlus* AstUnaryExpressionPlus::getPlus() const
{
    return plus_;
}

void AstUnaryExpressionPlus::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstUnaryExpressionPlus::type() const
{
    return "UnaryExpressionPlus";
}

AstUrshift::AstUrshift(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstUrshift::~AstUrshift()
{
}

string AstUrshift::type() const
{
    return "URSHIFT";
}

void AstUrshift::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstUrshift::length() const
{
    return 3;
}

AstUrshifteq::AstUrshifteq(char* position, File* const srcfile) :
        AstAssignmentOperator(position, srcfile)
{
}

AstUrshifteq::~AstUrshifteq()
{
}

string AstUrshifteq::type() const
{
    return "URSHIFTEQ";
}

void AstUrshifteq::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstUrshifteq::length() const
{
    return 4;
}

AstValue::AstValue(const string& text, char* position, File* const srcfile) :
        AstLeaf(position, srcfile),
     text_(text)
{
}

AstValue::~AstValue()
{
}

string AstValue::text() const
{
    return text_;
}

int AstValue::length() const
{
    return text_.length();
}

AstVariableDeclaratorAssign::AstVariableDeclaratorAssign(AstVariableDeclaratorId* const vadeid,
                                                         AstEq* const eq,
                                                         AstItem* const vain,
                                                         File* const srcfile) :
        AstVariableDeclarator(vadeid, eq, vain, srcfile)
{
    append(eq, vadeid);
    append(vain, eq);
}

AstVariableDeclaratorAssign::~AstVariableDeclaratorAssign()
{
}

void AstVariableDeclaratorAssign::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstVariableDeclaratorAssign::type() const
{
    return "VariableDeclaratorAssign";
}

AstVariableDeclarator::AstVariableDeclarator(AstVariableDeclaratorId* const vadeid,
                                             File* const srcfile) :
        AstNode(vadeid, srcfile), variableDeclaratorId_(vadeid), eq_(NULL), variableInitializer_(NULL)
{
}

AstVariableDeclarator::AstVariableDeclarator(AstVariableDeclaratorId* const vadeid,
                                             AstEq* const eq,
                                             AstItem* const vain,
                                             File* const srcfile) :
        AstNode(vadeid, srcfile), variableDeclaratorId_(vadeid), eq_(eq), variableInitializer_(vain)
{
}

AstVariableDeclarator::~AstVariableDeclarator()
{
}

AstVariableDeclaratorId* AstVariableDeclarator::getVariableDeclaratorId() const
{
    return variableDeclaratorId_;
}

AstEq* AstVariableDeclarator::getEq() const
{
    return eq_;
}

AstItem* AstVariableDeclarator::getVariableInitializer() const
{
    return variableInitializer_;
}

string AstVariableDeclarator::name() const
{
    AstIdentifier* id = getVariableDeclaratorId()->getIdentifier();
    return id->text();
}

AstVariableDeclaratorId::AstVariableDeclaratorId(AstIdentifier* const id,
                                                 AstDimensionList* const dili,
                                                 File* const srcfile) :
        AstNode(id, dili, srcfile), identifier_(id), dimensionList_(dili)
{
}

AstVariableDeclaratorId::~AstVariableDeclaratorId()
{
}

AstIdentifier* AstVariableDeclaratorId::getIdentifier() const
{
    return identifier_;
}

string AstVariableDeclaratorId::name() const
{
    AstIdentifier* id = getIdentifier();
    string name = id->text();
    return name;
}

AstDimensionList* AstVariableDeclaratorId::getDimensionList() const
{
    return dimensionList_;
}

AstVariableDeclaratorIdDimensions::AstVariableDeclaratorIdDimensions(AstIdentifier* const id,
                                                                     AstDimensionList* const dili,
                                                                     File* const srcfile) :
        AstVariableDeclaratorId(id, dili, srcfile)
{
}

AstVariableDeclaratorIdDimensions::~AstVariableDeclaratorIdDimensions()
{
}

void AstVariableDeclaratorIdDimensions::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstVariableDeclaratorIdDimensions::type() const
{
    return "VariableDeclaratorIdDimensions";
}

AstVariableDeclaratorIdNoDimensions::AstVariableDeclaratorIdNoDimensions(AstIdentifier* const id,
                                                                         File* const srcfile) :
        AstVariableDeclaratorId(id, NULL, srcfile)
{
}

AstVariableDeclaratorIdNoDimensions::~AstVariableDeclaratorIdNoDimensions()
{
}

void AstVariableDeclaratorIdNoDimensions::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstVariableDeclaratorIdNoDimensions::type() const
{
    return "VariableDeclaratorIdNoDimensions";
}

AstVariableDeclaratorList::AstVariableDeclaratorList(AstVariableDeclarator* const vd, 
                                               File* const srcfile) :
        AstNode(vd, srcfile)
{
    variableDeclarators_.push_back(vd);
}

AstVariableDeclaratorList::~AstVariableDeclaratorList()
{
}

AstVariableDeclarator* AstVariableDeclaratorList::getVariableDeclarator(int index) const
{
    return variableDeclarators_[index];
}

int AstVariableDeclaratorList::getVariableDeclaratorCount() const
{
    return variableDeclarators_.size();
}

void AstVariableDeclaratorList::add(AstVariableDeclarator* const vd)
{
    variableDeclarators_.push_back(vd);
    AstNode::append(vd);
}

AstComma* AstVariableDeclaratorList::getComma(int index) const
{
    return commas_[index];
}

int AstVariableDeclaratorList::getCommaCount() const
{
    return commas_.size();
}

void AstVariableDeclaratorList::add(AstComma* const c)
{
    commas_.push_back(c);
    AstNode::append(c);
}

void AstVariableDeclaratorList::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstVariableDeclaratorList::type() const
{
    return "VariableDeclaratorList";
}

AstVariableDeclaratorNoAssign::AstVariableDeclaratorNoAssign(AstVariableDeclaratorId* const vadeid,
                                                       File* const srcfile) :
        AstVariableDeclarator(vadeid, srcfile)
{
}

AstVariableDeclaratorNoAssign::~AstVariableDeclaratorNoAssign()
{
}

void AstVariableDeclaratorNoAssign::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstVariableDeclaratorNoAssign::type() const
{
    return "VariableDeclaratorNoAssign";
}

AstVariableInitializerList::AstVariableInitializerList(AstItem* const vi, 
                                                 File* const srcfile) :
        AstNode(vi, srcfile)
{
    variableInitializers_.push_back(vi);
}

AstVariableInitializerList::~AstVariableInitializerList()
{
}

AstItem* AstVariableInitializerList::getVariableInitializer(int index) const
{
    return variableInitializers_[index];
}

int AstVariableInitializerList::getVariableInitializerCount() const
{
    return variableInitializers_.size();
}

void AstVariableInitializerList::add(AstItem* const vi)
{
    variableInitializers_.push_back(vi);
    AstNode::append(vi);
}

AstComma* AstVariableInitializerList::getComma(int index) const
{
    return commas_[index];
}

int AstVariableInitializerList::getCommaCount() const
{
    return commas_.size();
}

void AstVariableInitializerList::add(AstComma* const c)
{
    commas_.push_back(c);
    AstNode::append(c);
}

void AstVariableInitializerList::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstVariableInitializerList::type() const
{
    return "VariableInitializerList";
}

AstVoid::AstVoid(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstVoid::~AstVoid()
{
}

string AstVoid::type() const
{
    return "VOID";
}

void AstVoid::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstVoid::length() const
{
    return 4;
}

AstVoidDotClass::AstVoidDotClass(AstVoid* const item0, 
             AstDot* const item1,
             AstClass* const item2,
             File* const srcfile) :
        AstNode(item0, item1, item2, srcfile), void_(item0), dot_(item1), class_(item2)
{
}

AstVoidDotClass::~AstVoidDotClass()
{
}

AstVoid* AstVoidDotClass::getVoid() const
{
    return void_;
}

AstDot* AstVoidDotClass::getDot() const
{
    return dot_;
}

AstClass* AstVoidDotClass::getClass() const
{
    return class_;
}

void AstVoidDotClass::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstVoidDotClass::type() const
{
    return "VoidDotClass";
}

AstVolatile::AstVolatile(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstVolatile::~AstVolatile()
{
}

string AstVolatile::type() const
{
    return "VOLATILE";
}

void AstVolatile::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstVolatile::length() const
{
    return 8;
}

AstWhile::AstWhile(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstWhile::~AstWhile()
{
}

string AstWhile::type() const
{
    return "WHILE";
}

void AstWhile::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstWhile::length() const
{
    return 5;
}

AstWhileStatement::AstWhileStatement(AstWhile* const w, 
                                     AstLparen* const l,
                                     AstItem* const e,
                                     AstRparen* const r,
                                     AstItem* const s,
                                     File* const srcfile) :
        AstNode(w, l, e, r, s, srcfile),
     while_(w), 
     lparen_(l), 
     expression_(e), 
     rparen_(r), 
     statement_(s)
{
}

AstWhileStatement::~AstWhileStatement()
{
}

AstWhile* AstWhileStatement::getWhile() const
{
    return while_;
}

AstLparen* AstWhileStatement::getLparen() const
{
    return lparen_;
}

AstItem* AstWhileStatement::getExpression() const
{
    return expression_;
}

AstRparen* AstWhileStatement::getRparen() const
{
    return rparen_;
}

AstItem* AstWhileStatement::getStatement() const
{
    return statement_;
}

void AstWhileStatement::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstWhileStatement::type() const
{
    return "WhileStatement";
}

AstXor::AstXor(char* position, File* const srcfile) :
        AstLeaf(position, srcfile)
{
}

AstXor::~AstXor()
{
}

string AstXor::type() const
{
    return "XOR";
}

void AstXor::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstXor::length() const
{
    return 1;
}

AstXoreq::AstXoreq(char* position, File* const srcfile) :
        AstAssignmentOperator(position, srcfile)
{
}

AstXoreq::~AstXoreq()
{
}

string AstXoreq::type() const
{
    return "XOREQ";
}

void AstXoreq::getProcessed(Processor* const proc)
{
    proc->process(this);
}

int AstXoreq::length() const
{
    return 2;
}

AstXorExpression::AstXorExpression(AstItem* const le, 
                                   AstXor* const x,
                                   AstItem* const re,
                                   File* const srcfile /* = NULL */) :
        AstNode(le, x, re, srcfile), lhsExpression_(le), xor_(x), rhsExpression_(re)
{
}

AstXorExpression::~AstXorExpression()
{
}

AstItem* AstXorExpression::getLhsExpression() const
{
    return lhsExpression_;
}

AstXor* AstXorExpression::getXor() const
{
    return xor_;
}

AstItem* AstXorExpression::getRhsExpression() const
{
    return rhsExpression_;
}

void AstXorExpression::getProcessed(Processor* const proc)
{
    proc->process(this);
}

string AstXorExpression::type() const
{
    return "XorExpression";
}
