#ifndef MethodDocumentationAnalyzer_h
#include "MethodDocumentationAnalyzer.h"
#endif

#ifndef AST_h
#include "AST.h"
#endif

#ifndef DocumentationErrors_h
#include "DocumentationErrors.h"
#endif

using namespace doctorj;

MethodDocumentationAnalyzer::MethodDocumentationAnalyzer(Reporter* const reporter,
                                                         AstMethodHeader* const header, 
                                                         AstItem* const returnType,
                                                         AstModifierList* const modifiers,
                                                         AstFormalParameterList* const parameters,
                                                         AstThrowsNameList* const throwsList) :
        MethodCtorDocumentationAnalyzer(reporter, 
                                        modifiers, 
                                        parameters,
                                        throwsList),
     header_(header),
     returnType_(returnType),
     gotReturnDoc_(false)
{
    check();
}

MethodDocumentationAnalyzer::~MethodDocumentationAnalyzer()
{
}

void MethodDocumentationAnalyzer::checkReturn(AstTaggedComment* const tc) 
{
    gotReturnDoc_ = true;
    if (returnType_) {
        AstTaggedDescribedComment* tdc = dynamic_cast<AstTaggedDescribedComment*>(tc);
        if (!tdc) {
            AstStringLeaf tlf = tc->tagLeaf();
            ErrorReturnWithoutDescription err(reporter(), &tlf);
            err.process();
        }
        else {
            // check for the return type being used as a target.
            string tgt = tdc->target();
            string retType = returnType_->text();
            if (tgt == retType) {
                AstStringLeaf tlf = tdc->targetLeaf();
                ErrorReturnTypeUsedAsTarget err(reporter(), &tlf, retType);
                err.process();
            }
        }
    }
    else {
        AstStringLeaf tlf = tc->tagLeaf();
        ErrorReturnDocumentedForMethodReturningVoid err(reporter(), &tlf);
        err.process();
    }
}

void MethodDocumentationAnalyzer::checkUndocumentedReturn()
{
    if (returnType_ && !gotReturnDoc_) {
        string msg("return not documented");
        ErrorReturnUndocumented err(reporter(), returnType_);
        err.process();
    }
}

string MethodDocumentationAnalyzer::type() const
{
    return "method";
}

AstItem* MethodDocumentationAnalyzer::getSubject()
{
    AstIdentifier* subject = header_->getMethodDeclarator()->getIdentifier();
    return subject;
}

AstFormalParameterList* MethodDocumentationAnalyzer::getFormalParameterList()
{
    AstMethodDeclarator* md = header_->getMethodDeclarator();
    AstMethodDeclaratorWArgs* mda = dynamic_cast<AstMethodDeclaratorWArgs*>(md);
    return mda == NULL ? NULL : mda->getFormalParameterList();
}

AstNoncode* MethodDocumentationAnalyzer::leadingNoncode()
{
    return header_->leadingNoncode();
}

ErrorInvalidJavadocTag* MethodDocumentationAnalyzer::makeUnknownTagError(AstItem* const item, 
                                                                         const string& tag) const
{
    return new ErrorInvalidJavadocTagMethod(reporter(), item, tag);
}
