#ifndef DocumentationErrors_h
#define DocumentationErrors_h

#ifndef Error_h
#include "Error.h"
#endif

#ifndef std_vector
#define std_vector
#include <vector>
#endif

namespace doctorj
{
    class Reporter;
    class StringEdit;

    class DocumentationError : public Error
    {
    public:
        DocumentationError();

        DocumentationError(Reporter* const reporter, 
                           AstItem* const item);

        virtual ~DocumentationError();

        virtual ErrorRegistry* getErrorRegistry() const;

        /**
         * The error level, 1 (mild warning), 2, or 3 (error). 
         */
        virtual int level() const = 0;

        /**
         * The number specific to the error type.
         */
        virtual int number() const = 0;

        /**
         * Returns the identifier.
         */
        virtual int identifier() const;

    protected:
        StringEdit* makeInsertion(const string& defaultValue) const;

        int getIndentation(AstItem* const context) const;

    };


    class ErrorAuthorWithoutNameText : public DocumentationError
    {
    public:
        ErrorAuthorWithoutNameText();

        /**
         * Constructs the error for the offensive item.
         */
        ErrorAuthorWithoutNameText(Reporter* const reporter, 
                                   AstItem* const item);
        
        virtual ~ErrorAuthorWithoutNameText();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;

        /**
         * Returns the edit, which would insert the default author text.
         */
        virtual StringEdit* getEdit();
            
    };


    class ErrorExceptionMismatchWithCode : public DocumentationError 
    {
    public:
        ErrorExceptionMismatchWithCode();

        /**
         * Takes a tag as the erroneous argument.
         */
        ErrorExceptionMismatchWithCode(Reporter* const reporter,
                                       AstItem* const item, 
                                       const string& tag);

        virtual ~ErrorExceptionMismatchWithCode();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * Returns the edit, which would remove the documented exception.
         */
        virtual StringEdit* getEdit();

    private:
        /**
         * The offensive tag.
         */
        string tag_;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorExceptionMisspelled : public DocumentationError
    {
    public:
        ErrorExceptionMisspelled();

        /**
         * Creates the error for the offensive set of parameters.
         */
        ErrorExceptionMisspelled(Reporter* const reporter, 
                                 AstItem* const item,
                                 const string& tag,
                                 const string& docThrow,
                                 const string& codeThrow);

        virtual ~ErrorExceptionMisspelled();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * Returns the edit, which would replace the misspelled exception.
         */
        virtual StringEdit* getEdit();

    private:

        /**
         * "@throw" or "@exception".
         */
        string tag_;

        /**
         * The exception name as in the documentation.
         */
        string docThrow_;

        /**
         * The exception name as in the code.
         */
        string codeThrow_;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorExceptionNotInCode : public DocumentationError
    {
    public:
        ErrorExceptionNotInCode();

        /**
         * Constructs the error for the offensive item and the given method
         * name.
         */
        ErrorExceptionNotInCode(Reporter* const reporter, 
                                AstItem* const item, 
                                const string& name);

        virtual ~ErrorExceptionNotInCode();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * Returns the edit, which would remove the documented exception.
         */
        virtual StringEdit* getEdit();

    private:

        /**
         * The method name.
         */
        string name_;

        /**
         * The error level (2). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class AstTaggedDescribedComment;

    class ErrorExceptionOrderNotAlphabetical : public DocumentationError
    {
    public:
        ErrorExceptionOrderNotAlphabetical();

        /**
         * Creates the error for the given tag ("@throws" or "@exception") and
         * documented throw name.
         */
        ErrorExceptionOrderNotAlphabetical(Reporter* const reporter, 
                                           AstTaggedDescribedComment* const current,
                                           AstItem* const item,
                                           const string& tag, 
                                           const string& docThrow,
                                           AstTaggedDescribedComment* const previous);

        virtual ~ErrorExceptionOrderNotAlphabetical();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * Returns the edit, which would remove the documented exception.
         *
         * @deprecated Not yet enabled.
         */
        // virtual StringEdit* getEdit();

    private:
        /**
         * The current exception in the documentation.
         */
        AstTaggedDescribedComment* current_;
        
        /**
         * The type of tag that was used.
         */
        string tag_;
        
        /**
         * The exception in the documentation.
         */
        string docThrow_;

        /**
         * The previous exception in the documentation.
         */
        AstTaggedDescribedComment* previous_;
        
        /**
         * The error level (2). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class AstName;
    class AstNoncode;
    class StringEditInsert;

    class ErrorExceptionUndocumented : public DocumentationError
    {
    public:
        ErrorExceptionUndocumented();

        ErrorExceptionUndocumented(Reporter* const reporter,
                                   AstNoncode* const noncode,
                                   AstName* const name, 
                                   const string& codeThrow);

        virtual ~ErrorExceptionUndocumented();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * Returns the edit for this error, which inserts the exception name
         * into the Javadoc comment.
         */
        StringEdit* getEdit();

    protected:
        /**
         * Creates an insertion edit for the exception.
         */
        StringEditInsert* makeInsert(char* pos) const;

    private:

        /**
         * The preceding noncode (comments and whitespace).
         */
        AstNoncode* noncode_;
        
        /**
         * The name of the exception, in the code.
         */
        string codeThrow_;

        /**
         * The error level (1). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };

    /**
     * Lack of a class name also implies lack of a description.
     */
    class ErrorExceptionWithoutClassName : public DocumentationError 
    {
    public:
        ErrorExceptionWithoutClassName();

        /**
         * Constructs the error for the offensive item.
         */
        ErrorExceptionWithoutClassName(Reporter* const reporter, 
                                       AstItem* const item, 
                                       const string& tag);

        virtual ~ErrorExceptionWithoutClassName();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * Returns the edit, which would remove the offensive line.
         */
        virtual StringEdit* getEdit();

    private:
        /**
         * The tag for the exception, either "@exception" or "@throws".
         */
        string tag_;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorExceptionWithoutDescription : public DocumentationError 
    {
    public:
        ErrorExceptionWithoutDescription();

        /**
         * Constructs the error for the offensive item.
         */
        ErrorExceptionWithoutDescription(Reporter* const reporter, 
                                         AstItem* const item, 
                                         const string& tag);

        virtual ~ErrorExceptionWithoutDescription();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * Inserts the default description, if any.
         */
        virtual StringEdit* getEdit();

    private:
        /**
         * The tag for the exception, either "@exception" or "@throws".
         */
        string tag_;
        
        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorFieldUndocumentedUncommented : public DocumentationError
    {
    public:
        ErrorFieldUndocumentedUncommented();

        /**
         * Creates the error, where the client has specified whether this was an
         * omission of documentation, or a commment altogether.
         */
        ErrorFieldUndocumentedUncommented(Reporter* const reporter, 
                                          AstItem* const item, 
                                          const string& itemType, 
                                          const string& errType,
                                          bool isPublic);

        virtual ~ErrorFieldUndocumentedUncommented();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * Returns the edit, which would insert the default field description.
         */
        virtual StringEdit* getEdit();
        
    private:
        /**
         * The type of the item.
         */
        string itemType_;

        /**
         * What type of error this was, i.e., "documentation" or "comment".
         */
        string errType_;

        /**
         * Whether the item has public access.
         */
        bool isPublic_;

        /**
         * The error level (1). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    
    class ErrorInvalidJavadocTag : public DocumentationError 
    {
    public:
        ErrorInvalidJavadocTag(Reporter* const reporter, 
                               AstItem* const item, 
                               const string& tag);

        /**
         * Exists only for the registration process.
         */
        ErrorInvalidJavadocTag();

        virtual ~ErrorInvalidJavadocTag();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message.
         */
        virtual string message() const;

    protected:

        /**
         * Returns the offensive tag.
         */
        virtual string tag() const;

    private:
        /**
         * The offensive tag.
         */
        string tag_;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorInvalidJavadocTagItem : public ErrorInvalidJavadocTag 
    {
    public:
        ErrorInvalidJavadocTagItem(Reporter* const reporter, 
                                   AstItem* const item, 
                                   const string& tag);

        /**
         * Exists only for the registration process.
         */
        ErrorInvalidJavadocTagItem();

        virtual ~ErrorInvalidJavadocTagItem();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message.
         */
        virtual string message() const;

        /**
         * By default, returns the valid Javadoc tags for any item. Subclasses
         * can define this for the specific type.
         */
        virtual vector<string> getValidTags() const = 0;

        /**
         * Deletes the line in question.
         */
        virtual StringEdit* getEdit();

    protected:

        /**
         * Returns the type of item to which this error applies.
         */
        virtual string type() const = 0;

        /**
         * Returns "a" or "an".
         */
        virtual string article() const;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorInvalidJavadocTagClass : public ErrorInvalidJavadocTagItem 
    {
    public:
        /**
         * Constructs the error for the given item and tag.
         */
        ErrorInvalidJavadocTagClass(Reporter* const reporter, 
                                    AstItem* const item, 
                                    const string& tag);

        /**
         * Exists only for the registration process.
         */
        ErrorInvalidJavadocTagClass();

        virtual ~ErrorInvalidJavadocTagClass();

        /**
         * Returns the valid Javadoc tags for a class.
         */
        virtual vector<string> getValidTags() const;

    protected:

        /**
         * Returns the message.
         */
        virtual string type() const;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorInvalidJavadocTagCtor : public ErrorInvalidJavadocTagItem 
    {
    public:
        /**
         * Constructs the error for the given item and tag.
         */
        ErrorInvalidJavadocTagCtor(Reporter* const reporter, 
                                   AstItem* const item, 
                                   const string& tag);

        /**
         * Exists only for the registration process.
         */
        ErrorInvalidJavadocTagCtor();

        virtual ~ErrorInvalidJavadocTagCtor();

        /**
         * Returns the valid Javadoc tags for a constructor.
         */
        virtual vector<string> getValidTags() const;

    protected:

        /**
         * Returns the message.
         */
        virtual string type() const;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorInvalidJavadocTagField : public ErrorInvalidJavadocTagItem 
    {
    public:
        /**
         * Constructs the error for the given item and tag.
         */
        ErrorInvalidJavadocTagField(Reporter* const reporter, 
                                    AstItem* const item, 
                                    const string& tag);

        /**
         * Exists only for the registration process.
         */
        ErrorInvalidJavadocTagField();

        virtual ~ErrorInvalidJavadocTagField();

        /**
         * Returns the valid Javadoc tags for a field.
         */
        virtual vector<string> getValidTags() const;

    protected:

        /**
         * Returns the message.
         */
        virtual string type() const;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorInvalidJavadocTagInterface : public ErrorInvalidJavadocTagItem 
    {
    public:
        /**
         * Constructs the error for the given item and tag.
         */
        ErrorInvalidJavadocTagInterface(Reporter* const reporter, 
                                        AstItem* const item, 
                                        const string& tag);

        /**
         * Exists only for the registration process.
         */
        ErrorInvalidJavadocTagInterface();

        virtual ~ErrorInvalidJavadocTagInterface();

        /**
         * Returns the valid Javadoc tags for an interface.
         */
        virtual vector<string> getValidTags() const;

    protected:

        /**
         * Returns the message.
         */
        virtual string type() const;

        /**
         * Returns "an".
         */
        virtual string article() const;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorInvalidJavadocTagMethod : public ErrorInvalidJavadocTagItem 
    {
    public:
        /**
         * Constructs the error for the given item and tag.
         */
        ErrorInvalidJavadocTagMethod(Reporter* const reporter, 
                                     AstItem* const item, 
                                     const string& tag);

        /**
         * Exists only for the registration process.
         */
        ErrorInvalidJavadocTagMethod();

        virtual ~ErrorInvalidJavadocTagMethod();

        /**
         * Returns the valid Javadoc tags for a method.
         */
        virtual vector<string> getValidTags() const;

    protected:

        /**
         * Returns the message.
         */
        virtual string type() const;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class AstJavadocComment;
    class AstTaggedComment;
    
    class ErrorMisorderedTags : public DocumentationError
    {
    public:
        ErrorMisorderedTags();

        ErrorMisorderedTags(Reporter* const reporter, 
                            AstJavadocComment* const jc,
                            const vector<AstTaggedComment*>& current,
                            const vector<AstTaggedComment*>& ideal);
        
        virtual ~ErrorMisorderedTags();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * Returns the edit, which would replace the tags.
         */
        virtual StringEdit* getEdit();

    private:
        /**
         * The error level (2). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;

        AstJavadocComment* comment_;

        vector<AstTaggedComment*> current_;
        
        vector<AstTaggedComment*> ideal_;
            
    };


    class ErrorMisspelledJavadocTag : public DocumentationError 
    {
    public:
        ErrorMisspelledJavadocTag();

        /**
         * Constructs the error for the tag and its near-matches.
         */
        ErrorMisspelledJavadocTag(Reporter* const reporter, 
                                  AstItem* const item, 
                                  const string& tag,
                                  const vector<string>& matches);

        virtual ~ErrorMisspelledJavadocTag();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * Replaces the given tag, if there is only one close match.
         */
        virtual StringEdit* getEdit();

    private:
        /**
         * The offensive tag.
         */
        string tag_;

        /**
         * The near matches.
         */
        vector<string> matches_;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorMisspelledWord : public DocumentationError 
    {
    public:
        ErrorMisspelledWord();

        /**
         * Constructs the error for the word and its near-matches.
         */
        ErrorMisspelledWord(Reporter* const reporter, 
                            AstItem* const item, 
                            const string& word,
                            const vector<string>& matches);
        
        virtual ~ErrorMisspelledWord();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * Returns the edit for this error.
         */
        virtual StringEdit* getEdit();

    private:
        /**
         * The offensive word.
         */
        string word_;

        /**
         * The near matches.
         */
        vector<string> matches_;
        
        /**
         * The error level (2). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorMultipleVersionTags : public DocumentationError 
    {
    public:
        ErrorMultipleVersionTags();
        
        ErrorMultipleVersionTags(Reporter* const reporter, 
                                 AstItem* const item);

        virtual ~ErrorMultipleVersionTags();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;
 
        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The error level (2). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;

        /**
         * Returns the edit, which would remove the description.
         */
        virtual StringEdit* getEdit();
            
    };


    class AstFormalParameterList;
    class AstThrowsNameList;

    class ErrorUndocumented : public DocumentationError
    {
    public:
        ErrorUndocumented();

        /**
         * Constructs the error for the given item and type.
         */
        ErrorUndocumented(Reporter* const reporter, 
                          AstItem* const item, 
                          const string& type);

        virtual ~ErrorUndocumented();

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * Returns the edit, which would insert the default generated code for
         * the item.
         */
        virtual StringEdit* getEdit();

    protected:
        /**
         * The error level (1). 
         */
        virtual int level() const;
        
        StringEdit* documentConstructor(AstItem* const it, AstItem* const context) const;

        StringEdit* documentField(AstItem* const it, AstItem* const context) const;

        StringEdit* documentMethod(AstItem* const it, AstItem* const context) const;
        
        void addDocumentation(string* const doc,
                              int indentation, 
                              AstFormalParameterList* const params,
                              AstThrowsNameList* const throwslist) const;

        StringEdit* documentTypeDeclaration(AstItem* const typedec,
                                            const string& defaultDescription) const;

        string makeDocumentation(int indentation, 
                                 const string& defaultDescription) const;

    private:
        /**
         * The type of the thing that wasn't commented.
         */
        string type_;
            
    };


    class ErrorNoCommentForPublic : public ErrorUndocumented
    {
    public:
        ErrorNoCommentForPublic();

        /**
         * Constructs the error for the given item and type, which has public
         * access.
         */
        ErrorNoCommentForPublic(Reporter* const reporter, 
                                AstItem* const item, 
                                const string& type);

        virtual ~ErrorNoCommentForPublic();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorNoComment : public ErrorUndocumented
    {
    public:
        ErrorNoComment();

        /**
         * Constructs the error for the given item and type.
         */
        ErrorNoComment(Reporter* const reporter, 
                       AstItem* const item, 
                       const string& type);

        virtual ~ErrorNoComment();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

    protected:
        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorNoDeprecatedText : public DocumentationError 
    {
    public:
        ErrorNoDeprecatedText();

        /**
         * Constructs the error for the offensive item.
         */
        ErrorNoDeprecatedText(Reporter* const reporter, 
                              AstItem* const item);
        
        virtual ~ErrorNoDeprecatedText();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorNoJavadocCommentForPublic : public ErrorUndocumented
    {
    public:
        ErrorNoJavadocCommentForPublic();

        /**
         * Constructs the error for the given item and type.
         */
        ErrorNoJavadocCommentForPublic(Reporter* const reporter, 
                                       AstItem* const item, 
                                       const string& type);

        virtual ~ErrorNoJavadocCommentForPublic();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorNoJavadocComment : public ErrorUndocumented
    {
    public:
        ErrorNoJavadocComment();

        /**
         * Constructs the error for the given item and type.
         */
        ErrorNoJavadocComment(Reporter* const reporter, 
                              AstItem* const item, 
                              const string& type);

        virtual ~ErrorNoJavadocComment();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorNoParameterDescription : public DocumentationError 
    {
    public:
        ErrorNoParameterDescription();

        /**
         * Constructs the error for the offensive item.
         */
        ErrorNoParameterDescription(Reporter* const reporter, 
                                    AstItem* const item);

        virtual ~ErrorNoParameterDescription();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;

        /**
         * Returns the edit, which would insert the default parameter description.
         */
        virtual StringEdit* getEdit();
            
    };


    class ErrorNoParameterName : public DocumentationError 
    {
    public:
        ErrorNoParameterName();

        /**
         * Constructs the error for the offensive item.
         */
        ErrorNoParameterName(Reporter* const reporter, 
                             AstItem* const item,
                             const string& paramName);

        virtual ~ErrorNoParameterName();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;

        /**
         * Returns the edit, which would insert the name and the default
         * parameter description.
         */
        virtual StringEdit* getEdit();

    private:
        
        /**
         * The expected name for this parameter.
         */
        string paramName_;
            
    };


    class ErrorNoPeriodEndingSummarySentence : public DocumentationError 
    {
    public:
        ErrorNoPeriodEndingSummarySentence();

        /**
         * Creates the error for the given Javadoc comment.
         */
        ErrorNoPeriodEndingSummarySentence(Reporter* const reporter,
                                           AstItem* const sentence);

        virtual ~ErrorNoPeriodEndingSummarySentence();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The error level (2). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;

        /**
         * Returns the edit, which would insert the period.
         */
        virtual StringEdit* getEdit();
            
    };


    class ErrorNoSeeReference : public DocumentationError 
    {
    public:
        ErrorNoSeeReference();

        /**
         * Constructs the error for the offensive item.
         */
        ErrorNoSeeReference(Reporter* const reporter, 
                            AstItem* const item);
        
        virtual ~ErrorNoSeeReference();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorNoSinceText : public DocumentationError 
    {
    public:
        ErrorNoSinceText();

        /**
         * Constructs the error for the offensive item.
         */
        ErrorNoSinceText(Reporter* const reporter, 
                         AstItem* const item);
        
        virtual ~ErrorNoSinceText();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class AstJavadocComment;

    class ErrorNoSummarySentence : public DocumentationError 
    {
    public:
        ErrorNoSummarySentence();

        /**
         * Constructs the error for the given Javadoc comment.
         */
        ErrorNoSummarySentence(Reporter* const reporter, 
                               AstJavadocComment* const jdc);

        virtual ~ErrorNoSummarySentence();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The error level (2). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorParameterDataTypeUsed : public DocumentationError 
    {
    public:
        ErrorParameterDataTypeUsed();

        /**
         * Constructs the error for the offensive item.
         */
        ErrorParameterDataTypeUsed(Reporter* const reporter, 
                                   AstItem* const item,
                                   const string& docName,
                                   const string& dataType,
                                   const string& codeName);

        virtual ~ErrorParameterDataTypeUsed();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * Returns the edit for this error.
         */
        virtual StringEdit* getEdit();
        
    private:
        /**
         * The parameter name in the documentation.
         */
        string docName_;

        /**
         * The data type in the code.
         */
        string dataType_;

        /**
         * The paramter name in the code, which should be the same as that in
         * the documentation.
         */
        string codeName_;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorParameterDocumentationMismatchWithCode : public DocumentationError 
    {
    public:
        ErrorParameterDocumentationMismatchWithCode();

        virtual ~ErrorParameterDocumentationMismatchWithCode();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;
 
        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorParameterMisspelled : public DocumentationError 
    {
    public:
        ErrorParameterMisspelled();

        /**
         * Constructs the error for the offensive item, taking the documented
         * name and the code name.
         */
        ErrorParameterMisspelled(Reporter* const reporter, 
                                 AstItem* const item,
                                 const string& docName,
                                 const string& codeName);

        virtual ~ErrorParameterMisspelled();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * Returns the fix for the misspelled parameter.
         */
        virtual StringEdit* getEdit();
        
    private:
        /**
         * The parameter name in the documentation.
         */
        string docName_;

        /**
         * The parameter name in the code.
         */
        string codeName_;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorParameterNotInCode : public DocumentationError 
    {
    public:
        ErrorParameterNotInCode();

        /**
         * Constructs the error for the offensive item, taking the parameter
         * name.
         */
        ErrorParameterNotInCode(Reporter* const reporter, 
                                AstItem* const item, 
                                const string& name);

        virtual ~ErrorParameterNotInCode();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * Returns the edit, which would delete the parameter description.
         */
        virtual StringEdit* getEdit();
        
    private:
        /**
         * The parameter name.
         */
        string name_;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorParameterOutOfOrderAndMispelled : public DocumentationError 
    {
    public:
        ErrorParameterOutOfOrderAndMispelled();

        /**
         * Constructs the error for the offensive item, taking the parameter
         * name as in the documentation and in the code.
         */
        ErrorParameterOutOfOrderAndMispelled(Reporter* const reporter, 
                                             AstItem* const item, 
                                             const string& docName, 
                                             const string& codeName);

        virtual ~ErrorParameterOutOfOrderAndMispelled();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;
        
    private:
        /**
         * The parameter name as in the documentation.
         */
        string docName_;

        /**
         * The parameter name as in the code.
         */
        string codeName_;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorParameterOutOfOrder : public DocumentationError 
    {
    public:
        ErrorParameterOutOfOrder();

        /**
         * Constructs the error for the offensive item, taking the parameter
         * name.
         */
        ErrorParameterOutOfOrder(Reporter* const reporter, 
                                 AstItem* const item, 
                                 const string& name);

        virtual ~ErrorParameterOutOfOrder();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;
        
    private:
        /**
         * The parameter name.
         */
        string name_;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorParameterRepeatedInDocumentation : public DocumentationError 
    {
    public:
        ErrorParameterRepeatedInDocumentation();

        /**
         * Constructs the error for the offensive item, taking the parameter
         * name.
         */
        ErrorParameterRepeatedInDocumentation(Reporter* const reporter, 
                                              AstItem* const item, 
                                              const string& name);

        virtual ~ErrorParameterRepeatedInDocumentation();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * Returns the edit, which would remove the description.
         */
        virtual StringEdit* getEdit();
        
    private:
        /**
         * The parameter name.
         */
        string name_;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorParametersDocumentedButNotInCode : public DocumentationError 
    {
    public:
        ErrorParametersDocumentedButNotInCode();

        /**
         * Creates the error for the offensive item, which is of the given type.
         */
        ErrorParametersDocumentedButNotInCode(Reporter* const reporter, 
                                              AstItem* const item, 
                                              const string& type);

        virtual ~ErrorParametersDocumentedButNotInCode();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * Returns the edit, which would remove the documented parameters.
         */
        virtual StringEdit* getEdit();
        
    private:
        /**
         * The type of the offensive parameter.
         */
        string type_;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorParameterUndocumented : public DocumentationError 
    {
    public:
        ErrorParameterUndocumented();

        /**
         * Constructs the error for the offensive argument name.
         */
        ErrorParameterUndocumented(Reporter* const reporter, 
                                   AstItem* const item, 
                                   const string& argName);

        virtual ~ErrorParameterUndocumented();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;
        
    private:
        /**
         * The name of the offensive parameter.
         */
        string argName_;

        /**
         * The error level (1). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorReturnDocumentedForMethodReturningVoid : public DocumentationError 
    {
    public:
        ErrorReturnDocumentedForMethodReturningVoid();

        /**
         * Constructs the error for the offensive item.
         */
        ErrorReturnDocumentedForMethodReturningVoid(Reporter* const reporter, 
                                                    AstItem* const item);

        virtual ~ErrorReturnDocumentedForMethodReturningVoid();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;

        /**
         * Returns the edit, which would remove the documented return.
         */
        virtual StringEdit* getEdit();
            
    };


    class ErrorReturnTypeUsedAsTarget : public DocumentationError 
    {
    public:
        ErrorReturnTypeUsedAsTarget();

        /**
         * Constructs the error for the offensive item and return type.
         */
        ErrorReturnTypeUsedAsTarget(Reporter* const reporter, 
                                    AstItem* const item, 
                                    const string& retType);

        virtual ~ErrorReturnTypeUsedAsTarget();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * Returns the edit, which would remove the return type from the text.
         */
        virtual StringEdit* getEdit();

    private:
        /**
         * The return type that was used.
         */
        string retType_;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorReturnUndocumented : public DocumentationError 
    {
    public:
        ErrorReturnUndocumented();

        /**
         * Constructs the error for the offensive item.
         */
        ErrorReturnUndocumented(Reporter* const reporter, 
                                AstItem* const item);

        virtual ~ErrorReturnUndocumented();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The error level (1). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorReturnUsedWithConstructor : public DocumentationError 
    {
    public:
        /**
         * For usage with code.
         */
        ErrorReturnUsedWithConstructor(Reporter* const reporter, 
                                       AstItem* const item);

        /**
         * For obtaining static information about instances of this class.
         */
        ErrorReturnUsedWithConstructor();

        virtual ~ErrorReturnUsedWithConstructor();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const /* = 0 */;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;

        /**
         * Returns the edit, which would remove the documented return.
         */
        virtual StringEdit* getEdit();
            
    };


    class ErrorReturnWithoutDescription : public DocumentationError 
    {
    public:
        ErrorReturnWithoutDescription();

        /**
         * Constructs the error for the offensive item.
         */
        ErrorReturnWithoutDescription(Reporter* const reporter, 
                                      AstItem* const item);

        virtual ~ErrorReturnWithoutDescription();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorSerialDataWithoutDescription : public DocumentationError 
    {
    public:
        ErrorSerialDataWithoutDescription();

        /**
         * Constructs the error for the offensive item.
         */
        ErrorSerialDataWithoutDescription(Reporter* const reporter, 
                                     AstItem* const item);

        virtual ~ErrorSerialDataWithoutDescription();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorSerialFieldWithoutDataDescription : public DocumentationError 
    {
    public:
        ErrorSerialFieldWithoutDataDescription();

        /**
         * Constructs the error for the offensive item.
         */
        ErrorSerialFieldWithoutDataDescription(Reporter* const reporter, 
                                               AstItem* const item);
        
        virtual ~ErrorSerialFieldWithoutDataDescription();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The error level (1). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorSerialFieldWithoutNameTypeOrDescription : public DocumentationError 
    {
    public:
        ErrorSerialFieldWithoutNameTypeOrDescription();

        /**
         * Constructs the error for the offensive item.
         */
        ErrorSerialFieldWithoutNameTypeOrDescription(Reporter* const reporter, 
                                                AstItem* const item);

        virtual ~ErrorSerialFieldWithoutNameTypeOrDescription();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorSerialFieldWithoutTypeOrDescription : public DocumentationError 
    {
    public:
        ErrorSerialFieldWithoutTypeOrDescription();

        /**
         * Constructs the error for the offensive item.
         */
        ErrorSerialFieldWithoutTypeOrDescription(Reporter* const reporter, 
                                            AstItem* const item);

        virtual ~ErrorSerialFieldWithoutTypeOrDescription();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorSerialWithoutDescription : public DocumentationError 
    {
    public:
        ErrorSerialWithoutDescription();

        /**
         * Constructs the error for the offensive item.
         */
        ErrorSerialWithoutDescription(Reporter* const reporter, 
                                    AstItem* const item);
        
        virtual ~ErrorSerialWithoutDescription();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The error level (1). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };


    class ErrorShortSummarySentence : public DocumentationError 
    {
    public:
        ErrorShortSummarySentence();

        /**
         * Constructs the error for the offensive item.
         */
        ErrorShortSummarySentence(Reporter* const reporter, 
                                  AstItem* const item);

        virtual ~ErrorShortSummarySentence();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The error level (2). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;
            
    };



    class ErrorVersionWithoutText : public DocumentationError
    {
    public:
        ErrorVersionWithoutText();

        /**
         * Constructs the error for the offensive item.
         */
        ErrorVersionWithoutText(Reporter* const reporter, 
                                AstItem* const item);
        
        virtual ~ErrorVersionWithoutText();

        /**
         * Returns the title of this error.
         */
        virtual string title() const;

        /**
         * Writes the long description of this error to the given output stream.
         */
        virtual void writeDescription(ostream& os) const;

        /**
         * Returns the message describing this error. 
         */
        virtual string message() const;

        /**
         * The error level (3). 
         */
        virtual int level() const;

        /**
         * The number specific to this error type.
         */
        virtual int number() const;

        /**
         * Returns the edit, which would insert the default version.
         */
        virtual StringEdit* getEdit();
            
    };
}

#endif //! DocumentationErrors_h
