#ifndef ErrorRegistry_h
#include "ErrorRegistry.h"
#endif

#ifndef doctorj_iterext_h
#define doctorj_iterext_h
#include "iterext.h"
#endif

#ifndef RegExp_h
#include "RegExp.h"
#endif

#ifndef doctorj_algorithm
#define doctorj_algorithm
#include <algorithm>
#endif

using namespace std;

using namespace doctorj;

namespace doctorj
{
    struct ErrorSettings
    {
        Check* check;
        int identifier;
        bool isEnabled;
    };
}

/**
 * Returns whether the code of the first precedes that of the other.
 */
static bool alpha(Check* const one, Check* const two) 
{
    if (one->category() == two->category()) {
        return one->identifier() < two->identifier();
    }
    else {
        return one->category() < two->category();
    }
}

NoSuchErrorCodeException::NoSuchErrorCodeException(const string& code) : code_(code)
{
}

NoSuchErrorCodeException::~NoSuchErrorCodeException()
{
}

string NoSuchErrorCodeException::code() const
{
    return code_;
}


ErrorRegistrant::ErrorRegistrant(Check* const error)
{
    error->getErrorRegistry()->add(error);
}


ErrorRegistry::StringToErrorSettings ErrorRegistry::errors_;

ErrorRegistry::StringToBool ErrorRegistry::suppressions_;

ErrorRegistry::ErrorRegistry() : identifier_(0)
{
}

ErrorRegistry::~ErrorRegistry()
{
    EACH(StringToErrorSettings, errors_, it) {
        ErrorSettings* es = it->second;
        Check* err = es->check;
        err->destroy();
        delete es;
    }
}

void ErrorRegistry::describe(ostream& os, const string& code) throw(NoSuchErrorCodeException)
{
    vector<Check*> errs = errors();
    EACHC(vector<Check*>, errs, it) {
        Check* e = *it;
        if (e->id() == code) {
            e->describe(os);
            return;
        }
    }

    cerr << "no such error code '" << code << "'" << endl;
    throw NoSuchErrorCodeException(code);
}

void ErrorRegistry::describeAll(ostream& os)
{
    vector<Check*> errs = errors();
    EACHC(vector<Check*>, errs, it) {
        Check* e = *it;
        e->describe(os);
        os << endl;
    }
}

void ErrorRegistry::add(Check* const check)
{
    StringToErrorSettings::iterator pos = errors_.find(check->title());
    if (pos == errors_.end()) {
        ErrorSettings* es = new ErrorSettings();
        es->identifier = identifier_;
        ++identifier_;
        // es->code = error->id();
        es->check = check;
        es->isEnabled = true;
        errors_[check->title()] = es;
        suppressions_[check->id()] = false;
    }
    else {
        cerr << "ERROR in ErrorRegistry:" << endl;
        cerr << "    error: " << check->title() << endl;
        cerr << "    already registered to: ";
        ErrorSettings* es = pos->second;
        cerr << es->check->title() << endl;
    }
}

void ErrorRegistry::setSuppressed(const string& code, bool suppress)
{
    if (code == "all" || code == "*") {
        EACH(StringToBool, suppressions_, it) {
            it->second = suppress;
        }
    }
    else {
        // allow wild cards in the code, such as "Dxxx", "S.8.", "D34*", and "D*"
        RegExp re(code);
        EACH(StringToBool, suppressions_, it) {
            string s = it->first;
            if (re.match(s)) {
                it->second = suppress;
            }
        }
    }
}

bool ErrorRegistry::isSuppressed(const string& code)
{
    StringToBool::const_iterator pos = suppressions_.find(code);
    if (pos == suppressions_.end()) {
        cerr << "ERROR in ErrorRegistry suppression list:" << endl;
        cerr << "    error: " << code << endl;
        cerr << "    sought but not registered." << endl;
        return false;
    }
    else {
        return pos->second;
    }
}

vector< Check* > ErrorRegistry::errors()
{
    vector< Check* > errs;
    EACHC(StringToErrorSettings, errors_, it) {
        ErrorSettings* es = it->second;
        errs.push_back(es->check);
    }
    sort(errs.begin(), errs.end(), alpha);
    return errs;
}

int ErrorRegistry::identifier(const string& errorTitle) const
{
    StringToErrorSettings::const_iterator pos = errors_.find(errorTitle);
    if (pos == errors_.end()) {
        cerr << "ERROR in ErrorRegistry:" << endl;
        cerr << "    error: " << errorTitle << endl;
        cerr << "    sought but not registered." << endl;
        return -1;
    }
    else {
        return pos->second->identifier;
    }
}
