/*======================================================================

    A driver for PCMCIA digimedia dmx512 devices

    digimedia_cs.c

    The contents of this file are subject to the Mozilla Public
    License Version 1.1 (the "License"); you may not use this file
    except in compliance with the License. You may obtain a copy of
    the License at http://www.mozilla.org/MPL/

    Software distributed under the License is distributed on an "AS
    IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
    implied. See the License for the specific language governing
    rights and limitations under the License.

    The initial developer of the original code is David A. Hinds
    <dahinds@users.sourceforge.net>.  Portions created by David A. Hinds
    are Copyright (C) 1999 David A. Hinds.  All Rights Reserved.

    Alternatively, the contents of this file may be used under the
    terms of the GNU General Public License version 2 (the "GPL"), in which
    case the provisions of the GPL are applicable instead of the
    above.  If you wish to allow the use of your version of this file
    only under the terms of the GPL and not to allow others to use
    your version of this file under the MPL, indicate your decision
    by deleting the provisions above and replace them with the notice
    and other provisions required by the GPL.  If you do not delete
    the provisions above, a recipient may use your version of this
    file under either the MPL or the GPL.

======================================================================*/

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <asm/io.h>

#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/ciscode.h>
#include <pcmcia/cisreg.h>
#include <pcmcia/ds.h>

#include <dmx/dmxdev.h>

#include "dgm_cs.h"

#define PCMCIA_DEBUG (10)

MODULE_AUTHOR("(c) Michael Stickel <michael@cubic.org> http://llg.cubic.org");
MODULE_DESCRIPTION("Driver for the Digimedia PCMCIA interface version " DMXVERSION);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,17)
MODULE_LICENSE("Dual MPL/GPL");
#endif

/*
#ifdef PCMCIA_DEBUG
static int pc_debug = PCMCIA_DEBUG;
MODULE_PARM(pc_debug, "i");
#define DO_DEBUG(n, args...) { if (pc_debug>(n)) args; }
#else
#define DO_DEBUG(n, args...)
#endif
*/
static int debug_flags   = 7;   /* debug messages trigger */
MODULE_PARM_DESC(debug_flags,    "Set trigger for debugging messages (see dgm.h)");
MODULE_PARM(debug_flags, "i");

/*====================================================================*/

/* add this to the driver-structure */
static dev_link_t *dev_list = NULL;

static dev_info_t dev_info = "digimedia_cs";

static void digimedia_config(dev_link_t *link);
static void digimedia_release(u_long arg);
static int  digimedia_event(event_t event, int priority, event_callback_args_t *args);

static dev_link_t *pcmcia_create_link(void);

static dev_link_t *digimedia_attach(void);
static void        digimedia_detach(dev_link_t *);

/*====================================================================*/

static void cs_error (client_handle_t handle, int func, int ret)
{
  error_info_t err = { func, ret };
  CardServices(ReportError, handle, &err);
}

/*====================================================================*/


/*--------------------------------------------------------
 *--
 *--
 *-------------------------------------------------------*/
int pcmcia_add_instance (dev_link_t *link)
{
  DO_DEBUG(DGM_DEBUG_PCMCIA, printk(DGM_DEBUG "dgm_cs : int pcmcia_add_instance (dev_link_t *link=%p)\n",link));
  if (link)
  {
    link->next = dev_list;
    dev_list = link;
    return CS_SUCCESS;
  }
  return -1;
}

/*--------------------------------------------------------
 *--
 *--
 *-------------------------------------------------------*/
int pcmcia_delete_instance (dev_link_t *link)
{
  static int in_function = 0;
  printk("int pcmcia_delete_instance (dev_link_t *link=%p, depth=%d)\n", link, in_function);

  if (in_function) return -1;
  in_function++;
  if (link)
  {
    int ret;

    /* find the link */
    dev_link_t **l;
    for (l = &dev_list; *l; l = &((*l)->next))
      if (*l == link)
        break;
    if (*l == NULL)
    {
      printk ("no matching instance found\n");
      in_function--;
      return -1;
    }

    if (link->handle)
    {
      del_timer(&link->release); /* not delaying release of instance. Deleting it now */
      printk(" I call digimedia_release(0x%08lX)\n", (u_long)link);
      if (link->state & DEV_CONFIG)
        digimedia_release((u_long)link);

      ret = CardServices(DeregisterClient, link->handle);
      if (ret != CS_SUCCESS)
      {
        cs_error(link->handle, DeregisterClient, ret);
        {
          in_function--;
          return -1;
        }
      }
      else
      {
        /* remove the link */
        if (l)
          *l = link->next;

        if (link->dev)
          kfree(link->dev);
        kfree(link);

        in_function--;
        return CS_SUCCESS;
      }
    }
  }
  in_function--;
  return -1;
}

/*--------------------------------------------------------
 *--
 *--
 *-------------------------------------------------------*/
static dev_link_t *pcmcia_create_link(void)
{
  dev_link_t *link = (dev_link_t *)kmalloc(sizeof(dev_link_t), GFP_KERNEL);
  if (link)
    memset(link, 0, sizeof(dev_link_t));
  return link;
}

/*====================================================================*/

/*--------------------------------------------------------
 *-- digimedia_attach() creates an "instance" of the driver,
 *-- allocating local data structures for one device.
 *-- The device is registered with Card Services.
 *-------------------------------------------------------*/
static dev_link_t *digimedia_attach(void)
{
  dev_link_t *link;

  DO_DEBUG(DGM_DEBUG_PCMCIA, printk(DGM_DEBUG "dgm_cs : digimedia_attach() {\n"));

  /* Create new digimedia device */
  link = pcmcia_create_link();
  if (link)
  {
    client_reg_t client_reg;
    int ret=0;

    link->release.function = &digimedia_release;
    link->release.data = (u_long)link;

    link->conf.Attributes = 0;
    link->conf.Vcc = 50;
    link->conf.IntType = INT_MEMORY_AND_IO;
    link->priv = NULL;

    /* Register with Card Services */
    if (pcmcia_add_instance (link) != CS_SUCCESS)
    {
      printk(__FILE__ ": failed to add instance\n");
      cs_error(link->handle, RegisterClient, ret);
      digimedia_detach(link);
      return NULL;
    }

    client_reg.dev_info   = &dev_info;
    client_reg.Attributes = INFO_IO_CLIENT | INFO_CARD_SHARE;
    client_reg.EventMask  = CS_EVENT_REGISTRATION_COMPLETE |
      CS_EVENT_CARD_INSERTION | CS_EVENT_CARD_REMOVAL |
      CS_EVENT_RESET_PHYSICAL | CS_EVENT_CARD_RESET |
      CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
    client_reg.event_handler = &digimedia_event;
    client_reg.Version = 0x0210;
    client_reg.event_callback_args.client_data = link;

    ret = CardServices(RegisterClient, &link->handle, &client_reg);
    if (ret != CS_SUCCESS)
    {
      cs_error(link->handle, RegisterClient, ret);
      digimedia_detach(link);
      return NULL;
    }
    return link;
  }
  return NULL;
} /* digimedia_attach */

/*--------------------------------------------------------
 *-- This deletes a driver "instance".  The device is de-registered
 *-- with Card Services.  If it has been released, all local data
 *-- structures are freed.  Otherwise, the structures will be freed
 *-- when the device is released.
 *-------------------------------------------------------*/
static void digimedia_detach(dev_link_t *link)
{
  DO_DEBUG(DGM_DEBUG_PCMCIA, printk(DGM_DEBUG "dgm_cs : digimedia_detach(0x%p)\n", link));
  if (pcmcia_delete_instance (link) != CS_SUCCESS)
    printk ("pcmcia_delete_instance failed\n");

} /* digimedia_detach */


/*======================[ Some CIS-Tuple Access Stuff ]================================*/

/*--------------------------------------------------------
 *--
 *--
 *-------------------------------------------------------*/
static int get_tuple(int fn, client_handle_t handle, tuple_t *tuple, cisparse_t *parse)
{
  int i;

  DO_DEBUG(DGM_DEBUG_PCMCIA, printk (DGM_DEBUG "dgm_cs : static int get_tuple(int fn=%d, client_handle_t handle, tuple_t *tuple=%p, cisparse_t *parse=%p)\n", fn, tuple, parse));

  i = CardServices(fn, handle, tuple);
  if (i != CS_SUCCESS) return CS_NO_MORE_ITEMS;
  i = CardServices(GetTupleData, handle, tuple);
  if (i != CS_SUCCESS) return i;
  return CardServices(ParseTuple, handle, tuple, parse);
}

#define first_tuple(a, b, c) get_tuple(GetFirstTuple, a, b, c)
#define next_tuple(a, b, c) get_tuple(GetNextTuple, a, b, c)


/*================[ Digimedia specific stuff ]========================*/

/*--------------------------------------------------------
 *--
 *-- This get things ready to call dgm_create_board
 *--
 *-------------------------------------------------------*/
static int digimedia_create_board(dev_link_t *link, unsigned long base, size_t size)
{
  int ret;
  dgm_board_t *brd = NULL;
  dgm_memory_t *dgm_mem = NULL;

  if (!link)
  {
    printk(__FILE__ ":unable to find dmnx-driver digimedia/digimedia\n");
    return -1;
  }

  /* get the board structure */
  brd = (dgm_board_t *)kmalloc(sizeof(dgm_board_t), GFP_KERNEL);
  if (brd == NULL)
  {
    DO_DEBUG(DGM_DEBUG_WARN_2, printk(DGM_WARN "dgm_cs : unable to allocate board structure\n"));
    return -ENOMEM;
  }

  /* fill some fields */
  brd->dev         = (void*)link;
  brd->type        = 2512;
  brd->if_count    = 1;

  /* remap memory */
  if (!(base && (size>=1024) && (size<=4096)))
  {
    DO_DEBUG(DGM_DEBUG_WARN_2, printk(DGM_WARN "dgm_cs : memory parameter NOT OK\n"));
    kfree(brd);
    return -1;
  }
  dgm_mem = (dgm_memory_t *)ioremap(base+DGM_CS_MEMORY_OFFSET, size);
  if (!dgm_mem)
  {
    DO_DEBUG(DGM_DEBUG_WARN_2, printk(DGM_WARN "dgm_cs : unable to remap I/O Memory\n"));
    kfree(brd);
    return -ENOMEM;
  }
  if (memcmp("DMXOEM",dgm_mem+0x01,6)!=0)
  {
    DO_DEBUG(DGM_DEBUG_WARN_2, printk(DGM_WARN "dgm_cs : remaped I/O Memory NOT signed\n"));
    iounmap(dgm_mem);
    kfree(brd);
    return -1;
  }
  DO_DEBUG(DGM_DEBUG_INFO_1,printk(DGM_INFO "dgm_cs : successfully remaped signed memory\n"));
  brd->mem_addr[0] = dgm_mem;

  ret = dgm_create_board(brd);
  if (ret)
  {
    DO_DEBUG(DGM_DEBUG_WARN_2, printk(DGM_WARN "dgm_cs : error at board creation by 'dgm'\n"));
    iounmap(dgm_mem);
    kfree(brd);
    return -1;
  }

  /* link board to link */
  link->priv = (void *)brd;

  return 0; /* anything succeded */
}

#define CS_CHECK(fn, args...) \
  while ((last_ret=CardServices(last_fn=(fn),args))!=0) goto cs_failed

#define CFG_CHECK(fn, args...) \
  if (CardServices(fn, args) != 0) goto next_entry


/*--------------------------------------------------------
 *-- digimedia_config() is scheduled to run after a
 *-- CARD_INSERTION event is received, to configure
 *-- the PCMCIA socket, and to make the dmx interface
 *-- available to the system.
 *-------------------------------------------------------*/
void digimedia_config(dev_link_t *link)
{
  client_handle_t  handle = link->handle;

  unsigned long    iomem_base=0;
  size_t           iomem_size=0;

  tuple_t          tuple;
  cisparse_t       parse;
  u_char           buf[255];
  config_info_t    conf;
  int              last_fn, last_ret;

  win_req_t        req;
  memreq_t         mem;

  int numchains = -1; /* no CIS present by default. */

  DO_DEBUG(DGM_DEBUG_PCMCIA, printk(DGM_DEBUG "dgm_cs : void digimedia_config(dev_link_t *link=%p)\n", link));

  if (!link) return;

  {
    cisinfo_t        cisinfo;
    int              ret;
    if ((ret=CardServices(ValidateCIS, handle, &cisinfo)) == CS_SUCCESS)
    {
      numchains = cisinfo.Chains;
      printk (KERN_INFO ":" __FILE__": CIS structure is valid and has %d entries\n", numchains);
    }
    else
    {
      char *estr = "unknown";
      if (ret==CS_BAD_HANDLE) estr="client handle is invalid";
      if (ret==CS_OUT_OF_RESOURCE) estr="unable to map CIS memory";
      printk("failed to validate CIS structure: %s\n", estr);
    }
  }

  /*
    This reads the card's CONFIG tuple to find its configuration registers.
  */
  tuple.DesiredTuple = CISTPL_CONFIG;
  tuple.Attributes   = 0;
  tuple.TupleData    = buf;
  tuple.TupleDataMax = sizeof(buf);
  tuple.TupleOffset  = 0;
  CS_CHECK(GetFirstTuple, handle, &tuple);
  CS_CHECK(GetTupleData,  handle, &tuple);
  CS_CHECK(ParseTuple,    handle, &tuple, &parse);
  link->conf.ConfigBase = parse.config.base;
  link->conf.Present    = parse.config.rmask[0];

  /* Configure card */
  link->state |= DEV_CONFIG;

  /* Look up the current Vcc */
  CS_CHECK(GetConfigurationInfo, handle, &conf);
  link->conf.Vcc = conf.Vcc;
  printk("Vcc=%d.%dV\n", conf.Vcc/10, conf.Vcc%10);


  if (0)
  {
    int j;
    int i;

    tuple.Attributes = TUPLE_RETURN_COMMON;
    tuple.TupleOffset = 0;
    tuple.TupleData = buf;
    tuple.TupleDataMax = sizeof(buf);
    tuple.DesiredTuple = RETURN_FIRST_TUPLE;

    i= first_tuple(handle, &tuple, &parse);
    for (j=0; i==CS_SUCCESS && j<10; j++)
    {
      DO_DEBUG(DGM_DEBUG_PCMCIA, printk(DGM_DEBUG "dgm_cs :    tuple%d: code=0x%x  desired=0x%x\n", j, tuple.TupleCode, tuple.DesiredTuple));
      switch(tuple.TupleCode)
      {
        case CISTPL_VERS_1:
          {
            int k;
            cistpl_vers_1_t *version = &parse.version_1;
            printk(__FILE__": Version: %d.%d:", version->major, version->minor);
            for (k=0; k<version->ns; k++)
              printk(" %s", &version->str[version->ofs[k]]);
            printk("\n");
          }
          break;

        case CISTPL_DEVICE_A:
          {
            int k;
            cistpl_device_t *device = &parse.device;
            printk(__FILE__ ": Device-Memory with %d device\n", device->ndev);
            for (k=0; k<device->ndev; k++)
            {
              int t = device->dev[k].type;
              char *typename =
                (t==CISTPL_DTYPE_NULL)?"no-device":
                (t==CISTPL_DTYPE_ROM)?"ROM":
                (t==CISTPL_DTYPE_OTPROM)?"OTMROM":
                (t==CISTPL_DTYPE_EPROM)?"EPROM":
                (t==CISTPL_DTYPE_EEPROM)?"EEPROM":
                (t==CISTPL_DTYPE_FLASH)?"FLASH":
                (t==CISTPL_DTYPE_SRAM)?"SRAM":
                (t==CISTPL_DTYPE_DRAM)?"DRAM":
                (t==CISTPL_DTYPE_FUNCSPEC)?"FUNCSPEC":
                (t==CISTPL_DTYPE_EXTEND)?"EXTENDED":"undefined";
              printk (__FILE__ ":  device# %d of type %s access %s speed=%uns  size=%u bytes\n",
                k,
                typename,
                device->dev[k].wp?"RO":"RW",
                device->dev[k].speed,
                device->dev[k].size);
            }
          }
          break;

        case CISTPL_CONFIG:
          break;

        case CISTPL_CFTABLE_ENTRY:
          {
            cistpl_cftable_entry_t *cftable = &parse.cftable_entry;
            int k;
            for (k=0; k<cftable->mem.nwin && k<CISTPL_MEM_MAX_WIN; k++)
              printk(__FILE__ ": Found a memory region of size %d bytes located at card base 0x%08X to be mapped at host address 0x%08X\n",
                cftable->mem.win[k].len,
                cftable->mem.win[k].card_addr,
                cftable->mem.win[k].host_addr);
          }
          break;

        default:
          printk(__FILE__ ": unexpected tuple-code 0x%x. please mail this line to michael@cubic.org\n", tuple.TupleCode);
          break;
      }
      i = next_tuple(handle, &tuple, &parse);
    }
  }

  req.Attributes = WIN_DATA_WIDTH_8 | WIN_MEMORY_TYPE_CM | WIN_ENABLE;
  req.Base = req.Size = 0;
  req.AccessSpeed = 0;
  link->win = (window_handle_t)link->handle;
  if (CardServices(RequestWindow, &link->win, &req) == CS_SUCCESS)
  {
    mem.CardOffset = mem.Page = 0;
    mem.CardOffset = link->conf.ConfigBase;
    DO_DEBUG(DGM_DEBUG_PCMCIA, printk (DGM_DEBUG "dgm_cs : req.Base=%lX  req.Size=%d  mem.CardOffset=%X\n", req.Base, req.Size, mem.CardOffset));
    if (CardServices(MapMemPage, link->win, &mem) == CS_SUCCESS)
    {
      link->conf.ConfigIndex = 0;
      CS_CHECK(RequestConfiguration, link->handle, &link->conf);

      /*
        typedef struct dev_node_t {
        char                dev_name[DEV_NAME_LEN];
        u_short             major, minor;
        struct dev_node_t   *next;
        } dev_node_t;
      */

      link->dev = DMX_ALLOC(dev_node_t);
      if (link->dev)
      {
        /* Oh what a hack - this is some kind of impossible */
        strcpy(link->dev->dev_name, "dmx");
        link->dev->major = 10;     /* misc */
        link->dev->minor = 210; /* /dev/dmx */
        link->dev->next = NULL;
      }
      link->state &= ~DEV_CONFIG_PENDING;
      iomem_base = req.Base;
      iomem_size = req.Size;

      DO_DEBUG(DGM_DEBUG_PCMCIA, printk (DGM_DEBUG "dgm_cs : after MapMemPage: req.Base=%lX  req.Size=%d  mem.CardOffset=%X\n", req.Base, req.Size, mem.CardOffset));
    }
  }

  if (iomem_size)
  {
    if (!digimedia_create_board(link, iomem_base, iomem_size))
      return;
  }
  if (link->state & DEV_CONFIG)
    digimedia_release((u_long)link);
  return;

 cs_failed:
  cs_error(link->handle, last_fn, last_ret);
  if (link->state & DEV_CONFIG)
    digimedia_release((u_long)link);
}

/*--------------------------------------------------------
 *-- After a card is removed, digimedia_release()
 *-- will unregister the device, and release
 *-- the PCMCIA configuration.
 *-------------------------------------------------------*/
void digimedia_release(u_long arg)
{
  dgm_board_t *brd  = NULL;
  dev_link_t  *link = (dev_link_t *)arg;

  DO_DEBUG(DGM_DEBUG_PCMCIA, printk (DGM_DEBUG "dgm_cs : digimedia_release(%p)\n", link));

  if (link)
  {
    brd = (dgm_board_t *)link->priv;

    DO_DEBUG(DGM_DEBUG_INFO_2, printk(DGM_INFO "dgm_cs : remove board with minor %d\n",brd->minor));
#if 0
    schedule();
#endif
    dgm_delete_board(brd);

    /* release memory mapping */
    if (brd->mem_addr[0])
      iounmap(brd->mem_addr[0]);

    /* unlink and free */
    link->priv = NULL;
    kfree(brd);
    link->dev = NULL;

    DO_DEBUG(DGM_DEBUG_PCMCIA, printk (DGM_DEBUG "dgm_cs : link->dev = NULL;\n"));

    CardServices(ReleaseWindow, link->win);
    CardServices(ReleaseConfiguration, link->handle);

    link->state &= ~DEV_CONFIG;
    DO_DEBUG(DGM_DEBUG_PCMCIA, printk (DGM_DEBUG "dgm_cs : link->state &= ~DEV_CONFIG;\n"));
  }
} /* digimedia_release */

/*--------------------------------------------------------
 *--
 *-- The card status event handler.  Mostly, this schedules other
 *-- stuff to run after an event is received.  A CARD_REMOVAL event
 *-- also sets some flags to discourage the dmx interface from
 *-- talking to the ports.
 *--
 *-------------------------------------------------------*/
static int digimedia_event(event_t event, int priority, event_callback_args_t *args)
{
  dev_link_t *link = args->client_data;

  DO_DEBUG(DGM_DEBUG_PCMCIA, printk(DGM_DEBUG "dgm_cs : digimedia_event : digimedia_event(0x%06x)\n", event));

  switch (event)
  {
    case CS_EVENT_CARD_REMOVAL:
      printk (KERN_INFO ":" __FILE__": card has been REMOVED\n");
      link->state &= ~DEV_PRESENT;
      if (link->state & DEV_CONFIG)
        mod_timer(&link->release, jiffies + HZ/20);
      printk (KERN_INFO ":" __FILE__": card remove finished\n");
      break;

    case CS_EVENT_CARD_INSERTION:
      printk (KERN_INFO ":" __FILE__": card has been INSERTED\n");
      link->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
      digimedia_config(link);
      printk (KERN_INFO ":" __FILE__ ": card insertion finished\n");
      break;

    case CS_EVENT_PM_SUSPEND:
      printk (KERN_INFO ":" __FILE__": card should SUSPEND\n");
      link->state |= DEV_SUSPEND;
      /* Fall through... */

    case CS_EVENT_RESET_PHYSICAL:
      printk (KERN_INFO ":" __FILE__": card should do a HARD-RESET\n");
      if ((link->state & DEV_CONFIG))
        CardServices(ReleaseConfiguration, link->handle);
      printk (KERN_INFO ":" __FILE__": HARD-RESET finished\n");
      break;

    case CS_EVENT_PM_RESUME:
      printk (KERN_INFO ":" __FILE__": card should RESUME\n");
      link->state &= ~DEV_SUSPEND;
      /* Fall through... */

    case CS_EVENT_CARD_RESET:
      printk (KERN_INFO ":" __FILE__": card should do a SOFT-RESET\n");
      if (DEV_OK(link))
        CardServices(RequestConfiguration, link->handle, &link->conf);
      printk (KERN_INFO ":" __FILE__": SOFT-RESET finished\n");
      break;

    case CS_EVENT_REGISTRATION_COMPLETE:
      printk (KERN_INFO ":" __FILE__": REGISTRATION_COMPLETE\n");
      break;

    default:
      printk (KERN_INFO ":" __FILE__": UNKNOWN EVENT %d\n", event);
      break;
  }
  DO_DEBUG(DGM_DEBUG_PCMCIA, printk(DGM_DEBUG "dgm_cs : digimedia_event : digimedia_event(0x%06x): AT-END\n", event));
  return 0;
} /* digimedia_event */

/*--------------------------------------------------------
 *-- init_digimedia_cs
 *--
 *-- This function is called after the module has loaded.
 *--
 *-------------------------------------------------------*/
static int __init init_digimedia_cs(void)
{
  int ret=0;
  servinfo_t serv;

  DO_DEBUG(DGM_DEBUG_PCMCIA, printk(DGM_DEBUG "dgm_cs : static int __init init_digimedia_cs(void)\n"));

  CardServices(GetCardServicesInfo, &serv);
  if (serv.Revision != CS_RELEASE_CODE)
  {
    printk(KERN_INFO ":" __FILE__ ": Card Services release does not match!\n");
    printk(KERN_INFO "Card Services Release = %X  driver version = %X\n", serv.Revision, CS_RELEASE_CODE);
    return -1;
  }

  ret=register_pccard_driver(&dev_info, &digimedia_attach, &digimedia_detach);
  printk ("register_pccard_driver() =%d\n", ret);
  if (ret!=CS_SUCCESS)
    return -1;
  return 0;
}

/*--------------------------------------------------------
 *--
 *-- exit_digimedia_cs
 *--
 *-------------------------------------------------------*/
static void __exit exit_digimedia_cs(void)
{
  int i;

  DO_DEBUG(DGM_DEBUG_PCMCIA, printk(DGM_DEBUG "dgm_cs : static void __exit exit_digimedia_cs(void)\n"));

  for (i=0; i<100 && dev_list != NULL; i++)
    digimedia_detach(dev_list);

  unregister_pccard_driver(&dev_info);
}


#ifdef MODULE
module_init(init_digimedia_cs);
module_exit(exit_digimedia_cs);
#endif
