#define _XOPEN_SOURCE 1
#define _BSD_SOURCE 1
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <pcap.h>
#include <inttypes.h>
#include <netinet/if_ether.h>
#if (__GLIBC__)
#include <net/ethernet.h>
#endif
#include <sys/wait.h>
#include <sys/time.h>
#include <libnet.h>
#include <pthread.h>

#include <linux/version.h>

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,1,50)
#warning divine needs Linux 2.2+
#endif

static unsigned char *device;
static char errbuf[256];

static void error(const char *message) {
  fprintf(stderr,"Error: %s\n",message);
  exit(23);
}

static void error2(const char *message) {
  fprintf(stderr,"Error: %s: %s\n",message,errbuf);
  exit(23);
}

static char *ip2a(unsigned int ip) {
  char *buf=malloc(20);
  sprintf(buf,"%d.%d.%d.%d",ip>>24,(ip>>16)&0xff,(ip>>8)&0xff,ip&0xff);
  return buf;
}

#ifndef OLD_LIBNET
static libnet_t *l;
#endif

void sendarp(u_long myip,u_long hisip) {
  static int initialized=0;
#ifdef OLD_LIBNET
  static struct libnet_link_int *l;
  static struct ether_addr *my_ether;
#else
  static struct libnet_ether_addr *my_ether;
#endif
  static unsigned char enet_bcast[6] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
  static unsigned char enet_nulls[6] = {0,0,0,0,0,0};
  static char *buf;

//  printf("sendarp who-has %s tell %s\n",ip2a(hisip),ip2a(myip));

  if (!initialized) {
    buf=malloc(1024);
    if (!buf) error("malloc failed");
#ifdef OLD_LIBNET
    l=libnet_open_link_interface(device, errbuf);
    if (!l) error2("open_link_interface failed");
    my_ether=libnet_get_hwaddr(l,device,buf);
    if (!my_ether) error2("libnet_get_hwaddr failed");
#else
    my_ether=libnet_get_hwaddr(l);
#endif
    initialized=1;
  }
#ifdef OLD_LIBNET
  /*
    *  Ethernet header
    */
  libnet_build_ethernet(
	  enet_bcast,             /* broadcast ethernet address */
	  my_ether->ether_addr_octet,    /* source ethernet address */
	  ETHERTYPE_ARP,          /* this frame holds an ARP packet */
	  NULL,                   /* payload */
	  0,                      /* payload size */
	  buf);                   /* packet header memory */

  /*
    *  ARP header
    */
  libnet_build_arp(
	  ARPHRD_ETHER,           /* hardware address type */
	  ETHERTYPE_IP,           /* protocol address type */
	  ETHER_ADDR_LEN,         /* hardware address length */
	  4,                      /* protocol address length */
	  ARPOP_REQUEST,          /* packet type - ARP request */
	  my_ether->ether_addr_octet,    /* source (local) ethernet address */
	  (u_char *)&myip,        /* source (local) IP address */
	  enet_nulls,             /* target's ethernet address (broadcast) */
	  (u_char *)&hisip,       /* target's IP address */
	  NULL,                   /* payload */
	  0,                      /* payload size */
	  buf + ETH_H);           /* packet header memory */

  if (libnet_write_link_layer(l, device, buf, LIBNET_ARP_H + LIBNET_ETH_H)==-1)
    error2("libnet_write_link_layer failed");
#else
  if (libnet_build_arp(
	  ARPHRD_ETHER,                           /* hardware addr */
	  ETHERTYPE_IP,                           /* protocol addr */
	  6,                                      /* hardware addr size */
	  4,                                      /* protocol addr size */
	  ARPOP_REQUEST,                          /* operation type */
	  my_ether->ether_addr_octet,             /* sender hardware addr */
	  (u_char *)&myip,                        /* sender protocol addr */
	  enet_nulls,                             /* target hardware addr */
	  (u_char *)&hisip,                       /* target protocol addr */
	  NULL,                                   /* payload */
	  0,                                      /* payload size */
	  l,                                      /* libnet handle */
	  0)==-1)                                 /* libnet id */
             error2("libnet_build_arp failed");
  if (libnet_autobuild_ethernet(
	  enet_bcast,                             /* ethernet destination */
	  ETHERTYPE_ARP,                          /* protocol type */
	  l)==-1)                                 /* libnet handle */
             error2("libnet_autobuild_ethernet failed");
  if (libnet_write(l)==-1)
    error2("libnet_write failed");
  libnet_clear_packet(l);
#endif
}

struct entry {
  unsigned int myip,netmask,gw;
  unsigned short pingips,proxyport;
  unsigned int pingip[10];
  char *description;
  char *proxy;
  char *resolvconf;
  char *script;
  struct entry *next;
};

int lines=0;

unsigned int parseip(const char *s) {
  unsigned int ip=0;
  unsigned char num=0;
  while (*s && isdigit(*s))
    num=num*10+*s++-'0';
  if (*s!='.')
    return -1;
  s++; ip=num; num=0;
  while (*s && isdigit(*s))
    num=num*10+*s++-'0';
  if (*s!='.')
    return -1;
  s++; ip=ip*256+num; num=0;
  while (*s && isdigit(*s))
    num=num*10+*s++-'0';
  if (*s!='.')
    return -1;
  s++; ip=ip*256+num; num=0;
  while (*s && isdigit(*s))
    num=num*10+*s++-'0';
  if (*s!=0)
    return -1;
  ip=ip*256+num;
  return ip;
}

void parse(char *ptr,struct entry *e) {
  char *colon=strchr(ptr,':');
  char *tmp=strchr(ptr,'/');
  memset(e,0,sizeof(e));
  if (tmp==0 || colon==0) goto parseerror;
  if (tmp>colon) goto parseerror;
  *tmp++=0;
  e->myip=parseip(ptr);
  {
    int bits=atoi(tmp);
    e->netmask=0xffffffff ^ ((1 << (32-bits))-1);
    ptr=colon+1;
  }

/*  printf("/sbin/ifconfig eth0 %s netmask %s\n",ip2a(e->myip),ip2a(e->netmask)); */

  colon=strchr(ptr,':');
  tmp=strchr(ptr,'|');
  if (colon==0) goto parseerror;
  *colon++=0;
  if (tmp && tmp<colon) {
    *tmp++=0;
    e->pingip[e->pingips++]=parseip(ptr);
    do {
      ptr=tmp;
      tmp=strchr(ptr,'|');
      if (tmp) {
	*tmp++=0;
	e->pingip[e->pingips++]=parseip(ptr);
	ptr=tmp;
      } else {
	e->pingip[e->pingips++]=parseip(ptr);
	ptr=colon;
      }
    } while (ptr<colon);
  } else {
    e->pingip[e->pingips++]=parseip(ptr);
    ptr=colon;
  }
  colon=strchr(ptr,':');
  if (colon==0) goto parseerror;
  *colon++=0;
  e->gw=parseip(ptr);
  ptr=colon;
  colon=strchr(ptr,':');
  if (colon==0) goto parseerror;
  *colon++=0;
  e->resolvconf=strdup(ptr);
  ptr=colon;
  colon=strchr(ptr,':');
  if (colon==0) goto parseerror;
  *colon++=0;
  e->proxy=strdup(ptr);
  ptr=colon;
  colon=strchr(ptr,':');
  if (colon==0) goto parseerror;
  *colon++=0;
  e->proxyport=atoi(ptr);
  ptr=colon;
  colon=strchr(ptr,':');
  if (colon==0) goto parseerror;
  *colon++=0;
  e->script=strdup(ptr);
  if (*(e->script) == 0) {
    free(e->script);
    e->script=0;
  }
  tmp=strchr(colon,'\n');
  if (tmp) *tmp=0;
  e->description=strdup(colon);
/*  e->dnsips--; e->pingips--; */
  return;
parseerror:
  fprintf(stderr,"parse error in /etc/divine.conf line %d\n",lines);
  exit(1);
}

struct entry *e;

void *send_arps(void *foo) {
  int count=6;
  for (;count;count--) {
    struct entry *tmp=e;
    struct timeval tv;
    while (tmp) {
      sendarp(0,htonl(tmp->myip));
      tmp=tmp->next;
    }
    tv.tv_sec=0;
    tv.tv_usec=300000;
    select(0,0,0,0,&tv);
    tmp=e;
    while (tmp) {
      int i;
      for (i=0; i<tmp->pingips; i++)
//	sendarp(htonl(tmp->myip),htonl(tmp->pingip[i]));
	sendarp(0,htonl(tmp->pingip[i]));
      tmp=tmp->next;
    }
    sleep(1);
    if (count>1)
      fprintf(stderr,"Resending ARP requests...\n");
  }
  fprintf(stderr,"Sorry, no answers received.\n");
  exit(2);
}

int main(int argc,char *argv[]) {
  FILE *f=fopen("/etc/divine.conf","r");
  char buf[1024];
  pthread_t t;
  int verbose=0;

  e=0;
  device=0;
  if (argc>1) {
    if (!strcmp(argv[1],"-v")) {
      verbose=1;
      argv[1]=argv[2];
      argc--;
    } else if (argc>2 && !strcmp(argv[2],"-v")) {
      verbose=1;
      argc--;
    }
  }
  if (argc>1) device=argv[1];
#ifdef OLD_LIBNET
  if (device==0) {
    struct sockaddr_in sin;
    if (libnet_select_device(&sin,&device,errbuf) == -1) {
      fprintf(stderr,"divine: could not select standard device\n");
      return -1;
    }
  }
#else
   l=libnet_init(LIBNET_LINK_ADV,device,errbuf);
   if (!l) error2("libnet_init failed");
   device=l->device;
#endif
  if (getuid()!=0) {
    fprintf(stderr,"Sorry, root only\n");
    exit(1);
  }
  putenv("PATH=/sbin:/bin:/usr/sbin:/usr/bin");
  if (f) {
    pcap_t *pd;
    u_char *packet;
    struct pcap_pkthdr pc_hdr;
    while (fgets(buf,1023,f)) {
      lines++;
      if (buf[0]=='#' || buf[0]=='\n')
	continue;
      if (e) {
	struct entry *tmp=malloc(sizeof(struct entry));
	parse(buf,tmp);
	tmp->next=e;
	e=tmp;
      } else {
	e=malloc(sizeof(struct entry));
	parse(buf,e);
      }
    }
    fclose(f);
    if (e==0) {
      fprintf(stderr,"No entries found.\n");
      exit(1);
    }
    if (pthread_create(&t,0,send_arps,0)) {
      fprintf(stderr,"Thread creation failed.\n");
      exit(1);
    }
    pd = pcap_open_live(device,50,0,500,errbuf);
    if (pd == NULL)
      error2("pcap_open_live failed");
    for (;(packet = ((u_char *)pcap_next(pd, &pc_hdr)));)
    {
      struct libnet_ethernet_hdr *p;
      struct ether_arp *a;
//      printf("packet!\n");
      p = (struct libnet_ethernet_hdr *)(packet);
      if (ntohs(p->ether_type) == ETHERTYPE_ARP) {
//	printf("arp packet!\n");
	a = (struct ether_arp *)(packet + 14);
	if (ntohs(a->arp_op) == ARPOP_REPLY) {
	  u_long source,target;
//	  printf("arp reply packet!1!!\n");
	  printf("hardware address length %d, protocol address length %d\n",a->arp_hln,a->arp_pln);
	  source=htonl(*(int*)(packet+14+8+a->arp_hln));
	  target=htonl(*(int*)(packet+14+8+a->arp_hln*2+a->arp_pln));
	  if (verbose)
	    printf("Got ARP reply from %s to %s\n",ip2a(source),ip2a(target));
	  {
	    struct entry *tmp=e;
	    while (tmp) {
	      int i;
	      if (tmp->myip==source) {	/* uh-oh: Ziel==Quelle, d.h. die IP ist schon vergeben */
		if (verbose)
		  printf("Blacklisting unavailable IP %s\n",ip2a(source));
		tmp->pingips=0;		/* sofort blacklisten */
	      }
	      for (i=0; i<tmp->pingips; i++)
		if (tmp->pingip[i]==source) {
		  int pid;
		  /* kill the sending thread, because we do not 
		     need further arps sent, and because some pcmcia cards
		     show weird failure behaviour, eg. xircom realport 
		     ethernet+modem: libnet_write_link_layer fails for 
		     the subsequent arps, and divine terminates despite
		     of successfully finding a config to setup. 
		     
		     az 2000-03-18
		     */
		  int result;
		  result = pthread_cancel(t);
		  if (result)
		    /* this is not good, but no reason to terminate: 
		       we'll just note it and continue */
		    fprintf(stderr,"pthread_cancel failed: %d\n",result);
		  fprintf(stderr,"Found: \"%s\"\n",tmp->description);
		  if (verbose) {
		    printf("ifconfig %s down\n",device);
		    fflush(stdout);
		  }
		  if ((pid=fork())==0) {
		    execlp("ifconfig","ifconfig",device,"down",0);
		    perror("ifconfig execlp failed");
		    _exit(0);
		  }
		  wait(0);
		  if (verbose) {
		    printf("ifconfig %s %s netmask %s broadcast %s\n",device,
			    ip2a(tmp->myip),ip2a(tmp->netmask),
			    ip2a((tmp->myip & tmp->netmask) + (0xffffffff-tmp->netmask)));
		    fflush(stdout);
		  }
		  if ((pid=fork())==0) {
		    execlp("ifconfig","ifconfig",device,ip2a(tmp->myip),
			   "netmask",ip2a(tmp->netmask),
			   "broadcast",ip2a((tmp->myip & tmp->netmask) +
					    (0xffffffff-tmp->netmask)),0);
		    perror("ifconfig execlp failed");
		    _exit(0);
		  }
		  wait(0);
		  if (verbose) {
		    printf("route add default gw %s\n",ip2a(tmp->gw));
		    fflush(stdout);
		  }
		  if ((pid=fork())==0) {
		    execlp("route","route","add","default","gw",ip2a(tmp->gw),0);
		    perror("route execlp failed");
		    _exit(0);
		  }
		  wait(0);
		  unlink("/etc/resolv.conf");
		  symlink(tmp->resolvconf,"/etc/resolv.conf");
		  if (tmp->proxyport) {
		    FILE *f=fopen("/etc/proxy","w");
		    fprintf(f,"%s:%d\n",tmp->proxy,tmp->proxyport);
		    fclose(f);
		  } else
		    unlink("/etc/proxy");
		  if (tmp->script) {
		    if (verbose) {
		      printf("%s \"%s\"\n",tmp->script,tmp->description);
		      fflush(stdout);
		    }
		    if ((pid=fork())==0) {
		      execlp(tmp->script,tmp->script,tmp->description,0);
		      perror("execlp failed");
		      _exit(0);
		    }
		    wait(0);
		  }
		  exit(0);
		}
	      tmp=tmp->next;
	    }
	  }
	}
      }
    }
  }
  fprintf(stderr,"Could not open /etc/divine.conf\n");
  exit(1);
}
