#!/usr/bin/env python
#
# upgrade-0.4-db.py - utility to convert database format from version 1 (0.4)
#
# Copyright (c) 2006-2007 The DITrack Project, www.ditrack.org.
#
# $Id: upgrade-0.4-db.py 1326 2007-02-26 03:46:34Z vss $
# $HeadURL: https://127.0.0.1/ditrack/src/tags/0.5/upgrade-0.4-db.py $
#
# Redistribution and use in source and binary forms, with or without 
# modification, are permitted provided that the following conditions are met:
#
#  * Redistributions of source code must retain the above copyright notice, 
# this list of conditions and the following disclaimer.
#  * Redistributions in binary form must reproduce the above copyright notice, 
# this list of conditions and the following disclaimer in the documentation 
# and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
# POSSIBILITY OF SUCH DAMAGE.
#

import email
import os
import re
import sys
import shutil

if len(sys.argv) != 2:
    sys.stderr.write("""
This is a utility to convert DITrack database from format 1 (DITrack versions
0.3, 0.3.1 and 0.4) to format 2 (DITrack 0.5).

Usage:
    %s DBPATH

""" % sys.argv[0])

    sys.exit(1)

dbpath = sys.argv[1]

version = os.popen("svn pg ditrack:format %s" % dbpath, "r").readline().strip()

if version != "1":
    sys.stderr.write("Unsupported database format version: %s" % version)
    sys.exit(1)


class UserCfg:
    """
    A representation of user accounts configuration.
    """

    def __contains__(self, key):
        return self.items.has_key(key)

    def __init__(self, fname):
        """
        Parses user accounts configuration file 'fname'.
        """

        users = {}
        f = open(fname)
        while 1:
            str = f.readline()
            if not str: break

            user = str.strip()

            if user in users:
                print >> sys.stderr, "Duplicate user entry '" + user + "' in" \
                    + " '" + fname + "'"

            users[user] = user

        f.close()

        self.items = users

    def __len__(self):
        return len(self.items)

    has_key = __contains__

class VersionCfg:
    """
    A representation of version sets configuration.
    """

    def __contains__(self, key):
        return self.items.has_key(key)

    def __getitem__(self, key):
        return self.items[key]

    def __init__(self, path):
        f = open(path)
        versions = email.message_from_file(f)
        f.close()

        if len(versions.get_payload()):
            print >> sys.stderr, "Empty line in version set configuration file"

        self.items = {}
        for s in versions.keys():

            if s in self.items:
                print >> sys.stderr, "Duplicate version set '" + s \
                    + "' definition"

            v = versions[s].strip().split("/")
            if len(v) != 3:
                print >> sys.stderr, "Invalid version set '" + s \
                    + "' definition"

            self.items[s] = dict.fromkeys(
                [ "past", "current", "future" ],
                map(lambda x: x.strip().split(), v)
                )

class Category:
    def __init__(self, version_set, versions, default_owner):
        self.version_set = version_set
        self.versions = versions
        self.default_owner = default_owner

class CategoryCfg(object):
    """
    Representation of per-database category configuration.
    """

    def __getitem__(self, key):
        return self.items[key]

    def __init__(self, cfgfile, users, versions):
        """
        Parse category configuration file 'cfgfile'.

        Defined versions and users are passed in 'versions' and 'users'
        parameters respectively. They are used only for consistency checks and
        are not stored anywhere inside the object.
        """

        f = open(cfgfile)
        categories = email.message_from_file(f)
        f.close()

        if len(categories.get_payload()):
            print >> sys.stderr, "Blank line in category configuration file '" \
                + cfgfile + "'"

        category = {}
        for c in categories.keys():
            s = categories[c].strip().split()
            if len(s) != 2:
                print >> sys.stderr, "Invalid category '" + c + "' definition"

            attr = {}
            for p in s:
                    av = p.split("=")
                    if len(av) != 2:
                        print >> sys.stderr, "Invalid category '" + c \
                            + "' definition (attribute/value pair)"

                    attr[av[0]] = av[1]

            if not attr.has_key("versions"):
                print >> sys.stderr, "Invalid category '" + c + "' " \
                    + "definition: no versions defined"

            if attr["versions"] not in versions:
                print >> sys.stderr, "Invalid category '" + c \
                    + "' definition: version set '" + attr["versions"] \
                    + "' undefined"

            if not attr.has_key("default-owner"):
                print >> sys.stderr, "Invalid category" + \
                    "'" + c + "' definition: no default owner defined"

            if not users.has_key(attr["default-owner"]):
                print >> sys.stderr, "Invalid category" + \
                    "'" + c + "' definition: default owner '" \
                    + attr["default-owner"] + "' undefined"

            if c in category:
                print >> sys.stderr, "Duplicate " + \
                    "category '" + c + "' definition"

            category[c] = Category(
                attr["versions"],
                versions[attr["versions"]],
                attr["default-owner"]
                )

        self.items = category

    def __iter__(self):
        return self.items.__iter__()

    def __len__(self):
        return len(self.items)


users = UserCfg(os.path.join(dbpath, "etc", "users"))
versions = VersionCfg(os.path.join(dbpath, "etc", "versions"))
categories_path = os.path.join(dbpath, "etc", "categories")
categories = CategoryCfg(categories_path, users, versions)

categories_path_new = categories_path + ".new"
categories_path_old = categories_path + ".old"
f = open(categories_path_new, "w")

for name in categories:
    c = categories[name]
    f.write("Category: %s\n" % name)
    f.write("Default-owner: %s\n" % c.default_owner)
    f.write("Version-set: %s\n" % c.version_set)
    f.write("\n")
f.close()

os.system("mv %s %s" % (categories_path, categories_path_old))
os.system("mv %s %s" % (categories_path_new, categories_path))

os.system("svn ps ditrack:format 2 %s" % dbpath)

issue_re = re.compile("i\\d+$")

datapath = os.path.join(dbpath, "data")

for iname in os.listdir(datapath):

    if not issue_re.match(iname):
        continue

    sys.stdout.write("issue: %s " % os.path.join(datapath,iname))

    if os.system("svn ls %s" % os.path.join(datapath,iname)):
        shutil.rmtree(os.path.join(datapath,iname))
        print("not in repositary. Deleted")
        continue

    print("converting")

    commentname = os.path.join(datapath, iname, "comment0")
    infoname = os.path.join(datapath, iname, "info")

    f = open(infoname)
    info = email.message_from_file(f)
    f.close()

    if "Resolution" in info:
        del info["Resolution"]

    header = {}

    cnum = 1
    while 1:
        cname = os.path.join(datapath, iname, "comment%d" % cnum)

        if not os.path.exists(cname):
            break

        f = open(cname)
        comment = email.message_from_file(f)
        f.close()

        remove = []
        for k in info.keys():
            h = "DT-Old-%s" % k

            if h in comment:
                header[k] = comment[h]
                remove.append(k)

        for k in remove:
            del info[k]

        cnum += 1

    for k in info.keys():
        header[k] = info[k]

    f = open(commentname, "w")

    for k in header:
        f.write("DT-Old-%s:\nDT-New-%s: %s\n" % (k, k, header[k]))

    f.write("\n%s" % info.get_payload())

    f.close()

    os.system("svn add %s" % commentname)
    os.system("svn rm %s" % infoname)
