/**
 * ViewPort.js - Code to create a viewport window in a web browser.
 *
 * Requires prototypejs 1.6+ and scriptaculous 1.8+
 * Requires DimpSlider.js.
 *
 * $Horde: dimp/js/src/ViewPort.js,v 1.194.2.33 2008/05/23 17:46:40 slusarz Exp $
 *
 * Copyright 2005-2008 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file COPYING for license information (GPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/gpl.html.
 */

/**
 * ViewPort
 */
var ViewPort = Class.create({

    // Required: content_container, lines, fetch_action, template,
    //           cachecheck_action, ajaxRequest, buffer_pages,
    //           limit_factor, content_class, row_class, selected_class
    // Optional: show_split_pane
    initialize: function(opts)
    {
        opts.content = $(opts.content_container);
        opts.empty = opts.empty_container ? $(opts.empty_container) : null;
        opts.error = opts.error_container ? $(opts.error_container) : null;
        this.opts = opts;

        this.scroller = new ViewPort_Scroller(this);
        this.template = new Template(opts.template);

        this.current_req_lookup = $H();
        this.current_req = $H();
        this.fetch_hash = $H();
        this.slice_hash = $H();
        this.views = $H();

        this.showSplitPane(opts.show_split_pane);

        // Initialize all other variables
        this.isbusy = this.line_height = this.page_size = this.splitbar = this.splitbar_loc = this.uc_run = this.view = this.viewport_init = null;
        this.request_num = 1;
    },

    // view = ID of view. Can not contain a '%' character.
    // params = TODO
    // search = (object) Search parameters
    // background = Load view in background?
    loadView: function(view, params, search, background)
    {
        var buffer, curr, init, opts = {}, ps;

        this.clearWait();

        // Need a page size before we can continue - this is what determines
        // the slice size to request from the server.
        if (this.page_size === null) {
            ps = (this.show_split_pane) ? this.getDefaultPageSize() : this.getMaxPageSize();
            if (isNaN(ps)) {
                this.loadView.bind(this, view, params, search, background).defer();
                return;
            }
            this.page_size = ps;
        }

        buffer = this._getBuffer();
        if (buffer) {
            if (!background && this.view) {
                // Need to store current buffer to save current offset
                this.views.set(this.view, { buffer: buffer, offset: this.currentOffset() });
            }
            curr = this.views.get(view);
        } else {
            init = true;
        }

        if (background) {
            opts = { noupdate: true, view: view };
        } else {
            this.view = view;
            if (!this.viewport_init) {
                this.viewport_init = 1;
                this.renderViewport();
            }
        }

        if (curr) {
            this.setMetaData('additional_params', $H(params), view);
            this.updateContent(curr.offset, opts);
            if (!background) {
                if (this.opts.onComplete) {
                    this.opts.onComplete();
                }
                this.checkCache();
            }
            return true;
        } else if (!init) {
            this.clear();
        }

        buffer = this._getBuffer(null, true);
        this.views.set(view, { buffer: buffer, offset: 0 });
        this.setMetaData({ additional_params: $H(params) }, view);
        this.fetchBuffer((search) ? 'search' : 'offset', (search) ? search : 0, opts);

        return false;
    },

    // view = ID of view
    deleteView: function(view)
    {
        this.views.unset(view);
    },

    // rownum = Row number
    scrollTo: function(rownum)
    {
        switch (this.isVisible(rownum)) {
        case -1:
            this.scroller.moveScroll(rownum - 1);
            break;

        // case 0:
        // noop

        case 1:
            this.scroller.moveScroll(rownum - this.getPageSize());
            break;
        }

        return true;
    },

    // rownum = Row number
    isVisible: function(rownum)
    {
        var offset = this.currentOffset();
        return (rownum < offset + 1) ? -1 :
               ((rownum > (offset + this.getCurrPageSize())) ? 1 : 0);
    },

    // params = TODO
    reload: function(params)
    {
        if (this.isFiltering()) {
            this.filter.filter(null, params);
        } else {
            this.fetchBuffer('offset', this.currentOffset(), null, $H(params).merge({ update: 1 }).toObject());
        }
    },

    refresh: function()
    {
        this.requestContentRefresh(this.currentOffset());
    },

    // vs = (Viewport_Selection) A Viewport_Selection object.
    // opts = (object) TODO [cacheid, noupdate, view]
    remove: function(vs, opts)
    {
        if (this.isbusy) {
            this.remove.bind(this, vs, cacheid, view).defer();
            return;
        }

        if (!vs.size()) {
            return;
        }

        opts = opts || {};
        this.isbusy = true;

        var args,
            i = 0,
            visible = vs.get('div'),
            vsize = visible.size();

        this.deselect(vs);

        if (opts.cacheid) {
            this.setMetaData({ cacheid: opts.cacheid }, opts.view);
        }

        // If we have visible elements to remove, only call refresh after
        // the last effect has finished.
        if (vsize) {
            // Set 'to' to a value slightly above 0 to prevent Effect.Fade
            // from auto hiding.  Hiding is unnecessary, since we will be
            // removing from the document shortly.
            args = { duration: 0.3, to: 0.01 };
            visible.each(function(v) {
                if (++i == vsize) {
                    args.afterFinish = this._removeids.bind(this, vs, opts);
                }
                Effect.Fade(v, args);
            }, this);
        } else {
            this._removeids(vs, opts);
        }
    },

    // vs = (Viewport_Selection) A Viewport_Selection object.
    // opts = (object) TODO [noupdate, view]
    _removeids: function(vs, opts)
    {
        this.setMetaData({ total_rows: this.getMetaData('total_rows', opts.view) - vs.size() }, opts.view);

        if (this.opts.onRemoveRows) {
            this.opts.onRemoveRows(vs);
        }

        this._getBuffer().remove(vs.get('rownum'));
        if (!opts.noupdate) {
            this.refresh();
        }
        this.isbusy = false;
    },

    // action = TODO
    // callback = TODO
    addFilter: function(action, callback)
    {
        this.filter = new ViewPort_Filter(this, action, callback);
    },

    // val = TODO
    // params = TODO
    runFilter: function(val, params)
    {
        if (this.filter) {
            this.filter.filter(Object.isUndefined(val) ? null : val, params);
        }
    },

    isFiltering: function()
    {
        return this.filter ? this.filter.isFiltering() : false;
    },

    // reset = (boolean) If true, don't update the viewport
    stopFilter: function(reset)
    {
        if (this.filter) {
            this.filter.clear(reset);
        }
    },

    // noupdate = (boolean) TODO
    onResize: function(noupdate)
    {
        if (!this.uc_run || !this.opts.content.visible()) {
            return;
        }

        if (this.opts.onBeforeResize) {
            this.opts.onBeforeResize();
        }

        this.renderViewport(noupdate);

        if (this.opts.onAfterResize) {
            this.opts.onAfterResize();
        }
    },

    // offset = (integer) TODO
    requestContentRefresh: function(offset, opts)
    {
        var b = this._getBuffer(), limit;

        if (b.sliceLoaded(offset)) {
            this.updateContent(offset);
            limit = b.isNearingLimit(offset);
            if (limit) {
                this.fetchBuffer('offset', offset, { noupdate: true }, { nearing: limit, page_size: this.getPageSize() });
            }
            return true;
        }

        this.fetchBuffer('offset', offset);
        return false;
    },

    // type = (string) 'search' or 'offset'
    // value = TODO
    // opts = (object) TODO [noupdate, view]
    // params = TODO
    fetchBuffer: function(type, value, opts, params)
    {
        opts = opts || {};

        // Only call onFetch() if we are loading in foreground.
        if (this.opts.onFetch && !opts.noupdate) {
            this.opts.onFetch();
        }

        var action = this.opts.fetch_action,
            cr,
            offset_list,
            request_id,
            request_string,
            request_vals,
            view = (opts.view || this.view);

        cr = this.current_req.get(view);
        params = this.addRequestParams(params, view);
        params.set(type, (Object.isNumber(value)) ? value.toJSON() : Object.toJSON(value));
        request_vals = [ view, type, value ];

        if (this.isFiltering()) {
            action = this.filter.getAction();
            params = this.filter.addFilterParams(params);
            // Need to capture filter params changes in the request ID
            request_vals.push(params.toJSON());
        }

        // Generate a unique request ID value based on the type, value, and
        // filter params. Since javascript does not have a native md5()
        // function, use a local lookup table instead.
        request_string = request_vals.join('|');
        request_id = this.fetch_hash.get(request_string);
        if (!request_id) {
            request_id = this.fetch_hash.set(request_string, this.request_num++);
        }
        params.set('request_id', request_id);

        if (cr) {
            if (cr.requests[request_id]) {
                // Check for repeat request.  We technically should never
                // reach here but if we do, make sure we don't go into an
                // infinite loop.
                if (++cr.requests[request_id].count == 4) {
                    this.displayFetchError();
                    return;
                }
            } else if (type == 'offset') {
                // Check for message list requests that are requesting
                // (essentially) the same message slice - such as two
                // scroll down requests sent in quick succession.  If the
                // original request will contain the slice needed by the
                // second request, ignore the later request and just
                // reposition the viewport on display.
                request_old_id = cr.offset_list.detect(function(p) {
                    return p.value.include(value + this.getPageSize());
                }, this);
                if (request_old_id) {
                    this.addRequest(view, request_old_id.key, { offset: value });
                    return;
                } else if (!opts.noupdate) {
                    // Set all other pending requests to noupdate, since the
                    // current request is now the active request.
                    Object.keys(cr.requests).each(function(k) {
                        this.addRequest(view, k, { noupdate: true });
                    }, this);
                }
            }
        }

        if (type == 'offset') {
            offset_list = $R(value, value + Math.min(value + params.get('buffer_size'), this.getMetaData('total_rows', view)));
        }

        this.addRequest(view, request_id, { o_list: offset_list, noupdate: opts.noupdate });

        this.opts.ajaxRequest(action, params);
        this.handleWait();
    },

    // args = (object) TODO
    // view = (string) The view requested.
    // Returns a Hash object
    addRequestParams: function(args, view)
    {
        var b = this._getBuffer(view),
            cid = this.getMetaData('cacheid', view),
            params = this.getMetaData('additional_params', view);
        if (cid) {
            params.update({ cacheid: cid });
        }
        return params.merge(args).merge({ buffer_size: b.bufferSize(), offset: this.currentOffset() });
    },

    // r = (Object) viewport response object.
    //     Common properties:
    //         id
    //         request_id
    //         type: 'list', 'slice' (DEFAULT: list)
    //
    //     Properties needed for type 'list':
    //         cacheid
    //         data
    //         label
    //         total_rows
    //         offset (optional)
    //         other
    //         rowlist
    //         totalrows
    //         update (optional)
    //
    //     Properties needed for type 'slice':
    //         data (object) - rownum is the only required property
    ajaxResponse: function(r)
    {
        if (this.isbusy) {
            this.ajaxResponse.bind(this, r).defer();
            return;
        }

        this.isbusy = true;
        this.clearWait();

        var buffer, cr, cr_id, data, datakeys, id, rowlist = {};

        if (r.type == 'slice') {
            data = r.data;
            datakeys = Object.keys(data);
            datakeys.each(function(k) {
                data[k].view = r.id;
                rowlist[k] = data[k].rownum;
            });
            buffer = this._getBuffer(r.id);
            buffer.update(data, rowlist, { slice: true });
            if (this.opts.onEndFetch) {
                this.opts.onEndFetch();
            }
            cr = this.slice_hash.get(r.request_id);
            if (cr) {
                cr(new ViewPort_Selection(buffer, 'uid', datakeys));
                this.slice_hash.unset(r.request_id);
            }
            this.isbusy = false;
            return;
        }

        id = (r.request_id) ? this.current_req_lookup.get(r.request_id) : r.id;
        cr = this.current_req.get(id);
        if (cr && r.request_id) {
            cr_id = cr.requests[r.request_id];
        }

        if (this.viewport_init) {
            this.viewport_init = 2;
        }

        buffer = this._getBuffer(id);
        buffer.update(r.data, r.rowlist, { update: r.update });
        buffer.setMetaData($H(r.other).merge({
            cacheid: r.cacheid,
            label: r.label,
            total_rows: r.totalrows
        }));

        if (r.request_id) {
            this.removeRequest(id, r.request_id);
        }

        this.isbusy = false;

        // Don't update the view if we are now in a different view, or if
        // we are loading in the background.
        if (!(this.view == id || r.search) ||
            (cr_id && cr_id.noupdate) ||
            !this.updateContent((cr_id && cr_id.offset) ? cr_id.offset : (r.offset ? parseInt(r.offset) : 0))) {
            return;
        }

        if (this.opts.onComplete) {
            this.opts.onComplete();
        }

        if (this.opts.onEndFetch) {
            this.opts.onEndFetch();
        }
    },

    // Adds a request to the current request queue.
    // Requests are stored by view ID. Under each ID is the following:
    //   offset_list: (array) TODO
    //   requests: (array) Stored by request ID
    //     count: (integer) Number of times slice has attempted to be loaded
    //     noupdate: (boolean) Do not update view
    //     offset: (integer) The offset to use
    // params = noupdate, o_list, offset
    addRequest: function(view, r_id, params)
    {
        var req = this.current_req.get(view);
        if (!req) {
            req = { offset_list: $H(), requests: {} };
        }
        if (params.o_list) {
            req.offset_list.set(r_id, params.o_list);
        }

        if (!req.requests[r_id]) {
            req.requests[r_id] = { count: 1 };
        }
        ['noupdate', 'offset'].each(function(p) {
            if (!Object.isUndefined(params[p])) {
                req.requests[r_id][p] = params[p];
            }
        });

        this.current_req.set(view, req);
        this.current_req_lookup.set(r_id, view);
    },

    // Removes a request to the current request queue.
    removeRequest: function(view, r_id)
    {
        var cr = this.current_req.get(view);
        if (cr) {
            if (Object.keys(cr.requests).size() == 1) {
                this.current_req.unset(view);
            } else {
                delete cr.requests[r_id];
                this.current_req.update(view, cr);
            }
        }
        this.current_req_lookup.unset(r_id);
    },

    // offset = (integer) TODO
    // opts = (object) TODO [view]
    updateContent: function(offset, opts)
    {
        opts = opts || {};

        if (!this._getBuffer(opts.view).sliceLoaded(offset)) {
            this.fetchBuffer('offset', offset, opts);
            return false;
        }

        if (!this.uc_run) {
            // Code for viewport that only needs to be initialized once.
            this.uc_run = true;
            if (this.opts.onFirstContent) {
                this.opts.onFirstContent();
            }
        }

        this.clearMsgList();

        var page_size = this.getPageSize(),
            rows,
            sel = this.getSelected();

        this.scroller.updateSize();
        this.scroller.moveScroll((this.getMetaData('total_rows', opts.view) > page_size) ? offset : 0, { noupdate: true });

        offset = this.currentOffset();
        rows = this.createSelection('rownum', $A($R(offset + 1, offset + page_size)));

        rows.get('dataob').each(function(row) {
            this.opts.content.insert(this.template.evaluate(row));
            if (sel.contains('uid', row.vp_id)) {
                $(row.domid).addClassName(this.opts.selected_class);
            }
        }, this);

        if (!rows.size() && this.opts.empty && this.viewport_init != 1) {
            // If loading a viewport for the first time, show a blank
            // viewport rather than the empty viewport status message.
            this.opts.content.update(this.opts.empty.innerHTML);
        }

        if (this.opts.onContent) {
            this.opts.onContent(rows);
        }

        return true;
    },

    displayFetchError: function()
    {
        if (this.opts.onFail) {
            this.opts.onFail();
        }
        if (this.opts.error) {
            this.opts.content.update(this.opts.error.innerHTML);
        }
    },

    checkCache: function()
    {
        this.opts.ajaxRequest(this.opts.fetch_action, this.addRequestParams({ checkcache: 1 }));
    },

    // rows = (array) An array of row numbers
    // callback = (function; optional) A callback function to run after we
    //            retrieve list of rows from server. Callback function
    //            receives one parameter - a ViewPort_Selection object
    //            containing the slice.
    // Return: Either a ViewPort_Selection object or false if the server needs
    //         to be queried.
    getSlice: function(rows, callback)
    {
        var params = { rangeslice: 1, start: rows.min(), length: rows.size() },
            r_id,
            slice;

        slice = this.createSelection('rownum', rows);
        if (rows.size() == slice.size()) {
            return slice;
        }

        if (this.opts.onFetch) {
            this.opts.onFetch();
        }
        if (callback) {
            r_id = this.request_num++;
            params.request_id = r_id;
            this.slice_hash.set(r_id, callback);
        }
        this.opts.ajaxRequest(this.opts.fetch_action, this.addRequestParams(params));
        return false;
    },

    handleWait: function(call)
    {
        this.clearWait();

        // Server did not respond in defined amount of time.  Alert the
        // callback function and set the next timeout.
        if (call && this.opts.onWait) {
            this.opts.onWait();
        }

        // Call wait handler every x seconds
        if (this.opts.viewport_wait) {
            this.waitHandler = this.handleWait.bind(this, true).delay(this.opts.viewport_wait);
        }
    },

    clearWait: function()
    {
        if (this.waitHandler) {
            clearTimeout(this.waitHandler);
            this.waitHandler = null;
        }
    },

    clearMsgList: function()
    {
        var c = this.opts.content;

        if (this.opts.onClearRows) {
            this.opts.onClearRows(c.childElements());
        }

        $A(c.childNodes).each(function(n) {
            c.removeChild(n);
        });
    },

    clear: function()
    {
        this.clearMsgList();
        this.scroller.clear();
    },

    visibleRows: function()
    {
        return this.opts.content.childElements();
    },

    getMetaData: function(id, view)
    {
        return this._getBuffer(view).getMetaData(id);
    },

    setMetaData: function(vals, view)
    {
        this._getBuffer(view).setMetaData(vals);
    },

    _getBuffer: function(view, create)
    {
        if (!create) {
            var b = this.views.get(view || this.view);
            if (b) {
                return b.buffer;
            }
        }
        return new ViewPort_Buffer(this, this.opts.buffer_pages, 0);
    },

    currentOffset: function()
    {
        return this.scroller.currentOffset();
    },

    // vs = (Viewport_Selection) A Viewport_Selection object.
    // flag = (string) Flag name.
    // add = (boolean) Whether to set/unset flag.
    updateFlag: function(vs, flag, add)
    {
        this._updateFlag(vs, flag, add, this.isFiltering());
        this._updateClass(vs, flag, add);
    },

    // vs = (Viewport_Selection) A Viewport_Selection object.
    // flag = (string) Flag name.
    // add = (boolean) Whether to set/unset flag.
    // filter = (boolean) Are we filtering results?
    _updateFlag: function(vs, flag, add, filter)
    {
        vs.get('dataob').each(function(r) {
            r.bg = (add) ? $w(r.bg).concat([ flag ]).join(' ') : $w(r.bg).without([ flag ]).join(' ');
            if (filter) {
                this._updateFlag(this.createSelection('uid', r.vp_id, r.view), flag, add);
            }
        }, this);
    },

    // vs = (Viewport_Selection) A Viewport_Selection object.
    // flag = (string) Flag name.
    // add = (boolean) Whether to set/unset flag.
    _updateClass: function(vs, flag, add)
    {
        vs.get('div').each(function(d) {
            if (add) {
                d.addClassName(flag);
            } else {
                d.removeClassName(flag);
            }
        });
    },

    getBufferSize: function(view)
    {
        return this._getBuffer(view).size;
    },

    getLineHeight: function()
    {
        if (this.line_height) {
            return this.line_height;
        }

        // To avoid hardcoding the line height, create a temporary row to
        // figure out what the CSS says.
        var d = new Element('DIV', { className: this.opts.content_class }).insert(new Element('DIV', { className: this.opts.row_class })).hide();
        document.body.appendChild(d);
        this.line_height = d.getHeight();
        d.remove();

        return this.line_height;
    },

    getPageSize: function()
    {
        return this.page_size;
    },

    getCurrPageSize: function()
    {
        return Math.min(this.getPageSize(), this.getMetaData('total_rows'));
    },

    getDefaultPageSize: function()
    {
        return Math.max(parseInt(this.getMaxPageSize() * 0.45), 5);
    },

    getMaxPageSize: function()
    {
        return parseInt(this.getMaxHeight() / this.getLineHeight());
    },

    getMaxHeight: function()
    {
        return document.viewport.getHeight() - this.opts.content.viewportOffset()[1];
    },

    showSplitPane: function(show)
    {
        this.show_split_pane = show;
        this.onResize();
    },

    renderViewport: function(noupdate)
    {
        if (!this.viewport_init) {
            return;
        }

        var lh, pane, panesize;

        // Get split pane dimensions
        if (this.opts.split_pane) {
            lh = this.getLineHeight();
            pane = $(this.opts.split_pane);
            if (this.show_split_pane) {
                if (!pane.visible()) {
                    this.initSplitBar();
                    this.page_size = (this.splitbar_loc) ? this.splitbar_loc : this.getDefaultPageSize();
                }
                panesize = this.getMaxHeight() - (lh * (this.page_size + 1));
            } else {
                if (pane.visible()) {
                    this.splitbar_loc = this.page_size;
                    $(pane, this.splitbar).invoke('hide');
                }
                this.page_size = this.getMaxPageSize();
            }
        }

        $(this.opts.content).setStyle({ height: (lh * this.page_size) + 'px' });
        if (panesize) {
            pane.setStyle({ height: panesize + 'px' });
            $(pane, this.splitbar).invoke('show');
        }

        if (!noupdate) {
            this.scroller.onResize();
        }
    },

    initSplitBar: function()
    {
        if (this.splitbar) {
            return;
        }

        this.splitbar = $(this.opts.splitbar);
        new Draggable(this.opts.splitbar, {
            constraint: 'vertical',
            ghosting: true,
            onStart: function() {
                // Cache these values since we will be using them multiple
                // times in snap().
                var lh = this.getLineHeight();
                this.sp = { lh: lh, pos: $(this.opts.content).positionedOffset()[1], max: parseInt((this.getMaxHeight() - 100) / lh) };
            }.bind(this),
            snap: function(x, y) {
                var l = parseInt((y - this.sp.pos) / this.sp.lh);
                if (l < 1) {
                    l = 1;
                } else if (l > this.sp.max) {
                    l = this.sp.max;
                }
                this.sp.lines = l;
                return [ 0, this.sp.pos + (l * this.sp.lh) ];
            }.bind(this),
            onEnd: function() {
                this.splitbar.setStyle({ top: 0 });
                this.page_size = this.sp.lines;
                this.renderViewport();
            }.bind(this)
        });
    },

    createSelection: function(format, data, view)
    {
        var buffer = this._getBuffer(view);
        return buffer ? new ViewPort_Selection(buffer, format, data) : new ViewPort_Selection(this._getBuffer(this.view));
    },

    getViewportSelection: function(view)
    {
        var buffer = this._getBuffer(view);
        return this.createSelection('uid', buffer ? buffer.getAllUIDs() : [], view);
    },

    // vs = (Viewport_Selection | array) A Viewport_Selection object -or-, if
    //       opts.range is set, an array of row numbers.
    // opts = (object) TODO [add, range]
    select: function(vs, opts)
    {
        opts = opts || {};

        if (opts.range) {
            vs = this.getSlice(vs, this.select.bind(this));
            if (vs === false) {
                return;
            }
        }

        var b = this._getBuffer(),
            sel;

        if (!opts.add) {
            sel = this.getSelected();
            b.deselect(sel, true);
            this._updateClass(sel, this.opts.selected_class, false);
        }
        b.select(vs);
        this._updateClass(vs, this.opts.selected_class, true);
        if (this.opts.selectCallback) {
            this.opts.selectCallback(vs, opts);
        }
    },

    // vs = (Viewport_Selection) A Viewport_Selection object.
    // opts = (object) TODO [clearall]
    deselect: function(vs, opts)
    {
        opts = opts || {};

        if (!vs.size()) {
            return;
        }

        this._getBuffer().deselect(vs, opts && opts.clearall);
        this._updateClass(vs, this.opts.selected_class, false);
        if (this.opts.deselectCallback) {
            this.opts.deselectCallback(vs, opts)
        }
    },

    getSelected: function()
    {
        return Object.clone(this._getBuffer().getSelected());
    }

});

/**
 * ViewPort_Scroller
 */
var ViewPort_Scroller = Class.create({

    initialize: function(vp)
    {
        this.vp = vp;

        // Initialize other variables
        this.pagesize = this.scrollsize = this.totalrows = null;
    },

    createScrollBar: function()
    {
        if (this.scrollDiv) {
            return;
        }
        var c = this.vp.opts.content;

        // Create the outer div.
        this.scrollsize = c.getHeight();
        this.scrollDiv = new Element('DIV', { className: 'sbdiv', style: 'height:' + this.scrollsize + 'px;' }).hide();

        // Create the cursor element.
        this.cursorDiv = new Element('DIV', { className: 'sbcursor' });

        // Create scrollbar object.
        this.scrollbar = new DimpSlider(this.cursorDiv, this.scrollDiv, { axis: 'vertical', onClickTrack: this.onClickTrackHandler.bind(this), onChange: this.onScroll.bind(this), onSlide: this.vp.opts.onSlide ? this.vp.opts.onSlide : null });

        // Add scrollbar to parent viewport
        this.scrollDiv.appendChild(this.cursorDiv);
        c.parentNode.insertBefore(this.scrollDiv, c.nextSibling);

        // Give our parent a right margin just big enough to
        // accomodate the scrollbar.
        c.setStyle({ marginRight: '-' + this.scrollDiv.getWidth() + 'px' });

        // Mouse wheel handler.
        c.observe(Prototype.Browser.IE ? 'mousewheel' : 'DOMMouseScroll', function(e) {
            var move_num = this.vp.getPageSize();
            move_num = (move_num > 3) ? 3 : move_num;
            this.moveScroll(this.currentOffset() + ((e.wheelDelta >= 0 || e.detail < 0) ? (-1 * move_num) : move_num));
            if (this.vp.opts.onMouseScroll) {
                this.vp.opts.onMouseScroll(e);
            }
            e.stop();
        }.bindAsEventListener(this));
    },

    onResize: function()
    {
        if (!this.scrollDiv) {
            return;
        }

        // Update the container div.
        this.scrollsize = this.vp.opts.content.getHeight();
        this.scrollDiv.setStyle({ height: this.scrollsize + 'px' });

        // Update the scrollbar size.
        this.scrollbar.updateTrackLength();

        // Update the cursor size
        this.updateSize();

        // Update displayed content.
        this.vp.requestContentRefresh(this.currentOffset());
    },

    updateSize: function()
    {
        this.createScrollBar();
        this.pagesize = this.vp.getPageSize();
        this.totalrows = this.vp.getMetaData('total_rows');
        if (this.totalrows <= this.pagesize) {
            this.vp.opts.content.setStyle({ cssFloat: 'none' });
            this.scrollDiv.hide();
        } else {
            this.scrollDiv.show();
            this.vp.opts.content.setStyle({ cssFloat: 'left' });
            // Minimum cursor size = 10px
            this.scrollbar.setHandleLength(Math.max(10, Math.round((this.pagesize / this.totalrows) * this.scrollsize)), true);
        }
    },

    clear: function()
    {
        if (this.scrollDiv) {
            this.totalrows = 0;
            this.scrollDiv.hide();
        }
    },

    // offset = (integer) Offset to move the scrollbar to
    // opts = (object) TODO [noupdate]
    moveScroll: function(offset, opts)
    {
        offset = this.offsetVal(offset);
        if (this.scrollDiv && this.scrollDiv.visible() && offset != this.scrollbar.getValue()) {
            this.scrollbar.setScrollPosition(offset, opts && opts.noupdate);
        }
    },

    offsetVal: function(offset)
    {
        var offset_rows = this.totalrows - this.pagesize;
        return Math.min(Math.max(offset, 0), offset_rows) / offset_rows;
    },

    onClickTrackHandler: function(v)
    {
        var dir = ((v - this.scrollbar.translateToPx(this.scrollbar.getValue()).replace(/px$/,"")) < 0) ? -1 : 1;
        this.moveScroll(this.currentOffset() - (1 * dir) + (this.pagesize * dir));
        this.scrollbar.clearClick();
    },

    onScroll: function()
    {
        if (this.vp.opts.onScroll) {
            this.vp.opts.onScroll();
        }

        this.vp.requestContentRefresh(this.currentOffset());

        if (this.vp.opts.onScrollIdle) {
            this.vp.opts.onScrollIdle();
        }
    },

    currentOffset: function()
    {
        return (!this.scrollDiv || !this.scrollDiv.visible()) ? 0 : Math.max(Math.round(this.scrollbar.getValue() * (this.totalrows - this.pagesize)), 0);
    }

});

/**
 * ViewPort_Buffer
 *
 * Note: recognize the difference between offset (current location in the
 * viewport - starts at 0) with start parameters (the row numbers - starts
 * at 1).
 */
var ViewPort_Buffer = Class.create({

    initialize: function(vp, b_pages, l_factor)
    {
        this.bufferPages = b_pages;
        this.limitFactor = l_factor;
        this.vp = vp;
        this.clear();
    },

    limitTolerance: function()
    {
        return Math.round(this.bufferSize() * (this.limitFactor / 100));
    },

    bufferSize: function()
    {
        // Buffer size must be at least the maximum page size.
        return Math.round(Math.max(this.vp.getMaxPageSize() + 1, this.bufferPages * this.vp.getPageSize()));
    },

    // d = TODO
    // l = TODO
    // opts = (object) TODO [slice, update]
    update: function(d, l, opts)
    {
        d = $H(d);
        l = $H(l);
        opts = opts || {};

        if (opts.slice) {
            d.each(function(o) {
                if (!this.data.get(o.key)) {
                    this.data.set(o.key, o.value);
                    this.inc.set(o.key, true);
                }
            }, this);
        } else {
            if (this.data.size()) {
                this.data.update(d);
                if (this.inc.size()) {
                    d.keys().each(function(k) {
                        this.inc.unset(k);
                    }, this);
                }
            } else {
                this.data = d;
            }
        }

        this.uidlist = (opts.update) ? l : (this.uidlist.size() ? this.uidlist.merge(l) : l);

        if (opts.update) {
            this.rowlist = $H();
        }
        l.each(function(o) {
            this.rowlist.set(o.value, o.key);
        }, this);
    },

    // offset = (integer) Offset of the beginning of the slice.
    sliceLoaded: function(offset)
    {
        return !this._rangeCheck($A($R(offset + 1, Math.min(offset + this.vp.getMaxPageSize() - 1, this.getMetaData('total_rows')))));
    },

    isNearingTopLimit: function(offset)
    {
        if (offset == 0) {
            return false;
        }
        return this._rangeCheck($A($R(Math.max(offset + 1 - this.limitTolerance(), 1), offset)));
    },

    isNearingBottomLimit: function(offset)
    {
        // Search for missing messages in reverse order since in normal usage
        // (sequential scrolling through the message list) messages are
        // more likely to be missing at furthest from the current view.
        return this._rangeCheck($A($R(offset + 1, Math.min(offset + this.limitTolerance() + this.vp.getMaxPageSize() - 1, this.getMetaData('total_rows')))).reverse());
    },

    isNearingLimit: function(offset)
    {
        if (this.uidlist.size() != this.getMetaData('total_rows')) {
            if (this.isNearingTopLimit(offset)) {
                return 'top';
            } else if (this.isNearingBottomLimit(offset)) {
                return 'bottom';
            }
        }
        return false;
    },

    _rangeCheck: function(range)
    {
        var i = this.inc.size();
        return range.any(function(o) {
            var g = this.rowlist.get(o);
            return (Object.isUndefined(g) || (i && this.inc.get(g)));
        }, this);
    },

    getData: function(uids)
    {
        return uids.collect(function(u) {
            var e = this.data.get(u);
            if (!Object.isUndefined(e)) {
                // We can directly write the rownum to the original object
                // since we will always rewrite when creating rows.
                e.domid = 'vp_row' + u;
                e.rownum = this.uidlist.get(u);
                e.vp_id = u;
                return e;
            }
        }, this).compact();
    },

    getAllUIDs: function()
    {
        return this.uidlist.keys();
    },

    rowsToUIDs: function(rows)
    {
        return rows.collect(function(n) {
            return this.rowlist.get(n);
        }, this).compact();
    },

    // vs = (Viewport_Selection) TODO
    select: function(vs)
    {
        this.selected.add('uid', vs.get('uid'));
    },

    // vs = (Viewport_Selection) TODO
    // clearall = (boolean) Clear all entries?
    deselect: function(vs, clearall)
    {
        if (clearall) {
            this.selected.clear();
        } else {
            this.selected.remove('uid', vs.get('uid'));
        }
    },

    getSelected: function()
    {
        return this.selected;
    },

    // rownums = (array) Array of row numbers to remove.
    remove: function(rownums)
    {
        var newsize,
            rowsize = this.rowlist.size(),
            rowsubtract = 0;
        newsize = rowsize - rownums.size();

        $A($R(rownums.min(), rowsize)).each(function(n) {
            var id = this.rowlist.get(n), r;
            if (rownums.include(n)) {
                this.data.unset(id);
                this.uidlist.unset(id);
                rowsubtract++;
            } else {
                r = n - rowsubtract;
                this.rowlist.set(r, id);
                this.uidlist.set(id, r);
            }
            if (n > newsize) {
                this.rowlist.unset(n);
            }
        }, this);
    },

    clear: function()
    {
        this.data = $H();
        this.inc = $H();
        this.mdata = $H({ total_rows: 0 });
        this.rowlist = $H();
        this.selected = new ViewPort_Selection(this);
        this.uidlist = $H();
    },

    getMetaData: function(id)
    {
        return this.mdata.get(id);
    },

    setMetaData: function(vals)
    {
        this.mdata.update(vals);
    }

});

/**
 * ViewPort_Filter
 */
var ViewPort_Filter = Class.create({

    initialize: function(vp, action, callback)
    {
        this.vp = vp;
        this.action = action;
        this.callback = callback;

        // Initialize other variables
        this.filterid = 0;
        this.filtering = this.last_filter = this.last_folder = this.last_folder_params = null;
    },

    filter: function(val, params)
    {
        params = params || {};
        val = (val === null) ? this.last_filter : val;
        if (!val) {
            this.clear();
            return;
        }

        val = this.last_filter = val.toLowerCase();

        if (this.filtering) {
            this.vp.fetchBuffer('offset', 0, {}, params);
            return;
        }

        this.filtering = ++this.filterid + '%search%';
        this.last_folder = this.vp.view;
        this.last_folder_params = this.vp.getMetaData('additional_params').merge(params);

        // Filter visible rows immediately.
        var c = this.vp.opts.content, delrows;
        delrows = c.childElements().findAll(function(n) {
            return n.getText(true).toLowerCase().indexOf(val) == -1;
        });
        if (this.vp.opts.onClearRows) {
            this.vp.opts.onClearRows(delrows);
        }
        delrows.invoke('remove');
        this.vp.scroller.clear();
        if (this.vp.opts.empty && !c.childElements().size()) {
            c.update(this.vp.opts.empty.innerHTML);
        }

        this.vp.loadView(this.filtering, this.last_folder_params);
    },

    isFiltering: function()
    {
        return this.filtering;
    },

    getAction: function()
    {
        return this.action;
    },

    // params is a Hash object
    addFilterParams: function(params)
    {
        if (!this.filtering) {
            return params;
        }

        params.update({ filter: this.last_filter });

        // Get parameters from a callback function, if defined.
        if (this.callback) {
            params.update(this.callback());
        }

        return params;
    },

    clear: function(reset)
    {
        if (this.filtering) {
            this.filtering = null;
            if (!reset) {
                this.vp.loadView(this.last_folder, this.last_folder_params);
            }
            this.vp.deleteView(this.filtering);
            this.last_filter = this.last_folder = null;
        }
    }

});

/**
 * ViewPort_Selection
 */
var ViewPort_Selection = Class.create({

    // Formats:
    //     'dataob' = Data objects
    //     'div' = DOM DIVs
    //     'domid' = DOM IDs
    //     'rownum' = Row numbers
    //     'uid' = Unique IDs
    initialize: function(buffer, format, data)
    {
        this.buffer = buffer;
        this.clear();
        if (!Object.isUndefined(format)) {
            this.add(format, data);
        }
    },

    add: function(format, d)
    {
        var c = this._convert(format, d);
        this.data = (this.data.size()) ? this.data.concat(c).uniq() : c;
    },

    remove: function(format, d)
    {
        this.data = this.data.without.apply(this.data, this._convert(format, d));
    },

    _convert: function(format, d)
    {
        d = Object.isArray(d) ? d : [ d ];
        switch (format) {
        case 'dataob':
            return d.pluck('vp_id');

        case 'div':
            return d.pluck('id').invoke('substring', 6);

        case 'domid':
            return d.invoke('substring', 6);

        case 'rownum':
            return this.buffer.rowsToUIDs(d);

        case 'uid':
            return d;
        }
    },

    clear: function()
    {
        this.data = [];
    },

    get: function(format)
    {
        format = Object.isUndefined(format) ? 'uid' : format;
        if (format == 'uid') {
            return this.data;
        }
        var d = this.buffer.getData(this.data);

        switch (format) {
        case 'dataob':
            return d;

        case 'div':
            return d.pluck('domid').collect(function(e) { return $(e); }).compact();

        case 'domid':
            return d.pluck('domid');

        case 'rownum':
            return d.pluck('rownum');
        }
    },

    contains: function(format, d)
    {
        return this.data.include(this._convert(format, d).first());
    },

    // params = (Hash) Search strings; key is search key, val is search string
    // The search string can be an array of values, a single value, or a
    // single RegExp.
    search: function(params)
    {
        return new ViewPort_Selection(this.buffer, 'uid', this.get('dataob').collect(function(i) {
            if ($H(params).all(function(s) {
                if (Object.isArray(s.value)) {
                    return s.value.include(i[s.key]);
                } else if (s.value instanceof RegExp) {
                    return i[s.key].match(s.value);
                } else {
                    return (i[s.key] == s.value);
                }
            })) {
                return i.vp_id;
            }
        }).compact());
    },

    size: function()
    {
        return this.data.size();
    }

});
