/**
 * DimpBase.js - Javascript used in the base DIMP page.
 *
 * $Horde: dimp/js/src/DimpBase.js,v 1.1.2.44 2008/05/23 17:48:07 slusarz Exp $
 *
 * Copyright 2005-2008 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file COPYING for license information (GPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/gpl.html.
 */

var DimpBase = {
    // Vars used and defaulting to null/false:
    //   filtertoggle, fl_visible, folder, folderswitch, isvisible,
    //   message_list_template, offset, pollpe, pp, searchobserve, uid,
    //   viewport
    delay_onload: (!Prototype.Browser.Gecko && !Prototype.Browser.Opera),
    drags: $H(),
    lastrow: -1,
    pivotrow: -1,
    showPreview: DIMP.conf.preview_pref,

    sfiltersfolder: $H({ sf_all: 'all', sf_current: 'current' }),
    sfilters: $H({ sf_msgall: 'msgall', sf_from: 'from', sf_to: 'to', sf_subject: 'subject' }),

    unseen_regex: new RegExp("(^|\\s)unseen(\\s|$)"),

    // Message selection functions

    // vs = (ViewPort_Selection) A ViewPort_Selection object.
    // opts = (object) Boolean options [delay, right]
    _select: function(vs, opts)
    {
        var d = vs.get('rownum');
        if (d.size() == 1) {
            this.lastrow = this.pivotrow = d.first();
        }

        this.toggleButtons();

        if ($('previewPane').visible()) {
            if (opts.right) {
                this.clearPreviewPane();
            } else {
                if (opts.delay) {
                    this.initPreviewPane.bind(this).delay(opts.delay);
                } else {
                    this.initPreviewPane();
                }
            }
        }
    },

    // vs = (ViewPort_Selection) A ViewPort_Selection object.
    // opts = (object) Boolean options [right]
    _deselect: function(vs, opts)
    {
        opts = opts || {};
        this.viewport.deselect(vs, opts);

        var sel = this.viewport.getSelected(),
            count = sel.size();
        if (!count) {
            this.lastrow = this.pivotrow = -1;
        }

        this.toggleButtons();
        if (opts.right || !count) {
            this.clearPreviewPane();
        } else if ((count == 1) && $('previewPane').visible()) {
            this._loadPreview(sel.get('dataob').first());
        }
    },

    // id = (string) DOM ID
    // opts = (Object) Boolean options [ctrl, right, shift]
    msgSelect: function(id, opts)
    {
        var bounds,
            row = this.viewport.createSelection('domid', id),
            rownum = row.get('rownum').first(),
            sel = this.isSelected('domid', id),
            selcount = this.selectedCount();

        this.lastrow = rownum;

        if (opts.shift) {
            if (selcount) {
                if (!sel || selcount != 1) {
                    bounds = [ rownum, this.pivotrow ];
                    this.viewport.select($A($R(bounds.min(), bounds.max())), { range: true });
                }
                return;
            }
        } else if (opts.ctrl) {
            this.pivotrow = rownum;
            if (sel) {
                this.viewport.deselect(row, { right: opts.right });
                return;
            } else if (opts.right || selcount) {
                this.viewport.select(row, { add: true, right: opts.right });
                return;
            }
        }

        this.viewport.select(row, { right: opts.right });
    },

    selectAll: function()
    {
        this.viewport.select($A($R(1, this.viewport.getMetaData('total_rows'))), { range: true });
    },

    isSelected: function(format, data)
    {
        return this.viewport.getSelected().contains(format, data);
    },

    selectedCount: function()
    {
        return (this.viewport) ? this.viewport.getSelected().size() : 0;
    },

    resetSelected: function()
    {
        if (this.viewport) {
            this.viewport.deselect(this.viewport.getSelected(), { clearall: true });
        }
        this.toggleButtons();
        this.clearPreviewPane();
    },

    // absolute = Is num an absolute row number - from 1 -> page_size (true) -
    //            or a relative change from the current selected value (false)
    //            If no current selected value, or more than 1 current
    //            selected value, the first message in the current viewport is
    //            selected.
    moveSelected: function(num, absolute)
    {
        var curr,
            curr_row,
            row,
            row_data,
            sel = this.viewport.getSelected();
        curr_row = (sel.size() == 1) ? sel.get('dataob').first() : null;

        if (absolute) {
            if (!this.viewport.getMetaData('total_rows')) {
                return;
            }
            curr = num;
        } else {
            if (num == 0) {
                return;
            }

            if (curr_row === null) {
                curr = this.viewport.currentOffset() + 1;
            } else {
                curr = curr_row.rownum + num;
                curr = (num > 1) ? Math.min(curr, this.viewport.getMetaData('total_rows')) : Math.max(curr, 1);
            }
        }

        row = this.viewport.createSelection('rownum', curr);
        if (row.size()) {
            row_data = row.get('dataob').first();
            if (!curr_row || row_data.imapuid != curr_row.imapuid) {
                this.viewport.scrollTo(row_data.rownum);
                this.viewport.select(row, { delay: 0.3 });
            }
        } else {
            this.offset = curr;
            this.viewport.requestContentRefresh(curr - 1);
        }
    },
    // End message selection functions

    go: function(loc, data)
    {
        var app, f, separator;

        if (loc.startsWith('compose:')) {
            return;
        }

        if (loc.startsWith('msg:')) {
            separator = loc.indexOf(':', 4);
            f = loc.substring(4, separator);
            this.uid = loc.substring(separator + 1);
            loc = 'folder:' + f;
            // Now fall through to the 'folder:' check below.
        }

        if (loc.startsWith('folder:')) {
            f = loc.substring(7);
            if (this.folder != f || !$('dimpmain_folder').visible()) {
                this.highlightSidebar(this.getFolderId(f));
                if (!$('dimpmain_folder').visible()) {
                    $('dimpmain_portal').hide();
                    $('dimpmain_folder').show();
                }
                this.addHistory(loc);
            }
            this.loadFolder(f);
            return;
        }

        this.folder = null;
        $('dimpmain_folder').hide();
        $('dimpmain_portal').update(DIMP.text.loading).show();

        if (loc.startsWith('app:')) {
            app = loc.substr(4);
            if (app == 'imp' || app == 'dimp') {
                this.go('folder:INBOX');
                return;
            }
            this.highlightSidebar('app' + app);
            this.addHistory(loc, data);
            if (data) {
                this.iframeContent(loc, data);
            } else if (DIMP.conf.app_urls[app]) {
                this.iframeContent(loc, DIMP.conf.app_urls[app]);
            }
            return;
        }

        switch (loc) {
        case 'portal':
            this.highlightSidebar('appportal');
            this.addHistory(loc);
            DimpCore.setTitle(DIMP.text.portal);
            DimpCore.doAction('ShowPortal', {}, [], this.portalCallback.bind(this));
            break;

        case 'options':
            this.highlightSidebar('appoptions');
            this.addHistory(loc);
            DimpCore.setTitle(DIMP.text.prefs);
            this.iframeContent(loc, DIMP.conf.prefs_url);
            break;
        }
    },

    addHistory: function(loc, data)
    {
        if (Horde.dhtmlHistory.getCurrentLocation() != loc) {
            Horde.dhtmlHistory.add(loc, data);
        }
    },

    highlightSidebar: function(id)
    {
        // Folder bar may not be fully loaded yet.
        if ($('foldersLoading').visible()) {
            this.highlightSidebar.bind(this, id).defer();
            return;
        }

        $('sidebarPanel').select('.on').invoke('removeClassName', 'on');

        var elt = $(id);
        if (!elt) {
            return;
        }
        if (!elt.match('LI')) {
            elt = elt.up();
            if (!elt) {
                return;
            }
        }
        elt.addClassName('on');

        // Make sure all subfolders are expanded
        elt.ancestors().each(function(n) {
            if (n.hasClassName('subfolders')) {
                this.toggleSubFolder(n.id.substring(3), 'exp');
            } else if (n.id == 'foldersSidebar') {
                $break;
            }
        }, this);
    },

    checkQuota: function()
    {
        DimpCore.doAction('CheckQuota', {}, [], this.checkQuotaCallback.bind(this));
    },

    checkQuotaCallback: function(r)
    {
        if (r.response.quota) {
            $('quota').update(r.response.quota);
        }
    },

    iframeContent: function(name, location)
    {
        if (name === null) {
            name = location;
        }

        var container = $('dimpmain_portal'), iframe;
        if (!container) {
            DimpCore.showNotifications([ { type: 'horde.error', message: 'Bad portal!' } ]);
            return;
        }

        iframe = new Element('IFRAME', { id: 'iframe' + name, className: 'iframe', frameborder: 0 });
        iframe.writeAttribute('src', location);

        // Hide menu in prefs pages.
        if (name == 'options') {
            iframe.observe('load', function() { $('iframeoptions').contentWindow.document.getElementById('menu').style.display = 'none'; });
        }

        container.insert(iframe);
    },

    // r = ViewPort row data
    msgWindow: function(r)
    {
        this.updateUnseenUID(r, 0);
        var url = DIMP.conf.message_url;
        url += (url.include('?') ? '&' : '?') +
               $H({ folder: r.view,
                    uid: r.imapuid }).toQueryString();
        DimpCore.popupWindow(url, 'msgview');
        return false;
    },

    composeMailbox: function(type)
    {
        var sel = this.viewport.getSelected();
        if (!sel.size()) {
            return;
        }
        sel.get('dataob').each(function(s) {
            DimpCore.compose(type, { folder: s.view, uid: s.imapuid });
        });
    },

    loadFolder: function(f, background)
    {
        if (!this.viewport) {
            this.createViewPort();
        }

        if (!background) {
            this.resetSelected();

            if (this.folder == f) {
                this.searchfilterClear(false);
                return;
            }

            this.searchfilterClear(true);
            $('folderName').update(DIMP.text.loading);
            $('msgHeader').update();
            this.folderswitch = true;
            this.folder = f;
        }

        this.viewport.loadView(f, { folder: f }, this.uid ? { imapuid: DimpCore.toUIDString(this.uid, f) } : null, background);
    },

    createViewPort: function()
    {
        var mf = $('msgList_filter'),
            settitle = this.setMessageListTitle.bind(this);

        this.viewport = new ViewPort({
            content_container: 'msgList',
            empty_container: 'msgList_empty',
            error_container: 'msgList_error',
            fetch_action: 'ListMessages',
            template: this.message_list_template,
            buffer_pages: DIMP.conf.buffer_pages,
            limit_factor: DIMP.conf.limit_factor,
            viewport_wait: DIMP.conf.viewport_wait,
            show_split_pane: this.showPreview,
            split_pane: 'previewPane',
            splitbar: 'splitBar',
            content_class: 'msglist',
            row_class: 'msgRow',
            selected_class: 'selectedRow',
            ajaxRequest: DimpCore.doAction.bind(DimpCore),
            onScrollIdle: settitle,
            onSlide: settitle,
            onViewChange: function() {
                DimpCore.addGC(this.viewport.visibleRows());
            }.bind(this),
            onContent: function(rows) {
                var thread = ((this.viewport.getMetaData('sortby') == DIMP.conf.sortthread) && this.viewport.getMetaData('thread'));
                rows.get('dataob').each(function(row) {
                    var r = $(row.domid), size, subj, tn, u;
                    // Add thread graphics
                    if (thread && thread[row.imapuid]) {
                        subj = r.down('.msgSubject');
                        u = thread[row.imapuid];
                        tn = subj.firstChild;
                        $R(0, u.length, true).each(function(i) {
                            subj.insertBefore($($('thread_img_' + u.charAt(i)).cloneNode(false)).writeAttribute('id', ''), tn);
                        });
                    }
                    // Add attachment graphics
                    if (row.atc) {
                        size = r.down('.msgSize');
                        size.insertBefore($($('atc_img_' + row.atc).cloneNode(false)).writeAttribute('id', ''), size.firstChild);
                    }
                }, this);
                this.setMessageListTitle();
            }.bind(this),
            onComplete: function() {
                var l, row;
                if (this.uid) {
                    row = this.viewport.getViewportSelection().search({ imapuid: this.uid, view: this.folder });
                    if (row.size() && this.viewport.scrollTo(row.get('rownum').first())) {
                        this.viewport.select(row);
                    }
                } else if (this.offset) {
                    this.viewport.select(this.viewport.createSelection('rownum', this.offset));
                }
                this.offset = this.uid = null;

                // 'label' will not be set if there has been an error
                // retrieving data from the server.
                l = this.viewport.getMetaData('label');
                if (l) {
                    $('folderName').update(l);
                }

                if (this.folderswitch) {
                    this.folderswitch = false;
                    if (this.folder == DIMP.conf.spam_folder) {
                        if (DimpCore.buttons.indexOf('button_spam') != -1) {
                            [ $('button_spam').up(), $('ctx_message_spam') ].invoke('hide');
                        }
                        if (DimpCore.buttons.indexOf('button_ham') != -1) {
                            [ $('button_ham').up(), $('ctx_message_ham') ].invoke('show');
                        }
                    } else {
                        if (DimpCore.buttons.indexOf('button_spam') != -1) {
                            [ $('button_spam').up(), $('ctx_message_spam') ].invoke('show');
                        }
                        if (DimpCore.buttons.indexOf('button_ham') != -1) {
                            [ $('button_ham').up(), $('ctx_message_ham') ].invoke('hide');
                        }
                    }

                    $('msgFromLink').update(this.viewport.getMetaData('special') ? DIMP.text.totext : DIMP.text.from);
                    this.setSortColumns();
                } else if (this.filtertoggle) {
                    if (this.filtertoggle == 1 &&
                        this.viewport.getMetaData('sortby') == DIMP.conf.sortthread) {
                        this.setSortColumns(DIMP.conf.sortdate);
                    } else {
                        this.setSortColumns();
                        this.updateTitle();
                    }
                    this.filtertoggle = 0;
                }

                if (this.viewport.isFiltering()) {
                    this.resetSelected();
                    this.updateTitle();
                } else {
                    this.setFolderLabel(this.folder, this.viewport.getMetaData('unseen'));
                }
            }.bind(this),
            onFetch: this.msgListLoading.bind(this, true),
            onEndFetch: this.msgListLoading.bind(this, false),
            onWait: function() {
                if ($('dimpmain_folder').visible()) {
                    DimpCore.showNotifications([ { type: 'horde.warning', message: DIMP.text.listmsg_wait } ]);
                }
            },
            onFail: function() {
                if ($('dimpmain_folder').visible()) {
                    DimpCore.showNotifications([ { type: 'horde.error', message: DIMP.text.listmsg_timeout } ]);
                }
                this.msgListLoading(false);
            }.bind(this),
            onFirstContent: function() {
                this.clearPreviewPane();
                var m = $('msgList');
                m.observe('dblclick', this.handleMsgListDblclick.bindAsEventListener(this));
                m.observe('mouseover', this.handleMsgListMouseover.bindAsEventListener(this));
            }.bind(this),
            onClearRows: function(r) {
                r.pluck('id').each(function(id) {
                    DimpCore.removeMouseEvents(id);
                });
                DimpCore.addGC(r);
            },
            onBeforeResize: function() {
                var sel = this.viewport.getSelected();
                this.isvisible = (sel.size() == 1) && (this.viewport.isVisible(sel.get('rownum').first()) == 0);
            }.bind(this),
            onAfterResize: function() {
                if (this.isvisible) {
                    this.viewport.scrollTo(this.viewport.getSelected().get('rownum').first());
                }
            }.bind(this),
            selectCallback: this._select.bind(this),
            deselectCallback: this._deselect.bind(this)
        });

        // DIMP's version of Droppables is smart enough to ignore Draggable
        // items that don't need any droppable listeners.
        $('splitBar').writeAttribute('nodrop', true);

        // If starting in no preview mode, need to set the no preview class
        if (!this.showPreview) {
            $('msgList').addClassName('msglistNoPreview');
        }

        // Set up viewport filter events.
        this.viewport.addFilter('ListMessages', this.addSearchfilterParams.bind(this));
        mf.observe('keyup', this.searchfilterOnKeyup.bind(this));
        mf.observe('focus', this.searchfilterOnFocus.bind(this));
        mf.observe('blur', this.searchfilterOnBlur.bind(this));
        mf.addClassName('msgFilterDefault');

        // Resize message pane on window size change.
        Event.observe(window, 'resize', this.viewport.onResize.bind(this.viewport));
    },

    addMouseEvents: function(parentfunc, p)
    {
        var elt;

        try {
            if (!this.drags.get(p.id)) {
                switch (p.type) {
                case 'draft':
                case 'message':
                    this.drags.set(p.id, new DraggableMail(p.id, DraggableMailOpts));
                    elt = $(p.id).down('div.msCheck');
                    if (elt.visible()) {
                        elt.observe('mousedown', this.handleMsgListCheckbox.bindAsEventListener(this));
                        elt.observe('contextmenu', Event.stop);
                    }
                    break;

                case 'container':
                case 'folder':
                    // TODO: Re-add folder dragging code when stable.
                    break;

                case 'special':
                    // For purposes of the contextmenu, treat special folders
                    // like regular folders.
                    p.type = 'folder';
                    break;
                }
            }
            p.onShow = this.onMenuShow.bind(this);
            parentfunc(p);
        } catch (e) {}
    },

    removeMouseEvents: function(parentfunc, id)
    {
        if (this.drags.get(id)) {
            this.drags.unset(id).destroy();
            DimpCore.addGC($(id).down('div.msCheck'));
        }
        parentfunc(id);
    },

    onMenuShow: function(ctx)
    {
        var elts, folder, ob, sel;

        switch (ctx.ctx) {
        case 'ctx_folder':
            elts = $('ctx_folder_create', 'ctx_folder_rename', 'ctx_folder_delete');
            folder = DimpCore.DMenu.element();
            if (folder.readAttribute('mbox') == 'INBOX') {
                elts.invoke('hide');
            } else if (DIMP.conf.fixed_folders.indexOf(folder.readAttribute('mbox')) != -1) {
                elts.shift();
                elts.invoke('hide');
            } else {
                elts.invoke('show');
            }

            if (folder.hasAttribute('u')) {
                $('ctx_folder_poll').hide();
                $('ctx_folder_nopoll').show();
            } else {
                $('ctx_folder_poll').show();
                $('ctx_folder_nopoll').hide();
            }
            break;

        case 'ctx_message':
            ob = this.viewport.createSelection('domid', ctx.id).get('dataob').first();
            if (ob.listmsg) {
                $('ctx_message_reply_list').show();
            } else {
                $('ctx_message_reply_list').hide();
            }
            break;

        case 'ctx_reply':
            sel = this.viewport.getSelected();
            if (sel.size() == 1) {
                ob = sel.get('dataob').first();
            }
            if (ob && ob.listmsg) {
                $('ctx_reply_reply_list').show();
            } else {
                $('ctx_reply_reply_list').hide();
            }
            break;

        case 'ctx_otheractions':
            $('oa_seen', 'oa_unseen', 'oa_flagged', 'oa_clear', 'oa_sep1', 'oa_blacklist', 'oa_whitelist', 'oa_sep2').compact().invoke(this.viewport.getSelected().size() ? 'show' : 'hide');
            break;
        }
        return true;
    },

    handleMsgListDblclick: function(e)
    {
        e = this.getMsgRow(e);
        if (!e) {
            return;
        }
        var row = this.viewport.createSelection('domid', e.id).get('dataob').first();
        return row.draft ? DimpCore.compose('resume', { folder: row.view, uid: row.imapuid }) : this.msgWindow(row);
    },

    handleMsgListMouseover: function(e)
    {
        e = this.getMsgRow(e);
        if (!e) {
            return;
        }
        DimpCore.addMouseEvents({ id: e.id, type: this.viewport.createSelection('domid', e.id).get('dataob').first().menutype });
    },

    handleMsgListCheckbox: function(e)
    {
        var elt = this.getMsgRow(e);
        if (!elt) {
            return;
        }
        this.msgSelect(elt.readAttribute('id'), { ctrl: true, right: true });
        e.stop();
    },

    getMsgRow: function(e)
    {
        e = e.element();
        if (e && !e.hasClassName('msgRow')) {
            e = e.up('.msgRow');
        }
        return e;
    },

    updateTitle: function()
    {
        var elt, label, unseen;
        if (this.viewport.isFiltering()) {
            label = DIMP.text.search + ' :: ' + this.viewport.getMetaData('total_rows') + ' ' + DIMP.text.resfound;
        } else {
            elt = $(this.getFolderId(this.folder));
            if (elt) {
                unseen = elt.readAttribute('u');
                label = elt.readAttribute('l');
                if (unseen > 0) {
                    label += ' (' + unseen + ')';
                }
            } else {
                label = this.viewport.getMetaData('label');
            }
        }
        DimpCore.setTitle(label);
    },

    sort: function(sortby, e)
    {
        // Don't change sort if we are past the sortlimit
        if (this.viewport.getMetaData('sortlimit')) {
            return;
        }

        // As of now, sort() is not always called by a prototype event
        // handler. Thus, we need to manually call extend() on the event ob.
        e = Event.extend(e);

        // See if we're switching between two sorts on the same
        // column.
        var clicked = e.element(), pn, s, text;
        if (clicked.nodeType == 3) {
            clicked = clicked.parentNode;
        }
        if (clicked.match('small')) {
            // Swap the major and minor sorts on the column.
            sortby = clicked.readAttribute('sortby');
            pn = clicked.up();
            text = clicked.getText();
            clicked.writeAttribute('sortby', pn.readAttribute('sortby'));
            pn.writeAttribute('sortby', sortby);
            clicked.setText(pn.getText());
            pn.setText(text);
        }
        if (sortby == this.viewport.getMetaData('sortby')) {
            s = { sortdir: (this.viewport.getMetaData('sortdir') ? 0 : 1) };
            this.viewport.setMetaData({ sortdir: s.sortdir });
        } else {
            s = { sortby: sortby };
            this.viewport.setMetaData({ sortby: s.sortby });
        }
        this.setSortColumns();
        this.viewport.reload(s);
    },

    setSortColumns: function(sortby)
    {
        if (Object.isUndefined(sortby)) {
            sortby = this.viewport.getMetaData('sortby');
        }
        var arrsort = (sortby == DIMP.conf.sortarrival),
            dsort = $('sortdate'),
            ssort = $('sortsubject'),
            tsort = (sortby == DIMP.conf.sortthread);

        // Set Subject/Thread sort
        ssort.update(tsort ? DIMP.text.thread : DIMP.text.subject);
        ssort.writeAttribute('sortby', tsort ? DIMP.conf.sortthread : DIMP.conf.sortsubject);

        // Set Date/Arrival sort
        dsort.update(arrsort ? DIMP.text.arrival : DIMP.text.datesort);
        dsort.writeAttribute('sortby', arrsort ? DIMP.conf.sortarrival : DIMP.conf.sortdate);

        if (!this.viewport.isFiltering()) {
            ssort.appendChild(new Element('SMALL', { id: 'sortsubject_small', sortby: (tsort ? DIMP.conf.sortsubject : DIMP.conf.sortthread) }).insert(tsort ? DIMP.text.subject : DIMP.text.thread));
            if (this.viewport.getMetaData('sortlimit')) {
                $('sortsubject_small').hide();
            } else {
                $('sortsubject_small').visible();
            }
        }

        dsort.appendChild(new Element('SMALL', { sortby: (arrsort ? DIMP.conf.sortdate : DIMP.conf.sortarrival) }).insert(arrsort ? DIMP.text.datesort : DIMP.text.arrival));

        $('dimpmain_folder').select('.item').invoke('childElements').flatten().each(function(node) {
            if (!node.match('div')) {
                return;
            }
            var setsort = false;
            if (node.hasClassName('msgFrom')) {
                setsort = (sortby == DIMP.conf.sortfrom);
            } else if (node.hasClassName('msgSubject')) {
                setsort = (sortby == DIMP.conf.sortsubject || sortby == DIMP.conf.sortthread);
            } else if (node.hasClassName('msgDate')) {
                setsort = (sortby == DIMP.conf.sortdate || sortby == DIMP.conf.sortarrival);
            } else if (node.hasClassName('msgSize')) {
                setsort = (sortby == DIMP.conf.sortsize);
            }

            if (setsort) {
                this.isSort(node);
            } else {
                this.notSort(node);
            }
        }, this);
    },

    isSort: function(node)
    {
        var sortdir = this.viewport.getMetaData('sortdir');
        node.removeClassName(sortdir ? 'sortdown' : 'sortup');
        node.addClassName(sortdir ? 'sortup' : 'sortdown');
    },

    notSort: function(node)
    {
        node.removeClassName('sortup');
        node.removeClassName('sortdown');
    },

    // Preview pane functions
    togglePreviewPane: function()
    {
        this.showPreview = !this.showPreview;
        $('previewtoggle').setText(this.showPreview ? DIMP.text.hide_preview : DIMP.text.show_preview);
        if (this.showPreview) {
            $('msgList').removeClassName('msglistNoPreview');
        } else {
            $('msgList').addClassName('msglistNoPreview');
        }
        new Ajax.Request(DIMP.conf.URI_PREFS, { parameters: { app: 'dimp', pref: 'show_preview', value: this.showPreview ? 1 : 0 } });
        this.viewport.showSplitPane(this.showPreview);
        if (this.showPreview) {
            this.initPreviewPane();
        }
        this.updateTitle();
    },

    _loadPreview: function(data)
    {
        var pp = $('previewPane'), pp_offset;
        if (!pp.visible()) {
            return;
        }
        if (this.pp &&
            this.pp == data) {
            return;
        }
        this.pp = data;

        pp_offset = pp.positionedOffset();
        $('msgLoading').setStyle({ position: 'absolute', top: (pp_offset.top + 10) + 'px', left: (pp_offset.left + 10) + 'px' }).show();

        DimpCore.doAction('ShowPreview', {}, DimpCore.toUIDArray(this.viewport.createSelection('dataob', data)), this._loadPreviewCallback.bind(this));
    },

    _loadPreviewCallback: function(resp)
    {
        var pm = $('previewMsg'), r = resp.response, row, search, tmp, tmp2;

        if (!r.error) {
            search = this.viewport.getViewportSelection(r.view).search({ vp_id: r.uid });
            if (search.size()) {
                row = search.get('dataob').first();
                this.updateUnseenUID(row, 0);
            }
        }

        if (this.pp &&
            this.pp.vp_id != r.uid) {
            return;
        }

        if (this.viewport.getSelected().size() != 1) {
            this.clearPreviewPane();
            return;
        }

        if (r.error) {
            DimpCore.showNotifications([ { type: r.errortype, message: r.error } ]);
            this.clearPreviewPane();
            return;
        }

        DimpCore.removeAddressLinks(pm);

        DIMP.conf.msg_index = r.index;
        DIMP.conf.msg_folder = r.folder;
        DIMP.conf.msg_source_link = r.source_link;

        // Add subject/priority
        tmp = pm.select('.subject');
        tmp.invoke('update', r.subject);
        switch (r.priority) {
        case 'high':
        case 'low':
            tmp.invoke('insert', { top: $($(r.priority + '_priority_img').cloneNode(false)).writeAttribute('id', false) });
            break;
        }

        // Add from/date
        pm.select('.from').invoke('update', r.from);
        $('msgHeadersColl').select('.date').invoke('update', r.minidate);
        $('msgHeaderDate').select('.date').invoke('update', r.fulldate);

        // Add to/cc
        [ 'to', 'cc' ].each(function(a) {
            if (r[a]) {
                pm.select('.' + a).invoke('update', r[a]);
            }
            [ $('msgHeader' + a.capitalize()) ].invoke(r[a] ? 'show' : 'hide');
        });

        // Add attachment information
        $('msgHeadersColl').select('.attachmentImage').invoke(r.atc_label ? 'show' : 'hide');
        if (r.atc_label) {
            tmp = $('msgAtc').show().down('.label');
            tmp2 = $('partlist');
            tmp2.hide().previous().update(r.atc_label + ' ' + r.atc_download);
            if (r.atc_list) {
                $('partlist_col').show();
                $('partlist_exp').hide();
                tmp.down().hide().next().show();
                tmp2.update(r.atc_list);
            } else {
                tmp.down().show().next().hide();
            }
        } else {
            $('msgAtc').hide();
        }

        $('msgBody').down().update(r.msgtext);
        $('msgLoading', 'previewInfo').invoke('hide');
        $('previewPane').scrollTop = 0;
        pm.show();

        if (r.js) {
            eval(r.response.js.join(';'));
        }
        DimpCore.buildAddressLinks(pm);
        this.addHistory('msg:' + row.view + ':' + row.imapuid);
    },

    initPreviewPane: function()
    {
        var sel = this.viewport.getSelected();
        if (sel.size() != 1) {
            this.clearPreviewPane();
        } else {
            this._loadPreview(sel.get('dataob').first());
        }
    },

    clearPreviewPane: function()
    {
        var p = $('previewPane');
        if (p.visible()) {
            $('msgLoading', 'previewMsg').invoke('hide');
            $('previewInfo').show();
            this.pp = null;
        }
    },

    // Labeling functions
    updateUnseenUID: function(r, setflag)
    {
        var sel,
            unseen = 0,
            unseenset = r.bg.match(this.unseen_regex);
        if ((setflag && unseenset) || (!setflag && !unseenset)) {
            return false;
        }

        sel = this.viewport.createSelection('dataob', r);
        if (setflag) {
            this.viewport.updateFlag(sel, 'unseen', true);
            ++unseen;
        } else {
            this.viewport.updateFlag(sel, 'unseen', false);
            --unseen;
        }

        return this.updateUnseenStatus(r.view, unseen);
    },

    updateUnseenStatus: function(mbox, change)
    {
        if (change == 0) {
            return false;
        }
        var unseen = parseInt($(this.getFolderId(mbox)).readAttribute('u')) + change;
        this.viewport.setMetaData({ unseen: unseen });
        this.setFolderLabel(mbox, unseen);
        return true;
    },

    setMessageListTitle: function()
    {
        var offset,
            rows = this.viewport.getMetaData('total_rows');
        if (rows > 0) {
            offset = this.viewport.currentOffset();
            $('msgHeader').update(DIMP.text.messages + ' ' + (offset + 1) + ' - ' + (Math.min(offset + this.viewport.getPageSize(), rows)) + ' ' + DIMP.text.of + ' ' + rows);
        } else {
            $('msgHeader').update(DIMP.text.nomessages);
        }
    },

    setFolderLabel: function(f, unseen)
    {
        var elt, fid = this.getFolderId(f);
        elt = $(fid);
        if (!elt || !elt.hasAttribute('u')) {
            return;
        }

        unseen = parseInt(unseen);
        elt.writeAttribute('u', unseen);

        if (f == 'INBOX' && window.fluid) {
            window.fluid.setDockBadge(unseen ? unseen : '');
        }

        $(fid + '_label').update((unseen > 0) ?
            new Element('STRONG').insert(elt.readAttribute('l')).insert('&nbsp;').insert(new Element('SPAN', { className: 'count', dir: 'ltr' }).insert('(' + unseen + ')')) :
            elt.readAttribute('l'));

        if (this.folder == f) {
            this.updateTitle();
        }
    },

    getFolderId: function(f)
    {
        return 'fld' + decodeURIComponent(f).replace(/_/g,'__').replace(/[^0-9a-z\-_:\.]/gi, '_');
    },

    getSubFolderId: function(f)
    {
        return 'sub' + f;
    },

    /* Folder list updates. */
    pollFolders: function()
    {
        // Reset poll folder counter.
        this.setPollFolders();
        var args = {};
        if (this.folder && $('dimpmain_folder').visible()) {
            args = this.viewport.addRequestParams({});
        }
        $('button_checkmail').setText('[' + DIMP.text.check + ']');
        DimpCore.doAction('PollFolders', args, [], this.pollFoldersCallback.bind(this));
    },

    pollFoldersCallback: function(r)
    {
        if (r.response.poll) {
            r = r.response.poll;
            $H(r).keys().each(function(f) {
                this.setFolderLabel(f, r[f].u);
                if (this.viewport) {
                    this.viewport.setMetaData({ unseen: r[f].u }, f);
                }
            }, this);
        }
        $('button_checkmail').setText(DIMP.text.getmail);
    },

    setPollFolders: function()
    {
        if (DIMP.conf.refresh_time) {
            if (this.pollPE) {
                this.pollPE.stop();
            }
            this.pollPE = new PeriodicalExecuter(this.pollFolders.bind(this), DIMP.conf.refresh_time);
        }
    },

    portalCallback: function(r)
    {
        if (r.response.linkTags) {
            var head = $A(document.getElementsByTagName('HEAD')).first();
            r.response.linkTags.each(function(newLink) {
                var link = new Element('LINK', { type: 'text/css', rel: 'stylesheet', href: newLink.href });
                if (newLink.media) {
                    link.media = newLink.media;
                }
                head.appendChild(link);
            });
        }
        $('dimpmain_portal').update(r.response.portal);

        /* Link portal block headers to the application. */
        $('dimpmain_portal').select('h1.header a').each(function(d) {
            d.observe('click', function(e, d) {
                this.go('app:' + d.getAttribute('app'));
                e.stop();
            }.bindAsEventListener(this, d));
        }.bind(this));
    },

    /* Search filter functions. */
    searchfilterOnKeyup: function()
    {
        if (this.searchobserve) {
            clearTimeout(this.searchobserve);
        }
        this.searchobserve = this.searchfilterRun.bind(this).delay(0.5);
    },

    searchfilterRun: function()
    {
        if (!this.viewport.isFiltering()) {
            this.filtertoggle = 1;
        }
        this.viewport.runFilter($F('msgList_filter'));
    },

    searchfilterOnFocus: function()
    {
        var q = $('qoptions');

        if ($('msgList_filter').hasClassName('msgFilterDefault')) {
            this.setFilterText(false);
        }

        if (!q.visible()) {
            $('sf_current').update(this.viewport.getMetaData('label'));
            this.setSearchfilterParams('to', 'msg');
            this.setSearchfilterParams('current', 'folder');
            q.show();
            this.viewport.onResize();
        }
    },

    searchfilterOnBlur: function()
    {
        if (!$F('msgList_filter')) {
            this.setFilterText(true);
        }
    },

    // reset = (boolean) TODO
    searchfilterClear: function(reset)
    {
        if (!$('qoptions').visible()) {
            return;
        }
        if (this.searchobserve) {
            clearTimeout(this.searchobserve);
            this.searchobserve = null;
        }
        this.setFilterText(true);
        $('qoptions').hide();
        this.filtertoggle = 2;
        this.resetSelected();
        this.viewport.onResize(reset);
        this.viewport.stopFilter(reset);
    },

    // d = (boolean) Deactivate filter input?
    setFilterText: function(d)
    {
        var mf = $('msgList_filter');
        if (d) {
            mf.setValue(DIMP.text.search);
            mf.addClassName('msgFilterDefault');
        } else {
            mf.setValue('');
            mf.removeClassName('msgFilterDefault');
        }
    },

    // type = 'folder' or 'msg'
    setSearchfilterParams: function(id, type)
    {
        var c = (type == 'folder') ? this.sfiltersfolder : this.sfilters;
        c.keys().each(function(i) {
            $(i).writeAttribute('className', (id == c.get(i)) ? 'qselected' : '');
        });
    },

    updateSearchfilter: function(id, type)
    {
        this.setSearchfilterParams(id, type);
        if ($F('msgList_filter')) {
            this.viewport.runFilter();
        }
    },

    addSearchfilterParams: function()
    {
        var sf = this.sfiltersfolder.keys().detect(function(s) {
            return $(s).hasClassName('qselected');
        }),
            sm = this.sfilters.keys().detect(function(s) {
            return $(s).hasClassName('qselected');
        });
        return { searchfolder: this.sfiltersfolder.get(sf), searchmsg: this.sfilters.get(sm) };
    },

    /* Enable/Disable DIMP action buttons as needed. */
    toggleButtons: function()
    {
        var disable = (this.selectedCount() == 0);
        DimpCore.buttons.each(function(b) {
            var b_up = $(b).up();
            if (disable) {
                b_up.addClassName('disabled');
            } else {
                b_up.removeClassName('disabled');
            }
            DimpCore.DMenu.disable(b + '_img', true, disable);
        });
    },

    /* Draggable/Droppable handlers. */
    folderHoverHandler: function(elt, folder, overlap)
    {
        if (overlap > 0.33 && overlap < 0.66 && $(this.getSubFolderId(folder.id))) {
            this.toggleSubFolder(folder.id, 'exp');
        }
    },

    folderDropHandler: function(elt, folder, e)
    {
        elt = $(elt);
        folder = $(folder);
        var foldername = folder.readAttribute('mbox'),
            sel = this.viewport.getSelected(),
            uids;

        // TODO: Re-add folder dragging code when stable.
        if (sel.size()) {
            // Dragging multiple selected messages.
            uids = sel;
        } else if (elt.readAttribute('mbox') != foldername) {
            // Dragging a single unselected message.
            uids = this.viewport.createSelection('domid', elt.id);
        }

        // Don't allow drag/drop to the current folder.
        if (uids.size() && this.folder != foldername) {
            this.viewport.updateFlag(uids, 'deletedmsg', true);
            DimpCore.doAction('MoveMessage', this.viewport.addRequestParams({ tofld: foldername }), DimpCore.toUIDArray(uids), this.deleteCallback.bind(this));
        }
    },

    /* Keydown event handler */
    keydownHandler: function(e)
    {
        // Only catch keyboard shortcuts in message list view. Also, disable
        // catching when we are in a form element.
        if (!$('dimpmain_folder').visible() || e.findElement('FORM')) {
            return;
        }

        var co,
            kc = e.keyCode || e.charCode,
            index,
            ps,
            r,
            row,
            rowoff,
            sel = this.viewport.getSelected();

        switch (kc) {
        case Event.KEY_DELETE:
        case Event.KEY_BACKSPACE:
            index;
            if (sel.size() == 1) {
                r = sel.get('dataob').first();
                if (e.shiftKey) {
                    this.moveSelected(r.rownum + ((r.rownum == this.viewport.getMetaData('total_rows')) ? -1 : 1), true);
                }
                this.flag('deleted', r);
            } else {
                this.flag('deleted');
            }
            e.stop();
            break;

        case Event.KEY_UP:
        case Event.KEY_DOWN:
            if (e.shiftKey && this.lastrow != -1) {
                row = this.viewport.createSelection('rownum', this.lastrow + ((kc == Event.KEY_UP) ? -1 : 1));
                if (row.size()) {
                    row = row.get('dataob').first();
                    this.viewport.scrollTo(row.rownum);
                    this.msgSelect(row.domid, { shift: true });
                }
            } else {
                this.moveSelected(kc == Event.KEY_UP ? -1 : 1);
            }
            e.stop();
            break;

        case Event.KEY_PAGEUP:
        case Event.KEY_PAGEDOWN:
            ps = this.viewport.getPageSize() - 1;
            move = ps * (kc == Event.KEY_PAGEUP ? -1 : 1);
            if (sel.size() == 1) {
                co = this.viewport.currentOffset();
                rowoff = sel.get('rownum').first() - 1;
                switch (kc) {
                case Event.KEY_PAGEUP:
                    if (co != rowoff) {
                        move = co - rowoff;
                    }
                    break;

                case Event.KEY_PAGEDOWN:
                    if ((co + ps) != rowoff) {
                        move = co + ps - rowoff;
                    }
                    break;
                }
            }
            this.moveSelected(move);
            e.stop();
            break;

        case Event.KEY_HOME:
        case Event.KEY_END:
            this.moveSelected(kc == Event.KEY_HOME ? 1 : this.viewport.getMetaData('total_rows'), true);
            e.stop();
            break;

        case Event.KEY_RETURN:
            if (!e.element().match('input')) {
                // Popup message window if single message is selected.
                if (sel.size() == 1) {
                    this.msgWindow(sel.get('dataob').first());
                }
                e.stop();
            }
            // Don't stop even by default since we may be in an INPUT
            // field, and we want the return to submit the form.
            break;

        case 65: // A
        case 97: // a
            if (e.ctrlKey) {
                this.selectAll();
                e.stop();
            }
            break;
        }
    },

    /* Handle rename folder actions. */
    renameFolder: function(folder)
    {
        if (Object.isUndefined(folder)) {
            return;
        }

        folder = $(folder);
        var n = this.createFolderForm(function(e) { this.folderAction(folder, e, 'rename'); return false; }.bindAsEventListener(this), DIMP.text.rename_prompt);
        n.down('input').setValue(folder.readAttribute('l'));
    },

    /* Handle insert folder actions. */
    createBaseFolder: function()
    {
        this.createFolderForm(function(e) { this.folderAction('', e, 'create'); return false; }.bindAsEventListener(this), DIMP.text.create_prompt);
    },

    createSubFolder: function(folder)
    {
        if (Object.isUndefined(folder)) {
            return false;
        }

        this.createFolderForm(function(e) { this.folderAction($(folder), e, 'createsub'); return false; }.bindAsEventListener(this), DIMP.text.createsub_prompt);
    },

    createFolderForm: function(action, text)
    {
        var n = new Element('FORM', { action: '#', id: 'RB_folder' }).insert(
                    new Element('P').insert(text)
                ).insert(
                    new Element('INPUT', { type: 'text', size: 15 })
                ).insert(
                    new Element('INPUT', { type: 'button', className: 'button', value: DIMP.text.ok }).observe('click', action)
                ).insert(
                    new Element('INPUT', { type: 'button', className: 'button', value: DIMP.text.cancel }).observe('click', this.closeRedBox.bind(this))
                ).observe('keydown', function(e) { if ((e.keyCode || e.charCode) == Event.KEY_RETURN) { e.stop(); action(e); } });

        RedBox.overlay = true;
        RedBox.onDisplay = Form.focusFirstElement.curry(n);
        RedBox.showHtml(n);
        return n;
    },

    closeRedBox: function()
    {
        var c = RedBox.getWindowContents();
        DimpCore.addGC([ c, c.descendants() ].flatten());
        RedBox.close();
    },

    folderAction: function(folder, e, mode)
    {
        this.closeRedBox();

        var action, form = e.findElement('form'), params, val;
        val = $F(form.down('input'));

        if (val) {
            switch (mode) {
            case 'rename':
                if (folder.readAttribute('l') != val) {
                    action = 'RenameFolder';
                    params = { old_name: folder.readAttribute('mbox'),
                               new_name: val };
                }
                break;

            case 'create':
            case 'createsub':
                action = 'CreateFolder';
                params = { folder: val };
                if (mode == 'createsub') {
                    params.parent = folder.readAttribute('mbox');
                }
                break;
            }
            if (action) {
                DimpCore.doAction(action, params, [], this.folderCallback.bind(this));
            }
        }
    },

    /* Folder action callback functions. */
    folderCallback: function(r)
    {
        r.response.d.each(function(i) {
            this.deleteFolder(i);
        }, this);

        r.response.c.each(function(i) {
            this.changeFolder(i);
        }, this);

        r.response.a.each(function(i) {
            this.createFolder(i);
        }, this);
    },

    deleteCallback: function(r)
    {
        this.msgListLoading(false);
        this.pollFoldersCallback(r);

        if (!r.response.uids ||
            !r.response.folder == this.folder) {
            return;
        }

        var search = this.viewport.getViewportSelection().search({ view: r.response.folder, imapuid: r.response.uids[r.response.folder] });
        if (search.size()) {
            if (r.response.remove) {
                this.viewport.remove(search, { cacheid: r.response.cacheid, noupdate: r.response.viewport });
            } else {
                // Need this to catch spam deletions.
                this.viewport.updateFlag(search, 'deletedmsg', true);
            }
        }
    },

    emptyFolderCallback: function(r)
    {
        if (r.response.mbox) {
            if (this.folder == r.response.mbox) {
                this.viewport.reload();
                this.clearPreviewPane();
            }
            this.setFolderLabel(r.response.mbox, 0);
        }
    },

    flagAllCallback: function(r)
    {
        if (r.response.mbox) {
            this.setFolderLabel(r.response.mbox, r.response.u);
        }
    },

    folderLoadCallback: function(r)
    {
        this.folderCallback(r);

        var clickevent = this.handleFolderMouseEvent.bindAsEventListener(this, 'click'),
            elts = $('specialfolders', 'normalfolders').compact(),
            fs = $('foldersSidebar'),
            fsheight = fs.getStyle('max-height'),
            outevent = this.handleFolderMouseEvent.bindAsEventListener(this, 'out'),
            overevent = this.handleFolderMouseEvent.bindAsEventListener(this, 'over');

        elts.invoke('observe', 'click', clickevent);
        elts.invoke('observe', 'mouseover', overevent);
        elts.invoke('observe', 'mouseout', outevent);

        $('foldersLoading').hide();
        fs.show();

        // Fix for IE6 - which doesn't support max-height.  We need to search
        // for height: 0px instead (comment in IE 6 CSS explains this is
        // needed for auto sizing).
        if (Prototype.Browser.IE && Object.isUndefined(fsheight)) {
            fsheight = fs.getStyle('height') == '0px' ? true : null;
        }

        if (fsheight !== null) {
            this.sizeFolderlist();
            Event.observe(window, 'resize', this.sizeFolderlist.bind(this));
        }
    },

    handleFolderMouseEvent: function(e, action)
    {
        var elt = e.element(), li, type;
        if (elt.nodeType == 3) {
            elt = elt.parentNode;
        }

        if (elt.hasClassName('folder') || elt.hasClassName('custom')) {
            li = elt;
        } else {
            li = elt.up('.folder');
            if (!li) {
                li = elt.up('.custom');
                if (!li) {
                    return;
                }
            }
        }

        type = li.readAttribute('ftype');

        switch (action) {
        case 'over':
            li.addClassName('over');
            if (type) {
                DimpCore.addMouseEvents({ id: li.id, type: type });
            }
            break;

        case 'out':
            li.removeClassName('over');
            break;

        case 'click':
            if (elt.hasClassName('exp') || elt.hasClassName('col')) {
                this.toggleSubFolder(li.id, 'tog');
            } else {
                switch (type) {
                case 'container':
                    e.stop();
                    break;

                case 'folder':
                case 'special':
                    return this.go('folder:' + li.readAttribute('mbox'));
                    break;
                }
            }
            break;
        }
    },

    toggleSubFolder: function(base, mode)
    {
        base = $(base);
        var s = $(this.getSubFolderId(base.id));
        if (mode == 'tog' ||
            (mode == 'exp' && !s.visible()) ||
            (mode == 'col' && s.visible())) {
            base.firstDescendant().className = s.toggle().visible() ? 'col' : 'exp';
        }
    },

    // Folder actions.
    // For format of the ob object, see DIMP::_createFolderElt().
    createFolder: function(ob)
    {
        var a,
            div,
            fid = this.getFolderId(ob.m),
            insert_node = 0,
            li,
            ll,
            mbox = decodeURIComponent(ob.m),
            parent,
            submbox;

        if (ob.s) {
            parent = $('specialfolders');
        } else {
            parent = $(this.getSubFolderId(this.getFolderId(ob.pa)));
            if (parent) {
                if (parent.firstDescendant().match('ul')) {
                    parent = parent.firstDescendant();
                }
            } else {
                parent = $('normalfolders');
            }
       }

        li = new Element('LI', { className: 'folder', id: fid, l: ob.l, mbox: mbox, ftype: ((ob.co) ? 'container' : ((ob.s) ? 'special' : 'folder')) });

        div = new Element('DIV', { className: ob.cl, id: fid + '_div' });
        if (ob.i) {
            div.update(ob.i);
        }
        if (ob.ch) {
            div.className = 'exp';
        }
        li.appendChild(div);

        a = new Element('A', { id: fid + '_label', nicetitle: ob.m }).insert(ob.l);
        li.appendChild(a);

        // Now walk through the parent <ul> to find the right place to
        // insert the new folder.
        submbox = $(this.getSubFolderId(fid));
        if (submbox) {
            parent.insertBefore(li, submbox);
            // If an expanded parent mailbox was deleted, we need to toggle the
            // icon accordingly.
            if (submbox.visible()) {
                div.removeClassName('exp');
                div.addClassName('col');
            }
        } else {
            ll = mbox.toLowerCase();
            parent.childElements().detect(function(node) {
                var nodembox = node.readAttribute('mbox');
                if (nodembox && (!ob.s || nodembox != 'INBOX')) {
                    if (ll < nodembox.toLowerCase()) {
                        insert_node = 1;
                        parent.insertBefore(li, node);
                        return true;
                    }
                }
            });

            if (!insert_node) {
                parent.appendChild(li);
            }
        }

        // Make the new folder a drop target.
        Droppables.add(fid, { hoverclass: 'drop', onHover: this.folderHoverHandler.bind(this), onDrop: this.folderDropHandler.bind(this), scroll: true, overlap: 'vertical' });

        // Need this because add() will make the children 'relative',
        // which breaks the folder list on IE.
        if (Prototype.Browser.IE) {
            $(fid).undoPositioned();
        }

        // Check for unseen messages
        if (ob.po) {
            li.writeAttribute('u', '');
            this.setFolderLabel(mbox, ob.u);
        }

        // Make sure the sub<mbox> ul is created if necessary.
        if (ob.ch && !submbox) {
            parent.insertBefore(new Element('LI', { className: 'subfolders', id: this.getSubFolderId(fid) }).insert(new Element('UL')).hide(), li.nextSibling);
        }
    },

    deleteFolder: function(folder)
    {
        var f = decodeURIComponent(folder), fid;
        if (this.folder == f) {
            this.go('folder:INBOX');
        }

        fid = this.getFolderId(folder);
        this.deleteFolderElt(fid, true);

        Effect.Fade(fid, { afterFinish: function(effect) {
            try {
                DimpCore.addGC(effect.element.remove());
            } catch (e) {
                DimpCore.debug('deleteFolder', e);
            }
        } });
    },

    changeFolder: function(ob)
    {
        var fid = this.getFolderId(ob.m), fdiv = $(fid + '_div'), oldexpand;
        this.deleteFolderElt(fid, !ob.ch);
        oldexpand = fdiv.hasClassName('col');
        if (ob.co && this.folder == ob.m) {
            this.go('folder:INBOX');
        }
        $(fid).remove();
        this.createFolder(ob);
        if (ob.ch && oldexpand) {
            fdiv.removeClassName('exp');
            fdiv.addClassName('col');
        }
    },

    deleteFolderElt: function(fid, sub)
    {
        DimpCore.addGC($(fid, fid + '_div', fid + '_label'));
        if (sub) {
            var submbox = $(this.getSubFolderId(fid));
            if (submbox) {
                submbox.remove();
            }
        }
        Droppables.remove(fid);
        DimpCore.removeMouseEvents(fid);
        if (this.viewport) {
            this.viewport.deleteView(fid);
        }
    },

    sizeFolderlist: function()
    {
        var fl = $('foldersSidebar');
        fl.setStyle({ height: (document.viewport.getHeight() - fl.cumulativeOffset()[1] - 10) + 'px' });
    },

    /* Flag actions for message list. */
    flag: function(action, index, folder)
    {
        var actionCall,
            args,
            obs = [],
            unseenstatus = 1,
            vs;

        if (index) {
            if (Object.isUndefined(folder)) {
                vs = this.viewport.createSelection('dataob', index);
            } else {
                vs = this.viewport.getViewportSelection().search({ imapuid: index, view: folder });
                if (!vs.size() && folder != this.folder) {
                    vs = this.viewport.getViewportSelection(folder).search({ imapuid: index });
                }
            }
        } else {
            vs = this.viewport.getSelected();
        }

        switch (action) {
        case 'allUnseen':
        case 'allSeen':
            DimpCore.doAction((action == 'allUnseen') ? 'MarkFolderUnseen' : 'MarkFolderSeen', { folder: folder }, [], this.flagAllCallback.bind(this));
            if (folder == this.folder) {
                this.viewport.updateFlag(this.createSelection('rownum', $A($R(1, this.viewport.getBufferSize()))), 'unseen', action == 'allUnseen');
            }
            break;

        case 'deleted':
        case 'undeleted':
        case 'spam':
        case 'ham':
        case 'blacklist':
        case 'whitelist':
            if (!vs.size()) {
                break;
            }
            args = this.viewport.addRequestParams({});
            if (action == 'deleted' || action == 'undeleted') {
                this.viewport.updateFlag(vs, 'deletedmsg', action == 'deleted');            }

            if (action == 'undeleted') {
                DimpCore.doAction('UndeleteMessage', args, DimpCore.toUIDArray(vs));
            } else {
                actionCall = { deleted: 'DeleteMessage', spam: 'ReportSpam', ham: 'ReportHam', blacklist: 'Blacklist', whitelist: 'Whitelist' };
                // This needs to be synchronous Ajax if we are calling from a
                // popup window because Mozilla will not correctly call the
                // callback function if the calling window has been closed.
                DimpCore.doAction(actionCall[action], args, DimpCore.toUIDArray(vs), this.deleteCallback.bind(this), { asynchronous: !(index && folder) });

                // If reporting spam, to indicate to the user that something is
                // happening (since spam reporting may not be instantaneous).
                if (action == 'spam' || action == 'ham') {
                    this.msgListLoading(true);
                }
            }
            break;

        case 'unseen':
        case 'seen':
            if (!vs.size()) {
                break;
            }
            args = { folder: this.folder, messageFlag: '-seen' };
            if (action == 'seen') {
                unseenstatus = 0;
                args.messageFlag = 'seen';
            }
            vs.get('dataob').each(function(s) {
                if (this.updateUnseenUID(s, unseenstatus)) {
                    obs.push(s);
                }
            }, this);

            if (obs.size()) {
                DimpCore.doAction('MarkMessage', args, DimpCore.toUIDArray(this.viewport.createSelection('dataob', obs)));
            }
            break;

        case 'flagged':
        case 'clear':
            if (!vs.size()) {
                break;
            }
            args = {
                folder: this.folder,
                messageFlag: ((action == 'flagged') ? 'flagged' : '-flagged')
            };
            this.viewport.updateFlag(vs, 'flagged', action == 'flagged');
            DimpCore.doAction('MarkMessage', args, DimpCore.toUIDArray(vs));
            break;

        case 'answered':
            this.viewport.updateFlag(vs, 'answered', true);
            this.viewport.updateFlag(vs, 'flagged', false);
            break;
        }
    },

    /* Miscellaneous folder actions. */
    purgeDeleted: function()
    {
        DimpCore.doAction('PurgeDeleted', this.viewport.addRequestParams({}), [], this.deleteCallback.bind(this));
    },

    modifyPollFolder: function(folder, add)
    {
        DimpCore.doAction('ModifyPollFolder', { folder: folder, add: (add) ? 1 : 0 }, [], this.modifyPollFolderCallback.bind(this));
    },

    modifyPollFolderCallback: function(r)
    {
        r = r.response;
        var f = r.folder, fid, p = { response: { poll: {} } };
        fid = $(this.getFolderId(f));

        if (r.add) {
            p.response.poll[f] = { u: r.poll.u };
            fid.writeAttribute('u', 0);
        } else {
            p.response.poll[f] = { u: 0 };
        }

        this.pollFoldersCallback(p);

        if (!r.add) {
            fid.removeAttribute('u');
        }
    },

    msgListLoading: function(show)
    {
        var ml_offset;

        if (this.fl_visible != show) {
            this.fl_visible = show;
            if (show) {
                ml_offset = $('msgList').positionedOffset();
                $('folderLoading').setStyle({ position: 'absolute', top: (ml_offset.top + 10) + 'px', left: (ml_offset.left + 10) + 'px' });
                Effect.Appear('folderLoading', { duration: 0.2 });
            } else {
                Effect.Fade('folderLoading', { duration: 0.2 });
            }
        }
    },

    /* Onload function. */
    onLoad: function() {
        var C = DimpCore.clickObserveHandler, tmp;

        if (Horde.dhtmlHistory.initialize()) {
            Horde.dhtmlHistory.addListener(this.go.bind(this));
        }

        this.setFilterText(true);

        /* Initialize the starting page if necessary. addListener() will have
         * already fired if there is a current location so only do a go()
         * call if there is no current location. */
        if (!Horde.dhtmlHistory.getCurrentLocation()) {
            if (DIMP.conf.login_view == 'inbox') {
                this.go('folder:INBOX');
            } else {
                this.go('portal');
                if (DIMP.conf.background_inbox) {
                    this.loadFolder('INBOX', true);
                }
            }
        }

        /* Add popdown menus. */
        DimpCore.addPopdown('button_reply', 'reply', 'cumulative');
        DimpCore.DMenu.disable('button_reply_img', true, true);
        DimpCore.addPopdown('button_forward', 'forward', 'cumulative');
        DimpCore.DMenu.disable('button_forward_img', true, true);
        DimpCore.addPopdown('button_other', 'otheractions', 'cumulative');

        /* Set up click event observers for elements on main page. */
        tmp = $('logo');
        if (tmp.visible()) {
            C({ d: tmp.down('a'), f: this.go.bind(this, 'portal') });
        }

        tmp = $('dimpbarActions');
        C({ d: tmp.down('.composelink'), f: DimpCore.compose.bind(DimpCore, 'new') });
        C({ d: tmp.down('.refreshlink'), f: this.pollFolders.bind(this) });

        tmp = $('serviceActions');
        tmp.select('.servicelink').each(this._sidebarMouseover.bind(this));
        [ 'portal', 'options' ].each(function(a) {
            var d = $('app' + a);
            if (d) {
                C({ d: d, f: this.go.bind(this, a) });
            }
        }.bind(this));
        tmp = $('applogout');
        if (tmp) {
            C({ d: tmp, f: DimpCore.logout.bind(DimpCore) });
        }

        tmp = $('applicationfolders');
        if (tmp) {
            tmp.select('ul li.custom').each(function(s) {
                var d = s.down('a');
                this._sidebarMouseover(s);
                C({ d: d, f: this.go.bind(this, 'app:' + d.getAttribute('app')) });
            }.bind(this));
        }

        tmp = $('sitemenu');
        if (tmp) {
            tmp.select('ul li.custom').each(this._sidebarMouseover.bind(this));
        }
        C({ d: $('normalfolders').down('li.folder a'), f: this.createBaseFolder.bind(this) });

        tmp = $('hometab');
        if (tmp) {
            C({ d: tmp, f: this.go.bind(this, 'portal') });
        }
        $('tabbar').select('a.applicationtab').each(function(a) {
            C({ d: a, f: function(a) { this.go('app:' + a.getAttribute('app')); }.bind(this, a) });
        }.bind(this));
        C({ d: $('button_reply'), f: this.composeMailbox.bind(this, 'reply') });
        C({ d: $('button_forward'), f: this.composeMailbox.bind(this, DIMP.conf.forward_default) });
        [ 'spam', 'ham', 'deleted' ].each(function(a) {
            var d = $('button_' + a);
            if (d) {
                C({ d: d, f: this.flag.bind(this, a) });
            }
        }.bind(this));
        C({ d: $('button_compose'), f: DimpCore.compose.bind(DimpCore, 'new') });
        C({ d: $('button_checkmail2'), f: this.pollFolders.bind(this) });
        C({ d: $('button_other'), f: function(e) { DimpCore.DMenu.trigger(e.findElement('A').next(), true); }, p: true });
        C({ d: $('qoptions').down('.qclose a'), f: this.searchfilterClear.bind(this, false) });
        [ 'all', 'current' ].each(function(a) {
            var d = $('sf_' + a);
            if (d) {
                C({ d: d, f: this.updateSearchfilter.bind(this, a, 'folder') });
            }
        }.bind(this));
        [ 'msgall', 'from', 'to', 'subject' ].each(function(a) {
            C({ d: $('sf_' + a), f: this.updateSearchfilter.bind(this, a, 'msg') });
        }.bind(this));
        C({ d: $('msgFromLink'), f: this.sort.bind(this, DIMP.conf.sortfrom), p: true });
        [ 'sortsubject', 'sortdate' ].each(function(a) {
           C({ d: $(a), f: this.sort.bind(this, $(a).readAttribute('sortby')), p: true });
        }.bind(this));
        C({ d: $('msglistHeader').down('.msgSize a'), f: this.sort.bind(this, DIMP.conf.sortsize), p: true });
        C({ d: $('ctx_folder_create'), f: function() { this.createSubFolder(DimpCore.DMenu.element()); }.bind(this) });
        C({ d: $('ctx_folder_rename'), f: function() { this.renameFolder(DimpCore.DMenu.element()); }.bind(this) });
        C({ d: $('ctx_folder_empty'), f: function() { if (window.confirm(DIMP.text.empty_folder)) { DimpCore.doAction('EmptyFolder', { folder: DimpCore.DMenu.element().readAttribute('mbox') }, [], this.emptyFolderCallback.bind(this)); } }.bind(this) });
        C({ d: $('ctx_folder_delete'), f: function() { if (window.confirm(DIMP.text.delete_folder)) { DimpCore.doAction('DeleteFolder', { folder: DimpCore.DMenu.element().readAttribute('mbox') }, [], this.folderCallback.bind(this)); } }.bind(this) });
        [ 'ctx_folder_seen', 'ctx_folder_unseen' ].each(function(a) {
            C({ d: $(a), f: function(type) { this.flag(type, null, DimpCore.DMenu.element().readAttribute('mbox')); }.bind(this, a == 'ctx_folder_seen' ? 'allSeen' : 'allUnseen') });
        }.bind(this));
        [ 'ctx_folder_poll', 'ctx_folder_nopoll' ].each(function(a) {
            C({ d: $(a), f: function(modify) { this.modifyPollFolder(DimpCore.DMenu.element().readAttribute('mbox'), modify); }.bind(this, a == 'ctx_folder_poll') });
        }.bind(this));
        C({ d: $('ctx_container_create'), f: function() { this.createSubFolder(DimpCore.DMenu.element()); }.bind(this) });
        C({ d: $('ctx_container_rename'), f: function() { this.renameFolder(DimpCore.DMenu.element()); }.bind(this) });
        [ 'reply', 'reply_all', 'reply_list', 'forward_all', 'forward_body', 'forward_attachments' ].each(function(a) {
            C({ d: $('ctx_message_' + a), f: this.composeMailbox.bind(this, a) });
        }.bind(this));
        [ 'seen', 'unseen', 'flagged', 'clear', 'spam', 'ham', 'blacklist', 'whitelist', 'deleted', 'undeleted' ].each(function(a) {
            var d = $('ctx_message_' + a);
            if (d) {
                C({ d: d, f: this.flag.bind(this, a) });
            }
        }.bind(this));
        C({ d: $('ctx_draft_resume'), f: this.composeMailbox.bind(this, 'resume') });
        [ 'flagged', 'clear', 'deleted', 'undeleted' ].each(function(a) {
            var d = $('ctx_draft_' + a);
            if (d) {
                C({ d: d, f: this.flag.bind(this, a) });
            }
        }.bind(this));
        [ 'reply', 'reply_all', 'reply_list' ].each(function(a) {
            C({ d: $('ctx_reply_' + a), f: this.composeMailbox.bind(this, a) });
        }.bind(this));
        [ 'forward_all', 'forward_body', 'forward_attachments' ].each(function(a) {
            C({ d: $('ctx_forward_' + a), f: this.composeMailbox.bind(this, a) });
        }.bind(this));
        C({ d: $('previewtoggle'), f: this.togglePreviewPane.bind(this) });
        [ 'seen', 'unseen', 'flagged', 'clear', 'blacklist', 'whitelist' ].each(function(a) {
            var d = $('oa_' + a);
            if (d) {
                C({ d: d, f: this.flag.bind(this, a) });
            }
        }.bind(this));
        C({ d: $('oa_selectall'), f: this.selectAll.bind(this) });

        tmp = $('oa_purge_deleted');
        if (tmp) {
            C({ d: tmp, f: this.purgeDeleted.bind(this) });
        }

        $('expandHeaders', 'collapseHeaders').each(function(a) {
            C({ d: a, f: function() { $('msgHeadersColl', 'msgHeaders').invoke('toggle'); } });
        }.bind(this));
        $('msg_newwin', 'msg_newwin_options').compact().each(function(a) {
            C({ d: a, f: function() { this.msgWindow(this.viewport.getViewportSelection().search({ imapuid: DIMP.conf.msg_index, view: DIMP.conf.msg_folder }).get('dataob').first()); }.bind(this) });
        }.bind(this));
        DimpCore.messageOnLoad();
    },

    _sidebarMouseover: function(e)
    {
        e.observe('mouseover', e.addClassName.bind(e, 'over'));
        e.observe('mouseout', e.removeClassName.bind(e, 'over'));
    }

};

/* Stuff to do immediately when page is ready. */
document.observe('dom:loaded', function() {
    $('dimpLoading').hide();
    $('dimpPage').show();

    /* Create the folder list. Any pending notifications will be caught via
     * the return from this call. */
    DimpCore.doAction('ListFolders', {}, [], DimpBase.folderLoadCallback.bind(DimpBase));

    /* Start message list loading as soon as possible. */
    if (!DimpBase.delay_onload) {
        DimpBase.onLoad();
    }

    /* Catch shift-selects in IE. */
    if (Prototype.Browser.IE) {
        document.observe('selectstart', function(e) { if (DIMP.conf.is_ie6 || !e.element().descendantOf('previewPane')) { e.stop(); } });
    }

    /* Remove unneeded search folders. */
    if (!DIMP.conf.search_all) {
        DimpBase.sfiltersfolder.unset('sf_all');
    }

    /* Check for new mail. */
    DimpBase.setPollFolders();

    /* Load quota bar. */
    if (DIMP.conf.show_quota) {
        DimpBase.checkQuota();
        if (DIMP.conf.refresh_time) {
            new PeriodicalExecuter(DimpBase.checkQuota.bind(DimpBase), DIMP.conf.refresh_time);
        }
    }

    /* Bind key shortcuts. */
    document.observe('keydown', DimpBase.keydownHandler.bind(DimpBase));

    /* In preview pane, clear any context menus on mousewheel scroll - it is
     * the only case we can not catch elsewhere. */
    $('previewPane').observe(Prototype.Browser.IE ? 'mousewheel' : 'DOMMouseScroll', DimpCore.DMenu.close);
});

/* Stuff to do after window is completely loaded. Don't init viewport until
 * now for non-Gecko/Opera browsers since sizing functions might not work
 * properly before. */
if (DimpBase.delay_onload) {
    Event.observe(window, 'load', DimpBase.onLoad.bind(DimpBase));
}

/* Need to register a callback function for doAction to catch viewport
 * information returned from the server. */
DimpCore.onDoActionComplete = function(r) {
    if (DimpBase.viewport && r.response.viewport) {
        DimpBase.viewport.ajaxResponse(r.response.viewport);
    }
};

/* Extend these functions from DimpCore since additional processing needs to
 * be done re: drag/drop and menu manipulation. */
DimpCore.addMouseEvents = DimpCore.addMouseEvents.wrap(DimpBase.addMouseEvents.bind(DimpBase));
DimpCore.removeMouseEvents = DimpCore.removeMouseEvents.wrap(DimpBase.removeMouseEvents.bind(DimpBase));
