/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/lib_annotations.hpp
 *
 * $Id: lib_front_end.hpp,v 1.18 2005/06/23 09:54:22 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @brief [LEVEL: beta] Conditional Macros for annotating a
 * library. This file is meant as a template for a library specific
 * interface. Copy the this file and replace XXX with the library
 * name.
 *
 * @test diagnostics/macros/lib_front_end.t.cpp
 */
#ifndef DIAGNOSTICS__LIB_ANNOTATIONS__INCLUDE_GUARD
#define DIAGNOSTICS__LIB_ANNOTATIONS__INCLUDE_GUARD

#ifndef XXX_DEBUG__LEVEL__
#  define XXX_DEBUG__LEVEL__ -1
#endif

/**
 * @brief DEBUG__LEVEL__ tells the test subsystem at which build level
 * we are.
 *
 * This is necessary since the test subsystem gives different verdicts
 * depending on DEBUG__LEVEL__. For instance, if DEBUG__LEVEL__ is 1
 * (LEVEL_DEBUG), then a failed XXX_DEBUG_CHECK is not a failure. But
 * if DEBUG__LEVEL__ is 0 (LEVEL_PROD), then a failed XXX_DEBUG_CHECK
 * causes a failure (since it would be never checked at production
 * level, i.e., should not fail at all).
 *
 * If you use diagnostics for several different libraries, then you
 * have to adjust DEBUG__LEVEL__ manually within each unit test: The
 * DEBUG__LEVEL__ of a unit test must be identical the build level of
 * the unit test. E.g., if you are in a unit test of library YYY, then you will use 
 *
 *   define DEBUG__LEVEL__ YYY_DEBUG__LEVEL__
 *
 */ 
#ifndef DEBUG__LEVEL__
#  define DEBUG__LEVEL__ XXX_DEBUG__LEVEL__
#endif

#if XXX_DEBUG__LEVEL__ >= 0
#  include <diagnostics/macros/check_annotation.hpp>
#  include <diagnostics/macros/assert_annotation.hpp>
#  include <diagnostics/macros/block_annotation.hpp>
#  include <diagnostics/macros/invariance_annotation.hpp>
#  include <diagnostics/macros/exceptionless_block_annotation.hpp>
#  include <diagnostics/macros/relation_modifier.hpp>
#else
#  define DIAGNOSTICS_RELATION(ARG1, ARG2, ARG3, ARG4, ARG5) ((void)0)
#endif

#ifndef XXX_INLINE
#  if XXX_DEBUG__LEVEL__ == 0
#    define XXX_INLINE inline 
#  else
#    define XXX_INLINE 
#  endif
#endif


#ifndef XXX_BASE_EXCEPTION_TYPE
#  define XXX_BASE_EXCEPTION_TYPE ::std::exception
#  include <exception>
#  define XXX_BASE_EXCEPTION_STR_WHAT(EX) (::std::string("EXCEPTION=\"") + "\" WHAT=\"" + (EX).what() + "\"")
#endif


#ifndef XXX_HANDLE_TEST_EXCEPTION_EXPLICITLY
#  define XXX_HANDLE_TEST_EXCEPTION_EXPLICITLY 1
#endif


#ifndef XXX_FAILED_ASSERT_ACTION
#  define XXX_FAILED_ASSERT_ACTION(LEVEL,NAME,COND) throw NAME(#COND)
#endif

#ifndef XXX_FAILED_ASSERT_ACTION1
#  define XXX_FAILED_ASSERT_ACTION1(LEVEL,NAME,COND,EXCEP_WHAT) throw NAME(EXCEP_WHAT)
#endif

#ifndef XXX_FAILED_CHECK_ACTION
#  define XXX_FAILED_CHECK_ACTION(LEVEL,NAME,COND) throw NAME(#COND)
#endif

#ifndef XXX_FAILED_CHECK_ACTION1
#  define XXX_FAILED_CHECK_ACTION1(LEVEL,NAME,COND,EXCEP_WHAT) throw NAME(EXCEP_WHAT)
#endif


#ifndef XXX_NR_WHAT_DEFAULT
#  define XXX_NR_WHAT_DEFAULT 0
#endif 

////////////////////////////////////////////////////////////////////////////////

// derived config

#if XXX_DEBUG__LEVEL__ >= 0
#if XXX_HANDLE_TEST_EXCEPTION_EXPLICITLY == 1
#  include <diagnostics/unittest/test_exception.hpp>
#endif
#endif

#ifndef XXX_BASE_EXCEPTION_STR_WHAT
#  warning XXX_BASE_EXCEPTION_STR_WHAT not defined
#endif

////////////////////////////////////////////////////////////////////////////////

// AUDIT

#if XXX_DEBUG__LEVEL__ > 1

#  define XXX_AUDIT_TRACE(STR_WHAT) \
    DIAGNOSTICS_BASE_LOG(::diagnostics::LEVEL_AUDIT,::diagnostics::TYPE_TRACE,XXX_NR_WHAT_DEFAULT,(STR_WHAT))
#  define XXX_AUDIT_BINARY_TRACE(STR_WHAT) \
    DIAGNOSTICS_BASE_LOG(::diagnostics::LEVEL_AUDIT,::diagnostics::TYPE_TRACE_BINARY,XXX_NR_WHAT_DEFAULT,(STR_WHAT))

#  define XXX_AUDIT_TRACE1(NR_WHAT,STR_WHAT) \
    DIAGNOSTICS_BASE_LOG(::diagnostics::LEVEL_AUDIT,::diagnostics::TYPE_TRACE,(NR_WHAT),(STR_WHAT))
#  define XXX_AUDIT_BINARY_TRACE1(NR_WHAT,STR_WHAT) \
    DIAGNOSTICS_BASE_LOG(::diagnostics::LEVEL_AUDIT,::diagnostics::TYPE_TRACE_BINARY,(NR_WHAT),(STR_WHAT))


#  define XXX_AUDIT_BLOCK_ENTER(STR_WHAT)  \
    DIAGNOSTICS_BASE_BLOCK_ENTER(::diagnostics::LEVEL_AUDIT, \
                                 ::diagnostics::TYPE_BLOCK_ENTER, \
                                 ::diagnostics::TYPE_BLOCK_EXIT,XXX_NR_WHAT_DEFAULT,(STR_WHAT))
#  define XXX_AUDIT_BLOCK_EXIT DIAGNOSTICS_BASE_BLOCK_EXIT
#  define XXX_AUDIT_BLOCK_GUARD(STR_WHAT)  \
    DIAGNOSTICS_BASE_BLOCK_GUARD(::diagnostics::LEVEL_AUDIT, \
                                 ::diagnostics::TYPE_BLOCK_ENTER, \
                                 ::diagnostics::TYPE_BLOCK_EXIT,XXX_NR_WHAT_DEFAULT,(STR_WHAT))

#  define XXX_AUDIT_PROCEDURE_ENTER(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_ENTER(::diagnostics::LEVEL_AUDIT, \
                                 ::diagnostics::TYPE_PROCEDURE_ENTER, \
                                 ::diagnostics::TYPE_PROCEDURE_EXIT,XXX_NR_WHAT_DEFAULT, \
                                 ::std::string("PROCEDURE=\"") + DIAGNOSTICS_FUNC_NAME + "\" " + (STR_WHAT))
#  define XXX_AUDIT_PROCEDURE_EXIT DIAGNOSTICS_BASE_BLOCK_EXIT
#  define XXX_AUDIT_PROCEDURE_GUARD(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_GUARD(::diagnostics::LEVEL_AUDIT, \
                                 ::diagnostics::TYPE_PROCEDURE_ENTER, \
                                 ::diagnostics::TYPE_PROCEDURE_EXIT,XXX_NR_WHAT_DEFAULT, \
                                 ::std::string("PROCEDURE=\"") + DIAGNOSTICS_FUNC_NAME + "\" " + (STR_WHAT))

#  define XXX_AUDIT_METHOD_ENTER(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_ENTER(::diagnostics::LEVEL_AUDIT, \
                                 ::diagnostics::TYPE_METHOD_ENTER, \
                                 ::diagnostics::TYPE_METHOD_EXIT,XXX_NR_WHAT_DEFAULT, \
                                 ::std::string("METHOD=\"") + DIAGNOSTICS_FUNC_NAME + "\" " + (STR_WHAT))
#  define XXX_AUDIT_METHOD_EXIT DIAGNOSTICS_BASE_BLOCK_EXIT
#  define XXX_AUDIT_METHOD_GUARD(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_GUARD(::diagnostics::LEVEL_AUDIT, \
                                 ::diagnostics::TYPE_METHOD_ENTER, \
                                 ::diagnostics::TYPE_METHOD_EXIT,XXX_NR_WHAT_DEFAULT, \
                                 ::std::string("METHOD=\"") + DIAGNOSTICS_FUNC_NAME + "\" " + (STR_WHAT))

#  define XXX_AUDIT_ASSERT(NAME,COND) \
    DIAGNOSTICS_BASE_ASSERT(::diagnostics::LEVEL_AUDIT,\
                            XXX_NR_WHAT_DEFAULT,\
                            "CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\"",\
                            (COND),\
                            XXX_FAILED_ASSERT_ACTION(::diagnostics::LEVEL_AUDIT,NAME,COND))
#  define XXX_AUDIT_ASSERT1(NAME,COND,EXCEP_WHAT) \
    DIAGNOSTICS_BASE_ASSERT(::diagnostics::LEVEL_AUDIT,\
                            XXX_NR_WHAT_DEFAULT,\
                            ::std::string("CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\" WHAT=\"") + (EXCEP_WHAT) + "\"",\
                            (COND),\
                            XXX_FAILED_ASSERT_ACTION1(::diagnostics::LEVEL_AUDIT,NAME,COND,EXCEP_WHAT))
#  define XXX_AUDIT_ASSERT_LOG(NAME,COND) \
    DIAGNOSTICS_BASE_ASSERT(::diagnostics::LEVEL_AUDIT,\
                            XXX_NR_WHAT_DEFAULT,\
                            "CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\"",\
                            (COND),\
                            ((void)0))
#  define XXX_AUDIT_ASSERT_LOG1(NAME,COND,EXCEP_WHAT) \
    DIAGNOSTICS_BASE_ASSERT(::diagnostics::LEVEL_AUDIT,\
                            XXX_NR_WHAT_DEFAULT,\
                            ::std::string("CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\" WHAT=\"") + (EXCEP_WHAT) + "\"",\
                            (COND),\
                            ((void)0))
#  define XXX_AUDIT_CHECK(NAME,COND) \
    DIAGNOSTICS_BASE_CHECK(::diagnostics::LEVEL_AUDIT,\
                           XXX_NR_WHAT_DEFAULT,\
                           "CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\"",\
                           (COND),\
                           XXX_FAILED_CHECK_ACTION(::diagnostics::LEVEL_AUDIT,NAME,COND))
#  define XXX_AUDIT_CHECK1(NAME,COND,STR_WHAT) \
    DIAGNOSTICS_BASE_CHECK(::diagnostics::LEVEL_AUDIT,\
                           XXX_NR_WHAT_DEFAULT,\
                           ::std::string("CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\" WHAT=\"") + (STR_WHAT) + "\"",\
                           (COND),\
                           XXX_FAILED_CHECK_ACTION1(::diagnostics::LEVEL_AUDIT,NAME,COND,STR_WHAT))

#  define XXX_AUDIT_CLASS_THROWING_INVARIANCE_ENTER DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_ENTER
#  define XXX_AUDIT_CLASS_THROWING_INVARIANCE_EXIT  \
    DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_EXIT(::diagnostics::LEVEL_AUDIT,\
                                           XXX_NR_WHAT_DEFAULT, \
                                           XXX_HANDLE_TEST_EXCEPTION_EXPLICITLY,\
                                           XXX_BASE_EXCEPTION_TYPE,\
                                           XXX_BASE_EXCEPTION_STR_WHAT)

#  define XXX_AUDIT_CLASS_INVARIANCE_GUARD DIAGNOSTICS_BASE_CLASS_INVARIANCE_GUARD
#  define XXX_AUDIT_CLASS_INVARIANCE_ENTER DIAGNOSTICS_BASE_CLASS_INVARIANCE_ENTER
#  define XXX_AUDIT_CLASS_INVARIANCE_EXIT  DIAGNOSTICS_BASE_CLASS_INVARIANCE_EXIT

#  define XXX_AUDIT_CLASS_INVARIANCE_ASSERT  DIAGNOSTICS_BASE_CLASS_INVARIANCE_ASSERT

#  define XXX_AUDIT_EXCEPTIONLESS_BLOCK_ENTER  DIAGNOSTICS_BASE_EXCEPTIONLESS_BLOCK_ENTER
#  define XXX_AUDIT_EXCEPTIONLESS_BLOCK_EXIT  \
    DIAGNOSTICS_BASE_EXCEPTIONLESS_BLOCK_EXIT(::diagnostics::LEVEL_AUDIT,\
                                              XXX_NR_WHAT_DEFAULT, \
                                              true,\
                                              XXX_HANDLE_TEST_EXCEPTION_EXPLICITLY,\
                                              XXX_BASE_EXCEPTION_TYPE,\
                                              XXX_BASE_EXCEPTION_STR_WHAT)
#  define XXX_AUDIT_EXEC(CMD) CMD
#  define XXX_AUDIT_EXEC_ENTER do { ((void)0)
#  define XXX_AUDIT_EXEC_EXIT  } while(false)

#else

#  define XXX_AUDIT_TRACE(STR_WHAT)        ((void)0)
#  define XXX_AUDIT_BINARY_TRACE(STR_WHAT) ((void)0)
#  define XXX_AUDIT_TRACE1(NR_WHAT,STR_WHAT)        ((void)0)
#  define XXX_AUDIT_BINARY_TRACE1(NR_WHAT,STR_WHAT) ((void)0)

#  define XXX_AUDIT_BLOCK_ENTER(STR_WHAT) do {
#  define XXX_AUDIT_BLOCK_EXIT            } while(false)
#  define XXX_AUDIT_BLOCK_GUARD(STR_WHAT) ((void)0)

#  define XXX_AUDIT_PROCEDURE_ENTER(STR_WHAT) do {
#  define XXX_AUDIT_PROCEDURE_EXIT            } while(false)
#  define XXX_AUDIT_PROCEDURE_GUARD(STR_WHAT) ((void)0)

#  define XXX_AUDIT_METHOD_ENTER(STR_WHAT) do {
#  define XXX_AUDIT_METHOD_EXIT            } while(false)
#  define XXX_AUDIT_METHOD_GUARD(STR_WHAT) ((void)0)

#  define XXX_AUDIT_ASSERT(NAME,COND)           ((void)0)
#  define XXX_AUDIT_ASSERT1(NAME,COND,STR_WHAT) ((void)0)
#  define XXX_AUDIT_ASSERT_LOG(NAME,COND)           ((void)0)
#  define XXX_AUDIT_ASSERT_LOG1(NAME,COND,STR_WHAT) ((void)0)
#  define XXX_AUDIT_CHECK(NAME,COND)            ((void)0)
#  define XXX_AUDIT_CHECK1(NAME,COND,STR_WHAT)  ((void)0)

#  define XXX_AUDIT_CLASS_THROWING_INVARIANCE_ENTER do {
#  define XXX_AUDIT_CLASS_THROWING_INVARIANCE_EXIT  } while(false)

#  define XXX_AUDIT_CLASS_INVARIANCE_GUARD ((void)0)
#  define XXX_AUDIT_CLASS_INVARIANCE_ENTER do {
#  define XXX_AUDIT_CLASS_INVARIANCE_EXIT  } while(false)

#  define XXX_AUDIT_CLASS_INVARIANCE_ASSERT  ((void)0)


#  define XXX_AUDIT_EXCEPTIONLESS_BLOCK_ENTER do {
#  define XXX_AUDIT_EXCEPTIONLESS_BLOCK_EXIT  } while(false)

#  define XXX_AUDIT_EXEC(CMD) ((void)0)
#  define XXX_AUDIT_EXEC_ENTER do { if(false) do { ((void)0)
#  define XXX_AUDIT_EXEC_EXIT  } while(false); } while(false)

#endif


////////////////////////////////////////////////////////////////////////////////

#if XXX_DEBUG__LEVEL__ > 0

#  define XXX_DEBUG_TRACE(STR_WHAT) \
    DIAGNOSTICS_BASE_LOG(::diagnostics::LEVEL_DEBUG,::diagnostics::TYPE_TRACE,XXX_NR_WHAT_DEFAULT,(STR_WHAT))
#  define XXX_DEBUG_BINARY_TRACE(STR_WHAT) \
    DIAGNOSTICS_BASE_LOG(::diagnostics::LEVEL_DEBUG,::diagnostics::TYPE_TRACE_BINARY,XXX_NR_WHAT_DEFAULT,(STR_WHAT))

#  define XXX_DEBUG_BLOCK_ENTER(STR_WHAT)  \
    DIAGNOSTICS_BASE_BLOCK_ENTER(::diagnostics::LEVEL_DEBUG, \
                                 ::diagnostics::TYPE_BLOCK_ENTER, \
                                 ::diagnostics::TYPE_BLOCK_EXIT,XXX_NR_WHAT_DEFAULT,(STR_WHAT))
#  define XXX_DEBUG_BLOCK_EXIT DIAGNOSTICS_BASE_BLOCK_EXIT
#  define XXX_DEBUG_BLOCK_GUARD(STR_WHAT)  \
    DIAGNOSTICS_BASE_BLOCK_GUARD(::diagnostics::LEVEL_DEBUG, \
                                 ::diagnostics::TYPE_BLOCK_ENTER, \
                                 ::diagnostics::TYPE_BLOCK_EXIT,XXX_NR_WHAT_DEFAULT,(STR_WHAT))

#  define XXX_DEBUG_PROCEDURE_ENTER(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_ENTER(::diagnostics::LEVEL_DEBUG, \
                                 ::diagnostics::TYPE_PROCEDURE_ENTER, \
                                 ::diagnostics::TYPE_PROCEDURE_EXIT,XXX_NR_WHAT_DEFAULT, \
                                 ::std::string("PROCEDURE=\"" DIAGNOSTICS_FUNC_NAME "\" ") + (STR_WHAT))
#  define XXX_DEBUG_PROCEDURE_EXIT DIAGNOSTICS_BASE_BLOCK_EXIT
#  define XXX_DEBUG_PROCEDURE_GUARD(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_GUARD(::diagnostics::LEVEL_DEBUG, \
                                 ::diagnostics::TYPE_PROCEDURE_ENTER, \
                                 ::diagnostics::TYPE_PROCEDURE_EXIT,XXX_NR_WHAT_DEFAULT, \
                                 ::std::string("PROCEDURE=\"" DIAGNOSTICS_FUNC_NAME "\" ") + (STR_WHAT))

#  define XXX_DEBUG_METHOD_ENTER(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_ENTER(::diagnostics::LEVEL_DEBUG, \
                                 ::diagnostics::TYPE_METHOD_ENTER, \
                                 ::diagnostics::TYPE_METHOD_EXIT,XXX_NR_WHAT_DEFAULT, \
                                 ::std::string("METHOD=\"" DIAGNOSTICS_FUNC_NAME "\" ") + (STR_WHAT))
#  define XXX_DEBUG_METHOD_EXIT DIAGNOSTICS_BASE_BLOCK_EXIT
#  define XXX_DEBUG_METHOD_GUARD(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_GUARD(::diagnostics::LEVEL_DEBUG, \
                                 ::diagnostics::TYPE_METHOD_ENTER, \
                                 ::diagnostics::TYPE_METHOD_EXIT,XXX_NR_WHAT_DEFAULT, \
                                 ::std::string("METHOD=\"" DIAGNOSTICS_FUNC_NAME "\" ") + (STR_WHAT))

#  define XXX_DEBUG_ASSERT(NAME,COND) \
    DIAGNOSTICS_BASE_ASSERT(::diagnostics::LEVEL_DEBUG,\
                            XXX_NR_WHAT_DEFAULT,\
                            "CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\"",\
                            (COND),\
                            XXX_FAILED_ASSERT_ACTION(::diagnostics::LEVEL_DEBUG,NAME,COND))
#  define XXX_DEBUG_ASSERT1(NAME,COND,STR_WHAT) \
    DIAGNOSTICS_BASE_ASSERT(::diagnostics::LEVEL_DEBUG,\
                            XXX_NR_WHAT_DEFAULT,\
                            ::std::string("CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\" WHAT=\"") + (STR_WHAT) + "\"",\
                            (COND),\
                            XXX_FAILED_ASSERT_ACTION1(::diagnostics::LEVEL_DEBUG,NAME,COND,STR_WHAT))
#  define XXX_DEBUG_CHECK(NAME,COND) \
    DIAGNOSTICS_BASE_CHECK(::diagnostics::LEVEL_DEBUG,\
                           XXX_NR_WHAT_DEFAULT,\
                           "CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\"",\
                           (COND),\
                           XXX_FAILED_CHECK_ACTION(::diagnostics::LEVEL_DEBUG,NAME,COND))
#  define XXX_DEBUG_CHECK1(NAME,COND,STR_WHAT) \
    DIAGNOSTICS_BASE_CHECK(::diagnostics::LEVEL_DEBUG,\
                           XXX_NR_WHAT_DEFAULT,\
                           ::std::string("CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\" WHAT=\"") + (STR_WHAT) + "\"",\
                           (COND),\
                           XXX_FAILED_CHECK_ACTION1(::diagnostics::LEVEL_DEBUG,NAME,COND,STR_WHAT))

#  define XXX_DEBUG_CLASS_THROWING_INVARIANCE_ENTER DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_ENTER
#  define XXX_DEBUG_CLASS_THROWING_INVARIANCE_EXIT  \
    DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_EXIT(::diagnostics::LEVEL_DEBUG,\
                                           XXX_NR_WHAT_DEFAULT, \
                                           XXX_HANDLE_TEST_EXCEPTION_EXPLICITLY,\
                                           XXX_BASE_EXCEPTION_TYPE,\
                                           XXX_BASE_EXCEPTION_STR_WHAT)

#  define XXX_DEBUG_CLASS_INVARIANCE_GUARD DIAGNOSTICS_BASE_CLASS_INVARIANCE_GUARD
#  define XXX_DEBUG_CLASS_INVARIANCE_ENTER DIAGNOSTICS_BASE_CLASS_INVARIANCE_ENTER
#  define XXX_DEBUG_CLASS_INVARIANCE_EXIT  DIAGNOSTICS_BASE_CLASS_INVARIANCE_EXIT

#  define XXX_DEBUG_CLASS_INVARIANCE_ASSERT DIAGNOSTICS_BASE_CLASS_INVARIANCE_ASSERT


#  define XXX_DEBUG_EXCEPTIONLESS_BLOCK_ENTER  DIAGNOSTICS_BASE_EXCEPTIONLESS_BLOCK_ENTER
#  define XXX_DEBUG_EXCEPTIONLESS_BLOCK_EXIT  \
    DIAGNOSTICS_BASE_EXCEPTIONLESS_BLOCK_EXIT(::diagnostics::LEVEL_DEBUG,\
                                              XXX_NR_WHAT_DEFAULT, \
                                              true,\
                                              XXX_HANDLE_TEST_EXCEPTION_EXPLICITLY,\
                                              XXX_BASE_EXCEPTION_TYPE,\
                                              XXX_BASE_EXCEPTION_STR_WHAT)

#  define XXX_DEBUG_EXEC(CMD) CMD
#  define XXX_DEBUG_EXEC_ENTER do { ((void)0)
#  define XXX_DEBUG_EXEC_EXIT  } while(false)

#else

#  define XXX_DEBUG_TRACE(STR_WHAT)        ((void)0)
#  define XXX_DEBUG_BINARY_TRACE(STR_WHAT) ((void)0)

#  define XXX_DEBUG_BLOCK_ENTER(STR_WHAT) do {
#  define XXX_DEBUG_BLOCK_EXIT            } while(false)
#  define XXX_DEBUG_BLOCK_GUARD(STR_WHAT) ((void)0)

#  define XXX_DEBUG_PROCEDURE_ENTER(STR_WHAT) do {
#  define XXX_DEBUG_PROCEDURE_EXIT            } while(false)
#  define XXX_DEBUG_PROCEDURE_GUARD(STR_WHAT) ((void)0)

#  define XXX_DEBUG_METHOD_ENTER(STR_WHAT) do {
#  define XXX_DEBUG_METHOD_EXIT            } while(false)
#  define XXX_DEBUG_METHOD_GUARD(STR_WHAT) ((void)0)

#  define XXX_DEBUG_ASSERT(NAME,COND)           ((void)0)
#  define XXX_DEBUG_ASSERT1(NAME,COND,STR_WHAT) ((void)0)
#  define XXX_DEBUG_CHECK(NAME,COND)            ((void)0)
#  define XXX_DEBUG_CHECK1(NAME,COND,STR_WHAT)  ((void)0)

#  define XXX_DEBUG_CLASS_THROWING_INVARIANCE_ENTER do {
#  define XXX_DEBUG_CLASS_THROWING_INVARIANCE_EXIT  } while(false)

#  define XXX_DEBUG_CLASS_INVARIANCE_GUARD ((void)0)
#  define XXX_DEBUG_CLASS_INVARIANCE_ENTER do {
#  define XXX_DEBUG_CLASS_INVARIANCE_EXIT  } while(false)

#  define XXX_DEBUG_CLASS_INVARIANCE_ASSERT  ((void)0)


#  define XXX_DEBUG_EXCEPTIONLESS_BLOCK_ENTER do {
#  define XXX_DEBUG_EXCEPTIONLESS_BLOCK_EXIT  } while(false)

#  define XXX_DEBUG_EXEC(CMD) ((void)0)
#  define XXX_DEBUG_EXEC_ENTER do { if(false) do { ((void)0)
#  define XXX_DEBUG_EXEC_EXIT  } while(false); } while(false)

#endif


////////////////////////////////////////////////////////////////////////////////

#if XXX_DEBUG__LEVEL__ >= 0

#define XXX_PROD_TRACE(STR_WHAT) \
    DIAGNOSTICS_BASE_LOG(::diagnostics::LEVEL_PROD,::diagnostics::TYPE_TRACE,XXX_NR_WHAT_DEFAULT,(STR_WHAT))
#define XXX_PROD_BINARY_TRACE(STR_WHAT) \
    DIAGNOSTICS_BASE_LOG(::diagnostics::LEVEL_PROD,::diagnostics::TYPE_TRACE_BINARY,XXX_NR_WHAT_DEFAULT,(STR_WHAT))

#define XXX_PROD_TRACE1(NR_WHAT,STR_WHAT) \
    DIAGNOSTICS_BASE_LOG(::diagnostics::LEVEL_PROD,::diagnostics::TYPE_TRACE,(NR_WHAT),(STR_WHAT))
#define XXX_PROD_BINARY_TRACE1(NR_WHAT,STR_WHAT) \
    DIAGNOSTICS_BASE_LOG(::diagnostics::LEVEL_PROD,::diagnostics::TYPE_TRACE_BINARY,(NR_WHAT),(STR_WHAT))


#define XXX_PROD_BLOCK_ENTER(STR_WHAT)  \
    DIAGNOSTICS_BASE_BLOCK_ENTER(::diagnostics::LEVEL_PROD, \
                                 ::diagnostics::TYPE_BLOCK_ENTER, \
                                 ::diagnostics::TYPE_BLOCK_EXIT,XXX_NR_WHAT_DEFAULT,(STR_WHAT))
#define XXX_PROD_BLOCK_EXIT DIAGNOSTICS_BASE_BLOCK_EXIT
#define XXX_PROD_BLOCK_GUARD(STR_WHAT)  \
    DIAGNOSTICS_BASE_BLOCK_GUARD(::diagnostics::LEVEL_PROD, \
                                 ::diagnostics::TYPE_BLOCK_ENTER, \
                                 ::diagnostics::TYPE_BLOCK_EXIT,XXX_NR_WHAT_DEFAULT,(STR_WHAT))

#define XXX_PROD_PROCEDURE_ENTER(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_ENTER(::diagnostics::LEVEL_PROD, \
                                 ::diagnostics::TYPE_PROCEDURE_ENTER, \
                                 ::diagnostics::TYPE_PROCEDURE_EXIT,XXX_NR_WHAT_DEFAULT, \
                                 ::std::string("PROCEDURE=\"" DIAGNOSTICS_FUNC_NAME "\" ") + (STR_WHAT))
#define XXX_PROD_PROCEDURE_EXIT DIAGNOSTICS_BASE_BLOCK_EXIT
#define XXX_PROD_PROCEDURE_GUARD(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_GUARD(::diagnostics::LEVEL_PROD, \
                                 ::diagnostics::TYPE_PROCEDURE_ENTER, \
                                 ::diagnostics::TYPE_PROCEDURE_EXIT,XXX_NR_WHAT_DEFAULT, \
                                 ::std::string("PROCEDURE=\"" DIAGNOSTICS_FUNC_NAME "\" ") + (STR_WHAT))

#define XXX_PROD_METHOD_ENTER(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_ENTER(::diagnostics::LEVEL_PROD, \
                                 ::diagnostics::TYPE_METHOD_ENTER, \
                                 ::diagnostics::TYPE_METHOD_EXIT,XXX_NR_WHAT_DEFAULT, \
                                 ::std::string("METHOD=\"" DIAGNOSTICS_FUNC_NAME "\" ") + (STR_WHAT))
#define XXX_PROD_METHOD_EXIT DIAGNOSTICS_BASE_BLOCK_EXIT
#define XXX_PROD_METHOD_GUARD(STR_WHAT) \
    DIAGNOSTICS_BASE_BLOCK_GUARD(::diagnostics::LEVEL_PROD, \
                                 ::diagnostics::TYPE_METHOD_ENTER, \
                                 ::diagnostics::TYPE_METHOD_EXIT,XXX_NR_WHAT_DEFAULT, \
                                 ::std::string("METHOD=\"" DIAGNOSTICS_FUNC_NAME "\" ") + (STR_WHAT))

#define XXX_PROD_ASSERT(NAME,COND) \
    DIAGNOSTICS_BASE_ASSERT(::diagnostics::LEVEL_PROD,\
                            XXX_NR_WHAT_DEFAULT,\
                            "CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\"",\
                            (COND),\
                            XXX_FAILED_ASSERT_ACTION(::diagnostics::LEVEL_PROD,NAME,COND))
#define XXX_PROD_ASSERT1(NAME,COND,STR_WHAT) \
    DIAGNOSTICS_BASE_ASSERT(::diagnostics::LEVEL_PROD,\
                            XXX_NR_WHAT_DEFAULT,\
                            ::std::string("CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\" WHAT=\"") + (STR_WHAT) + "\"",\
                            (COND),\
                            XXX_FAILED_ASSERT_ACTION1(::diagnostics::LEVEL_PROD,NAME,COND,STR_WHAT))

#define XXX_PROD_CLASS_THROWING_INVARIANCE_ENTER DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_ENTER
#define XXX_PROD_CLASS_THROWING_INVARIANCE_EXIT  \
    DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_EXIT(::diagnostics::LEVEL_PROD,\
                                           XXX_NR_WHAT_DEFAULT, \
                                           XXX_HANDLE_TEST_EXCEPTION_EXPLICITLY,\
                                           XXX_BASE_EXCEPTION_TYPE,\
                                           XXX_BASE_EXCEPTION_STR_WHAT)

#define XXX_PROD_CLASS_INVARIANCE_GUARD DIAGNOSTICS_BASE_CLASS_INVARIANCE_GUARD
#define XXX_PROD_CLASS_INVARIANCE_ENTER DIAGNOSTICS_BASE_CLASS_INVARIANCE_ENTER
#define XXX_PROD_CLASS_INVARIANCE_EXIT  DIAGNOSTICS_BASE_CLASS_INVARIANCE_EXIT

#define XXX_PROD_CLASS_INVARIANCE_ASSERT  DIAGNOSTICS_BASE_CLASS_INVARIANCE_ASSERT

#define XXX_PROD_EXCEPTIONLESS_BLOCK_ENTER  DIAGNOSTICS_BASE_EXCEPTIONLESS_BLOCK_ENTER
#define XXX_PROD_EXCEPTIONLESS_BLOCK_EXIT  \
    DIAGNOSTICS_BASE_EXCEPTIONLESS_BLOCK_EXIT(::diagnostics::LEVEL_PROD,\
                                              XXX_NR_WHAT_DEFAULT, \
                                              true,\
                                              XXX_HANDLE_TEST_EXCEPTION_EXPLICITLY,\
                                              XXX_BASE_EXCEPTION_TYPE,\
                                              XXX_BASE_EXCEPTION_STR_WHAT)

#else

#  define XXX_PROD_TRACE(STR_WHAT)        ((void)0)
#  define XXX_PROD_BINARY_TRACE(STR_WHAT) ((void)0)

#  define XXX_PROD_BLOCK_ENTER(STR_WHAT) do {
#  define XXX_PROD_BLOCK_EXIT            } while(false)
#  define XXX_PROD_BLOCK_GUARD(STR_WHAT) ((void)0)

#  define XXX_PROD_PROCEDURE_ENTER(STR_WHAT) do {
#  define XXX_PROD_PROCEDURE_EXIT            } while(false)
#  define XXX_PROD_PROCEDURE_GUARD(STR_WHAT) ((void)0)

#  define XXX_PROD_METHOD_ENTER(STR_WHAT) do {
#  define XXX_PROD_METHOD_EXIT            } while(false)
#  define XXX_PROD_METHOD_GUARD(STR_WHAT) ((void)0)

#  define XXX_PROD_ASSERT(NAME,COND)           if( ! ( COND ) ) throw NAME ( #COND )
#  define XXX_PROD_ASSERT1(NAME,COND,STR_WHAT) if( ! ( COND ) ) throw NAME ( STR_WHAT )

#  define XXX_PROD_CLASS_THROWING_INVARIANCE_ENTER do {
#  define XXX_PROD_CLASS_THROWING_INVARIANCE_EXIT  } while(false)

#  define XXX_PROD_CLASS_INVARIANCE_GUARD ((void)0)
#  define XXX_PROD_CLASS_INVARIANCE_ENTER do {
#  define XXX_PROD_CLASS_INVARIANCE_EXIT  } while(false)

#  define XXX_PROD_CLASS_INVARIANCE_ASSERT  ((void)0)


#  define XXX_PROD_EXCEPTIONLESS_BLOCK_ENTER do {
#  define XXX_PROD_EXCEPTIONLESS_BLOCK_EXIT  } while(false)

#endif


#if XXX_DEBUG__LEVEL__ > 0

#  define XXX_PROD_CHECK(NAME,COND) \
    DIAGNOSTICS_BASE_CHECK(::diagnostics::LEVEL_PROD,\
                           XXX_NR_WHAT_DEFAULT,\
                           "CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\"",\
                           (COND),\
                           XXX_FAILED_CHECK_ACTION(::diagnostics::LEVEL_PROD,NAME,COND))
#  define XXX_PROD_CHECK1(NAME,COND,STR_WHAT) \
    DIAGNOSTICS_BASE_CHECK(::diagnostics::LEVEL_PROD,\
                           XXX_NR_WHAT_DEFAULT,\
                           ::std::string("CONDITION=\"" #COND "\" EXCEPTION=\"" #NAME "\" WHAT=\"") + (STR_WHAT) + "\"",\
                           (COND),\
                           XXX_FAILED_CHECK_ACTION1(::diagnostics::LEVEL_PROD,NAME,COND,STR_WHAT))

#else

#  define XXX_PROD_CHECK(NAME,COND) \
    do { if(!(COND)) { XXX_FAILED_CHECK_ACTION(::diagnostics::LEVEL_PROD,NAME,COND); } } while(false)
#  define XXX_PROD_CHECK1(NAME,COND,STR_WHAT) \
    do { if(!(COND)) { XXX_FAILED_CHECK_ACTION1(::diagnostics::LEVEL_PROD,NAME,COND,STR_WHAT); } } while(false)

#endif

#define XXX_METHOD_GUARD(STR_WHAT) XXX_AUDIT_METHOD_GUARD(STR_WHAT); XXX_AUDIT_CLASS_INVARIANCE_GUARD 


#define XXX_AUDIT_ASSERT_RELATION(NAME,OP1,REL,OP2) \
       DIAGNOSTICS_RELATION(XXX_AUDIT_ASSERT1,NAME,OP1,REL,OP2)
#define XXX_AUDIT_CHECK_RELATION(NAME,OP1,REL,OP2) \
       DIAGNOSTICS_RELATION(XXX_AUDIT_CHECK1,NAME,OP1,REL,OP2)

#define XXX_DEBUG_ASSERT_RELATION(NAME,OP1,REL,OP2) \
       DIAGNOSTICS_RELATION(XXX_DEBUG_ASSERT1,NAME,OP1,REL,OP2)
#define XXX_DEBUG_CHECK_RELATION(NAME,OP1,REL,OP2) \
       DIAGNOSTICS_RELATION(XXX_DEBUG_CHECK1,NAME,OP1,REL,OP2)

#define XXX_PROD_ASSERT_RELATION(NAME,OP1,REL,OP2) \
       DIAGNOSTICS_RELATION(XXX_PROD_ASSERT1,NAME,OP1,REL,OP2)
#define XXX_PROD_CHECK_RELATION(NAME,OP1,REL,OP2) \
       DIAGNOSTICS_RELATION(XXX_PROD_CHECK1,NAME,OP1,REL,OP2)


#endif

// vim:ts=4:sw=4
