/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */



#include <diagnostics/extensions/memory/allocation_database.hpp>

#include <diagnostics/extensions/memory/allocation_record.hpp>
#include <diagnostics/extensions/memory/malloc_allocator.hpp>

#include <diagnostics/frame/logging_implementation.hpp>
#include <diagnostics/frame/logging_mutex.hpp>

#include <diagnostics/util/to_string.hpp>

#include <map>
#include <utility>

DIAGNOSTICS_NAMESPACE_BEGIN;
MEMORY_NAMESAPCE_BEGIN;

using ::diagnostics::internal::to_string;
using ::diagnostics::internal::Logging_Implementation;

#define MEMORY_LOG(TYPE,STR_WHAT) Logging_Implementation::instance()->log(LEVEL_AUDIT,TYPE,0,STR_WHAT,"","",0)
#define MEMORY_LOG_SYS(TYPE,STR_WHAT) Logging_Implementation::instance()->log(LEVEL_SYSTEM,TYPE,0,STR_WHAT,"","",0)

class Allocation_Database_Implementation
{
	//typedef ::diagnostics::memory::internal::Malloc_Allocator<void*> Allocator_t;

	typedef ::diagnostics::memory::internal::Malloc_Allocator< ::std::pair<void* const, ::diagnostics::memory::Allocation_Record> > Allocator_t;

    typedef ::std::map<void*,Allocation_Record, ::std::less<void*>, 
					   Allocator_t> Allocations_t;

    typedef Allocation_Database_Implementation Self;
    
    int m_log_nesting;
    static int m_check_nesting;
    Tick_t m_creation_tick;
    Allocations_t m_allocations;
    ::std::size_t m_size;
private:
    friend class Allocation_Database;
    
    Allocation_Database_Implementation()
		:  m_log_nesting(0),
		   m_creation_tick(0),
		   m_allocations(),
		   m_size(0)
    {
		Logging_Implementation::instance()->log(LEVEL_SYSTEM,TYPE_TRACE,0,"MEM: Allocation_Database initialized","","",0);
    }

    static Self * m_instance;
public:
    static inline Allocation_Database_Implementation* instance()
    {
		if(m_instance==NULL) 
			m_instance=new Self;
		return m_instance;
    }

    inline void register_allocation(void * const p, ::std::size_t const len) 
    {
		m_allocations[p]=Allocation_Record(len,++m_creation_tick);
		m_size+=len;
		if(m_log_nesting!=0)
			MEMORY_LOG(TYPE_TRACE,
					   "MEMORY: Allocation:   "
					   "address=" + to_string(p)
					   + " size=" + to_string(len)
					   + " tick=" + to_string(m_creation_tick));
    }
    inline bool register_deallocation(void * const p) 
    {
		Allocations_t::iterator const pos(m_allocations.find(p));
		if(pos==m_allocations.end()) {
			MEMORY_LOG(TYPE_FAILED_ASSERTION,
					   "MEMORY: Unknown_Block_Deallocation: "
					   "address=" + to_string(p));
			return false;
		}

		// we know the block
		m_size-=pos->second.m_len;
		if(m_log_nesting!=0)
			MEMORY_LOG(TYPE_TRACE,"MEMORY: Deallocation: "
					   "address=" + to_string(p)
					   + " size=" + to_string(pos->second.m_len)
					   + " tick=" + to_string(pos->second.m_creation_tick));
		m_allocations.erase(pos);
	
		return true;
    }

    inline void balance(::std::size_t & size,
						Tick_t & number)
    {
		size=m_size;
		number=m_allocations.size();
    }

    inline void balance(Tick_t const start_from,
						::std::size_t & size,
						Tick_t & number)
    {
		size=0;
		number=0;
		Allocations_t::const_iterator begin(m_allocations.begin());
		Allocations_t::const_iterator const end(m_allocations.end());
		for(;begin!=end;++begin) {
			if(start_from<=begin->second.m_creation_tick) {
				++number;
				size+=begin->second.m_len;
			}
		}
    }


    inline Tick_t * unallocated_blocks(Tick_t const start_from,Tick_t * b, Tick_t * const e)
    {
		Allocations_t::const_iterator begin(m_allocations.begin());
		Allocations_t::const_iterator const end(m_allocations.end());
		for(;begin!=end;++begin) {
			if(start_from<=begin->second.m_creation_tick) {
				*b=begin->second.m_creation_tick;
				++b;
				if(b==e) return b;
			}
		}
		return b;
    }


    inline void reset()
    {
		MEMORY_LOG(TYPE_TRACE,
				   "MEMORY: Reset:"
				   " tick="+to_string(m_creation_tick)
				   +" number="+to_string(m_allocations.size())
				   +" size="+to_string(m_size));
		m_allocations=Allocations_t();
		m_creation_tick=0;
		m_size=0;
    }

    inline void analysis_on(bool const logging)
    {
		++m_check_nesting;
		if(logging) ++m_log_nesting;
    }
    
    inline void analysis_off(bool const logging)
    {
		--m_check_nesting;
		// we reset the database if no checking/logging is going on
		if(m_check_nesting==0) reset();
		if(logging) --m_log_nesting;
    }
   
    inline Tick_t creation_tick()
    {
		return m_creation_tick;
    }    
};

Allocation_Database_Implementation* Allocation_Database_Implementation::m_instance=NULL;
int Allocation_Database_Implementation::m_check_nesting=0;

////////////////////////////////////////////////////////////////////////////////

void Allocation_Database::register_allocation(void* const p, ::std::size_t const len) 
{
	// this check MUST come before doing the guard:
	// the guard might allocate its mutex -- and thus we had an 
	// endless loop.
	//
	// This is NOT TOTALLY thread safe: m_check_nesting can only be
	// changed while being in the critical section -- but it could
	// change after this check. However, if a memory anlysis depends
	// on a race condition, it is unusable anyhow.
    if(Allocation_Database_Implementation::m_check_nesting==0) return;
	
	DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD(lock_guard);
	if(DIAGNOSTICS_LOGGING_FACILITY_LOCK_IS_NESTED(lock_guard))
		return;

	Allocation_Database_Implementation::instance()->register_allocation(p,len);
}

bool Allocation_Database::register_deallocation(void* const p) 
{
	// see above
	if(Allocation_Database_Implementation::m_check_nesting==0) return true;
	
	DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD(lock_guard);
	if(DIAGNOSTICS_LOGGING_FACILITY_LOCK_IS_NESTED(lock_guard))
		return true;

	return Allocation_Database_Implementation::instance()->register_deallocation(p);
}

void Allocation_Database::balance(Tick_t const start_from,
								  ::std::size_t & size,
								  Tick_t & number)
{
    DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD(lock_guard);
    Allocation_Database_Implementation::instance()->balance(start_from,size,number);
}

Tick_t * Allocation_Database::unallocated_blocks(Tick_t const start_from,
												 Tick_t * const b,
												 Tick_t * const e)
{
    DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD(lock_guard);
    return Allocation_Database_Implementation::instance()->unallocated_blocks(start_from,b,e);
}


void Allocation_Database::balance(::std::size_t & size,
								  Tick_t & number)
{
    DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD(lock_guard);
    Allocation_Database_Implementation::instance()->balance(size,number);
}


void Allocation_Database::reset()
{
    DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD(lock_guard);
    Allocation_Database_Implementation::instance()->reset();
}

void Allocation_Database::analysis_on(bool const logging)
{
    DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD(lock_guard);
    Allocation_Database_Implementation::instance()->analysis_on(logging);
}

void Allocation_Database::analysis_off(bool const logging)
{
    DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD(lock_guard);
    Allocation_Database_Implementation::instance()->analysis_off(logging);
}

Tick_t Allocation_Database::creation_tick()
{
    DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD(lock_guard);
    return Allocation_Database_Implementation::instance()->creation_tick();
}


MEMORY_NAMESAPCE_END;
DIAGNOSTICS_NAMESPACE_END;

// vim:ts=4:sw=4
