/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/file_test_data_source.t.cpp
 *
 * @brief [LEVEL: alpha] testing @ref diagnostics::unittest::File_Test_Data_Source
 *
 * $Id: file_test_data_source.t.cpp,v 1.11 2005/06/23 09:54:25 esdentem Exp $
 *
 * @author Christian Schallhart
 */
#include <diagnostics/unittest.hpp>

// tested component
#include <diagnostics/unittest/test_system/file_test_data_source.hpp>

#include <diagnostics/unittest/test_system_exception.hpp>

// test support
#include <diagnostics/unittest/test_system/test_data_source.ts.hpp>

#define NONEXISTINGTESTFILENAME "test/file_test_data_source.nonexisting.testfile"
#define TESTFILENAME "test/scratch/file_test_data_source.testfile"

#define TEST_COMPONENT_NAME File_Test_Data_Source
#define TEST_COMPONENT_NAMESPACE diagnostics::unittest

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;
TEST_COMPONENT_TEST_NAMESPACE_BEGIN;

/**
 * @brief [PRO] fct/normal: testing the default-cstr.
 */
void creation(Test_Data & data)
{
    File_Test_Data_Source s;
    TEST_ASSERT(s.is_initialized()==false);

    TEST_THROWING_BLOCK_ENTER;
    s.is_modified();
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    s.file_name();
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    s.exists_entry("x");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    s.get_entry("x");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    s.set_entry("x","y");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_ASSERT(s.is_initialized()==false);
}


/**
 * @brief [PRO] fct/normal: testing init
 */
void init(Test_Data & data)
{
    File_Test_Data_Source s;
    TEST_ASSERT(s.is_initialized()==false);

    TEST_THROWING_BLOCK_ENTER;
    s.is_modified();
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    s.init(NONEXISTINGTESTFILENAME);
    TEST_ASSERT(s.is_initialized()==true);
    TEST_ASSERT(s.is_modified()==false);
    TEST_ASSERT(s.file_name()==NONEXISTINGTESTFILENAME);

    TEST_ASSERT(s.exists_entry("x")==false);
    
    TEST_THROWING_BLOCK_ENTER;
    s.get_entry("x");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
}

/**
 * @brief [PRO] fct/normal: the methods of @ref Test_Data_Source
 * interface -- together with the is_modified()
 *
 * @attention: One property can only be tested in the conext of
 * save/load, namely the the interplay between set_entry and
 * is_modified in the context of an already loaded database.
 */
void exists_set_get_modified(Test_Data & data)
{
    File_Test_Data_Source s;
    s.init(NONEXISTINGTESTFILENAME);
    TEST_ASSERT(s.is_modified()==false);

    TEST_ASSERT(s.exists_entry("x")==false);
    
    TEST_THROWING_BLOCK_ENTER;
    s.get_entry("x");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_ASSERT(s.is_modified()==false);
    // set now
    s.set_entry("x","y");
    TEST_ASSERT(s.is_modified()==true);
    TEST_ASSERT(s.get_entry("x")=="y");
    TEST_THROWING_BLOCK_ENTER;
    s.get_entry("x1");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);


    // set another one
    s.set_entry("x1","y1");
    TEST_ASSERT(s.is_modified()==true);
    TEST_ASSERT(s.get_entry("x")=="y");
    TEST_ASSERT(s.get_entry("x1")=="y1");
    TEST_THROWING_BLOCK_ENTER;
    s.get_entry("x2");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    // overwrite
    s.set_entry("x","Y");
    TEST_ASSERT(s.is_modified()==true);
    TEST_ASSERT(s.get_entry("x")=="Y");
    TEST_ASSERT(s.get_entry("x1")=="y1");
    TEST_THROWING_BLOCK_ENTER;
    s.get_entry("x2");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    // reinit
    s.init(NONEXISTINGTESTFILENAME);
    TEST_ASSERT(s.is_modified()==false);
    TEST_THROWING_BLOCK_ENTER;
    s.get_entry("x");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    TEST_THROWING_BLOCK_ENTER;
    s.get_entry("x1");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
}

/**
 * @brief [PRO] fct/normal: saving and loading. This test generates
 * two files test/file_test_data_source.testfile{2|} -- but is
 * independent of whether these files exist or not.
 */
void save_load(Test_Data & data)
{
    File_Test_Data_Source s;

    // loading non existing file
    TEST_THROWING_BLOCK_ENTER;
    s.load(NONEXISTINGTESTFILENAME);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    
    // init
    s.init(TESTFILENAME);
    TEST_ASSERT(s.is_modified()==false);

    TEST_THROWING_BLOCK_ENTER;
    s.get_entry("x");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    // set now
    s.set_entry("x","y");
    s.set_entry("x1","y1");
    s.set_entry("x","Y");

    TEST_ASSERT(s.is_modified()==true);
    s.save(TESTFILENAME);
    TEST_ASSERT(s.is_initialized()==true);
    TEST_ASSERT(s.is_modified()==false);
    TEST_ASSERT(s.file_name()==TESTFILENAME);

    // load into another source
    File_Test_Data_Source s2;
    s2.load(TESTFILENAME);
    TEST_ASSERT(s2.is_initialized()==true);
    TEST_ASSERT(s2.is_modified()==false);
    TEST_ASSERT(s2.file_name()==TESTFILENAME);
    TEST_ASSERT(s2.get_entry("x")=="Y");
    TEST_ASSERT(s2.get_entry("x1")=="y1");

    // overwrite -- no change -- still is_modified turns true
    s2.set_entry("x1","y1");
    TEST_ASSERT(s2.get_entry("x1")=="y1");
    TEST_ASSERT(s2.is_modified()==true);
    
    // save into another file
    s2.save(TESTFILENAME "2");
    TEST_ASSERT(s2.is_initialized()==true);
    TEST_ASSERT(s2.is_modified()==false);
    TEST_ASSERT(s2.file_name()==TESTFILENAME "2");


    // init to TESTFILENAME "2" /////////////////////////////////
    s2.init(TESTFILENAME "2");
    TEST_ASSERT(s2.is_initialized()==true);
    TEST_ASSERT(s2.is_modified()==false);
    TEST_ASSERT(s2.file_name()==TESTFILENAME "2");

	// save to TESTFILENAME "2" -- NO FORCE
	s2.save(TESTFILENAME "2");

    // load TESTFILENAME "2" and check that it did not change
    s2.load(TESTFILENAME "2");
    TEST_ASSERT(s2.is_initialized()==true);
    TEST_ASSERT(s2.is_modified()==false);
    TEST_ASSERT(s2.file_name()==TESTFILENAME "2");
    TEST_ASSERT(s2.get_entry("x")=="Y");
    TEST_ASSERT(s2.get_entry("x1")=="y1");


    // init to TESTFILENAME "2" /////////////////////////////////
    s2.init(TESTFILENAME "2");
    TEST_ASSERT(s2.is_initialized()==true);
    TEST_ASSERT(s2.is_modified()==false);
    TEST_ASSERT(s2.file_name()==TESTFILENAME "2");

	// save to TESTFILENAME "2" -- WITH FORCE //////////////////
	s2.save(TESTFILENAME "2",true);

    // load TESTFILENAME "2" and check that it did not change
    s2.load(TESTFILENAME "2");
    TEST_ASSERT(s2.is_initialized()==true);
    TEST_ASSERT(s2.is_modified()==false);
    TEST_ASSERT(s2.file_name()==TESTFILENAME "2");
    TEST_ASSERT(!s2.exists_entry("x"));
    TEST_ASSERT(!s2.exists_entry("x1"));
}


TEST_COMPONENT_TEST_NAMESPACE_END;
TEST_NAMESPACE_END;
UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

TEST_SUITE_BEGIN;
TEST_NORMAL_CASE(&creation,LEVEL_PROD);
TEST_NORMAL_CASE(&init,LEVEL_PROD);
TEST_NORMAL_CASE(&exists_set_get_modified,LEVEL_PROD);
TEST_NORMAL_CASE(&save_load,LEVEL_PROD);

File_Test_Data_Source* source(new File_Test_Data_Source);
source->init(NONEXISTINGTESTFILENAME);
TEST_NORMAL_CASE(Test_Data_Source_Compliance(source, new File_Test_Data_Source),LEVEL_PROD);
TEST_SUITE_END;

STREAM_TEST_SYSTEM_MAIN;
// vim:ts=4:sw=4
