/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/frame/logger.hpp
 *
 * @brief [LEVEL: beta] The @ref diagnostics::Logger Interface
 *
 * $Id: logger.hpp,v 1.11 2005/06/23 09:54:19 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @test no testing -- no compliance test
 */

#ifndef DIAGNOSTICS__FRAME__LOGGER_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__FRAME__LOGGER_HPP__INCLUDE_GUARD

#include <diagnostics/frame/namespace.hpp>

DIAGNOSTICS_NAMESPACE_BEGIN;

// used in the interface by reference
// <diagnostics/frame/record.hpp>
class Record;

/**
 * @class Logger diagnostics/frame/logger.hpp
 *
 * @brief A Logger can be registered in the logging framework (see
 * @ref Logging_Config). Each @ref Record being logged subsequently is
 * passed to the Logger (by invocations of @ref Logging_Facility::log).
 *
 * In general, each Logger has to work without throwing any exceptions
 * since Loggers are used to report errors -- error reporting which
 * has error conditions is not a good idea. However, to provide a
 * possibility to report problems, each Logger can use the @ref
 * DIAGNOSTICS_PANIC_LOG macro to report a problem. If a logger runs
 * into a problem, it should try to go on as well as possible.
 *
 * Logger do not have to take care for mutual exclusion: The framework
 * is only processing one log message after the other, a logger has
 * the guarantee that only one thread is accessing it at a time.
 *
 * @attention Loggers are not allowed to use the @ref
 * Logging_Facility::log call directly or indirectly. If @ref
 * DIAGNOSTICS_SWITCH_CHECK_FOR_NESTING is set to 1, then such nested
 * logging attempts are checked for and in case, a message is written
 * to the panic log (see @ref DIAGNOSTICS_PANIC_LOG).
 *
 * @nosubgrouping
 */
class Logger
{
	////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Creation
     * @{
     */
public:
    /**
     * @throw never
     */
    virtual ~Logger();
	// @}
    
    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Logging Interface
     * @{
     */
public:
    /**
     * @brief Logs a @ref Record. 
     *
     * @throw never
     */
    virtual void log(Record const & record) =0;
    
    /** 
     * @brief Called by the framework, if the framework is shutting down,
     * and the logger is still registered in the framework.
     *
     * Once this method is called, no other method on the Logger will
     * be invoked again (not even the destructor). This happens
     * immediately before the process exits. Therefore, this method
     * might be used to call the destructor internally.
     *
     * @throw never
     */
    virtual void close() =0;
    // @}
};


DIAGNOSTICS_NAMESPACE_END;

#endif

// vim:ts=4:sw=4
