/* utils.cpp
 * Functions useful across the different modules of
 * drawing and non-drawing code.
 *
 * for Denemo, a gtk+ frontend to GNU Lilypond
 * (c) 1999, 2000, 2001, 2002 Matthew Hiller
 */

#include <stdio.h>
#include <gtk/gtk.h>
#include "accwidths.h"
#include <denemo/denemo.h>
#include "notewidths.h"
#include "utils.h"

void
drawbitmapinverse (GdkPixmap * pixmap, GdkGC * gc, GdkBitmap * mask, gint x,
		   gint y, gint width, gint height)
{
  gdk_gc_set_clip_mask (gc, mask);
  gdk_gc_set_clip_origin (gc, x, y);
  gdk_draw_rectangle (pixmap, gc, TRUE, x, y, width, height);
  gdk_gc_set_clip_mask (gc, NULL);	/* Removes clip mask */
}

void
set_tuplefied_numticks (mudelaobject * theobj, gint numerator,
			gint denominator)
{
  theobj->durinticks = theobj->basic_durinticks * numerator / denominator;
  /* Though WHOLENUMTICKS is chosen strategically so that in common
   * cases, this division works out evenly, things should also work
   * out properly if it doesn't; i.e., durinticks_untupletized is
   * rounded down */
}

void
set_grace_numticks (mudelaobject * theobj, gint multiplier)
{

  theobj->durinticks = theobj->basic_durinticks / multiplier;


}

void
set_basic_numticks (mudelaobject * theobj)
{
  gint power;
  gint withoutdots;
  gint addperdot, i;

  switch (theobj->type)
    {
    case CHORD:
      power = 1 << ((chord *)theobj->object)->baseduration;
      withoutdots = WHOLE_NUMTICKS / power;
      addperdot = withoutdots / 2;
      theobj->basic_durinticks = withoutdots;
      for (i = 0; i < ((chord *)theobj->object)->numdots; addperdot /= 2, i++)
	theobj->basic_durinticks += addperdot;
      break;
    default:
      theobj->basic_durinticks = 0;
      theobj->durinticks = 0;
      /* There's no reason not to set that as well */
      break;
    }
}

/* Returns the amount of space to be left after a note or rest, only
 * taking the width of the measure into consideration */

gint
space_after (gint numticks, gint wholenotewidth)
{
  return MAX (numticks * wholenotewidth / WHOLE_NUMTICKS, 0);
}

#define EXTRAWIDTH 5

/* Sets the minimum space that needs to be allocated for drawing a mudela
 * object */

void
setpixelmin (mudelaobject * theobj)
{
  gint i, baseduration, headtype;
  chord chordval;
  GList *tnode;
  note *thetone;
  /* And these static declaration are copied right out of drawnotes.c
   * and drawaccidentals.c */

  switch (theobj->type)
    {
    case CHORD:
      chordval = *(chord *)theobj->object;
      baseduration = chordval.baseduration;
      headtype = MIN (baseduration, 2);
      if (chordval.tones)
	{
	  /* if (theobj->isstart_beamgroup && theobj->isend_beamgroup &&
	     chordval.is_stemup)
	     theobj->minpixelsalloted = headwidths[headtype] + STEM_WIDTH;
	     else
	     theobj->minpixelsalloted = headwidths[headtype];  */
	  /* We can get away with that because the stems are narrower
	   * than even the narrowest notes; upstemmed notes are
	   * the unusual case here */

	  /* The above code will allow extra space for the stem of
	   * stemup notes. It's commented out 'cause we no longer want
	   * that behavior */
	  theobj->minpixelsalloted = headwidths[headtype];
	}
      else			/* a rest */
	theobj->minpixelsalloted = restwidths[baseduration];

      /* 12 pixels for the first dot, 6 for each dot thereafter */
      if (chordval.numdots)
	theobj->minpixelsalloted += 6;
      for (i = 0; i < chordval.numdots; i++)
	theobj->minpixelsalloted += 6;
      theobj->space_before = 0;
      if (chordval.hasanacc)
	for (tnode = chordval.tones; tnode; tnode = tnode->next)
	  {
	    thetone = (note *)tnode->data;
	    if (thetone->showaccidental)
	      theobj->space_before =
		MAX (theobj->space_before, thetone->position_of_accidental);
	  }
      if (chordval.is_reversealigned)
	if (chordval.is_stemup)
	  theobj->minpixelsalloted += headwidths[headtype];
	else if (!chordval.hasanacc)
	  /* Accidental positioning already accounts for the extra leading
	     space that we need for reverse-aligned noteheads, provided
	     the chord has an accidental in it somewhere. We only have to
	     remark upon noteheads to the left of the stem if there weren't
	     any accidentals to position.  */
	  theobj->space_before += headwidths[headtype];
      theobj->minpixelsalloted += EXTRAWIDTH;
      break;
    case TUPOPEN:
    case TUPCLOSE:
      /* The real way do this will be with a gdk_string_width. Until
       * then, though: */
      theobj->minpixelsalloted = 40;
      theobj->space_before = 0;
      break;
    case CLEF:
      theobj->minpixelsalloted = 35;
      theobj->space_before = 0;
      break;
    case KEYSIG:
      theobj->space_before = 0;
      break;
    case TIMESIG:
      theobj->minpixelsalloted = 40;
      theobj->space_before = 0;
      break;
    case STEMDIRECTIVE:
      /* The real way do this will be with a gdk_string_width. Until
       * then, though: */
      theobj->minpixelsalloted = 40;
      theobj->space_before = 0;
      break;
    case LILYDIRECTIVE:
      theobj->minpixelsalloted = 40;
      theobj->space_before = 0;
      break;
    case DYNAMIC:
      theobj->minpixelsalloted = 40;
      theobj->space_before = 0;
      break;
    case GRACE_START:
    case GRACE_END:
      theobj->minpixelsalloted = 40;
      theobj->space_before = 0;
      break;
    default:
      theobj->minpixelsalloted = 0;
      theobj->space_before = 0;
      break;
    }
}

/* Height of a tone based on its mid_c_offset and the clef that it's in */

gint calculateheight (gint mid_c_offset, gint dclef)
{
  switch (dclef)
    {
    case DENEMO_TREBLE_CLEF:
      return 5 * LINE_SPACE - HALF_LINE_SPACE * mid_c_offset;
      break;			/* Probably gratuitous */
    case DENEMO_ALTO_CLEF:
      return 2 * LINE_SPACE - HALF_LINE_SPACE * mid_c_offset;
      break;
    case DENEMO_G_8_CLEF:
      return LINE_SPACE - HALF_LINE_SPACE * (mid_c_offset - 1);
      break;
    case DENEMO_BASS_CLEF:
      return -LINE_SPACE - HALF_LINE_SPACE * mid_c_offset;
      break;
    case DENEMO_TENOR_CLEF:
      return LINE_SPACE - HALF_LINE_SPACE * mid_c_offset;
      break;
    case DENEMO_SOPRANO_CLEF:
      return LINE_SPACE - HALF_LINE_SPACE * (mid_c_offset - 6);
      break;
    }

  return (0);
}

gint offsettonumber (gint n)
{
  if (n >= 0)
    return n % 7;
  else
    return (7 - (-n % 7)) % 7;
  /* Not all C implementations conform to the more recent standard on how %
     should operate on negative operands.  */
}

gchar mid_c_offsettoname (gint mid_c_offset)
{
  int otn = offsettonumber (mid_c_offset);

  return ((otn + 2) % 7) + 'a';
}

gint mid_c_offsettooctave (gint mid_c_offset)
{
  if (mid_c_offset < 0)
    return -((-mid_c_offset + 6) / 7) + 1;
  else
    return (mid_c_offset / 7) + 1;
}

void
freeit (gpointer data, gpointer user_data)
{
  g_free (data);
}

/************* routines for calling from debug code ***************/
#include "staffops.h"

G_GNUC_UNUSED void 
printobj (objnode *obj)
{
  mudelaobject *curObj;

  curObj = (mudelaobject *)(obj->data);
  switch (curObj->type) {
  case CHORD:
    fprintf (stderr, "\t\t%s type\n","CHORD");
    break;
  case TUPOPEN:
    fprintf (stderr, "\t\t%s type\n","TUPOPEN");
    break;
  case TUPCLOSE:
    fprintf (stderr, "\t\t%s type\n","TUPCLOSE");
    break;
  case CLEF:
    fprintf (stderr, "\t\t%s type\n","CLEF");
    break;
  case TIMESIG:
    fprintf (stderr, "\t\t%s type\n","TIMESIG");
    break;
  case KEYSIG:
    fprintf (stderr, "\t\t%s type\n","KEYSIG");
    break;
  case BARLINE:
    fprintf (stderr, "\t\t%s type\n","BARLINE");
    break;
  case LILYDIRECTIVE:
    fprintf (stderr, "\t\t%s type\n","LILYDIRECTIVE");
    break;
  case COMMENT:
    fprintf (stderr, "\t\t%s type\n","COMMENT");
    break;
  case STEMDIRECTIVE:
    fprintf (stderr, "\t\t%s type\n","STEMDIRECTIVE");
    break;
  case MEASUREBREAK:
    fprintf (stderr, "\t\t%s type\n","MEASUREBREAK");
    break;
  case DYNAMIC:
    fprintf (stderr, "\t\t%s type\n","DYNAMIC");
    break;
  case GRACE_START:
    fprintf (stderr, "\t\t%s type\n","GRACE_START");
    break;
  case GRACE_END:
    fprintf (stderr, "\t\t%s type\n","GRACE_END");
    break;
  case LYRIC:
    fprintf (stderr, "\t\t%s type\n","LYRIC");
    break;
  case FIGURE:
    fprintf (stderr, "\t\t%s type\n","FIGURE");
    break;
  default: /* needs to be up to date with enum in include/denemo/denemo.h */
    fprintf (stderr, "!!!!!unknown object type %x - see enum in denemo.h\n", curObj->type);
    break;
  }
}

G_GNUC_UNUSED void 
printobjs (objnode *obj)
{
  objnode *curobj;
  if(obj == NULL) {
    fprintf(stderr, "NULL object\n");
    return;
  }    
  printobj (obj);
  fprintf(stderr, "previous objects\n");
  curobj = obj;
  while(curobj->prev) {
    printobj(curobj->prev);
    curobj = curobj->prev;
  }
  fprintf(stderr, "next objects\n");
  curobj = obj;
  while (curobj->next) {
    printobj(curobj->next);
    curobj = curobj->next;
  } 
}

G_GNUC_UNUSED void
printmeasure (measurenode * mnode)
{
  if(mnode == NULL) {
    fprintf(stderr, "Empty measure\n");
    return;
  }    
  printobjs (firstobjnode (mnode));
}
G_GNUC_UNUSED void
printmeasures (staffnode * thestaff)
{
  GList* measure = firstmeasurenode(thestaff);
  gint measurenum=1;
  for (measure = firstmeasurenode(thestaff); measure; measure=measure->next) {
    fprintf(stderr, "*************Measure %d *************\n", measurenum++);
    printmeasure(measure);
  } 
}
G_GNUC_UNUSED void
printscoreinfo (scoreinfo *si)
{
  if(si->thescore == NULL) {
    fprintf(stderr, "Staff with NULL thescore field\n");
    return;
  }  
  printmeasures (si->thescore);
}
