/* moveviewport.cpp
   functions that change leftmeasurenum, rightmeasurenum, top_measure,
   bottom_measure

   for Denemo, a gtk+ frontend to GNU Lilypond
   (c) 2000, 2001, 2002 Matthew Hiller
*/

#include "commandfuncs.h"
#include "contexts.h"
#include "moveviewport.h"
#include "staffops.h"
#include "utils.h"

/* update_hscrollbar should be called as a cleanup whenever
   si->leftmeasurenum or si->rightmeasurenum may have been altered,
   e.g., by preceding calls to set_rightmeasurenum; or when the
   number of measures may have changed.  */

void
update_hscrollbar (struct scoreinfo *si)
{
  GtkAdjustment *adj = GTK_ADJUSTMENT (si->hadjustment);

  adj->upper = g_list_length (si->measurewidths) + 1.0;
  adj->page_size = adj->page_increment
    = si->rightmeasurenum - si->leftmeasurenum + 1.0;
  adj->value = si->leftmeasurenum;
#if GTK_MAJOR_VERSION > 1
  /* what if anything needs be done here? */
#else
  gtk_range_slider_update (GTK_RANGE (si->hscrollbar));
#endif
}

/* update_vscrollbar should be called as a cleanup whenever
   si->top_staff or si->bottom_staff may have been altered,
   e.g., by preceding calls to set_bottom_staff; or when the number of
   staffs may have changed.

   For simplicity, this function treats nonprimary voices as
   full-fledged staffs, which'll be visually confusing. I'll fix it
   soon.  */

void
update_vscrollbar (struct scoreinfo *si)
{
  GtkAdjustment *adj = GTK_ADJUSTMENT (si->vadjustment);

  adj->upper = g_list_length (si->thescore) + 1.0;
  adj->page_size = adj->page_increment
    = si->bottom_staff - si->top_staff + 1.0;
  adj->value = si->top_staff;
#if GTK_MAJOR_VERSION > 1
  /* what if anything needs be done here? */
#else
  gtk_range_slider_update (GTK_RANGE (si->vscrollbar));
#endif
}

void
set_rightmeasurenum (struct scoreinfo *si)
{
  gint spaceleft = si->widthtoworkwith;
  GList *mwidthiterator =
    g_list_nth (si->measurewidths, si->leftmeasurenum - 1);

  for (si->rightmeasurenum = si->leftmeasurenum;
       mwidthiterator && spaceleft >= GPOINTER_TO_INT (mwidthiterator->data);
       spaceleft -=
       (GPOINTER_TO_INT (mwidthiterator->data) + SPACE_FOR_BARLINE),
       mwidthiterator = mwidthiterator->next, si->rightmeasurenum++)
    ;
  si->rightmeasurenum = MAX (si->rightmeasurenum - 1, si->leftmeasurenum);
}

/* Utility function for advancing a staff number and staff iterator to
   the next primary voice, or to one off the end and NULL if there are
   none remaining.  */

static void
to_next_primary_voice (gint * staff_number, staffnode ** staff_iterator)
{
  do
    {
      (*staff_number)++;
      *staff_iterator = (*staff_iterator)->next;
    }
  while (*staff_iterator
	 && ((staff *) (*staff_iterator)->data)->voicenumber == 2);
}

/* This function also has a side effect of bumping si->top_staff
   up to the staff number of the next primary voice if si->top_staff
   initially points to a nonprimary voice.  */

void
set_bottom_staff (struct scoreinfo *si)
{
  gint space_left;
  staffnode *staff_iterator;
  gint staff_number;

  /* Bump up si->top_staff, if necessary.  */
  staff_iterator = g_list_nth (si->thescore, si->top_staff - 1);
  if (((staff *) staff_iterator->data)->voicenumber == 2)
    to_next_primary_voice (&si->top_staff, &staff_iterator);

  /* With that settled, now determine how many additional (primary)
     staves will fit into the window.  */
  staff_number = si->top_staff;
  space_left = si->scorearea->allocation.height;
  do
    {
      space_left -= si->staffspace;
      to_next_primary_voice (&staff_number, &staff_iterator);
    }
  while (staff_iterator && space_left >= si->staffspace);

  si->bottom_staff = staff_number - 1;
}

void
isoffleftside (struct scoreinfo *si)
{
  while (si->currentmeasurenum < si->leftmeasurenum)
    {
      si->leftmeasurenum
	-= MAX ((si->rightmeasurenum - si->leftmeasurenum + 1) / 2, 1);
      if (si->leftmeasurenum < 1)
	si->leftmeasurenum = 1;
      set_rightmeasurenum (si);
    }
  find_leftmost_allcontexts (si);
  update_hscrollbar (si);
}

void
isoffrightside (struct scoreinfo *si)
{
  while (si->currentmeasurenum > si->rightmeasurenum)
    {
      si->leftmeasurenum
	+= MAX ((si->rightmeasurenum - si->leftmeasurenum + 1) / 2, 1);
      set_rightmeasurenum (si);
    }
  find_leftmost_allcontexts (si);
  update_hscrollbar (si);
}

void
move_viewport_up (struct scoreinfo *si)
{
  staffnode *staff_iterator;

  staff_iterator = g_list_nth (si->thescore, si->top_staff - 1);
  while (si->currentstaffnum < si->top_staff
	 || ((staff *) staff_iterator->data)->voicenumber == 2)
    {
      si->top_staff--;
      staff_iterator = staff_iterator->prev;
    }
  set_bottom_staff (si);
  update_vscrollbar (si);
}

void
move_viewport_down (struct scoreinfo *si)
{
  staffnode *staff_iterator;

  staff_iterator = g_list_nth (si->thescore, si->top_staff - 1);
  while (si->currentstaffnum > si->bottom_staff)
    {
      to_next_primary_voice (&si->top_staff, &staff_iterator);
      set_bottom_staff (si);
    }
  update_vscrollbar (si);
}

void
set_currentmeasurenum (struct scoreinfo *si, gint dest)
{
  if ((dest  > 0) && (dest <= gint(g_list_length (si->measurewidths))))
    {
      si->leftmeasurenum = dest;
      si->currentmeasurenum = dest;
      setcurrents (si);
      set_rightmeasurenum (si);
      find_leftmost_allcontexts (si);
      update_hscrollbar (si);
      gtk_widget_draw (si->scorearea, NULL);
    }
}

void
vertical_scroll (GtkAdjustment * adjust, struct scoreinfo *si)
{
  gint dest;

  if ((dest = (gint) (adjust->value + 0.5)) != si->top_staff)
    {
      si->top_staff = dest;
      set_bottom_staff (si);
      if (si->currentstaffnum > si->bottom_staff)
	{
	  si->currentstaffnum = si->bottom_staff;
	  si->currentstaff = g_list_nth (si->thescore, si->bottom_staff - 1);
	  setcurrentprimarystaff (si);
	  setcurrents (si);
	}
      else if (si->currentstaffnum < si->top_staff)
	{
	  si->currentstaffnum = si->top_staff;
	  si->currentstaff = g_list_nth (si->thescore, si->top_staff - 1);
	  setcurrentprimarystaff (si);
	  setcurrents (si);
	}
      gtk_widget_draw (si->scorearea, NULL);
    }
  update_vscrollbar (si);
}

void
horizontal_scroll (GtkAdjustment * adjust, struct scoreinfo *si)
{
  gint dest;

  if ((dest = (gint) (adjust->value + 0.5)) != si->leftmeasurenum)
    {
      si->leftmeasurenum = dest;
      set_rightmeasurenum (si);
      if (si->currentmeasurenum > si->rightmeasurenum)
	{
	  si->currentmeasurenum = si->rightmeasurenum;
	  setcurrents (si);
	}
      else if (si->currentmeasurenum < si->leftmeasurenum)
	{
	  si->currentmeasurenum = si->leftmeasurenum;
	  setcurrents (si);
	}
      find_leftmost_allcontexts (si);
      gtk_widget_draw (si->scorearea, NULL);
    }
  update_hscrollbar (si);
}
