/*	file.cpp
 * 	Denemo File IO 
 *
 * 	for Denemo, a gtk+ frontend to GNU Lilypond
 * 	(c) Adam Tee, Matthew Hiller 2000, 2001, 2002
 * 	(c) University of Leeds 2000, 2001, 2002
 */

#include "calculatepositions.h"
#include "contexts.h"
#include <denemo/denemo.h>
#include "dialogs.h"
#include "exportabc.h"
#include "exportmudela.h"
#include "file.h"
#include "frogio.h"
#include "frogdefs.h"
#include "moveviewport.h"
#include "staffops.h"
#include "scoreops.h"
#include "utils.h"
#include "exportxml.h"
#include "exportmidi.h"
#include "importxml.h"
#include "exportcsound.h"
#include "importmidi.h"
#include "lyparserfuncs.h"
#include "lyparser.h"
#include "prefops.h"

#include <stdio.h>
#include <string.h>
#include <sys/stat.h>          /* check existance and type of files */
#include <dirent.h>            /* filter and sort filenames */
#include <fnmatch.h>           /* find filenames matches */
#include <malloc.h>            /* I use free() */


typedef enum
{ DENEMO_FORMAT = 0,
  DNM_FORMAT,
  MUDELA_FORMAT,
  ABC_FORMAT,
  JTF_FORMAT,
  MIDI_FORMAT,
  JTF_POLY_FORMAT,
  CSOUND_FORMAT
}FileFormatNames;

/* Keep this up to date ! */

#define FIRST_FORMAT_NAME DENEMO_FORMAT
#define LAST_FORMAT_NAME JTF_POLY_FORMAT 

struct FileFormatData
{
  gchar *filename_mask;
  gchar *description;
  gchar *filename_extension;
};

static struct FileFormatData supported_file_formats[] = {
  {"*.denemo", N_("Denemo XML format (*.denemo)"), ".denemo"},
  {"*.dnm", N_("Denemo XML format (*.dnm)"), ".dnm"},
  {"*.ly", N_("Lilypond (formerly Mudela) (*.ly)"), ".ly"},
  {"*.abc", N_("ABC (*.abc)"), ".abc"},
  {"*.jtf", N_("Unnamed file format (*.jtf)"), ".jtf"},
  {"*.midi", N_("Midi (*.midi)"), ".midi"},
  {"*.jtf", N_("Unnamed file format (*.jtf) Poly"), ".jtf"},
  {"*.sco", N_("CSound Score File (*.sco)"), ".sco"}
};
/* Some macros just to shorten lines */
#define FORMAT_MASK(i) supported_file_formats[i].filename_mask
#define FORMAT_DESCRIPTION(i) supported_file_formats[i].description
#define FORMAT_EXTENSION(i) supported_file_formats[i].filename_extension

struct callbackdata
{
  struct scoreinfo *si;
  GtkWidget *fs;
  GtkWidget *comboentry;
};

static gchar *file_selection_path = NULL;

/* Prototypes for non-exported functions */
static gint guess_file_format (gchar * file_name);
static void update_file_selection_path (gchar * filename);
static void formats_list_item_selected (GtkList * list, GtkWidget * child,
				 gpointer user_data);
static void populate_file_list (GtkWidget * file_selection, gchar * the_pattern); 
static int not_a_dot_file (const struct dirent *a_directory_entry);
static void new_format_selected (GtkWidget * widget, gpointer data);
static void new_dir_selected (GtkWidget * widget,
		       gint row,
		       gint column,
		       GdkEventButton * bevent, gpointer user_data);
static void safe_file_save_as (GtkWidget * widget, gpointer data);
 





void
confirmbox (struct scoreinfo *si, GtkSignalFunc signalfunc)
{
  GtkWidget *dialog, *yes, *no, *label;
  dialog = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dialog), _("Confirm"));
  
  label = gtk_label_new
    (_("The current score has changes in it which you have not saved."));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label,
		      TRUE, TRUE, 0);
  gtk_widget_show (label);
  label = gtk_label_new (_("Proceed with the operation nonetheless?"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label,
		      TRUE, TRUE, 0);
  gtk_widget_show (label);
  yes = gtk_button_new_with_label (_("Yes"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area),
		      yes, TRUE, TRUE, 0);
  gtk_widget_show (yes);
  
  no = gtk_button_new_with_label (_("No"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area),
		      no, TRUE, TRUE, 0);
  gtk_widget_show (no);
 #if GTK_MAJOR_VERSION > 1 
  g_signal_connect (GTK_OBJECT (yes), "clicked",
		      G_CALLBACK (signalfunc), si);
  g_signal_connect_swapped (GTK_OBJECT (yes), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     (gpointer) dialog);
  g_signal_connect_swapped (GTK_OBJECT (no), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     (gpointer) dialog);
#else
  gtk_signal_connect (GTK_OBJECT (yes), "clicked",
		      GTK_SIGNAL_FUNC (signalfunc), si);
  gtk_signal_connect_object (GTK_OBJECT (yes), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (dialog));
  gtk_signal_connect_object (GTK_OBJECT (no), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (dialog));
#endif

  gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
  gtk_widget_show (dialog);
}

/* This function is to be called after a file load of any sort.  */

void
updatescoreinfo (struct scoreinfo *si)
{
  staffnode *curstaff;
  for (curstaff = si->thescore; curstaff; curstaff = curstaff->next)
    {
      beamsandstemdirswholestaff ((staff *)curstaff->data);
      showwhichaccidentalswholestaff ((staff *)curstaff->data);
    }
  find_xes_in_all_measures (si);
  find_leftmost_allcontexts (si);

  si->currentstaff = si->thescore;
  si->currentmeasure = firstmeasurenode (si->currentstaff);
  si->currentobject = firstobjnode (si->currentmeasure);
  if (!si->currentobject)
    si->cursor_appending = TRUE;
  else
    si->cursor_appending = FALSE;
  si->leftmeasurenum = si->currentstaffnum = si->currentmeasurenum = 1;
  si->cursor_x = 0;
  si->haschanged = FALSE;
  set_rightmeasurenum (si);
  set_bottom_staff (si);
  update_hscrollbar (si);
  update_vscrollbar (si);
  gtk_widget_draw (si->scorearea, NULL);
}

void
set_si_filename (struct scoreinfo *si, gchar * filename)
{
  gchar *dialog_title = g_strconcat ("Denemo - ", filename, NULL);;
  gtk_window_set_title (GTK_WINDOW (si->window), dialog_title);
  g_free (dialog_title);
  g_string_assign (si->filename, filename);
}

/* The function that actually determines the file type and calls the
   function that opens the file.  (So many layers of indirection...)  */

void
open_for_real (gchar * filename, struct scoreinfo *si)
{
  gint result;
  if(si->lily_file!=NULL) abandon_lily_tree(si);
  if (strcmp (filename + strlen (filename) - 7, ".denemo") == 0)
    result = importXML (filename, si);
  else if (strcmp (filename + strlen (filename) - 4, ".dnm") == 0)
    result = importXML (filename, si);
  else if (strcmp (filename + strlen (filename) - 3, ".ly") == 0)
    result = lyinput (filename, si);
  else if(strcmp (filename + strlen (filename) -4, ".mid") ==0)
	result = importMidi (filename,si);
  else if(strcmp (filename + strlen(filename) - 4 , ".jtf") == 0)
    result = froginput (filename, si);
  if (result != -1)
    {
      set_si_filename (si, filename);
      updatescoreinfo (si);
      gtk_signal_emit_by_name (GTK_OBJECT (si->hadjustment), "changed");
      gtk_signal_emit_by_name (GTK_OBJECT (si->vadjustment), "changed");
    }
}

/* A callback that acts as a wrapper to open_for_real */

void
filesel_open (GtkWidget * widget, gpointer data)
{
  struct callbackdata *cbdata = (struct callbackdata *)data;
  struct scoreinfo *si = cbdata->si;
  struct stat file_information;

  gchar *file_name =
    (gchar*)gtk_file_selection_get_filename (GTK_FILE_SELECTION (cbdata->fs));
  update_file_selection_path (file_name);
  if (!stat (file_name, &file_information))
    {
      /* ...and it's not a directory */
      if (!S_ISDIR (file_information.st_mode))
	open_for_real (file_name, si);
    };
  
}

/* A callback that actually does the saving based on what was entered
   in the file selection widget.  */

void
filesel_save (GtkWidget * widget, gpointer data)
{
  gint format_id;
  struct callbackdata *cbdata = (struct callbackdata *)data;
  struct scoreinfo *si = cbdata->si;
  gchar *file_name =
    (gchar*)gtk_file_selection_get_filename (GTK_FILE_SELECTION (cbdata->fs));
  format_id =
    GPOINTER_TO_INT (gtk_object_get_user_data
		     (GTK_OBJECT (cbdata->comboentry)));
  

  if (fnmatch (FORMAT_MASK (format_id), file_name, 0) == 0)
    file_name = g_strdup (file_name);
  else
    file_name = g_strconcat (file_name, FORMAT_EXTENSION (format_id), NULL);
  /* we don't want to save scores under ".denemo", ".jtf" and so on. do we? */
  if (*(strrchr (file_name, '/') + 1) != '.')
    {
      switch (format_id)
	{
	case DENEMO_FORMAT:
	case DNM_FORMAT:
	  {
	    exportXML (file_name, si, 0, 0);
	    break;
	  };
	case MUDELA_FORMAT:
	  {
	    exportmudela (file_name, si, 0, 0);
	    break;
	  };
	case ABC_FORMAT:
	  {
	    exportabc (file_name, si, 0, 0);
	    break;
	  };
	case JTF_FORMAT:
	  {
	    filesave (file_name, si, 0, 0, 0);
	    break;
	  };
	case MIDI_FORMAT:
	  {
	    exportmidi (file_name, si, 0, 0);
	    break;
	  };
	case JTF_POLY_FORMAT:
	  {
	    filesave (file_name, si, 0, 0, 1);
	    break;
	  };
	case CSOUND_FORMAT:
	  {
	    exportcsound (file_name, si, 0, 0);
	    break;
	  };
	default:
	  break;
	};
      si->haschanged = FALSE;
      set_si_filename (si, file_name);
      g_free (file_name);
      /* Beware of this: file_selection dialog is closed only if we've actually
       * saved a file. (Or if the user pressed Cancel, of course. */
      gtk_widget_destroy (cbdata->fs);
    }
  
}



void
file_openwrapper (gpointer callback_data,
		  guint callback_action, GtkWidget * widget)
{
  struct scoreinfo *si = (struct scoreinfo *)callback_data;
  if (si->haschanged)
    {
      confirmbox (si, GTK_SIGNAL_FUNC (file_open));
    }
  else
    {
      file_open (NULL, si);
    }
}

/* File open dialog - opened where appropriate */

void
file_open (GtkWidget * widget, gpointer callback_data)
{
  GtkWidget *file_selection;
  GtkWidget *label;
  GtkWidget *combobox;
  GtkWidget *item;
  static struct callbackdata cbdata;

  file_selection = gtk_file_selection_new (_("Open"));

  /* Open the last visited directory, if any. */
  if (file_selection_path != NULL)
    {
      gtk_file_selection_set_filename (GTK_FILE_SELECTION (file_selection),
				       file_selection_path);
    };

#if GTK_MAJOR_VERSION > 1 
   g_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (file_selection)->ok_button),
                     "clicked",
                     G_CALLBACK (filesel_open),
                     &cbdata);
   			   
   /* Ensure that the dialog box is destroyed when the user clicks a button. */
   
   g_signal_connect_swapped (GTK_OBJECT (GTK_FILE_SELECTION (file_selection)->ok_button),
                             "clicked",
                             G_CALLBACK (gtk_widget_destroy), 
                             (gpointer) file_selection); 

   g_signal_connect_swapped (GTK_OBJECT (GTK_FILE_SELECTION (file_selection)->cancel_button),
                             "clicked",
                             G_CALLBACK (gtk_widget_destroy),
                             (gpointer) file_selection); 
#else
   gtk_signal_connect (GTK_OBJECT
		      (GTK_FILE_SELECTION (file_selection)->ok_button),
		      "clicked", GTK_SIGNAL_FUNC(filesel_open), &cbdata);

  gtk_signal_connect_object (GTK_OBJECT
			     (GTK_FILE_SELECTION (file_selection)->ok_button),
			     "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
			     GTK_OBJECT(file_selection));

  gtk_signal_connect_object (GTK_OBJECT
			     (GTK_FILE_SELECTION (file_selection)->
			      cancel_button), "clicked", 
			     GTK_SIGNAL_FUNC(gtk_widget_destroy),
			     GTK_OBJECT(file_selection));

  gtk_signal_connect_after (GTK_OBJECT
			    (GTK_FILE_SELECTION (file_selection)->dir_list),
			    "select_row", GTK_SIGNAL_FUNC (new_dir_selected),
			    &cbdata);
#endif

  label = gtk_label_new (_("Select file format:"));
  gtk_box_pack_start (GTK_BOX
		      (GTK_FILE_SELECTION (file_selection)->main_vbox), label,
		      TRUE, TRUE, 0);
  gtk_widget_show (label);

  combobox = gtk_combo_new ();

  item = gtk_list_item_new_with_label (_(FORMAT_DESCRIPTION (DENEMO_FORMAT)));
  gtk_object_set_user_data
    (GTK_OBJECT (item), GINT_TO_POINTER ((int) DENEMO_FORMAT));
  gtk_widget_show (item);
  gtk_container_add (GTK_CONTAINER (GTK_COMBO (combobox)->list), item);
  item = gtk_list_item_new_with_label (_(FORMAT_DESCRIPTION (DNM_FORMAT)));
    gtk_object_set_user_data
          (GTK_OBJECT (item), GINT_TO_POINTER ((int) DNM_FORMAT));
      gtk_widget_show (item);
        gtk_container_add (GTK_CONTAINER (GTK_COMBO (combobox)->list), item);

  item = gtk_list_item_new_with_label (_(FORMAT_DESCRIPTION (MUDELA_FORMAT)));
  gtk_object_set_user_data
    (GTK_OBJECT (item), GINT_TO_POINTER ((int) MUDELA_FORMAT));
  gtk_widget_show (item);
  gtk_container_add (GTK_CONTAINER (GTK_COMBO (combobox)->list), item);

  item = gtk_list_item_new_with_label (_(FORMAT_DESCRIPTION (JTF_FORMAT)));
  gtk_object_set_user_data
    (GTK_OBJECT (item), GINT_TO_POINTER ((int) JTF_FORMAT));
  gtk_widget_show (item);
  gtk_container_add (GTK_CONTAINER (GTK_COMBO (combobox)->list), item);
  item = gtk_list_item_new_with_label (_(FORMAT_DESCRIPTION (MIDI_FORMAT)));
  gtk_object_set_user_data
    (GTK_OBJECT (item), GINT_TO_POINTER ((int) MIDI_FORMAT));
  gtk_widget_show (item);
  gtk_container_add (GTK_CONTAINER (GTK_COMBO (combobox)->list), item);

  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combobox)->entry),
		      _(FORMAT_DESCRIPTION (DENEMO_FORMAT)));

  gtk_object_set_user_data (GTK_OBJECT (GTK_COMBO (combobox)->entry),
			    GINT_TO_POINTER ((int) DENEMO_FORMAT));

  cbdata.si = (struct scoreinfo *)callback_data;
  cbdata.fs = file_selection;
  cbdata.comboentry = GTK_COMBO (combobox)->entry;

  gtk_editable_set_editable (GTK_EDITABLE (cbdata.comboentry), FALSE);
  gtk_box_pack_start (GTK_BOX
		      (GTK_FILE_SELECTION (file_selection)->main_vbox),
		      combobox, TRUE, TRUE, 0);
#if GTK_MAJOR_VERSION > 1
  g_signal_connect (GTK_OBJECT (GTK_COMBO (combobox)->entry), "changed",
		      G_CALLBACK (new_format_selected), &cbdata);
  g_signal_connect (GTK_OBJECT (GTK_COMBO (combobox)->list), "select_child",
		      G_CALLBACK (formats_list_item_selected), &cbdata);
#else
  gtk_signal_connect (GTK_OBJECT (GTK_COMBO (combobox)->entry), "changed",
		      GTK_SIGNAL_FUNC (new_format_selected), &cbdata);
  gtk_signal_connect (GTK_OBJECT (GTK_COMBO (combobox)->list), "select_child",
		      GTK_SIGNAL_FUNC (formats_list_item_selected), &cbdata);
#endif
  gtk_widget_show (combobox);
  gtk_window_set_modal (GTK_WINDOW (file_selection), TRUE);
  gtk_window_set_position (GTK_WINDOW (file_selection), GTK_WIN_POS_MOUSE);
  gtk_widget_show (file_selection);
  /* Force the initial file_list to show only those files matching the current pattern */

  gtk_signal_emit_by_name (GTK_OBJECT (GTK_COMBO (combobox)->entry),
			   "changed", &cbdata);

}

void
file_saveaswrapper (gpointer callback_data, guint callback_action,
		    GtkWidget * widget)
{
  file_saveas (widget, callback_data);
}

void
file_savewrapper (gpointer callback_data, guint callback_action,
		  GtkWidget * widget)
{
  file_save (widget, callback_data);
}

void
file_save (GtkWidget * widget, gpointer callback_data)
{
  struct scoreinfo *si = (struct scoreinfo *)callback_data;
  if (strcmp (si->filename->str, "") == 0)
    /* No filename's been given */
    file_saveas (widget, callback_data);
  else
    switch (guess_file_format (si->filename->str))
      {
      case DENEMO_FORMAT:
      case DNM_FORMAT:
	{
	  exportXML (si->filename->str, si, 0, 0);
	  break;
	};
      case MUDELA_FORMAT:
	{
	  exportmudela (si->filename->str, si, 0, 0);
	  break;
	};
      case ABC_FORMAT:
	{
	  exportabc (si->filename->str, si, 0, 0);
	  break;
	};
      case JTF_POLY_FORMAT:
	{
	  filesave (si->filename->str, si, 0, 0, 1);
	  break;
	};
      case JTF_FORMAT:
	{
	  filesave (si->filename->str, si, 0, 0, 0);
	  break;
	};
      case MIDI_FORMAT:
	{
	  exportmidi (si->filename->str, si, 0, 0);
	  break;
	};
      case CSOUND_FORMAT:
	{
	  exportcsound (si->filename->str, si, 0, 0);
	  break;
	};
	
      default:
	{
	  exportXML (si->filename->str, si, 0, 0);
	  break;
	};
      };

  si->haschanged = FALSE;
}

void
file_saveas (GtkWidget * widget, gpointer callback_data)
{
  GtkWidget *file_selection;
  GtkWidget *label;
  GtkWidget *item;
  GtkWidget *combobox;
  static struct callbackdata cbdata;

  file_selection = gtk_file_selection_new (_("Save As"));

  /* Open the last visited directory, if any. */
  if (file_selection_path != NULL)
    {
      gtk_file_selection_set_filename (GTK_FILE_SELECTION (file_selection),
				       file_selection_path);
    };

  gtk_window_set_modal (GTK_WINDOW (file_selection), TRUE);
#if GTK_MAJOR_VERSION > 1
  g_signal_connect (GTK_OBJECT
		      (GTK_FILE_SELECTION (file_selection)->ok_button),
		      "clicked", G_CALLBACK (safe_file_save_as),
		      &cbdata);
  g_signal_connect_swapped (GTK_OBJECT (GTK_FILE_SELECTION (file_selection)->cancel_button),
                             "clicked",
                             G_CALLBACK (gtk_widget_destroy),
                            (gpointer) file_selection);
#else
  gtk_signal_connect (GTK_OBJECT
		      (GTK_FILE_SELECTION (file_selection)->ok_button),
		      "clicked", GTK_SIGNAL_FUNC (safe_file_save_as),
		      &cbdata);

  /* Beware we don't connect a callback to destroy the file_selection dialog
   * when the user presses 'Ok'. Instead, we destroy this widget within
   * filesave_sel, and only when the file's been actually saved. Thus, this
   * dialog remains opened until a file is (over)written or the user cancels
   * the action */

  gtk_signal_connect_object (GTK_OBJECT
			     (GTK_FILE_SELECTION (file_selection)->
			      cancel_button), "clicked", 
			     GTK_SIGNAL_FUNC(gtk_widget_destroy),
			     GTK_OBJECT(file_selection));

  gtk_signal_connect_after (GTK_OBJECT
			    (GTK_FILE_SELECTION (file_selection)->dir_list),
			    "select_row", GTK_SIGNAL_FUNC (new_dir_selected),
			    &cbdata);
#endif

  label = gtk_label_new (_("Select file format:"));
  gtk_box_pack_start (GTK_BOX
		      (GTK_FILE_SELECTION (file_selection)->main_vbox), label,
		      TRUE, TRUE, 0);
  gtk_widget_show (label);

  combobox = gtk_combo_new ();
  if( ((struct scoreinfo *)callback_data)->lily_file) {
    item = gtk_list_item_new_with_label (_(FORMAT_DESCRIPTION (MUDELA_FORMAT)));
    gtk_object_set_user_data (GTK_OBJECT (item),
			    GINT_TO_POINTER ((int) MUDELA_FORMAT));
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (GTK_COMBO (combobox)->list), item);
    gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combobox)->entry),
		      _(FORMAT_DESCRIPTION (MUDELA_FORMAT)));
    
    gtk_object_set_user_data (GTK_OBJECT (GTK_COMBO (combobox)->entry),
			    GINT_TO_POINTER ((int) MUDELA_FORMAT));
  } else {

  item = gtk_list_item_new_with_label (_(FORMAT_DESCRIPTION (DENEMO_FORMAT)));
  gtk_object_set_user_data (GTK_OBJECT (item),
			    GINT_TO_POINTER ((int) DENEMO_FORMAT));
  gtk_widget_show (item);
  gtk_container_add (GTK_CONTAINER (GTK_COMBO (combobox)->list), item);

   item = gtk_list_item_new_with_label (_(FORMAT_DESCRIPTION (DNM_FORMAT)));
     gtk_object_set_user_data (GTK_OBJECT (item),
	                             GINT_TO_POINTER ((int) DNM_FORMAT));
       gtk_widget_show (item);
         gtk_container_add (GTK_CONTAINER (GTK_COMBO (combobox)->list), item);
	 
  item = gtk_list_item_new_with_label (_(FORMAT_DESCRIPTION (MUDELA_FORMAT)));
  gtk_object_set_user_data (GTK_OBJECT (item),
			    GINT_TO_POINTER ((int) MUDELA_FORMAT));
  gtk_widget_show (item);
  gtk_container_add (GTK_CONTAINER (GTK_COMBO (combobox)->list), item);

  item = gtk_list_item_new_with_label (_(FORMAT_DESCRIPTION (ABC_FORMAT)));
  gtk_object_set_user_data (GTK_OBJECT (item),
			    GINT_TO_POINTER ((int) ABC_FORMAT));
  gtk_widget_show (item);
  gtk_container_add (GTK_CONTAINER (GTK_COMBO (combobox)->list), item);

  item = gtk_list_item_new_with_label (_(FORMAT_DESCRIPTION (JTF_FORMAT)));
  gtk_object_set_user_data (GTK_OBJECT (item),
			    GINT_TO_POINTER ((int) JTF_FORMAT));
  gtk_widget_show (item);
  gtk_container_add (GTK_CONTAINER (GTK_COMBO (combobox)->list), item);

  item =
    gtk_list_item_new_with_label (_(FORMAT_DESCRIPTION (JTF_POLY_FORMAT)));
  gtk_object_set_user_data (GTK_OBJECT (item),
			    GINT_TO_POINTER ((int) JTF_POLY_FORMAT));
  gtk_widget_show (item);
  gtk_container_add (GTK_CONTAINER (GTK_COMBO (combobox)->list), item);

  item = gtk_list_item_new_with_label (_(FORMAT_DESCRIPTION (MIDI_FORMAT)));
  gtk_object_set_user_data (GTK_OBJECT (item),
			    GINT_TO_POINTER ((int) MIDI_FORMAT));
  gtk_widget_show (item);
  gtk_container_add (GTK_CONTAINER (GTK_COMBO (combobox)->list), item);
  item = gtk_list_item_new_with_label (_(FORMAT_DESCRIPTION (CSOUND_FORMAT)));
  gtk_object_set_user_data (GTK_OBJECT (item),
			    GINT_TO_POINTER ((int) CSOUND_FORMAT));
  gtk_widget_show (item);
  gtk_container_add (GTK_CONTAINER (GTK_COMBO (combobox)->list), item);


  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combobox)->entry),
		      _(FORMAT_DESCRIPTION (DENEMO_FORMAT)));

  gtk_object_set_user_data (GTK_OBJECT (GTK_COMBO (combobox)->entry),
			    GINT_TO_POINTER ((int) DENEMO_FORMAT));

  }
  cbdata.si = (struct scoreinfo *)callback_data;
  cbdata.fs = file_selection;
  cbdata.comboentry = GTK_COMBO (combobox)->entry;


  gtk_editable_set_editable (GTK_EDITABLE (cbdata.comboentry), FALSE);
  gtk_box_pack_start (GTK_BOX
		      (GTK_FILE_SELECTION (file_selection)->main_vbox),
		      combobox, TRUE, TRUE, 0);
#if GTK_MAJOR_VERSION > 1
  g_signal_connect (GTK_OBJECT (GTK_COMBO (combobox)->entry), "changed",
		      G_CALLBACK (new_format_selected), &cbdata);
  g_signal_connect (GTK_OBJECT (GTK_COMBO (combobox)->list), "select_child",
		      G_CALLBACK (formats_list_item_selected), &cbdata);
#else
  gtk_signal_connect (GTK_OBJECT (GTK_COMBO (combobox)->entry), "changed",
		      GTK_SIGNAL_FUNC (new_format_selected), &cbdata);
  gtk_signal_connect (GTK_OBJECT (GTK_COMBO (combobox)->list), "select_child",
		      GTK_SIGNAL_FUNC (formats_list_item_selected), &cbdata);
#endif
  gtk_widget_show (combobox);
  gtk_window_set_position (GTK_WINDOW (file_selection), GTK_WIN_POS_MOUSE);
  gtk_widget_show (file_selection);
  /* Force the intial file_list to show only those files matching the current pattern */

  gtk_signal_emit_by_name (GTK_OBJECT (GTK_COMBO (combobox)->entry),
			   "changed", file_selection);

}

void
file_newwrapper (gpointer callback_data,
		 guint callback_action, GtkWidget * widget)
{
  struct scoreinfo *si = (struct scoreinfo *)callback_data;
  if (si->haschanged)
    {
      confirmbox (si, GTK_SIGNAL_FUNC (deletescore));
    }
  else
    {
      deletescore (NULL, (struct scoreinfo *)callback_data);
    };
}

void
deletescore (GtkWidget * widget, struct scoreinfo *si)
{
  free_score (si);
  newstaff (si, INITIAL, NULL);
  set_rightmeasurenum (si);
  update_hscrollbar (si);
  update_vscrollbar (si);
  gtk_widget_draw (si->scorearea, NULL);
}

/* Stores the last visited directory, so we can go right there if the
 * File Selection dialog is opened again */
void
update_file_selection_path (gchar * filename)
{
  gint path_size;

  g_free (file_selection_path);	/* free previously allocated memory. */

  path_size =
    ((strrchr (filename, G_DIR_SEPARATOR)) -  filename + 2);
  file_selection_path = g_new (gchar, path_size);
  memset (file_selection_path, '\0', path_size);	/*ensure null-terminated string */
  memcpy ((void *) file_selection_path, filename, path_size - 1);
}


/* We don't want to show 'dot-files' (hidden) in our file_list. do we? */
int
not_a_dot_file (const struct dirent *a_directory_entry)
{
  return (a_directory_entry->d_name[0] != '.');
}


/* Populates the file_list with files whose names match the given pattern */
void
populate_file_list (GtkWidget * file_selection, gchar * the_pattern)
{
  struct dirent **directory_entries;
  struct stat file_information;
  GtkCList *the_file_list;
  gchar *the_directory;
  gchar *full_path_name;
  int files_count;
  int entry;
  gchar *file_entry_name[1];	/* to be compatible with gtk_clist_append */

#if GTK_MAJOR_VERSION > 1
  return; /* needs porting to GTK-2.0 */
#else

  the_file_list = GTK_CLIST (GTK_FILE_SELECTION (file_selection)->file_list);
  gtk_clist_clear (the_file_list);
  the_directory =
    (gchar*)gtk_file_selection_get_filename (GTK_FILE_SELECTION (file_selection));

  files_count =
    scandir (the_directory, &directory_entries, not_a_dot_file,
	     alphasort);
  gtk_clist_freeze (the_file_list);
  for (entry = 0; entry < files_count; ++entry)
    {
      file_entry_name[0] = directory_entries[entry]->d_name;
      full_path_name =
	g_strjoin ("/", the_directory, file_entry_name[0], NULL);

      /* file exists, that should be obvious but... */
      if (!stat (full_path_name, &file_information))
	{
	  /* file is not a directory... */
	  if (!S_ISDIR (file_information.st_mode))
	    {
	      /* ...and matches the pattern */
	      if (fnmatch (the_pattern, file_entry_name[0], 0) == 0)
		{
		  gtk_clist_append (the_file_list, file_entry_name);
		}
	    };
	};
      /* don't use g_free, cause that's allocated with malloc () */
      free (directory_entries[entry]);
      g_free (full_path_name);
    };
  gtk_clist_thaw (the_file_list);
#endif
}

/* Whenever the user navigates through the list of our combobox,
 * this callback updates the format_id attached to the entry widget
 * so that it's always correct */
void
formats_list_item_selected (GtkList * list, GtkWidget * child,
			    gpointer user_data)
{
  struct callbackdata *cbdata;
  GtkWidget *combobox_entry;
  gpointer format_id;

  cbdata = (struct callbackdata *) user_data;
  combobox_entry = GTK_WIDGET (cbdata->comboentry);
  format_id = gtk_object_get_user_data (GTK_OBJECT (child));
  gtk_object_set_user_data (GTK_OBJECT (combobox_entry), format_id);
}


/* When the users changes the desired format in the format's entry
 * of our combobox this callback is called to update the file_list
 * so that it shows file names matching with the new pattern selected */
void
new_format_selected (GtkWidget * widget, gpointer data)
{
  struct callbackdata *cbdata;
  GtkWidget *file_selection;
  gint format_id;

  cbdata = (struct callbackdata *) data;
  file_selection = cbdata->fs;
  format_id =
    GPOINTER_TO_INT (gtk_object_get_user_data (GTK_OBJECT (widget)));
  populate_file_list (file_selection, FORMAT_MASK (format_id));
}


/* When the user double-clicks a directory in the dir_list, this callback is
 * called. It just populates the file_list with those files in the new directory
 * matching the pattern selected in our format's list entry. For directories having
 * many files inside, the file_list 'flicks'. That's because our callback is
 * called after the default one, which populates the list with all the files
 * in the directory. This can be avoided by disconnecting the default callback
 * and writing it again from scracth... */
void
new_dir_selected (GtkWidget * widget, gint row, gint column,
		  GdkEventButton * bevent, gpointer user_data)
{
  struct callbackdata *cbdata;
  GtkWidget *file_selection;
  GtkWidget *combobox_entry;
  gint format_id;

  if (bevent)
    {
      if (bevent->type == GDK_2BUTTON_PRESS)
	{
	  cbdata = (struct callbackdata *) user_data;
	  file_selection = GTK_WIDGET (cbdata->fs);
	  gtk_clist_freeze (GTK_CLIST
			    (GTK_FILE_SELECTION (file_selection)->file_list));
	  combobox_entry = GTK_WIDGET (cbdata->comboentry);
	  format_id =
	    GPOINTER_TO_INT (gtk_object_get_user_data
			     (GTK_OBJECT (combobox_entry)));
	  populate_file_list (file_selection, FORMAT_MASK (format_id));
	  gtk_clist_thaw (GTK_CLIST
			  (GTK_FILE_SELECTION (file_selection)->file_list));
	};
    };
};

/* Try to suggest the format of a given file, after its file name extension. A
 * more powerful function could be written to guess the format after the
 * file contents */
gint guess_file_format (gchar * file_name)
{
  gint name_iterator;
  gboolean format_match;

  name_iterator = FIRST_FORMAT_NAME;
  format_match = FALSE;

  while (!format_match && name_iterator <= LAST_FORMAT_NAME)
    {
      format_match =
	(fnmatch (FORMAT_MASK (name_iterator++), file_name, 0) == 0);
    };

  /* In case no match could be found, we just give a 'default' format.
   * Chances are that all formats will be wrong, however ;-) */
  if (!format_match)
    return (DENEMO_FORMAT);
  else
    return (--name_iterator);
};

void
safe_file_save_as (GtkWidget * widget, gpointer data)
{
  struct stat file_information;
  GtkWidget *dialog;
  GtkWidget *ok_button;
  GtkWidget *cancel_button;
  GtkWidget *label;
  struct callbackdata *cbdata;
  gint format_id;
  gchar *file_name;

  cbdata = (struct callbackdata *) data;
  file_name =
    (gchar*)gtk_file_selection_get_filename (GTK_FILE_SELECTION (cbdata->fs));
  update_file_selection_path (file_name);
  format_id = GPOINTER_TO_INT (gtk_object_get_user_data
			       (GTK_OBJECT (cbdata->comboentry)));

  /* Prevent names with double extensions: whatever.denemo.denemo */
  if (fnmatch (FORMAT_MASK (format_id), file_name, 0) == 0)
    file_name = g_strdup (file_name);
  else
    file_name = g_strconcat (file_name, FORMAT_EXTENSION (format_id), NULL);

  if (!stat (file_name, &file_information))
    {
      /* if we arrive in here, the suggested file_name already exists, if
       * it's a directory we just do nothing (that's should never happen)
       * otherwise, we ask the user */
      if (!S_ISDIR (file_information.st_mode))
	{
	  dialog = gtk_dialog_new ();
	  gtk_window_set_title (GTK_WINDOW (dialog),
				_("Confirm file replacement"));

	  label =
	    gtk_label_new (_
			   ("There already exists a file by that name at the chosen location"));
	  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label,
			      TRUE, TRUE, 0);
	  gtk_widget_show (label);

	  label =
	    gtk_label_new (_
			   ("and it will be overwritten. Proceed nonetheless?"));
	  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label,
			      TRUE, TRUE, 0);
	  gtk_widget_show (label);

	  ok_button = gtk_button_new_with_label (_("Ok"));
	  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area),
			      ok_button, TRUE, TRUE, 0);
	  gtk_widget_show (ok_button);

	  cancel_button = gtk_button_new_with_label (_("Cancel"));
	  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area),
			      cancel_button, TRUE, TRUE, 0);
	  gtk_widget_show (cancel_button);
#if GTK_MAJOR_VERSION > 1
	  g_signal_connect (GTK_OBJECT (ok_button), "clicked",
			    G_CALLBACK (filesel_save), data);
	  g_signal_connect_swapped (GTK_OBJECT (ok_button),
				    "clicked",
				    G_CALLBACK (gtk_widget_destroy),
				    (gpointer) dialog); 
	  g_signal_connect_swapped (GTK_OBJECT (cancel_button),
				    "clicked",
				    G_CALLBACK (gtk_widget_destroy),
				    (gpointer) dialog); 
#else
	  gtk_signal_connect (GTK_OBJECT (ok_button), "clicked",
			      GTK_SIGNAL_FUNC (filesel_save), data);
	  gtk_signal_connect_object (GTK_OBJECT (ok_button), "clicked",
				     GTK_SIGNAL_FUNC (gtk_widget_destroy),
				     GTK_OBJECT (dialog));
	  gtk_signal_connect_object (GTK_OBJECT (cancel_button), "clicked",
				     GTK_SIGNAL_FUNC (gtk_widget_destroy),
				     GTK_OBJECT (dialog));
#endif
	  gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
	  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
	  gtk_widget_show (dialog);
	}
    }
  else
    {
      /* Suggested file didn't exist, just save it */
      filesel_save (cbdata->fs, data);
    };
}



void
reload_lily_file (GtkWidget * button, gpointer data)
{
  scoreinfo *si = (scoreinfo *) data;
  GString *filename;
  GString *realfilename;
  gint target_measure;
  if (si->lily_file)
    {
      if (!GTK_WIDGET_IS_SENSITIVE (si->scorearea))
	{
	  gtk_widget_set_sensitive (si->textview, FALSE);
	  gtk_widget_set_sensitive (si->scorearea, TRUE);
	}
      target_measure = si->leftmeasurenum;
      realfilename = g_string_new (si->filename->str);
      filename = g_string_new (locatedotdenemo ());
      filename = g_string_append (filename, "/reloadfile.ly");
      exportmudela (filename->str, si, 0, 0);
      open_for_real (filename->str, si);
      set_si_filename (si, realfilename->str);
      si->haschanged = TRUE;
      if (si->lily_file && (ntype (si->lily_file) == PARSE_ERROR))
	return;
      set_currentmeasurenum (si, target_measure);
      gtk_widget_show_all (si->window);	/* may have been hidden when lily file was unparseable */
      gtk_widget_hide (si->musicdatabutton);	/* only shown when on DENEMO_MEASURES */
    }
}
