# -*- coding: utf-8 -*-
#
# Author: Ingelrest François (Francois.Ingelrest@gmail.com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import cgi, gtk, gui.window, media, modules, os, tools, urllib

from gui     import fileChooser, help, extTreeview, extListview, selectPath
from tools   import consts, prefs
from media   import playlist
from gettext import gettext as _
from os.path import isdir, isfile
from gobject import idle_add, TYPE_STRING, TYPE_INT

MOD_INFO = ('File Explorer', _('File Explorer'), _('Browse your file system'), [], True, True)
MOD_L10N = MOD_INFO[modules.MODINFO_L10N]

# Default preferences
PREFS_DEFAULT_MEDIA_FOLDERS     = {_('Home'): consts.dirBaseUsr}    # List of media folders that are used as roots for the file explorer
PREFS_DEFAULT_SHOW_HIDDEN_FILES = False                             # True if hidden files should be shown


# The format of a row in the treeview
(
    ROW_PIXBUF,    # Item icon
    ROW_NAME,      # Item name
    ROW_TYPE,      # The type of the item (e.g., directory, file)
    ROW_FULLPATH   # The full path to the item
) = range(4)


# The possible types for a node of the tree
(
    TYPE_DIR,   # A directory
    TYPE_PLIST, # A playlist
    TYPE_FILE,  # A media file
    TYPE_NONE   # A fake item, used to display a '+' in front of a directory when needed
) = range(4)


class FileExplorer(modules.Module):
    """ This explorer lets the user browse the disk from a given root directory (e.g., ~/, /) """

    def __init__(self):
        """ Constructor """
        modules.Module.__init__(self, (consts.MSG_EVT_APP_STARTED, consts.MSG_EVT_EXPLORER_CHANGED, consts.MSG_EVT_APP_QUIT))


    def onModLoaded(self):
        """ The module has been loaded """
        self.trees           = {}     # A [tree, state] per root folder
        self.popup           = None
        self.cfgWin          = None
        self.folders         = prefs.get(__name__, 'media-folders', PREFS_DEFAULT_MEDIA_FOLDERS)
        self.scrolled        = gtk.ScrolledWindow()
        self.currFolder      = None
        self.showHiddenFiles = prefs.get(__name__, 'show-hidden-files', PREFS_DEFAULT_SHOW_HIDDEN_FILES)
        # Explorer
        self.scrolled.set_shadow_type(gtk.SHADOW_IN)
        self.scrolled.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        self.scrolled.show()

        for name in self.folders:
            modules.postMsg(consts.MSG_CMD_EXPLORER_ADD, {'modName': MOD_L10N, 'expName': name, 'icon': None, 'widget': self.scrolled})


    def createTree(self, rootFolder):
        """ Return a new tree for the given root folder """
        columns = (('',   [(gtk.CellRendererPixbuf(), gtk.gdk.Pixbuf), (gtk.CellRendererText(), TYPE_STRING)], True),
                   (None, [(None, TYPE_INT)],                                                                  False),
                   (None, [(None, TYPE_STRING)],                                                               False))

        tree = extTreeview.ExtTreeView(columns, True)
        tree.setIsDraggableFunc(self.isPlayable)
        tree.setDNDSources([consts.DND_TARGETS[consts.DND_DAP_URI]])
        # GTK handlers
        tree.connect('drag-data-get',              self.onDragDataGet)
        tree.connect('key-press-event',            self.onKeyPressed)
        tree.connect('exttreeview-row-expanded',   self.onRowExpanded)
        tree.connect('exttreeview-row-collapsed',  self.onRowCollapsed)
        tree.connect('exttreeview-button-pressed', self.onMouseButton)

        return tree


    def onModUnloaded(self):
        """ The module is going to be unloaded """
        prefs.set(__name__, 'media-folders',     self.folders)
        prefs.set(__name__, 'show-hidden-files', self.showHiddenFiles)


    def sortKey(self, row):
        """ Key function used to compare two rows of the tree """
        return row[ROW_NAME].lower()


    def isPlayable(self):
        """ Only playable rows can be dragged """
        for row in self.trees[self.currFolder][0].iterSelectedRows():
            if row[ROW_TYPE] in (TYPE_FILE, TYPE_PLIST) or (row[ROW_TYPE] == TYPE_DIR and row[ROW_PIXBUF] == consts.icoMediaDir):
                return True
        return False


    def setShowHiddenFiles(self, showHiddenFiles):
        """ Show/hide hidden files """
        if showHiddenFiles != self.showHiddenFiles:
            # Update the configuration window if needed
            if self.cfgWin is not None and self.cfgWin.isVisible():
                self.cfgWin.getWidget('chk-hidden').set_active(showHiddenFiles)

            self.showHiddenFiles = showHiddenFiles
            self.refresh(self.trees[self.currFolder][0])


    def getFilesFromPaths(self, tree, paths):
        """
            Return a list of playable files from:
                * The list 'paths' if it is not None
                * The selected rows if paths is None
        """
        if paths is None: return [row[ROW_FULLPATH] for row in tree.getSelectedRows() if row[ROW_TYPE] != TYPE_NONE]
        else:             return [row[ROW_FULLPATH] for row in tree.getRows(paths) if row[ROW_TYPE] != TYPE_NONE]


    def playPaths(self, tree, paths, replace):
        """
            Replace/extend the tracklist
            If the list 'paths' is None, use the current selection
        """
        if self.isPlayable():
            tracks = media.getTracks(self.getFilesFromPaths(tree, paths))
            if replace: modules.postMsg(consts.MSG_CMD_TRACKLIST_SET, {'tracks': tracks, 'playNow': True})
            else:       modules.postMsg(consts.MSG_CMD_TRACKLIST_ADD, {'tracks': tracks})


    def renameFolder(self, oldName, newName):
        """ Rename a folder """
        # Remove immediately the tree from the scrolled window if needed
        # We can't do that later, because we have no longer access to the old name after that
        if self.currFolder == oldName:
            self.scrolled.remove(self.trees[oldName][0])
            self.currFolder = None

        # Rename the root folder
        self.folders[newName] = self.folders[oldName]
        del self.folders[oldName]

        # Move the tree associated to that folder, if any
        if oldName in self.trees:
            self.trees[newName] = self.trees[oldName]
            del self.trees[oldName]

        modules.postMsg(consts.MSG_CMD_EXPLORER_REMOVE, {'modName': MOD_L10N, 'expName': oldName})
        modules.postMsg(consts.MSG_CMD_EXPLORER_ADD,    {'modName': MOD_L10N, 'expName': newName, 'icon': None, 'widget': self.scrolled})


    # --== Tree management ==--


    def __startLoading(self, tree, row):
        """ Tell the user that the contents of row is being loaded """
        name = tree.getItem(row, ROW_NAME)
        # TODO Put this style somewhere, we're using it in multiple places
        tree.setItem(row, ROW_NAME, '%s  <span size="smaller" foreground="#909090">%s</span>' % (name, _('loading...')))


    def __stopLoading(self, tree, row):
        """ Tell the user that the contents of row has been loaded"""
        name  = tree.getItem(row, ROW_NAME)
        index = name.find('<')

        if index != -1:
            tree.setItem(row, ROW_NAME, name[:index-2])


    def getDirContents(self, directory):
        """ Return a tuple of sorted rows (directories, playlists, mediaFiles) for the given directory """
        playlists   = []
        mediaFiles  = []
        directories = []

        for (file, path) in tools.listDir(directory, self.showHiddenFiles):
            if isdir(path):
                directories.append((consts.icoDir, cgi.escape(unicode(file, errors='replace')), TYPE_DIR, path))
            elif isfile(path):
                if media.isSupported(file):
                    mediaFiles.append((consts.icoMediaFile, cgi.escape(unicode(file, errors='replace')), TYPE_FILE, path))
                elif playlist.isSupported(file):
                    playlists.append((consts.icoMediaFile, cgi.escape(unicode(file, errors='replace')), TYPE_PLIST, path))

        playlists.sort(key=self.sortKey)
        mediaFiles.sort(key=self.sortKey)
        directories.sort(key=self.sortKey)

        return (directories, playlists, mediaFiles)


    def exploreDir(self, tree, parent, directory, fakeChild=None):
        """
            List the contents of the given directory and append it to the tree as a child of parent
            If fakeChild is not None, remove it when the real contents has been loaded
        """
        directories, playlists, mediaFiles = self.getDirContents(directory)

        tree.appendRows(directories, parent)
        tree.appendRows(playlists,   parent)
        tree.appendRows(mediaFiles,  parent)

        if fakeChild is not None:
            tree.removeRow(fakeChild)

        idle_add(self.updateChildren(tree, parent).next)


    def updateChildren(self, tree, parent):
        """ This generator updates (e.g., icon, fake child for directories) all children of the given parent """
        for child in tree.iterChildren(parent):
            # Only directories need to be updated and since they are always on top, we can stop if we find something else
            if tree.getItem(child, ROW_TYPE) != TYPE_DIR:
                break

            # Make sure it's readable
            directory = tree.getItem(child, ROW_FULLPATH)
            if not os.access(directory, os.R_OK | os.X_OK):
                continue

            hasContent      = False
            hasMediaContent = False
            for (file, path) in tools.listDir(directory, self.showHiddenFiles):
                if isdir(path):
                    hasContent = True
                elif isfile(path) and (media.isSupported(file) or playlist.isSupported(file)):
                    hasContent      = True
                    hasMediaContent = True
                    break

            # Append/remove children if needed
            if hasContent and tree.getNbChildren(child) == 0:      tree.appendRow((consts.icoDir, '', TYPE_NONE, ''), child)
            elif not hasContent and tree.getNbChildren(child) > 0: tree.removeAllChildren(child)

            # Change the icon based on whether the folder contains some media content
            if hasMediaContent: tree.setItem(child, ROW_PIXBUF, consts.icoMediaDir)
            else:               tree.setItem(child, ROW_PIXBUF, consts.icoDir)

            yield True


        if parent is not None:
            self.__stopLoading(tree, parent)

        yield False


    def refresh(self, tree, treePath=None):
        """ Refresh the tree, starting from treePath """
        if treePath is None: directory = self.folders[self.currFolder]
        else:                directory = tree.getItem(treePath, ROW_FULLPATH)

        directories, playlists, mediaFiles = self.getDirContents(directory)

        disk       = directories + playlists + mediaFiles
        diskIndex  = 0
        childIndex = 0
        while diskIndex < len(disk):
            file    = disk[diskIndex]
            rowPath = tree.getChild(treePath, childIndex)

            # We've reached the end of the tree, append the file and switch to the next one
            if rowPath is None:
                tree.appendRow(file, treePath)
                continue

            cmpResult = cmp(self.sortKey(tree.getRow(rowPath)), self.sortKey(file))
            if cmpResult < 0:
                tree.removeRow(rowPath)
            else:
                if cmpResult > 0:
                    tree.insertRowBefore(file, treePath, rowPath)
                diskIndex  += 1
                childIndex += 1

        # If there are tree rows left, all the corresponding files are no longer there
        while childIndex < tree.getNbChildren(treePath):
            tree.removeRow(tree.getChild(treePath, childIndex))

        # Update nodes' appearance
        if len(directories) != 0:
            idle_add(self.updateChildren(tree, treePath).next)

        # Recursively refresh expanded rows
        for child in tree.iterChildren(treePath):
            if tree.row_expanded(child):
                idle_add(self.refresh, tree, child)


    # --== GTK handlers ==--


    def onMouseButton(self, tree, event, path):
        """ A mouse button has been pressed """
        if event.button == 3:
            self.onShowPopupMenu(tree, event.button, event.time, path)
        elif path is not None:
            if event.button == 2:
                 self.playPaths(tree, [path], False)
            elif event.button == 1 and event.type == gtk.gdk._2BUTTON_PRESS:
                if   tree.getItem(path, ROW_PIXBUF) != consts.icoDir: self.playPaths(tree, None, True)
                elif tree.row_expanded(path):                         tree.collapse_row(path)
                else:                                                 tree.expand_row(path, False)


    def onShowPopupMenu(self, tree, button, time, path):
        """ Show a popup menu """
        if self.popup is None:
            self.popup = tools.loadGladeFile('FileExplorerMenu.glade')
            self.popup.get_widget('item-refresh').connect('activate', lambda widget: self.refresh(tree))
            self.popup.get_widget('item-collapse').connect('activate', lambda widget: tree.collapse_all())
            self.popup.get_widget('item-add').connect('activate', lambda widget: self.playPaths(tree, None, False))
            self.popup.get_widget('item-play').connect('activate', lambda widget: self.playPaths(tree, None, True))
            self.popup.get_widget('item-hidden').connect('toggled', lambda item: self.setShowHiddenFiles(item.get_active()))
            self.popup.get_widget('menu-popup').show_all()

        playable = path is not None and tree.getItem(path, ROW_PIXBUF) != consts.icoDir

        self.popup.get_widget('item-add').set_sensitive(playable)
        self.popup.get_widget('item-play').set_sensitive(playable)
        self.popup.get_widget('item-hidden').set_active(self.showHiddenFiles)
        self.popup.get_widget('menu-popup').popup(None, None, None, button, time)


    def onKeyPressed(self, tree, event):
        """ A key has been pressed """
        keyname = gtk.gdk.keyval_name(event.keyval)

        if keyname == 'F5':       self.refresh(tree)
        elif keyname == 'plus':   tree.expandRows()
        elif keyname == 'Left':   tree.collapseRows()
        elif keyname == 'Right':  tree.expandRows()
        elif keyname == 'minus':  tree.collapseRows()
        elif keyname == 'space':  tree.switchRows()
        elif keyname == 'Return': self.playPaths(tree, None, True)


    def onRowExpanded(self, tree, path):
        """ Replace the fake child by the real children """
        idle_add(self.__startLoading, tree, path)
        idle_add(self.exploreDir, tree, path, tree.getItem(path, ROW_FULLPATH), tree.getChild(path, 0))


    def onRowCollapsed(self, tree, path):
        """ Replace all children by a fake child """
        tree.removeAllChildren(path)
        tree.appendRow((consts.icoDir, '', TYPE_NONE, ''), path)


    def onDragDataGet(self, tree, context, selection, info, time):
        """ Provide information about the data being dragged """
        selection.set('text/uri-list', 8, ' '.join([urllib.pathname2url(file) for file in self.getFilesFromPaths(tree, None)]))


   # --== Message handler ==--


    def handleMsg(self, msg, params):
        """ Handle messages sent to this module """
        if msg == consts.MSG_EVT_EXPLORER_CHANGED and params['modName'] == MOD_L10N and self.currFolder != params['expName']:
            newFolder = params['expName']

            # Create a tree for this root folder if there's not already one
            if newFolder not in self.trees:
                self.trees[newFolder] = [self.createTree(newFolder), None]

            tree, state = self.trees[newFolder]

            # Make sure this tree is the current child of the scrolled window
            if self.currFolder is not None:
                currTree = self.trees[self.currFolder][0]
                self.trees[self.currFolder] = [currTree, currTree.saveState(ROW_NAME)]
                self.scrolled.remove(currTree)
            self.scrolled.add(tree)

            # Display the contents of the root folder
            self.currFolder = params['expName']

            # If the tree has just been created, we need to explore the root folder to populate the tree
            if state is not None:
                # FIXME This doesn't work, the scrollbar stays at the top...
                idle_add(tree.restoreState, state, ROW_NAME)
            else:
                self.exploreDir(tree, None, self.folders[self.currFolder])
                if len(tree) != 0:
                    tree.scroll_to_cell(0)
        elif msg == consts.MSG_EVT_APP_STARTED:
            self.onModLoaded()
        elif msg == consts.MSG_EVT_APP_QUIT:
            self.onModUnloaded()


    # --== Configuration ==--


    def configure(self, parent):
        """ Show the configuration dialog """
        if self.cfgWin is None:
            self.cfgWin = gui.window.Window('FileExplorer.glade', 'vbox1', __name__, MOD_L10N, 370, 400)
            # Create the list of folders
            txtRdr  = gtk.CellRendererText()
            pixRdr  = gtk.CellRendererPixbuf()
            columns = ((None, [(txtRdr, TYPE_STRING)],                           0, False, False),
                       ('',   [(pixRdr, gtk.gdk.Pixbuf), (txtRdr, TYPE_STRING)], 2, False, True))

            self.cfgList = extListview.ExtListView(columns, sortable=False, useMarkup=True, canShowHideColumns=False)
            self.cfgList.set_headers_visible(False)
            self.cfgWin.getWidget('scrolledwindow1').add(self.cfgList)
            # Connect handlers
            self.cfgList.connect('key-press-event', self.onCfgKeyPressed)
            self.cfgList.get_selection().connect('changed', self.onCfgSelectionChanged)
            self.cfgWin.getWidget('btn-add').connect('clicked', self.onAddFolder)
            self.cfgWin.getWidget('btn-remove').connect('clicked', lambda btn: self.onRemoveSelectedFolder(self.cfgList))
            self.cfgWin.getWidget('btn-ok').connect('clicked', self.onBtnOk)
            self.cfgWin.getWidget('btn-rename').connect('clicked', self.onRenameFolder)
            self.cfgWin.getWidget('btn-cancel').connect('clicked', lambda btn: self.cfgWin.hide())
            self.cfgWin.getWidget('btn-help').connect('clicked', self.onHelp)

        if not self.cfgWin.isVisible():
            self.populateFolderList()
            self.cfgWin.getWidget('chk-hidden').set_active(self.showHiddenFiles)
            self.cfgWin.getWidget('btn-ok').grab_focus()

        self.cfgWin.show()


    def populateFolderList(self):
        """ Populate the list of known folders """
        self.cfgList.replaceContent([(name, consts.icoBtnDir, '<b>%s</b>\n<small>%s</small>' % (cgi.escape(name), cgi.escape(path)))
                                     for name, path in sorted(self.folders.iteritems())])


    def onAddFolder(self, btn):
        """ Let the user add a new folder to the list """
        result = selectPath.SelectPath(MOD_L10N, self.cfgWin, self.folders.keys()).run()

        if result is not None:
            name, path = result
            self.folders[name] = path
            self.populateFolderList()
            modules.postMsg(consts.MSG_CMD_EXPLORER_ADD, {'modName': MOD_L10N, 'expName': name, 'icon': None, 'widget': self.scrolled})


    def onRemoveSelectedFolder(self, list):
        """ Remove the selected media folder """
        if list.getSelectedRowsCount() == 1:
            remark   = _('You will be able to add this root folder again later on if you wish so.')
            question = _('Remove the selected entry?')
        else:
            remark   = _('You will be able to add these root folders again later on if you wish so.')
            question = _('Remove all selected entries?')

        if gui.questionMsgBox(self.cfgWin, question, '%s %s' % (_('Your media files will not be deleted.'), remark)) == gtk.RESPONSE_YES:
            for row in self.cfgList.getSelectedRows():
                name = row[0]
                modules.postMsg(consts.MSG_CMD_EXPLORER_REMOVE, {'modName': MOD_L10N, 'expName': name})
                del self.folders[name]

                # Remove the tree, if any, from the scrolled window
                if self.currFolder == name:
                    self.scrolled.remove(self.trees[name][0])
                    self.currFolder = None

                # Remove the tree associated to this root folder, if any
                if name in self.trees:
                    del self.trees[name]

            self.cfgList.removeSelectedRows()


    def onRenameFolder(self, btn):
        """ Let the user rename a folder """
        name         = self.cfgList.getSelectedRows()[0][0]
        forbidden    = [rootName for rootName in self.folders if rootName != name]
        pathSelector = selectPath.SelectPath(MOD_L10N, self.cfgWin, forbidden)

        pathSelector.setPathSelectionEnabled(False)
        result = pathSelector.run(name, self.folders[name])

        if result is not None and result[0] != name:
            self.renameFolder(name, result[0])
            self.populateFolderList()


    def onCfgKeyPressed(self, list, event):
        """ Remove the selection if possible """
        if gtk.gdk.keyval_name(event.keyval) == 'Delete':
            self.onRemoveSelectedFolder(list)


    def onCfgSelectionChanged(self, selection):
        """ The selection has changed """
        self.cfgWin.getWidget('btn-remove').set_sensitive(selection.count_selected_rows() != 0)
        self.cfgWin.getWidget('btn-rename').set_sensitive(selection.count_selected_rows() == 1)


    def onBtnOk(self, btn):
        """ The user has clicked on the OK button """
        self.cfgWin.hide()
        self.setShowHiddenFiles(self.cfgWin.getWidget('chk-hidden').get_active())


    def onHelp(self, btn):
        """ Display a small help message box """
        helpDlg = help.HelpDlg(MOD_L10N)
        helpDlg.addSection(_('Description'),
                           _('This module allows you to browse the files on your drives.'))
        helpDlg.addSection(_('Usage'),
                           _('At least one root folder must be added to browse your files. This folder then becomes the root of the '
                             'file explorer tree in the main window.'))
        helpDlg.show(self.cfgWin)
