# -*- coding: utf-8 -*-
#
# Author: Ingelrest François (Athropos@gmail.com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import dbus, modules, os.path

from gui        import infoMsgBox
from tools      import consts, prefs
from media      import track
from gettext    import gettext as _
from modules    import ThreadedModule
from gui.window import Window


# Module configuration
MOD_NAME            = _('Instant Messenger Status')
MOD_DESC            = _('Update the status message of your IM client')
MOD_IS_MANDATORY    = False
MOD_IS_CONFIGURABLE = True


# Possible actions upon stopping or quitting
(
    STOP_DO_NOTHING,
    STOP_SET_STATUS
) = range(2)


# Default preferences
DEFAULT_STATUS_MSG       = '♫ {artist} - {album} ♫'
DEFAULT_STOP_ACTION      = STOP_SET_STATUS
DEFAULT_STOP_STATUS      = _('Decibel is stopped')
DEFAULT_SANITIZED_WORDS  = ''
DEFAULT_UPDATE_ON_PAUSED = True
DEFAULT_UPDATE_WHEN_AWAY = False


##############################################################################


class Gaim:

    def __init__(self, dbusInterface):
        """ Constructor """
        self.dbusInterface = dbusInterface


    def listAccounts(self):
        """ Return a default account, since Gaim merges all of them """
        return ['GenericAccount']


    def setStatusMsg(self, account, msg):
        """ Change the status message of the given account """
        try:
            current    = self.dbusInterface.GaimSavedstatusGetCurrent()
            statusType = self.dbusInterface.GaimSavedstatusGetType(current)
            statusId   = self.dbusInterface.GaimPrimitiveGetIdFromType(statusType)
            if statusId in ['available'] or prefs.get(__name__, 'update-when-away', DEFAULT_UPDATE_WHEN_AWAY):
                saved = self.dbusInterface.GaimSavedstatusNew('', statusType)
                self.dbusInterface.GaimSavedstatusSetMessage(saved, msg)
                self.dbusInterface.GaimSavedstatusActivate(saved)
        except:
            pass


##############################################################################


class Gajim:

    def __init__(self, dbusInterface):
        """ Constructor """
        self.dbusInterface = dbusInterface


    def listAccounts(self):
        """ Return a list of existing accounts """
        try:    return [account for account in self.dbusInterface.list_accounts()]
        except: return []


    def setStatusMsg(self, account, msg):
        """ Change the status message of the given account """
        try:
            currentStatus = self.dbusInterface.get_status(account)
            if currentStatus in ['online', 'chat'] or prefs.get(__name__, 'update-when-away', DEFAULT_UPDATE_WHEN_AWAY):
                self.dbusInterface.change_status(currentStatus, msg, account)
        except:
            pass


##############################################################################


class Gossip:

    def __init__(self, dbusInterface):
        """ Constructor """
        self.dbusInterface = dbusInterface


    def listAccounts(self):
        """ Return a default account, since Gossip merges all of them """
        return ['GenericAccount']


    def setStatusMsg(self, account, msg):
        """ Change the status message of the given account """
        try:
            currentStatus, currentMsg = self.dbusInterface.GetPresence('')
            if currentStatus in ['available'] or prefs.get(__name__, 'update-when-away', DEFAULT_UPDATE_WHEN_AWAY):
                self.dbusInterface.SetPresence(currentStatus, msg)
        except:
            pass


##############################################################################


class Pidgin:

    def __init__(self, dbusInterface):
        """ Constructor """
        self.dbusInterface = dbusInterface


    def listAccounts(self):
        """ Return a default account, since Pidgin merges all of them """
        return ['GenericAccount']


    def setStatusMsg(self, account, msg):
        """ Change the status message of the given account """
        try:
            current    = self.dbusInterface.PurpleSavedstatusGetCurrent()
            statusType = self.dbusInterface.PurpleSavedstatusGetType(current)
            statusId   = self.dbusInterface.PurplePrimitiveGetIdFromType(statusType)
            if statusId in ['available'] or prefs.get(__name__, 'update-when-away', DEFAULT_UPDATE_WHEN_AWAY):
                saved = self.dbusInterface.PurpleSavedstatusNew('', statusType)
                self.dbusInterface.PurpleSavedstatusSetMessage(saved, msg)
                self.dbusInterface.PurpleSavedstatusActivate(saved)
        except:
            pass


##############################################################################


# Elements associated with each supported IM clients
(
    IM_DBUS_SERVICE_NAME,
    IM_DBUS_OBJECT_NAME,
    IM_DBUS_INTERFACE_NAME,
    IM_CLASS,
    IM_INSTANCE,
    IM_ACCOUNTS
) = range(6)


# All specific classes have been defined, so we can now populate the list of supported IM clients
CLIENTS = (
            ['org.gajim.dbus',                 '/org/gajim/dbus/RemoteObject',   'org.gajim.dbus.RemoteInterface',    Gajim,  None, []],
            ['org.gnome.Gossip',               '/org/gnome/Gossip',              'org.gnome.Gossip',                  Gossip, None, []],
            ['net.sf.gaim.GaimService',        '/net/sf/gaim/GaimObject',        'net.sf.gaim.GaimInterface',         Gaim,   None, []],
            ['im.pidgin.purple.PurpleService', '/im/pidgin/purple/PurpleObject', 'im.pidgin.purple.PurpleInterfacep', Pidgin, None, []]
          )


class IMStatus(ThreadedModule):


    def __init__(self, wTree):
        """ Constructor """
        ThreadedModule.__init__(self)
        self.status     = ''     # The currently used status
        self.paused     = False  # True if the current track is paused
        self.clients    = []     # Clients currently active
        self.cfgWindow  = None   # Configuration window
        self.trackInfo  = None   # Information about the current track
        # Messages handled by this module
        modules.register(self, [modules.MSG_EVT_NEW_TRACK, modules.MSG_EVT_PAUSED, modules.MSG_EVT_UNPAUSED, modules.MSG_EVT_STOPPED,
                                modules.MSG_EVT_APP_STARTED, modules.MSG_EVT_APP_QUIT, modules.MSG_EVT_MOD_LOADED, modules.MSG_EVT_MOD_UNLOADED])


    def __format(self, string, trackInfo):
        """ Replace the special fields in the given string by their corresponding value and sanitize the result """
        result = trackInfo.format(string)

        if len(prefs.get(__name__, 'sanitized-words', DEFAULT_SANITIZED_WORDS)) != 0:
            lowerResult = result.lower()
            for word in [w.lower() for w in prefs.get(__name__, 'sanitized-words', DEFAULT_SANITIZED_WORDS).split('\n') if len(w) > 2]:
                pos = lowerResult.find(word)
                while pos != -1:
                    result      = result[:pos+1] + ('*' * (len(word)-2)) + result[pos+len(word)-1:]
                    lowerResult = lowerResult[:pos+1] + ('*' * (len(word)-2)) + lowerResult[pos+len(word)-1:]
                    pos         = lowerResult.find(word)

        return result


    def setStatusMsg(self, status):
        """ Try to update the status of all accounts of all active IM clients """
        for client in self.clients:
            for account in client[IM_ACCOUNTS]:
                client[IM_INSTANCE].setStatusMsg(account, status)


    def onNewTrack(self, trackInfo):
        """ A new track is being played """
        self.status    = self.__format(prefs.get(__name__, 'status-msg', DEFAULT_STATUS_MSG), trackInfo)
        self.paused    = False
        self.trackInfo = trackInfo
        self.setStatusMsg(self.status)


    def onStopped(self):
        """ The current track has been stopped """
        self.paused    = False
        self.trackInfo = None
        if prefs.get(__name__, 'stop-action', DEFAULT_STOP_ACTION) == STOP_SET_STATUS:
            self.setStatusMsg(prefs.get(__name__, 'stop-status', DEFAULT_STOP_STATUS))


    def onPaused(self):
        """ The current track has been paused """
        self.paused = True
        if prefs.get(__name__, 'update-on-paused', DEFAULT_UPDATE_ON_PAUSED):
            self.setStatusMsg(self.status + ' ' + _('[paused]'))


    def onUnpaused(self):
        """ The current track has been unpaused """
        self.paused = False
        self.setStatusMsg(self.status)


    def detectActiveClients(self):
        """ Find supported active IM clients """
        # Seems to randomly crash when not executed in the GTK main loop?
        activeServices = dbus.SessionBus().get_object('org.freedesktop.DBus', '/org/freedesktop/DBus').ListNames()
        for activeClient in [client for client in CLIENTS if client[IM_DBUS_SERVICE_NAME] in activeServices]:
            obj       = dbus.SessionBus().get_object(activeClient[IM_DBUS_SERVICE_NAME], activeClient[IM_DBUS_OBJECT_NAME])
            interface = dbus.Interface(obj, activeClient[IM_DBUS_INTERFACE_NAME])

            activeClient[IM_INSTANCE] = activeClient[IM_CLASS](interface)
            activeClient[IM_ACCOUNTS] = activeClient[IM_INSTANCE].listAccounts()

            self.clients.append(activeClient)


    # --== Message handler ==--


    def handleMsg(self, msg, params):
        """ Handle messages sent to this module """
        if   msg == modules.MSG_EVT_PAUSED:       self.onPaused()
        elif msg == modules.MSG_EVT_STOPPED:      self.onStopped()
        elif msg == modules.MSG_EVT_APP_QUIT:     self.onStopped()
        elif msg == modules.MSG_EVT_UNPAUSED:     self.onUnpaused()
        elif msg == modules.MSG_EVT_NEW_TRACK:    self.onNewTrack(params['track'])
        elif msg == modules.MSG_EVT_MOD_LOADED:   self.gtkExecute(self.detectActiveClients)
        elif msg == modules.MSG_EVT_APP_STARTED:  self.gtkExecute(self.detectActiveClients)
        elif msg == modules.MSG_EVT_MOD_UNLOADED: self.onStopped()


    # --== Configuration ==--


    def configure(self, parent):
        """ Show the configuration window """
        if self.cfgWindow is None:
            self.cfgWindow            = Window('IMStatus.glade', 'vbox1', __name__, MOD_NAME, 440, 290)
            self.cfgBtnOk             = self.cfgWindow.getWidget('btn-ok')
            self.cfgBtnHelp           = self.cfgWindow.getWidget('btn-help')
            self.cfgBtnCancel         = self.cfgWindow.getWidget('btn-cancel')
            self.cfgTxtStatus         = self.cfgWindow.getWidget('txt-status')
            self.cfgTxtStopStatus     = self.cfgWindow.getWidget('txt-stopStatus')
            self.cfgRadStopDoNothing  = self.cfgWindow.getWidget('rad-stopDoNothing')
            self.cfgRadStopSetStatus  = self.cfgWindow.getWidget('rad-stopSetStatus')
            self.cfgChkUpdateOnPaused = self.cfgWindow.getWidget('chk-updateOnPaused')
            self.cfgChkUpdateWhenAway = self.cfgWindow.getWidget('chk-updateWhenAway')
            self.cfgTxtSanitizedWords = self.cfgWindow.getWidget('txt-sanitizedWords')
            # GTK handlers
            self.cfgRadStopDoNothing.connect('toggled', self.onRadToggled)
            self.cfgRadStopSetStatus.connect('toggled', self.onRadToggled)
            self.cfgBtnOk.connect('clicked', self.onBtnOk)
            self.cfgBtnCancel.connect('clicked', lambda btn: self.cfgWindow.hide())
            self.cfgBtnHelp.connect('clicked', self.onBtnHelp)


        if not self.cfgWindow.isVisible():
            self.cfgTxtStatus.set_text(prefs.get(__name__, 'status-msg', DEFAULT_STATUS_MSG))
            self.cfgChkUpdateOnPaused.set_active(prefs.get(__name__, 'update-on-paused', DEFAULT_UPDATE_ON_PAUSED))
            self.cfgChkUpdateWhenAway.set_active(prefs.get(__name__, 'update-when-away', DEFAULT_UPDATE_WHEN_AWAY))
            self.cfgRadStopDoNothing.set_active(prefs.get(__name__, 'stop-action', DEFAULT_STOP_ACTION) == STOP_DO_NOTHING)
            self.cfgRadStopSetStatus.set_active(prefs.get(__name__, 'stop-action', DEFAULT_STOP_ACTION) == STOP_SET_STATUS)
            self.cfgTxtStopStatus.set_sensitive(prefs.get(__name__, 'stop-action', DEFAULT_STOP_ACTION) == STOP_SET_STATUS)
            self.cfgTxtStopStatus.set_text(prefs.get(__name__, 'stop-status', DEFAULT_STOP_STATUS))
            self.cfgTxtSanitizedWords.get_buffer().set_text(prefs.get(__name__, 'sanitized-words', DEFAULT_SANITIZED_WORDS))

        self.cfgBtnOk.grab_focus()
        self.cfgWindow.show()


    def onRadToggled(self, btn):
        """ A radio button has been toggled """
        self.cfgTxtStopStatus.set_sensitive(self.cfgRadStopSetStatus.get_active())


    def onBtnOk(self, btn):
        """ Save new preferences """
        prefs.set(__name__, 'status-msg', self.cfgTxtStatus.get_text())
        prefs.set(__name__, 'update-on-paused', self.cfgChkUpdateOnPaused.get_active())
        prefs.set(__name__, 'update-when-away', self.cfgChkUpdateWhenAway.get_active())
        (start, end) = self.cfgTxtSanitizedWords.get_buffer().get_bounds()
        prefs.set(__name__, 'sanitized-words', self.cfgTxtSanitizedWords.get_buffer().get_text(start, end).strip())
        if self.cfgRadStopDoNothing.get_active():
            prefs.set(__name__, 'stop-action', STOP_DO_NOTHING)
        else:
            prefs.set(__name__, 'stop-action', STOP_SET_STATUS)
            prefs.set(__name__, 'stop-status', self.cfgTxtStopStatus.get_text())
        self.cfgWindow.hide()
        # Update status
        if self.trackInfo is not None:
            self.status = self.__format(prefs.get(__name__, 'status-msg', DEFAULT_STATUS_MSG), self.trackInfo)
            if self.paused: self.setStatusMsg(self.status + ' ' + _('[paused]'))
            else:           self.setStatusMsg(self.status)


    def onBtnHelp(self, btn):
        """ Display a small help message box """
        helpMsg  = _('Any field in the status message of the form <b>{<tt>field</tt>}</b> will be replaced by the corresponding value. Available fields are:')
        helpMsg += '\n\n'
        helpMsg += track.getFormatSpecialFields()
        helpMsg += '\n\n'
        helpMsg += _('You can also define some words to sanitize before using them to change your status. The middle characters of matching words are automatically replaced with asterisks (e.g., "Metallica - Live S**t Binge &amp; Purge"). Put one word per line.')
        infoMsgBox(self.cfgWindow, helpMsg)
