# Written by Bram Cohen
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: CurrentRateMeasure.py 342 2008-01-23 03:21:02Z camrdale-guest $

"""Measuring rates of download and upload."""

from clock import clock

class Measure:
    """The measurement of one rate.
    
    @type max_rate_period: C{float}
    @ivar max_rate_period: maximum amount of time to guess the current rate 
        estimate represents
    @type ratesince: C{float}
    @ivar ratesince: the oldest time the rate estimate is for
    @type last: C{float}
    @ivar last: the last time the rate was updated
    @type rate: C{float}
    @ivar rate: the latest calculated rate
    @type total: C{long}
    @ivar total: the total amount that went in to calculating the rate
    
    """
    
    def __init__(self, max_rate_period, fudge = 1, saved_total = 0L):
        """Initialize the measurement.
        
        @type max_rate_period: C{float}
        @param max_rate_period: maximum amount of time to guess the current 
            rate estimate represents
        @type fudge: C{int}
        @param fudge: time equivalent of writing to kernel-level TCP buffer, 
            for rate adjustment (optional, defaults to 1)
        @type saved_total: C{long}
        @param saved_total: the saved amount measured from a previous run
            (optional, defaults to 0)
        
        """
        
        self.max_rate_period = max_rate_period
        self.ratesince = clock() - fudge
        self.last = self.ratesince
        self.rate = 0.0
        self.total = long(saved_total)

    def update_rate(self, amount):
        """Update the rate with new data.
        
        @type amount: C{long}
        @param amount: the new data to add into the rate calculation
        
        """
        
        self.total += amount
        t = clock()
        self.rate = (self.rate * (self.last - self.ratesince) + 
            amount) / (t - self.ratesince + 0.0001)
        self.last = t
        if self.ratesince < t - self.max_rate_period:
            self.ratesince = t - self.max_rate_period

    def get_rate(self):
        """Get the current rate measurement.
        
        @rtype: C{float}
        @return: the current rate
        
        """
        
        self.update_rate(0)
        return self.rate

    def get_rate_noupdate(self):
        """Get the current rate measurement without updating it.
        
        @rtype: C{float}
        @return: the current rate
        
        """
        
        return self.rate

    def time_until_rate(self, newrate):
        """Calculate how long until the rate drops to the target.
        
        @type newrate: C{float}
        @param newrate: the target rate
        @rtype: C{float}
        @return: the number of seconds until the rate decreases to the target 
            rate, or 0 if it's already there (or below it)
        
        """
        
        if self.rate <= newrate:
            return 0
        t = clock() - self.ratesince
        return ((self.rate * t) / newrate) - t

    def get_total(self):
        """Get the total amount used to calculate the rate..
        
        @rtype: C{float}
        @return: the total amount
        
        """
        
        return self.total