# Written by Bram Cohen
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: Uploader.py 266 2007-08-18 02:06:35Z camrdale-guest $

"""Manage uploading to a single peer."""

from DebTorrent.CurrentRateMeasure import Measure

class Upload:
    """Manage uploading to a single peer.
    
    @type connection: L{Connecter.Connection}
    @ivar connection: the connection to the peer
    @type ratelimiter: L{RateLimiter.RateLimiter}
    @ivar ratelimiter: the RateLimiter instance to use
    @type totalup: L{Debtorrent.CurrentRateMeasure.Measure}
    @ivar totalup: the Measure instance to use
    @type choker: L{Choker.Choker}
    @ivar choker: the Choker instance to use
    @type storage: L{BT1.StorageWrapper.StorageWrapper}
    @ivar storage: the StorageWrapper instance
    @type picker: L{BT1.PiecePicker.PiecePicker}
    @ivar picker: the PiecePicker instance
    @type config: C{dictionary}
    @ivar config: the configration information
    @type max_slice_length: C{int}
    @ivar max_slice_length: maximum length chunk to send to peers
    @type choked: C{boolean}
    @ivar choked: whether we are choking the connection
    @type cleared: C{boolean}
    @ivar cleared: whether requests are allowed to be appended to the buffer
    @type interested: C{boolean}
    @ivar interested: whether the peer is interested
    @type super_seeding: C{boolean}
    @ivar super_seeding: whether we are in super-seed mode
    @type buffer: C{list} of (C{int}, C{int}, C{int})
    @ivar buffer: the pending requests for the peer, the piece index, offset
        within the piece, and chunk length requested
    @type measure: L{DebTorrent.CurrentRateMeasure.Measure}
    @ivar measure: for measuring the upload rate to the peer
    @type was_ever_interested: C{boolean}
    @ivar was_ever_interested: whether the peer has ever been interested
    @type seed_have_list: C{list} of C{int}
    @ivar seed_have_list: the list of pieces the peer is allowed to request
        in super-seed mode
    @type skipped_count: C{int}
    @ivar skipped_count: the number of pieces the peer has refused to request
        from us in super-seed mode
    @type piecedl: C{int}
    @ivar piecedl: the current piece being downloaded by the peer
    @type piecebuf: L{DebTorrent.piecebuffer.SingleBuffer}
    @ivar piecebuf: the buffer containing the entire piece currently being
        downloaded by the peer
    
    """
    
    def __init__(self, connection, ratelimiter, totalup, choker, storage,
                 picker, config):
        """Initialize the instance and send the initial bitfield.
        
        @type connection: L{Connecter.Connection}
        @param connection: the connection to the peer
        @type ratelimiter: L{RateLimiter.RateLimiter}
        @param ratelimiter: the RateLimiter instance to use
        @type totalup: L{Debtorrent.CurrentRateMeasure.Measure}
        @param totalup: the Measure instance to use
        @type choker: L{Choker.Choker}
        @param choker: the Choker instance to use
        @type storage: L{BT1.StorageWrapper.StorageWrapper}
        @param storage: the StorageWrapper instance
        @type picker: L{BT1.PiecePicker.PiecePicker}
        @param picker: the PiecePicker instance
        @type config: C{dictionary}
        @param config: the configration information
        
        """
        
        self.connection = connection
        self.ratelimiter = ratelimiter
        self.totalup = totalup
        self.choker = choker
        self.storage = storage
        self.picker = picker
        self.config = config
        self.max_slice_length = config['max_slice_length']
        self.choked = True
        self.cleared = True
        self.interested = False
        self.super_seeding = False
        self.buffer = []
        self.measure = Measure(config['max_rate_period'], config['upload_rate_fudge'])
        self.was_ever_interested = False
        if storage.get_amount_left() == 0:
            if choker.super_seed:
                self.super_seeding = True   # flag, and don't send bitfield
                self.seed_have_list = []    # set from piecepicker
                self.skipped_count = 0
            else:
                if config['breakup_seed_bitfield']:
                    bitfield, msgs = storage.get_have_list_cloaked()
                    connection.send_bitfield(bitfield)
                    for have in msgs:
                        connection.send_have(have)
                else:
                    connection.send_bitfield(storage.get_have_list())
        else:
            if storage.do_I_have_anything():
                connection.send_bitfield(storage.get_have_list())
        self.piecedl = None
        self.piecebuf = None

    def got_not_interested(self):
        """Process a received not interested message."""
        if self.interested:
            self.interested = False
            del self.buffer[:]
            self.piecedl = None
            if self.piecebuf:
                self.piecebuf.release()
            self.piecebuf = None
            self.choker.not_interested(self.connection)

    def got_interested(self):
        """Process a received interested message."""
        if not self.interested:
            self.interested = True
            self.was_ever_interested = True
            self.choker.interested(self.connection)

    def get_upload_chunk(self):
        """Get a chunk to upload to the peer.
        
        @rtype: (C{int}, C{int}, C{string})
        @return: the piece index, offset within the piece, and the chunk
        
        """
        
        if self.choked or not self.buffer:
            return None
        index, begin, length = self.buffer.pop(0)
        if self.config['buffer_reads']:
            if index != self.piecedl:
                if self.piecebuf:
                    self.piecebuf.release()
                self.piecedl = index
                self.piecebuf = self.storage.get_piece(index, 0, -1)
            try:
                piece = self.piecebuf[begin:begin+length]
                assert len(piece) == length
            except:     # fails if storage.get_piece returns None or if out of range
                self.connection.close()
                return None
        else:
            if self.piecebuf:
                self.piecebuf.release()
                self.piecedl = None
            piece = self.storage.get_piece(index, begin, length)
            if piece is None:
                self.connection.close()
                return None
        self.measure.update_rate(len(piece))
        self.totalup.update_rate(len(piece))
        return (index, begin, piece)

    def got_request(self, index, begin, length):
        """Add a received request for a chunk to the buffer.
        
        @type index: C{int}
        @param index: the piece index
        @type begin: C{int}
        @param begin: the offset within the piece
        @type length: C{int}
        @param length: the amount of data to return
        
        """
        
        if ( (self.super_seeding and not index in self.seed_have_list)
                   or not self.interested or length > self.max_slice_length ):
            self.connection.close()
            return
        if not self.cleared:
            self.buffer.append((index, begin, length))
        if not self.choked and self.connection.next_upload is None:
                self.ratelimiter.queue(self.connection)


    def got_cancel(self, index, begin, length):
        """Cancel a request for a chunk.
        
        @type index: C{int}
        @param index: the piece index
        @type begin: C{int}
        @param begin: the offset within the piece
        @type length: C{int}
        @param length: the amount of data to return
        
        """
        
        try:
            self.buffer.remove((index, begin, length))
        except ValueError:
            pass

    def choke(self):
        """Start choking the connection."""
        if not self.choked:
            self.choked = True
            self.connection.send_choke()
        self.piecedl = None
        if self.piecebuf:
            self.piecebuf.release()
            self.piecebuf = None

    def choke_sent(self):
        """Remove all requests after a choke is sent."""
        del self.buffer[:]
        self.cleared = True

    def unchoke(self):
        """Unchoke the connection."""
        if self.choked:
            self.choked = False
            self.cleared = False
            self.connection.send_unchoke()
        
    def disconnected(self):
        """Clean up for disconnection from the peer."""
        if self.piecebuf:
            self.piecebuf.release()
            self.piecebuf = None

    def is_choked(self):
        """Check whether we are choking the connection.
        
        @rtype: C{boolean}
        @return: whether the connection is being choked
        
        """
        
        return self.choked
        
    def is_interested(self):
        """Check whether the peer is interested in downloading.
        
        @rtype: C{boolean}
        @return: whether the connected peer is interested
        
        """
        
        return self.interested

    def has_queries(self):
        """Check whether we have pending requests for chunks from the peer.
        
        @rtype: C{boolean}
        @return: whether there are requests pending
        
        """
        
        return not self.choked and len(self.buffer) > 0

    def get_rate(self):
        """Get the current upload rate to the peer.
        
        @rtype: C{float}
        @return: the current upload rate
        
        """
        
        return self.measure.get_rate()
    
