# Written by Bram Cohen
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: download_bt1.py 416 2010-03-20 18:29:37Z camrdale $

"""Manage a single download.

@type logger: C{logging.Logger}
@var logger: the logger to send all log messages to for this module
@type defaults: C{list} of (C{string}, unknown, C{string})
@var defaults: the default configuration variables, including descriptions
@type argslistheader: C{string}
@var argslistheader: the header to print before the default config

"""

from zurllib import urlopen
from urlparse import urlparse
from BT1.btformats import check_message
from BT1.Choker import Choker
from BT1.Storage import Storage
from BT1.StorageWrapper import StorageWrapper
from BT1.FileSelector import FileSelector
from BT1.Uploader import Upload
from BT1.Downloader import Downloader
from BT1.HTTPDownloader import HTTPDownloader
from BT1.Connecter import Connecter
from RateLimiter import RateLimiter
from BT1.Encrypter import Encoder
from RawServer import RawServer, autodetect_ipv6, autodetect_socket_style
from BT1.Rerequester import Rerequester
from BT1.DownloaderFeedback import DownloaderFeedback
from RateMeasure import RateMeasure
from CurrentRateMeasure import Measure
from BT1.PiecePicker import PiecePicker
from BT1.Statistics import Statistics
from ConfigDir import ConfigDir
from bencode import bencode, bdecode
from sha import sha
from os import path, makedirs, listdir, walk
from parseargs import parseargs, formatDefinitions, defaultargs
from socket import error as socketerror
from random import seed
from threading import Thread, Event
from clock import clock
from BTcrypto import CRYPTO_OK
from __init__ import createPeerID
from BT1.makemetafile import getpieces, getsubpieces, uniconvert, convert_all
from gzip import GzipFile
from StringIO import StringIO
import binascii, logging

logger = logging.getLogger('DebTorrent.download_bt1')

defaults = [
    # Not in the config file
    ('configfile', '', 'the configuration file to use, if not specified then ' +
        'a file in /etc/debtorrent will be used, followed by ' +
        'a file in the .DebTorrent directory in the user\'s home directory'),
    # Locations
    ('download_dir', '',
        'directory to save the download in, blank indicates use the user\'s ' +
        'home directory'),
    ('saveas_style', 1,
        'How to name torrent download directory ' + 
        '(1 = <mirror>_dists_<suite>_<section>_binary-<arch>, ' +
        '2 = <mirror> (caution: experimental))' ),
    ('cache_dir', '', 'the directory to use to get/store cache files, if not ' + 
        'specified then a .DebTorrent directory in the user\'s home directory ' +
        'will be used'),
    ('expire_cache_data', 10,
        'the number of days after which you wish to expire old cache data ' +
        '(0 = disabled)'),
    ('log_dir', '',
        'directory to write the logfiles to (default is to use the cache directory)'),
    ('log_level', 20,
        'level to write the logfiles at, varies from 10 (debug) to 50 (critical)'),
    # Rate limits
    ('max_upload_rate', 0,
        'maximum kB/s to upload at (0 = no limit, -1 = automatic)'),
    ('max_download_rate', 0,
        'maximum kB/s to download at (0 = no limit)'),
    ('upload_rate_fudge', 5.0, 
        'time equivalent of writing to kernel-level TCP buffer, for rate adjustment'),
    ('tcp_ack_fudge', 0.03,
        'how much TCP ACK download overhead to add to upload rate calculations ' +
        '(0 = disabled)'),
    ('max_rate_period', 20.0,
        'maximum amount of time to guess the current rate estimate represents'),
    ('upload_unit_size', 1460,
        'when limiting upload rate, how many bytes to send at a time'),
    # Torrent Connections
    ('bind', '', 
        'comma-separated list of ips/hostnames to bind to locally'),
    ('ip', '',
        'ip to report you have to the tracker'),
    ('minport', 10000, 'minimum port to listen on, counts up if unavailable'),
    ('maxport', 60000, 'maximum port to listen on'),
    ('random_port', 1, 'whether to choose randomly inside the port range ' +
        'instead of counting up linearly'),
#    ('ipv6_enabled', autodetect_ipv6(),
    ('ipv6_enabled', 0,
        'allow the client to connect to peers via IPv6'),
    ('ipv6_binds_v4', autodetect_socket_style(),
        'set if an IPv6 server socket will also field IPv4 connections'),
    ('timeout', 300.0,
        'time to wait between closing sockets which nothing has been received on'),
    ('timeout_check_interval', 60.0,
        'time to wait between checking if any connections have timed out'),
    # Encryption
    ('crypto_allowed', int(CRYPTO_OK),
        'whether to allow the client to accept encrypted connections'),
    ('crypto_only', 0,
        'whether to only create or allow encrypted connections'),
    ('crypto_stealth', 0,
        'whether to prevent all non-encrypted connection attempts; ' +
        'will result in an effectively firewalled state on older trackers'),
    # APT Requests
    ('apt_port', 9988, 'port to listen for apt on'),
    ('apt_bind', '', 
        'comma-separated list of ips/hostnames to bind to locally ' +
        'to listen for APT requests'),
    ('show_infopage', 1, 'whether to display an info page when the ' +
        'APT requester\'s root directory is loaded'),
    ('infopage_redirect', '', 'a URL to redirect the info page to'),
    ('favicon', '', 
        'file containing x-icon data to return when browser requests favicon.ico'),
    ('display_path', 0,
        'whether to display the full path or the torrent contents for each torrent'),
    ('allow_get', 0, 
        'adds a /file?infohash={hash} url that allows users to download the ' +
        'dtorrent file'),
    ('min_time_between_log_flushes', 3.0,
        'minimum time it must have been since the last flush to do another one'),
    ('hupmonitor', 0, 
        'whether to reopen the log file upon receipt of HUP signal'),
    ('allowed_ips', '', 
        'only allow connections from IPs specified in the given file; '+
        'file contains subnet data in the format: aa.bb.cc.dd/len'),
    ('banned_ips', '', 
        'don\'t allow connections from IPs specified in the given file; ' +
        'file contains IP range data in the format: xxx:xxx:ip1-ip2'),
    ('parse_ip_files', 60, 
        'seconds between reloading of allowed_ips and banned_ips lists for apt'),
    # Other Things
    ('disable_http_downloader', 0, 
        '(for testing purposes only) whether to disable the backup HTTP downloader'),
    ('separate_all', 3, 'whether to separate the architecture:all packages, ' +
        '0 = don\'t separate, 1 = separate and run architecture:all, ' +
        '2 = separate and run all architectures but all, ' + 
        '3 = separate and run both'),
    ('pieces_url', 'merkel.debian.org/~camrdale/extrapieces/',
        'the remote URL to use to fetch information on the splitting of ' +
        'large packages into pieces'),
    ('long_lived_torrent_url', 'merkel.debian.org/~camrdale/uniquepieces/',
        'the remote URL to use to fetch information on creating a long-lived ' +
        'torrent for testing or unstable using unique piece numbers'),
    # End of Normal Options
    # BitTorrent Options
    ('keepalive_interval', 120.0,
        'number of seconds to pause between sending keepalives'),
    ('download_slice_size', 2 ** 14,
        'How many bytes to query for per request.'),
    ('max_slice_length', 2 ** 17,
        'maximum length slice to send to peers, larger requests are ignored'),
    ('max_message_length', 2 ** 23,
        'maximum length prefix encoding you\'ll accept over the wire - ' +
        'larger values get the connection dropped.'),
    ('rarest_first_cutoff', 2,
        "number of downloads at which to switch from random to rarest first"),
    ('rarest_first_priority_cutoff', 5,
        'the number of peers which need to have a piece before other ' +
        'partials take priority over rarest first'),
    ('breakup_seed_bitfield', 1,
        'sends an incomplete bitfield and then fills with have messages, '
        'in order to get around stupid ISP manipulation'),
    ('snub_time', 30.0,
        'seconds to wait for data to come in over a connection before ' +
        'assuming it\'s semi-permanently choked'),
    # Incentive Mechanism
    ('max_uploads', 7,
        'the maximum number of uploads to allow at once.'),
    ('min_uploads', 4,
        'the number of uploads to fill out to with extra optimistic unchokes'),
    ('round_robin_period', 30,
        'the number of seconds between the client\'s switching upload targets'),
    ('auto_kick', 1,
        'whether to allow the client to automatically kick/ban peers that send bad data'),
    ('security', 1,
        'whether to enable extra security features intended to prevent abuse'),
    # Tracker Connections
    ('default_tracker', 'http://dttracker.debian.net:6969/announce', 
        'the default tracker address to use for new torrents'),
    ('force_tracker', '', 
        'set this to the tracker address to use for all torrents'),
    ('rerequest_interval', 5 * 60,
        'time to wait between requesting more peers'),
    ('min_peers', 20, 
        'minimum number of peers to not do rerequesting'),
    ('max_initiate', 40,
        'number of peers at which to stop initiating new connections'),
    ('max_connections', 0,
        'the absolute maximum number of peers to connect with (0 = no limit)'),
    ('http_timeout', 60, 
        'number of seconds to wait before assuming that an http connection ' +
        'has timed out'),
    # File System
    ('buffer_reads', 1,
        'whether to buffer disk reads'),
    ('write_buffer_size', 4,
        'the maximum amount of space to use for buffering disk writes ' +
        '(in megabytes, 0 = disabled)'),
    ('max_files_open', 50,
        'the maximum number of files to keep open at a time, 0 means no limit'),
    ('check_hashes', 1,
        'whether to check hashes on disk'),
    ('double_check', 1,
        'whether to double-check data being written to the disk for errors ' +
        '(may increase CPU load)'),
    ('triple_check', 0,
        'whether to thoroughly check data being written to the disk ' +
        '(may slow disk access)'),
    ('lock_files', 1,
        'whether to lock files the client is working with'),
    ('lock_while_reading', 0,
        'whether to lock access to files being read'),
    ('auto_flush', 0,
        'minutes between automatic flushes to disk (0 = disabled)'),
    # Advanced Features
    ('super_seeder', 0,
        'whether to use special upload-efficiency-maximizing routines ' +
        '(only for dedicated seeds)'),
    ('dedicated_seed_id', '',
        'code to send to tracker identifying as a dedicated seed'),
    ]

argslistheader = 'Arguments are:\n\n'


def parse_params(params, presets = {}):
    """Parse the command-line parameters.
    
    @type params: C{list} of C{string}
    @param params: the command-line parameters
    @type presets: C{dictionary}
    @param presets: the preset values to use (optional)
    @rtype: C{dictionary}
    @return: the configuration variables
    @raise ValueError: if the parameters are not properly specified
    
    """
    
    if len(params) == 0:
        return None
    config, args = parseargs(params, defaults, 0, 1, presets = presets)
    if args:
        if config['responsefile'] or config['url']:
            raise ValueError,'must have responsefile or url as arg or parameter, not both'
        if path.isfile(args[0]):
            config['responsefile'] = args[0]
        else:
            try:
                urlparse(args[0])
            except:
                raise ValueError, 'bad filename or url'
            config['url'] = args[0]
    elif (config['responsefile'] == '') == (config['url'] == ''):
        raise ValueError, 'need responsefile or url, must have one, cannot have both'
    return config


def get_usage(defaults = defaults, cols = 100, presets = {}):
    """Print the usage information for the program.
    
    @type defaults: C{list} of C{tuple}
    @param defaults: the default configuration variables
        (optional, default is to use L{defaults})
    @type cols: C{int}
    @param cols: the width of the print out (optional, default is 100)
    @type presets: C{dictionary}
    @param presets: the preset values to use (optional)
    
    """
    
    return (argslistheader + formatDefinitions(defaults, cols, presets))


def get_response(file, url, separate_all = 0, 
                 default_tracker = 'http://dttracker.debian.net:6969/announce'):
    """Get the response data from a metainfo or Packages file.
    
    First checks to see if the data is in the Packages file format, and
    returns the extracted response data if it is. Otherwise, assumes it is
    a metainfo file and tries to get the response data from it.
    
    @type file: C{string}
    @param file: the file name to use, or None to indicate that the url is
        to be used
    @type url: C{string}
    @param url: the URL to download the metainfo file from
    @type separate_all: C{int}
    @param separate_all: whether to separate the architecture:all packages into
        a separate torrent (optional, defaults to no)
    @type default_tracker: C{string}
    @param default_tracker: the default tracker to use for created torrents
        (optional, defaults to 'http://dttracker.debian.net:6969/announce')
    @rtype: C{dictionary}, C{dictionary}
    @return: the metainfo data
    
    """
    
    (response, response_all) = get_packages(file, url, separate_all, default_tracker)
    if response:
        try:
            check_message(response)
        except ValueError, e:
            logger.exception("got bad file info")
            return (None, None)
        return (response, response_all)
    if response_all:
        try:
            check_message(response_all)
        except ValueError, e:
            logger.exception("got bad file info")
            return (None, None)
        return (response, response_all)
    try:
        if file:
            h = open(file, 'rb')
            try:
                line = h.read(10)   # quick test to see if responsefile contains a dict
                front,garbage = line.split(':',1)
                assert front[0] == 'd'
                int(front[1:])
            except:
                logger.exception(file+' is not a valid responsefile')
                return (None, None)
            try:
                h.seek(0)
            except:
                try:
                    h.close()
                except:
                    pass
                h = open(file, 'rb')
        else:
            try:
                h = urlopen(url)
            except:
                logger.exception(url+' bad url')
                return (None, None)
        response = h.read()
    
    except IOError, e:
        logger.eexception('problem getting response info')
        return (None, None)
    try:    
        h.close()
    except:
        pass
    try:
        try:
            response = bdecode(response)
        except:
            logger.warning("bad data in responsefile, switching to sloppy decoding")
            response = bdecode(response, sloppy=1)
        check_message(response)
    except ValueError, e:
        logger.exception("got bad file info")
        return (None, None)

    return (response, None)

def get_packages(file, url, separate_all = 0, 
                 default_tracker = 'http://dttracker.debian.net:6969/announce'):
    """Extract the response data from a Packages file.
    
    @type file: C{string}
    @param file: the file name to use, or None to indicate that the url is
        to be used
    @type url: C{string}
    @param url: the URL to download the metainfo file from
    @type separate_all: C{int}
    @param separate_all: whether to separate the architecture:all packages into
        a separate torrent (optional, defaults to no)
    @type default_tracker: C{string}
    @param default_tracker: the default tracker to use for created torrents
        (optional, defaults to 'http://dttracker.debian.net:6969/announce')
    @rtype: C{dictionary}, C{dictionary}
    @return: the metainfo data
    
    """

    try:
        if file:
            name = path.split(file)[1]
            h = open(file, 'r')
            try:
                for line in h:   # quick test to see if packages file is correct
                    if len(line.strip()) > 0:
                        break
                assert line[:8] == "Package:"
            except:
                logger.warning(file+' is not a valid Packages file')
                return (None, None)
            try:
                h.seek(0)
            except:
                try:
                    h.close()
                except:
                    pass
                h = open(file, 'r')
        else:
            urlp = urlparse(url)
            name = urlp[1] + urlp[2].replace('/','_')
            try:
                h = urlopen(url)
                file = url
                data = h.read()
                assert len(data) > 8
                if url[-3:] == ".gz":
                    name = urlp[1] + urlp[2][:-3].replace('/','_')
                    if data[:8] != "Package:":
                        compressed = StringIO(data)
                        f = GzipFile(fileobj = compressed)
                        data = f.read()
                assert data[:8] == "Package:"
                h = data.split('\n')
            except:
                logger.warning(url+' bad url')
                return (None, None)

        sub_pieces = getsubpieces(name)
        
        piece_ordering_all = {}
        ordering_all_headers = {}
        piece_ordering = {}
        ordering_headers = {}
        if separate_all:
            (piece_ordering, ordering_headers) = getordering(name)
            if separate_all in [1, 3]:
                (piece_ordering_all, ordering_all_headers) = getordering(name, all = True)
    
        (info, info_all) = getpieces(h, separate_all = separate_all, sub_pieces = sub_pieces,
                                     piece_ordering = piece_ordering,
                                     piece_ordering_all = piece_ordering_all,
                                     num_pieces = int(ordering_headers.get('NextPiece', 0)),
                                     num_all_pieces = int(ordering_all_headers.get('NextPiece', 0)))

        response = None
        response_all = None
        if info:
            response = {'info': info,
                        'announce': default_tracker, 
                        'name': uniconvert(name)}

            if "Tracker" in ordering_headers:
                response['announce'] = ordering_headers["Tracker"].strip()
                del ordering_headers["Tracker"]
            if "Torrent" in ordering_headers:
                response['identifier'] = binascii.a2b_hex(ordering_headers["Torrent"].strip())
                del ordering_headers["Torrent"]
            for header, value in ordering_headers.items():
                response[header] = value.strip()

        if info_all:
            response_all = {'info': info_all,
                            'announce': default_tracker, 
                            'name': uniconvert(convert_all(name))}
    
            if "Tracker" in ordering_all_headers:
                response_all['announce'] = ordering_all_headers["Tracker"].strip()
                del ordering_all_headers["Tracker"]
            if "Torrent" in ordering_all_headers:
                response_all['identifier'] = binascii.a2b_hex(ordering_all_headers["Torrent"].strip())
                del ordering_all_headers["Torrent"]
            for header, value in ordering_all_headers.items():
                response_all[header] = value.strip()

    except IOError, e:
        logger.exception('problem getting Packages info')
        return (None, None)
    try:    
        h.close()
    except:
        pass

    return (response, response_all)


class BT1Download:    
    """Manage a single download.
    
    @type statusfunc: C{method}
    @ivar statusfunc: the method to call to print status updates
    @type finfunc: C{method}
    @ivar finfunc: the method to call when the download is completed
    @type errorfunc: C{method}
    @ivar errorfunc: the method to call when an error occurs
    @type doneflag: C{threading.Event}
    @ivar doneflag: the flag that indicates when the program is to be shutdown
    @type config: C{dictionary}
    @ivar config: the configuration variables
    @type response: C{dictionary}
    @ivar response: the response data from the metainfo file
    @type infohash: C{string}
    @ivar infohash: the hash of the info from the response data
    @type myid: C{string}
    @ivar myid: the peer ID to use
    @type rawserver: L{RawServer.RawServer}
    @ivar rawserver: the server controlling the program
    @type port: C{int}
    @ivar port: the port being listened to
    @type info: C{dictionary}
    @ivar info: the info data from the response data
    @type pieces: C{list} of C{string}
    @ivar pieces: the hashes of the pieces
    @type piece_lengths: C{list} of C{int}
    @ivar piece_lengths: the lengths of the pieces
    @type len_pieces: C{int}
    @ivar len_pieces: the number of pieces
    @type argslistheader: C{string}
    @ivar argslistheader: the header to print before the default config
    @type unpauseflag: C{threading.Event}
    @ivar unpauseflag: the flag to unset to pause the download
    @type pickled_data: C{dictionary}
    @ivar pickled_data: the saved data from a previous run
    @type downloader: L{BT1.Downloader.Downloader}
    @ivar downloader: the Downloader instance
    @type storagewrapper: L{BT1.StorageWrapper.StorageWrapper}
    @ivar storagewrapper: the StorageWrapper instance
    @type fileselector: L{BT1.FileSelector.FileSelector}
    @ivar fileselector: the FileSelector instance
    @type super_seeding_active: C{boolean}
    @ivar super_seeding_active: whether the download is in super-seed mode
    @type filedatflag: C{threading.Event}
    @ivar filedatflag: not used
    @type superseedflag: C{threading.Event}
    @ivar superseedflag: indicates the upload is in super-seed mode
    @type whenpaused: C{float}
    @ivar whenpaused: the time when the download was paused
    @type finflag: C{threading.Event}
    @ivar finflag: whether the download is complete
    @type rerequest: L{BT1.Rerequester.Rerequester}
    @ivar rerequest: the Rerequester instance to use to communicate with the tracker
    @type tcp_ack_fudge: C{float}
    @ivar tcp_ack_fudge: the fraction of TCP ACK download overhead to add to 
        upload rate calculations
    @type appdataobj: L{ConfigDir.ConfigDir}
    @ivar appdataobj: the configuration and cache directory manager
    @type excflag: C{threading.Event}
    @ivar excflag: whether an exception has occurred
    @type failed: C{boolean}
    @ivar failed: whether the download failed
    @type checking: C{boolean}
    @ivar checking: whether the download is in the initialization phase
    @type started: C{boolean}
    @ivar started: whether the download has been started
    @type picker: L{BT1.PiecePicker.PiecePicker}
    @ivar picker: the PiecePicker instance
    @type choker: L{BT1.Choker.Choker}
    @ivar choker: the Choker instance
    @type filename: C{string}
    @ivar filename: the save location
    @type files: C{list} of (C{string}, C{long})
    @ivar files: the full file names and lengths of all the files in the download
    @type datalength: C{long}
    @ivar datalength: the total length of the download
    @type storage: L{BT1.Storage.Storage}
    @ivar storage: the file storage instance
    @type upmeasure: L{CurrentRateMeasure.Measure}
    @ivar upmeasure: the measure of the upload rate
    @type downmeasure: L{CurrentRateMeasure.Measure}
    @ivar downmeasure: the measure of the download rate
    @type ratelimiter: L{RateLimiter.RateLimiter}
    @ivar ratelimiter: the RateLimiter instance to limit the upload rate
    @type ratemeasure: L{RateMeasure.RateMeasure}
    @ivar ratemeasure: the RateMeasure instance
    @type ratemeasure_datarejected: C{method}
    @ivar ratemeasure_datarejected: the method to call when incoming data failed
    @type connecter: L{BT1.Connecter.Connecter}
    @ivar connecter: the Connecter instance to manage all the connections
    @type encoder: L{BT1.Encrypter.Encoder}
    @ivar encoder: the port listener for connections
    @type encoder_ban: C{method}
    @ivar encoder_ban: the method to call to ban and IP address
    @type httpdownloader: L{BT1.HTTPDownloader.HTTPDownloader}
    @ivar httpdownloader: the backup HTTP downloader
    @type statistics: L{BT1.Statistics.Statistics}
    @ivar statistics: the statistics gathering instance
    
    """
    
    def __init__(self, statusfunc, finfunc, errorfunc, doneflag,
                 config, response, infohash, id, rawserver, port,
                 appdataobj):
        """Initialize the instance.
        
        @type statusfunc: C{method}
        @param statusfunc: the method to call to print status updates
        @type finfunc: C{method}
        @param finfunc: the method to call when the download is completed
        @type errorfunc: C{method}
        @param errorfunc: the method to call when an error occurs
        @type doneflag: C{threading.Event}
        @param doneflag: the flag that indicates when the program is to be shutdown
        @type config: C{dictionary}
        @param config: the configuration variables
        @type response: C{dictionary}
        @param response: the response data from the metainfo file
        @type infohash: C{string}
        @param infohash: the hash of the info from the response data
        @type id: C{string}
        @param id: the peer ID to use
        @type rawserver: L{RawServer.RawServer}
        @param rawserver: the server controlling the program
        @type port: C{int}
        @param port: the port being listened to
        @type appdataobj: L{ConfigDir.ConfigDir}
        @param appdataobj: the configuration and cache directory manager
        
        """
        
        self.statusfunc = statusfunc
        self.finfunc = finfunc
        self.errorfunc = errorfunc
        self.doneflag = doneflag
        self.config = config
        self.response = response
        self.infohash = infohash
        self.myid = id
        self.rawserver = rawserver
        self.port = port
        
        self.info = self.response['info']
        self.pieces = [self.info['pieces'][x:x+20]
                       for x in xrange(0, len(self.info['pieces']), 20)]
        self.piece_lengths = self.info['piece lengths']
        self.len_pieces = len(self.pieces)
        self.argslistheader = argslistheader
        self.unpauseflag = Event()
        self.unpauseflag.set()
        self.pickled_data = None
        self.downloader = None
        self.storagewrapper = None
        self.fileselector = None
        self.super_seeding_active = False
        self.filedatflag = Event()
        self.superseedflag = Event()
        self.whenpaused = None
        self.finflag = Event()
        self.rerequest = None
        self.tcp_ack_fudge = config['tcp_ack_fudge']

        self.appdataobj = appdataobj

        self.excflag = self.rawserver.get_exception_flag()
        self.failed = False
        self.checking = False
        self.started = False

        self.picker = PiecePicker(self.len_pieces, config['rarest_first_cutoff'],
                             config['rarest_first_priority_cutoff'])
        self.choker = Choker(config, rawserver.add_task,
                             self.picker, self.finflag.isSet)


    def checkSaveLocation(self, loc):
        """Check whether the download location exists.
        
        For multiple files, returns true if a single one exists.
        
        @type loc: C{string}
        @param loc: the save location to check
        @rtype: C{boolean}
        @return: whether the location exists.
        
        """
        
        if self.info.has_key('length'):
            return path.exists(loc)
        for x in self.info['files']:
            if path.exists(path.join(loc, x['path'][0])):
                return True
        return False
                

    def saveAs(self, filefunc, pathfunc = None):
        """Initialize the location to save the download to.
        
        @type filefunc: C{method}
        @param filefunc: the method to call to get the location to save the 
            download
        @type pathfunc: C{method}
        @param pathfunc: the method to call to alert the UI to any possible 
            change in the download location
        @rtype: C{string}
        @return: the location to save to
        
        """
        
        try:
            def make(f, forcedir = False):
                """Create the parent directories of a file.
                
                @type f: C{string}
                @param f: the file name
                @type forcedir: C{boolean}
                @param forcedir: set to True if f is a directory name
                    (optional, defaults to False)
                
                """
                
                if not forcedir:
                    f = path.split(f)[0]
                if f != '' and not path.exists(f):
                    makedirs(f)

            if self.info.has_key('length'):
                file_length = self.info['length']
                file = filefunc(self.response['name'], file_length,
                                self.config['download_dir'], False)
                if file is None:
                    return None
                make(file)
                files = [(file, file_length)]
            else:
                file_length = 0L
                for x in self.info['files']:
                    file_length += x['length']
                file = filefunc(self.response['name'], file_length,
                                self.config['download_dir'], True)
                if file is None:
                    return None

                # if this path exists, and no files from the info dict exist, we assume it's a new download and 
                # the user wants to create a new directory with the default name
                existing = 0
                if path.exists(file):
                    if not path.isdir(file):
                        logger.error(file + 'is not a dir')
                        self.errorfunc(file + 'is not a dir')
                        return None
#                    Don't create a directory with the default name within it
#                    if len(listdir(file)) > 0:  # if it's not empty
#                        for x in self.info['files']:
#                            if path.exists(path.join(file, x['path'][0])):
#                                existing = 1
#                        if not existing:
#                            file = path.join(file, self.response['name'])
#                            if path.exists(file) and not path.isdir(file):
#                                if file[-8:] == '.dtorrent':
#                                    file = file[:-8]
#                                if path.exists(file) and not path.isdir(file):
#                                    logger.error("Can't create dir - " + self.response['name'])
#                                    self.errorfunc("Can't create dir - " + self.response['name'])
#                                    return None
                make(file, True)

                # alert the UI to any possible change in path
                if pathfunc != None:
                    pathfunc(file)

                files = []
                for x in self.info['files']:
                    if x['path']:
                        n = file
                        for i in x['path']:
                            n = path.join(n, i)
                        files.append((n, x['length']))
                    else:
                        files.append(('', 0L))
                    #Move directory creation to Storage
                    #make(n)
        except OSError, e:
            logger.exception("Couldn't allocate dir")
            self.errorfunc("Couldn't allocate dir - " + str(e))
            return None

        self.filename = file
        self.files = files
        self.datalength = file_length
        return file
    

    def getFilename(self):
        """Get the download location.
        
        @rtype: C{string}
        @return: the download location
        
        """
        
        return self.filename


    def _finished(self):
        """Finish the download.
        
        Set the files to read-only, update the Choker for seeding, stop the 
        piece requester.
        
        """
        
        self.finflag.set()
        try:
            self.storage.set_readonly()
        except (IOError, OSError), e:
            logger.exception('trouble setting readonly at end')
            self.errorfunc('trouble setting readonly at end - ' + str(e))
        if self.superseedflag.isSet():
            self._set_super_seed()
        self.choker.set_round_robin_period(
            max( self.config['round_robin_period'],
                 self.config['round_robin_period'] *
                           (float(self.datalength) / len(self.piece_lengths)) / 200000 ) )
        self.rerequest_complete()
        self.finfunc()

    def _data_flunked(self, amount, index):
        """Process a failed hash check on a piece.
        
        @type amount: C{int}
        @param amount: the amount of failed data
        @type index: C{int}
        @param index: the piece that failed
        
        """
        
        self.ratemeasure_datarejected(amount)
        if not self.doneflag.isSet():
            logger.warning('piece %d failed hash check, re-downloading it', index)
            self.errorfunc('piece %d failed hash check, re-downloading it' % index)

    def _failed(self, reason):
        """Stop the failed download.
        
        @type reason: C{string}
        @param reason: the reason for the failure
        
        """
        
        self.failed = True
        self.doneflag.set()
        if reason is not None:
            logger.critical(reason)
            self.errorfunc(reason)
        

    def loadState(self):
        """Load the save download state from a previous run.
        
        @rtype: C{list} of C{boolean}
        @return: a list of the enabled files in the torrent
            (first call only, subsequent calls will return None)
        
        """

        # If it's already loaded, do nothing
        if self.pickled_data is not None:
            return
        
        try:
            self.pickled_data = self.appdataobj.getTorrentData(self.infohash)
        except:
            logger.exception('Could not retrieve the pickled data')
            self.pickled_data = None
            
        must_find_files = False

        # Upgrade the saved state
        if self.pickled_data and self.pickled_data.get('version', 0) < 1:
            if 'resume data' in self.pickled_data:
                new_priority = {}
                if 'priority' in self.pickled_data['resume data']:
                    try:
                        old_priority = self.pickled_data['resume data']['priority']
                        for i in xrange(min(len(old_priority), len(self.files))):
                            if old_priority[i] >= 0:
                                new_priority[self.files[i][0]] = old_priority[i]
                        logger.info('Upgraded the old priority state')
                    except:
                        logger.exception('Previously saved priority state is corrupt, resetting')
                        must_find_files = True
                else:
                    must_find_files = True
                self.pickled_data['resume data']['priority'] = new_priority
                new_files = {}
                if 'files' in self.pickled_data['resume data']:
                    try:
                        old_files = self.pickled_data['resume data']['files']
                        assert len(old_files) % 3 == 0
                        old_files = [old_files[x:x+3] for x in xrange(0,len(old_files),3)]
                        for i, size, mtime in old_files:
                            if i < len(self.files):
                                new_files[self.files[i][0]] = [size, mtime]
                        logger.info('Upgraded the old saved files state')
                    except:
                        logger.exception('Previously saved file state is corrupt, resetting')
                self.pickled_data['resume data']['files'] = new_files
                if 'partial files' in self.pickled_data['resume data']:
                    del self.pickled_data['resume data']['partial files']
            else:
                self.pickled_data['resume data'] = {'priority': {}, 'files': {}}
                must_find_files = True
            self.pickled_data['stats'] = {'upload': 0L, 'download': 0L, 'download_http': 0L}
            self.pickled_data['version'] = 1

        # Initialize the saved state it if it wasn't found
        if not self.pickled_data:
            must_find_files = True
            self.pickled_data = {}
            self.pickled_data['resume data'] = {'priority': {}, 'files': {}}
            self.pickled_data['stats'] = {'upload': 0L, 'download': 0L, 'download_http': 0L}
            self.pickled_data['version'] = 1

        enabled_files = []
        found = 0
        if must_find_files:
            # No state, so search the download directory for old files to use
            logger.info('no cached data, manually finding and hash checking old files')
            self.pickled_data['resume data']['priority'] = {}
            try:
                logger.info('Searching %s for already downloaded files' % self.filename)
                found_files = {}
                for root, dirs, files in walk(self.filename):
                    for file in files:
                        found_files[path.join(root, file)] = 1
                
                if found_files:
                    logger.info('Found %d downloaded files' % len(found_files.keys()))
                    priority = {}
                    for file, length in self.files:
                        if file in found_files:
                            found += 1
                            priority[file] = 1
                            enabled_files.append(True)
                        else:
                            enabled_files.append(False)
                    logging.info('%d of the found files were also present in the %d files of the torrent' % (found, len(self.files)))
                    self.pickled_data['resume data']['priority'] = priority
                else:
                    logger.info('No downloaded files were found')
                
            except:
                logger.exception('Error occurred when manually finding the old files')
                enabled_files = None
        else:
            # Have state, so build the list of enabled files
            d = self.pickled_data['resume data']['priority']
            for file, length in self.files:
                if file in d and d[file] >= 0:
                    found += 1
                    enabled_files.append(True)
                else:
                    enabled_files.append(False)
            logger.info('Of %d previous files, %d are still valid for this torrent' 
                        % (len(d.keys()), found))
        
        return enabled_files

    def initFiles(self, old_style = False):
        """Initialize the files for the download.
        
        Initialize the priorities, then create the Storage, StorageWrapper, 
        and FileSelector. Then initialize the StorageWrapper.
        
        @type old_style: C{boolean}
        @param old_style: whether to use the old-style StorageWrapper 
            initialization (optional, defaults to False)
        @rtype: C{boolean}
        @return: whether the initialization was successful
        
        """
        
        if self.doneflag.isSet():
            return None

        enabled_files = self.loadState()

        try:
            try:
                self.storage = Storage(self.files, self.piece_lengths,
                                       self.doneflag, self.config, enabled_files)
            except IOError, e:
                logger.exception('trouble accessing files')
                self.errorfunc('trouble accessing files - ' + str(e))
                return None
            if self.doneflag.isSet():
                return None

            self.storagewrapper = StorageWrapper(self.storage, self.config['download_slice_size'],
                self.pieces, self.piece_lengths, self.datalength, self._finished, self._failed,
                self.statusfunc, self.doneflag, self.config['check_hashes'],
                self._data_flunked, self.rawserver.add_task,
                self.config, self.unpauseflag)
            
        except ValueError, e:
            logger.exception('bad data')
            self._failed('bad data - ' + str(e))
        except IOError, e:
            logger.exception('IOError')
            self._failed('IOError - ' + str(e))
        if self.doneflag.isSet():
            return None

        self.fileselector = FileSelector(self.files, self.piece_lengths,
                                         self.appdataobj.getPieceDir(self.infohash),
                                         self.storage, self.storagewrapper,
                                         self.rawserver.add_task, self.picker,
                                         self._failed)
        self.fileselector.unpickle(self.pickled_data.get('resume data', {}))

        self.checking = True
        if old_style:
            return self.storagewrapper.old_style_init()
        return self.storagewrapper.initialize


    def _make_upload(self, connection, ratelimiter, totalup):
        """Create a new Upload instance
        
        @type connection: L{BT1.Connecter.Connection}
        @param connection: the connection to upload on
        @type ratelimiter: L{BT1.RateLimiter.RateLimiter}
        @param ratelimiter: the RateLimiter instance to use
        @type totalup: L{CurrentRateMeasure.Measure}
        @param totalup: the Measure instance to use to calculate the total 
            upload rate
        @rtype: L{BT1.Uploader.Upload}
        @return: the new Upload instance
        
        """
        
        return Upload(connection, ratelimiter, totalup,
                      self.choker, self.storagewrapper, self.picker,
                      self.config)

    def _kick_peer(self, connection):
        """Disconnect a peer.
        
        @type connection: L{BT1.Connecter.Connection}
        @param connection: the connection of the peer to disconnect
        
        """
        
        def k(connection = connection):
            """Close a connection.
            
            @type connection: L{Connecter.Connection}
            @param connection: the connection of the peer to disconnect
                (optional, defaults to the _kick_peer connection)
            
            """
        
            connection.close()
        self.rawserver.add_task(k,0)

    def _ban_peer(self, ip):
        """Ban a peer from the download.
        
        @type ip: C{string}
        @param ip: the IP address of the peer to ban
        
        """
        
        self.encoder_ban(ip)

    def _received_raw_data(self, x):
        """Update the rate limiter when data comes in.
        
        @type x: C{int}
        @param x: the number of bytes that were received
        
        """
        
        if self.tcp_ack_fudge:
            x = int(x*self.tcp_ack_fudge)
            self.ratelimiter.adjust_sent(x)

    def _received_data(self, x):
        """Add received data to the rate measures.
        
        @type x: C{int}
        @param x: the number of bytes that were received
        
        """
        
        self.downmeasure.update_rate(x)
        self.ratemeasure.data_came_in(x)

    def _received_http_data(self, x):
        """Add received HTTP data to the rate measures.
        
        @type x: C{int}
        @param x: the number of bytes that were received
        
        """
        
        self.downmeasure.update_http_rate(x)
        self.ratemeasure.data_came_in(x)
        self.downloader.external_data_received(x)

    def _cancelfunc(self, pieces):
        """Cancel the download of pieces.
        
        @type pieces: C{list} of C{int}
        @param pieces: the pieces to stop downloading
        
        """
        
        self.downloader.cancel_piece_download(pieces)
        self.httpdownloader.cancel_piece_download(pieces)

    def _reqmorefunc(self, pieces):
        """Request to download the pieces.
        
        @type pieces: C{list} of C{int}
        @param pieces: the pieces to request
        
        """
        
        self.downloader.requeue_piece_download(pieces)

    def startEngine(self, ratelimiter = None):
        """Start various downloader engines.
        
        Starts the upload and download L{CurrentRateMeasure.Measure}, 
        L{RateLimiter.RateLimiter}, L{BT1.Downloader.Downloader}, 
        L{BT1.Connecter.Connecter}, L{BT1.Encrypter.Encoder}, and
        L{BT1.HTTPDownloader.HTTPDownloader}.
        
        @type ratelimiter: L{RateLimiter.RateLimiter}
        @param ratelimiter: the RateLimiter instance to use 
            (optional, defaults to starting a new one)
        @rtype: C{boolean}
        @return: whether the engines were started
        
        """

        if self.doneflag.isSet():
            return False

        self.checking = False

        if not CRYPTO_OK:
            if self.config['crypto_allowed']:
                logger.warning('crypto library not installed')
                self.errorfunc('warning - crypto library not installed')
            self.config['crypto_allowed'] = 0
            self.config['crypto_only'] = 0
            self.config['crypto_stealth'] = 0

        for i in xrange(self.len_pieces):
            if self.storagewrapper.do_I_have(i):
                self.picker.complete(i)
        
        # Just to be sure
        self.loadState()
        
        total_up = long(self.pickled_data['stats']['upload'])
        total_down = long(self.pickled_data['stats']['download'])
        # The total_http stat isn't in files created with debtorrent-0.1.8
        total_http = 0L;
        if self.pickled_data['stats'].has_key('download_http'):
            total_http = long(self.pickled_data['stats']['download_http'])
        self.upmeasure = Measure(self.config['max_rate_period'],
                            self.config['upload_rate_fudge'], saved_total = total_up)
        self.downmeasure = Measure(self.config['max_rate_period'], saved_total = total_down, http_total = total_http)

        if ratelimiter:
            self.ratelimiter = ratelimiter
        else:
            self.ratelimiter = RateLimiter(self.rawserver.add_task,
                                           self.config['upload_unit_size'],
                                           self.setConns)
            self.ratelimiter.set_upload_rate(self.config['max_upload_rate'])
        
        self.ratemeasure = RateMeasure()
        self.ratemeasure_datarejected = self.ratemeasure.data_rejected

        self.downloader = Downloader(self.storagewrapper, self.picker,
            self.config['max_rate_period'],
            self.len_pieces, self.config['download_slice_size'],
            self._received_data, self.config['snub_time'], self.config['auto_kick'],
            self._kick_peer, self._ban_peer)
        self.downloader.set_download_rate(self.config['max_download_rate'])
        self.connecter = Connecter(self._make_upload, self.downloader, self.choker,
                            self.len_pieces, self.upmeasure, self.config,
                            self.ratelimiter, self.rawserver.add_task)
        self.encoder = Encoder(self.connecter, self.rawserver,
            self.myid, self.config['max_message_length'], self.rawserver.add_task,
            self.config['keepalive_interval'], self.infohash,
            self._received_raw_data, self.config)
        self.encoder_ban = self.encoder.ban

        self.httpdownloader = HTTPDownloader(self.storagewrapper, self.picker,
            self.rawserver, self.finflag, self.errorfunc, self.downloader,
            self.config['max_rate_period'], self.len_pieces, self._received_http_data,
            self.connecter.got_piece, self.getFilename)
        if self.response.has_key('deb_mirrors') and not self.finflag.isSet():
            if not self.config['disable_http_downloader']:
                for u in self.response['deb_mirrors']:
                    self.httpdownloader.make_download(u)

        self.fileselector.tie_in(self._cancelfunc, self._reqmorefunc, 
                                 self.rerequest_ondownloadmore)

        self.appdataobj.deleteTorrentData(self.infohash)
                            # erase old data once you've started modifying it

        if self.config['super_seeder']:
            self.set_super_seed()

        self.started = True
        return True


    def rerequest_complete(self):
        """Send the completed event to the tracker."""
        if self.rerequest:
            self.rerequest.announce(1)

    def rerequest_stopped(self):
        """Send the stopped event to the tracker."""
        if self.rerequest:
            self.rerequest.announce(2)

    def rerequest_lastfailed(self):
        """Check if the last tracker request failed.
        
        @rtype: C{boolean}
        @return: whether it failed (or False if there is no Rerequester)
        
        """
        
        if self.rerequest:
            return self.rerequest.last_failed
        return False

    def rerequest_ondownloadmore(self):
        """Try to trigger a tracker request."""
        if self.rerequest:
            self.rerequest.hit()

    def startRerequester(self, seededfunc = None, force_rapid_update = False):
        """Start the tracker requester.
        
        @type seededfunc: C{method}
        @param seededfunc: method to call if the tracker reports the torrent
            is seeded (optional, defaults to not checking)
        @type force_rapid_update: C{boolean}
        @param force_rapid_update: whether to do quick tracker updates when 
            requested (optional, defaults to False)
        
        """
        
        if self.response.has_key('announce-list'):
            trackerlist = self.response['announce-list']
        else:
            trackerlist = [[self.response['announce']]]

        self.rerequest = Rerequester(self.port, self.myid, self.infohash, 
            trackerlist, self.config, 
            self.rawserver.add_task, self.rawserver.add_task,
            self.errorfunc,
            self.encoder.start_connections,
            self.connecter.how_many_connections, 
            self.storagewrapper.get_amount_left, 
            self.upmeasure.get_total, self.downmeasure.get_total,
            self.upmeasure.get_rate, self.downmeasure.get_rate,
            self.doneflag, self.unpauseflag, seededfunc, force_rapid_update,
            self.response.get('name', ''))

        self.rerequest.start()


    def _init_stats(self):
        """Start the statistics aggregater."""
        self.statistics = Statistics(self.upmeasure, self.downmeasure,
                    self.connecter, self.httpdownloader, self.ratelimiter,
                    self.rerequest_lastfailed, self.filedatflag)
        if self.info.has_key('files'):
            self.statistics.set_dirstats(self.files, self.piece_lengths)

    def startStats(self):
        """Start a statistics gatherer.
        
        @rtype: C{method}
        @return: the method to call to get the gathered statistics
        
        """
        
        self._init_stats()
        d = DownloaderFeedback(self.choker, self.httpdownloader, self.rawserver.add_task,
            self.upmeasure.get_rate, self.downmeasure.get_rate,
            self.ratemeasure, self.storagewrapper.get_stats,
            self.datalength, self.finflag, self.statistics)
        return d.gather


    def getPortHandler(self):
        """Get the object that is called when a connection comes in.
        
        @rtype: L{BT1.Encrypter.Encoder}
        @return: the object responsible for listening to a port
        
        """
        
        return self.encoder


    def shutdown(self):
        """Shutdown the running download.
        
        @rtype: C{boolean}
        @return: False if a failure or exception occurred
        
        """
        
        logger.info('Download shutdown')
        if self.checking or self.started:
            self.storagewrapper.sync()
            self.storage.close()
            self.rerequest_stopped()
        if self.fileselector and self.started:
            torrentdata = {}
            torrentdata['version'] = 1
            torrentdata['stats'] = {'upload': self.upmeasure.get_total(),
                                    'download': self.downmeasure.get_total(),
                                    'download_http': self.downmeasure.get_http_subtotal()}
            if not self.failed:
                self.fileselector.finish()
                torrentdata['resume data'] = self.fileselector.pickle()
            try:
                self.appdataobj.writeTorrentData(self.infohash,torrentdata)
            except:
                self.appdataobj.deleteTorrentData(self.infohash) # clear it
        return not self.failed and not self.excflag.isSet()
        # if returns false, you may wish to auto-restart the torrent


    def setUploadRate(self, rate):
        """Set a new maximum upload rate.
        
        @type rate: C{float}
        @param rate: the new upload rate (kB/s)
        
        """
        
        try:
            def s(self = self, rate = rate):
                """Worker function to actually set the rate.
                
                @type self: L{BT1Download}
                @param self: the BT1Download instance to set the rate of 
                    (optional, defaults to the current instance)
                @type rate: C{float}
                @param rate: the new rate to set 
                    (optional, defaults to the L{setUploadRate} rate
                
                """
                
                self.config['max_upload_rate'] = rate
                self.ratelimiter.set_upload_rate(rate)
 
            self.rawserver.add_task(s)
        except AttributeError:
            pass

    def setConns(self, conns, conns2 = None):
        """Set the number of connections limits.
        
        @type conns: C{int}
        @param conns: the number of uploads to fill out to with extra 
            optimistic unchokes
        @type conns2: C{int}
        @param conns2: the maximum number of uploads to allow at once 
            (optional, defaults to the value of L{conns})
        
        """
        
        if not conns2:
            conns2 = conns
        try:
            def s(self = self, conns = conns, conns2 = conns2):
                """Worker function to actually set the connection limits.
                
                @type self: L{BT1Download}
                @param self: the BT1Download instance to set the rate of 
                    (optional, defaults to the current instance)
                @type conns: C{int}
                @param conns: the number of uploads to fill out to with extra 
                    optimistic unchokes
                @type conns2: C{int}
                @param conns2: the maximum number of uploads to allow at once
                
                """
                
                self.config['min_uploads'] = conns
                self.config['max_uploads'] = conns2
                if (conns > 30):
                    self.config['max_initiate'] = conns + 10
                    
            self.rawserver.add_task(s)
        except AttributeError:
            pass
        
    def setDownloadRate(self, rate):
        """Set a new maximum download rate.
        
        @type rate: C{float}
        @param rate: the new download rate (kB/s)
        
        """
        
        try:
            def s(self = self, rate = rate):
                """Worker function to actually set the rate.
                
                @type self: L{BT1Download}
                @param self: the BT1Download instance to set the rate of 
                    (optional, defaults to the current instance)
                @type rate: C{float}
                @param rate: the new rate to set 
                    (optional, defaults to the L{setDownloadRate} rate
                
                """
                
                self.config['max_download_rate'] = rate
                self.downloader.set_download_rate(rate)

            self.rawserver.add_task(s)
        except AttributeError:
            pass

    def startConnection(self, ip, port, id):
        """Start a new connection to a peer.
        
        @type ip: C{string}
        @param ip: the IP address of the peer
        @type port: C{int}
        @param port: the port to contact the peer on
        @type id: C{string}
        @param id: the peer's ID
        
        """
        
        self.encoder._start_connection((ip, port), id)
      
    def _startConnection(self, ipandport, id):
        """Start a new connection to a peer.
        
        @type ipandport: (C{string}, C{int})
        @param ipandport: the IP address and port to contact the peer on
        @type id: C{string}
        @param id: the peer's ID
        
        """
        
        self.encoder._start_connection(ipandport, id)
        
    def setInitiate(self, initiate):
        """Set the maximum number of connections to initiate.
        
        @type initiate: C{int}
        @param initiate: the new maximum
        
        """
        
        try:
            def s(self = self, initiate = initiate):
                """Worker function to actually set the maximum.
                
                @type self: L{BT1Download}
                @param self: the BT1Download instance to set the max connections of 
                    (optional, defaults to the current instance)
                @type initiate: C{int}
                @param initiate: the new maximum
                
                """
        
                self.config['max_initiate'] = initiate

            self.rawserver.add_task(s)
        except AttributeError:
            pass

    def getConfig(self):
        """Get the current configuration parameters.
        
        @rtype: C{dictionary}
        @return: the configuration parameters
        
        """
        
        return self.config

    def getDefaults(self):
        """Get the default configuration parameters.
        
        @rtype: C{dictionary}
        @return: the default configuration parameters
        
        """
        
        return defaultargs(defaults)

    def getUsageText(self):
        """Get the header only for the usage text (not used).
        
        @rtype: C{string}
        @return: the header of the usage text
        
        """
        
        return self.argslistheader

    def reannounce(self, special = None):
        """Reannounce to the tracker.
        
        @type special: C{string}
        @param special: the URL of the tracker to announce to 
            (optional, defaults to the tracker list from the metainfo file)
        
        """
        
        try:
            def r(self = self, special = special):
                """Worker function to actually do the announcing.
                
                @type self: L{BT1Download}
                @param self: the BT1Download instance to announce
                    (optional, defaults to the current instance)
                @type special: C{string}
                @param special: the URL of the tracker to announce to 
                
                """

                if special is None:
                    self.rerequest.announce()
                else:
                    self.rerequest.announce(specialurl = special)

            self.rawserver.add_task(r)
        except AttributeError:
            pass

    def getResponse(self):
        """Get the response data from the metainfo file.
        
        @rtype: C{dictionary}
        @return: the response data (or None if there isn't any
        
        """
        
        try:
            return self.response
        except:
            return None

    def getFilepath(self, f):
        """Get the stored file name from a file index.
        
        @type f: C{int}
        @param f: the index of the file
        @rtype: C{string}
        @return: the file name where the indexed file is stored.
        
        """
        
        try:
            return self.files[f][0]
        except:
            return None

    def Pause(self):
        """Schedule the pausing of the download.

        @rtype: C{boolean}
        @return: whether the download pause was schedules
        
        """
        
        if not self.storagewrapper:
            return False
        self.unpauseflag.clear()
        self.rawserver.add_task(self.onPause)
        return True

    def onPause(self):
        """Pause the download."""
        
        logger.info('Download paused')
        self.whenpaused = clock()
        if not self.downloader:
            return
        self.downloader.pause(True)
        self.encoder.pause(True)
        self.choker.pause(True)
    
    def Unpause(self):
        """Schedule the resuming of the download."""
        self.unpauseflag.set()
        self.rawserver.add_task(self.onUnpause)

    def onUnpause(self):
        """Resume the download."""
        logger.info('Download unpaused')
        if not self.downloader:
            return
        self.downloader.pause(False)
        self.encoder.pause(False)
        self.choker.pause(False)
        if self.rerequest and self.whenpaused and clock()-self.whenpaused > 60:
            self.rerequest.announce(3)      # rerequest automatically if paused for >60 seconds

    def set_super_seed(self):
        """Schedule the change of the upload into super-seed mode."""
        try:
            self.superseedflag.set()
            def s(self = self):
                """Worker function to actually call the change to super-seed.
                
                @type self: L{BT1Download}
                @param self: the BT1Download instance to change to super-seed
                    (optional, defaults to the current instance)
                
                """
                if self.finflag.isSet():
                    self._set_super_seed()

            self.rawserver.add_task(s)
        except AttributeError:
            pass

    def _set_super_seed(self):
        """Change the upload into super-seed mode."""
        if not self.super_seeding_active:
            self.super_seeding_active = True
            logger.info('SUPER-SEED OPERATION ACTIVE: ' +
                        'please set Max uploads so each peer gets 6-8 kB/s')
            self.errorfunc('SUPER-SEED OPERATION ACTIVE: ' +
                           'please set Max uploads so each peer gets 6-8 kB/s')
            def s(self = self):
                """Worker function to actually change to super-seed.
                
                @type self: L{BT1Download}
                @param self: the BT1Download instance to change to super-seed
                    (optional, defaults to the current instance)
                
                """

                self.downloader.set_super_seed()
                self.choker.set_super_seed()

            self.rawserver.add_task(s)
            if self.finflag.isSet():        # mode started when already finished
                def r(self = self):
                    """Worker function to actually do the announcing.
                    
                    @type self: L{BT1Download}
                    @param self: the BT1Download instance to announce
                        (optional, defaults to the current instance)
                    
                    """

                    self.rerequest.announce(3)  # so after kicking everyone off, reannounce

                self.rawserver.add_task(r)

    def am_I_finished(self):
        """Determine if the download is complete or still under way.
        
        @rtype: C{boolean}
        @return: whether the download is complete
        
        """
        
        return self.finflag.isSet()

    def get_transfer_stats(self):
        """Get the total amount of data transferred.
        
        @rtype: (C{long}, C{long})
        @return: the measured total upload and download bytes
        
        """
        
        return self.upmeasure.get_total(), self.downmeasure.get_total()
