#written by John Hoffman
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: ConfigDir.py 266 2007-08-18 02:06:35Z camrdale-guest $

"""Manage configuration and cache files.

@type DIRNAME: C{string}
@var DIRNAME: the directory name to use for storing config files

"""

from inifile import ini_write, ini_read
from bencode import bencode, bdecode
from types import IntType, LongType, StringType, FloatType
from parseargs import defaultargs
from __init__ import product_name, version_short
import sys,os
from time import time, strftime
from binascii import b2a_hex, a2b_hex

DIRNAME = '.'+product_name
MASTER_CONFIG = '/etc/debtorrent'

def copyfile(oldpath, newpath):
    """Simple file copy, all in RAM.
    
    @type oldpath: C{string}
    @param oldpath: the file name to copy from
    @type newpath: C{string}
    @param newpath: the file name to copy to
    @rtype: C{boolean}
    @return: whether the copy was successful
    
    """
    
    try:
        f = open(oldpath,'rb')
        r = f.read()
        success = True
    except:
        success = False
    try:
        f.close()
    except:
        pass
    if not success:
        return False
    try:
        f = open(newpath,'wb')
        f.write(r)
    except:
        success = False
    try:
        f.close()
    except:
        pass
    return success


class ConfigDir:
    """Manage configuration and cache files.
    
    @type config_type: C{string}
    @ivar config_type: the extension to include in the saved files' names
    @type home_dir: C{string}
    @ivar home_dir: the user's home directory
    @type cache_dir: C{string}
    @ivar cache_dir: the directory to save cache files in
    @type dir_torrentcache: C{string}
    @ivar dir_torrentcache: the directory to save torrent files in
    @type dir_datacache: C{string}
    @ivar dir_datacache: the directory to save stopped torrent's state in
    @type dir_piececache: C{string}
    @ivar dir_piececache: the directory to store temporary piece files in
    @type configfile: C{string}
    @ivar configfile: the file name for the saved configuration data
    @type statefile: C{string}
    @ivar statefile: the file name for the saved state
    @type TorrentDataBuffer: C{dictionary}
    @ivar TorrentDataBuffer: any torrent data read/written, keys are the
        torrent info hashes, values are the torrent data (C{dictionary})
    @type config: C{dictionary}
    @ivar config: the current configuration variables
    
    @group Config Handling: setDefaults, setCacheDir, checkConfig, loadConfigFile, loadConfig, saveConfig, getConfig
    @group State: getState, saveState
    @group Torrent Files: getTorrents, getTorrentVariations, getTorrentFile, getTorrent, writeTorrent
    @group Torrent Data: getTorrentData, writeTorrentData, deleteTorrentData, getPieceDir
    @group Expire Cache: deleteOldCacheData, deleteOldTorrents
    
    """

    ###### INITIALIZATION TASKS ######

    def __init__(self, config_type = None):
        """Initialize the instance, create directories and file names.
        
        @type config_type: C{string}
        @param config_type: the extension to include in the saved files' names
            (optional, default is to use no extension)
        
        """
        
        if config_type:
            self.config_type = '.'+config_type
        else:
            self.config_type = ''

        def check_sysvars(x):
            """Check a system variable to see if it expands to something.
            
            @type x: C{string}
            @param x: the system variable to check
            @rtype: C{string}
            @return: the expanded variable, or None if it doesn't expand
            
            """
            
            y = os.path.expandvars(x)
            if y != x and os.path.isdir(y):
                return y
            return None

        for d in ['${HOME}']:
            self.home_dir = check_sysvars(d)
            if self.home_dir:
                break
        else:
            self.home_dir = os.path.expanduser('~')
            if not os.path.isdir(self.home_dir):
                self.home_dir = os.path.abspath(os.path.dirname(sys.argv[0]))

        if not os.path.isdir(os.path.join(self.home_dir,DIRNAME)):
            os.mkdir(os.path.join(self.home_dir,DIRNAME))    # exception if failed

        self.TorrentDataBuffer = {}


    ###### CONFIG HANDLING ######

    def setDefaults(self, defaults, ignore=[]):
        """Set the default values to use for the configuration.
        
        @type defaults: C{dictionary}
        @param defaults: the default config values
        @type ignore: C{list}
        @param ignore: the keys in the defaults to ignore
            (optional, default is to ignore none of them)
        
        """
        
        self.config = defaultargs(defaults)
        for k in ignore:
            if self.config.has_key(k):
                del self.config[k]

    def setCacheDir(self, cache_dir, create_dirs = True):
        """Sets the various cache directory locations.
        
        @type cache_dir: C{string}
        @param cache_dir: the directory to save cache files in
        @type create_dirs: C{boolean}
        @param create_dirs: whether to create the client cache directories
            (optional, defaults to True)
        
        """

        if cache_dir:
            self.cache_dir = cache_dir
        else:
            self.cache_dir = os.path.join(self.home_dir,DIRNAME)

        if not os.path.isdir(self.cache_dir):
            os.mkdir(self.cache_dir)    # exception if failed

        if create_dirs:
            self.dir_torrentcache = os.path.join(self.cache_dir,'torrentcache')
            if not os.path.isdir(self.dir_torrentcache):
                os.mkdir(self.dir_torrentcache)
    
            self.dir_datacache = os.path.join(self.cache_dir,'datacache')
            if not os.path.isdir(self.dir_datacache):
                os.mkdir(self.dir_datacache)
    
            self.dir_piececache = os.path.join(self.cache_dir,'piececache')
            if not os.path.isdir(self.dir_piececache):
                os.mkdir(self.dir_piececache)

        self.statefile = os.path.join(self.cache_dir,'state'+self.config_type)

    def checkConfig(self, configfile):
        """Check if a config file already exists.
        
        @type configfile: C{string}
        @param configfile: the config file to use
        @rtype: C{boolean}
        @return: whether the config file exists
        
        """
        
        return os.path.exists(configfile)

    def loadConfigFile(self, configfile):
        """Load a configuration from a config file.
        
        @type configfile: C{string}
        @param configfile: the config file to use
        
        """
        
        try:
            r = ini_read(self.configfile)['']
        except:
            return self.config
        l = self.config.keys()
        for k,v in r.items():
            if self.config.has_key(k):
                t = type(self.config[k])
                try:
                    if t == StringType:
                        self.config[k] = v
                    elif t == IntType or t == LongType:
                        self.config[k] = long(v)
                    elif t == FloatType:
                        self.config[k] = float(v)
                    l.remove(k)
                except:
                    pass

    def loadConfig(self, params):
        """Load the configuration from any config files.
        
        @type params: C{list} of C{strings}
        @param params: a list of the command-line arguments
        @rtype: C{dictionary}
        @return: the loaded configuration variables
        @raise IOError: if the specified config file can not be found
        
        """
        
        try:
            self.configfile = params[params.index('--configfile')+1]
            configfile = True
        except:
            configfile = False
            self.configfile = os.path.join(os.path.join(self.home_dir,DIRNAME),
                                           self.config_type+'.conf')
        
            master_configfile = os.path.join(MASTER_CONFIG,
                                             self.config_type+'.conf')
            
            if self.checkConfig(master_configfile):
                self.loadConfigFile(master_configfile)

        if self.checkConfig(self.configfile):
            self.loadConfigFile(self.configfile)
        elif configfile:
            raise IOError('config file could not be found')

        return self.config

    def saveConfig(self, new_config = None):
        """Sets and writes to the file the new configuration.
        
        @type new_config: C{dictionary}
        @param new_config: the configuration to set and write
            (optional, default is to use the previously set one)
        @rtype: boolean
        @return: whether writing to the file was successful
        
        """
        
        if new_config:
            for k,v in new_config.items():
                if self.config.has_key(k):
                    self.config[k] = v
        try:
            ini_write( self.configfile, self.config,
                       'Generated by '+product_name+'/'+version_short+'\n'
                       + strftime('%x %X') )
            return True
        except:
            return False

    def getConfig(self):
        """Get the current configuration variables.
        
        @rtype: C{dictionary}
        @return: the current configuration variables
        
        """

        return self.config


    ###### STATE HANDLING ######

    def getState(self):
        """Get the state from the state file.
        
        @rtype: C{dictionary}
        @return: the previosuly saved state, or None if there was no previously
            saved state
        
        """
        
        try:
            f = open(self.statefile,'rb')
            r = f.read()
        except:
            r = None
        try:
            f.close()
        except:
            pass
        try:
            r = bdecode(r)
        except:
            r = None
        return r        

    def saveState(self, state):
        """Saves the state to the state file.
        
        @type state: C{dictionary}
        @param state: the state to save
        @rtype: boolean
        @return: whether the saving was successful
        
        """

        try:
            f = open(self.statefile,'wb')
            f.write(bencode(state))
            success = True
        except:
            success = False
        try:
            f.close()
        except:
            pass
        return success


    ###### TORRENT HANDLING ######

    def getTorrents(self):
        """Get a list of the torrents that have cache data.
        
        @rtype: C{list} of C{string}
        @return: the torrent hashes found
        
        """
        
        d = {}
        for f in os.listdir(self.dir_torrentcache):
            f = os.path.basename(f)
            try:
                f, garbage = f.split('.')
            except:
                pass
            d[a2b_hex(f)] = 1
        return d.keys()

    def getTorrentVariations(self, t):
        """Get the torrent variations in the cache data for a given hash.

        @type t: C{string}
        @param t: the torrent hash to check for
        @rtype: C{list} of C{int}
        @return: the variations of the hash found
        
        """
        
        t = b2a_hex(t)
        d = []
        for f in os.listdir(self.dir_torrentcache):
            f = os.path.basename(f)
            if f[:len(t)] == t:
                try:
                    garbage, ver = f.split('.')
                except:
                    ver = '0'
                d.append(int(ver))
        d.sort()
        return d

    def getTorrentFile(self, t, v = -1):
        """Get the undecoded torrent file for the hash.

        @type t: C{string}
        @param t: the torrent hash to lookup
        @type v: C{int}
        @param v: the variation to get (optional, default is the largest)
        @rtype: C{string}
        @return: the contents of the torrent file
        
        """
        
        if v == -1:
            v = max(self.getTorrentVariations(t))   # potential exception
        t = b2a_hex(t)
        if v:
            t += '.'+str(v)
        try:
            f = open(os.path.join(self.dir_torrentcache,t),'rb')
            r = f.read()
        except:
            r = None
        try:
            f.close()
        except:
            pass
        return r

    def getTorrent(self, t, v = -1):
        """Get the torrent data for the hash.

        @type t: C{string}
        @param t: the torrent hash to lookup
        @type v: C{int}
        @param v: the variation to get (optional, default is the largest)
        @rtype: C{dictionary}
        @return: the torrent metainfo found
        
        """
        
        f = self.getTorrentFile(t, v)
        if f:
            try:
                r = bdecode(f)
            except:
                r = None
        else:
            r = None
        return r

    def writeTorrent(self, data, t, v = -1):
        """Save the torrent data.

        @type data: C{dictionary}
        @param data: the torrent metainfo
        @type t: C{string}
        @param t: the hash of the torrent metainfo
        @type v: C{int}
        @param v: the variation to save as, or None for no variation 
            (optional, default is the next after the largest)
        @rtype: C{int}
        @return: the variation used, or None if the write failed
        
        """

        if v == -1:
            try:
                v = max(self.getTorrentVariations(t))+1
            except:
                v = 0
        t = b2a_hex(t)
        if v:
            t += '.'+str(v)
        try:
            f = open(os.path.join(self.dir_torrentcache,t),'wb')
            f.write(bencode(data))
        except:
            v = None
        try:
            f.close()
        except:
            pass
        return v


    ###### TORRENT DATA HANDLING ######

    def getTorrentData(self, t):
        """Retrieve cached data for a torrent.
        
        @type t: C{string}
        @param t: the info hash to retrieve cached data for
        @rtype: C{dictionary}
        @return: the bdecoded cached data
        
        """
        
        if self.TorrentDataBuffer.has_key(t):
            return self.TorrentDataBuffer[t]
        t = os.path.join(self.dir_datacache,b2a_hex(t))
        if not os.path.exists(t):
            return None
        try:
            f = open(t,'rb')
            r = bdecode(f.read())
        except:
            r = None
        try:
            f.close()
        except:
            pass
        self.TorrentDataBuffer[t] = r
        return r

    def writeTorrentData(self, t, data):
        """Write cached data for a torrent.
        
        @type t: C{string}
        @param t: the info hash to write cached data for
        @type data: C{dictionary}
        @param data: the data to cache
        @rtype: C{boolean}
        @return: whether the write was successful
        
        """

        self.TorrentDataBuffer[t] = data
        try:
            f = open(os.path.join(self.dir_datacache,b2a_hex(t)),'wb')
            f.write(bencode(data))
            success = True
        except:
            success = False
        try:
            f.close()
        except:
            pass
        if not success:
            self.deleteTorrentData(t)
        return success

    def deleteTorrentData(self, t):
        """Delete the cached data for a torrent.
        
        @type t: C{string}
        @param t: the info hash to delete the cached data of
        
        """

        try:
            os.remove(os.path.join(self.dir_datacache,b2a_hex(t)))
        except:
            pass

    def getPieceDir(self, t):
        """Get the directory to save temporary pieces for a torrent.
        
        @type t: C{string}
        @param t: the info hash to get the piece cache data of
        @rtype: C{string}
        @return: the directory to save temporary pieces in
        
        """

        return os.path.join(self.dir_piececache,b2a_hex(t))


    ###### EXPIRATION HANDLING ######

    def deleteOldCacheData(self, days, still_active = [], delete_torrents = False):
        """Delete old cache data after a period of time.
        
        @type days: C{int}
        @param days: the number of days to delete cached data after
        @type still_active: C{list} of C{string}
        @param still_active: the hashes of torrents that are still running
            (optional, default is to delete all torrent's cached data)
        @type delete_torrents: C{boolean}
        @param delete_torrents: whether to delete the torrent files as well
        
        """
        
        if not days:
            return
        exptime = time() - (days*24*3600)
        names = {}
        times = {}

        for f in os.listdir(self.dir_torrentcache):
            p = os.path.join(self.dir_torrentcache,f)
            f = os.path.basename(f)
            try:
                f, garbage = f.split('.')
            except:
                pass
            try:
                f = a2b_hex(f)
                assert len(f) == 20
            except:
                continue
            if delete_torrents:
                names.setdefault(f,[]).append(p)
            try:
                t = os.path.getmtime(p)
            except:
                t = time()
            times.setdefault(f,[]).append(t)
        
        for f in os.listdir(self.dir_datacache):
            p = os.path.join(self.dir_datacache,f)
            try:
                f = a2b_hex(os.path.basename(f))
                assert len(f) == 20
            except:
                continue
            names.setdefault(f,[]).append(p)
            try:
                t = os.path.getmtime(p)
            except:
                t = time()
            times.setdefault(f,[]).append(t)

        for f in os.listdir(self.dir_piececache):
            p = os.path.join(self.dir_piececache,f)
            try:
                f = a2b_hex(os.path.basename(f))
                assert len(f) == 20
            except:
                continue
            for f2 in os.listdir(p):
                p2 = os.path.join(p,f2)
                names.setdefault(f,[]).append(p2)
                try:
                    t = os.path.getmtime(p2)
                except:
                    t = time()
                times.setdefault(f,[]).append(t)
            names.setdefault(f,[]).append(p)

        for k,v in times.items():
            if max(v) < exptime and not k in still_active and k in names:
                for f in names[k]:
                    try:
                        os.remove(f)
                    except:
                        try:
                            os.removedirs(f)
                        except:
                            pass


    def deleteOldTorrents(self, days, still_active = []):
        """Delete old cached data and torrents after a period of time.
        
        @type days: C{int}
        @param days: the number of days to delete cached data after
        @type still_active: C{list} of C{string}
        @param still_active: the hashes of torrents that are still running
            (optional, default is to delete all torrent's cached data)
        
        """
        
        self.deleteOldCacheData(days, still_active, True)
