# Written by Bram Cohen
# multitracker extensions by John Hoffman
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: makemetafile.py 426 2010-03-21 05:03:59Z camrdale $

"""Create a torrent file or data structure.

@type logger: C{logging.Logger}
@var logger: the logger to send all log messages to for this module
@type defaults: C{list} of (C{string}, unknown, C{string})
@var defaults: the default configuration variables, including descriptions
@type default_piece_len_exp: C{int}
@var default_piece_len_exp: the exponent of the default piece size to use
@type ignore: C{list} of C{string}
@var ignore: file names to ignore when creating torrents

"""

from os.path import getsize, split, join, abspath, isdir, exists
from os import listdir
from sha import sha
from copy import copy
from string import strip
from DebTorrent.bencode import bencode
from btformats import check_info, check_message
from threading import Event, Thread
from time import time
from traceback import print_exc
from DebTorrent.zurllib import urlopen
import gzip
from bz2 import BZ2File
from StringIO import StringIO
from re import subn
from debian import deb822
import binascii, logging
try:
    from sys import getfilesystemencoding
    ENCODING = getfilesystemencoding()
except:
    from sys import getdefaultencoding
    ENCODING = getdefaultencoding()

logger = logging.getLogger('DebTorrent.BT1.makemetafile')

defaults = [
    ('announce', '', 'the announce URL to use for the torrent'),
    ('announce_list', '',
        'a list of announce URLs - explained below'),
    ('deb_mirrors', '',
        'a list of mirror URLs - explained below'),
    ('piece_size_pow2', 0,
        "which power of 2 to set the piece size to (0 = automatic)"),
    ('comment', '',
        "optional human-readable comment to put in .dtorrent"),
    ('filesystem_encoding', '',
        "optional specification for filesystem encoding " +
        "(set automatically in recent Python versions)"),
    ('target', '',
        "optional target file for the torrent"),
    ('pieces_file', '', 'the file that contains the sub-package piece information'),
    ('separate_all', 1, 'create a separate torrent for the architecture:all packages'),
    ('ordering_file', '', 'the file that contains the piece ordering info'),
    ('ordering_all_file', '', 'the file that contains the piece ordering info ' + 
         'for the architecture:all torrent (only used if separate_all = 1)'),
    ]

default_piece_len_exp = 18

ignore = ['core', 'CVS']

def print_announcelist_details():
    """Print the configuration options for the announce list and deb mirrors."""
    print ('    announce_list = optional list of redundant/backup tracker URLs, in the format:')
    print ('           url[,url...][|url[,url...]...]')
    print ('                where URLs separated by commas are all tried first')
    print ('                before the next group of URLs separated by the pipe is checked.')
    print ("                If none is given, it is assumed you don't want one in the metafile.")
    print ('                If announce_list is given, clients which support it')
    print ('                will ignore the <announce> value.')
    print ('           Examples:')
    print ('                http://tracker1.com|http://tracker2.com|http://tracker3.com')
    print ('                     (tries trackers 1-3 in order)')
    print ('                http://tracker1.com,http://tracker2.com,http://tracker3.com')
    print ('                     (tries trackers 1-3 in a randomly selected order)')
    print ('                http://tracker1.com|http://backup1.com,http://backup2.com')
    print ('                     (tries tracker 1 first, then tries between the 2 backups randomly)')
    print ('')
    print ('    deb_mirrors = optional list of mirror URLs, in the format:')
    print ('            url[|url...]')
    
def uniconvertl(l, e):
    """Convert a list of strings to Unicode.
    
    @type l: C{list} of C{string}
    @param l: the strings to convert to unicode
    @type e: C{string}
    @param e: the encoding to use for converting the input data
    @rtype: C{list} of C{string}
    @return: the converted strings encoded in UTF-8
    @raise UnicodeError: if a conversion error occurs
    
    """
    
    r = []
    try:
        for s in l:
            r.append(uniconvert(s, e))
    except UnicodeError:
        raise UnicodeError('bad filename: '+join(l))
    return r

def uniconvert(s, e = None):
    """Convert a string to Unicode.
    
    @type s: C{string}
    @param s: the string to convert to unicode
    @type e: C{string}
    @param e: the encoding to use for converting the input data
        (optional, defaults to the current file system encoding, or ASCII if
        it cannot be determined)
    @rtype: C{string}
    @return: the converted string encoded in UTF-8
    @raise UnicodeError: if a conversion error occurs
    
    """
    
    if not e:
        e = ENCODING
    if not e:
        e = 'ascii'
    
    if type(s) != unicode:
        try:
            s = unicode(s,e)
        except UnicodeError:
            raise UnicodeError('bad filename: '+s)
    return s.encode('utf-8')

def convert_all(f):
    """Find the architecture and replace it with 'all'.
    
    @type f: C{string}
    @param f: the string to search and replace the architecture in
    @rtype: C{string}
    @return: the converted string
    
    """
    
    (f_all, n) = subn(r'binary-[a-zA-Z0-9-]+([^a-zA-Z0-9-]?)', r'binary-all\1', f)
    if n == 0:
        # Otherwise add '-all' before the extension
        (f_all, n) = subn(r'\.([^.]*)$', r'-all.\1', f)
        if n == 0:
            # Otherwise add '-all' to the end
            f_all = f + '-all'
    return f_all

def universal_open(filename):
    """Open the file for reading, works for compressed files.
    
    Will check for a decompressed version of compressed files, otherwise uses
    an on-the-fly decompression file-like object.
    
    @type filename: C{string}
    @param filename: the path of the file to open
    @rtype: C{file}
    @return: the read-only file-like object
    
    """
    
    # Open and possibly decompress the file
    if filename.endswith('.gz'):
        if exists(filename[:-3]):
            f = open(filename[:-3])
        else:
            f = gzip.open(filename, 'r')
    elif filename.endswith('.bz2'):
        if exists(filename[:-4]):
            f = open(filename[:-4])
        else:
            f = BZ2File(filename)
    else:
        f = open(filename)
    return f

def make_meta_file(file, params = {}, progress = lambda x: None):
    """Create the torrent files from a Packages file.
    
    @type file: C{string}
    @param file: the Packages file to parse to create the torrent
    @type params: C{dictionary}
    @param params: the command-line parameters to use
    @type progress: C{method}
    @param progress: report the progress of the creation
    
    """
    
    if params.has_key('piece_size_pow2'):
        piece_len_exp = params['piece_size_pow2']
    else:
        piece_len_exp = default_piece_len_exp
    if params.has_key('target') and params['target'] != '':
        f = params['target']
    else:
        a, b = split(file)
        if b == '':
            f = a + '.dtorrent'
            name = a
        else:
            f = join(a, b + '.dtorrent')
            name = b
            
    if piece_len_exp == 0:  # automatic
        size = calcsize(file)
        if   size > 8L*1024*1024*1024:   # > 8 gig =
            piece_len_exp = 21          #   2 meg pieces
        elif size > 2*1024*1024*1024:   # > 2 gig =
            piece_len_exp = 20          #   1 meg pieces
        elif size > 512*1024*1024:      # > 512M =
            piece_len_exp = 19          #   512K pieces
        elif size > 64*1024*1024:       # > 64M =
            piece_len_exp = 18          #   256K pieces
        elif size > 16*1024*1024:       # > 16M =
            piece_len_exp = 17          #   128K pieces
        elif size > 4*1024*1024:        # > 4M =
            piece_len_exp = 16          #   64K pieces
        else:                           # < 4M =
            piece_len_exp = 15          #   32K pieces
    piece_length = 2 ** piece_len_exp

    encoding = None
    if params.has_key('filesystem_encoding'):
        encoding = params['filesystem_encoding']

    (info, info_all, ordering_headers, ordering_all_headers) = \
        makeinfo(file, piece_length, encoding, progress, 
                 params['separate_all'], params['pieces_file'],
                 params['ordering_file'], params['ordering_all_file'])

    if info:
        create_file(f, info, uniconvert(name, encoding), params,
                    ordering_headers)
        
    if info_all:
        create_file(convert_all(f), info_all,
                    uniconvert(convert_all(name), encoding), params,
                    ordering_all_headers)
        
def create_file(f, info, name, params, ordering_headers):
    """Actually write the torrent data to a file.
    
    @type f: C{string}
    @param f: the file name to write
    @type info: C{dictionary}
    @param info: the torrent data to write
    @type name: C{string}
    @param name: the internal name of the torrent
    @type params: C{dictionary}
    @param params: the command-line parameters
    @type ordering_headers: C{dictionary}
    @param ordering_headers: the headers from the ordering file for the torrent
    @raise ValueError: if the data is not correct
   
    """
    
    check_info(info)
    data = {'info': info, 'name': name, 'creation date': long(time())}
    
    if "Tracker" in ordering_headers:
        data['announce'] = ordering_headers["Tracker"].strip()
        del ordering_headers["Tracker"]
    if "Torrent" in ordering_headers:
        data['identifier'] = binascii.a2b_hex(ordering_headers["Torrent"].strip())
        del ordering_headers["Torrent"]
    for header, value in ordering_headers.items():
        data[header] = value.strip()
    
    if params.has_key('announce') and params['announce']:
        data['announce'] = params['announce'].strip()
    if params.has_key('comment') and params['comment']:
        data['comment'] = params['comment']
        
    if params.has_key('real_announce_list'):    # shortcut for progs calling in from outside
        data['announce-list'] = params['real_announce_list']
    elif params.has_key('announce_list') and params['announce_list']:
        l = []
        for tier in params['announce_list'].split('|'):
            l.append(tier.split(','))
        data['announce-list'] = l
        
    if params.has_key('real_deb_mirrors'):    # shortcut for progs calling in from outside
        data['deb_mirrors'] = params['real_deb_mirrors']
    elif params.has_key('deb_mirrors') and params['deb_mirrors']:
        data['deb_mirrors'] = params['deb_mirrors'].split('|')
        
    if "announce" not in data:
        raise ValueError, "The announce URL must be specified, either on the command line or in the downloaded torrrent file"
        
    h = open(f, 'wb')
    h.write(bencode(data))
    h.close()

def calcsize(file):
    """Calculate the size of a file/directory.
    
    @type file: C{string}
    @param file: the file/directory to calculate the size of
    @rtype: C{long}
    @return: the size of the file/directory
    
    """
    
    if not isdir(file):
        return getsize(file)
    total = 0L
    for s in subfiles(abspath(file)):
        total += getsize(s[1])
    return total

class ExtraPieces(deb822._multivalued):
    """For reading sub-piece data from an extrapieces file."""
    _multivalued_fields = {
        "sha1-pieces": [ "SHA1", "size" ],
    }

def getsubpieces(pieces_file):
    """Process the sub-package piece information for the Packages file.
    
    @type pieces_file: C{string}
    @param pieces_file: the file that contains the piece information
    @rtype: C{dictionary}
    @return: the piece info, keys are the file names, values are tuples of 
        a list of piece SHA1 hashes and a list of piece sizes

    """
    
    pieces = {}
    packages = 0
    piece_url = ''
    
    try:
        f = universal_open(pieces_file)
    except:
        logger.exception('sub-pieces file not found: '+pieces_file)
        return {}

    for pkg in ExtraPieces.iter_paragraphs(f):
        if len(str(pkg.get('SHA1', ''))) > 0 and len(pkg.get('SHA1-pieces', [])) > 0:
            sha1 = binascii.a2b_hex(str(pkg['SHA1']))
            pieces[sha1] = ([], [])
            for piece in pkg['SHA1-pieces']:
                pieces[sha1][0].append(binascii.a2b_hex(str(piece['SHA1'])))
                pieces[sha1][1].append(int(str(piece['size'])))

    try:
        f.close()
    except:
        pass

    logger.info('successfully retrieved sub-piece data for '+str(len(pieces))+' files')

    return pieces

class Torrent(deb822._multivalued):
    """For reading piece numbers from a unique piece number torrent file."""
    _multivalued_fields = {
        "piecenumbers": [ "number", "file" ],
    }

def getordering(torrent_file):
    """Retrieve unique piece piece ordering information for the Packages file.
    
    @type torrent_file: C{string}
    @param torrent_file: the file that contains the piece ordering information
    @rtype: C{dictionary}
    @return: the piece ordering info, keys are the starting piece numbers to
        use for the files, values are the file names

    """
    
    pieces = {}
    headers = {}
    piece_url = ''
    
    try:
        f = universal_open(torrent_file)
    except:
        logger.exception('torrent ordering file not found: '+torrent_file)
        return (pieces, headers)

    tor = Torrent(f)
    for header in tor:
        if str(header.lower()) != 'piecenumbers':
            # Read the headers from the file
            headers[str(header)] = str(tor[header])
            
    # Read the piece ordering data from the file
    for piece in tor['PieceNumbers']:
        pieces[int(str(piece['number']))] = str(piece['file'])
            
    logger.info('successfully retrieved torrent ordering data for '+str(len(pieces))+' files')
    logger.debug('headers: %r', headers)
    return (pieces, headers)

def orderpieces(fs, pieces, lengths, separate_all = 1, piece_ordering = {}, num_pieces = 0):
    """Order the pieces appropriately in the info dictionary.
    
    @type fs: C{dictionary}
    @param fs: the files data for the torrent, keys are the file names from the
        Packages file, values are the dictionaries to use for them in the torrent
    @type pieces: C{dictionary}
    @param pieces: piece hashes for the torrent, keys are the file names from the
        Packages file, values are lists of the piece hashes for the file
    @type lengths: C{dictionary}
    @param lengths: lengths of the pieces for the torrent, keys are the file
        names from the Packages file, values are lists of the piece lengths
    @type separate_all: C{boolean}
    @param separate_all: whether to separate the architecture:all packages into
        a separate torrent (optional, defaults to True)
    @type piece_ordering: C{dictionary}
    @param piece_ordering: the piece ordering info to use for the torrent
        (optional, defaults to being ordered by file name)
    @type num_pieces: C{int}
    @param num_pieces: the number of pieces in the piece ordering 
        (optional, but must be specified if using ordering data)
    @rtype: C{dictionary}
    @return: the properly ordered info section of the torrent
    
    """

    if piece_ordering and separate_all:
        pieces_list = ['\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00']*num_pieces
        lengths_list = [0]*num_pieces
        fs_list = []
        
        ordered_pieces = piece_ordering.keys()
        ordered_pieces.sort()
        cur_piece = 0
        for next_piece in ordered_pieces:
            if piece_ordering[next_piece] not in pieces:
                # Skip files we don't know about
                continue

            if next_piece > cur_piece:
                fs_list.append({'length': 0, 'path': []})
                cur_piece = next_piece
            elif next_piece < cur_piece:
                raise ValueError, 'piece ordering is invalid'

            len_pieces = len(pieces[piece_ordering[cur_piece]])
            pieces_list[cur_piece:(cur_piece+len_pieces)] = \
                pieces[piece_ordering[cur_piece]]
            lengths_list[cur_piece:(cur_piece+len_pieces)] = \
                lengths[piece_ordering[cur_piece]]
            fs_list.append(fs[piece_ordering[cur_piece]])
            cur_piece = cur_piece + len_pieces
        if num_pieces > cur_piece:
            fs_list.append({'length': 0, 'path': []})
            cur_piece = num_pieces
    else:
        pieces_list = []
        lengths_list = []
        fs_list = []

        files = fs.keys()
        files.sort()
        for file in files:
            pieces_list.extend(pieces[file])
            lengths_list.extend(lengths[file])
            fs_list.append(fs[file])
        
    return {'pieces': ''.join(pieces_list), 'piece lengths': lengths_list,
            'files': fs_list}


def getpieces(f, encoding = None, progress = lambda x: None, separate_all = 1, 
              sub_pieces = {}, piece_ordering = {}, piece_ordering_all = {},
              num_pieces = 0, num_all_pieces = 0):
    """Extract the piece information from the Packages file.
    
    @type f: C{iterable}
    @param f: the already opened file or file data as a list of strings
    @type encoding: C{string}
    @param encoding: the encoding to use for the file names
        (optional, defaults to the default encoding, or ASCII)
    @type progress: C{method}
    @param progress: the method to call with updates on the progress
        (optional, defaults to not printing progress updates)
    @type separate_all: C{boolean}
    @param separate_all: whether to separate the architecture:all packages into
        a separate torrent (optional, defaults to True)
    @type sub_pieces: C{dictionary}
    @param sub_pieces: the sub-package piece info, keys are the file names,
        values are tuples of a list of piece SHA1 hashes and a list of piece
        sizes (optional, defaults to not using sub-package pieces)
    @type piece_ordering: C{dictionary}
    @param piece_ordering: the piece ordering info to use for the architecture
        specific torrent (optional, defaults to being ordered by file name)
    @type piece_ordering_all: C{dictionary}
    @param piece_ordering_all: the piece ordering info to use for the architecture:all
        torrent (optional, defaults to being ordered by file name)
    @type num_pieces: C{int}
    @param num_pieces: the number of pieces in the piece ordering for the architecture
        specific torrent (optional, but must be specified if using ordering data)
    @type num_all_pieces: C{int}
    @param num_all_pieces: the number of pieces in the piece ordering for the architecture:all
        torrent (optional, but must be specified if using ordering data)
    @rtype: (C{dictionary}, C{dictionary})
    @return: the two torrents, the second is the architecture:all one, if that
        was requested, otherwise it is None
    
    """
    
    if not encoding:
        encoding = ENCODING
    if not encoding:
        encoding = 'ascii'
    
    pieces = ({}, {})
    lengths = ({}, {})
    fs = ({}, {})
    packages = [0, 0]
    info = None
    info_all = None
    
    for pkg in deb822.Packages.iter_paragraphs(f, fields = ['Filename', 'Size', 'SHA1', 'Architecture']):
        if pkg.get('Size', '') and pkg.get('SHA1', '') and pkg.get('Filename', ''):
            # Check which torrent to add the info to
            all = 0
            if (separate_all and str(pkg.get('Architecture', '')) == 'all'):
                all = 1

            sha1 = binascii.a2b_hex(str(pkg['SHA1']))
            if sub_pieces.has_key(sha1):
                lengths[all][str(pkg['Filename'])] = sub_pieces[sha1][1]
                pieces[all][str(pkg['Filename'])] = sub_pieces[sha1][0]
            else:
                lengths[all][str(pkg['Filename'])] = [long(str(pkg['Size']))]
                pieces[all][str(pkg['Filename'])] = [sha1]

            path = []
            temp = str(pkg['Filename'])
            while temp:
                temp,d = split(temp)
                path.insert(0,d)
            fs[all][str(pkg['Filename'])] = {'length': long(str(pkg['Size'])), 'path': uniconvertl(path, encoding)}
            packages[all] += 1
            progress(packages[0] + packages[1])
            
    if packages[0] > 0:
        info = orderpieces(fs[0], pieces[0], lengths[0], separate_all,
                           piece_ordering, num_pieces)
        logger.info('got metainfo for torrent of '+str(len(info['piece lengths']))+
                    ' pieces for '+str(len(info['files']))+' files')

    if packages[1] > 0:
        info_all = orderpieces(fs[1], pieces[1], lengths[1], separate_all,
                               piece_ordering_all, num_all_pieces)
        logger.info('got metainfo for arch:all torrent of ' + 
                    str(len(info_all['piece lengths'])) +
                    ' pieces for '+str(len(info_all['files']))+' files')

    return (info, info_all)

def makeinfo(file, piece_length, encoding, progress, separate_all = 1,
             pieces_file = '', torrent_file = '', torrent_all_file = ''):
    """Open the file and pass it to the getpieces function.
    
    @type file: C{string}
    @param file: the file name of the Packages file to make into a torrent
    @type piece_length: C{int}
    @param piece_length: not used
    @type encoding: C{string}
    @param encoding: the encoding to use for the file names
    @type progress: C{method}
    @param progress: the method to call with updates on the progress
    @type separate_all: C{boolean}
    @param separate_all: whether to separate the architecture:all packages into
        a separate torrent (optional, defaults to True)
    @type pieces_file: C{string}
    @param pieces_file: the file that contains the piece information
        (optional, defaults to not using sub pieces)
    @type torrent_file: C{string}
    @param torrent_file: the file that contains the piece ordering information
        (optional, defaults to retrieving the info from the web)
    @type torrent_all_file: C{string}
    @param torrent_all_file: the file that contains the piece ordering information
        for arch:all (optional, defaults to retrieving the info from the web)
    @rtype: (C{dictionary}, C{dictionary}, C{dictionary}, C{dictionary})
    @return: the two torrents, the second is the architecture:all one, if that
        was requested, otherwise it is None, followed by the ordering headers
        for the torrents
    
    """

    sub_pieces = {}
    if pieces_file:
        sub_pieces = getsubpieces(pieces_file)

    piece_ordering_all = {}
    ordering_all_headers = {}
    piece_ordering = {}
    ordering_headers = {}
    if separate_all:
        if torrent_file:
            (piece_ordering, ordering_headers) = getordering(torrent_file)
        if torrent_all_file:
            (piece_ordering_all, ordering_all_headers) = getordering(torrent_all_file)

    file = abspath(file)
    f = universal_open(file)
    (info, info_all) = getpieces(f, encoding, progress, separate_all, sub_pieces,
                                 piece_ordering, piece_ordering_all,
                                 int(ordering_headers.get('NextPiece', 0)),
                                 int(ordering_all_headers.get('NextPiece', 0)))
    f.close()
    
    return (info, info_all, ordering_headers, ordering_all_headers)

def subfiles(d):
    """Process a directory structure to find all the files in it.
    
    Files in a directory are parsed first before the sub-directory files.
    
    @type d: C{string}
    @param d: the top-level directory to start at
    @rtype: C{list} of (C{list} of C{string}, C{string})
    @return: all the files found in the directory, both as a path list and a
        file name
    
    """
    
    r = []
    stack = [([], d)]
    while len(stack) > 0:
        p, n = stack.pop()
        if isdir(n):
            for s in listdir(n):
                if s not in ignore and s[:1] != '.':
                    stack.append((copy(p) + [s], join(n, s)))
        else:
            r.append((p, n))
    return r


def completedir(dir, params = {}, vc = lambda x: None, fc = lambda x: None):
    """Create a torrent for each file in a directory.
    
    Does not recurse into sub-directories.
    
    @type dir: C{string}
    @param dir: the directory to find files in
    @type params: C{dictionary}
    @param params: the configuration options (optional, defaults to None)
    @type vc: C{method}
    @param vc: progress report while the torrent generation is underway
    @type fc: C{method}
    @param fc: progress report when a new torrent generation is started
    
    """
    
    files = listdir(dir)
    files.sort()
    ext = '.dtorrent'
    if params.has_key('target'):
        target = params['target']
    else:
        target = ''

    togen = []
    for f in files:
        if f[-len(ext):] != ext and (f + ext) not in files:
            togen.append(join(dir, f))
        
    for i in togen:
        fc(i)
        try:
            t = split(i)[-1]
            if t not in ignore and t[0] != '.':
                if target != '':
                    params['target'] = join(target,t+ext)
                make_meta_file(i, params, progress = vc)
        except ValueError:
            print_exc()

class TorrentCreator:
    """Create a torrent metainfo from a downloaded Packages file (threaded).
    
    """
    
    def __init__(self, path, filename, callback, sched, cache, config, identifier):
        """Process a downloaded Packages file and start the torrent making thread.
        
        @type path: C{list} of C{string}
        @param path: the path of the file to download, starting with the mirror name
        @type filename: C{string}
        @param filename: the file containing the downloaded Packages file
        @type callback: C{method}
        @param callback: the method to call with the torrent when it has been created
        @type sched: C{method}
        @param sched: the method to call to schedule future invocation of a function
        @type cache: L{DebTorrent.HTTPCache.HTTPCache}
        @param cache: the cache of downloaded files
        @type config: C{dictionary}
        @param config: the configuration parameters
        @type identifier: C{string}
        @param identifier: an identifier of the origin and label of the repository
        
        """

        self.path = path
        self.filename = filename
        self.callback = callback
        self.sched = sched
        self.cache = cache
        self.config = config
        self.name = '_'.join(self.path[:-1])
        self.responses = []
        self.download = []
        self.subpieces_url = None
        self.subpieces_file = None
        self.torrent_url = None
        self.torrent_all_url = None
        self.torrent_file = None
        self.torrent_all_file = None
        
        if identifier == ('Debian', 'Debian') and 'dists' in self.path:
            parts = self.path[:]
            try:
                parts[parts.index('stable', parts.index('dists'))] = 'etch'
            except:
                pass
            try:
                parts[parts.index('testing', parts.index('dists'))] = 'lenny'
            except:
                pass
            try:
                parts[parts.index('unstable', parts.index('dists'))] = 'sid'
            except:
                pass
            url = 'dists_'
            url += '_'.join(parts[parts.index('dists')+1:])
            if url.endswith('.gz'):
                url = url[:-3]
            if url.endswith('.bz2'):
                url = url[:-4]
            self.subpieces_url = self.config['pieces_url']
            if not self.subpieces_url.endswith('/'):
                self.subpieces_url += '/'
            self.subpieces_url += url + '-extrapieces.gz'
            self.download.append('subpieces')
            if self.config['separate_all']:
                self.torrent_url = self.config['long_lived_torrent_url']
                if not self.torrent_url.endswith('/'):
                    self.torrent_url += '/'
                self.torrent_url += url + '-torrent.gz'
                self.download.append('torrent')
                if self.config['separate_all'] in [1, 3]:
                    self.torrent_all_url = convert_all(self.torrent_url)
                    self.download.append('torrent_all')
                    
        for down in self.download[:]:
            down_path = getattr(self, down + '_url').split('/')
            r, filename = self.cache.cache_get(down_path, True)
            if filename:
                setattr(self, down + '_file', filename)
                self.download.remove(down)
            else:
                self.cache.download_get(down_path, self._download_callback, True)
                
        if not self.download:
            self._start()

    def _download_callback(self, path, r, filename):
        """Save the newly downloaded file's name and move on to creation if downloading is complete.
        
        @type path: C{list} of C{string}
        @param path: the path of the file to download, starting with the mirror name
        @type r: (C{int}, C{string}, C{dictionary}, C{string})
        @param r: the HTTP status code, status message, headers, and cached data
        @type filename: C{string}
        @param filename: the file containing the successfully downloaded file
        
        """

        found = False
        for down in self.download:
            if getattr(self, down + '_url') == '/'.join(path):
                if filename:
                    setattr(self, down + '_file', filename)
                else:
                    logger.warning('Failed to download file: %s', '/'.join(path))
                found = True
                self.download.remove(down)
                break
        
        if not found:
            logger.warning('Got response for unrequested download: %s', '/'.join(path))

        if not self.download:
            self._start()
        
    def _start(self):
        """Create and start the thread to create the torrent metainfo."""
        
        logger.debug('starting thread to create torrent for: '+self.name)
        rq = Thread(target = self._create, name = 'TorrentCreator('+self.name+')')
        rq.setDaemon(False)
        rq.start()
    
    def _create(self):
        """Process a downloaded Packages file and start a torrent."""

        try:
            f = universal_open(self.filename)
        except:
            logger.warning('Packages file could not be opened')
            self.sched(self._finished)
            return

        logger.debug('Packages file successfully opened')
        try:
            sub_pieces = {}
            if self.subpieces_file:
                sub_pieces = getsubpieces(self.subpieces_file)
    
            piece_ordering_all = {}
            ordering_all_headers = {}
            piece_ordering = {}
            ordering_headers = {}
            if self.torrent_file:
                (piece_ordering, ordering_headers) = getordering(self.torrent_file)
            if self.torrent_all_file:
                (piece_ordering_all, ordering_all_headers) = getordering(self.torrent_all_file)
        
            (info, info_all) = getpieces(f, separate_all = self.config['separate_all'],
                                         sub_pieces = sub_pieces,
                                         piece_ordering = piece_ordering,
                                         piece_ordering_all = piece_ordering_all,
                                         num_pieces = int(ordering_headers.get('NextPiece', 0)),
                                         num_all_pieces = int(ordering_all_headers.get('NextPiece', 0)))
            f.close()
        except:
            logger.exception('Failed to create torrent for: %s', self.name)
            f.close()
            return

        name = self.name
        if info and self.config['separate_all'] in (0, 2, 3):
            try:
                self.responses.append(self._create_response(info, ordering_headers, name))
            except:
                logger.exception('Creating the response dictionary failed.')
            
        name = convert_all(self.name)
        if info_all and self.config['separate_all'] in (1, 3):
            try:
                self.responses.append(self._create_response(info_all, ordering_all_headers, name))
            except:
                logger.exception('Creating the response dictionary failed.')
        
        self.sched(self._finished)

    def _create_response(self, info, headers, name):
        """Create the response dictionary for the torrent.
        
        @type info: C{dictionary}
        @param info: the info dictionary to use for the torrent
        @type headers: C{dictionary}
        @param headers: the headers from the torrent file
        @type name: C{string}
        @param name: the name to use for the torrent
        
        """

        response = {'info': info,
                    'announce': self.config['default_tracker'],
                    'name': uniconvert(name)}
        
        if "Tracker" in headers:
            response['announce'] = headers["Tracker"].strip()
            del headers["Tracker"]
        if "Torrent" in headers:
            response['identifier'] = binascii.a2b_hex(headers["Torrent"].strip())
            del headers["Torrent"]
        for header, value in headers.items():
            response[header] = value.strip()

        if self.config["force_tracker"]:
            response['announce'] = self.config["force_tracker"]
            
        if self.path.count('dists'):
            mirror = 'http://' + '/'.join(self.path[:self.path.index('dists')]) + '/'
            response.setdefault('deb_mirrors', []).append(mirror)
            
        return (response, name)

    def _finished(self):
        """Wrap up the creation and call the callback function."""
        
        for (response, name) in self.responses:
            try:
                check_message(response)
            except:
                logger.exception('Poorly formatted torrent, not starting')
            else:
                self.callback(response, name, self.path)
        
        del self.responses[:]
